/* pcimem_utility.c
 *
 * 03/18/2021 Pushkar Singh
 *
 * Sangoma PCI cards debug registers interface.
 *
 * This tool provides a debug interface for reading and writing
 * to PCI registers via the device base address registers (BARs)
 * for Sangoma cards.
 * The tool uses the PCI resource nodes automatically created
 * by recently Linux kernels.
 *
 * The readline library is used for the command line interface
 * so that up-arrow command recall works. Command-line history
 * is not implemented. Use -lreadline -lcurses when building.
 *
 * And is only supported on Linux as of now
 *
 * Upgrade on: 04/30/2021 by Pushkar Singh <psingh@sangoma.com>
 * ----------------------------------------------------------------
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <byteswap.h>
#include <stdbool.h>
#include <string.h>
#include <stdlib.h>
/* Readline support */
#include <readline/readline.h>
#include <readline/history.h>

#define AFT_MCPU_INTERFACE_ADDR         0x46
#define AFT_MCPU_INTERFACE              0x44

#define READ_CMD 0x01
#define WRITE_CMD 0x02

#define FILE_PARAMS 0x03

/* PCI device */
typedef struct {
    /* Base address region */
    unsigned int bar;

    /* Slot info */
    unsigned int domain;
    unsigned int bus;
    unsigned int slot;
    unsigned int function;

    /* Resource filename */
    char         filename[100];

    /* File descriptor of the resource */
    int          fd;

    /* Memory mapped resource */
    unsigned char *maddr;
    unsigned int   size;
    unsigned int   offset;

    /* PCI physical address */
    unsigned int   phys;

    /* Address to pass to read/write (includes offset) */
    unsigned char *addr;
} device_t;

void display_help(device_t *dev);
void parse_command(device_t *dev);
int process_command(device_t *dev, char *cmd);
int change_mem(device_t *dev, char *cmd);
int write_verify_mem(device_t *dev, char *cmd);
int display_mem(device_t *dev, char *cmd);
int execute_file_commands(device_t *dev, char *cmd);
int change_endian(device_t *dev, char *cmd);
int verify_fill_cmd(char *cmd, int *operation, char *file_cmd);
int set_sleep_time(char *cmd);

/* Endian read/write mode */
static int big_endian = 0;

/* sleep time */
int s_time = 0;

/* Low-level access functions */
static void write_8
(
    device_t     *dev,
    unsigned int  addr,
    unsigned char data
);

static unsigned char read_8
(
    device_t    *dev,
    unsigned int addr
);

static void write_le16
(
    device_t          *dev,
    unsigned int       addr,
    unsigned short int data
);

static unsigned short int read_le16
(
    device_t    *dev,
    unsigned int addr
);

static void write_be16
(
    device_t          *dev,
    unsigned int       addr,
   unsigned short int data
);

static unsigned short int read_be16
(
    device_t    *dev,
    unsigned int addr
);

static void write_le32
(
    device_t    *dev,
    unsigned int addr,
    unsigned int data
);

static unsigned int read_le32
(
    device_t    *dev,
    unsigned int addr
);

static void write_be32
(
    device_t    *dev,
    unsigned int addr,
    unsigned int data
);

static unsigned int read_be32
(
    device_t    *dev,
    unsigned int addr
);


/* Usage */
static void show_usage()
{
    printf("\nUsage: pcimem_utility -s <device>\n"\
            "                       -h Help       (this message)\n"\
            "                       -s <device>   Slot/device (as per lspci)\n" \
            "                       -b <BAR>      Base address region (BAR) to access, eg. 0 for BAR0\n\n");
}

int main(int argc, char *argv[])
{
    int opt;
    char *slot = 0;
    int status;
    struct stat statbuf;
    device_t device;
    device_t *dev = &device;

    /* Clear the structure fields */
    memset(dev, 0, sizeof(device_t));

    while ((opt = getopt(argc, argv, "b:hs:")) != -1)
    {
        switch (opt)
	{
	    case 'b':
    		/* Defaults to BAR0 if not provided */
    		dev->bar = atoi(optarg);
   	    break;

    	    case 'h':
    		show_usage();
    	        return -1;
   	    break;

    	    case 's':
    		slot = optarg;
   	    break;

    	    default:
    		show_usage();
    		return -1;
   	    break;
    	}
    }

    if (slot == 0)
    {
    	show_usage();
    	return -1;
    }
    
    /* ------------------------------------------------------------
     * Open and map the PCI region
     * ------------------------------------------------------------
     */
    
    /* Extract the PCI parameters from the slot string */
    status = sscanf(slot, "%2x:%2x.%1x",
    		&dev->bus, &dev->slot, &dev->function);

    if (status != 3)
    {
    	printf("Error parsing slot information!\n");
    	show_usage();
    	return -1;
    }
    
    /* Convert to a sysfs resource filename and open the resource */
    snprintf(dev->filename, 99, "/sys/bus/pci/devices/%04x:%02x:%02x.%1x/resource%d",
    		dev->domain, dev->bus, dev->slot, dev->function, dev->bar);
    
    printf("filename = %s\n", dev->filename);
    
    dev->fd = open(dev->filename, O_RDWR | O_SYNC);
    if (dev->fd < 0)
    {
    	printf("Open failed for file '%s': errno %d, %s\n",
    		dev->filename, errno, strerror(errno));
    	return -1;
    }
    
    
    /* PCI memory size */
    status = fstat(dev->fd, &statbuf);
    if (status < 0)
    {
    	printf("fstat() failed: errno %d, %s\n",
    		errno, strerror(errno));
    	return -1;
    }

    dev->size = statbuf.st_size;
    
    /* Map */
    dev->maddr = (unsigned char *)mmap( NULL, (size_t)(dev->size), PROT_READ|PROT_WRITE, MAP_SHARED, dev->fd, 0);

    if (dev->maddr == (unsigned char *)MAP_FAILED)
    {
    	printf("failed (mmap returned MAP_FAILED)\n");
    	printf("BARs that are I/O ports are not supported by this tool\n");
    	dev->maddr = 0;
    	close(dev->fd);
    	return -1;
    }
    
    /* Device regions smaller than a 4k page in size can be offset
     * relative to the mapped base address. The offset is
     * the physical address modulo 4k
     */
    {
    	char configname[100];
    	int fd;
    
    	snprintf(configname, 99, "/sys/bus/pci/devices/%04x:%02x:%02x.%1x/config",
    			dev->domain, dev->bus, dev->slot, dev->function);

    	fd = open(configname, O_RDWR | O_SYNC);
    	if (dev->fd < 0)
	{
    		printf("Open failed for file '%s': errno %d, %s\n",
    			configname, errno, strerror(errno));
    		return -1;
    	}
    
    	status = lseek(fd, 0x10 + 4*dev->bar, SEEK_SET);
    	if (status < 0)
	{
    		printf("Error: configuration space lseek failed\n");
    		close(fd);
    		return -1;
    	}

    	status = read(fd, &dev->phys, 4);
    	if (status < 0)
	{
    		printf("Error: configuration space read failed\n");
    		close(fd);
    		return -1;
    	}

    	dev->offset = ((dev->phys & 0xFFFFFFF0) % 0x1000);
    	dev->addr = dev->maddr + dev->offset;
    	close(fd);
    }
    
    
    /* ------------------------------------------------------------
     * Tests
     * ------------------------------------------------------------
     */
    
    printf("\n");
    printf("PCI Device %s Info:\n");
    printf("----------------------------------------------------------------\n\n");
    printf(" - device virually map to %p\n", dev->addr);
    printf(" - accessing BAR%d\n", dev->bar);
    printf(" - region size is %d-bytes\n", dev->size);
    printf(" - offset into region is %d-bytes\n", dev->offset);
    
    /* Display help */
    display_help(dev);
    
    /* Process commands */
    parse_command(dev);
    
    /* Cleanly shutdown */
    munmap(dev->maddr, dev->size);
    close(dev->fd);
    return 0;
}

void parse_command(device_t *dev)
{
    char *line;
    int len;
    int status;
    
    while(1)
    {
    	line = readline("PCI> ");

    	/* Ctrl-D check */
    	if (line == NULL) {
    		printf("\n");
    		continue;
    	}

    	/* Empty line check */
    	len = strlen(line);
    	if (len == 0)
	{
    		continue;
    	}

    	/* Process the line */
    	status = process_command(dev, line);
    	if (status < 0)
	{
    		break;
    	}
    
    	/* Add it to the history */
    	add_history(line);
    	free(line);
    }
    return;
}

/*--------------------------------------------------------------------
 * User interface
 *--------------------------------------------------------------------
 */
void display_help( device_t *dev )
{
    printf("\n");
    printf("  ?                                            Help\n");
    printf("  r[width] [device to be read] addr len        Display/Read memory starting from addr\n");
    printf("                                               [width]\n");
    printf("                                                 b  - byte / 8-bit access (default)\n");
    printf("                                                 h  - word / 16-bit access\n");
    printf("                                                 w  - word / 32-bit access\n");
    printf("                                               [device to be read]\n");
    printf("                                                 c  - card registers (default)\n");
    printf("                                                 f  - framer registers\n");
    printf("                                               len  - number of items to be displayed\n");
    printf("  w[width] [device to be read] addr val len v  Write/Fill memory\n");
    printf("                                               [width]\n");
    printf("                                                 b  - byte / 8-bit access (default)\n");
    printf("                                                 h  - word / 16-bit access\n");
    printf("                                                 w  - word / 32-bit access\n");
    printf("                                               [device to be read]\n");
    printf("                                                 c  - card registers (default)\n");
    printf("                                                 f  - framer registers\n");
    printf("                                               addr - start address\n");
    printf("                                               val  - value to be written\n");
    printf("                                               len  - number of items to be displayed\n");
    printf("                                               v    - if verification is required \n");
    printf(" f [<path-with-filename]                       execute command as preset within file \n");
    printf("                                               [operations supported in file]\n");
    printf("                                                 r - read command operation \n");
    printf("                                                 w - write command operation \n");
    printf(" s [time]                                      set sleep time interval in milli second between executing command from file \n");
    printf("  q                                            Quit\n");
    printf("\n");
}

int process_command(device_t *dev, char *cmd)
{
    if (cmd[0] == '\0')
    {
    	return 0;
    }

    switch (cmd[0])
    {
    	case '?':
    	case 'h':
    	case 'H':
    		display_help(dev);
    	break;

    	case 'r':
    	case 'R':
    		return display_mem(dev, cmd);
    	break;

    	case 'w':
    	case 'W':
    		return write_verify_mem(dev, cmd);
    	break;

    	case 'f':
    	case 'F':
		execute_file_commands(dev, cmd);
	break;

    	case 's':
    	case 'S':
		set_sleep_time(cmd);
	break;
    	case 'q':
    	case 'Q':
    		return -1;
    	break;

    	default:
	break;
    }

    return 0;
}

int set_sleep_time(char *cmd)
{
    int sleep_time = 0;
    int status = 0;
    char dummy ;
    if (cmd[1] == ' ')
    {
        status = sscanf(cmd, "%c %d", &dummy, &sleep_time);
        if (status != 2)
        {
	        printf("Syntax error (use ? for help)\n");
            /* Don't break out of command processing loop */
            return 0;
        }
    }

    printf("Setting sleep time:%d\n", sleep_time);
    s_time = sleep_time;

}

int execute_file_commands(device_t *dev, char *cmd)
{
    char filename[256] = { '\0' };
    char file_cmd[512] = { '\0' };
    char curr_dir[64] = { '\0' };
    char *tmp_name = NULL;
    int operation = 0x00;
    char *line = NULL;
    FILE *fp = NULL;
    size_t len = 0;
    int ret = 0;
    ssize_t read;

    /* incrementing cmd by 1 as we expect cmd[0] to be f i.e. for file operation */
    cmd++; 
    tmp_name = strtok(cmd, " \n");

    line = strchr(tmp_name, '/');

    if( line == NULL )
    {
        getcwd( curr_dir, 64 );
        sprintf(filename, "%s/%s", curr_dir, tmp_name);
    }
    else
        strcpy(filename, line);

    fp = fopen( filename, "r");
    if (fp == NULL)
    {
        printf("Not able to open file: %s\n", filename);
        return -1;
    }

    line = NULL;
    tmp_name = NULL;

    
    while( (read = getline(&line, &len, fp) ) != -1)
    {
        printf("*********************************************************************************\n");

        if( line[read -2] == '\r' )
            line[read-2]='\0';
        if( line[read -1] == '\n' )
            line[read-1]='\0';
        else
            line[read-1]='\0';

        if( line[0] == '#' || line[1] == '#' )
        {
	    fflush(stdout);
            printf("Ignoring comment: [%s]\n", line);
            printf("*********************************************************************************\n\n");
            continue;
        }
	else if ( line[0] == '\r' || line [0] == '\n' )
        {
	    fflush(stdout);
            printf("Ignoring empty line\n");
            printf("*********************************************************************************\n\n");
            continue;
        } 
//      printf("Retrieved line of length %zu:\n", read);
        ret = verify_fill_cmd(line, &operation, file_cmd);
        if ( ret )
        {
	    fflush(stdout);
            if( ret == 1 )
                printf("Invalid Command found: [%s]\n", line);
            
            printf("*********************************************************************************\n\n");
            continue;
        }

	fflush(stdout);
        switch (operation)
        {
            case READ_CMD:
                printf("Executing Read command: [%s]\n", line);
                display_mem(dev, file_cmd);
            break;

            case WRITE_CMD:
                printf("Executing Write command: [%s]\n", line);
                write_verify_mem(dev, file_cmd);
            break;

            default:
               printf("Invalid operation type [%d]\n", operation);
            break;
        }
        printf("*********************************************************************************\n\n");

	sleep(s_time/1000);
    }

    fclose(fp);
    if (line)
        free(line);

    return 0;
}

int verify_fill_cmd(char *cmd, int *operation, char *file_cmd)
{
    char str[512] = { '\0' };
    int num_params = 0;
    char *token = NULL;
    int idx = 0;
    bool found_comment = false;

    strcpy(str, cmd);
    token = strtok(str, " \r\n");

    if( token != NULL )
    {
        num_params++;
        switch (token[0])
        {
            case 'r':
            case 'R':
                *operation = READ_CMD;
                sprintf(file_cmd, "%s", "r");
            break;
    
            case 'w':
            case 'W':
                *operation = WRITE_CMD;
                sprintf(file_cmd, "%s", "w");
            break;
	
	    case 's':
	    case 'S':
		/* Sleep setting*/
                printf("Sleep setting !!\n");
		set_sleep_time(cmd);
		return 0;
		break;
    
            default:
                printf("Only Read/Write i.e. r/w operations supported\n");
                return 1;
            break;
        }

	switch (token[1])
        {
            case 'b':
            case 'B':
                sprintf(file_cmd, "%sb", file_cmd);
            break;
    
            case 'h':
            case 'H':
                sprintf(file_cmd, "%sh", file_cmd);
            break;
    
            case 'w':
            case 'W':
                sprintf(file_cmd, "%sw", file_cmd);
            break;

            default:
                sprintf(file_cmd, "%sb", file_cmd);
            break;
        }
    }
    else
    {
        printf("Ignoring Empty line\n", cmd);
        return 2;
    }

    while( (token = strtok(NULL, " \r\n")) != NULL )
    {
        num_params++;

        printf("token = %s and num_params = %d\n", token, num_params);
	fflush(stdout);
        switch (num_params)
        {
            case 2:
                if( ((token[0] == 'c') || (token[0] == 'C')) && (strlen(token) == 1) )
                {
                    sprintf(file_cmd, "%s c", file_cmd);
                    printf("Performing Operation on Card PCI config space\n");
                }
                else if ( ((token[0] == 'f') || (token[0] == 'F')) && (strlen(token) == 1) )
                {
                    sprintf(file_cmd, "%s f", file_cmd);
                    printf("Performing Operation on Framer config space\n");
                }
                else
                {
                    printf("Unknown device type[%s] to perform operation\n", token);
                    return 1;
                }
            break;

            case 3:
            case 4:
                for( idx = 0; token[idx] != '\0'; idx++ )
                {
                    if ( isdigit(token[idx]) )
                        continue;
                    else if ( (token[idx] == 'x' || token[idx] == 'X') && (idx == 1) )
                        continue;
                    else if ( (token[idx] >= 'a' && token[idx] <= 'f') || (token[idx] >= 'A' && token[idx] <= 'F') )
                        continue;
                    else
                    {
                        printf("Invalid %s[%s] passed in command\n", (num_params==2) ? "address":"value/length", token);
                        return 1;
                    }
                }

                sprintf(file_cmd, "%s %s", file_cmd, token);
            break;
    
            default:
                if( token[0] == '#' )
		{
		    found_comment = true;
                    break;
		}

                printf("Invalid num of params %d passed. Maximum 3 params are allowd [device type] [address] [value/length]\n");
                return 1;
            break;
        }

	if( found_comment )
	    break;
    }

    if ( *operation == WRITE_CMD )
        sprintf(file_cmd, "%s 1 v", file_cmd);

    fflush(stdout);
    return 0;
}

#define MOD_TYPE_FXO 0x02
#define SPI_INTERFACE_REG	0x54

#define MOD_SPI_ADDR_FXS_READ	0x80
#define MOD_SPI_ADDR_FXS_WRITE	0x00

#define MOD_SPI_CS_FXS_CHIP_0	0x01
#define MOD_SPI_CS_FXS_CHIP_1	0x02

#define MOD_SPI_CS_FXO_CHIP_1	0x08
#define MOD_SPI_CS_FXO_WRITE	0x00
#define MOD_SPI_CS_FXO_READ	0x40

#define MOD_SPI_CTRL_V3_START	0x08	/* bit 27 used as START in version 1-2-3 */
#define MOD_SPI_CTRL_START	0x80
#define MOD_SPI_CTRL_FXS	0x10
#define MOD_SPI_CTRL_CHAIN	0x20

#define MOD_SPI_RESET		0x40000000
#define MOD_SPI_BUSY		0x80000000
#define MOD_SPI_V3_STAT		0x08000000	/* bit 27 used as START in version 1-2-3 */
#define MOD_SPI_START		0x08000000

unsigned int write_spi_remora(device_t *dev, int addr, unsigned char wvalue, int mod_no, int chain){
	int		type;
	int		reg, value;
	unsigned int	data = 0;
	unsigned char	cs = 0x00, ctrl_byte = 0x00;
	int		i;

	reg	= addr;
	value	= wvalue;
	type	= MOD_TYPE_FXO;

	if(1)printf("%s:%d: Module %d: chain:%d Write RM FE code (reg %d, value %02X)!\n",
				__FUNCTION__,__LINE__,
				/*FIXME: hw->devname,*/mod_no, chain, reg, (unsigned char )value);
	//mod_no	= 2;
	//chain	= 1;
#if 0
	if (!wan_test_bit(mod_no, card->fe.fe_param.remora.module_map)){
		DEBUG_ERROR("%s: %s:%d: Internal Error: Module %d\n",
			card->devname, __FUNCTION__,__LINE__,mod_no);
		return -EINVAL;
	}
#endif
	
	/* bit 0-7: data byte */
	data = value & 0xFF;
	if (type == MOD_TYPE_FXO){

		/* bit 8-15: register number */
		data |= (reg & 0xFF) << 8;

		/* bit 16-23: chip select byte
		** bit 16
		**
		**
		**			*/
		cs = 0x20;
		cs |= MOD_SPI_CS_FXO_WRITE;
		if (mod_no % 2 == 0){
			/* Select second chip in a chain */
			cs |= MOD_SPI_CS_FXO_CHIP_1;
		}
		data |= (cs & 0xFF) << 16;

		/* bit 24-31: ctrl byte
		** bit 24
		**
		**
		**			*/
		ctrl_byte = mod_no / 2;
#if !defined(SPI2STEP)
		//if (hw->hwcpu->hwcard->core_rev > 3){
		if (1){
			ctrl_byte |= MOD_SPI_CTRL_START;
		}else{
			ctrl_byte |= MOD_SPI_CTRL_V3_START;
		}
#endif
		ctrl_byte |= MOD_SPI_CTRL_CHAIN;	/* always chain */
		data |= ctrl_byte << 24;

	}
#if 0
	else if (type == MOD_TYPE_FXS){

		/* bit 8-15: register byte */
		reg = reg & 0x7F;
		reg |= MOD_SPI_ADDR_FXS_WRITE; 
		data |= (reg & 0xFF) << 8;
		
		/* bit 16-23: chip select byte
		** bit 16
		**
		**
		**			*/
		if (mod_no % 2){
			/* Select first chip in a chain */
			cs = MOD_SPI_CS_FXS_CHIP_0;
		}else{
			/* Select second chip in a chain */
			cs = MOD_SPI_CS_FXS_CHIP_1;
		}
		data |= cs << 16;

		/* bit 24-31: ctrl byte
		** bit 24
		**
		**
		**			*/
		ctrl_byte = mod_no / 2;
#if !defined(SPI2STEP)
		//if (hw->hwcpu->hwcard->core_rev > 3){
		if(1){
			ctrl_byte |= MOD_SPI_CTRL_START;
		}else{
			ctrl_byte |= MOD_SPI_CTRL_V3_START;
		}
#endif
		ctrl_byte |= MOD_SPI_CTRL_FXS;
		if (chain){
			ctrl_byte |= MOD_SPI_CTRL_CHAIN;
		}
		data |= ctrl_byte << 24;

	}
#endif 
	else{
		printf("%s: Module %d: Unsupported module type %d!\n",
				dev->filename, mod_no, type);
		return -EINVAL;
	}

	//sdla_bus_write_4(hw, SPI_INTERFACE_REG, data);	
    	if (big_endian == 0)
		write_le32(dev, SPI_INTERFACE_REG, data);	
	else
		write_be32(dev, SPI_INTERFACE_REG, data);	
#if defined(SPI2STEP)
	sleep(0.1);
	//if (hw->hwcpu->hwcard->core_rev > 3){
	if (1){
		data |= MOD_SPI_START;
	}else{
		data |= MOD_SPI_V3_START;
	}
	//sdla_bus_write_4(hw, SPI_INTERFACE_REG, data);	
    	if (big_endian == 0)
		write_le32(dev, SPI_INTERFACE_REG, data);	
	else
		write_be32(dev, SPI_INTERFACE_REG, data);	
#endif
#if 0
	DEBUG_EVENT("%s: %s: Module %d - Execute SPI command %08X\n",
					card->fe.name,
					__FUNCTION__,
					mod_no,
					data);
#endif

	for (i=0;i<10;i++){	
		sleep(0.5);
		//sdla_bus_read_4(hw, SPI_INTERFACE_REG, &data);
    		if (big_endian == 0)
		{
			data = read_le32(dev, SPI_INTERFACE_REG);	
    		}
		else
		{
			data = read_be32(dev, SPI_INTERFACE_REG);	
    		}

		if (data & MOD_SPI_BUSY){
			continue;
		}
	}

	if (data & MOD_SPI_BUSY) {
		printf("%s: Module %d: Critical Error (%s:%d)!\n",
					dev->filename, mod_no,
					__FUNCTION__,__LINE__);
		return -EINVAL;
	}
        return 0;
}

unsigned int read_spi_remora(device_t *dev, int addr, int mod_no, int chain){
	int		type, reg;
	unsigned int 	data = 0;
	unsigned char	cs = 0x00, ctrl_byte = 0x00;
	int		i;

	//mod_no	= 2;
	type	= MOD_TYPE_FXO;
	//chain	= 1;
	reg	= addr;

	/* bit 0-7: data byte */
	data = 0x00;
	if (type == MOD_TYPE_FXO){
		//printf("%s: Remora Read Reg %X mod_no:%02X, type:%02X, chain:%02X \n", dev->filename, reg, mod_no, type, chain);

		/* bit 8-15: register byte */
		data |= (reg & 0xFF) << 8;

		/* bit 16-23: chip select byte
		** bit 16
		**
		**
		**			*/
		cs = 0x20;
		cs |= MOD_SPI_CS_FXO_READ;
		if (mod_no % 2 == 0){
			/* Select second chip in a chain */
			cs |= MOD_SPI_CS_FXO_CHIP_1;
		}
		data |= (cs & 0xFF) << 16;

		/* bit 24-31: ctrl byte
		** bit 24
		**
		**
		**			*/
		ctrl_byte = mod_no / 2;
#if !defined(SPI2STEP)
		//printf("DEBUG %s:%d  reg:0x%02X wdata:0x%08X\n", __func__, __LINE__, reg, data);
		//if (hw->hwcpu->hwcard->core_rev > 3){
		if (1){
			ctrl_byte |= MOD_SPI_CTRL_START;
		}else{
			ctrl_byte |= MOD_SPI_CTRL_V3_START;
		}
#endif
		ctrl_byte |= MOD_SPI_CTRL_CHAIN;	/* always chain */
		data |= ctrl_byte << 24;
		//printf("DEBUG %s:%d  reg:0x%02X wdata:0x%08X\n", __func__, __LINE__, reg, data);

	}
#if 0 
	else if (type == MOD_TYPE_FXS){

		/* bit 8-15: register byte */
		reg = reg & 0x7F;
		reg |= MOD_SPI_ADDR_FXS_READ; 
		data |= (reg & 0xFF) << 8;
		
		/* bit 16-23: chip select byte
		** bit 16
		**
		**
		**			*/
		if (mod_no % 2){
			/* Select first chip in a chain */
			cs = MOD_SPI_CS_FXS_CHIP_0;
		}else{
			/* Select second chip in a chain */
			cs = MOD_SPI_CS_FXS_CHIP_1;
		}
		data |= cs << 16;

		/* bit 24-31: ctrl byte
		** bit 24
		**
		**
		**			*/
		ctrl_byte = mod_no / 2;
#if !defined(SPI2STEP)
		if (hw->hwcpu->hwcard->core_rev > 3){
			ctrl_byte |= MOD_SPI_CTRL_START;
		}else{
			ctrl_byte |= MOD_SPI_CTRL_V3_START;
		}
#endif
		ctrl_byte |= MOD_SPI_CTRL_FXS;
		if (chain){
			ctrl_byte |= MOD_SPI_CTRL_CHAIN;
		}
		data |= ctrl_byte << 24;

	}
#endif
	else{
		printf("%s: Module %d: Unsupported module type %d!\n",
				dev->filename, mod_no, type);
		return -EINVAL;
	}

	//sdla_bus_write_4(hw, SPI_INTERFACE_REG, data);	
    	if (big_endian == 0)
	{
		write_le32(dev, SPI_INTERFACE_REG, data);	
    	}
	else
	{
		write_be32(dev, SPI_INTERFACE_REG, data);	
    	}
#if defined(SPI2STEP)
	sleep(0.2);
	//printf("DEBUG %s:%d  reg:0x%02X wdata:0x%08X\n", __func__, __LINE__, reg, data);
	//if (hw->hwcpu->hwcard->core_rev > 3){
	if (1){
		data |= MOD_SPI_START;
	}else{
		data |= MOD_SPI_V3_START;
	}
    	if (big_endian == 0)
	{
		write_le32(dev, SPI_INTERFACE_REG, data);	
    	}
	else
	{
		write_be32(dev, SPI_INTERFACE_REG, data);	
    	}
#endif
#if 0
	DEBUG_EVENT("%s: %s: Module %d - Execute SPI command %08X\n",
					hw->devname,
					__FUNCTION__,
					mod_no,
					data);
#endif
	for (i=0;i<10;i++){
		sleep(0.5);
    		if (big_endian == 0)
		{
			data = read_le32(dev, SPI_INTERFACE_REG);	
    		}
		else
		{
			data = read_be32(dev, SPI_INTERFACE_REG);	
    		}
		//printf("DEBUG %s:%d  reg:0x%02X rdata:0x%08X\n", __func__, __LINE__, reg, data);
		if (!(data & MOD_SPI_BUSY)) 
			break;
	}

	if (data & MOD_SPI_BUSY){
		printf("%s: Module %d: Critical Error (%s:%d)!\n",
					dev->filename, mod_no,
					__FUNCTION__,__LINE__);
		return 0xFF;
	}

	return (unsigned char)(data & 0xFF);
}

int display_mem(device_t *dev, char *cmd)
{
    int width = 'b';
    int reg_type = 'c';
    int addr = 0;
    int len = 0;
    int status;
    int i;
    unsigned char d8;
    unsigned short d16, tmp_d16;
    unsigned int d32;

    /* d, db, dw, dd */
    if (cmd[1] == ' ')
    {
        status = sscanf(cmd, "%*c %c %x %x", &reg_type, &addr, &len);
        if (status != 3)
        {
	        printf("Syntax error (use ? for help)\n");
            /* Don't break out of command processing loop */
            return 0;
        }
    }
    else
    {
	     status = sscanf(cmd, "%*c%c %c %x %x", &width, &reg_type, &addr, &len);

        if (status != 4)
        {
            printf("Syntax error (use ? for help)\n");
            /* Don't break out of command processing loop */
	        return 0;
        }
    }

    if (addr > dev->size)
    {
    	printf("Error: invalid address (maximum allowed is %.8X\n", dev->size);
    	return 0;
    }

    if ( (reg_type == 'c') || (reg_type == 'C') )
        reg_type='c';
    else if ( (reg_type == 'f') || (reg_type == 'F') )
        reg_type='f';

    /* Length is in bytes */
    if ((addr + len) > dev->size)
    {
    	/* Truncate */
    	len = dev->size;
    }
    
    switch (width)
    {
    	case 'b':
    	case 'B':
    		for (i = 0; i < len; i++)
		{
 		    if ((i%16) == 0)
		    {
    		        printf("\n%.8X: ", addr+i);
 		    }
    
                    if( reg_type == 'f' )
                    {
    			/* First Read AFT_MCPU_INTERFACE_ADDR */
    			if (big_endian == 0)
			{
    				tmp_d16 = read_le16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
			else
			{
    				tmp_d16 = read_be16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
    
                        /* Write CHIP Reg address that needs to be read */
                        d16 = (unsigned short)(addr + i);
    			if (big_endian == 0)
			{
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, d16);
    			}
			else
			{
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, d16);
    			}

                        /* READ Framer Reg address that needs to be read */
    			d8 = read_8(dev, AFT_MCPU_INTERFACE);
                    }else if(reg_type == 's'){
			unsigned char dummy[8];
			int modno, chain;
	     		status = sscanf(cmd, "%*c%c %c %x %x %x %x", &dummy[0], &dummy[1], &dummy[2], &dummy[4], &modno, &chain);
			d8 = read_spi_remora(dev, addr+i, modno, chain);
		    }
                    else if(reg_type == 'c')
                        /* READ Card Reg address that needs to be read */
    			d8 = read_8(dev, addr+i);

    			printf("%.2X ", d8);
    
                    if( reg_type == 'f' )
                    {
                        /* Write Back the orig value */
    			if (big_endian == 0) {
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d16);
    			} else {
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d16);
    			}
                    }
    		}
    		printf("\n");
    		break;
    
    	case 'h':
    	case 'H':
    		for (i = 0; i < len; i+=2)
		{
    			if ((i%16) == 0)
			{
    				printf("\n%.8X: ", addr+i);
    			}
    
                    if( reg_type == 'f' )
                    {
    			/* First Read AFT_MCPU_INTERFACE_ADDR */
    			if (big_endian == 0)
                        {
    				tmp_d16 = read_le16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
                        else
                        {
    				tmp_d16 = read_be16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
    
                        /* Write CHIP Reg address that needs to be read */
                        d16 = (unsigned short)(addr + i);
    			if (big_endian == 0)
                        {
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, d16);
    			}
                        else
                        {
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, d16);
    			}
   
                    }

  		    if (big_endian == 0)
                    {
                        if( reg_type == 'f' )
                            /* READ Framer Reg address that needs to be read */
    		 	    d16 = read_le16(dev, AFT_MCPU_INTERFACE);
                        else if(reg_type == 'c')
                            /* READ Card Reg address that needs to be read */
    		 	    d16 = read_le16(dev, addr+i);
    		    }
                    else
                    {
                        if( reg_type == 'f' )
                            /* READ Framer Reg address that needs to be read */
    			    d16 = read_be16(dev, AFT_MCPU_INTERFACE);
                        else if(reg_type == 'c')
                            /* READ Card Reg address that needs to be read */
    			    d16 = read_be16(dev, addr+i);
    		    }
		    printf("%.4X ", d16);
    
                    if( reg_type == 'f' )
                    {
                        /* Write Back the orig value */
    			if (big_endian == 0)
                        {
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d16);
    			}
                        else
                        {
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d16);
    			}
                    } 
    		}
    		printf("\n");
    		break;
    
            case 'w':
            case 'W':
                    if( reg_type == 'f' )
                        printf("Utility does not support word read/write operation for Framer");
                    else if( reg_type == 'c' )
                    {
                        for (i = 0; i < len; i+=4)
                        {
                            if ((i%16) == 0)
                            {
                                printf("\n%.8X: ", addr+i);
                            }

                            if (big_endian == 0)
                            {
                                d32 = read_le32(dev, addr+i);
                            }
                            else
                            {
                                d32 = read_be32(dev, addr+i);
                            }
                            printf("%.8X ", d32);
                        }
                    }

                    printf("\n");
    		break;
    
    	default:
    		printf("Syntax error (use ? for help)\n");
    		/* Don't break out of command processing loop */
    		break;
    }
    printf("\n");
    return 0;
}

int write_verify_mem(device_t *dev, char *cmd)
{
    int width = 'b';
    int reg_type = 'c';
    int verify = 't';
    int addr = 0, tmp_addr = 0;
    int len = 0;
    int status;
    int i;
    unsigned char d8;
    unsigned short d16;
    unsigned int d32, tmp_d32;
    
    if (cmd[1] == ' ')
    {
    	status = sscanf(cmd, "%*c %c %x %x %x %c", &reg_type, &addr, &d32, &len, &verify);
    	if ((status != 4) && (status != 5))
	{
                if( status == 3 )
                {
			len = 1;
			verify='v';
		}
		else
		{
	    		printf("Syntax error (use ? for help)\n");
    			/* Don't break out of command processing loop */

	    		return 0;
		}
    	}
    }
    else
    {
    	status = sscanf(cmd, "%*c%c %c %x %x %x %c", &width, &reg_type, &addr, &d32, &len, &verify);

    	if ((status != 5) && (status != 6))
	{
		if( status == 4 )
                {
			len = 1;
			verify='v';
                }
                else
                {
    		    printf("Syntax error (use ? for help)\n");
      		    /* Don't break out of command processing loop */
    		    return 0;
                }
    	}
    }

    if (addr > dev->size)
    {
    	printf("Error: invalid address (maximum allowed is %.8X\n", dev->size);
    	return 0;
    }

    if ( (reg_type == 'c') || (reg_type == 'C') )
        reg_type='c';
    else if ( (reg_type == 'f') || (reg_type == 'F') )
        reg_type='f';

    /* Length is in bytes */
    if ((addr + len) > dev->size)
    {
    	/* Truncate */
    	len = dev->size;
    }
    
    switch (width)
    {
    	case 'b':
    	case 'B':
    		for (i = 0; i < len; i++)
		{
                    if( reg_type == 'f' )
                    {
    			/* First Read AFT_MCPU_INTERFACE_ADDR */
    			if (big_endian == 0)
			{
    				tmp_d32 = read_le16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
			else
			{
    				tmp_d32 = read_be16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
    
                        /* Write CHIP Reg address that needs to be read */
                        tmp_addr = (unsigned short)(addr + i);

    			if (big_endian == 0)
			{
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_addr);
    			}
			else
			{
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_addr);
    			}
    
                        /* Write the value */
    			d8 = (unsigned char)(d32 + i);
    			write_8(dev, AFT_MCPU_INTERFACE, d8);
                    }else if(reg_type == 's'){
			unsigned char dummy[8];
			int modno, chain;
                        d8 = (unsigned char)(d32 + i);
	     		status = sscanf(cmd, "%*c%c %c %x %x %x %x %x %x", &dummy[0], &dummy[1], &dummy[2], &dummy[4], &dummy[5], &dummy[6], &modno, &chain);
                        write_spi_remora(dev, addr+i, d8, modno, chain);

		    } else if(reg_type == 'c')
                    {
                        d8 = (unsigned char)(d32 + i);
                        write_8(dev, addr+i, d8);
                    }
    
                    if( reg_type == 'f' )
                    {
                        /* Write Back the orig value */
    			if (big_endian == 0)
			{
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d32);
    			}
			else
			{
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d32);
    			}
                    }
    		}
    		break;

    	case 'h':
    	case 'H':
                if( reg_type == 'f' )
                {
                    printf("Utility does not support half word read/write operation for Framer");
                }
                else if( reg_type == 'c' )
                {
                    for (i = 0; i < len; i++)
                    {
                        d16 = (unsigned short)(d32 + i);

                        if (big_endian == 0)
                        {
                            write_le16(dev, addr+2*i, d16);
                        }
                        else
                        {
                            write_be16(dev, addr+2*i, d16);
                        }
                    }
                }
    		break;

    
    	case 'w':
    	case 'W':
                if( reg_type == 'f' )
                {
                    printf("Utility does not support word read/write operation for Framer");
                }
                else if( reg_type == 'c' )
                {
                    for (i = 0; i < len; i++)
                    {
                        if (big_endian == 0)
                        {
                            write_le32(dev, addr+4*i, d32 + i);
                        }
                        else
                        {
                            write_be32(dev, addr+4*i, d32 + i);
                        }
                    }
                }
    		break;
    
    	default:
    		printf("Syntax error (use ? for help)\n");
    		/* Don't break out of command processing loop */
    		break;
    }
    
    switch (verify) {
    	case 'v':
    	case 'V':
                if( reg_type == 'f' )
                {
                    if( (width == 'w') || (width == 'h') )
                    {
                        printf("Utility does not support half word/word read/write operation for Framer");
                        break;
                    }
       		    for (i = 0; i < len; i++)
		    {
    			if ((i%16) == 0)
			{
    				printf("\n%.8X: ", addr+i);
    			}
    
    			/* First Read AFT_MCPU_INTERFACE_ADDR */
    			if (big_endian == 0)
			{
    				tmp_d32 = read_le16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
			else
			{
    				tmp_d32 = read_be16(dev, AFT_MCPU_INTERFACE_ADDR);
    			}
    
                        /* Write CHIP Reg address that needs to be read */
                        tmp_addr = (unsigned short)(addr + i);
    			if (big_endian == 0)
			{
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_addr);
    			}
			else
			{
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_addr);
    			}
    
                        /* READ CHIP Reg address that needs to be read */
    			d8 = read_8(dev, AFT_MCPU_INTERFACE);
    			printf("%.2X ", d8);
    
                        /* Write Back the orig value */
    			if (big_endian == 0)
			{
    				write_le16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d32);
    			}
			else
			{
    				write_be16(dev, AFT_MCPU_INTERFACE_ADDR, tmp_d32);
    			}
    		    }
    		    printf("\n");
                }
                else if( reg_type == 'c' )
                {
                    if( width == 'b' )
                    {
                        for (i = 0; i < len; i++)
                        {
                            if ((i%16) == 0)
                            {
                                printf("\n%.8X: ", addr+i);
                            }
                            d8 = read_8(dev, addr+i);
                            printf("%.2X ", d8);
                        }
                        printf("\n");
                    }
                    else if( width == 'h' )
                    {
                        for (i = 0; i < len; i+=2)
                        {
                            if ((i%16) == 0)
                            {
                                printf("\n%.8X: ", addr+i);
                            }
                            if (big_endian == 0)
                            {
                                d16 = read_le16(dev, addr+i);
                            }
                            else
                            {
                                d16 = read_be16(dev, addr+i);
                            }
                            printf("%.4X ", d16);
                        }
                        printf("\n");
                    }
                    else if( width == 'w' )
                    {
                        for (i = 0; i < len; i+=4)
                        {
                            if ((i%16) == 0)
                            {
                                printf("\n%.8X: ", addr+i);
                            }
                            if (big_endian == 0)
                            {
                                d32 = read_le32(dev, addr+i);
                            }
                            else
                            {
                                d32 = read_be32(dev, addr+i);
                            }
                            printf("%.8X ", d32);
                        }
                        printf("\n");
                    }
                }
    		break;
    
    	default:
    		printf("Syntax error (use ? for help)\n");
    		/* Don't break out of command processing loop */
    		break;
    }
    
    return 0;
}

int change_endian(device_t *dev, char *cmd)
{
    char endian = 0;
    int status;
    
    /* e, el, eb */
    status = sscanf(cmd, "%*c%c", &endian);
    if (status < 0)
    {
            /* Display the current setting */
            if (big_endian == 0)
	    {
                    printf("Endian mode: little-endian\n");
            }
	    else
	    {
                    printf("Endian mode: big-endian\n");
            }

            return 0;
    }
    else if (status == 1)
    {
            switch (endian)
	    {
                    case 'b':
                            big_endian = 1;
                            break;
                    case 'l':
                            big_endian = 0;
                            break;
                    default:
                            printf("Syntax error (use ? for help)\n");
                            /* Don't break out of command processing loop */
                            break;
            }
    }
    else
    {
            printf("Syntax error (use ? for help)\n");
            /* Don't break out of command processing loop */
    }
    return 0;
}

/* ----------------------------------------------------------------
 * Raw pointer read/write access
 * ----------------------------------------------------------------
 */
static void write_8( device_t *dev, unsigned int addr, unsigned char  data)
{
    *(volatile unsigned char *)(dev->addr + addr) = data;
    msync((void *)(dev->addr + addr), 1, MS_SYNC | MS_INVALIDATE);
}

static unsigned char read_8( device_t *dev, unsigned int addr)
{
    return *(volatile unsigned char *)(dev->addr + addr);
}

static void write_le16( device_t *dev, unsigned int addr, unsigned short int data)
{
    if (__BYTE_ORDER != __LITTLE_ENDIAN)
    {
    	data = bswap_16(data);
    }

    *(volatile unsigned short int *)(dev->addr + addr) = data;
    msync((void *)(dev->addr + addr), 2, MS_SYNC | MS_INVALIDATE);
}

static unsigned short int read_le16( device_t *dev, unsigned int addr)
{
    unsigned int data = *(volatile unsigned short int *)(dev->addr + addr);
    if (__BYTE_ORDER != __LITTLE_ENDIAN)
    {
    	data = bswap_16(data);
    }

    return data;
}

static void write_be16( device_t *dev, unsigned int addr, unsigned short int data)
{
    if (__BYTE_ORDER == __LITTLE_ENDIAN)
    {
        data = bswap_16(data);
    }

    *(volatile unsigned short int *)(dev->addr + addr) = data;
    msync((void *)(dev->addr + addr), 2, MS_SYNC | MS_INVALIDATE);
}

static unsigned short int read_be16( device_t *dev, unsigned int addr)
{
    unsigned int data = *(volatile unsigned short int *)(dev->addr + addr);
    if (__BYTE_ORDER == __LITTLE_ENDIAN)
    {
    	data = bswap_16(data);
    }

    return data;
}

static void write_le32( device_t *dev, unsigned int addr, unsigned int data)
{
    if (__BYTE_ORDER != __LITTLE_ENDIAN)
    {
            data = bswap_32(data);
    }

    //printf("write_le32: data = 0x%x and for addr = 0x%x:0x%x\n", data, dev->addr, addr);
    *(volatile unsigned int *)(dev->addr + addr) = data;
    msync((void *)(dev->addr + addr), 4, MS_SYNC | MS_INVALIDATE);
}

static unsigned int read_le32( device_t *dev, unsigned int addr)
{
    unsigned int data = *(volatile unsigned int *)(dev->addr + addr);
    if (__BYTE_ORDER != __LITTLE_ENDIAN)
    {
    	data = bswap_32(data);
    }

    return data;
}

static void write_be32( device_t *dev, unsigned int addr, unsigned int data)
{
    if (__BYTE_ORDER == __LITTLE_ENDIAN)
    {
        data = bswap_32(data);
    }

    *(volatile unsigned int *)(dev->addr + addr) = data;
    msync((void *)(dev->addr + addr), 4, MS_SYNC | MS_INVALIDATE);
}

static unsigned int read_be32( device_t *dev, unsigned int addr)
{
    unsigned int data = *(volatile unsigned int *)(dev->addr + addr);
    if (__BYTE_ORDER == __LITTLE_ENDIAN)
    {
    	data = bswap_32(data);
    }

    return data;
}
