/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2007 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisISR.c - NDIS driver interrupt processing.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         09/29/05    Changed descriptor structure.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         06/02/06    Report statistics from MIB counters.
    THa         10/04/06    Report link state in interrupt routine.
    THa         12/18/07    Add hardware checksumming support.
   ---------------------------------------------------------------------------
*/


#include "target.h"
#include "NdisDevice.h"


#if 0
#define RCV_UNTIL_NONE
#endif


#ifdef DEBUG_OVERRUN
int receive_overrun = 0;
#endif

/* -------------------------------------------------------------------------- */

/*
    ProcessReceive

    Description:
        This function processes the buffer receiving.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (BOOLEAN):
        TRUE if a packet is received; FALSE otherwise.
*/


int ProcessReceive (
    IN  PNDIS_ADAPTER pAdapter )
{
    int           iNext;
    TDescStat     status;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    PTDescInfo    pInfo = &pHardware->m_RxDescInfo;
    PBUFFER_INFO  pBufInfo = &pAdapter->m_RxBufInfo;
    PNDIS_PACKET* PacketArray = pAdapter->m_PacketArray;
    PNDIS_PACKET  pPacket;
    PTDesc        pDesc;
    PDMA_BUFFER   pDma;

#ifndef RCV_UNTIL_NONE
    int           cnLeft = pInfo->cnAlloc;
#endif
    UINT          i;
    UINT          uiPacketCount = 0;

#ifndef NO_STATS
    PUCHAR        pBuffer;
#endif
    int           port = MAIN_PORT;

#ifdef DEBUG_RX_DATA
    int           nLength;
#endif

    iNext = pInfo->iNext;

#ifdef RCV_UNTIL_NONE
    while ( 1 )

#else
    while ( cnLeft-- )
#endif
    {
        /* Get next descriptor which is not hardware owned. */
        GetReceivedPacket( pInfo, iNext, pDesc, status.ulData );
        if ( status.rx.fHWOwned )
            break;

#ifdef CHECK_OVERRUN
        if ( !( pDesc->pCheck->Control.ulData & CPU_TO_LE32( DESC_HW_OWNED )) )
        {

#ifdef DEBUG_OVERRUN
            receive_overrun = 1;
            pHardware->m_ulDropped++;
#endif
            ReleasePacket( pDesc );
            FreeReceivedPacket( pInfo, iNext );

#ifndef RCV_UNTIL_NONE
            cnLeft = pInfo->cnAlloc;
#endif
            continue;
        }
#endif

        pDesc->sw.Control.ulData = status.ulData;
        pInfo->pCurrent = pDesc;

        HardwareReceive( pHardware );

        if ( pHardware->m_nPacketLen )
        {

#ifdef DEBUG_OVERRUN
            pHardware->m_ulReceived++;
#endif

            /* Get received packet information. */
            pDma = DMA_BUFFER( pDesc );

            /* Adjust the length of NDIS buffer to the length of the packet. */
            NdisAdjustBufferLength( pDma->pNdisBuffer,
                pHardware->m_nPacketLen );

            /* Flush the receive buffer. */
            NdisFlushBuffer( pDma->pNdisBuffer, FALSE );
            NdisMUpdateSharedMemory(
                pAdapter->m_hAdapter,
                pDma->ulSize,
                pDma->sm.pVirtual,
                pDma->PhysicalAddr );

#ifndef NO_STATS
            pBuffer = ( PUCHAR ) pDma->sm.pVirtual;

#ifdef DEBUG_RX_DATA
            for ( nLength = 0; nLength < pHardware->m_nPacketLen; nLength++ )
            {
                DbgPrint( "%02X ", pBuffer[ nLength ]);
                if ( ( nLength % 16 ) == 15 )
                {
                    DbgPrint( "\n" );
                }
            }
            DbgPrint( "\n" );
#endif

            /* Descriptor does not have broadcast indication. */
            if ( 0xFF == pBuffer[ 0 ] )
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_FRAMES_RCV ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_BYTES_RCV ] +=
                        pHardware->m_nPacketLen;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_MULTICAST_FRAMES_RCV ]--;
            }
            else if ( pBuffer[ 0 ] != 0x01 )
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_UNICAST_FRAMES_RCV ]++;

                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_FRAMES_RCV ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_BYTES_RCV ] +=
                        pHardware->m_nPacketLen;
            }
#endif
            pHardware->m_cnCounter[ port ][ OID_COUNTER_RCV_OK ]++;

            /* Get pointer of associated packet for further processing. */
            pPacket = pDma->pNdisPacket;

            if ( pHardware->m_dwReceiveConfig &
                    ( DMA_RX_CTRL_CSUM_IP | DMA_RX_CTRL_CSUM_TCP ) )
            {
                PNDIS_TCP_IP_CHECKSUM_PACKET_INFO pChecksum =
                    ( PNDIS_TCP_IP_CHECKSUM_PACKET_INFO ) &
                    NDIS_PER_PACKET_INFO_FROM_PACKET( pPacket,
                        TcpIpChecksumPacketInfo );

                if ( pChecksum )
                {
                    pChecksum->Value = 0;

#if defined( CHECK_RCV_ERRORS ) || defined( RCV_HUGE_FRAME )
                    if ( status.rx.fCsumErrIP )
                        pChecksum->Receive.NdisPacketIpChecksumFailed =
                            TRUE;
                    else
#endif
                        pChecksum->Receive.NdisPacketIpChecksumSucceeded =
                            TRUE;

#if defined( CHECK_RCV_ERRORS ) || defined( RCV_HUGE_FRAME )
                    if ( status.rx.fCsumErrTCP )
                        pChecksum->Receive.NdisPacketTcpChecksumFailed =
                            TRUE;
                    else
#endif
                        pChecksum->Receive.NdisPacketTcpChecksumSucceeded =
                            TRUE;
#if 0
#if defined( CHECK_RCV_ERRORS ) || defined( RCV_HUGE_FRAME )
                    if ( status.rx.fCsumErrUDP )
                        pChecksum->Receive.NdisPacketUdpChecksumFailed =
                            TRUE;
                    else
#endif
                        pChecksum->Receive.NdisPacketUdpChecksumSucceeded =
                            TRUE;
#endif
                }
            }

            /* There are buffers available for using in descriptors. */
#ifndef UNDER_CE
            if ( NdisQueryDepthSList( &pBufInfo->listHead ) )

#else
            if ( pBufInfo->cnAvail )
#endif
            {
                /* Upper layer can keep this packet. */
                NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_SUCCESS );

                /* Reset this pointer so that we can debug the RX descriptor if
                   error happens.
                */
                RESERVED( pPacket )->pDesc = NULL;

                /* Get a new buffer for this descriptor. */
                pDma = Alloc_DMA_Buffer( pBufInfo, pDesc );

                /* Set descriptor for packet receive. */
                SetReceiveBuffer( pDesc, pDma->sm.ulPhysical );
                SetReceiveLength( pDesc, pDma->ulSize );
                ReleasePacket( pDesc );
            }
            else
            {
                /* Upper layer must copy this packet immediately. */
                NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_RESOURCES );
                RESERVED( pPacket )->pDesc = pDesc;

                /* Cannot release the descriptor yet. */
            }

            PacketArray[ uiPacketCount ] = pPacket;
            uiPacketCount++;
        }
        else
        {
            ReleasePacket( pDesc );
        }

        FreeReceivedPacket( pInfo, iNext );
    }
    pInfo->iNext = iNext;

    if ( uiPacketCount )
    {
        /* We have packets to indicate. */
        NdisMIndicateReceivePacket( pAdapter->m_hAdapter, PacketArray,
            uiPacketCount );

        /* Determine which packets are kept for reuse and which ones are needed
           to put back to the list.
        */
        for ( i = 0; i < uiPacketCount; i++ )
        {
            pPacket = PacketArray[ i ];
            if ( NDIS_STATUS_PENDING != NDIS_GET_PACKET_STATUS( pPacket ) )
            {
                /* If the status code for the packet is not status pending,
                   which should be modified by NdisMIndicateReceivePacket
                   function, then we can place the resouces back on the free
                   list.
                */
                pDma = RESERVED( pPacket )->pDma;

#if 0
                /* Adjust the buffer length. */
                NdisAdjustBufferLength( pDma->pNdisBuffer, pDma->ulSize );
#endif

                /* If we indicate to the upper that it could not keep this
                   packet then do not place it back to the list.
                */
                if ( NDIS_STATUS_RESOURCES ==
                        NDIS_GET_PACKET_STATUS( pPacket ) )
                {
                    pDesc = RESERVED( pPacket )->pDesc;

                    /* It is set by us so release the descriptor. */
                    if ( pDesc )
                    {
                        RESERVED( pPacket )->pDesc = NULL;
                        ReleasePacket( pDesc );
                    }

                    /* It should not happen unless upper layer really has no
                       resources.
                    */
                    else
                    {
                        /* Place the buffer back to the free list for use next
                           time.
                        */
                        Free_DMA_Buffer( pBufInfo, pDma );
                    }
                }
                else
                {
                    /* Place the buffer back to the free list for use next
                       time.
                    */
                    Free_DMA_Buffer( pBufInfo, pDma );
                }
            }
        }
    }
    return( uiPacketCount != 0 );
}  // ProcessReceive


/*
    MiniportHandleInterrupt

    Description:
        This routine is used by NDIS for deferred interrupt processing.  It
        reads from the Interrupt Status Register any outstanding interrupts and
        handles them.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

VOID MiniportHandleInterrupt (
    IN  NDIS_HANDLE hAdapterContext )
{
    UINT          IntEnable;
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    int           port = MAIN_PORT;

#ifdef DEBUG_DRIVER_INTERRUPT
DbgPrint( "MiniportHandleInterrupt"NEWLINE );
#endif

    do
    {
        // Get the interrupt bits and save them.
        // Mask only interesting bits.
        HardwareReadInterrupt( pHardware, &IntEnable );
        if ( !IntEnable )
        {
            break;
        }

#ifdef DEBUG_INTERRUPT
DbgPrint( " intr:%08x"NEWLINE, IntEnable );
#endif
        HardwareAcknowledgeInterrupt( pHardware, IntEnable );

        /* All packets have been sent. */
        if ( ( IntEnable & INT_TX_EMPTY ) )
        {
            /* Also acknowledge transmit complete interrupt. */
            HardwareAcknowledgeTransmit( pHardware );
            TransmitDone( pAdapter );

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_TX ]++;
#endif
        }

        /* Do not need to process transmit complete as all transmitted
           descriptors are freed.
        */
        else
        if ( ( IntEnable & INT_TX ) )
        {
            TransmitDone( pAdapter );

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_TX ]++;
#endif
        }

        if ( ( IntEnable & INT_RX ) )
        {

            // Receive the packet
            ProcessReceive( pAdapter );

#ifdef RCV_UNTIL_NONE
            // Acknowledge the interrupt
            HardwareAcknowledgeReceive( pHardware );
#endif

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_RX ]++;
#endif
        }

        if ( ( IntEnable & INT_PHY ) )
        {
            pHardware->m_bLinkIntWorking = TRUE;
            SwitchGetLinkStatus( pHardware );
            if ( pAdapter->m_ulNdisMediaState !=
                    pHardware->m_ulHardwareState )
            {
                NDIS_STATUS nsStatus;

                nsStatus = ( NdisMediaStateConnected ==
                    pHardware->m_ulHardwareState ) ?
                    NDIS_STATUS_MEDIA_CONNECT : NDIS_STATUS_MEDIA_DISCONNECT;

                NdisMIndicateStatus( pAdapter->m_hAdapter, nsStatus, NULL, 0 );
                NdisMIndicateStatusComplete( pAdapter->m_hAdapter );
                pAdapter->m_ulNdisMediaState = pHardware->m_ulHardwareState;
            }
            if ( ( pAdapter->m_ulDriverState & DRIVER_STATE_RESET )  &&
                    NdisMediaStateConnected == pAdapter->m_ulNdisMediaState )
            {
                pAdapter->m_ulDriverState &= ~DRIVER_STATE_RESET;
                NdisMResetComplete( pAdapter->m_hAdapter, NDIS_STATUS_SUCCESS,
                    FALSE );
            }
        }

        if ( ( IntEnable & INT_RX_OVERRUN ) )
        {

#ifdef DBG
            DBG_PRINT( "Rx overrun"NEWLINE );
#endif
            pHardware->m_cnCounter[ port ][ OID_COUNTER_RCV_NO_BUFFER ]++;

            // Acknowledge the interrupt
            HardwareAcknowledgeOverrun( pHardware );
            HardwareResumeReceive( pHardware );

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_RX_OVERRUN ]++;
#endif
        }

        if ( ( IntEnable & INT_RX_STOPPED ) )
        {

#ifdef DBG
            DBG_PRINT( "Rx stopped"NEWLINE );
#endif

            /* Receive just has been stopped. */
            if ( 0 == pHardware->m_bReceiveStop ) {
                pHardware->m_ulInterruptMask &= ~INT_RX_STOPPED;
            }
            else if ( pHardware->m_bReceiveStop > 1 ) {
                if ( pHardware->m_bEnabled  &&
                        ( pHardware->m_dwReceiveConfig &
                        DMA_RX_CTRL_ENABLE ) ) {

#ifdef DBG
                    DBG_PRINT( "Rx disabled"NEWLINE );
#endif
                    HardwareStartReceive( pHardware );
                }
                else {

#ifdef DBG
                    DBG_PRINT( "Rx stop disabled:%d"NEWLINE,
                        pHardware->m_bReceiveStop );
#endif
                    pHardware->m_ulInterruptMask &= ~INT_RX_STOPPED;
                    pHardware->m_bReceiveStop = 0;
                }
            }

            /* Receive just has been started. */
            else {
                pHardware->m_bReceiveStop++;
            }
            HardwareAcknowledgeInterrupt( pHardware, INT_RX_STOPPED );
            break;
        }

#ifdef DBG
        if ( ( IntEnable & INT_TX_STOPPED ) )
        {
            UINT dwData;

            pHardware->m_ulInterruptMask &= ~INT_TX_STOPPED;
            DBG_PRINT( "Tx stopped"NEWLINE );
            HW_READ_DWORD( pHardware, REG_DMA_TX_CTRL, &dwData );
            if ( !( dwData & DMA_TX_CTRL_ENABLE ) ) {
                DBG_PRINT( "Tx disabled"NEWLINE );
            }
            HardwareAcknowledgeInterrupt( pHardware, INT_TX_STOPPED );
            break;
        }
#endif
    } while ( 1 );
}  // MiniportHandleInterrupt


/*
    MiniportISR

    Description:
        This routine is used by NDIS for the hardware to check if the interrupt
        is its own in the shared interrupt environment.

    Parameters:
        PBOOLEAN pbInterruptRecognized
            Buffer to indicate the interrupt is recognized

        PBOOLEAN pbQueueDpc
            Buffer to indicate the deferred interrupt processing should be
            scheduled.

        PVOID pContext
            Pointer to adapter information structure.

    Return (None):
*/

VOID MiniportISR (
    OUT PBOOLEAN pbInterruptRecognized,
    OUT PBOOLEAN pbQueueDpc,
    IN  PVOID    pContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    UINT          ulIntStatus = 0;

    HardwareReadInterrupt( pHardware, &ulIntStatus );

    /* Hardware interrupts are disabled. */
    ulIntStatus &= pHardware->m_ulInterruptSet;

    /* Got interrupt and not already queued for interrupt handling. */
    if ( ulIntStatus )
    {
        HardwareDisableInterrupt( pHardware );

        *pbInterruptRecognized = TRUE;
        *pbQueueDpc = TRUE;
    }
    else
    {
        // not our interrupt!
        *pbInterruptRecognized = FALSE;
        *pbQueueDpc = FALSE;
    }
}  // MiniportISR

/* -------------------------------------------------------------------------- */

/*
    MiniportReturnPacket

    Description:
        This routine attempts to return to the receive free list the packet
        passed to us by NDIS.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

        PNDIS_PACKET pPacket
            Pointer to NDIS packet.

    Return (None):
*/

VOID MiniportReturnPacket (
    IN  NDIS_HANDLE  hAdapterContext,
        PNDIS_PACKET pPacket )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PDMA_BUFFER   pDma;

#ifdef DBG_
    DBG_PRINT( "MiniportReturnPacket"NEWLINE );
#endif

#if 0
    NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_SUCCESS );
#endif

    /* Get the pointer to receive buffer. */
    pDma = RESERVED( pPacket )->pDma;

#if 0
    /* Adjust the buffer length. */
    NdisAdjustBufferLength( pDma->pNdisBuffer, pDma->ulSize );
#endif

    Free_DMA_Buffer( &pAdapter->m_RxBufInfo, pDma );
}  // MiniportReturnPacket
