/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2008 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisDevice.c - NDIS driver device functions.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         08/16/05    Added PCI configuration I/O.
    THa         09/29/05    Changed descriptor structure.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         04/13/06    Add EEPROM access support.
    THa         06/02/06    Report statistics from MIB counters.
    THa         07/27/06    Version 1.4 supports NDIS 5.
    THa         09/22/06    Version 1.6 fixes big UDP packet transmit problem.
    THa         10/04/06    Version 1.6 delays reporting driver reset status.
    THa         12/21/06    Check LED setting for hardware hang if
                            SET_DEFAULT_LED is set to a non-default setting.
    THa         12/18/07    Update for 64-bit systems.
    THa         01/22/08    Update for 64-bit Windows.
    THa         02/15/08    Add force link support.
    THa         12/15/08    Fix getting SpeedDuplex parameter problem.
    DAVIDCAI    12-18-2009  Change TDeviceInfo structure, so new Micrel LinkMD driver can pass
    												port index through the new TDeviceInfo structure, and get link status
    												for each port. 
   ---------------------------------------------------------------------------
*/


#include "target.h"
#include "NdisDevice.h"


PHARDWARE phw;

/* -------------------------------------------------------------------------- */

/*
    GetConfigEntryUlong

    Description:
        This function retrieves a double-word value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PUINT pulValue
            Buffer to store the value.

    Return (BOOLEAN):
        TRUE if successful; otherwise FALSE.
*/

BOOLEAN GetConfigEntryUlong (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PUINT        pulValue )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    NDIS_PARAMETER_TYPE           parameter = NdisParameterHexInteger;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        parameter );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        *pulValue = pReturnedValue->ParameterData.IntegerData;
        return TRUE;
    }
    return FALSE;
}  // GetConfigEntryUlong


/*
    GetConfigEntryUshort

    Description:
        This function retrieves a word value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PUSHORT pusValue
            Buffer to store the value.

    Return (BOOLEAN):
        TRUE if successful; otherwise FALSE.
*/

BOOLEAN GetConfigEntryUshort (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PUSHORT      pusValue )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    NDIS_PARAMETER_TYPE           parameter = NdisParameterHexInteger;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        parameter );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        *pusValue = ( USHORT ) pReturnedValue->ParameterData.IntegerData;
        return TRUE;
    }
    return FALSE;
}  // GetConfigEntryUshort


/*
    GetConfigEntryUchar

    Description:
        This function retrieves a byte value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PUCHAR pbValue
            Buffer to store the value.

    Return (BOOLEAN):
        TRUE if successful; otherwise FALSE.
*/

BOOLEAN GetConfigEntryUchar (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PUCHAR       pbValue )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    NDIS_PARAMETER_TYPE           parameter = NdisParameterInteger;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        parameter );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        *pbValue = ( UCHAR ) pReturnedValue->ParameterData.IntegerData;
        return TRUE;
    }
    return FALSE;
}  // GetConfigEntryUchar


/*
    GetConfigEntryString

    Description:
        This function retrieves a string value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PCHAR szValue
            Buffer to store the string value.

        USHORT wLength
            The length of the buffer.

    Return (int):
        The length of the string.
*/

int GetConfigEntryString (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PCHAR        szValue,
    USHORT       wLength )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    ANSI_STRING                   anszString;

    anszString.MaximumLength = wLength;
    anszString.Buffer = szValue;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        NdisParameterString );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        NdisUnicodeStringToAnsiString( &anszString,
            &pReturnedValue->ParameterData.StringData );
        return( anszString.Length );
    }
    return( 0 );
}  // GetConfigEntryString

/* -------------------------------------------------------------------------- */

#define nszcBusType                 NDIS_STRING_CONST( "BusType" )
#define nszcIoBaseAddress           NDIS_STRING_CONST( "IoBaseAddress" )
#define nszcInterruptNumber         NDIS_STRING_CONST( "InterruptNumber" )
#define nszcIoBaseLength            NDIS_STRING_CONST( "IoLen" )
#define nszcMemoryAddress           NDIS_STRING_CONST( "MemoryBaseAddress" )
#define nszcMemorySize              NDIS_STRING_CONST( "MemorySize" )
#define nszcSlotNumber              NDIS_STRING_CONST( "SlotNumber" )

#define nszcChecksumRx              NDIS_STRING_CONST( "ChecksumRx" )
#define nszcChecksumTx              NDIS_STRING_CONST( "ChecksumTx" )
#define nszcNetworkAddress          NDIS_STRING_CONST( "NetworkAddress" )
#define nszcSpeedDuplex             NDIS_STRING_CONST( "SpeedDuplex" )

#define DEFAULT_IO_BASE_ADDRESS     0x300
#define DEFAULT_IO_BASE_LENGTH      0x10
#define DEFAULT_MEMORY_SIZE         0x1000
#define DEFAULT_MEMORY_WINDOW       0xd4000
#define DEFAULT_INTERRUPT_NUMBER    3
#define MAX_IOBASEADDR              0xFFFFFFFF
#define MIN_IOBASEADDR              0x0100
#define MAX_IRQ                     15
#define MIN_IRQ                     2


/*
    GetResources

    Description:
        This function gets adapter resources from the operating system and the
        registry.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        NDIS_HANDLE hConfiguration
            Handle to adapter configuration.

        NDIS_HANDLE hRegistry
            Handle to registry key.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS GetResources (
        PNDIS_ADAPTER pAdapter,
    IN  NDIS_HANDLE   hConfiguration,
    IN  NDIS_HANDLE   hRegistry )
{
    UCHAR               bBuffer[ sizeof( NDIS_RESOURCE_LIST ) +
        ( sizeof( CM_PARTIAL_RESOURCE_DESCRIPTOR ) * 10 )];
    PNDIS_RESOURCE_LIST pResource = ( PNDIS_RESOURCE_LIST ) bBuffer;
    UINT                uiBufferSize = sizeof( bBuffer );
    NDIS_STATUS         nsStatus;
    UINT                ulIndex;
    int                 NextIo = 0;
    PHARDWARE           pHardware = &pAdapter->m_Hardware;
    NDIS_STRING         nszSlotNumber = nszcSlotNumber;

    NdisMQueryAdapterResources( &nsStatus, hConfiguration, pResource,
        &uiBufferSize );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {

#if 0
        // Assume Windows 9X
        pAdapter->ucOSEnvironment = WIN9X;
        // if NdisMQueryAdapterResources returns success check the OS environment
        NdisReadConfiguration ( &nsStatus,
                                &pncpReturnValue,
                                hRegistryHandle,
                                &nsEnvironment,
                                NdisParameterInteger);

        if (nsStatus == NDIS_STATUS_SUCCESS)
            // NT = NdisEnvironmentWindowsNt, 95 = NdisEnvironmentWindows.
            if (NdisEnvironmentWindowsNt == (UINT) pncpReturnValue->ParameterData.IntegerData)
                // must be Win2k
                pAdapter->ucOSEnvironment = WIN2K;
#endif

    }
    else
    {
#if 0
        // if NdisMQueryAdapterResources returns fail the OS is WinNT
        pAdapter->ucOSEnvironment = WINNT40;
#endif
    }

    pAdapter->m_ulSlot = 0;
    GetConfigEntryUlong( hRegistry, &nszSlotNumber,
        &pAdapter->m_ulSlot );

#if DBG
DbgPrint( "  slot: %d"NEWLINE, pAdapter->m_ulSlot );
#endif
    NdisReadPciSlotInformation( pAdapter->m_hAdapter, pAdapter->m_ulSlot, CFIT,
        &pAdapter->m_ulInterruptVector, sizeof( UCHAR ));

#if DBG
DbgPrint( "  vect: %d"NEWLINE, pAdapter->m_ulInterruptVector );
#endif
    if ( pResource )
    {
        // Parse the NDIS_RESOURCE_LIST (CM_PARTIAL_RESOURCE_LIST/
        // CM_PARTIAL_RESOURCE_DESCRIPTOR).
        for ( ulIndex = 0; ulIndex < pResource->Count; ulIndex++ )
        {
            // We handle 3 types of resources: I/O port, interrupt, and
            // memory base.  Ignore all others.
            switch ( pResource->PartialDescriptors[ ulIndex ].Type )
            {
                case CmResourceTypePort:
                        pAdapter->m_ulIOBaseAddress = ( UINT )
                            pResource->PartialDescriptors[ ulIndex ].u.Port.
                            Start.u.LowPart;
                        pAdapter->m_ulIOLength =
                            pResource->PartialDescriptors[ ulIndex ].u.Port.
                            Length;

#ifdef DBG
DbgPrint( "  io: %x, %x"NEWLINE, pAdapter->m_ulIOBaseAddress,
    pAdapter->m_ulIOLength );
#endif
                    break;

                case CmResourceTypeInterrupt:
                    pAdapter->m_ulInterruptNumber =
                        pResource->PartialDescriptors[ ulIndex ].u.Interrupt.
                        Vector;

#ifdef DBG
DbgPrint( "  intr: %d"NEWLINE, pAdapter->m_ulInterruptNumber );
#endif
                    break;

                case CmResourceTypeMemory:
#ifdef _WIN64
                    pAdapter->m_ulMemoryAddress =
                        pResource->PartialDescriptors[ ulIndex ].u.Memory.Start.QuadPart;
#else
                    pAdapter->m_ulMemoryAddress = ( ULONG )
                        pResource->PartialDescriptors[ ulIndex ].u.Memory.
                        Start.u.LowPart;
#endif
                    pAdapter->m_ulMemorySize =
                        pResource->PartialDescriptors[ ulIndex ].u.Memory.
                        Length;

#ifdef DBG
DbgPrint( "  mem: %lx, %x"NEWLINE, pAdapter->m_ulMemoryAddress,
    pAdapter->m_ulMemorySize );
#endif
                    break;
            }
        }
    }
    else
    {
        NDIS_STRING nszIoBaseAddress = nszcIoBaseAddress;
        NDIS_STRING nszIoBaseLength = nszcIoBaseLength;
        NDIS_STRING nszInterruptNumber = nszcInterruptNumber;
        NDIS_STRING nszMemoryAddress = nszcMemoryAddress;
        NDIS_STRING nszMemorySize = nszcMemorySize;
        UINT        uiAddr;

        pAdapter->m_ulIOBaseAddress = DEFAULT_IO_BASE_ADDRESS;
        GetConfigEntryUlong( hRegistry, &nszIoBaseAddress,
            &pAdapter->m_ulIOBaseAddress );
        if ( pAdapter->m_ulIOBaseAddress < MIN_IOBASEADDR  ||
                pAdapter->m_ulIOBaseAddress > MAX_IOBASEADDR )
        {
            NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
                NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1,
                pAdapter->m_ulIOBaseAddress );
            return( NDIS_STATUS_FAILURE );
        }

        pAdapter->m_ulIOLength = DEFAULT_IO_BASE_LENGTH;
        GetConfigEntryUlong( hRegistry, &nszIoBaseLength,
            &pAdapter->m_ulIOLength );

        pAdapter->m_ulInterruptNumber = DEFAULT_INTERRUPT_NUMBER;
        GetConfigEntryUlong( hRegistry, &nszInterruptNumber,
            &pAdapter->m_ulInterruptNumber );
        if ( pAdapter->m_ulInterruptNumber < MIN_IRQ  ||
                pAdapter->m_ulInterruptNumber > MAX_IRQ )
        {
            NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
                NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1,
                pAdapter->m_ulInterruptNumber );
            return( NDIS_STATUS_FAILURE );
        }

        pAdapter->m_ulMemoryAddress = DEFAULT_MEMORY_WINDOW;
        GetConfigEntryUlong( hRegistry, &nszMemoryAddress, &uiAddr );
        pAdapter->m_ulMemoryAddress = uiAddr;

        pAdapter->m_ulMemorySize = DEFAULT_MEMORY_SIZE;
        GetConfigEntryUlong( hRegistry, &nszMemorySize,
            &pAdapter->m_ulMemorySize );
    }
    return NDIS_STATUS_SUCCESS;
}  // GetResources


/*
    ReadConfiguration

    Description:
        This function gets configuration for the adapter from the operating
        system and the registry.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        NDIS_HANDLE hConfiguration
            Handle to adapter configuration.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS ReadConfiguration (
        PNDIS_ADAPTER pAdapter,
    IN  NDIS_HANDLE   hConfiguration )
{
    // Handle for reading from registry.
    NDIS_HANDLE hRegistry;
    NDIS_STATUS nsStatus;
    BOOLEAN     bStatus;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    NDIS_STRING nszBusType = nszcBusType;
    NDIS_STRING nszAddress = nszcNetworkAddress;
    NDIS_STRING nszChecksumRx = nszcChecksumRx;
    NDIS_STRING nszChecksumTx = nszcChecksumTx;
    NDIS_STRING nszSpeed = nszcSpeedDuplex;
    CHAR        szAddress[ 16 ];
    int         len;
    ULONG       ulData;
    USHORT      wData;

    // Open the configuration space.
    NdisOpenConfiguration( &nsStatus, &hRegistry, hConfiguration );
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {
        NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
            NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 0 );
        return( nsStatus );
    }

    // get bus type
    pAdapter->m_BusType = NdisInterfacePci;
    bStatus = GetConfigEntryUlong( hRegistry, &nszBusType,
        ( PUINT ) &pAdapter->m_BusType );

#ifdef DBG
    if ( !bStatus )
        DbgPrint( " def" );
DbgPrint( "  bustype: %x"NEWLINE, pAdapter->m_BusType );
#endif

    bStatus = GetConfigEntryUshort( hRegistry, &nszChecksumRx, &wData );
    if ( bStatus )
    {
        if ( wData )
        {
            pAdapter->m_ulRxChecksum = TASK_OFFLOAD_CHECKSUM;
        }
        else
            pAdapter->m_ulRxChecksum = 0;
    }
    bStatus = GetConfigEntryUshort( hRegistry, &nszChecksumTx, &wData );
    if ( bStatus )
    {
        if ( wData )
        {
            pAdapter->m_ulTxChecksum = TASK_OFFLOAD_CHECKSUM;
        }
        else
            pAdapter->m_ulTxChecksum = 0;
    }
#if DBG
    DBG_PRINT( "Checksum = %u %u"NEWLINE,
        pAdapter->m_ulRxChecksum, pAdapter->m_ulTxChecksum );
#endif

    len = GetConfigEntryString( hRegistry, &nszAddress, szAddress, 16 );
    if ( 12 == len )
    {
        UCHAR bData;
        int   index = 5;

#if DBG
        DBG_PRINT( "MAC: %s"NEWLINE, szAddress );
#endif
        while ( len-- > 0 )
        {
            if ( '0' <= szAddress[ len ]  &&  szAddress[ len ] <= '9' )
            {
                bData = szAddress[ len ] - '0';
                if ( ( len & 1 ) )
                    pHardware->m_bOverrideAddress[ index ] = bData;
                else
                    pHardware->m_bOverrideAddress[ index-- ] |= ( bData << 4 );
            }
            else if ( 'A' <= szAddress[ len ]  &&  szAddress[ len ] <= 'F' )
            {
                bData = szAddress[ len ] - 'A' + 10;
                if ( ( len & 1 ) )
                    pHardware->m_bOverrideAddress[ index ] = bData;
                else
                    pHardware->m_bOverrideAddress[ index-- ] |= ( bData << 4 );
            }
            else
                break;
        }
        if ( len < 0 )
        {
#if DBG
            PrintMacAddress( pHardware->m_bOverrideAddress );
            DBG_PRINT( NEWLINE );
#endif
            pHardware->m_bMacOverrideAddr = TRUE;
        }
    }

    bStatus = GetConfigEntryUlong( hRegistry, &nszSpeed, &ulData );
    if ( bStatus )
    {
        if ( ulData )
        {
            pHardware->m_bDuplex = ( UCHAR )( ulData & 3 );
            if ( ulData >= 0x10000 )
            {
                pHardware->m_bForceLink = 1;
                ulData -= 0x10000;
            }
            if ( ulData >= 0x100 )
            {
                pHardware->m_bSpeed = 100;
            }
            else if ( ulData >= 0x10 )
            {
                pHardware->m_bSpeed = 10;
            }
        }
#if DBG
        DBG_PRINT( "SpeedDuplex = %x; %u %u %u"NEWLINE, ulData,
            pHardware->m_bSpeed, pHardware->m_bDuplex,
			pHardware->m_bForceLink );
#endif
    }

    pHardware->m_RxDescInfo.cnAlloc = NUM_OF_RX_DESC;
    pHardware->m_TxDescInfo.cnAlloc = NUM_OF_TX_DESC;
    pHardware->m_pPciCfg = pAdapter;
    pHardware->m_pDevice = pAdapter;

    if ( ( nsStatus = GetResources( pAdapter, hConfiguration, hRegistry )) !=
            NDIS_STATUS_SUCCESS )
    {
        goto ReadConfigurationDone;
    }

ReadConfigurationDone:
    NdisCloseConfiguration( hRegistry );

    return( nsStatus );
}  // ReadConfiguration


/*
    DeRegisterAdapter

    Description:
        This routine de-registers the adapter from the operating system.  It
        releases the I/O address range used by the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID DeRegisterAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

#ifdef DBG
DbgPrint( "DeRegister"NEWLINE );
#endif

    if ( pHardware->m_pVirtualMemory )
    {
#ifdef UNDER_CE
        VirtualFree( pHardware->m_pVirtualMemory, 0, MEM_RELEASE );
        pHardware->m_ulVIoAddr = 0;

#else
        NdisMUnmapIoSpace( pAdapter->m_hAdapter,
            pHardware->m_pVirtualMemory,
            pAdapter->m_ulMemorySize );
#endif
        pHardware->m_pVirtualMemory = NULL;
    }
    if ( pHardware->m_ulVIoAddr )
    {
        NdisMDeregisterIoPortRange( pAdapter->m_hAdapter,
            pAdapter->m_ulIOBaseAddress, pAdapter->m_ulIOLength,
            ( PVOID ) pHardware->m_ulVIoAddr );
        pHardware->m_ulVIoAddr = 0;
    }
}  // DeRegisterAdapter


/*
    RegisterAdapter

    Description:
        This function registers the adapter to the NDIS wrapper.  It acquires
        the I/O address range to run the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS RegisterAdapter (
    PNDIS_ADAPTER pAdapter )
{
    NDIS_PHYSICAL_ADDRESS PhysicalAddress;
    NDIS_STATUS           nsStatus = NDIS_STATUS_SUCCESS;
    PHARDWARE             pHardware = &pAdapter->m_Hardware;

#ifdef UNDER_CE
    BOOL                  bResult;
#endif

#ifdef DBG
DbgPrint( "Register"NEWLINE );
#endif

    if ( pAdapter->m_ulMemorySize )
    {
#ifdef _WIN64
        PhysicalAddress.QuadPart = pAdapter->m_ulMemoryAddress;
#else
        NdisSetPhysicalAddressLow( PhysicalAddress,
            pAdapter->m_ulMemoryAddress );
        NdisSetPhysicalAddressHigh( PhysicalAddress, 0 );
#endif
        nsStatus = NdisMMapIoSpace(( PVOID* ) &pHardware->m_pVirtualMemory,
            pAdapter->m_hAdapter, PhysicalAddress,
            pAdapter->m_ulMemorySize );
        if ( nsStatus != NDIS_STATUS_SUCCESS )
        {
            goto RegisterAdapterError;
        }
    }
    if ( pAdapter->m_ulIOLength )
    {
#ifdef UNDER_CE
        pHardware->m_pVirtualMemory = VirtualAlloc( 0, PAGE_SIZE, MEM_RESERVE,
            PAGE_NOACCESS );
        DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: VirtualAlloc at 0x%x.\r\n" ),
            pHardware->m_pVirtualMemory ));
        if ( !pHardware->m_pVirtualMemory )
            goto RegisterAdapterError;
        bResult = VirtualCopy( pHardware->m_pVirtualMemory,
            ( PVOID )( pAdapter->m_ulIOBaseAddress & ( ~( PAGE_SIZE - 1 ))),
            PAGE_SIZE, PAGE_READWRITE | PAGE_NOCACHE );
        pHardware->m_ulVIoAddr = ( ULONG ) pHardware->m_pVirtualMemory +
            ( pAdapter->m_ulIOBaseAddress & ( PAGE_SIZE - 1 ));
        DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: VirtualCopy at 0x%x.\r\n" ),
            pHardware->m_ulVIoAddr ));

#else
        nsStatus = NdisMRegisterIoPortRange(( PVOID* ) &pHardware->m_ulVIoAddr,
            pAdapter->m_hAdapter, pAdapter->m_ulIOBaseAddress,
            pAdapter->m_ulIOLength );
#endif
        if ( nsStatus != NDIS_STATUS_SUCCESS )
        {
#ifndef UNDER_CE
            NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
                NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 3, 0x10560010, 0x1,
                pAdapter->m_ulIOBaseAddress );
#endif
            goto RegisterAdapterError;
        }
    }

    return( nsStatus );

RegisterAdapterError:
    DeRegisterAdapter( pAdapter );
    return( nsStatus );
}  // RegisterAdapter

/* -------------------------------------------------------------------------- */

/*
    InitMapRegisters

    Description:
        This local routine initializes the map register information structure.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

static void InitMapRegisters (
        PNDIS_ADAPTER pAdapter )
{
    int i;

    for ( i = 0; i < pAdapter->m_cnMapRegAlloc; i++ )
    {
        pAdapter->m_MapReg[ i ].uiRegister = i;
        pAdapter->m_MapReg[ i ].pNdisBuffer = NULL;
        pAdapter->m_MapReg[ i ].pNext = &pAdapter->m_MapReg[ i + 1 ];
    }
    pAdapter->m_MapReg[ i - 1 ].pNext = NULL;

    pAdapter->m_pMapRegTop = pAdapter->m_MapReg;
    pAdapter->m_cnMapRegAvail = pAdapter->m_cnMapRegAlloc;
}  // InitMapRegisters


/*
    AllocateMapRegisters

    Description:
        This local function allocates the mapping registers for use in
        transmission.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

static NDIS_STATUS AllocateMapRegisters (
        PNDIS_ADAPTER pAdapter )
{
    PHARDWARE             pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS           NdisStatus;
    NDIS_PHYSICAL_ADDRESS HighestAcceptableMax =
        NDIS_PHYSICAL_ADDRESS_CONST( -1, -1 );

    pAdapter->m_cnMapRegAlloc = pHardware->m_TxDescInfo.cnAlloc * 3 / 8;

    /* Need to call NdisMAllocateMapRegisters before NdisMAllocateSharedMemory
       is used.
    */
    NdisStatus = NdisMAllocateMapRegisters(
        pAdapter->m_hAdapter,
        0,
        TRUE,
        pAdapter->m_cnMapRegAlloc,
        NDIS_BUFFER_SIZE );

    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        pAdapter->m_cnMapRegAlloc = MIN_MAP_REGISTERS;
        if ( NdisMAllocateMapRegisters(
                pAdapter->m_hAdapter,
                0,
                TRUE,
                pAdapter->m_cnMapRegAlloc,
                NDIS_BUFFER_SIZE ) != NDIS_STATUS_SUCCESS )
        {
            pAdapter->m_cnMapRegAlloc = 0;
            return NdisStatus;
        }
    }

#if (NDISVER >= 50)
    NdisStatus = NdisAllocateMemoryWithTag(( PVOID* ) &pAdapter->m_MapReg,
        sizeof( MAP_REG ) * pAdapter->m_cnMapRegAlloc, 'rciM' );
#else
    NdisStatus = NdisAllocateMemory(( PVOID* ) &pAdapter->m_MapReg,
        sizeof( MAP_REG ) * pAdapter->m_cnMapRegAlloc,
        0, HighestAcceptableMax );
#endif
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return( NdisStatus );
    }

    InitMapRegisters( pAdapter );

    pAdapter->m_cnPackets = pHardware->m_RxDescInfo.cnAlloc;

#if (NDISVER >= 50)
    NdisStatus = NdisAllocateMemoryWithTag(( PVOID* ) &pAdapter->m_PacketArray,
        sizeof( PNDIS_PACKET ) * pAdapter->m_cnPackets, 'rciM' );
#else
    NdisStatus = NdisAllocateMemory(( PVOID* ) &pAdapter->m_PacketArray,
        sizeof( PNDIS_PACKET ) * pAdapter->m_cnPackets,
        0, HighestAcceptableMax );
#endif
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        pAdapter->m_cnPackets = 0;
        return( NdisStatus );
    }

    return NDIS_STATUS_SUCCESS;
}  /* AllocateMapRegisters */


/*
    AllocateSoftDescriptors

    Description:
        This local function allocates software descriptors for manipulation in
        memory.

    Parameters:
        PTDescInfo pDescInfo
            Pointer to descriptor information structure.

        int fTransmit
            Indication that descriptors are for transmit.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

static NDIS_STATUS AllocateSoftDescriptors (
        PTDescInfo pDescInfo,
        int        fTransmit )
{
    NDIS_STATUS           NdisStatus;
    NDIS_PHYSICAL_ADDRESS HighestAcceptableMax =
        NDIS_PHYSICAL_ADDRESS_CONST( -1, -1 );

#if (NDISVER >= 50)
    NdisStatus = NdisAllocateMemoryWithTag(( PVOID* ) &pDescInfo->pRing,
        sizeof( TDesc ) * pDescInfo->cnAlloc, 'rciM' );
#else
    NdisStatus = NdisAllocateMemory(( PVOID* ) &pDescInfo->pRing,
        sizeof( TDesc ) * pDescInfo->cnAlloc,
        0, HighestAcceptableMax );
#endif
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return( NdisStatus );
    }
    NdisZeroMemory( pDescInfo->pRing, sizeof( TDesc ) * pDescInfo->cnAlloc );
    HardwareInitDescriptors( pDescInfo, fTransmit );

    return NDIS_STATUS_SUCCESS;
}  /* AllocateSoftDescriptors */


/*
    AllocateDescriptors

    Description:
        This local function allocates hardware descriptors for receiving and
        transmitting.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

static NDIS_STATUS AllocateDescriptors (
        PNDIS_ADAPTER pAdapter )
{
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    int         nOffset;
    NDIS_STATUS NdisStatus;

    /* Allocate memory for RX & TX descriptors. */
    pAdapter->m_DescPool.ulAllocSize =
        pHardware->m_RxDescInfo.nSize * pHardware->m_RxDescInfo.cnAlloc +
        pHardware->m_TxDescInfo.nSize * pHardware->m_TxDescInfo.cnAlloc +
        DESC_ALIGNMENT;
    pAdapter->m_DescPool.bCached = FALSE;

    NdisMAllocateSharedMemory( pAdapter->m_hAdapter,
        pAdapter->m_DescPool.ulAllocSize,
        pAdapter->m_DescPool.bCached,
        &pAdapter->m_DescPool.pAllocVirtual,
        &pAdapter->m_DescPool.AllocPhysicalAddr );

    if ( pAdapter->m_DescPool.pAllocVirtual == NULL )
    {
        pAdapter->m_DescPool.ulAllocSize = 0;
        return NDIS_STATUS_RESOURCES;
    }
    else if ( NdisGetPhysicalAddressHigh(
            pAdapter->m_DescPool.AllocPhysicalAddr ) != 0 )
    {
        NdisMFreeSharedMemory( pAdapter->m_hAdapter,
            pAdapter->m_DescPool.ulAllocSize,
            pAdapter->m_DescPool.bCached,
            pAdapter->m_DescPool.pAllocVirtual,
            pAdapter->m_DescPool.AllocPhysicalAddr );
        pAdapter->m_DescPool.ulAllocSize = 0;
        pAdapter->m_DescPool.pAllocVirtual = NULL;
        return NDIS_STATUS_RESOURCES;
    }

    NdisZeroMemory( pAdapter->m_DescPool.pAllocVirtual,
        pAdapter->m_DescPool.ulAllocSize );

    /* Align to the next cache line boundary. */
    nOffset = ((( ULONG_PTR ) pAdapter->m_DescPool.pAllocVirtual % DESC_ALIGNMENT )
        ? ( DESC_ALIGNMENT - ( int )
        (( ULONG_PTR ) pAdapter->m_DescPool.pAllocVirtual % DESC_ALIGNMENT )) : 0 );
    pAdapter->m_DescPool.pVirtual = pAdapter->m_DescPool.pAllocVirtual +
        nOffset;
    pAdapter->m_DescPool.ulPhysical = NdisGetPhysicalAddressLow(
        pAdapter->m_DescPool.AllocPhysicalAddr ) + nOffset;

    /* Allocate receive/transmit descriptors. */
    pHardware->m_RxDescInfo.phwRing = ( PTHw_Desc )
        pAdapter->m_DescPool.pVirtual;
    pHardware->m_RxDescInfo.ulRing = pAdapter->m_DescPool.ulPhysical;
    nOffset = pHardware->m_RxDescInfo.cnAlloc * pHardware->m_RxDescInfo.nSize;
    pHardware->m_TxDescInfo.phwRing = ( PTHw_Desc )
        ( pAdapter->m_DescPool.pVirtual + nOffset );
    pHardware->m_TxDescInfo.ulRing = pAdapter->m_DescPool.ulPhysical + nOffset;

    NdisStatus = AllocateSoftDescriptors( &pHardware->m_RxDescInfo, FALSE );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return( NdisStatus );
    }
    NdisStatus = AllocateSoftDescriptors( &pHardware->m_TxDescInfo, TRUE );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return( NdisStatus );
    }

    return NDIS_STATUS_SUCCESS;
}  /* AllocateDescriptors */


/*
    InitBuffers

    Description:
        This local routine initializes the DMA buffer information structure.

    Parameters:
        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

    Return (None):
*/

static void InitBuffers (
    PBUFFER_INFO pBufInfo )
{
    PDMA_BUFFER pDma;
    int         i;

#ifdef UNDER_CE
    PDMA_BUFFER pPrevious = NULL;

    pBufInfo->pTop = pBufInfo->BufArray;
    pBufInfo->cnAvail = pBufInfo->cnAlloc;
#endif

    pDma = pBufInfo->BufArray;
    for ( i = 0; i < pBufInfo->cnAlloc; i++ )
    {
#ifndef UNDER_CE
        NdisInterlockedPushEntrySList( &pBufInfo->listHead, &pDma->listEntry,
            &pBufInfo->listLock );
        pDma++;

#else
        pPrevious = pDma;
        pDma++;
        pPrevious->pNext = pDma;
#endif
    }

#ifdef UNDER_CE
    pPrevious->pNext = NULL;
#endif
}  /* InitBuffers */


/*
    AllocateBuffers

    Description:
        This local function allocates DMA buffers for use in receiving and
        transmitting.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

        int fPacket
            Indication that packets are allocated for receiving.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

static NDIS_STATUS AllocateBuffers (
    PNDIS_ADAPTER pAdapter,
    PBUFFER_INFO  pBufInfo,
    int           fPacket )
{
    NDIS_PHYSICAL_ADDRESS AllocPhysicalAddr;
    NDIS_PHYSICAL_ADDRESS PhysicalAddr;
    NDIS_STATUS           NdisStatus;
    PDMA_BUFFER           pDma;
    PUCHAR                pAllocVirtual;
    PUCHAR                pVirtual;
    UINT                  ulAllocSize;
    UINT                  ulPhysical;
    int                   i;
    int                   nOffset;
    PSHARED_MEM           pBufPool = &pBufInfo->MemPool;
    NDIS_PHYSICAL_ADDRESS HighestAcceptableMax =
        NDIS_PHYSICAL_ADDRESS_CONST( -1, -1 );

    /* Attempt to allocate all the buffer space in one big block. */
    /* If it fails, allocate each buffer individually. */
    pBufPool->ulAllocSize = ( pBufInfo->cnAlloc * NDIS_BUFFER_SIZE ) +
        BUFFER_ALIGNMENT;
    pBufPool->bCached = TRUE;

    NdisMAllocateSharedMemory( pAdapter->m_hAdapter,
        pBufPool->ulAllocSize,
        pBufPool->bCached,
        &pBufPool->pAllocVirtual,
        &pBufPool->AllocPhysicalAddr );

    /* Big buffer not allocated. */
    if ( pBufPool->pAllocVirtual == NULL )
    {
        pBufPool->ulAllocSize = 0;
    }
    else if ( NdisGetPhysicalAddressHigh( pBufPool->AllocPhysicalAddr ) != 0 )
    {
        NdisMFreeSharedMemory( pAdapter->m_hAdapter,
            pBufPool->ulAllocSize,
            pBufPool->bCached,
            pBufPool->pAllocVirtual,
            pBufPool->AllocPhysicalAddr );
        pBufPool->pAllocVirtual = NULL;
        pBufPool->ulAllocSize = 0;
    }
    else
    {
        NdisZeroMemory( pBufPool->pAllocVirtual, pBufPool->ulAllocSize );
        nOffset = ((( ULONG_PTR ) pBufPool->pAllocVirtual % BUFFER_ALIGNMENT ) ?
            ( BUFFER_ALIGNMENT - ( int )
            (( ULONG_PTR ) pBufPool->pAllocVirtual % BUFFER_ALIGNMENT )) : 0 );
        pBufPool->pVirtual = pBufPool->pAllocVirtual + nOffset;
        pBufPool->ulPhysical = NdisGetPhysicalAddressLow(
            pBufPool->AllocPhysicalAddr ) + nOffset;
        nOffset = 0;
    }

#if (NDISVER >= 50)
    NdisStatus = NdisAllocateMemoryWithTag(( PVOID* ) &pBufInfo->BufArray,
        sizeof( DMA_BUFFER ) * pBufInfo->cnAlloc, 'rciM' );
#else
    NdisStatus = NdisAllocateMemory(( PVOID* ) &pBufInfo->BufArray,
        sizeof( DMA_BUFFER ) * pBufInfo->cnAlloc,
        0, HighestAcceptableMax );
#endif
    NdisZeroMemory( pBufInfo->BufArray, sizeof( DMA_BUFFER ) *
        pBufInfo->cnAlloc );

    /* Set up each buffer. */
    pAllocVirtual = NULL;
    NdisSetPhysicalAddressHigh( AllocPhysicalAddr, 0 );
    NdisSetPhysicalAddressLow( AllocPhysicalAddr, 0 );
    NdisSetPhysicalAddressHigh( PhysicalAddr, 0 );
    ulAllocSize = pBufPool->ulAllocSize;

    pDma = pBufInfo->BufArray;
    for ( i = 0; i < pBufInfo->cnAlloc; i++ )
    {
        /* Big buffer allocated. */
        if ( pBufPool->pAllocVirtual )
        {
            /* Allocate each small buffer from big buffer. */
            pVirtual = pBufPool->pVirtual + nOffset;
            ulPhysical = pBufPool->ulPhysical + nOffset;
            NdisSetPhysicalAddressLow( PhysicalAddr, ulPhysical );
            nOffset += NDIS_BUFFER_SIZE;
        }
        else
        {
            ulAllocSize = NDIS_BUFFER_SIZE + BUFFER_ALIGNMENT;

            /* Allocate each small buffer from system. */
            NdisMAllocateSharedMemory( pAdapter->m_hAdapter,
                ulAllocSize,
                TRUE,
                &pAllocVirtual,
                &AllocPhysicalAddr );
            if ( pAllocVirtual == NULL )
            {
                return NDIS_STATUS_RESOURCES;
            }
            else if ( NdisGetPhysicalAddressHigh( AllocPhysicalAddr ) != 0 )
            {
                NdisMFreeSharedMemory( pAdapter->m_hAdapter,
                    ulAllocSize,
                    TRUE,
                    pAllocVirtual,
                    AllocPhysicalAddr );
                return NDIS_STATUS_RESOURCES;
            }

            NdisZeroMemory( pAllocVirtual, ulAllocSize );
            nOffset = ((( ULONG_PTR ) pAllocVirtual % BUFFER_ALIGNMENT ) ?
                ( BUFFER_ALIGNMENT - ( int )
                (( ULONG_PTR ) pAllocVirtual % BUFFER_ALIGNMENT )) : 0 );
            pVirtual = pAllocVirtual + nOffset;
            ulPhysical = NdisGetPhysicalAddressLow( AllocPhysicalAddr ) +
                nOffset;
            NdisSetPhysicalAddressLow( PhysicalAddr, ulPhysical );
        }

        /* Save the allocated memory information. */
        pDma->sm.ulAllocSize = ulAllocSize;
        pDma->sm.pAllocVirtual = pAllocVirtual;
        pDma->sm.AllocPhysicalAddr = AllocPhysicalAddr;

        /* These addresses point to the data buffer. */
        pDma->sm.pVirtual = pVirtual;
        pDma->sm.ulPhysical = ulPhysical;
        pDma->PhysicalAddr = PhysicalAddr;
        pDma->ulSize = NDIS_BUFFER_SIZE;

        /* Allocate an NDIS flush buffer for each buffer. */
        NdisAllocateBuffer(
            &NdisStatus,
            &pDma->pNdisBuffer,
            pAdapter->m_hBufferPool,
            pDma->sm.pVirtual,
            pDma->ulSize );
        if ( NdisStatus != NDIS_STATUS_SUCCESS )
        {
            pDma->pNdisBuffer = NULL;
            return NDIS_STATUS_RESOURCES;
        }

        if ( fPacket )
        {
            /* Allocate a packet for each receive buffer. */
            NdisAllocatePacket(
                &NdisStatus,
                &pDma->pNdisPacket,
                pAdapter->m_hPacketPool );
            if ( NdisStatus != NDIS_STATUS_SUCCESS )
            {
                pDma->pNdisPacket = NULL;
                return NDIS_STATUS_RESOURCES;
            }

            /* Set the header size in the packet's Out-Of-Band information. */
            /* All other fields in the Out-Of-Band information have been    */
            /* initialized to 0 by NdisAllocatePacket().                    */
            NDIS_SET_PACKET_HEADER_SIZE(( pDma->pNdisPacket ),
                ETHERNET_HEADER_SIZE );

            /* Chain the buffer on the packet. */
            NdisChainBufferAtFront( pDma->pNdisPacket, pDma->pNdisBuffer );

            /* Save a pointer to the receive buffer with the packet. */
            RESERVED( pDma->pNdisPacket )->pDma = pDma;
        }
        pDma++;
    }
    NdisAllocateSpinLock( &pBufInfo->listLock );

#ifndef UNDER_CE
    NdisInitializeSListHead( &pBufInfo->listHead );
#endif
    InitBuffers( pBufInfo );

    return NDIS_STATUS_SUCCESS;
}  /* AllocateBuffers */


#ifndef INLINE
/*
    Alloc_DMA_Buffer

    Description:
        This function allocates a DMA buffer information block for either
        transmitting or receiving.

    Parameters:
        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

        PTDesc pCurrent
            Pointer to current descriptor.

    Return (PDMA_BUFFER):
        Pointer to the DMA buffer information block if successful; NULL
        otherwise.
*/

PDMA_BUFFER Alloc_DMA_Buffer (
    PBUFFER_INFO pBufInfo,
    PTDesc       pCurrent )
{
    PDMA_BUFFER        pDma;

#ifndef UNDER_CE
    PSINGLE_LIST_ENTRY pEntry;

    pEntry = NdisInterlockedPopEntrySList( &pBufInfo->listHead,
        &pBufInfo->listLock );
    ASSERT( pEntry != NULL );
    pDma = CONTAINING_RECORD( pEntry, DMA_BUFFER, listEntry );

#else
    ASSERT( pBufInfo->cnAvail );
    pDma = pBufInfo->pTop;
    pBufInfo->pTop = pDma->pNext;
    pBufInfo->cnAvail--;
#endif

    /* Link to current descriptor. */
    pCurrent->pReserved = pDma;
    return( pDma );
}  // Alloc_DMA_Buffer


/*
    Free_DMA_Buffer

    Description:
        This routine frees the DMA buffer information block.

    Parameters:
        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

        PDMA_BUFFER pDma
            Pointer to DMA buffer information block.

    Return (None):
*/

VOID Free_DMA_Buffer (
    PBUFFER_INFO pBufInfo,
    PDMA_BUFFER  pDma )
{
#ifndef UNDER_CE
    NdisInterlockedPushEntrySList( &pBufInfo->listHead, &pDma->listEntry,
        &pBufInfo->listLock );

#else
    pDma->pNext = pBufInfo->pTop;
    pBufInfo->pTop = pDma;
    pBufInfo->cnAvail++;
#endif
}  // Free_DMA_Buffer
#endif


/*
    InitReceiveBuffers

    Description:
        This routine initializes DMA buffers for receiving.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void InitReceiveBuffers (
        PNDIS_ADAPTER pAdapter )
{
    int         i;
    PTDesc      pDesc;
    PDMA_BUFFER pDma;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    PTDescInfo  pInfo = &pHardware->m_RxDescInfo;

    for ( i = 0; i < pHardware->m_RxDescInfo.cnAlloc; i++ )
    {
        GetRxPacket( pInfo, pDesc );

        /* Get a new buffer for this descriptor. */
        pDma = Alloc_DMA_Buffer( &pAdapter->m_RxBufInfo, pDesc );

        /* Set descriptor. */
        SetReceiveBuffer( pDesc, pDma->sm.ulPhysical );
        SetReceiveLength( pDesc, pDma->ulSize );
        ReleasePacket( pDesc );
    }
}  /* InitReceiveBuffers */


/*
    AllocateMemory

    Description:
        This function allocates memory for use by hardware descriptors for
        receiving and transmitting.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS AllocateMemory (
        PNDIS_ADAPTER pAdapter )
{
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS NdisStatus;

    /* Determine the number of receive and transmit descriptors. */

#ifdef SKIP_TX_INT
    pHardware->m_TxIntCnt = 0;
    pHardware->m_TxIntMask = pHardware->m_TxDescInfo.cnAlloc / 4;
    if ( pHardware->m_TxIntMask > 8 )
    {
        pHardware->m_TxIntMask = 8;
    }
    while ( pHardware->m_TxIntMask ) {
        pHardware->m_TxIntCnt++;
        pHardware->m_TxIntMask >>= 1;
    }
    if ( pHardware->m_TxIntCnt ) {
        pHardware->m_TxIntMask = ( 1 << ( pHardware->m_TxIntCnt - 1 )) - 1;
        pHardware->m_TxIntCnt = 0;
    }
#endif

    /* Determine the descriptor size. */
    pHardware->m_RxDescInfo.nSize =
        ((( sizeof( THw_Desc ) + DESC_ALIGNMENT - 1 ) / DESC_ALIGNMENT ) *
        DESC_ALIGNMENT );
    pHardware->m_TxDescInfo.nSize =
        ((( sizeof( THw_Desc ) + DESC_ALIGNMENT - 1 ) / DESC_ALIGNMENT ) *
        DESC_ALIGNMENT );

    if ( pHardware->m_RxDescInfo.nSize != sizeof( THw_Desc ) )
    {
        DBG_PRINT( "Hardware descriptor size not right!"NEWLINE );
    }
    CheckDescriptorNum( &pHardware->m_RxDescInfo );
    CheckDescriptorNum( &pHardware->m_TxDescInfo );

    /* TX/RX buffers we need. */
#if 1
    /* More receive buffers so that they can be used by upper layer and
       returned later in the MiniportReturnPacket call.
    */
    pAdapter->m_RxBufInfo.cnAlloc = pHardware->m_RxDescInfo.cnAlloc << 1;
#else
    pAdapter->m_RxBufInfo.cnAlloc = pHardware->m_RxDescInfo.cnAlloc;
#endif
    pAdapter->m_TxBufInfo.cnAlloc = pHardware->m_TxDescInfo.cnAlloc;

    /* Allocate map registers. */
    NdisStatus = AllocateMapRegisters( pAdapter );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return NdisStatus;
    }

    /* Allocate descriptors */
    NdisStatus = AllocateDescriptors( pAdapter );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return NdisStatus;
    }

    /* Allocate a pool of NDIS buffers. */
    NdisAllocateBufferPool(
        &NdisStatus,
        &pAdapter->m_hBufferPool,
        pAdapter->m_RxBufInfo.cnAlloc + pAdapter->m_TxBufInfo.cnAlloc );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return NdisStatus;
    }

    /* Allocate a pool of packets. */
    NdisAllocatePacketPool(
        &NdisStatus,
        &pAdapter->m_hPacketPool,
        pAdapter->m_RxBufInfo.cnAlloc,
        0 );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return NdisStatus;
    }

    /* Allocate receive buffers. */
    NdisStatus = AllocateBuffers( pAdapter, &pAdapter->m_RxBufInfo, TRUE );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return NdisStatus;
    }

    /* Allocate transmit buffers. */
    NdisStatus = AllocateBuffers( pAdapter, &pAdapter->m_TxBufInfo, FALSE );
    if ( NdisStatus != NDIS_STATUS_SUCCESS )
    {
        return NdisStatus;
    }

    InitReceiveBuffers( pAdapter );

#ifdef DBG_
    CheckDescriptors( &pAdapter->m_Hardware.m_RxDescInfo );
    CheckDescriptors( &pAdapter->m_Hardware.m_TxDescInfo );
#endif

    return NDIS_STATUS_SUCCESS;
}  /* AllocateMemory */


/*
    FreeMapRegisters

    Description:
        This local routine frees the mapping registers allocated by
        AllocateMapRegisters().

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

static void FreeMapRegisters (
        PNDIS_ADAPTER pAdapter )
{
    /* free map registers */
    if ( pAdapter->m_cnMapRegAlloc )
    {
        NdisMFreeMapRegisters( pAdapter->m_hAdapter );
        NdisFreeMemory( pAdapter->m_MapReg,
            sizeof( MAP_REG ) * pAdapter->m_cnMapRegAlloc, 0 );
        pAdapter->m_cnMapRegAlloc = 0;
    }

    if ( pAdapter->m_cnPackets )
    {
        NdisFreeMemory( pAdapter->m_PacketArray,
            sizeof( PNDIS_PACKET ) * pAdapter->m_cnPackets, 0 );
        pAdapter->m_cnPackets = 0;
    }
}  /* FreeMapRegisters */


/*
    FreeDescriptors

    Description:
        This local routine frees the software and hardware descriptors
        allocated by AllocateDescriptors().

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

static void FreeDescriptors (
        PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    /* reset descriptor */
    pHardware->m_RxDescInfo.phwRing = NULL;
    pHardware->m_TxDescInfo.phwRing = NULL;
    pHardware->m_RxDescInfo.ulRing = 0;
    pHardware->m_TxDescInfo.ulRing = 0;

    /* Free memory */
    if ( pAdapter->m_DescPool.pAllocVirtual )
        NdisMFreeSharedMemory( pAdapter->m_hAdapter,
            pAdapter->m_DescPool.ulAllocSize,
            pAdapter->m_DescPool.bCached,
            pAdapter->m_DescPool.pAllocVirtual,
            pAdapter->m_DescPool.AllocPhysicalAddr );

    /* reset resource pool */
    pAdapter->m_DescPool.ulAllocSize = 0;
    pAdapter->m_DescPool.pAllocVirtual = NULL;

    if ( pHardware->m_RxDescInfo.pRing )
    {
        NdisFreeMemory( pHardware->m_RxDescInfo.pRing,
            sizeof( TDesc ) * pHardware->m_RxDescInfo.cnAlloc, 0 );
        pHardware->m_RxDescInfo.pRing = NULL;
    }
    if ( pHardware->m_TxDescInfo.pRing )
    {
        NdisFreeMemory( pHardware->m_TxDescInfo.pRing,
            sizeof( TDesc ) * pHardware->m_TxDescInfo.cnAlloc, 0 );
        pHardware->m_TxDescInfo.pRing = NULL;
    }
}  /* FreeDescriptors */


/*
    FreeBuffers

    Description:
        This local routine frees the DMA buffers allocated by
        AllocateBuffers().

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

    Return (None):
*/

static void FreeBuffers (
        PNDIS_ADAPTER pAdapter,
        PBUFFER_INFO  pBufInfo )
{
    int         i;
    PDMA_BUFFER pDma;
    PSHARED_MEM pBufPool = &pBufInfo->MemPool;

    if ( !pBufInfo->BufArray )
        return;

    pDma = pBufInfo->BufArray;
    for ( i = 0; i < pBufInfo->cnAlloc; i++ )
    {
        if ( pDma->pNdisPacket )
        {
            NdisFreePacket( pDma->pNdisPacket );
            pDma->pNdisPacket = NULL;
        }
        if ( pDma->pNdisBuffer )
        {
            /* Need to restore to original length before releasing. */
            NdisAdjustBufferLength( pDma->pNdisBuffer, pDma->ulSize );
            NdisFreeBuffer( pDma->pNdisBuffer );
            pDma->pNdisBuffer = NULL;
        }
        if ( pDma->sm.pAllocVirtual )
        {
            NdisMFreeSharedMemory( pAdapter->m_hAdapter,
                pDma->sm.ulAllocSize,
                pDma->sm.bCached,
                pDma->sm.pAllocVirtual,
                pDma->sm.AllocPhysicalAddr );
        }
        pDma++;
    }
    NdisFreeMemory( pBufInfo->BufArray,
        sizeof( DMA_BUFFER ) * pBufInfo->cnAlloc, 0 );
    pBufInfo->BufArray = NULL;

    if ( pBufPool->pAllocVirtual )
    {
        NdisMFreeSharedMemory( pAdapter->m_hAdapter,
            pBufPool->ulAllocSize,
            pBufPool->bCached,
            pBufPool->pAllocVirtual,
            pBufPool->AllocPhysicalAddr );
        pBufPool->ulAllocSize = 0;
        pBufPool->pAllocVirtual = NULL;
    }
    NdisFreeSpinLock( &pBufInfo->listLock );
}  /* FreeBuffers */


/*
    FreeMemory

    Description:
        This local routine frees all the resources allocated by
        AllocateMemory().

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID FreeMemory (
        PNDIS_ADAPTER pAdapter )
{
    /* Free transmit buffers. */
    FreeBuffers( pAdapter, &pAdapter->m_TxBufInfo );

    /* Free receive buffers */
    FreeBuffers( pAdapter, &pAdapter->m_RxBufInfo );

    /* Free map registers. */
    FreeMapRegisters( pAdapter );

    /* Free descriptors. */
    FreeDescriptors( pAdapter );

    /* Free packet pool handle. */
    if ( pAdapter->m_hPacketPool )
    {
        NdisFreePacketPool( pAdapter->m_hPacketPool );
        pAdapter->m_hPacketPool = 0;
    }

    /* Free buffer pool handle. */
    if ( pAdapter->m_hBufferPool )
    {
        NdisFreeBufferPool( pAdapter->m_hBufferPool );
        pAdapter->m_hBufferPool = 0;
    }
}  /* FreeMemory */

/* -------------------------------------------------------------------------- */

/*
    InitAdapter

    Description:
        This function initializes the adapter when the adapter is started.  It
        calls ReadConfiguration() to get all information required to start the
        adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        NDIS_HANDLE hAdapter
            Handle to adapter context containing adapter information.

        NDIS_HANDLE hConfiguration
            Handle to adapter configuration.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS InitAdapter (
        PNDIS_ADAPTER pAdapter,
    IN  NDIS_HANDLE   hAdapter,
    IN  NDIS_HANDLE   hConfiguration )
{
    NDIS_STATUS nsStatus;

    // Save the adapter handle furnished by NDIS.
    pAdapter->m_hAdapter = hAdapter;

    nsStatus = ReadConfiguration( pAdapter, hConfiguration );
    if ( !NT_SUCCESS( nsStatus ) )
    {
        return( nsStatus );
    }

    /* Need to set bus master else NdisMAllocateSharedMemory will not succeed.
    */
#if (NDISVER < 51)
    NdisMSetAttributes( pAdapter->m_hAdapter, ( NDIS_HANDLE ) pAdapter, TRUE,
        pAdapter->m_BusType );

#else
    NdisMSetAttributesEx( pAdapter->m_hAdapter, ( NDIS_HANDLE ) pAdapter, 0,
        NDIS_ATTRIBUTE_SURPRISE_REMOVE_OK | NDIS_ATTRIBUTE_BUS_MASTER,
        pAdapter->m_BusType );
#endif

    if ( !NT_SUCCESS( AllocateMemory( pAdapter )) )
    {
        return NDIS_STATUS_FAILURE;
    }

    return NDIS_STATUS_SUCCESS;
}  // InitAdapter


/*
    SetupAdapter

    Description:
        This function setups the adapter when the adapter is started.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS SetupAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;
    UCHAR     bPromiscuous;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }
    if ( !HardwareInitialize( pHardware )  ||  !HardwareReset( pHardware ) )
    {
        NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
            NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1, 0x10560100 );
        ReleaseAdapter( pAdapter );
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }

    HardwareDisable( pHardware );
    HardwareSetPowerSaving( pHardware, FALSE );

#if 0
    HardwareGetVersion( pHardware );
#endif

    // Read the Ethernet address.
    if ( !HardwareReadAddress( pHardware ) )
    {
        NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
            NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1, 0x10560101 );
        ReleaseAdapter( pAdapter );
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }

    pHardware->m_ulHardwareState = MediaStateDisconnected;

    /* Initialize to invalid value so that link detection is done. */
    pHardware->m_PortInfo[ MAIN_PORT ].bLinkPartner = 0xFF;
    pHardware->m_PortInfo[ MAIN_PORT ].ulHardwareState =
        MediaStateDisconnected;

#ifdef DEF_KS8842
    pHardware->m_PortInfo[ OTHER_PORT ].bLinkPartner = 0xFF;
    pHardware->m_PortInfo[ OTHER_PORT ].ulHardwareState =
        MediaStateDisconnected;
#endif

    HardwareSetup( pHardware );
    if ( !pAdapter->m_ulRxChecksum )
    {
        pHardware->m_dwReceiveConfig &= ~( DMA_RX_CTRL_CSUM_TCP |
            DMA_RX_CTRL_CSUM_UDP | DMA_RX_CTRL_CSUM_IP );
    }
    HardwareSwitchSetup( pHardware );
    HardwareSetDescriptorBase( pHardware,
        pHardware->m_TxDescInfo.ulRing,
        pHardware->m_RxDescInfo.ulRing );

    if ( ( pAdapter->m_ulPacketFilter &
            ( NDIS_PACKET_TYPE_PROMISCUOUS | NDIS_PACKET_TYPE_ALL_LOCAL )) )
        bPromiscuous = TRUE;
    else
        bPromiscuous = FALSE;
    if ( !HardwareSetPromiscuous( pHardware, bPromiscuous ) )
    {
        ReleaseAdapter( pAdapter );
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }

    pAdapter->m_ulNdisMediaState = pAdapter->m_Hardware.m_ulHardwareState;

    PortInitCounters( pHardware, MAIN_PORT );

#if defined( DEF_KS8842 )  &&  !defined( TWO_NETWORK_INTERFACE )
    PortInitCounters( pHardware, OTHER_PORT );
#endif
    PortInitCounters( pHardware, HOST_PORT );

    HardwareSetupInterrupt( pHardware );

#ifdef DBG
DbgPrint( "Setup completed"NEWLINE );
#endif

#if 0
    HardwareGetInfo( pHardware );
#endif

    ReleaseAdapter( pAdapter );

#if (NDISVER >= 50)
    /* Specifies the lowest device power state from which the
       miniport's NIC can signal a wake-up on receipt of a Magic
       Packet.  (A Magic Packet is a packet that contains 16 contiguous
       copies of the receiving NIC's Ethernet address.)
    */
#ifdef DEF_KS8841
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinMagicPacketWakeUp =
        NdisDeviceStateD3;
#else
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinMagicPacketWakeUp =
        NdisDeviceStateUnspecified;
#endif

    /* Specifies the lowest device power state from which the
       miniport's NIC can signal a wake-up event on receipt of a
       network frame that contains a pattern specified by the protocol
       driver.
    */
#ifdef DEF_KS8841
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinPatternWakeUp =
        NdisDeviceStateD3;
#else
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinPatternWakeUp =
        NdisDeviceStateUnspecified;
#endif

    /* Specifies the lowest device power state from which the
       miniport's NIC can signal a wake-up event in response to a link
       change (the connection or disconnection of the NIC's network
       cable).
    */
#ifdef DEF_KS8841
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinLinkChangeWakeUp =
        NdisDeviceStateD3;
#else
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinLinkChangeWakeUp =
        NdisDeviceStateUnspecified;
#endif

    pAdapter->CurrentPowerState = pAdapter->NextPowerState =
        NdisDeviceStateD0;
#endif

    phw = pHardware;

    return NDIS_STATUS_SUCCESS;
}  // SetupAdapter


/*
    ShutdownAdapter

    Description:
        This routine shutdowns the adapter when the driver is unloaded.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void ShutdownAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

#ifndef UNDER_CE
#ifdef DBG_
    int       i;
#endif
#endif

    AcquireAdapter( pAdapter, FALSE );
    HardwareDisableInterrupt( pHardware );
    HardwareDisable( pHardware );
    ReleaseAdapter( pAdapter );

#ifndef UNDER_CE
#ifdef DBG_
    DbgPrint( "counters:\n" );
    for ( i = OID_COUNTER_FIRST; i < OID_COUNTER_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_cnCounter[ 0 ][ i ]);
    }
    DbgPrint( "wait delays:\n" );
    for ( i = WAIT_DELAY_FIRST; i < WAIT_DELAY_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_nWaitDelay[ i ]);
    }
    DbgPrint( "bad:\n" );
    for ( i = COUNT_BAD_FIRST; i < COUNT_BAD_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_nBad[ i ]);
    }
    DbgPrint( "good:\n" );
    for ( i = COUNT_GOOD_FIRST; i < COUNT_GOOD_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_nGood[ i ]);
    }
#endif
#endif
}  // ShutdownAdapter


/*
    StartAdapter

    Description:
        This routine starts the adapter when the driver is woke up.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void StartAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    AcquireAdapter( pAdapter, FALSE );
    HardwareEnable( pHardware );
    HardwareEnableInterrupt( pHardware );
    ReleaseAdapter( pAdapter );
}  // StartAdapter

/* -------------------------------------------------------------------------- */

/*
    MiniportSynchronizeISR

    Description:
        This function is used to do things not interruptable by software
        interrupt.

    Parameters:
        PVOID Context
            Pointer to hardware information structure.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN MiniportSynchronizeISR (
    IN  PVOID Context )
{
    PHARDWARE pHardware = ( PHARDWARE ) Context;

    HardwareDisableInterrupt( pHardware );
    return TRUE;
}  // MiniportSynchronizeISR


/*
    HardwareDisableInterruptSync

    Description:
        This function is used to disable hardware interrupt atomically so that
        the operation is not interrupted.

    Parameters:
        void* pHardware
            Pointer to hardware information structure.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

void HardwareDisableInterruptSync (
    void* pHardware )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER )(( PHARDWARE ) pHardware )->
        m_pDevice;

    NdisMSynchronizeWithInterrupt( &pAdapter->m_Interrupt,
        MiniportSynchronizeISR, pHardware );
}  /* HardwareDisableInterruptSync */

/* -------------------------------------------------------------------------- */

#define ADAPTER_LOCK_TIMEOUT  100

/*
    AcquireAdapter

    Description:
        This function is used to acquire the adapter so that only one process
        has access to the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        BOOLEAN fWait
            Indicate whether to wait or not.  User functions should wait, while
            periodic timer routine should not.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN AcquireAdapter (
    IN  PNDIS_ADAPTER pAdapter,
    IN  BOOLEAN       fWait )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;
    int       cnTimeOut = ADAPTER_LOCK_TIMEOUT;

    while ( --cnTimeOut )
    {
        NdisAcquireSpinLock( &pAdapter->m_lockHardware );

        if ( !pHardware->m_bAcquire )
        {
            pHardware->m_bAcquire = TRUE;
            NdisReleaseSpinLock( &pAdapter->m_lockHardware );
            return TRUE;
        }
        NdisReleaseSpinLock( &pAdapter->m_lockHardware );
        NdisMSleep( 10 );
    }
    if ( fWait )
    {

    }
    return FALSE;
}  // AcquireAdapter


/*
    ReleaseAdapter

    Description:
        This routine is used to release the adapater so that other process can
        have access to the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void ReleaseAdapter (
    IN  PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    NdisAcquireSpinLock( &pAdapter->m_lockHardware );
    pHardware->m_bAcquire = FALSE;
    NdisReleaseSpinLock( &pAdapter->m_lockHardware );
}  // ReleaseAdapter

/* -------------------------------------------------------------------------- */

/*
    AdapterCheckForHang

    Description:
        This function checks the adapter is working properly.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (BOOLEAN):
        TRUE if adapter is not working; FALSE if working properly.
*/

BOOLEAN AdapterCheckForHang (
    IN  PNDIS_ADAPTER pAdapter )
{
#if SET_DEFAULT_LED
    PHARDWARE pHardware = &pAdapter->m_Hardware;
    USHORT    wData;

    HW_READ_WORD( pHardware, REG_SWITCH_CTRL_5_OFFSET, &wData );
    if ( ( TO_LO_BYTE( wData ) & SET_DEFAULT_LED ) != SET_DEFAULT_LED )
    {
        return TRUE;
    }
#endif

    return FALSE;
}  // AdapterCheckForHang


/*
    AdapterGetDescription

    Description:
        This routine returns the adapter description.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID AdapterGetDescription (
    PVOID* ppSource,
    PUINT  puiLength )
{
#ifdef DEF_KS8841
    static char gszDescription[] = "KSZ8841 PCI Ethernet";

#else
    static char gszDescription[] = "KSZ8842 PCI Ethernet";
#endif

    *ppSource = gszDescription;
    *puiLength = strlen( gszDescription );
}  // AdapterGetDescription


/*
    AdapterReadCounter

    Description:
        This function retrieves the statistics counter value.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        EOidCounter OidCounter
            The counter index.

    Return (ULONGLONG):
        The counter value.
*/

ULONGLONG AdapterReadCounter (
    PNDIS_ADAPTER pAdapter,
    EOidCounter   OidCounter )
{
    PHARDWARE    pHardware = &pAdapter->m_Hardware;
    PPORT_CONFIG pPort = &pHardware->m_Port[ MAIN_PORT ];
    ULONGLONG    qCounter = pHardware->m_cnCounter[ MAIN_PORT ][ OidCounter ];

    switch ( OidCounter )
    {
        case OID_COUNTER_XMIT_OK:

#if 0
#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_RX_MULTICAST ] +
                pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_TX_MULTICAST ] +
                pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];
#endif
#endif
#if DBG
            DBG_PRINT( "xmit ok: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_RCV_OK:

#if 0
#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];
#endif
#endif
#if DBG
            DBG_PRINT( "rcv ok: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_DIRECTED_FRAMES_XMIT:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];
#endif
#if DBG
            DBG_PRINT( "xmit frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_BROADCAST_FRAME_XMIT:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ];
#endif
#if DBG
            DBG_PRINT( "broadcast frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_DIRECTED_FRAMES_RCV:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];
#endif
#if DBG
            DBG_PRINT( "rcv frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_BROADCAST_FRAMES_RCV:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ];
#endif
#if DBG
            DBG_PRINT( "rcv broadcast frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_RCV_ERROR_CRC:
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_CRC_ERR ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_RX_CRC_ERR ];
#endif
            break;
    }

#ifndef UNDER_CE
#ifdef DBG
    switch ( OidCounter )
    {
        case OID_COUNTER_DIRECTED_BYTES_XMIT:
            DBG_PRINT( "xmit bytes: %u"NEWLINE, ( UINT ) qCounter );
            break;
        case OID_COUNTER_BROADCAST_BYTES_XMIT:
            DBG_PRINT( "broadcast bytes: %u"NEWLINE, ( UINT ) qCounter );
            break;
        case OID_COUNTER_DIRECTED_BYTES_RCV:
            DBG_PRINT( "rcv bytes: %u"NEWLINE, ( UINT ) qCounter );
            break;
        case OID_COUNTER_BROADCAST_BYTES_RCV:
            DBG_PRINT( "rcv broadcast: %u"NEWLINE, ( UINT ) qCounter );
            break;
    }
#endif
#endif

    return( qCounter );
}  // AdapteReadCounter


/*
    AdapterReset

    Description:
        This function resets the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN AdapterReset (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    AcquireAdapter( pAdapter, FALSE );
    if ( NdisDeviceStateD0 == pAdapter->CurrentPowerState )
    {
        HardwareDisableInterrupt( pHardware );
        HardwareDisable( pHardware );

        HardwareReset( pHardware );
    }
    {
        pHardware->m_ulHardwareState = MediaStateDisconnected;

        /* Initialize to invalid value so that link detection is done. */
        pHardware->m_PortInfo[ MAIN_PORT ].bLinkPartner = 0xFF;
        pHardware->m_PortInfo[ MAIN_PORT ].ulHardwareState =
            MediaStateDisconnected;

#ifdef DEF_KS8842
        pHardware->m_PortInfo[ OTHER_PORT ].bLinkPartner = 0xFF;
        pHardware->m_PortInfo[ OTHER_PORT ].ulHardwareState =
            MediaStateDisconnected;
#endif

        HardwareSetup( pHardware );
        if ( !pAdapter->m_ulRxChecksum )
        {
            pHardware->m_dwReceiveConfig &= ~( DMA_RX_CTRL_CSUM_TCP |
                DMA_RX_CTRL_CSUM_UDP | DMA_RX_CTRL_CSUM_IP );
        }
        HardwareSwitchSetup( pHardware );
        HardwareSetDescriptorBase( pHardware,
            pHardware->m_TxDescInfo.ulRing,
            pHardware->m_RxDescInfo.ulRing );

        /* Reset may wipe out these registers. */
        if ( pHardware->m_bMacOverrideAddr )
            HardwareSetAddress( pHardware );
        if ( pHardware->m_bMulticastListSize )
            HardwareSetGroupAddress( pHardware );

        TransmitReset( pAdapter );
        HardwareResetRxPackets( pHardware );
        HardwareResetTxPackets( pHardware );

#ifndef UNDER_CE
        NdisInterlockedFlushSList( &pAdapter->m_RxBufInfo.listHead );
        NdisInterlockedFlushSList( &pAdapter->m_TxBufInfo.listHead );
#endif
        InitBuffers( &pAdapter->m_RxBufInfo );
        InitBuffers( &pAdapter->m_TxBufInfo );
        InitMapRegisters( pAdapter );
        InitReceiveBuffers( pAdapter );

#if 0
        memset(( void* ) pHardware->m_cnCounter[ MAIN_PORT ], 0,
            ( sizeof( ULONGLONG ) * OID_COUNTER_LAST ) );
        PortInitCounters( pHardware, MAIN_PORT );

#if defined( DEF_KS8842 )  &&  !defined( TWO_NETWORK_INTERFACE )
        PortInitCounters( pHardware, OTHER_PORT );
#endif
        PortInitCounters( pHardware, HOST_PORT );
#endif

        HardwareEnable( pHardware );
        HardwareEnableInterrupt( pHardware );
        ReleaseAdapter( pAdapter );
        return TRUE;
    }
    ReleaseAdapter( pAdapter );
    return FALSE;
}  // AdapterReset

/* -------------------------------------------------------------------------- */

/*
    MiniportReset

    Description:
        This function is used by NDIS to reset the adapter.

    Parameters:
        PBOOLEAN AddresssingReset
            Flag to indicate adapter needs the addressing information reloaded.

        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise, an error code indicating
        failure.
*/

NDIS_STATUS MiniportReset (
    OUT PBOOLEAN    AddressingReset,
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

    *AddressingReset = FALSE;
    if ( AdapterReset( pAdapter ) )
    {
        pAdapter->m_ulDriverState |= DRIVER_STATE_RESET;
        NdisMSetTimer( &pAdapter->m_ResetTimer, 2000 );
        return( NDIS_STATUS_PENDING );
    }
    return( NDIS_STATUS_FAILURE );
}  // MiniportReset

/* -------------------------------------------------------------------------- */

/*
    MiniportCheckForHang

    Description:
        This function is used by NDIS to check if the adapter is not working.
        The NDIS library calls this function every 2 seconds.  If TRUE is
        returned, it will call MiniportReset to force an adapter reset.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (BOOLEAN):
        TRUE if adapter requires a reset; FALSE if adapter is functioning
        properly.
*/

BOOLEAN MiniportCheckForHang (
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

    return( AdapterCheckForHang( pAdapter ));
}  // MiniportCheckForHang

/* -------------------------------------------------------------------------- */

/*
    MiniportDisableInterrupt

    Description:
        This routine is used by NDIS to disable hardware interrupts.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

VOID MiniportDisableInterrupt (
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

#ifdef DEBUG_INTERRUPT
    DbgPrint( "dis int"NEWLINE );
#endif
    HardwareDisableInterrupt( &pAdapter->m_Hardware );
}  // MiniportDisableInterrupt


/*
    MiniportEnableInterrupt

    Description:
        This routine is used by NDIS to enable hardware interrupts.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

VOID MiniportEnableInterrupt (
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

#ifdef DEBUG_INTERRUPT_
    DbgPrint( "ena int"NEWLINE );
#endif
    HardwareEnableInterrupt( &pAdapter->m_Hardware );
}  // MiniportEnableInterrupt

/* -------------------------------------------------------------------------- */

#pragma pack(1)
typedef struct
{
    // Total command length
    UINT   dwLength;

    // Driver Extension Command
    USHORT wCommand;

    // Option flags
    USHORT wFlags;

    // Driver Extension Return Code
    UINT   dwReturnCode;

    // Parameters
    //UCHAR  bParam[ 4 ];
    BYTE  bParam[ 6 ]; //DAVIDCAI change from 4 to 6 12-17-2009 to pass port index through the bParam[5]
} TDeviceInfo, *PTDeviceInfo;
#pragma pack()


#define DEVICE_COMMAND_INIT       0
#define DEVICE_COMMAND_GET        1
#define DEVICE_COMMAND_SET        2
#define DEVICE_FLAG_OFF           0
#define DEVICE_FLAG_CABLE_STATUS  1
#define DEVICE_FLAG_LINK_STATUS   2
#define DEVICE_FLAG_EEPROM_REG    3

#define DEVICE_FLAG_CAPABILITIES  1

#define DEVICE_COMMAND_COUNTERS   3

#define DEVICE_INIT_SIZE          ( sizeof( TDeviceInfo ) + 8 )
#define DEVICE_INIT_RETURN_SIZE   ( sizeof( TDeviceInfo ) + 4 )
//#define DEVICE_GET_SIZE          ( sizeof( TDeviceInfo ) + 16 )
#define DEVICE_GET_SIZE           ( sizeof( TDeviceInfo ) + 0 )
#define DEVICE_CABLE_STATUS_SIZE  ( sizeof( TDeviceInfo ) + 10 * sizeof( UINT ))
#define DEVICE_LINK_STATUS_SIZE   ( sizeof( TDeviceInfo ) + 6 * sizeof( UINT ))
#define DEVICE_COUNTERS_SIZE      ( sizeof( TDeviceInfo ) + 4 * sizeof( UINT ))


NDIS_STATUS GetDeviceMode (
    IN  PNDIS_ADAPTER pAdapter,
    IN  NDIS_OID      Oid,
    IN  PVOID         pBuffer,
    IN  ULONG         ulLen,
    OUT PULONG        pulBytesWritten,
    OUT PULONG        pulBytesNeeded )
{
    NDIS_STATUS status = NDIS_STATUS_SUCCESS;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;

    // make sure input buffer is valid
    if ( pBuffer  &&  ulLen >= sizeof( TDeviceInfo ) )
    {
        PTDeviceInfo pInfo = ( PTDeviceInfo ) pBuffer;

        *pulBytesNeeded = 0;
        *pulBytesWritten = sizeof( TDeviceInfo );
        if ( Oid == pInfo->dwReturnCode )
        {
            if ( DEVICE_COMMAND_GET == pInfo->wCommand )
            {
                switch ( pInfo->wFlags )
                {
                    case DEVICE_FLAG_CABLE_STATUS:
                        if ( ulLen >= DEVICE_CABLE_STATUS_SIZE )
                        {
                            AcquireAdapter( pAdapter, TRUE );
                            //HardwareGetCableStatus( pHardware, 0,
                              //  pInfo->bParam );
                           //DAVIDCAI changed 12-17-2009  pInfo->bParam[5] is port index  
                            HardwareGetCableStatus( pHardware, pInfo->bParam[5],
                                pInfo->bParam );
                            
                            ReleaseAdapter( pAdapter );
                            *pulBytesWritten = DEVICE_CABLE_STATUS_SIZE;
                        }
                        else
                        {
                            *pulBytesNeeded = DEVICE_CABLE_STATUS_SIZE;
                            status = NDIS_STATUS_BUFFER_TOO_SHORT;
                        }
                        break;
                    case DEVICE_FLAG_LINK_STATUS:
                        if ( ulLen >= DEVICE_LINK_STATUS_SIZE )
                        {
                            AcquireAdapter( pAdapter, TRUE );
                            //HardwareGetLinkStatus( pHardware, 0,
                              //  pInfo->bParam );
                            //DAVIDCAI changed 12-17-2009  pInfo->bParam[5] is port index 
                            
                            HardwareGetLinkStatus( pHardware, pInfo->bParam[5],
                                pInfo->bParam );
                            
                            ReleaseAdapter( pAdapter );
                            *pulBytesWritten = DEVICE_LINK_STATUS_SIZE;
                        }
                        else
                        {
                            *pulBytesNeeded = DEVICE_LINK_STATUS_SIZE;
                            status = NDIS_STATUS_BUFFER_TOO_SHORT;
                        }
                        break;
                    case DEVICE_FLAG_EEPROM_REG:
                    {
                        PUSHORT pwParam = ( PUSHORT ) pInfo->bParam;

                        AcquireAdapter( pAdapter, TRUE );
                        pwParam[ 1 ] = EepromReadWord( pHardware,
                            ( UCHAR ) pwParam[ 0 ]);
                        ReleaseAdapter( pAdapter );
                        *pulBytesWritten = sizeof( TDeviceInfo );
                        break;
                    }
                    default:
                        status = NDIS_STATUS_INVALID_DATA;
                }
            }
            else if ( DEVICE_COMMAND_INIT == pInfo->wCommand )
            {
                if ( ulLen >= DEVICE_INIT_SIZE )
                {
                    pInfo->bParam[ 0 ] = 'M';
                    pInfo->bParam[ 1 ] = 'i';
                    pInfo->bParam[ 2 ] = 'c';
                    pInfo->bParam[ 3 ] = 'r';
                    //DAVIDCAI added 12-17-2009, using bparam[4] to pass back the port information
                    //to new Mirecl LinkMD driver.
                    #ifdef DEF_KS8842
                      pInfo->bParam[ 4 ]=2;
                    #else
                     pInfo->bParam[ 4 ]=1;
                    #endif
                    //DAVIDCAI end
                    *pulBytesWritten = DEVICE_INIT_RETURN_SIZE;
                }
                else
                {
                    *pulBytesNeeded = DEVICE_INIT_SIZE;
                    status = NDIS_STATUS_BUFFER_TOO_SHORT;
                }
            }
            else
            {
                status = NDIS_STATUS_INVALID_DATA;
            }
        }
        else
        {
            status = NDIS_STATUS_INVALID_DATA;
        }
        pInfo->dwLength = *pulBytesWritten;
        pInfo->dwReturnCode = status;
    }
    else
    {
        *pulBytesNeeded = sizeof( TDeviceInfo );
        *pulBytesWritten = 0;
        status = NDIS_STATUS_BUFFER_TOO_SHORT;
    }
    return( status );
}  // GetDeviceMode


NDIS_STATUS SetDeviceMode (
    IN  PNDIS_ADAPTER pAdapter,
    IN  NDIS_OID      Oid,
    IN  PVOID         pBuffer,
    IN  ULONG         ulLen,
    OUT PULONG        pulBytesRead,
    OUT PULONG        pulBytesNeeded )
{
    NDIS_STATUS status = NDIS_STATUS_SUCCESS;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;

    if ( pBuffer  &&  ulLen >= sizeof( TDeviceInfo ) )
    {
        PTDeviceInfo pInfo = ( PTDeviceInfo ) pBuffer;

        *pulBytesNeeded = 0;
        *pulBytesRead = sizeof( TDeviceInfo );
        if ( Oid == pInfo->dwReturnCode  &&
                DEVICE_COMMAND_SET == pInfo->wCommand )
        {
            switch ( pInfo->wFlags )
            {
                case DEVICE_FLAG_CAPABILITIES:
                {
                    PUINT pulData = ( PUINT ) pInfo->bParam;

                    AcquireAdapter( pAdapter, TRUE );
                    HardwareSetCapabilities( pHardware, 0, *pulData );
                    ReleaseAdapter( pAdapter );
                    break;
                }
                case DEVICE_FLAG_EEPROM_REG:
                {
                    PUSHORT pwParam = ( PUSHORT ) pInfo->bParam;

                    AcquireAdapter( pAdapter, TRUE );
                    EepromWriteWord( pHardware, ( UCHAR ) pwParam[ 0 ],
                        pwParam[ 1 ]);
                    ReleaseAdapter( pAdapter );
                    break;
                }
                default:
                    status = NDIS_STATUS_INVALID_DATA;
            }
        }
        else
            status = NDIS_STATUS_INVALID_DATA;
        pInfo->dwLength = *pulBytesRead;
        pInfo->dwReturnCode = status;
    }
    else
    {
        *pulBytesNeeded = sizeof( TDeviceInfo );
        *pulBytesRead = 0;
        status = NDIS_STATUS_BUFFER_TOO_SHORT;
    }
    return( status );
}  // SetDeviceMode


BOOLEAN PciReadConfig (
    IN  PVOID  pAdapterContext,
    IN  int    offset,
    OUT PUINT  pulData )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pAdapterContext;

    if ( NdisReadPciSlotInformation( pAdapter->m_hAdapter, pAdapter->m_ulSlot,
            offset, pulData, sizeof( UINT )) == sizeof( UINT ) )
        return TRUE;
    return FALSE;
}  /* PciReadConfig */


BOOLEAN PciReadConfigByte (
    IN  PVOID  pAdapterContext,
    IN  int    offset,
    OUT PUCHAR pbData )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pAdapterContext;

    if ( NdisReadPciSlotInformation( pAdapter->m_hAdapter, pAdapter->m_ulSlot,
            offset, pbData, sizeof( UCHAR )) == sizeof( UCHAR ) )
        return TRUE;
    return FALSE;
}  /* PciReadConfigByte */


BOOLEAN PciReadConfigWord (
    IN  PVOID   pAdapterContext,
    IN  int     offset,
    OUT PUSHORT pwData )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pAdapterContext;

    if ( NdisReadPciSlotInformation( pAdapter->m_hAdapter, pAdapter->m_ulSlot,
            offset, pwData, sizeof( USHORT )) == sizeof( USHORT ) )
        return TRUE;
    return FALSE;
}  /* PciReadConfigWord */


BOOLEAN PciWriteConfig (
    IN  PVOID pAdapterContext,
    IN  int   offset,
    IN  UINT  ulData )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pAdapterContext;

    if ( NdisWritePciSlotInformation( pAdapter->m_hAdapter, pAdapter->m_ulSlot,
            offset, &ulData, sizeof( UINT )) == sizeof( UINT ) )
        return TRUE;
    return FALSE;
}  /* PciWriteConfig */


BOOLEAN PciWriteConfigByte (
    IN  PVOID pAdapterContext,
    IN  int   offset,
    IN  UCHAR bData )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pAdapterContext;

    if ( NdisWritePciSlotInformation( pAdapter->m_hAdapter, pAdapter->m_ulSlot,
            offset, &bData, sizeof( UCHAR )) == sizeof( UCHAR ) )
        return TRUE;
    return FALSE;
}  /* PciWriteConfigByte */


BOOLEAN PciWriteConfigWord (
    IN  PVOID  pAdapterContext,
    IN  int    offset,
    IN  USHORT wData )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pAdapterContext;

    if ( NdisWritePciSlotInformation( pAdapter->m_hAdapter, pAdapter->m_ulSlot,
            offset, &wData, sizeof( USHORT )) == sizeof( USHORT ) )
        return TRUE;
    return FALSE;
}  /* PciWriteConfigWord */
