/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2006 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisISR.c - NDIS driver interrupt processing.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         06/02/06    Report statistics from MIB counters.
    THa         10/04/06    Report link state in interrupt routine.
   ---------------------------------------------------------------------------
*/


#include "target.h"
#include "NdisDevice.h"


/* -------------------------------------------------------------------------- */

/*
    ProcessReceive

    Description:
        This function processes the buffer receiving.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (BOOLEAN):
        TRUE if a packet is received; FALSE otherwise.
*/

BOOLEAN ProcessReceive (
    IN  PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;
    BOOLEAN   bReceiveDone = FALSE;
    UINT      uiIndicateLength;
    UINT      uiLength;
    USHORT    wData;
    int       port = MAIN_PORT;

    // At this point, receive interrupts are disabled.
    do
    {
        HardwareReceive( pHardware );
        if ( !pHardware->m_nPacketLen )
            goto next_packet;

        uiLength = pHardware->m_nPacketLen - ETHERNET_HEADER_SIZE;
        uiIndicateLength = uiLength;

        // Only allow received packets after Filter is defined.
        if ( pAdapter->m_bPacketFilterSet )
        {
#if defined( DEBUG_RX_DATA )  &&  !defined( UNDER_CE )
            UINT   wLength;
#endif
            PUCHAR pBuffer = &pHardware->m_bLookahead[ ETHERNET_HEADER_SIZE ];

            NdisMEthIndicateReceive( pAdapter->m_hAdapter,
                ( NDIS_HANDLE ) pAdapter, pHardware->m_bLookahead,
                ETHERNET_HEADER_SIZE,
                &pHardware->m_bLookahead[ ETHERNET_HEADER_SIZE ],
                uiIndicateLength, uiLength );

#if defined( DEBUG_RX_DATA )  &&  !defined( UNDER_CE )
            for ( wLength = 0; wLength < uiIndicateLength; wLength++ ) {
                DbgPrint( "%02X ", pBuffer[ wLength ]);
                if ( ( wLength % 16 ) == 15 ) {
                    DbgPrint( "\n" );
                }
            }
            DbgPrint( "\n" );
#endif
            bReceiveDone |= TRUE;

#ifndef NO_STATS
            if ( pHardware->m_bLookahead[ 0 ] == 0xFF )
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_FRAMES_RCV ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_BYTES_RCV ] +=
                        pHardware->m_nPacketLen;
            }
            else
            if ( pHardware->m_bLookahead[ 0 ] == 0x01 )
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_MULTICAST_FRAMES_RCV ]++;
            }
            else
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_UNICAST_FRAMES_RCV ]++;

                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_FRAMES_RCV ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_BYTES_RCV ] +=
                        pHardware->m_nPacketLen;
            }
#endif
            pHardware->m_cnCounter[ port ][ OID_COUNTER_RCV_OK ]++;
        }

next_packet:

        // Acknowledge the interrupt
        HardwareAcknowledgeReceive( pHardware );

        HardwareReadRegWord( pHardware, REG_RX_STATUS_BANK,
            REG_RX_STATUS_OFFSET, &wData );
    } while ( wData );

    return( bReceiveDone );
}  // ProcessReceive


/*
    MiniportHandleInterrupt

    Description:
        This routine is used by NDIS for deferred interrupt processing.  It
        reads from the Interrupt Status Register any outstanding interrupts and
        handles them.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

VOID MiniportHandleInterrupt (
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    USHORT        IntEnable;
    BOOLEAN       bReceiveDone = FALSE;
    int           port = MAIN_PORT;

#ifdef DEBUG_DRIVER_INTERRUPT
DbgPrint( "MiniportHandleInterrupt"NEWLINE );
#endif

#ifdef DEBUG_COUNTER
    pHardware->m_nGood[ COUNT_GOOD_INT ]++;
#endif
    HardwareReadInterrupt( pHardware, &IntEnable );
    while ( IntEnable )
    {

#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_INT_LOOP ]++;
#endif
        HardwareAcknowledgeInterrupt( pHardware, IntEnable );

#ifdef DEBUG_INTERRUPT
DbgPrint( " intr:%04x"NEWLINE, IntEnable );
#endif

        if ( ( IntEnable & INT_TX ) )
        {
            HardwareTransmitDone( pHardware );

            // Acknowledge the interrupt
            HardwareAcknowledgeTransmit( pHardware );

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_TX ]++;
#endif

#ifdef SEND_QUEUE
            SendNextPacket( pAdapter );
#endif
        }

        if ( ( IntEnable & INT_RX ) )
        {
            // Receive the packet
            bReceiveDone |= ProcessReceive( pAdapter );

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_RX ]++;
#endif

#ifdef EARLY_RECEIVE
            if ( ( IntEnable & INT_RX_EARLY ) )
            {
                // Acknowledge the interrupt
                HardwareAcknowledgeEarly( pHardware );
            }
#endif
        }

        if ( ( IntEnable & INT_PHY ) )
        {
            pHardware->m_bLinkIntWorking = TRUE;
            SwitchGetLinkStatus( pHardware );
            if ( pAdapter->m_ulNdisMediaState !=
                    pHardware->m_ulHardwareState )
            {
                NDIS_STATUS nsStatus;

                nsStatus = ( NdisMediaStateConnected ==
                    pHardware->m_ulHardwareState ) ?
                    NDIS_STATUS_MEDIA_CONNECT : NDIS_STATUS_MEDIA_DISCONNECT;

                NdisMIndicateStatus( pAdapter->m_hAdapter, nsStatus, NULL, 0 );
                NdisMIndicateStatusComplete( pAdapter->m_hAdapter );
                pAdapter->m_ulNdisMediaState = pHardware->m_ulHardwareState;
            }
            if ( ( pAdapter->m_ulDriverState & DRIVER_STATE_RESET )  &&
                    NdisMediaStateConnected == pAdapter->m_ulNdisMediaState )
            {
                pAdapter->m_ulDriverState &= ~DRIVER_STATE_RESET;
                NdisMResetComplete( pAdapter->m_hAdapter, NDIS_STATUS_SUCCESS,
                    FALSE );
            }
        }

#ifdef EARLY_RECEIVE
        if ( ( IntEnable & INT_RX_EARLY ) )
        {
            HardwareReceiveEarly( pHardware );

            // Acknowledge the interrupt
            HardwareAcknowledgeEarly( pHardware );

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_RX_EARLY ]++;
#endif
        }
#endif

        if ( ( IntEnable & INT_RX_OVERRUN ) )
        {

#ifdef EARLY_RECEIVE
            if ( pHardware->m_bReceiveDiscard )
                pHardware->m_bReceiveDiscard = FALSE;
            else
#endif
            pHardware->m_cnCounter[ port ][ OID_COUNTER_RCV_NO_BUFFER ]++;

            // Acknowledge the interrupt
            HardwareAcknowledgeOverrun( pHardware );

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_RX_OVERRUN ]++;
#endif
        }

#ifdef DBG
        if ( ( IntEnable & INT_RX_STOPPED ) )
        {
            USHORT wData;

            DBG_PRINT( "Rx stopped"NEWLINE );
            HardwareReadRegWord( pHardware, REG_RX_CTRL_BANK,
                REG_RX_CTRL_OFFSET, &wData );
            if ( ( wData & RX_CTRL_ENABLE ) )
            {
                DBG_PRINT( "Rx disabled"NEWLINE );
            }
            if ( ( pHardware->m_wReceiveConfig & RX_CTRL_ENABLE ) )
            {
                HW_WRITE_WORD( pHardware, REG_RX_CTRL_OFFSET, 0 );
                HW_WRITE_WORD( pHardware, REG_RX_CTRL_OFFSET,
                    pHardware->m_wReceiveConfig );
            }
            else
                pHardware->m_wInterruptMask &= ~INT_RX_STOPPED;
            HardwareAcknowledgeInterrupt( pHardware, INT_RX_STOPPED );
            break;
        }

        if ( ( IntEnable & INT_TX_STOPPED ) )
        {
            USHORT wData;

            DBG_PRINT( "Tx stopped"NEWLINE );
            HardwareReadRegWord( pHardware, REG_TX_CTRL_BANK,
                REG_TX_CTRL_OFFSET, &wData );
            if ( ( wData & TX_CTRL_ENABLE ) )
            {
                DBG_PRINT( "Tx disabled"NEWLINE );
            }
            if ( ( pHardware->m_wTransmitConfig & TX_CTRL_ENABLE ) )
            {
                HW_WRITE_WORD( pHardware, REG_TX_CTRL_OFFSET, 0 );
                HW_WRITE_WORD( pHardware, REG_TX_CTRL_OFFSET,
                    pHardware->m_wTransmitConfig );
            }
            else
                pHardware->m_wInterruptMask &= ~INT_TX_STOPPED;
            HardwareAcknowledgeInterrupt( pHardware, INT_TX_STOPPED );
            break;
        }
#endif
        HardwareReadInterrupt( pHardware, &IntEnable );
    }

    // Finally, indicate ReceiveComplete to all protocols which received
    // packets.
    if ( bReceiveDone )
    {
        // only allow received packets after Filter is defined.
        if ( pAdapter->m_bPacketFilterSet )
        {
#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_RCV_COMPLETE ]++;
#endif
            NdisMEthIndicateReceiveComplete( pAdapter->m_hAdapter );
        }
    }
}  // MiniportHandleInterrupt


/*
    MiniportISR

    Description:
        This routine is used by NDIS for the hardware to check if the interrupt
        is its own in the shared interrupt environment.

    Parameters:
        PBOOLEAN pbInterruptRecognized
            Buffer to indicate the interrupt is recognized

        PBOOLEAN pbQueueDpc
            Buffer to indicate the deferred interrupt processing should be
            scheduled.

        PVOID pContext
            Pointer to adapter information structure.

    Return (None):
*/

VOID MiniportISR (
    OUT PBOOLEAN pbInterruptRecognized,
    OUT PBOOLEAN pbQueueDpc,
    IN  PVOID    pContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) pContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    USHORT        wIntStatus;

    HardwareSaveBank( pHardware );
    HardwareReadInterrupt( pHardware, &wIntStatus );

    if ( wIntStatus )
    {
        HardwareDisableInterrupt( pHardware );

        *pbInterruptRecognized = TRUE;
        *pbQueueDpc = TRUE;
    }
    else
    {
        // not our interrupt!
        *pbInterruptRecognized = FALSE;
        *pbQueueDpc = FALSE;
    }

    HardwareRestoreBank( pHardware );
}  // MiniportISR

/* -------------------------------------------------------------------------- */

/*
    MiniportTransferData

    Description:
        A protocol driver calls the TransferData request (indirectly via
        NdisTransferData) from within its Receive event handler to instruct the
        driver to copy the contents of the received packet to a specified
        packet buffer.

    Parameters:
        PNDIS_PACKET pPacket
            Pointer to NDIS packet.

        PUINT puiBytesTransferred
            Number of bytes written into the packet buffer.  This value is not
            valid if the return status is STATUS_PENDING.

        NDIS_HANDLE hAdapterContext
            Handle to adapter context containing adapter information.

        NDIS_HANDLE hReceiveContext
            Handle to receive context passed by the driver on its call to
            NdisMEthIndicateReceive.  The driver can use this value to
            determine which packet, on which adapter, is being received.

        UINT uiBytesOffset
            A number specifying the offset within the received packet at which
            the copy is to begin.  If the entire packet is to be copied,
            uiByteOffset must be zero.

        UINT uiBytesToTransfer
            A number specifying the number of bytes to copy.  It is legal to
            transfer zero bytes; this has no effect.  If the sum of
            uiByteOffset and uiBytesToTransfer is greater than the size of the
            received packet, then the remainder of the packet (starting from
            uiByteOffset) is transferred, and the trailing portion of the
            receive buffer is not modified.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS MiniportTransferData (
    OUT PNDIS_PACKET pPacket,
    OUT PUINT        puiBytesTransferred,
    IN  NDIS_HANDLE  hAdapterContext,
    IN  NDIS_HANDLE  hReceiveContext,
    IN  UINT         uiByteOffset,
    IN  UINT         uiBytesToTransfer )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS   nsStatus;
    PNDIS_BUFFER  pndisBuffer;          // Current NDIS_BUFFER to copy into
    PUCHAR        pBuffer;              // Virtual address of the buffer
    UINT          uiBufferLength;       // Length of the buffer
    UINT          uiBufferOffset;       // Offset into the buffer
    UINT          uiBytesCopy;
    UINT          uiBytesLeft;
    UINT          uiLength;
    UINT          uiOffset;

#ifdef DBG
DbgPrint( "MiniportTransferData: %u, %u"NEWLINE, uiByteOffset,
    uiBytesToTransfer );
#endif

    // Take care of boundary condition of zero length copy.
    if ( !uiBytesToTransfer )
    {
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_XFER_ZERO ]++;
#endif
        return NDIS_STATUS_SUCCESS;
    }
    uiLength = pHardware->m_nPacketLen - ETHERNET_HEADER_SIZE;

    // See how much data there is to transfer.
    if ( uiByteOffset + uiBytesToTransfer > uiLength )
    {
        if ( uiByteOffset >= uiLength )
        {
#ifdef DEBUG_COUNTER
            pHardware->m_nBad[ COUNT_BAD_XFER_ZERO ]++;
#endif
            goto TransferDataError;
        }
        uiBytesToTransfer = uiLength - uiByteOffset;
    }

    // Set the number of bytes left to transfer.
    uiBytesLeft = uiBytesToTransfer;

    // Address on the adapter to copy from.
    uiOffset = ETHERNET_HEADER_SIZE + uiByteOffset;

    // Get location to copy into.
    NdisQueryPacket( pPacket, NULL, NULL, &pndisBuffer, NULL );
    NdisQueryBuffer( pndisBuffer, ( PVOID* ) &pBuffer, &uiBufferLength );
    uiBufferOffset = 0;

    // Loop, filling each buffer in the packet until there
    // are no more buffers or the data has all been copied.
    while ( uiBytesLeft > 0 )
    {
        // See how much data to read into this buffer.
        uiBytesCopy = uiBufferLength - uiBufferOffset;
        if ( uiBytesCopy > uiBytesLeft )
        {
            uiBytesCopy = uiBytesLeft;
        }

        NdisMoveMemory( pBuffer + uiBufferOffset,
            &pHardware->m_bLookahead[ uiOffset ], uiBytesCopy );

        // Update offsets and counts
        uiOffset += uiBytesCopy;
        uiBytesLeft -= uiBytesCopy;

        // Is the transfer done now?
        if ( !uiBytesLeft )
            break;

        // Was the end of this packet buffer reached?
        uiBufferOffset += uiBytesCopy;
        if ( uiBufferOffset == uiBufferLength )
        {
            NdisGetNextBuffer( pndisBuffer, &pndisBuffer );
            if ( !pndisBuffer )
                break;

            NdisQueryBuffer( pndisBuffer, ( PVOID* ) &pBuffer,
                &uiBufferLength );
            uiBufferOffset = 0;
        }
    }
    nsStatus = NDIS_STATUS_SUCCESS;
    *puiBytesTransferred = uiBytesToTransfer - uiBytesLeft;
    return( nsStatus );

TransferDataError:
    *puiBytesTransferred = 0;
    return NDIS_STATUS_FAILURE;
}  // MiniportTransferData

/* -------------------------------------------------------------------------- */

/*
    MiniportReturnPacket

    Description:
        This routine attempts to return to the receive free list the packet
        passed to us by NDIS.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

        PNDIS_PACKET pPacket
            Pointer to NDIS packet.

    Return (None):
*/

VOID MiniportReturnPacket (
    IN  NDIS_HANDLE  hAdapterContext,
        PNDIS_PACKET pPacket )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

#ifdef DBG
    DBG_PRINT( "MiniportReturnPacket"NEWLINE );
#endif
    NdisAcquireSpinLock( &pAdapter->m_lockAdapter );

    NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_SUCCESS );

    NdisReleaseSpinLock( &pAdapter->m_lockAdapter );
}  // MiniportReturnPacket
