/* ---------------------------------------------------------------------------
             Copyright (c) 2006-2007 Micrel, Inc.  All rights reserved.
------------------------------------------------------------------------------

ks_dma.c - KSZ88xx generic bus transmit/receive by platform host CPU's DMA functions.
------------------------------------------------------------------------------

Author  Date      Version  Description
PCD     04/08/05  1.0.5    Created.
-----------------------------------------------------------------------------

################## DMA DRIVER PARAMETERS #######################


dma_align_mask
    specifies the DMA transfer address alignment mask. 
    the dma bus_width must be configured as 32bit.

f_dma_rx
    0 = the driver will use PIO (CPU time) to move packet data 
        from KSZ88xxM to host memory.
    1 = the driver will use CPU dma function to move packet data 
        from KSZ88xxM to host memory.
    The configuration value is decided by the platform layer with the 
    macro PLATFORM_RX_DMA.
        
f_dma_tx
    0 = the driver will use PIO (CPU time) to move packet data 
        from host memory to KSZ88xxM.
    1 = the driver will use CPU dma function to move packet data 
        from host memory to KSZ88xxM.
    The configuration value is decided by the platform layer with the 
    macro PLATFORM_TX_DMA.

dma_rx_channel
    specifies the dma channel to use for receiving packets. 
    The configuration value is decided by the platform layer with the 
    macro PLATFORM_RX_DMA_CHAN.
    the driver will call the Platform_IsValidDmaChannel
    to varify an available dma channel.
        
dma_tx_channel
    specifies the dma channel to use for transmitting packets. 
    The configuration value is decided by the platform layer with the 
    macro PLATFORM_TX_DMA_CHAN.
    the driver will call the Platform_IsValidDmaChannel
    to varify an available dma channel.

dma_threshold
    specifies the minimum size a packet must be for using DMA. 
    Otherwise the driver will use PIO. For small packets PIO may
    be faster than DMA because DMA requires a certain amount of set
    up time where as PIO can start almost immediately. Setting this
    value to 0 means dma will always be used where dma has been enabled.

    The configuration value is decided by the platform layer with the
    macro PLATFORM_DMA_THRESHOLD
*/

#include "target.h"
#include "hardware.h"
#ifdef DEF_VXWORKS
#include "endLib.h"
#include "intLib.h"
#include "config.h"
#include "ks884xEnd_shBus.h"
#endif /* #ifdef DEF_VXWORKS */
#ifdef DEF_LINUX
#include "device.h"
#endif

/* define DMA driver default configuration parameters */

#define PLATFORM_RX_DMA          TRUE
#define PLATFORM_TX_DMA          TRUE
#define PLATFORM_RX_DMA_CHAN        0
#define PLATFORM_TX_DMA_CHAN        1
#define PLATFORM_DMA_THRESHOLD      0

/* Platform SH7751R DMAC (DMA controller) register addresses */

#define DMAC_SAR0_ADDR	((volatile ULONG *)0xffa00000UL)	/* src address 0 */
#define DMAC_DAR0_ADDR	((volatile ULONG *)0xffa00004UL)	/* dst address 0 */
#define DMAC_TCR0_ADDR	((volatile ULONG *)0xffa00008UL)	/* x'fer count 0 */
  #define CHCR_TCR0_MASK   0x00FFFFFF       /* Dma transfer count mask bits */

#define DMAC_CHCR0_ADDR	((volatile ULONG *)0xffa0000cUL)	/* channel ctrl 0 */
  #define CHCR_DA_INC      0x00004000       /* Dma destination address increase */
  #define CHCR_DA_FIX      0x00000000       /* Dma destination address fixed */
  #define CHCR_SA_INC      0x00001000       /* Dma source address increase */
  #define CHCR_SA_FIX      0x00000000       /* Dma source address fixed */
  #define CHCR_AUTO_REQ    0x00000400       /* Dma Auto-request (ext addr space - ext addr space */
  #define CHCR_BURST_MODE  0x00000080       /* Dma burst mode */
  #define CHCR_SIZE_8      0x00000010       /* Dma transfer data size - 8bit */
  #define CHCR_SIZE_16     0x00000020       /* Dma transfer data size - 16bit */
  #define CHCR_SIZE_32     0x00000030       /* Dma transfer data size - 32bit */
  #define CHCR_QCL         0x00000008       /* Dma request queue clear */
  #define CHCR_IE          0x00000004       /* Dma interrupt generate after transfer done */
  #define CHCR_TE          0x00000002       /* Dma transfer end (done) status */
  #define CHCR_DE          0x00000001       /* Dma channel enable */

  #define CHCR_DM_MASK     0x0000C000       /* Dma DA mask bits */
  #define CHCR_SM_MASK     0x00003000       /* Dma SA mask bits */
  #define CHCR_RS_MASK     0x00000F00       /* Dma RS mask bits */
  #define CHCR_TS_MASK     0x00000070       /* Dma TS mask bits */

#define DMAC_DMAOR_ADDR	((volatile ULONG *)0xffa00040)	/* op. register */
  #define DMAOR_DDT        0x00008000       /* Dma On-demand data transfer mode */
  #define DMAOR_CH0        0x00000000       /* Dma channel priority mode - ch0>ch1>ch2>ch3 */
  #define DMAOR_ROBIN      0x00000300       /* Dma channel priority mode - Round robin */
  #define DMAOR_AE         0x00000004       /* Dma controller AE error */
  #define DMAOR_NMIF       0x00000002       /* Dma controller NMIF error */
  #define DMAOR_DEM        0x00000001       /* Dma controller enabled */

  #define DMAOR_PR_MASK    0x00000300       /* Dma PR mask bits */
  #define DMAOR_ERROR      (DMAOR_AE | DMAOR_NMIF) 

#define DMA_BASE_ADDR		    0xffa00000UL
#define DMA_CHAN_BASE_ADDR(n)	(DMA_BASE_ADDR + (n * 0x10UL) )

#define DMA_CHAN_SAR(n)		(*(volatile unsigned long *)(DMA_CHAN_BASE_ADDR(n) + 0x0UL))
#define DMA_CHAN_DAR(n)		(*(volatile unsigned long *)(DMA_CHAN_BASE_ADDR(n) + 0x4UL))
#define DMA_CHAN_COUNT(n)	(*(volatile unsigned long *)(DMA_CHAN_BASE_ADDR(n) + 0x8UL))
#define DMA_CHAN_CHCR(n)	(*(volatile unsigned long *)(DMA_CHAN_BASE_ADDR(n) + 0xCUL))
#define DMA_DMAOR_STATUS    (*(volatile unsigned long *)(DMA_BASE_ADDR + 0x40UL))

#define DMA_CHAN_DISABLE(n)	                        \
{                                                   \
    ULONG  dmaRegValue;                             \
    dmaRegValue = (ULONG )(DMA_CHAN_CHCR(n) );      \
    dmaRegValue |= (CHCR_QCL);      \
    dmaRegValue &= ~(CHCR_IE | CHCR_TE | CHCR_DE);      \
    DMA_CHAN_CHCR(n) = dmaRegValue;      \
}

#define DMA_CHAN_ENABLE(n)	                        \
{                                                   \
    ULONG  dmaRegValue;                             \
    dmaRegValue = (ULONG )(DMA_CHAN_CHCR(n) );      \
    DMA_CHAN_CHCR(n) = dmaRegValue | (CHCR_IE | CHCR_DE );       \
}

#define DMA_CHAN_INT_DISABLE(n)	                        \
{                                                   \
    ULONG  dmaRegValue;                             \
    dmaRegValue = (ULONG )(DMA_CHAN_CHCR(n) );      \
    DMA_CHAN_CHCR(n) = dmaRegValue & ~CHCR_IE;      \
}

#define DMA_CHAN_INT_ENABLE(n)	                        \
{                                                   \
    ULONG  dmaRegValue;                             \
    dmaRegValue = (ULONG )(DMA_CHAN_CHCR(n) );      \
    DMA_CHAN_CHCR(n) = dmaRegValue | CHCR_IE;       \
}

#define DMA_CLEAR_ERROR 	                        \
{                                                   \
    DMA_DMAOR_STATUS = dma_oper_cntl_reg & ~DMAOR_ERROR;  \
}


/* Platform SH7751R INTC (Interrupt controller) register addresses */

#define INT_IPRAC_ADDR      ((volatile USHORT *)0xffd0000c)    /* Interrupt Priority register C */
  #define INT_DMA_LEVEL       0x0E00             /* DMA channel interrupt priority level - 14 */

#define INT_PRI_ADDR        ((volatile UINT32 *)0xfe080000)    /* Interrupt Priority level setting register 00 */
#define INT_REQ_ADDR        ((volatile UINT32 *)0xfe080020)    /* Interrupt Factor  register 00    */
#define INT_MSK_ADDR        ((volatile UINT32 *)0xfe080040)    /* Interrupt Mask  register 00      */
#define INT_MSKCLR_ADDR     ((volatile UINT32 *)0xfe080060)    /* Interrupt Mask Clear register 00 */
  #define INT_DMA0            0x00000010UL       /* DMA channel 0 interrupt bit */
  #define INT_DMA1            0x00000008UL       /* DMA channel 1 interrupt bit */
  #define INT_DMA2            0x00000004UL       /* DMA channel 2 interrupt bit */
  #define INT_DMA3            0x00000002UL       /* DMA channel 3 interrupt bit */


#define MAX_DMA_CHANNEL     4
ULONG  dma_chan_cntl_reg[MAX_DMA_CHANNEL];
ULONG  dma_oper_cntl_reg;


#define DMA_DEBUG
/*
 */
#ifdef DMA_DEBUG        /* if debugging DMA driver */
#    define DMA_DEBUG_OFF           0x000000
#    define DMA_DEBUG_XFER          0x000001
#    define DMA_DEBUG_INT           0x000002
#    define DMA_DEBUG_INIT          0x000004
#    define DMA_DEBUG_ERROR         0x800000
#    define DMA_DEBUG_ALL           0xffffff

#ifdef DEF_LINUX
#define LOCAL static
#endif
LOCAL int dmaDebug = ( DMA_DEBUG_ERROR /*| DMA_DEBUG_XFER | DMA_DEBUG_INT | DMA_DEBUG_INIT */ );


#define DMA_PRINT(FLG,X)                                                \
	if (dmaDebug & FLG) DBG_PRINT X;

#ifdef DEF_VXWORKS
#define DMA_LOG(FLG, X0, X1, X2, X3, X4, X5, X6)       \
    if (dmaDebug & FLG)                              \
        logMsg((char *) X0, (int)X1, (int)X2, (int)X3, \
                            (int)X4, (int)X5, (int)X6);
#else
#define DMA_LOG(DBG_SW, X0, X1, X2, X3, X4, X5, X6)
#endif /* DEF_VXWORKS */

#define DMA_ASSERT(condition)													\
	if(!(condition)) {																\
		DBG_PRINT("DMA_ASSERTION_FAILURE: File=" __FILE__ ", Line=%d\n",__LINE__);	\
		while(1);																	\
	}

#else /* DMA_DEBUG */

#define DMA_PRINT(FLG,X)                                                \
#define DMA_LOG(DBG_SW, X0, X1, X2, X3, X4, X5, X6)
#define DMA_ASSERT(condition)

#endif /* DRV_DEBUG */


/* OS dependent external function protocol */
extern void  Tx_DmaCompletionCallback ( int, int );
extern void  Rx_DmaCompletionCallback ( int, int );


/* Platform dependent function protocol */

void    Platform_DmaSetParameters ( PHARDWARE );
BOOLEAN Platform_DmaInitialize ( PHARDWARE );
BOOLEAN Platform_IsValidDmaChannel ( UCHAR );
BOOLEAN Platform_DmaDisable ( PHARDWARE, UCHAR );
ULONG   Platform_RequestDmaChannel( PHARDWARE );
void    Platform_ReleaseDmaChannel( PHARDWARE, UCHAR );
BOOLEAN Platform_DmaStartXfer ( PHARDWARE, UCHAR, ULONG, ULONG, ULONG, BOOLEAN );
ULONG   Platform_DmaGetDwCnt ( PHARDWARE, UCHAR );
BOOLEAN Platform_IsDmaComplete ( PHARDWARE, UCHAR );
BOOLEAN Platform_DmaComplete ( PHARDWARE, UCHAR );
void    Platform_DmaInterruptEnable ( UCHAR );
void    Platform_DmaInterruptClear ( UCHAR );

int     Platform_DmaMakeIrq ( PHARDWARE, int );
void    Platform_DmaTxISR ( void * );
void    Platform_DmaRxISR ( void * );



/* -------------------------------------------------------------------------- *
 *       Platform Dependent DMA APIs                                          *
 *                                                                            *
 *  The user has to modify the contents for their specify platform.           *
 *  The example here use Renesas SH7751R DMAC                                 *
 *  dma_rx_channel is 1                                                       *
 *  dma_tx_channel is 2                                                       *
 *  dma_threshold  is 128 byte                                                *
 * -------------------------------------------------------------------------- */


/********************************************************
FUNCTION: Platform_DmaSetParameters

Description:
    This routine configures DMA driver parameters.

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

RETURN VALUE:
	TRUE, on Success
	FALSE, on Failure
********************************************************/
void Platform_DmaSetParameters
(
    PHARDWARE pHardware
)
{
    PDMA_XFER pDmaInfo = &pHardware->dmaInfo;
    int i;

    pDmaInfo->f_dma_rx = PLATFORM_RX_DMA;
    pDmaInfo->f_dma_tx = PLATFORM_TX_DMA;

    pDmaInfo->dma_rx_channel = PLATFORM_RX_DMA_CHAN;
    pDmaInfo->dma_tx_channel = PLATFORM_TX_DMA_CHAN;

    pDmaInfo->dma_threshold = PLATFORM_DMA_THRESHOLD;

    pDmaInfo->dma_align_mask = 0x03;
    pDmaInfo->dma_tx.dma_txBuf = (void *)NULL;
    pDmaInfo->dma_rx.dma_rxBuf = (void *)NULL;

    for (i=0; i<MAX_DMA_CHANNEL; i++)
        dma_chan_cntl_reg[i] = 0;
    dma_oper_cntl_reg = 0;

    DMA_PRINT(DMA_DEBUG_INIT, 
     ("Platform_DmaSetParameters(): f_dma_rx=%d, f_dma_tx=%d, dma_rx_channel=%d, dma_tx_channel=%d, dma_threshold=%ld"NEWLINE, 
       pDmaInfo->f_dma_rx, pDmaInfo->f_dma_tx, pDmaInfo->dma_rx_channel, pDmaInfo->dma_tx_channel, 
       pDmaInfo->dma_threshold ));
}



/*********************************************************
FUNCTION: Platform_IsValidDmaChannel

DESCRIPTION:
    This function is used to test the validity of the 
    channels request with parameters dma_rx_channel,
    and dma_rx_channel

PARAMETERS:
	UCHAR dwDmaCh
	      the dma channel number to test for validity

RETURN VALUE:
	TRUE if the dma channel may be used
	FALSE if the dma channel may not be used
*********************************************************/

BOOLEAN Platform_IsValidDmaChannel
(
	UCHAR     dwDmaCh
)
{
	/* for SH7751R only use channels 0-3 */
    if((dwDmaCh < MAX_DMA_CHANNEL)) 
    {
        return TRUE;
    }
    return FALSE;
}


/********************************************************

FUNCTION: Platform_DmaInitialize

DESCRIPTION:
    This function is used to initialize platform DMAC by DMA channel 
    from dma_rx_channel, and dma_tx_channel.

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

RETURN VALUE:
	TRUE, on Success
	FALSE, on Failure
********************************************************/
BOOLEAN Platform_DmaInitialize
(
    PHARDWARE pHardware
)
{
    PDMA_XFER pDmaInfo = &pHardware->dmaInfo;
    ULONG  dmaRegAddr;
    ULONG  dmaRegValue;
#if 0
    STATUS status = ERROR;
#endif


    /* Configures DMA driver parameters */
    Platform_DmaSetParameters ( pHardware );

    /* Configure platform DMAC receive channel */
    if ( pDmaInfo->f_dma_rx )
    {
        /* Disable DMA channel */
        if ( !(Platform_DmaDisable(pHardware, pDmaInfo->dma_rx_channel)) )
            return (FALSE);

        /* Configure SH7751R DMA Channel Control Reg */

        dmaRegAddr = (ULONG)((ULONG)DMAC_CHCR0_ADDR + (ULONG)(pDmaInfo->dma_rx_channel * 0x10) );
        dmaRegValue = *(volatile ULONG *)(dmaRegAddr );             
        
        /* DMA destination address increased, source address fixed */
        dmaRegValue &= ~(CHCR_DM_MASK | CHCR_SM_MASK);
        dmaRegValue |= (CHCR_DA_INC | CHCR_SA_FIX)   /* (CHCR_DA_INC | CHCR_SA_INC) */;      

        /* DMA transfer request - Auto-request dual address mode 
           (external address space -> external address space) */
        dmaRegValue &= ~CHCR_RS_MASK;
        dmaRegValue |= CHCR_AUTO_REQ;      

        /* DMA bus mode - burst mode */
        dmaRegValue |= CHCR_BURST_MODE;      

        /* DMA transfer data size */
        dmaRegValue &= ~CHCR_TS_MASK;
        dmaRegValue |= CHCR_SIZE_32;      

        /* Write to DMA Channel Control Reg */
        *(volatile ULONG *)(dmaRegAddr ) = dmaRegValue;
        dma_chan_cntl_reg[pDmaInfo->dma_rx_channel] = dmaRegValue;

        DMA_PRINT(DMA_DEBUG_INIT, 
                 ("Platform_DmaInitialize() receive channel %d: dmaRegAddr=%08lx, dmaRegValue=%08lx"NEWLINE, 
                   pDmaInfo->dma_rx_channel, dmaRegAddr, *(volatile ULONG *)(dmaRegAddr ) ));


        /* Configure SH7751R DMA Transfer Count Reg */

        dmaRegAddr = (ULONG)((ULONG)DMAC_TCR0_ADDR + ((ULONG)pDmaInfo->dma_rx_channel * 0x10) );
        
        /* Clear DMA Transfer count to zero */
        *(volatile ULONG *)(dmaRegAddr ) = 0;

        DMA_PRINT(DMA_DEBUG_INIT, 
                 ("Platform_DmaInitialize() receive channel %d: dmaRegAddr=%08lx, dmaRegValue=%08lx"NEWLINE, 
                   pDmaInfo->dma_rx_channel, dmaRegAddr, *(volatile ULONG *)(dmaRegAddr ) ));

    }

    /* Configure platform DMAC transmit channel */
    if ( pDmaInfo->f_dma_tx )
    {
        /* Disable DMA channel */
        if ( !(Platform_DmaDisable(pHardware, pDmaInfo->dma_tx_channel)) )
            return (FALSE);

        /* Configure SH7751R DMA Channel Control Reg */

        dmaRegAddr = (ULONG)((ULONG)DMAC_CHCR0_ADDR + (ULONG)(pDmaInfo->dma_tx_channel * 0x10UL) );
        dmaRegValue = *(volatile ULONG *)(dmaRegAddr );             

        /* DMA destination address fix, source address increase */
        dmaRegValue &= ~(CHCR_DM_MASK | CHCR_SM_MASK);
        dmaRegValue |=  (CHCR_DA_FIX | CHCR_SA_INC) /* (CHCR_DA_INC | CHCR_SA_INC)  */;      

        /* DMA transfer request - Auto-request dual address mode 
           (external address space -> external address space) */
        dmaRegValue &= ~CHCR_RS_MASK;
        dmaRegValue |= CHCR_AUTO_REQ;      

        /* DMA bus mode - burst mode */
        dmaRegValue |= CHCR_BURST_MODE;      

        /* DMA transfer data size */
        dmaRegValue &= ~CHCR_TS_MASK;
        dmaRegValue |= CHCR_SIZE_32;      

        /* Write to DMA Channel Control Reg */
        *(volatile ULONG *)(dmaRegAddr ) = dmaRegValue;
        dma_chan_cntl_reg[pDmaInfo->dma_tx_channel] = dmaRegValue;

        DMA_PRINT(DMA_DEBUG_INIT, 
                 ("Platform_DmaInitialize() transmit channel %d: dmaRegAddr=%08lx, dmaRegValue=%08lx"NEWLINE, 
                   pDmaInfo->dma_tx_channel, dmaRegAddr, *(volatile ULONG *)(dmaRegAddr ) ));


        /* Configure SH7751R DMA Transfer Count Reg */

        dmaRegAddr = (ULONG)((ULONG)DMAC_TCR0_ADDR + (ULONG)(pDmaInfo->dma_tx_channel * 0x10) );
        
        /* Clear DMA Transfer count to zero */
        *(volatile ULONG *)(dmaRegAddr ) = 0;

        DMA_PRINT(DMA_DEBUG_INIT, 
                 ("Platform_DmaInitialize() transmit channel %ld: dmaRegAddr=%08lx, dmaRegValue=%08lx"NEWLINE, 
                   (ULONG)(pDmaInfo->dma_tx_channel * 0x10), dmaRegAddr, *(volatile ULONG *)(dmaRegAddr ) ));
       
    }

    if ( ( pDmaInfo->f_dma_rx ) || ( pDmaInfo->f_dma_tx ) )
    {
       /* Configure SH7751R DMA Operation Reg */

       dmaRegAddr = (ULONG)DMAC_DMAOR_ADDR;
       dmaRegValue = *(volatile ULONG *)(dmaRegAddr );             
        
       /* DMA transfer mode - Normal DMA mode */
       /* dmaRegValue &= ~DMAOR_DDT; */
       dmaRegValue |= DMAOR_DDT;

       /* DMA channel priority mode - Round robin */ 
       dmaRegValue &= ~DMAOR_PR_MASK;
       dmaRegValue |= DMAOR_CH0;      

       /* Enable DMA controller */
       dmaRegValue |= DMAOR_DEM;      

       /* Write to DMA Operation Reg */
       *(volatile ULONG *)(dmaRegAddr ) = dmaRegValue;
       dma_oper_cntl_reg = dmaRegValue; 

       DMA_PRINT(DMA_DEBUG_INIT, 
              ("Platform_DmaInitialize() operation reg: dmaRegAddr=%08lx, dmaRegValue=%08lx"NEWLINE, 
                dmaRegAddr, *(volatile ULONG *)(dmaRegAddr ) ));
    }

    return TRUE;
}


/********************************************************
FUNCTION: Platform_DmaDisable

DESCRIPTION:
    This function is used to disable DMA channel 'dwDmaCh'.

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

	ULONG dwDmaCh
	    the Dma channel to disable

RETURN VALUE:
	TRUE on success
	FALSE on failure
********************************************************/
BOOLEAN Platform_DmaDisable
(
    PHARDWARE pHardware,
	UCHAR dwDmaCh
)
{
    ULONG  dmaRegAddr;
    ULONG  dmaRegValue;

    DMA_ASSERT(Platform_IsValidDmaChannel(dwDmaCh))
 
    dmaRegAddr = (ULONG)((ULONG)DMAC_CHCR0_ADDR + (dwDmaCh * 0x10) );
    dmaRegValue = *(volatile ULONG *)(dmaRegAddr );             

    dmaRegValue &= ~CHCR_DE;
    *(volatile ULONG *)(dmaRegAddr ) = dmaRegValue;
 
	return TRUE;
}


/******************************************************
FUNCTION: Platform_RequestDmaChannel

DESCRIPTION:
	If the OS supports Dma Channel allocation then this function
	  will use that support to reserve a dma channel.
	If the OS does not support Dma Channel allocation, or a
	  channel can not be reserved then this function 
	  should nerver called.

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

RETURN VALUE:
	returns the DMA channel number that has been reserved

******************************************************/
ULONG Platform_RequestDmaChannel
(
    PHARDWARE pHardware
)
{
	/* Since Platform_RequestDmaChannel never returns a dma channel
	   in SH7751R platform, then this function should never be called */
	DMA_ASSERT(FALSE)
	return 0;
}


/*****************************************************
FUNCTION: Platform_ReleaseDmaChannel

DESCRIPTION:
	Releases the DMA channel specified by dwDmaChannel,
	  which was previously returned by Platform_RequestDmaChannel
    If the OS supports it this function will notify the OS
	  that the dma channel is free to be used by other devices

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

	DWORD dwDmaChannel
	    the Dma channel number to be released
******************************************************/
void Platform_ReleaseDmaChannel
(
    PHARDWARE pHardware,
	UCHAR dwDmaChannel
)
{
	/* Since Platform_RequestDmaChannel never returns a dma channel
	   in SH7751R platform, then this function should never be called */
	DMA_ASSERT(FALSE)
}


/*****************************************************
FUNCTION: Platform_DmaStartXfer

DESCRIPTION:
    Begins a new dma transfer,
	Should not be called if another transfer is in progress.
    Checking 'dwDmaCh' is right channel for read or write since 
    in SH7751R platform, we use seperate channels for transmit/receive.

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

	ULONG dwDmaCh
	    dma transfer at this channel.

	ULONG sourceAddr
	    source address of a DMA transfer.

	ULONG destAddr
	    destination address of a DMA transfer.

	ULONG length
	    bytes to transfer.

	BOOLEAN   fMemWr
        True, DMA transmiting transfer operation.

RETURN VALUE:
    TRUE if dma has begun the transfer
	FALSE for any error
******************************************************/
BOOLEAN Platform_DmaStartXfer
(
    PHARDWARE pHardware,
	UCHAR     dwDmaCh,
	ULONG     sourceAddr,
	ULONG     destAddr,
	ULONG     length,
	BOOLEAN   fMemWr
)
{
    PDMA_XFER pDmaInfo = &pHardware->dmaInfo;
#if 0
    ULONG  dmaRegAddr;
    ULONG  dmaRegValue;
#endif

    /* 1. validate the requested channel */

#if (0)
    DMA_PRINT(DMA_DEBUG_XFER, 
	          ("Platform_DmaStartXfer -- fMemWr=%d,dwDmaCh=%d,length=%ld,sourceAddr=%08lx,destAddr=%08lx "NEWLINE, 
                 fMemWr, dwDmaCh, length, sourceAddr, destAddr));
#endif

    if ( fMemWr ) 
    {
   
        DMA_ASSERT(dwDmaCh==pDmaInfo->dma_tx_channel)
    }
    else
    {
        DMA_ASSERT(dwDmaCh==pDmaInfo->dma_rx_channel)
    }

    /* 2. make sure the channel's not already running */

    if ( DMA_CHAN_COUNT(dwDmaCh) != 0UL )
    {
        DMA_PRINT(DMA_DEBUG_ERROR, 
		          ("Platform_DmaStartXfer -- requested channel (%d) is still running"NEWLINE, dwDmaCh));
        return FALSE;
    }

    /* 3. calculate the physical transfer addresses */


    /* 4. validate the address alignments */

	if ((sourceAddr & pDmaInfo->dma_align_mask) != 0UL)
	{
        DMA_PRINT(DMA_DEBUG_ERROR, 
		          ("Platform_DmaStartXfer -- bad sourceAddr (0x%08lX) alignment"NEWLINE, sourceAddr));
		return FALSE;
	}

	if ((destAddr & pDmaInfo->dma_align_mask) != 0UL)
	{
        DMA_PRINT(DMA_DEBUG_ERROR, 
		          ("Platform_DmaStartXfer -- bad destAddr (0x%08lX) alignment"NEWLINE, destAddr));
		return FALSE;
	}

	/* 6. Config Control reg */

	/* Disable the selected channel first */	
     DMA_CHAN_CHCR(dwDmaCh) = (ULONG)(dma_chan_cntl_reg[dwDmaCh] & ~(CHCR_IE | CHCR_TE | CHCR_DE) ) ;
    /*DMA_CHAN_DISABLE(dwDmaCh); */

	/* Set Source and destination addresses */
	DMA_CHAN_SAR(dwDmaCh) = sourceAddr;
	DMA_CHAN_DAR(dwDmaCh) = destAddr;

	/* Set the transmit size in terms of the xfer mode */
	DMA_CHAN_COUNT(dwDmaCh) = (length >> 2) & CHCR_TCR0_MASK;

#if (0)
    DMA_PRINT(DMA_DEBUG_XFER, 
             ("Platform_DmaStartXfer() SAR %08lX:%08lx "NEWLINE, 
             ((ULONG)DMAC_SAR0_ADDR + (dwDmaCh * 0x10) ), DMA_CHAN_SAR(dwDmaCh) ));

    DMA_PRINT(DMA_DEBUG_XFER, 
             ("Platform_DmaStartXfer() DAR %08lX:%08lx "NEWLINE, 
              ((ULONG)DMAC_DAR0_ADDR + (dwDmaCh * 0x10) ), DMA_CHAN_DAR(dwDmaCh) ));

    DMA_PRINT(DMA_DEBUG_XFER, 
             ("Platform_DmaStartXfer() COUNT %08lX:%08lx "NEWLINE, 
              ((ULONG)DMAC_TCR0_ADDR + (dwDmaCh * 0x10) ), DMA_CHAN_COUNT(dwDmaCh) ));

    DMA_PRINT(DMA_DEBUG_XFER, 
             ("Platform_DmaStartXfer() DMAC_CHCR0_ADDR: %08lX:%08lx "NEWLINE, 
              ((ULONG)DMAC_CHCR0_ADDR + (dwDmaCh * 0x10) ), DMA_CHAN_CHCR(dwDmaCh) ));
#endif

	/* Enable DMA controller ch x */
#ifndef DEF_LINUX
    DMA_CHAN_CHCR(dwDmaCh) = (ULONG)(dma_chan_cntl_reg[dwDmaCh] | (/*CHCR_IE |*/ CHCR_DE) ) ;
#else
    DMA_CHAN_CHCR(dwDmaCh) = (ULONG)(dma_chan_cntl_reg[dwDmaCh] | (CHCR_IE | CHCR_DE) ) ;
#endif
	/* DMA_CHAN_ENABLE(dwDmaCh); */

	/* DMA Transfering.... */
	return TRUE;
}

/*******************************************************
FUNCTION: Platform_DmaGetDwCnt

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

	UCHAR dwDmaCh
	    dma transfer at this channel.

RETURN VALUE:
    0, if the DMA channel is ready to handle another 
	   request from Platform_DmaStartXfer
   non zero, is the number of DWORDS left for the 
       dma channel to transfer
*******************************************************/
ULONG Platform_DmaGetDwCnt
(
    PHARDWARE pHardware,
	UCHAR dwDmaCh
)
{
	ULONG  dwCount;

    DMA_ASSERT(Platform_IsValidDmaChannel(dwDmaCh))
    dwCount = DMA_CHAN_COUNT(dwDmaCh);
    
    return (dwCount);
}


/******************************************************
FUNCTION: Platform_IsDmaComplete

DESCRIPTION:
	Check if the specified dma channel to finish 
	transfering data. 

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

	UCHAR dwDmaCh
	    dma transfer at this channel.

RETURN VALUE:
    TRUE if dma has finished the transfer
	FALSE for any error

******************************************************/
BOOLEAN Platform_IsDmaComplete
(
    PHARDWARE pHardware,
	UCHAR dwDmaCh
)
{
    BOOLEAN status = TRUE;

    /* DMA_ASSERT(Platform_IsValidDmaChannel(dwDmaCh)) */

	/* if channel is disable */
	if ((DMA_CHAN_CHCR(dwDmaCh) & CHCR_DE) == 0UL)
         status = FALSE;
    
    /* if transfer counter is zero */
    if ( DMA_CHAN_COUNT(dwDmaCh) != 0UL )
         status = FALSE;
 
    /* if transfer without any error */
    if ( (DMA_DMAOR_STATUS & DMAOR_ERROR) != 0UL) 
         status = FALSE;

    if (status )
    {
       /* Disable the selected channel */
       DMA_CHAN_CHCR(dwDmaCh) = (ULONG)(dma_chan_cntl_reg[dwDmaCh] & ~(CHCR_IE | CHCR_TE | CHCR_DE) ) ;

       /* Clear the DMA error bits */
       DMA_CLEAR_ERROR;
    }
    return (status);
}


/******************************************************
FUNCTION: Platform_DmaComplete

DESCRIPTION:
	Waits for the specified dma channel to finish 
	transfering data. Upon return the dma channel should
	be ready to handle another request from 
	Platform_DmaStartXfer
	This function should be the same as waiting for 
	Platform_DmaGetDwCnt to return 0

PARAMETERS:
    PHARDWARE pHardware
        Pointer to hardware instance.

	ULONG dwDmaCh
	    dma transfer at this channel.

******************************************************/
BOOLEAN Platform_DmaComplete
(
    PHARDWARE pHardware,
	UCHAR dwDmaCh
)
{
    ULONG dwTimeOut = 1 /*1000000 */;

    /* DMA_ASSERT(Platform_IsValidDmaChannel(dwDmaCh)) */

	/* if channel is disable */
	if ((DMA_CHAN_CHCR(dwDmaCh) & CHCR_DE) == 0UL)
    {
         DMA_PRINT(DMA_DEBUG_ERROR, 
                  ("Platform_DmaComplete: the DMA channel %d is not enabled"NEWLINE, dwDmaCh ));
         return (FALSE);
    }

    if ( (DMA_DMAOR_STATUS & DMAOR_ERROR) != 0UL) 
    {
        DMA_PRINT(DMA_DEBUG_ERROR, 
                  ("Platform_DmaComplete: the transfer AE or NMIF error %08lx."NEWLINE, 
                    DMA_DMAOR_STATUS ));
        return (FALSE);
	}

	while( (dwTimeOut) && ( DMA_CHAN_COUNT(dwDmaCh) != 0UL) )
	{
		DelayMicrosec(1);
		dwTimeOut--;
	}

    /* Disable the selected channel */
    DMA_CHAN_CHCR(dwDmaCh) = (ULONG)(dma_chan_cntl_reg[dwDmaCh] & ~(CHCR_IE | CHCR_TE | CHCR_DE) ) ;

    /* Clear the DMA error bits */
    DMA_CLEAR_ERROR;

    if(dwTimeOut == 0)
    {
        DMA_PRINT(DMA_DEBUG_ERROR, 
		          ("Platform_DmaComplete: Timed out"NEWLINE));
         return (FALSE);
    }
    return (TRUE);
}



void Platform_DmaInterruptEnable
(
	UCHAR dwDmaCh
)
{
    ULONG   int_bit=0x00000010UL;
    ULONG   int_value;
    USHORT  wValue;

    Platform_DmaInterruptClear( dwDmaCh);

    /* Priority platform Interrupt Controller DMAC interrupt */
/*
    int_value = *(ULONG *)INT_PRI_ADDR;    
    int_value |= 0x00000F00UL;
    *INT_PRI_ADDR = int_value;

    DMA_PRINT(DMA_DEBUG_INT, 
              ("Platform_DmaInterruptEnable: INT_PRI_ADDR 0x%08X:%08x."NEWLINE, 
                    INT_PRI_ADDR, int_value ));
*/

    /* Accept DMAC channel interrupt */
    int_value = *(ULONG *)INT_MSK_ADDR;  
    int_bit >>= dwDmaCh;
    int_value &= ~int_bit; 
    *INT_MSK_ADDR = int_value;

    DMA_PRINT(DMA_DEBUG_INT, 
              ("Platform_DmaInterruptEnable: INT_MSK_ADDR 0x%08X:%08lx."NEWLINE, 
                    (int) INT_MSK_ADDR, int_value ));

    /* Set Priority level for DMAC interrupt */
    wValue = *(USHORT *)INT_IPRAC_ADDR;    
    wValue |= INT_DMA_LEVEL;
    *INT_IPRAC_ADDR = wValue;

    DMA_PRINT(DMA_DEBUG_INT, 
              ("Platform_DmaInterruptEnable: INT_IPRAC_ADDR 0x%08X:%04x."NEWLINE, 
                    (int) INT_IPRAC_ADDR, wValue ));

    /* Enable DMA channel interrupt IE */
    /* DMA_CHAN_INT_ENABLE(dwDmaCh); */
}

void Platform_DmaInterruptClear
(
	UCHAR dwDmaCh
)
{
    ULONG  int_bit=0x00000010UL;
    ULONG  int_value;

    /* Clear platform Interrupt Controller DMAC interrupt */
    int_value = *(ULONG *)INT_MSKCLR_ADDR;    
    int_bit >>= dwDmaCh;
    int_value |= int_bit; 
    *INT_MSKCLR_ADDR = int_value;
}

/*---------------------------------------------------------------------------*/

/******************************************************
FUNCTION: Platform_DmaTxISR

DESCRIPTION:
     Connect DMA Transmit Channel ISR to interrupt vector table.

PARAMETERS:
     int parameter
          Pointer to network device.

******************************************************/
int Platform_DmaMakeIrq
(
   PHARDWARE pHardware,
   int parameter
)
{
    PDMA_XFER pDmaInfo = &pHardware->dmaInfo;
    int  rc = 0;

    DMA_PRINT(DMA_DEBUG_INT, 
             ("Platform_DmaMakeIrq: is called."NEWLINE));

    if ( pDmaInfo->f_dma_rx )
    {
#ifdef DEF_VXWORKS
    {
#if (0)
        DRV_CTRL *pDrvCtrl= (DRV_CTRL *)parameter;  /* pointer to DRV_CTRL structure */

        rc = intConnect((VOIDFUNCPTR *)INUM_TO_IVEC(INUM_DMAC_DMTE0 /* 50 */),
                                 Platform_DmaRxISR, (int)(pDrvCtrl));

       if ( rc != 0)
          DMA_PRINT(DMA_DEBUG_INIT, 
                   ("Platform_DmaMakeIrq() connect DMA Platform_DmaRxISR() fail"NEWLINE ));
#endif

    }
#endif /* #ifdef DEF_VXWORKS */

#ifdef DEF_LINUX
#endif /* #ifdef DEF_LINUX */
    }

    if ( pDmaInfo->f_dma_tx )
    {
#ifdef DEF_VXWORKS
    {
#if (0)
        DRV_CTRL *pDrvCtrl= (DRV_CTRL *)parameter;  /* pointer to DRV_CTRL structure */

        rc = intConnect((VOIDFUNCPTR *)INUM_TO_IVEC(INUM_DMAC_DMTE1 /* 51 */),
                                 Platform_DmaTxISR, (int)(pDrvCtrl));
       if ( rc != 0)
          DMA_PRINT(DMA_DEBUG_INIT, 
                   ("Platform_DmaMakeIrq() connect DMA Platform_DmaTxISR() fail"NEWLINE ));
#endif

    }
#endif /* #ifdef DEF_VXWORKS */

#ifdef DEF_LINUX
#endif /* #ifdef DEF_LINUX */
    }

   return (rc);
}


/******************************************************
FUNCTION: Platform_DmaTxISR

DESCRIPTION:
     This routine is called at interrupt level in response 
     to an interrupt from the DMA transmit channel controller.

PARAMETERS:
      void* dev_id
            Pointer to network device.

******************************************************/
void Platform_DmaTxISR
(
    void *dev_id
)
{
#ifdef DEF_VXWORKS
    DRV_CTRL *pDrvCtrl= (DRV_CTRL *)dev_id;  /* pointer to DRV_CTRL structure */
    PHARDWARE  pHardware = pDrvCtrl->pHW;
#endif
#ifdef DEF_LINUX
    struct net_device* dev = ( struct net_device* ) dev_id;
    struct dev_priv*   priv = ( struct dev_priv* ) dev->priv;
    struct dev_info*   hw_priv = priv->pDevInfo;
    PHARDWARE          pHardware = &hw_priv->hw;
#endif

    PDMA_XFER pDmaInfo = &pHardware->dmaInfo;
    PDMA_TX   pDmaTxInfo = &pDmaInfo->dma_tx;
	UCHAR     dwDmaCh    = pDmaInfo->dma_tx_channel;


    DMA_PRINT(DMA_DEBUG_INT, 
             ("Platform_DmaTxISR: is called."NEWLINE));

    if ((DMA_CHAN_CHCR(dwDmaCh) & CHCR_DE) == 0UL)
    {
         DMA_PRINT(DMA_DEBUG_ERROR, 
                  ("Platform_DmaTxISR: the DMA channel %d is not enabled"NEWLINE, 
                    dwDmaCh ));
    }

    if ( DMA_CHAN_COUNT(dwDmaCh) != 0UL )
	{
         DMA_PRINT(DMA_DEBUG_ERROR, 
                  ("Platform_DmaTxISR: the transfer count %ld is not zero."NEWLINE, 
                    DMA_CHAN_COUNT(dwDmaCh) ));
	}

    if ( (DMA_DMAOR_STATUS & DMAOR_ERROR) != 0UL) 
    {
         DMA_PRINT(DMA_DEBUG_ERROR, 
                  ("Platform_DmaTxISR: the transfer AE or NMIF error %08lx."NEWLINE, 
                    DMA_DMAOR_STATUS ));
	}

    /* Clear the error bits */
    DMA_CLEAR_ERROR;

    /* Clear platform Interrupt Controller DMAC interrupt */
    Platform_DmaInterruptClear ( dwDmaCh ); 

    /* Disable DMA channel */
    DMA_CHAN_CHCR(dwDmaCh) = (ULONG)(dma_chan_cntl_reg[dwDmaCh] & ~(CHCR_IE | CHCR_TE | CHCR_DE) ) ;

    /* Free transmit data buffer */
    if ( pDmaTxInfo->dma_txBuf)
    {
#ifdef DEF_VXWORKS

#if (0)
       netJobAdd ((FUNCPTR)Tx_DmaCompletionCallback,
		          (int) pDrvCtrl, (int)pDmaTxInfo->dma_txBuf, 0, 0, 0);
#endif
#endif /* #ifdef DEF_VXWORKS */

#ifdef DEF_LINUX
#endif /* #ifdef DEF_VXWORKS */

    } /* if ( pDmaInfo->dma_txBuf) */

}


/******************************************************
FUNCTION: Platform_DmaTxISR

DESCRIPTION:
     This routine is called at interrupt level in response 
     to an interrupt from the DMA receive channel controller.

PARAMETERS:
      void* dev_id
            Pointer to network device.

******************************************************/
void Platform_DmaRxISR
(
    void *dev_id
)
{
#ifdef DEF_VXWORKS
    DRV_CTRL *         pDrvCtrl= (DRV_CTRL *)dev_id;  /* pointer to DRV_CTRL structure */
    PHARDWARE          pHardware = pDrvCtrl->pHW;
#endif
#ifdef DEF_LINUX
    struct net_device* dev = ( struct net_device* ) dev_id;
    struct dev_priv*   priv = ( struct dev_priv* ) dev->priv;
    struct dev_info*   hw_priv = priv->pDevInfo;
    PHARDWARE          pHardware = &hw_priv->hw;
#endif
    PDMA_XFER pDmaInfo   = &pHardware->dmaInfo;
    PDMA_RX   pDmaRxInfo = &pDmaInfo->dma_rx;
    UCHAR     dwDmaCh    = pDmaInfo->dma_rx_channel;


    DMA_PRINT(DMA_DEBUG_INT, 
             ("Platform_DmaRxISR: is called."NEWLINE  ));

    if ((DMA_CHAN_CHCR(dwDmaCh) & CHCR_DE) == 0UL)
    {
        DMA_PRINT(DMA_DEBUG_ERROR, 
                 ("Platform_DmaRxISR: the DMA channel %d is not enabled"NEWLINE, 
                    dwDmaCh ));
    }

    if ( DMA_CHAN_COUNT(dwDmaCh) != 0UL )
	{
        DMA_PRINT(DMA_DEBUG_ERROR, 
                 ("Platform_DmaRxISR: the transfer count %ld is not zero."NEWLINE, 
                    DMA_CHAN_COUNT(dwDmaCh) ));
	}

    if ( (DMA_DMAOR_STATUS & DMAOR_ERROR) != 0UL) 
    {
        DMA_PRINT(DMA_DEBUG_ERROR, 
                  ("Platform_DmaRxISR: the transfer AE or NMIF error %08lx."NEWLINE, 
                    DMA_DMAOR_STATUS ));
	}

    /* Clear the error bits */
    DMA_CLEAR_ERROR;

    /* Clear platform Interrupt Controller DMAC interrupt */
    Platform_DmaInterruptClear ( dwDmaCh ); 

    /* Disable DMA channel */
    DMA_CHAN_CHCR(dwDmaCh) = (ULONG)(dma_chan_cntl_reg[dwDmaCh] & ~(CHCR_IE | CHCR_TE | CHCR_DE) ) ;

    /* Pass recive data buffer to upper layer */
    if ( pDmaRxInfo->dma_rxBuf)
    {
#ifdef DEF_VXWORKS
#if (0)
       netJobAdd ((FUNCPTR)Rx_DmaCompletionCallback,
		          (int) pDrvCtrl, (int)pDmaRxInfo->dma_rxBuf, 0, 0, 0);
#endif

#if (0)
       Rx_DmaCompletionCallback (
		          (int) pDrvCtrl, (int)pDmaRxInfo->dma_rxBuf);
#endif
#endif /* #ifdef DEF_VXWORKS */

#ifdef DEF_LINUX
#endif /* #ifdef DEF_LINUX */

       DMA_PRINT(DMA_DEBUG_INT, 
             ("Platform_DmaRxISR: pass dma_rxBuf %p to uplayer."NEWLINE, pDmaRxInfo->dma_rxBuf ));
    }

}


#if (0)

static void Tx_DmaCompletionCallback
(   void *param
)
{
	PPRIVATE_DATA privateData = param;

	BUG_ON(privateData->TxSkb == NULL);
	dev_kfree_skb(privateData->TxSkb);
	privateData->TxSkb = NULL;
	Tx_WakeQueue(privateData,0x04UL);
}

static void Rx_DmaCompletionCallback
(
    void* param
)
{
	PPRIVATE_DATA privateData = param;

	BUG_ON(privateData->RxSkb == NULL);
	Rx_HandOffSkb(privateData,privateData->RxSkb);
	privateData->RxSkb = NULL;
	Lan_EnableInterrupt(privateData,privateData->RxInterrupts);
}
#endif
