<?php
/*
	SMG WebGUI

	Software distributed under the License is distributed on an "AS IS" basis,
	WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
	for the specific language governing rights and limitations under the
	License.
	
	Licence information can be found within the file LICENCE.php
	
	The Original Code is SMG WebGUI

	The Initial Developer of the Original Code is
	Jonathan Maurice <jmaurice@sangoma.com>

	Contributor(s):
*/


///////////////////////////////////////////////////////////////////////////////
//
// BgProcess class
//
///////////////////////////////////////////////////////////////////////////////


class BgProcess
{
  private $cmd;
  private $args;
  private $processName;
  private $workDir;
  private $installDir;
  private $logFile;
  private $pidFile;
  private $pid;

 /*  
  * @param string $in_cmd command to excecute (or path to command)
  * @param string $in_args command arguments
  * @param string $in_processName name of the process. If null, the basename of
  * $in_cmd will be use. The process name is what is use to keep track of the 
  * background process by reading the corresponding log and pid file. Bottom 
  * line, this must be the same name for each process that are the same. If you 
  * want to run 2 of the same command (at the same time) you must supply 
  * different process name of it will fail.
  * @param string $in_workDir directory where the command must be launched.
  */
  public function __construct($in_cmd, $in_args = null, $in_processName = null, $in_workDir = null)
  {
    $this->cmd = $in_cmd; 
    $this->args = $args; 

    if ($in_processName === null){
      $this->processName = pathinfo($in_processName, PATHINFO_FILENAME);
    }
    else {
      $this->processName = $in_processName;
    }

    $this->workDir = $in_workDir;
    $this->installDir = $this->getInstallDir();
    $this->logFile = $this->installDir . "/log/". $this->processName;
    $this->pidFile = $this->installDir . "/run/". $this->processName .".pid";
    $this->pid = null;
  }

  public function run()
  {
    $cmd = "nohup " . $this->cmd;
    if($this->args !== null){
      $cmd .= " " . $this->args;
    }
    $cmd .= " > " . $this->logFile . " 2>&1 & echo $! > " . $this->pidFile;

    if ($this->workDir === null){
      exec($cmd);
    }
    else {
      $cwd = getcwd();
      chdir($this->workDir);
      exec($cmd);
      chdir($cwd);
    }

    $this->pid = $this->getPid();
  }

  public function getArgs()
  {
    return $this->args;
  }

  public function setArgs($in_args)
  {
    $this->args = $in_args;
  }

  public function isRunning()
  {
    if($this->pid === null){
      $pid = $this->getPid();
    }
    else {
      $pid = $this->pid;
    }

    if ($pid === null){
      return false;
    }

    $running = posix_kill($pid, 0);

    if ($running){
      return true;
    }
    else {
      $this->deletePidFile();
      return false;
    }
  }

  public function killProcessAndChilds($pid,$signal) { 
        exec("ps -ef| awk '\$3 == '$pid' { print  \$2 }'", $output, $ret); 
        if($ret) return 'you need ps, grep, and awk'; 
        while(list(,$t) = each($output)) { 
            if ( $t != $pid ) { 
                $this->killProcessAndChilds($t,$signal); 
            } 
        } 
        //echo "killing ".$pid."\n"; 
        $kill = posix_kill($pid, 9); 

        return $kill;
    }

  public function getLog()
  {
    if(!file_exists($this->logFile)){
      return null;
    }

    return file($this->logFile);
  }

  public function abort($in_waitTime = 0)
  {
    if($this->pid === null){
      $pid = $this->getPid();
    }
    else {
      $pid = $this->pid;
    }

    if ($pid === null){
      return true;
    }

    //$killed = posix_kill($pid, 9);
    $killed = $this->killProcessAndChilds($pid,9);


    if($killed){
      sleep(1);
      return $this->waitForAbort($in_waitTime);
    }
    else {
      return false;
    }
  }

  public function getInstallDir()
  {
    $cwd = getcwd();
    $match = array();

    preg_match("/(.*?)\/nginx.*/", $cwd, $match);
    return $match[1]. "/nginx";
  }

  private function waitForAbort($in_time)
  {
    $loop = 0;
    while ($this->isRunning() && $loop <= $in_time) {
      sleep(1);
      $loop++;
    }

    return !$this->isRunning();
  }

  private function deletePidFile()
  {
    if(!file_exists($this->pidFile)){
      return true;
    }
    return unlink($this->pidFile);
  }

  private function getPid()
  {
    if(!file_exists($this->pidFile)){
      return null;
    }
    $content = file_get_contents($this->pidFile);

    $pid = null;
    $match = array();

    if(preg_match("/(\d+)/", $content, $match)){
      $pid = $match[1];
    }

    return trim($pid);
  }

}

?>
