/*
 * Sangoma Transcoder Library
 *
 * Nenad Corbic <ncorbic@sangoma.com>
 *
 * Copyright (C) 2010 Sangoma Technologies
 *
 * Redistribution and use is permitted only in binary forms without
 * modification and provided that the following conditions are met:
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * * Neither the name of the original author; nor the names of any contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Contributors:
 *
 * Moises Silva <moy@sangoma.com>
 *
 */
#ifndef _LIB_SANGOMA_TRANSCODE_UTILS_
#define _LIB_SANGOMA_TRANSCODE_UTILS_

#include "sng_tc_if.h"

#define MAX_CODEC_ID_SZ 127

/*!
  \enum sngtc_iana_code_types
  \brief IANA Transcoding types
  \note http://www.iana.org/assignments/rtp-parameters
*/  
enum sngtc_iana_code_types {
	IANA_PCMU_A_8000_1 	= 0,
	IANA_RESERVED_1		= 1,
	IANA_RESERVED_2		= 2,
	IANA_GSM_A_8000_1	= 3,
	IANA_G723_A_8000_1	= 4,
	IANA_DVI4_A_8000_1 	= 5,
	IANA_DIV4_A_16000_1	= 6,
	IANA_LPC_A_8000_1	= 7,
	IANA_PCMA_A_8000_1	= 8,
	IANA_G722_A_8000_1	= 9,
	/* FIXME: Asterisk incorrectly 
	 * uses payload 10 for L16 one channel. However
	 * we're at fault too because our mapping function
	 * is not working, sngtc_set_iana_code_based_on_codec_id
	 * should go over soap, or we should now include also
	 * the desired IANA code in each codec request.
	 * I am setting the IANA for both L16 codecs to 10
	 * since we do not support 2 channels anyways */
	IANA_L16_A_8000_2 	= 10,
	IANA_L16_A_8000_1	= 10,
	IANA_QCELP_A_8000_1	= 12,
	IANA_CN_A_8000_1	= 13,
	IANA_MPA_A_90000_1	= 14,
	IANA_G728_A_8000_1	= 15,
	IANA_DVI4_A_11025_1 = 16,
	IANA_DVI4_A_22050_1 = 17,
	IANA_G729_AB_8000_1  = 18,
	IANA_G729_A_8000_1  = 19,
	IANA_RESERVED_19	= 19,
	IANA_UNASSIGED_20	= 20,
	IANA_UNASSIGND_21	= 21,
	IANA_UNASSIGED_22	= 22,
	IANA_UNASSIGND_23	= 23,
	IANA_UNASSIGED_24	= 24,
	IANA_CELB_V_90000	= 25,
	IANA_JPEG_V_90000	= 26,
	IANA_UNASSIGED_27	= 27,
	IANA_NV_28			= 28,
	IANA_UNASSIGED_28	= 28,
	IANA_UNASSIGED_29	= 30,
	IANA_H261_V_90000   = 31,
	IANA_MPV_V_90000    = 32,
	IANA_MP2T_AV_90000  = 33,
	IANA_H263_V_90000   = 34,
	IANA_G726_16_A_8000_1	= 85,

	IANA_AMR_A_8000_1   = 96,   /* Works */
	IANA_ILBC_133_8000_1	= 97,
	IANA_ILBC_152_8000_1	= 98,
	
	IANA_G726_40_8000_1	= 121,  	/* Does not work - does not crash */
	IANA_G726_24_8000_1	= 123,   /* This is wrong */ /* Does not work - crashes vocallo */
	IANA_G726_16_8000_1	= 124,	/* Does not work - crashes vocallo */
	

	IANA_AMR_WB_16000_1 = 100,   /* ??? */

	IANA_SIREN7             = 102,
	IANA_SIREN14            = 115,

	IANA_EFR_A_8000_1	= 102,

	IANA_EVRC_B_8000_1	= 103,
	IANA_EVRC_8000_1	= 104,
	IANA_G726_32_8000_1 = 111,  /* Works */
	IANA_L16_A_16000_1    = 118,
};

/*!
  \enum SNGTC_LOGLEVEL_DEBUG
  \enum SNGTC_LOGLEVEL_WARN
  \enum SNGTC_LOGLEVEL_INFO
  \enum SNGTC_LOGLEVEL_STATS
  \enum SNGTC_LOGLEVEL_ERROR
  \enum SNGTC_LOGLEVEL_CRIT
 */
enum {
	SNGTC_LOGLEVEL_DEBUG=1,
	SNGTC_LOGLEVEL_STATS,
	SNGTC_LOGLEVEL_WARN,
	SNGTC_LOGLEVEL_INFO,
	SNGTC_LOGLEVEL_ERROR,
	SNGTC_LOGLEVEL_CRIT,
	SNGTC_LOGLEVEL_PUBLIC=100,
};

#if 1

extern sngtc_log_func_t _SNGTC_API sngtc_log_func;

#define sngtc_print(level,fmt,...) if (sngtc_log_func) sngtc_log_func(level,fmt,##__VA_ARGS__);
#define SNGTC_DEBUG(a,...) 		sngtc_print(SNGTC_LOGLEVEL_DEBUG,a,##__VA_ARGS__)
#define SNGTC_WARN(a,...) 		sngtc_print(SNGTC_LOGLEVEL_WARN,a,##__VA_ARGS__)
#define SNGTC_INFO(a,...) 		sngtc_print(SNGTC_LOGLEVEL_INFO,a,##__VA_ARGS__)
#define SNGTC_STATS(a,...) 		sngtc_print(SNGTC_LOGLEVEL_STATS,a,##__VA_ARGS__)
#define SNGTC_ERROR(a,...)		sngtc_print(SNGTC_LOGLEVEL_ERROR,a,##__VA_ARGS__)
#define SNGTC_CRIT(a,...) 		sngtc_print(SNGTC_LOGLEVEL_CRIT,a,##__VA_ARGS__)
#define SNGTC_TEST(a,...)
#else
#define SNGTC_PRINT printf
#define SNGTC_DEBUG printf
#define SNGTC_WARN  printf
#define SNGTC_INFO  printf
#define SNGTC_STATS printf
#define SNGTC_ERROR printf
#define SNGTC_CRIT printf
#define SNGTC_TEST printf
#endif




#define SNGTC_NIPV4(addr) \
	((unsigned char *)&addr)[3], \
	((unsigned char *)&addr)[2], \
	((unsigned char *)&addr)[1], \
	((unsigned char *)&addr)[0]

#define SNGTC_NIPV4_FMT "%u.%u.%u.%u"


#define SNGTC_LOGLEVEL_DECODE(level) \
		((level==SNGTC_LOGLEVEL_DEBUG) ? "SNGTC_DEBUG" : \
		(level==SNGTC_LOGLEVEL_WARN) ? "SNGTC_WARN" : \
		(level==SNGTC_LOGLEVEL_INFO) ? "SNGTC_INFO" : \
		(level==SNGTC_LOGLEVEL_STATS) ? "SNGTC_STATS" : \
		(level==SNGTC_LOGLEVEL_ERROR) ? "SNGTC_ERROR" : \
		(level==SNGTC_LOGLEVEL_CRIT) ? "SNGTC_CRIT" : "Unkown")
		

static SNG_INLINE int sngtc_codec_ipv4_str_hex(const char *ip_str, uint32_t *ip_addr)
{
	char t_ip_str[50];
	char *p;
	uint32_t temp;
	char cnt=3;
#ifndef WIN32
	char *saveptr;
#endif
	*ip_addr=0;

	if (strlen(ip_str) > 15) {
		return 1;
	}

	sngtc_snprintf(t_ip_str, sizeof(t_ip_str), "%s",ip_str);

	p = sngtc_strtok_r(t_ip_str, ".", &saveptr);
	while (p != NULL) {

		if (atoi(p) > 255) {
			SNGTC_ERROR( "%s(): Error: Invalid IP %s\n",__FUNCTION__,ip_str);
			return 1;
		}
	
		temp = atoi(p);
		temp= temp&0xFF;

		//SNGTC_DEBUG("GOT IP NIBBLE 0x%02X Shifting by %i Shift=0x%08X\n",temp,8*cnt, (temp<<(8*cnt)));
		p=sngtc_strtok_r(NULL, ".", &saveptr);

		*ip_addr |= (temp<<(8*cnt));

		if (cnt > 0) {
			cnt--;
		}
	}

	if (cnt != 0) { 
		SNGTC_ERROR("libsangoma_codec_ipv4_str_hex(): CNT ERROR IP=%s = IP=0x%08X cnt=%i p=%s tip=%s\n",
					ip_str, *ip_addr, cnt, p, t_ip_str);
		return 1;
	}

	return 0;
}


static SNG_INLINE int sngtc_codec_ipv4_hex_to_str(uint32_t ip_addr, char *ip_str)
{
	sprintf(ip_str,"%d.%d.%d.%d",
			(ip_addr>>24)&0xFF,
			(ip_addr>>16)&0xFF,
			(ip_addr>>8)&0xFF,
			(ip_addr)&0xFF);


	return 0;
}

enum {
	SNGTC_MODE_LIBRARY,
	SNGTC_MODE_SOAP_CLIENT,
	SNGTC_MODE_SOAP_SERVER
};


#endif

