/*******************************************************************************//**
 * \file sng_tc.h
 * \brief Sangoma Transcoding API 
 * \brief Defines Sangoma Transcoding API functions and structures
 *              
 * Sangoma Transcoder Library
 *
 * Author(s): Nenad Corbic <ncorbic@sangoma.com>
 *            Moises Silva <moy@sangoma.com>
 *
 *
 * Copyright (C) 2010 Sangoma Technologies
 *
 * Redistribution and use is permitted only in binary forms without
 * modification and provided that the following conditions are met:
 *
 * * Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * * Neither the name of the original author; nor the names of any contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * David Rokhvarg  <davidr@sangoma.com>		ported to MS Windows
 */

#ifndef __LIB_SANGOMA_TRANSCODE__
#define __LIB_SANGOMA_TRANSCODE__

#include "sng_tc_os.h"
#include "sng_tc_version.h"
#include "sng_tc_if.h"
#include "sng_tc_utils.h"


enum sngtc_rpt_session_type
{
	SNGTC_RTP_SESSION_A,
	SNGTC_RTP_SESSION_B
};

struct sngtc_codec_reply;


typedef struct sngtc_vocallo_map {
	uint8_t init;

	sngtc_eth_mac_t v_mac;
	sngtc_ip_t 		v_ip;
	uint32_t		v_eth_port;

	uint32_t		tdm_timestamp_reg;
	uint32_t		error;
}sngtc_vocallo_map_t;

#define MAX_IP_STR_SIZE 100
typedef struct sngtc_nic_vocallo_map{
	uint8_t init;
	sngtc_eth_mac_t nic_mac;
	sngtc_ip_t nic_ip;
	sngtc_ip_t nic_netmask;

	sngtc_ip_t voc_base_ip;
	sngtc_ip_t voc_base_netmask;
	int        voc_base_udp_prot;

	int voc_mods_sz;
	sngtc_vocallo_map_t voc_map[SNGTC_MAX_MAC_RANGE];
}sngtc_nic_vocallo_map_t;






/***************************
 Statistics
***************************/
/*!
	\struct sngtc_rtcp_sender_stats
	\brief RTCP Sender Statistics
	\typedef sngtc_rtcp_sender_stats_t
*/
typedef struct sngtc_rtcp_sender_stats
{
	uint32_t SentPktCnt[2];
	uint32_t SentByteCnt[2];
	uint32_t NtpTimestamp[2];
	uint32_t RtpTimestamp;
	uint32_t SenderSsrc;
	uint32_t SenderStartTime;
	uint32_t SenderReportCnt;
	uint32_t SenderState;
	uint32_t LastSenderReportTime;
	uint32_t SsrcChangeCnt;
} sngtc_rtcp_sender_stats_t;

/*!
	\struct sngtc_rtcp_receiver_stats
	\brief RTCP Receiver Statistics
	\typedef sngtc_rtcp_receiver_stats_t
*/
typedef struct sngtc_rtcp_receiver_stats
{
	uint32_t ulHighestSeqNumber;
	uint32_t ulCumulativeLostCnt;
	uint32_t ulFractionLost;
	uint32_t ulInterArrivalJitter;
	uint32_t ulMaxFractionLost;
	uint32_t ulMaxInterArrivalJitter;
	uint32_t ulAverageFractionLost;
	uint32_t ulAverageInterArrivalJitter;
	uint32_t ulReceiverStartTime;
	uint32_t ulReceiverReportCnt;
	uint32_t ulLastReceiverReportTime;
	uint32_t ulRemoteSenderSsrc;
	uint32_t ulAlarmRxThreshold;
	uint32_t ulAlarmRxThresholdCnt;
} sngtc_rtcp_receiver_stats_t;

/*!
	\struct sngtc_rtp_global_stats
	\brief RTP Global Statistics
	\typedef sngtc_rtp_global_stats_t
*/
typedef struct sngtc_rtp_global_stats
{
	uint32_t NumMember;
	uint32_t NumSender;
	uint32_t SessionStartTime;
	uint32_t RxByePktCnt;
	uint32_t RxSenderJoinCnt;
	uint32_t RxMemberJoinCnt;
	uint32_t ErrCollisionCnt;
	uint32_t ErrLoopCnt;
	uint32_t ErrRtcpPktInvalidCnt;
} sngtc_rtp_global_stats_t ;

/*!
	\struct sngtc_rtp_local_stats
	\brief Local RTP Statistics
	\typedef sngtc_rtp_local_stats_t
*/
typedef struct sngtc_rtp_local_stats
{
	/* Local Statistics */
	uint32_t ReceivedPktCnt[2];
	uint32_t ReceivedByteCnt[2];
	sngtc_rtcp_sender_stats_t SenderStats;
	sngtc_rtcp_receiver_stats_t ReceiverStats;
} sngtc_rtp_local_stats_t;


/*!
	\struct sngtc_rtp_remote_stats
	\brief Remote RTP Statistics
	\typedef sngtc_rtp_remote_stats_t
*/
typedef struct sngtc_rtp_remote_stats
{
	uint32_t RoundTripTime;
	uint32_t MaxRoundTripTime;
	uint32_t AverageRoundTripTime;
	sngtc_rtcp_sender_stats_t SenderStats;
	sngtc_rtcp_receiver_stats_t ReceiverStats;
} sngtc_rtp_remote_stats_t;

/*!
	\struct sngtc_rtp_stats
	\brief Full RTP Statistics both Local & Remote
	\typedef sngtc_rtp_stats_t
*/
typedef struct sngtc_rtp_stats
{
	sngtc_rtp_global_stats_t stats;
	sngtc_rtp_local_stats_t	 local_stats;
	sngtc_rtp_remote_stats_t remote_stats;

} sngtc_rtp_stats_t;


int _SNGTC_API sngtc_get_nic_vocallo_map(sngtc_init_cfg_t *cfg,
							  sngtc_nic_vocallo_map_t **nic_vocallo_map, 
							  uint32_t *nic_voc_map_idx);


int _SNGTC_API sngtc_rtp_get_status(uint16_t rtp_session_idx, int rtp_a, sngtc_rtp_stats_t *rtp_stats);

int _SNGTC_API sngtc_all_free_sanity_check(void);

int _SNGTC_API sngtc_module_get_instance(uint16_t instance_idx, uint16_t *module_idx);
int _SNGTC_API sngtc_module_reset(uint16_t module_idx);
int _SNGTC_API sngtc_module_restore_session(uint16_t module_idx);
int _SNGTC_API sngtc_module_create_host(uint16_t module_idx);
int _SNGTC_API sngtc_module_write_file(uint16_t module_idx, char *local_file, char *remote_file, int progress);
int _SNGTC_API sngtc_module_get_sw_version(uint16_t module_idx, char *sw_ver_str, int sw_ver_str_sz);

int _SNGTC_API sngtc_set_iana_code_based_on_codec_id(uint32_t codec_id, uint32_t iana);
void _SNGTC_API sngtc_mac_addr_to_str(char *mac_str, uint8_t mac[SNGTC_MAX_ETH_SZ]);
int _SNGTC_API sngtc_free_transcoding_sessions_by_tag (uint64_t tag);
int _SNGTC_API sngtc_module_read_sid(uint16_t module_idx, char *local_file);
int _SNGTC_API sng_tc_get_gateway_ip(sngtc_ip_t *gateway);
int _SNGTC_API sngtc_detect_eth_dev(void);

/* provide either a eth name or mac and get back the full configuration */
int _SNGTC_API sngtc_find_ethconf(char *ethname, uint8_t *mac, system_eth_cfg_t **eth_cfg);
int _SNGTC_API sngtc_module_set_cpu_threshold(uint16_t module_idx, int high, int low);
int _SNGTC_API sngtc_poll(void);
int _SNGTC_API sngtc_module_write_bin_file(uint16_t module_idx, char *bin_name,
											char *bin_file, char * bin_file_end,
											char *remote_file, int progress);
int _SNGTC_API sngtc_module_get_capacity(uint16_t module_idx, int *capacity);
int _SNGTC_API sngtc_is_module_active(uint16_t module_idx);
#endif

