/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2007 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisSend.c - NDIS driver send functions.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         09/29/05    Changed descriptor structure.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         04/03/06    Disable send queue because of crashes.
    THa         06/02/06    Report statistics from MIB counters.
    THa         09/22/06    Version 1.6 fixes big UDP packet transmit problem.
    THa         12/18/07    Add hardware checksumming support.
   ---------------------------------------------------------------------------
*/


#include "target.h"
#include "NdisDevice.h"


#if 0
#define SEND_QUEUE
#endif

/* -------------------------------------------------------------------------- */

USHORT ntohs (
    USHORT wNetValue )
{
    USHORT wHostValue;

    wHostValue = (( UCHAR ) wNetValue << 8 ) | ( UCHAR )( wNetValue >> 8 );
    return( wHostValue );
}  // ntohs


/*
    GetFirstBuffer

    Description:
        This function retrieves the first NDIS buffer in the NDIS packet.

    Parameters:
        PNDIS_BUFFER* ppndisBuffer
            Pointer to NDIS buffer pointer.

        PUINT puiLength
            Buffer to store the length of the data buffer.

    Return (PUCHAR):
        Pointer to data buffer in the NDIS buffer.
*/

__inline
PUCHAR GetFirstBuffer (
    PNDIS_BUFFER* ppndisBuffer,
    PUINT         puiLength )
{
    PUCHAR pBuffer;

    // Each packet may be made up of several buffers, linked together.  Get the
    // virtual address (in pBuffer) and the length (in uiLength) of the first
    // buffer in the packet by calling NdisQueryBuffer.
    NdisQueryBuffer( *ppndisBuffer, ( PVOID* ) &pBuffer, puiLength );

    // If the pointer to the buffer is not NULL and the length returned from
    // the above call is zero, skip to the next buffer and query it for its
    // length.
    while ( !*puiLength )
    {
        NdisGetNextBuffer( *ppndisBuffer, ppndisBuffer );
        if ( !*ppndisBuffer )
        {
            return NULL;
        }
        NdisQueryBuffer( *ppndisBuffer, ( PVOID* ) &pBuffer, puiLength );
    }
    return( pBuffer );
}  // GetFirstBuffer


/*
    GetNextBuffer

    Description:
        This function retrieves the next NDIS buffer in the NDIS packet.

    Parameters:
        PNDIS_BUFFER* ppndisBuffer
            Pointer to NDIS buffer pointer.

        PUINT puiLength
            Buffer to store the length of the data buffer.

    Return (PUCHAR):
        Pointer to data buffer in the NDIS buffer.
*/

__inline
PUCHAR GetNextBuffer (
    PNDIS_BUFFER* ppndisBuffer,
    PUINT         puiLength )
{
    PUCHAR pBuffer;

    *puiLength = 0;

    // Go through the process of eliminating buffers with no data in them.
    // Exit the following while loop when we run out of buffers or have one
    // that contains some data.
    while ( !*puiLength )
    {
        NdisGetNextBuffer( *ppndisBuffer, ppndisBuffer );
        if ( !*ppndisBuffer )
        {
            return NULL;
        }
        NdisQueryBuffer( *ppndisBuffer, ( PVOID* ) &pBuffer, puiLength );
    }
    return( pBuffer );
}  // GetNextBuffer


/*
    AdapterCopyDownPacket

    Description:
        This function copies the packet down to the adapter buffer.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_BUFFER pndisBuffer
            Pointer to NDIS buffer.

        UINT uiPacketLength
            The length of the buffer.

        PUINT puiLength
            Buffer to store the actual length of data copied.

        BOOLEAN* pbBroadcast
            Buffer to store the broadcast flag.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN AdapterCopyDownPacket (
    IN  PNDIS_ADAPTER pAdapter,
    IN  PNDIS_BUFFER  pndisBuffer,
    IN  UINT          uiPacketLength,
    OUT PUINT         puiLength,
    OUT BOOLEAN*      pbBroadcast )
{
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    PDMA_BUFFER pDma = pAdapter->m_TxBufInfo.pCurrent;

    // Length of current source buffer
    UINT        uiBufferLength;

    // Address of current source buffer
    PUCHAR      pBuffer;

    *puiLength = 0;

    pBuffer = GetFirstBuffer( &pndisBuffer, &uiBufferLength );

    if ( pBuffer )
    {
        if ( ( pBuffer[ 0 ] & 0x01 ) )
        {
            *pbBroadcast = TRUE;
        }
        else
        {
            *pbBroadcast = FALSE;
        }
        HardwareSetTransmitBuffer( pHardware, pDma->sm.ulPhysical );
        HardwareSetTransmitLength( pHardware, uiPacketLength );
    }

    while ( pBuffer )
    {
        NdisMoveMemory( &pDma->sm.pVirtual[ *puiLength ], pBuffer,
            uiBufferLength );
        *puiLength += uiBufferLength;

        pBuffer = GetNextBuffer( &pndisBuffer, &uiBufferLength );
    }

    if ( !*puiLength )
    {
        goto CopyDownPacketError;
    }

    /* Flush buffer. */
    NdisFlushBuffer( pDma->pNdisBuffer, TRUE );

    return TRUE;

CopyDownPacketError:

#ifdef DEBUG_COUNTER
    pHardware->m_nBad[ COUNT_BAD_COPY_DOWN ]++;
#endif
    return FALSE;
}  // AdapterCopyDownPacket


/*
    AllocMapReg

    Description:
        This function allocates a mapping register information block for
        transmission.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (PMAP_REG):
        Pointer to mapping register information block if one is allocated, else
        NULL.
*/

__inline
PMAP_REG AllocMapReg (
    PNDIS_ADAPTER pAdapter )
{
    PMAP_REG pMapReg = NULL;

    ASSERT( pAdapter->m_cnMapRegAvail > 0 );
#if 0
    if ( pAdapter->m_cnMapRegAvail )
#endif
    {
        pMapReg = pAdapter->m_pMapRegTop;
        pAdapter->m_pMapRegTop = pMapReg->pNext;
        pAdapter->m_cnMapRegAvail--;
    }
    return( pMapReg );
}  // AllocMapReg


/*
    FreeMapReg

    Description:
        This routine frees the mapping register information block.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PMAP_REG pMapReg
            Pointer to the mapping register information block.

    Return (None):
*/

__inline
VOID FreeMapReg (
    PNDIS_ADAPTER pAdapter,
    PMAP_REG      pMapReg )
{
    pMapReg->pNdisBuffer = NULL;
    pMapReg->pNext = pAdapter->m_pMapRegTop;
    pAdapter->m_pMapRegTop = pMapReg;
    pAdapter->m_cnMapRegAvail++;
}  // FreeMapReg

/* -------------------------------------------------------------------------- */

__inline
VOID HardwareChecksum (
    PNDIS_PACKET pPacket,
    PHARDWARE    pHardware )
{
    if ( NDIS_PROTOCOL_ID_TCP_IP == NDIS_GET_PACKET_PROTOCOL_TYPE( pPacket ) )
    {
        PNDIS_TCP_IP_CHECKSUM_PACKET_INFO pChecksum =
            ( PNDIS_TCP_IP_CHECKSUM_PACKET_INFO ) &
            NDIS_PER_PACKET_INFO_FROM_PACKET( pPacket,
                TcpIpChecksumPacketInfo );
        if ( pChecksum->Transmit.NdisPacketChecksumV4 )
        {
            if ( pChecksum->Transmit.NdisPacketTcpChecksum )
            {
                pHardware->m_TxDescInfo.pCurrent->sw.BufSize.tx.fCsumGenTCP =
                    TRUE;
            }
            if ( pChecksum->Transmit.NdisPacketUdpChecksum )
            {
                pHardware->m_TxDescInfo.pCurrent->sw.BufSize.tx.fCsumGenUDP =
                    TRUE;
            }
            if ( pChecksum->Transmit.NdisPacketIpChecksum )
            {
                pHardware->m_TxDescInfo.pCurrent->sw.BufSize.tx.fCsumGenIP =
                    TRUE;
            }
        }
    }
}  /* HardwareChecksum */


/*
    SendPacket

    Description:
        This function sends a packet to hardware.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_BUFFER pBuffer
            Pointer to NDIS buffer.

        UINT uiPacketLength
            The length of the buffer.

    Return (NDIS_STATUS):
        NDIS_STATUS_PENDING if successful; otherwise, an error code indicating
        failure.
*/

NDIS_STATUS SendPacket (
    PNDIS_ADAPTER pAdapter,
    PNDIS_PACKET  pPacket,
    PNDIS_BUFFER  pBuffer,
    UINT          uiPacketLength )
{
    PBUFFER_INFO pBufInfo = &pAdapter->m_TxBufInfo;
    PHARDWARE    pHardware = &pAdapter->m_Hardware;
    PTDescInfo   pInfo = &pHardware->m_TxDescInfo;
    NDIS_STATUS  nsStatus = NDIS_STATUS_FAILURE;
    UINT         uiLength;
    BOOLEAN      bBroadcast;
    BOOLEAN      bResult;
    int          port = MAIN_PORT;

    pBufInfo->pCurrent = Alloc_DMA_Buffer( pBufInfo, pInfo->pCurrent );

    bResult = AdapterCopyDownPacket( pAdapter, pBuffer, uiPacketLength,
        &uiLength, &bBroadcast );

    if ( !bResult )
    {
        Free_DMA_Buffer( pBufInfo, pBufInfo->pCurrent );
        pInfo->pCurrent->pReserved = NULL;

        HardwareFreeTxPacket( pInfo );
        goto SendIntrDone;
    }
    pBufInfo->pCurrent->pNdisPacket = pPacket;

    if ( pAdapter->m_ulTxChecksum )
        HardwareChecksum( pPacket, pHardware );
    if ( HardwareSendPacket( pHardware ) )
    {
        nsStatus = NDIS_STATUS_PENDING;
        if ( uiLength )
        {
            pHardware->m_cnCounter[ port ][ OID_COUNTER_XMIT_OK ]++;

#ifndef NO_STATS
            if ( bBroadcast )
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_FRAME_XMIT ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_BROADCAST_BYTES_XMIT ] += uiLength;
            }
            else
            {
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_FRAMES_XMIT ]++;
                pHardware->m_cnCounter[ port ]
                    [ OID_COUNTER_DIRECTED_BYTES_XMIT ] += uiLength;
            }
#endif
        }
    }

SendIntrDone:

    return( nsStatus );
}  // SendPacket


#define DRV_PACKET_ARRAY_SIZE  32


/*
    SendBuffers

    Description:
        This function sends a packet in several buffer descriptors to hardware.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_Packet pPacket
            Pointer to NDIS packet.

        PNDIS_BUFFER pndisBuffer
            Pointer to NDIS buffer.

        UINT uiPacketLength
            Length of the packet.

        UINT uiBufferCount
            Number of buffers.

    Return (NDIS_STATUS):
        NDIS_STATUS_PENDING if successful; otherwise, an error code indicating
        failure.
*/

NDIS_STATUS SendBuffers (
    PNDIS_ADAPTER pAdapter,
    PNDIS_PACKET  pPacket,
    PNDIS_BUFFER  pndisBuffer,
    UINT          uiPacketLength,
    UINT          uiBufferCount )
{
    PTDesc                     pDesc;
    PTDesc                     pLastDesc;
    PTDesc                     pFirstDesc;
    PDMA_BUFFER                pDma;
    PHARDWARE                  pHardware = &pAdapter->m_Hardware;
    PTDescInfo                 pInfo = &pHardware->m_TxDescInfo;
    NDIS_STATUS                nsStatus = NDIS_STATUS_FAILURE;
    TDesc                      BufDesc;
    NDIS_PHYSICAL_ADDRESS_UNIT PhySegArray[ DRV_PACKET_ARRAY_SIZE ];
    UINT                       uiPhySegCount;
    UINT                       uiBuf;
    UINT                       uiSeg;
#if 0
    UINT                       uiLength;
    BOOLEAN                    bBroadcast;
#endif
    int                        port = MAIN_PORT;

    /* Remember the very first descriptor. */
    pFirstDesc = pInfo->pCurrent;
    pDesc = pFirstDesc;
    pLastDesc = pFirstDesc;

#if 0
{
    UINT   uiBufferLength;
    PUCHAR pBuffer = GetFirstBuffer( &pndisBuffer, &uiBufferLength );

    if ( pBuffer  &&  ( pBuffer[ 0 ] & 0x01 ) )
    {
        bBroadcast = TRUE;
    }
    else
    {
        bBroadcast = FALSE;
    }
}
#endif

    for ( uiBuf = 0; uiBuf < uiBufferCount; uiBuf++ )
    {

#if 0
        // want length only
        NdisQueryBuffer( pndisBuffer, NULL, &uiLength );
        if ( 0 == uiLength )
        {
            continue;
        }
#endif

        pDma = Alloc_DMA_Buffer( &pAdapter->m_TxBufInfo, &BufDesc );

        /* Get free map register. */
        pDma->pMapReg = AllocMapReg( pAdapter );
        pDma->pMapReg->pNdisBuffer = pndisBuffer;

        /* Get the mapping of the physical segments. */
        NdisMStartBufferPhysicalMapping(
            pAdapter->m_hAdapter,
            pndisBuffer,                // Buffer
            pDma->pMapReg->uiRegister,  // PhysicalMapRegister
            TRUE,                       // WriteToDevice
            PhySegArray,                // PhysicalAddressArray
            &uiPhySegCount              // ArraySize
            );
#ifdef DBG
        if ( uiPhySegCount )
#endif
        {
            for ( uiSeg = 0; uiSeg < uiPhySegCount; uiSeg++ )
            {
#ifdef DBG
                if ( PhySegArray[ uiSeg ].Length )
#endif
                {

                    /* Release this descriptor to hardware if not the very
                       first one, not wanting the hardware to start
                       transmitting this packet when descriptors are being set.
                    */
                    if ( pLastDesc != pFirstDesc )
                        ReleasePacket( pLastDesc );

                    SetTransmitBuffer( pDesc,
                        ( UINT32 ) NdisGetPhysicalAddressLow(
                        PhySegArray[ uiSeg ].PhysicalAddress ));
                    SetTransmitLength( pDesc,
                        PhySegArray[ uiSeg ].Length );

                    /* Remember the last descriptor. */
                    pLastDesc = pDesc;

                    /* Allocate more descriptor. */
                    GetTxPacket( pInfo, pDesc );

#ifdef SKIP_TX_INT
                    ++pHardware->m_TxIntCnt;
#endif
                }
            }

            /* If more than one descriptor is used for this buffer, move the
               buffer so that the mapping is freed only after all descriptors
               are transmitted.
            */
            pLastDesc->pReserved = BufDesc.pReserved;
        }

#ifdef DBG
        else
        {
            NdisMCompleteBufferPhysicalMapping(
                pAdapter->m_hAdapter, pndisBuffer, pDma->pMapReg->uiRegister );

            /* Recycle this map register. */
            FreeMapReg( pAdapter, pDma->pMapReg );
            pDma->pMapReg = NULL;

            Free_DMA_Buffer( &pAdapter->m_TxBufInfo, pDma );

            do
            {
                HardwareFreeTxPacket( pInfo );
                pDesc = &pInfo->pRing[ pInfo->iNext ];
                ReleaseDescriptor( pDesc, pDesc->sw.Control );
            } while ( pDesc != pFirstDesc );
            goto SendBuffersDone;
        }
#endif

        /* Flush buffer. */
        NdisFlushBuffer( pndisBuffer, TRUE );

        /* Get next buffer. */
        NdisGetNextBuffer( pndisBuffer, &pndisBuffer );
    }

    /* One more descriptor is allocated. */
    HardwareFreeTxPacket( pInfo );

    /* The last descriptor holds the packet so that it can be returned to
       network subsystem after all descriptors are transmitted.
    */
    pDma->pNdisPacket = pPacket;

    /* pCurrent points to the last descriptor. */
    pInfo->pCurrent = pLastDesc;

    /* Release the first descriptor. */
    ReleasePacket( pFirstDesc );

    if ( pAdapter->m_ulTxChecksum )
        HardwareChecksum( pPacket, pHardware );
    if ( HardwareSendPacket( pHardware ) )
    {
        nsStatus = NDIS_STATUS_PENDING;
        pHardware->m_cnCounter[ port ][ OID_COUNTER_XMIT_OK ]++;

#ifndef NO_STATS
        if ( bBroadcast )
        {
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_BROADCAST_FRAME_XMIT ]++;
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_BROADCAST_BYTES_XMIT ] += uiPacketLength;
        }
        else
        {
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_DIRECTED_FRAMES_XMIT ]++;
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_DIRECTED_BYTES_XMIT ] += uiPacketLength;
        }
#endif
    }

#ifdef DBG
SendBuffersDone:
#endif

    return( nsStatus );
}  // SendBuffers

/* -------------------------------------------------------------------------- */

/*
    TransmitDone

    Description:
        This routine cleans up the descriptors after transmission.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID TransmitDone (
    PNDIS_ADAPTER pAdapter )
{
    int         iLast;
    TDescStat   status;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    PTDescInfo  pInfo = &pHardware->m_TxDescInfo;
    PTDesc      pDesc;
    PDMA_BUFFER pDma;

    iLast = pInfo->iLast;

    while ( pInfo->cnAvail < pInfo->cnAlloc )
    {
        /* Get next descriptor which is not hardware owned. */
        GetTransmittedPacket( pInfo, iLast, pDesc, status.ulData );
        if ( status.tx.fHWOwned )
            break;

        pDma = DMA_BUFFER( pDesc );

        /* This descriptor contains a buffer. */
        if ( pDma )
        {
            /* It is a mapped buffer. */
            if ( pDma->pMapReg )
            {
                /* Ask network subsystem to free the mapping. */
                NdisMCompleteBufferPhysicalMapping(
                    pAdapter->m_hAdapter,
                    pDma->pMapReg->pNdisBuffer,
                    pDma->pMapReg->uiRegister );

                /* Free the map register. */
                FreeMapReg( pAdapter, pDma->pMapReg );
                pDma->pMapReg = NULL;
            }

            /* This descriptor contains the last buffer in the packet. */
            if ( pDma->pNdisPacket )
            {
                /* Notify the network subsystem that the packet has been sent.
                */
                NdisMSendComplete( pAdapter->m_hAdapter, pDma->pNdisPacket,
                    NDIS_STATUS_SUCCESS );

                /* Release the packet back to network subsystem. */
                pDma->pNdisPacket = NULL;
            }

            /* Free the transmitted buffer. */
            Free_DMA_Buffer( &pAdapter->m_TxBufInfo, pDma );
            pDesc->pReserved = NULL;
        }

        /* Free the transmitted descriptor. */
        FreeTransmittedPacket( pInfo, iLast );
    }
    pInfo->iLast = iLast;

#ifdef SEND_QUEUE
    if ( pHardware->m_TxDescInfo.cnAvail >
            pHardware->m_TxDescInfo.cnAlloc / 4 )
        SendNextPacket( pAdapter );
#endif
}  // TransmitDone


/*
    TransmitReset

    Description:
        This routine resets the descriptors to starting state after a software
        reset.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID TransmitReset (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    PTDescInfo  pInfo = &pHardware->m_TxDescInfo;
    int         iLast;
    PTDesc      pDesc;
    PDMA_BUFFER pDma;
    TDescStat   status;

    iLast = pInfo->iLast;

    while ( pInfo->cnAvail < pInfo->cnAlloc )
    {
        /* Get next descriptor which is hardware owned. */
        GetTransmittedPacket( pInfo, iLast, pDesc, status.ulData );
        if ( status.tx.fHWOwned )
        {
            ReleaseDescriptor( pDesc, status );
        }

        pDma = DMA_BUFFER( pDesc );

        /* This descriptor contains a buffer. */
        if ( pDma )
        {
            /* It is a mapped buffer. */
            if ( pDma->pMapReg )
            {
                /* Ask network subsystem to free the mapping. */
                NdisMCompleteBufferPhysicalMapping(
                    pAdapter->m_hAdapter,
                    pDma->pMapReg->pNdisBuffer,
                    pDma->pMapReg->uiRegister );

                /* Free the map register. */
                FreeMapReg( pAdapter, pDma->pMapReg );
                pDma->pMapReg = NULL;
            }

            /* This descriptor contains the last buffer in the packet. */
            if ( pDma->pNdisPacket )
            {
                /* Notify the network subsystem that the packet has not been
                   sent.
                */
                NdisMSendComplete( pAdapter->m_hAdapter, pDma->pNdisPacket,
                    NDIS_STATUS_FAILURE );

                /* Release the packet back to network subsystem. */
                pDma->pNdisPacket = NULL;
            }

            /* Free the buffer. */
            Free_DMA_Buffer( &pAdapter->m_TxBufInfo, pDma );
            pDesc->pReserved = NULL;
        }

        /* Free the descriptor. */
        FreeTransmittedPacket( pInfo, iLast );
    }
    pInfo->iLast = iLast;
}  // TransmitReset

/* -------------------------------------------------------------------------- */

/*
    MiniportSend

    Description:
        This function sends one packet in NDIS 3.0.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_PACKET pPacket
            Pointer to NDIS packet.

        UINT uiFlags
            Send flags.

    Return (NDIS_STATUS):
        NDIS_STATUS_PENDING if successful; otherwise, an error code indicating
        failure.
*/

NDIS_STATUS MiniportSend (
    IN  NDIS_HANDLE  hAdapterContext,
    IN  PNDIS_PACKET pPacket,
    IN  UINT         uiFlags )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS   nsStatus = NDIS_STATUS_FAILURE;
    PNDIS_BUFFER  pndisBuffer;
    UINT          uiPacketLength;
    UINT          uiPhysicalBufferCount;
    UINT          uiBufferCount;
    UINT          InterruptMask;
    int           nCount;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        goto SendDone;
    }

#ifdef DEBUG_COUNTER
    pHardware->m_nGood[ COUNT_GOOD_SEND_PACKET ]++;
#endif
    NdisQueryPacket( pPacket, &uiPhysicalBufferCount, &uiBufferCount,
        &pndisBuffer, &uiPacketLength );

    // If the returned packet length is zero, there is nothing to transmit.
    if ( !uiPacketLength )
    {
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_SEND_ZERO ]++;
#endif
        nsStatus = NDIS_STATUS_SUCCESS;
        goto SendLockDone;
    }

    /* Save the current interrupt mask and block all interrupts. */
    InterruptMask = HardwareBlockInterrupt( pHardware );

    if ( ( int ) uiBufferCount > pAdapter->m_cnMapRegAvail )
    {
        nCount = HardwareAllocPacket( pHardware, uiPacketLength, 1 );
    }
    else
        nCount = HardwareAllocPacket( pHardware, uiPacketLength,
            uiPhysicalBufferCount );
    if ( !nCount )
    {
        nsStatus = NDIS_STATUS_RESOURCES;
        goto SendBlockDone;
    }

    if ( nCount > ( int ) uiBufferCount )
    {
        nsStatus = SendBuffers( pAdapter, pPacket, pndisBuffer, uiPacketLength,
            uiBufferCount );
    }
    else
        nsStatus = SendPacket( pAdapter, pPacket, pndisBuffer, uiPacketLength );

SendBlockDone:
    /* Restore the interrupt mask. */
    HardwareSetInterrupt( pHardware, InterruptMask );

SendLockDone:
    ReleaseAdapter( pAdapter );

SendDone:
    NDIS_SET_PACKET_STATUS( pPacket, nsStatus );
    return( nsStatus );
}  // MiniportSend

/* -------------------------------------------------------------------------- */

/*
    AdapterTransmitQueueLength

    Description:
        This function returns the number of packets waiting in the transmit
        queue.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (UINT):
        The number of packets in the send queue.
*/

UINT AdapterTransmitQueueLength (
    IN  PNDIS_ADAPTER pAdapter )
{
    PNDIS_PACKET pPacket;
    UINT         ulCount = 0;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
        return( NUM_OF_TX_DESC );

    pPacket = pAdapter->m_pPacketQueueHead;
    while ( pPacket )
    {
        ++ulCount;
        pPacket = RESERVED( pPacket )->Next;
    }
    ReleaseAdapter( pAdapter );
    return( ulCount );
}  // AdapterTransmitQueueLength


#ifdef SEND_QUEUE
/*
    AddPacketToTransmitQueue

    Description:
        This routine adds packets to the transmit queue.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PNDIS_PACKET pPacket
            Pointer to NDIS packet.

    Return (None):
*/

void AddPacketToTransmitQueue (
    PNDIS_ADAPTER pAdapter,
    PNDIS_PACKET  pPacket )
{
    if ( pAdapter->m_pPacketQueueTail )
        RESERVED( pAdapter->m_pPacketQueueTail )->Next = pPacket;
    else
        pAdapter->m_pPacketQueueHead = pPacket;
    RESERVED( pPacket )->Next = NULL;
    pAdapter->m_pPacketQueueTail = pPacket;
    NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_PENDING );
}  // AddPacketToTransmitQueue


/*
    SendNextPacket

    Description:
        This routine is used to send the next packet waiting in the queue.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void SendNextPacket (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE    pHardware = &pAdapter->m_Hardware;
    NDIS_STATUS  nsStatus;
    PNDIS_PACKET pPacket;
    PNDIS_BUFFER pndisBuffer;
    UINT         uiPacketLength;
    UINT         uiBufferCount;
    UINT         uiPhysicalBufferCount;
    int          nCount;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        return;
    }

    // Return if we don't have a packet to send.
    if ( !pAdapter->m_pPacketQueueHead )
    {
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_NO_NEXT_PACKET ]++;
#endif
        goto SendNextPacketDone;
    }

    pPacket = pAdapter->m_pPacketQueueHead;

    NdisQueryPacket( pPacket, &uiPhysicalBufferCount, &uiBufferCount,
        &pndisBuffer, &uiPacketLength );

    // If the returned packet length is zero, there is nothing to transmit.
    if ( !uiPacketLength )
    {
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_SEND_ZERO ]++;
#endif
        nsStatus = NDIS_STATUS_SUCCESS;
        NDIS_SET_PACKET_STATUS( pPacket, nsStatus );
        goto SendNextDone;
    }

    if ( ( int ) uiBufferCount > pAdapter->m_cnMapRegAvail )
    {
        nCount = HardwareAllocPacket( pHardware, uiPacketLength, 1 );
    }
    else
        nCount = HardwareAllocPacket( pHardware, uiPacketLength,
            uiPhysicalBufferCount );
    if ( !nCount )
    {
        goto SendNextPacketDone;
    }

    if ( nCount > ( int ) uiBufferCount )
    {
        nsStatus = SendBuffers( pAdapter, pPacket, pndisBuffer, uiPacketLength,
            uiBufferCount );
    }
    else
        nsStatus = SendPacket( pAdapter, pPacket, pndisBuffer, uiPacketLength );

SendNextDone:

    // Remove the packet from the queue.
#ifdef DEBUG_COUNTER
    pHardware->m_nGood[ COUNT_GOOD_NEXT_PACKET ]++;
#endif
    pAdapter->m_pPacketQueueHead = RESERVED( pPacket )->Next;
    if ( pPacket == pAdapter->m_pPacketQueueTail )
        pAdapter->m_pPacketQueueTail = NULL;

SendNextPacketDone:
    ReleaseAdapter( pAdapter );
}  // SendNextPacket
#endif


/*
    MiniportSendPackets

    Description:
        This routine sends one or more packets in NDIS 4.0.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        PPNDIS_PACKET pPacketArray
            Array of pointers to NDIS packets.

        UINT uiCount
            Number of packets in the array.

    Return (None):
*/

VOID MiniportSendPackets (
    IN  NDIS_HANDLE   hAdapterContext,
    IN  PPNDIS_PACKET pPacketArray,
    IN  UINT          uiCount )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    NDIS_STATUS   nsStatus;
    PNDIS_PACKET  pPacket;
    PNDIS_BUFFER  pndisBuffer;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    UINT          uiIndex;
    UINT          uiPacketLength;
    UINT          uiBufferCount;
    UINT          uiPhysicalBufferCount;
    UINT          InterruptMask;
    int           nCount;

    // If the count of packets in the array is zero, exit after noting this
    // fact.
    if ( !uiCount )
    {

#ifdef DEBUG_COUNTER
        pHardware->m_nBad[ COUNT_BAD_SEND_ZERO ]++;
#endif
        return;
    }

    // Acquire the hardware to avoid having 2 threads messing with the adapter
    // at the same time.  If this can't happen, return all of the packets with
    // a failure indication and exit.
    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        for ( uiIndex = 0; uiIndex < uiCount; uiIndex++ )
        {
            pPacket = pPacketArray[ uiIndex ];
            NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_FAILURE );
        }
        return;
    }

    /* Save the current interrupt mask and block all interrupts. */
    InterruptMask = HardwareBlockInterrupt( pHardware );

    // The following 'for' loop handles one packet in the array each time
    // through.

    for ( uiIndex = 0; uiIndex < uiCount; uiIndex++ )
    {
        // Count packets sent.  Point pPacket to the next packet in the array.
#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_SEND_PACKET ]++;
#endif
        pPacket = pPacketArray[ uiIndex ];

#ifdef SEND_QUEUE
        // If there are packets waiting in the queue to be transmitted add this
        // packet to the end of that queue.  We have to keep things in order.
        if ( pAdapter->m_pPacketQueueTail )
        {
#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_SEND_QUEUE ]++;
#endif
            AddPacketToTransmitQueue( pAdapter, pPacket );
        }
        else
#endif
        {
            NdisQueryPacket( pPacket, &uiPhysicalBufferCount, &uiBufferCount,
                &pndisBuffer, &uiPacketLength );

            // If the returned packet length is zero, there is nothing to
            // transmit.
            if ( !uiPacketLength )
            {
#ifdef DEBUG_COUNTER
                pHardware->m_nGood[ COUNT_GOOD_SEND_ZERO ]++;
#endif
                NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_SUCCESS );
                continue;
            }

#if 1
            if ( ( int ) uiBufferCount > pAdapter->m_cnMapRegAvail )
#else
            if ( ( int ) uiBufferCount > 0  )
#endif
            {
                uiBufferCount = 2;
                nCount = HardwareAllocPacket( pHardware, uiPacketLength, 1 );
            }
            else
                nCount = HardwareAllocPacket( pHardware, uiPacketLength,
                    uiPhysicalBufferCount );
            if ( nCount > ( int ) uiBufferCount )
            {
                nsStatus = SendBuffers( pAdapter, pPacket, pndisBuffer,
                    uiPacketLength, uiBufferCount );

                NDIS_SET_PACKET_STATUS( pPacket, nsStatus );
            }
            else if ( !nCount )
            {

#ifdef SEND_QUEUE
                AddPacketToTransmitQueue( pAdapter, pPacket );

#else
                NDIS_SET_PACKET_STATUS( pPacket, NDIS_STATUS_RESOURCES );
#endif
            }
            else
            {
                nsStatus = SendPacket( pAdapter, pPacket, pndisBuffer,
                    uiPacketLength );

                NDIS_SET_PACKET_STATUS( pPacket, nsStatus );
            }
        }
    }

    /* Restore the interrupt mask. */
    HardwareSetInterrupt( pHardware, InterruptMask );

    ReleaseAdapter( pAdapter );
}  // MiniportSendPackets
