/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2007 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisOid.c - NDIS driver OID query functions.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         04/13/06    Add EEPROM access support.
    THa         06/02/06    Report statistics from MIB counters.
    THa         07/27/06    Version 1.4 supports NDIS 5.
    THa         12/18/07    Add NDIS 5.1 support.
   ---------------------------------------------------------------------------
*/


#include "target.h"
#include "NdisDevice.h"
#include "NdisOid.h"


/* -------------------------------------------------------------------------- */

#define OID_DEVICE_MODE  0xff0100cc

// List of supported OID for this driver

NDIS_OID SupportedOids[] =
{
    OID_GEN_SUPPORTED_LIST,
    OID_GEN_HARDWARE_STATUS,
    OID_GEN_MEDIA_SUPPORTED,
    OID_GEN_MEDIA_IN_USE,
    OID_GEN_MAXIMUM_LOOKAHEAD,
    OID_GEN_MAXIMUM_FRAME_SIZE,
    OID_GEN_LINK_SPEED,
    OID_GEN_TRANSMIT_BUFFER_SPACE,
    OID_GEN_RECEIVE_BUFFER_SPACE,
    OID_GEN_TRANSMIT_BLOCK_SIZE,
    OID_GEN_RECEIVE_BLOCK_SIZE,
    OID_GEN_VENDOR_ID,
    OID_GEN_VENDOR_DESCRIPTION,
    OID_GEN_CURRENT_PACKET_FILTER,
    OID_GEN_CURRENT_LOOKAHEAD,
    OID_GEN_DRIVER_VERSION,
    OID_GEN_MAXIMUM_TOTAL_SIZE,
    OID_GEN_PROTOCOL_OPTIONS,
    OID_GEN_MAC_OPTIONS,
    OID_GEN_MEDIA_CONNECT_STATUS,
    OID_GEN_MAXIMUM_SEND_PACKETS,
    OID_GEN_VENDOR_DRIVER_VERSION,

    OID_802_3_PERMANENT_ADDRESS,
    OID_802_3_CURRENT_ADDRESS,
    OID_802_3_MAXIMUM_LIST_SIZE,
    OID_802_3_MULTICAST_LIST,

    OID_GEN_XMIT_OK,
    OID_GEN_RCV_OK,
    OID_GEN_XMIT_ERROR,
    OID_GEN_RCV_ERROR,
    OID_GEN_RCV_NO_BUFFER,

    OID_GEN_DIRECTED_FRAMES_XMIT,
    OID_GEN_MULTICAST_FRAMES_XMIT,
    OID_GEN_BROADCAST_FRAMES_XMIT,
    OID_GEN_DIRECTED_FRAMES_RCV,
    OID_GEN_MULTICAST_FRAMES_RCV,
    OID_GEN_BROADCAST_FRAMES_RCV,

    OID_GEN_RCV_CRC_ERROR,

    OID_802_3_RCV_ERROR_ALIGNMENT,
    OID_802_3_XMIT_ONE_COLLISION,
    OID_802_3_XMIT_MORE_COLLISIONS,

    OID_802_3_XMIT_DEFERRED,
    OID_802_3_XMIT_MAX_COLLISIONS,
    OID_802_3_XMIT_LATE_COLLISIONS,

#if 0
    OID_GEN_PHYSICAL_MEDIUM,
#endif

#if (NDISVER >= 50)
     OID_PNP_CAPABILITIES,
     OID_PNP_SET_POWER,
     OID_PNP_QUERY_POWER,
     OID_PNP_ADD_WAKE_UP_PATTERN,
     OID_PNP_REMOVE_WAKE_UP_PATTERN,
     OID_PNP_ENABLE_WAKE_UP,
     OID_PNP_WAKE_UP_OK,
     OID_PNP_WAKE_UP_ERROR,
#endif
};

/* -------------------------------------------------------------------------- */

#if (NDISVER >= 50)
static
NDIS_STATUS AddWakeUpPattern (
    IN  PNDIS_ADAPTER           pAdapter,
    IN  PNDIS_PM_PACKET_PATTERN pPattern,
    OUT PUINT                   pulBytesRead )
{
    NDIS_STATUS     NdisStatus;
    PWAKE_UP_PACKET pWakeUpPacket;

    if ( pAdapter->m_cnWakeUpPacket < MAX_WAKE_UP_PACKET  &&
            pPattern->MaskSize <= MAX_PATTERN_LENGTH / 8  &&
            pPattern->PatternSize <= MAX_PATTERN_LENGTH )
    {
        int nIndex;

        for ( nIndex = 0; nIndex < MAX_WAKE_UP_PACKET; nIndex++ )
        {
            pWakeUpPacket = &pAdapter->WakeUpPacket[ nIndex ];
            if ( !pWakeUpPacket->NdisPacketPattern.MaskSize )
                break;
        }

        /* Move the packet pattern separately in case they are not
           continuous.
        */
        pAdapter->m_cnWakeUpPacket++;

        /* Copy the NDIS_PM_PACKET_PATTERN. */
        pWakeUpPacket->NdisPacketPattern = *pPattern;
        pWakeUpPacket->NdisPacketPattern.PatternOffset =
            sizeof( NDIS_PM_PACKET_PATTERN ) + pPattern->MaskSize;
        *pulBytesRead += sizeof( NDIS_PM_PACKET_PATTERN );

        /* Copy the MASK. */
        NdisMoveMemory(
            &pWakeUpPacket->MaskAndPattern[ 0 ], ( pPattern + 1 ),
            pPattern->MaskSize );
        *pulBytesRead += pPattern->MaskSize;

        /* Copy the PATTERN. */
        NdisMoveMemory(
            &pWakeUpPacket->MaskAndPattern[ pPattern->MaskSize ],
            ( PUCHAR ) pPattern + pPattern->PatternOffset,
            pPattern->PatternSize );
        *pulBytesRead += pPattern->PatternSize;

#if DBG_
{
    UINT i;

    for ( i = 0; i < pPattern->MaskSize; i++ )
        DBG_PRINT( "%02X ", pWakeUpPacket->MaskAndPattern[ i ]);
    DBG_PRINT( NEWLINE );
    for ( ; i < pPattern->MaskSize + pPattern->PatternSize; i++ )
        DBG_PRINT( "%02X ", pWakeUpPacket->MaskAndPattern[ i ]);
    DBG_PRINT( NEWLINE );
}
DBG_PRINT( "Added: %u"NEWLINE, *pulBytesRead );
#endif
        HardwareSetWolFrame( &pAdapter->m_Hardware, nIndex,
            pPattern->MaskSize,
            pWakeUpPacket->MaskAndPattern,
            pPattern->PatternSize,
            &pWakeUpPacket->MaskAndPattern[ pPattern->MaskSize ]);

        NdisStatus = NDIS_STATUS_SUCCESS;
    }
    else
    {
        *pulBytesRead = 0;
        NdisStatus = NDIS_STATUS_RESOURCES;
    }

    return NdisStatus;
}  /* AddWakeUpPattern */


static
NDIS_STATUS RemoveWakeUpPattern (
    IN  PNDIS_ADAPTER           pAdapter,
    IN  PNDIS_PM_PACKET_PATTERN pPattern,
    OUT PUINT                   pulBytesRead )
{
    NDIS_STATUS     NdisStatus;
    PWAKE_UP_PACKET pWakeUpPacket;
    UINT            nCount;
    int             nIndex;
    PUCHAR          pSource;
    PUCHAR          pTarget;

    for ( nIndex = 0; nIndex < MAX_WAKE_UP_PACKET; nIndex++ )
    {
        pWakeUpPacket = &pAdapter->WakeUpPacket[ nIndex ];
        if ( pPattern->MaskSize == pWakeUpPacket->NdisPacketPattern.MaskSize
                &&  pPattern->PatternSize ==
                pWakeUpPacket->NdisPacketPattern.PatternSize )
        {
            /* Compare the mask. */
            for ( nCount = 0,
                    pSource = ( PUCHAR ) pPattern +
                    sizeof( NDIS_PM_PACKET_PATTERN ),
                    pTarget = pWakeUpPacket->MaskAndPattern;
                    nCount < pWakeUpPacket->NdisPacketPattern.MaskSize;
                    nCount++ )
                if ( pSource[ nCount ] != pTarget[ nCount ] )
                    break;
            if ( nCount < pWakeUpPacket->NdisPacketPattern.MaskSize )
                continue;

            /* Compare the pattern. */
            for ( nCount = 0,
                    pSource = ( PUCHAR ) pPattern + pPattern->PatternOffset,
                    pTarget = &pWakeUpPacket->MaskAndPattern[
                        pPattern->MaskSize ];
                    nCount < pWakeUpPacket->NdisPacketPattern.PatternSize;
                    nCount++ )
                if ( pSource[ nCount ] != pTarget[ nCount ] )
                    break;
            if ( nCount == pWakeUpPacket->NdisPacketPattern.PatternSize )
            {
                pWakeUpPacket->NdisPacketPattern.MaskSize = 0;
                break;
            }
        }
    }

    if ( nIndex < MAX_WAKE_UP_PACKET )
    {
        pAdapter->m_cnWakeUpPacket--;

        *pulBytesRead = sizeof( NDIS_PM_PACKET_PATTERN ) +
            pPattern->MaskSize + pPattern->PatternSize;

#if DBG
DBG_PRINT( "Remove: %d, %u"NEWLINE, nIndex, *pulBytesRead );
#endif
        HardwareSetWolEnable( &pAdapter->m_Hardware,
            ( USHORT )( 1 << nIndex ), FALSE );
        NdisStatus = NDIS_STATUS_SUCCESS;
    }
    else
    {
        *pulBytesRead = 0;
        NdisStatus = NDIS_STATUS_NOT_ACCEPTED;
    }

    return NdisStatus;
}  /* RemoveWakeUpPattern */
#endif

/* -------------------------------------------------------------------------- */

/*
    MiniportQueryInformation

    Description:
        This function is used by NDIS to query adapter information by
        pre-defined NDIS_OIDs.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

        NDIS_OID Oid
            The NDIS_OID to process.

        PVOID pBuffer
            Pointer to NdisRequest->InformationBuffer into which store the
            result of the query.

        ULONG ulBufferLength
            The number of bytes in pBuffer.

        PULONG pulBytesWritten
            Buffer to store the number of bytes written into pBuffer.

        PULONG pulBytesNeeded
            Buffer to store the number of bytes needed to complete the request
            if the specified buffer is not big enough.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS MiniportQueryInformation (
    IN  NDIS_HANDLE hAdapterContext,
    IN  NDIS_OID    Oid,
    IN  PVOID       pBuffer,
    IN  ULONG       ulBufferLength,
    OUT PULONG      pulBytesWritten,
    OUT PULONG      pulBytesNeeded )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    NDIS_STATUS   nsStatus = NDIS_STATUS_SUCCESS;
    PHARDWARE     pHardware = &pAdapter->m_Hardware;
    PPORT_CONFIG  pPort = &pHardware->m_Port[ MAIN_PORT ];
    ULONGLONG     qCounter;
    UINT          ulGeneric;
    USHORT        usGeneric;
    UINT          uiMoveBytes = sizeof( UINT );
    PVOID         pMoveSource = ( PVOID ) &ulGeneric;

    *pulBytesWritten = 0;
    *pulBytesNeeded = 0;

    switch ( Oid )
    {
        case OID_GEN_MEDIA_CONNECT_STATUS:
            ulGeneric = pAdapter->m_ulNdisMediaState;
            break;

        case OID_GEN_PROTOCOL_OPTIONS:

#ifdef DBG
            DBG_PRINT( "protocol options"NEWLINE );
#endif
            break;

        case OID_GEN_MAXIMUM_SEND_PACKETS:
            ulGeneric = 4;

#ifdef DBG
            DBG_PRINT( "max send packets: %d"NEWLINE, ulGeneric );
#endif
            break;

        // counters
        case OID_GEN_XMIT_OK:
            qCounter = AdapterReadCounter( pAdapter, OID_COUNTER_XMIT_OK );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_RCV_OK:
            qCounter = AdapterReadCounter( pAdapter, OID_COUNTER_RCV_OK );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_XMIT_ERROR:
            qCounter = AdapterReadCounter( pAdapter, OID_COUNTER_XMIT_ERROR );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_RCV_ERROR:
            qCounter = AdapterReadCounter( pAdapter, OID_COUNTER_RCV_ERROR );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_RCV_NO_BUFFER:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_RCV_NO_BUFFER );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_DIRECTED_BYTES_XMIT:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_DIRECTED_BYTES_XMIT );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_DIRECTED_FRAMES_XMIT:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_DIRECTED_FRAMES_XMIT );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_BROADCAST_BYTES_XMIT:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_BROADCAST_BYTES_XMIT );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_BROADCAST_FRAMES_XMIT:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_BROADCAST_FRAME_XMIT );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_MULTICAST_FRAMES_XMIT:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_MULTICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_MULTICAST ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_DIRECTED_BYTES_RCV:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_DIRECTED_BYTES_RCV );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_DIRECTED_FRAMES_RCV:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_DIRECTED_FRAMES_RCV );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_BROADCAST_BYTES_RCV:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_BROADCAST_BYTES_RCV );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_BROADCAST_FRAMES_RCV:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_BROADCAST_FRAMES_RCV );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_MULTICAST_FRAMES_RCV:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_MULTICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_MULTICAST ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_RCV_CRC_ERROR:
            qCounter = AdapterReadCounter( pAdapter,
                OID_COUNTER_RCV_ERROR_CRC );
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_TRANSMIT_QUEUE_LENGTH:
            ulGeneric = AdapterTransmitQueueLength( pAdapter );

#ifdef DBG
            DBG_PRINT( "transmit queue len: %u"NEWLINE, ulGeneric );
#endif
            break;

        case OID_802_3_RCV_ERROR_ALIGNMENT:
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_ALIGNMENT_ERR ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_RX_ALIGNMENT_ERR ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_802_3_XMIT_ONE_COLLISION:
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_SINGLE_COLLISION ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_TX_SINGLE_COLLISION ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_802_3_XMIT_MORE_COLLISIONS:
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_MULTI_COLLISION ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_TX_MULTI_COLLISION ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_802_3_XMIT_DEFERRED:
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_DEFERRED ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_TX_DEFERRED ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_802_3_XMIT_MAX_COLLISIONS:
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_TOTAL_COLLISION ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_TX_TOTAL_COLLISION ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_802_3_XMIT_LATE_COLLISIONS:
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_LATE_COLLISION ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_TX_LATE_COLLISION ];
#endif
            uiMoveBytes = sizeof( ULONGLONG );
            pMoveSource = ( PVOID ) &qCounter;
            break;

        case OID_GEN_PHYSICAL_MEDIUM:
#if (NDISVER >= 51)
            ulGeneric = NdisPhysicalMediumUnspecified;
#else
            ulGeneric = NdisPhysicalMediumReserved0;
#endif
            break;

        case OID_GEN_SUPPORTED_LIST:
            pMoveSource = ( PVOID ) SupportedOids;
            uiMoveBytes = sizeof( SupportedOids );
            break;

        case OID_GEN_HARDWARE_STATUS:
        {
            NDIS_HARDWARE_STATUS HardwareStatus = NdisHardwareStatusReady;

#if 0
            HardwareStatus = NdisHardwareStatusNotReady;
            HardwareStatus = NdisHardwareStatusReset;
#endif
            pMoveSource = ( PVOID ) &HardwareStatus;
            uiMoveBytes = sizeof( NDIS_HARDWARE_STATUS );
            break;
        }

        case OID_GEN_MEDIA_SUPPORTED:
        case OID_GEN_MEDIA_IN_USE:
        {
            NDIS_MEDIUM Medium = NdisMedium802_3;

            pMoveSource = ( PVOID ) &Medium;
            uiMoveBytes = sizeof( NDIS_MEDIUM );
            break;
        }

        case OID_GEN_MAXIMUM_LOOKAHEAD:
            ulGeneric = NDIS_MAX_LOOKAHEAD;

#ifdef DBG
            DBG_PRINT( "max lookahead: %u"NEWLINE, ulGeneric );
#endif
            break;

        case OID_GEN_MAXIMUM_FRAME_SIZE:
            ulGeneric = MAX_ETHERNET_BODY_SIZE;
            break;

        case OID_GEN_LINK_SPEED:
            ulGeneric = pAdapter->m_Hardware.m_ulTransmitRate;
            break;

        case OID_GEN_TRANSMIT_BUFFER_SPACE:
            ulGeneric = TX_BUF_SIZE;
            break;

        case OID_GEN_RECEIVE_BUFFER_SPACE:
            ulGeneric = RX_BUF_SIZE;
            break;

        case OID_GEN_TRANSMIT_BLOCK_SIZE:
            ulGeneric = MAX_ETHERNET_BODY_SIZE + ETHERNET_HEADER_SIZE;
            break;

        case OID_GEN_RECEIVE_BLOCK_SIZE:
            ulGeneric = MAX_ETHERNET_BODY_SIZE + ETHERNET_HEADER_SIZE;
            break;

        case OID_GEN_VENDOR_ID:

#ifdef DBG
            DBG_PRINT( "vendor id"NEWLINE );
#endif
            NdisMoveMemory( &ulGeneric,
                pAdapter->m_Hardware.m_bPermanentAddress, 3 );
            ulGeneric &= 0x00FFFFFF;
            break;

        case OID_GEN_VENDOR_DESCRIPTION:

#ifdef DBG
            DBG_PRINT( "vendor descrip"NEWLINE );
#endif
            AdapterGetDescription( &pMoveSource, &uiMoveBytes );
            break;

        case OID_GEN_VENDOR_DRIVER_VERSION:
            ulGeneric = (( USHORT ) DRIVER_MAJOR_VERSION << 16 ) |
                ( USHORT ) DRIVER_MINOR_VERSION;
            break;

        case OID_GEN_CURRENT_PACKET_FILTER:
            ulGeneric = pAdapter->m_ulPacketFilter;
            break;

        case OID_GEN_CURRENT_LOOKAHEAD:

#ifdef DBG
            DBG_PRINT( "get max lookahead: %u"NEWLINE,
                pAdapter->m_ulMaxLookAhead );
#endif
            ulGeneric = pAdapter->m_ulMaxLookAhead;
            break;

        case OID_GEN_DRIVER_VERSION:
            usGeneric = (( USHORT ) NDIS_MAJOR_VERSION << 8 ) |
                ( USHORT ) NDIS_MINOR_VERSION;
            pMoveSource = &usGeneric;
            uiMoveBytes = sizeof( usGeneric );
            break;

        case OID_GEN_MAXIMUM_TOTAL_SIZE:
            ulGeneric = MAX_ETHERNET_BODY_SIZE + ETHERNET_HEADER_SIZE;
            break;

        case OID_GEN_MAC_OPTIONS:
            ulGeneric = ( UINT )( NDIS_MAC_OPTION_COPY_LOOKAHEAD_DATA |
                NDIS_MAC_OPTION_NO_LOOPBACK |
                NDIS_MAC_OPTION_RECEIVE_SERIALIZED );
#if 0
            ulGeneric |= NDIS_MAC_OPTION_FULL_DUPLEX;
            ulGeneric |= NDIS_MAC_OPTION_TRANSFERS_NOT_PEND;
#endif
            break;

        case OID_802_3_PERMANENT_ADDRESS:
            pMoveSource = ( PVOID ) pAdapter->m_Hardware.m_bPermanentAddress;
            uiMoveBytes = sizeof( pAdapter->m_Hardware.m_bPermanentAddress );
            break;

        case OID_802_3_CURRENT_ADDRESS:
            pMoveSource = ( PVOID ) pAdapter->m_Hardware.m_bOverrideAddress;
            uiMoveBytes = sizeof( pAdapter->m_Hardware.m_bOverrideAddress );
            break;

        case OID_802_3_MULTICAST_LIST:
            ulGeneric = ( UINT ) pAdapter->m_Hardware.m_bMulticastListSize;
            break;

        case OID_802_3_MAXIMUM_LIST_SIZE:
            ulGeneric = MAX_MULTICAST_LIST;
            break;

#if (NDISVER >= 50)
        case OID_PNP_CAPABILITIES:
        {
            uiMoveBytes = sizeof( NDIS_PNP_CAPABILITIES );
            pMoveSource = ( PVOID ) &pAdapter->PNP_Capabilities;
            break;
        }
        case OID_PNP_QUERY_POWER:
        {
            NDIS_DEVICE_POWER_STATE PowerState =
                *(( PNDIS_DEVICE_POWER_STATE ) pBuffer );

#if DBG
DBG_PRINT( "query power: %d"NEWLINE, PowerState );
#endif
            switch ( PowerState )
            {
                case NdisDeviceStateUnspecified:
                    break;
                case NdisDeviceStateD0:
                    break;
                case NdisDeviceStateD1:
                    break;
                case NdisDeviceStateD2:
                    break;
                case NdisDeviceStateD3:
                    break;
            }
            uiMoveBytes = 0;
            break;
        }
        case OID_PNP_ENABLE_WAKE_UP:
            ulGeneric = pAdapter->m_ulWakeUpEnable;
            break;
        case OID_PNP_WAKE_UP_OK:
            ulGeneric = pAdapter->m_ulWakeUpOk;
            break;
        case OID_PNP_WAKE_UP_ERROR:
            ulGeneric = pAdapter->m_ulWakeUpError;
            break;
#endif

        case OID_TCP_TASK_OFFLOAD:
            nsStatus = NDIS_STATUS_NOT_SUPPORTED;
            break;

        case OID_FFP_SUPPORT:

#if DBG
            DBG_PRINT( "Fast Forward Packet"NEWLINE );
#endif
            nsStatus = NDIS_STATUS_NOT_SUPPORTED;
            break;

        case OID_GEN_SUPPORTED_GUIDS:
            uiMoveBytes = 0;
            break;

        case OID_DEVICE_MODE:
            nsStatus = GetDeviceMode( pAdapter, Oid, pBuffer, ulBufferLength,
                pulBytesWritten, pulBytesNeeded );
            uiMoveBytes = 0;
            break;

        default:

#ifdef DBG
            DBG_PRINT( "query oid: %08x"NEWLINE, Oid );
#endif
            return NDIS_STATUS_FAILURE;
    }

    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        if ( uiMoveBytes > ulBufferLength )
        {
            // Not enough room in pBuffer. Punt
            *pulBytesNeeded = uiMoveBytes;
            nsStatus = NDIS_STATUS_INVALID_LENGTH;
        }
        else
        {
            // Store result.
            NdisMoveMemory( pBuffer, pMoveSource, uiMoveBytes );
            *pulBytesWritten += uiMoveBytes;
        }
    }
    return( nsStatus );
}  // MiniportQueryInformation

/* -------------------------------------------------------------------------- */

/*
    CheckInputBufferSize

    Description:
        This helper function checks the size of the buffer is big enough to
        be processed.

    Parameters:
        ULONG ulBufferLength
            The length of buffer.

        ULONG ulBufferSize
            The size of the buffer needed.

        PULONG pulBytesNeeded
            Buffer to store the number of bytes needed for the buffer.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS         Buffer is okay
        NDIS_STATUS_INVALID_LENGTH  Invalid input buffer data length
*/

static
NDIS_STATUS CheckInputBufferSize (
    ULONG  ulBufferLength,
    ULONG  ulBufferSize,
    PULONG pulBytesNeeded )
{
    NDIS_STATUS nsStatus = NDIS_STATUS_SUCCESS;

    if ( ulBufferLength != ulBufferSize )
    {
        nsStatus = NDIS_STATUS_INVALID_LENGTH;
        *pulBytesNeeded = ulBufferSize;
    }
    return( nsStatus );
}  // CheckInputBufferSize


/*
    MiniportSetInformation

    Description:
        This function is used by NDIS to set adapter information by
        pre-defined NDIS_OIDs.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

        NDIS_OID Oid
            The NDIS_OID to be set.

        PVOID pBuffer
            Pointer to NdisRequest->InformationBuffer which holds the data to
            be set.

        ULONG ulBufferLength
            The number of bytes in pBuffer.

        PULONG pulBytesRead
            Buffer to store the number of bytes read from pBuffer if the call
            is successful.

        PULONG pulBytesNeeded
            Buffer to store the number of bytes needed to complete the request
            if the specified buffer is not big enough.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS         Set Information was successful
        NDIS_STATUS_INVALID_LENGTH  Invalid input buffer data length
        NDIS_STATUS_INVALID_OID     Invalid OID
        NDIS_STATUS_NOT_SUPPORTED   Valid OID, operation not currently
                                    supported
        NDIS_STATUS_INVALID_DATA    Invalid data in the input buffer
*/

NDIS_STATUS MiniportSetInformation (
    IN  NDIS_HANDLE hAdapterContext,
    IN  NDIS_OID    Oid,
    IN  PVOID       pBuffer,
    IN  ULONG       ulBufferLength,
    OUT PULONG      pulBytesRead,
    OUT PULONG      pulBytesNeeded )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;
    NDIS_STATUS   nsStatus = NDIS_STATUS_SUCCESS;
    PUINT         pUlong = ( PUINT ) pBuffer;

    *pulBytesNeeded = 0;
    *pulBytesRead = 0;
    switch ( Oid )
    {
        case OID_802_3_MULTICAST_LIST:
            if ( ( ulBufferLength % MAC_ADDRESS_LENGTH ) != 0 )
            {
                nsStatus = NDIS_STATUS_INVALID_LENGTH;
                break;
            }

            if ( ( ulBufferLength / MAC_ADDRESS_LENGTH ) > MAX_MULTICAST_LIST )
            {

#if 0
                nsStatus = NDIS_STATUS_MULTICAST_FULL;

#else
                if ( !AcquireAdapter( pAdapter, TRUE ) )
                {
                    nsStatus = NDIS_STATUS_FAILURE;
                    break;
                }
                if ( !HardwareSetMulticast( &pAdapter->m_Hardware, TRUE ) )
                {
                    nsStatus = NDIS_STATUS_FAILURE;
                }
                ReleaseAdapter( pAdapter );
                *pulBytesRead = ulBufferLength;
#endif
                break;
            }

            pAdapter->m_Hardware.m_bMulticastListSize = ( UCHAR )
                ( ulBufferLength / MAC_ADDRESS_LENGTH );
            NdisMoveMemory( pAdapter->m_Hardware.m_bMulticastList, pBuffer,
                ulBufferLength );
            if ( !AcquireAdapter( pAdapter, TRUE ) )
            {
                nsStatus = NDIS_STATUS_FAILURE;
                break;
            }
            if ( !HardwareSetGroupAddress( &pAdapter->m_Hardware ) )
            {
                nsStatus = NDIS_STATUS_FAILURE;
            }
            ReleaseAdapter( pAdapter );
            *pulBytesRead = ulBufferLength;
            break;

        case OID_GEN_CURRENT_PACKET_FILTER:
        {
            UCHAR bPromiscuous;

            if ( !NT_SUCCESS( nsStatus = CheckInputBufferSize( ulBufferLength,
                    sizeof( UINT ), pulBytesNeeded )) )
            {
                break;
            }

#ifdef DBG
            DBG_PRINT( "set filter:%08x"NEWLINE, pUlong[ 0 ] );
#endif

            if ( pUlong[ 0 ] & pAdapter->m_ulPacketsNotAllowed )
            {
                nsStatus = NDIS_STATUS_NOT_SUPPORTED;
                break;
            }

            pAdapter->m_ulPacketFilter = pUlong[ 0 ];
            if ( pAdapter->m_ulPacketFilter )
            {
                if ( !AcquireAdapter( pAdapter, TRUE ) )
                {
                    nsStatus = NDIS_STATUS_FAILURE;
                    break;
                }
                if ( ( pAdapter->m_ulPacketFilter &
                        NDIS_PACKET_TYPE_ALL_MULTICAST ) )
                    bPromiscuous = TRUE;
                else
                    bPromiscuous = FALSE;
                if ( !HardwareSetMulticast( &pAdapter->m_Hardware,
                        bPromiscuous ) )
                {
                    nsStatus = NDIS_STATUS_FAILURE;
                }
                if ( ( pAdapter->m_ulPacketFilter &
                        ( NDIS_PACKET_TYPE_PROMISCUOUS |
                        NDIS_PACKET_TYPE_ALL_LOCAL )) )
                    bPromiscuous = TRUE;
                else
                    bPromiscuous = FALSE;
                if ( !HardwareSetPromiscuous( &pAdapter->m_Hardware,
                        bPromiscuous ) )
                {
                    nsStatus = NDIS_STATUS_FAILURE;
                }
                ReleaseAdapter( pAdapter );

                // Now that the packet filter has been received, allow receive
                // packets to be indicated to the protocol stack.
                pAdapter->m_bPacketFilterSet = TRUE;
            }
            else
            {
                pAdapter->m_bPacketFilterSet = FALSE;
            }
            *pulBytesRead = sizeof( UINT );
            break;
        }

        case OID_GEN_CURRENT_LOOKAHEAD:
            if ( !NT_SUCCESS( nsStatus = CheckInputBufferSize( ulBufferLength,
                    sizeof( UINT ), pulBytesNeeded )) )
            {
                break;
            }

            if ( pUlong[ 0 ] > NDIS_MAX_LOOKAHEAD )
            {
                nsStatus = NDIS_STATUS_INVALID_DATA;
                break;
            }

#ifdef DBG
            DBG_PRINT( "set max lookahead: %u, %u"NEWLINE,
                pAdapter->m_ulMaxLookAhead, pUlong[ 0 ]);
#endif
            if ( pAdapter->m_ulMaxLookAhead < pUlong[ 0 ] )
                pAdapter->m_ulMaxLookAhead = pUlong[ 0 ];

            *pulBytesRead = sizeof( UINT );
            break;

#if (NDISVER >= 51)
        case OID_GEN_MACHINE_NAME:

#ifdef DBG
            DBG_PRINT( "gen_machine_name"NEWLINE );
#endif
            *pulBytesRead = ulBufferLength;
            break;
#endif

        case OID_802_3_CURRENT_ADDRESS:
            if ( !NT_SUCCESS( nsStatus = CheckInputBufferSize( ulBufferLength,
                    MAC_ADDRESS_LENGTH, pulBytesNeeded )) )
            {
                break;
            }

            NdisMoveMemory( pAdapter->m_Hardware.m_bPermanentAddress, pBuffer,
                MAC_ADDRESS_LENGTH );
            *pulBytesRead = MAC_ADDRESS_LENGTH;
            break;

#if (NDISVER >= 50)
        case OID_PNP_SET_POWER:
            if ( !NT_SUCCESS( nsStatus = CheckInputBufferSize( ulBufferLength,
                    sizeof( NDIS_DEVICE_POWER_STATE ), pulBytesNeeded )) )
            {
                break;
            }

            // D0 power state (full power) is the only state supported
            pAdapter->NextPowerState =
                *(( PNDIS_DEVICE_POWER_STATE ) pBuffer );

#if DBG
DBG_PRINT( "set power: %u, %u; %u, %u"NEWLINE, ulBufferLength,
    sizeof( NDIS_DEVICE_POWER_STATE ), pAdapter->CurrentPowerState,
    pAdapter->NextPowerState );
#endif
            if ( pAdapter->CurrentPowerState != pAdapter->NextPowerState )
            {
                switch ( pAdapter->NextPowerState )
                {
                    case NdisDeviceStateD0:
                        DriverWake( pAdapter, pAdapter->CurrentPowerState );
                        break;
                    case NdisDeviceStateD1:
                    case NdisDeviceStateD2:
                    case NdisDeviceStateD3:
                        if ( NdisDeviceStateD0 == pAdapter->CurrentPowerState )
                        {
                            DriverSleep( pAdapter );
                        }
                        break;
                }
                pAdapter->CurrentPowerState = pAdapter->NextPowerState;
            }
            *pulBytesRead = ulBufferLength;
            break;

        case OID_PNP_ADD_WAKE_UP_PATTERN:

#if DBG_
DBG_PRINT( "add wakeup: %u"NEWLINE, ulBufferLength );
#endif
            return( AddWakeUpPattern( pAdapter,
                ( PNDIS_PM_PACKET_PATTERN ) pBuffer, pulBytesRead ));

        case OID_PNP_REMOVE_WAKE_UP_PATTERN:

#if DBG_
DBG_PRINT( "remove wakeup: %u"NEWLINE, ulBufferLength );
#endif
            return( RemoveWakeUpPattern( pAdapter,
                ( PNDIS_PM_PACKET_PATTERN ) pBuffer, pulBytesRead ));

        case OID_PNP_ENABLE_WAKE_UP:
            if ( !NT_SUCCESS( nsStatus = CheckInputBufferSize( ulBufferLength,
                    sizeof( UINT ), pulBytesNeeded )) )
            {
                break;
            }

            pAdapter->m_ulWakeUpEnable = pUlong[ 0 ];

#if DBG
            if ( pAdapter->m_ulWakeUpEnable & NDIS_PNP_WAKE_UP_MAGIC_PACKET )
                DBG_PRINT( "NDIS_PNP_WAKE_UP_MAGIC_PNP"NEWLINE );
            if ( pAdapter->m_ulWakeUpEnable & NDIS_PNP_WAKE_UP_PATTERN_MATCH )
                DBG_PRINT( "NDIS_PNP_WAKE_UP_PATTERN_MATCH"NEWLINE );
            if ( pAdapter->m_ulWakeUpEnable & NDIS_PNP_WAKE_UP_LINK_CHANGE )
                DBG_PRINT( "NDIS_PNP_WAKE_UP_LINK_CHANGE"NEWLINE );
#endif
            if ( pAdapter->m_ulWakeUpEnable & NDIS_PNP_WAKE_UP_MAGIC_PACKET )
                HardwareSetWolEnable( &pAdapter->m_Hardware, WOL_MAGIC_ENABLE,
                    TRUE );
            if ( pAdapter->m_ulWakeUpEnable & NDIS_PNP_WAKE_UP_PATTERN_MATCH )
            {
                int nIndex;

                for ( nIndex = 0; nIndex < MAX_WAKE_UP_PACKET; nIndex++ )
                    if ( pAdapter->WakeUpPacket[ nIndex ].
                            NdisPacketPattern.MaskSize )
                        HardwareSetWolEnable( &pAdapter->m_Hardware,
                            ( USHORT )( 1 << nIndex ), TRUE );
            }
            *pulBytesRead = ulBufferLength;
            break;
#endif

        case OID_GEN_NETWORK_LAYER_ADDRESSES:
        {
#if DBG
            PNETWORK_ADDRESS_LIST pAddrList =
                ( PNETWORK_ADDRESS_LIST ) pBuffer;
            int i, j;

DBG_PRINT( "network addr: %d"NEWLINE, pAddrList->AddressCount );
            for ( i = 0; i < pAddrList->AddressCount; i++ ) {
DBG_PRINT( "%d: ", pAddrList->Address[ i ].AddressType );
                for ( j = 0; j < pAddrList->Address[ i ].AddressLength; j++ ) {
DBG_PRINT( "%02X ", pAddrList->Address[ i ].Address[ j ]);
                }
DBG_PRINT( NEWLINE );
            }
#endif
            *pulBytesRead = ulBufferLength;
            break;
        }

        case OID_GEN_TRANSPORT_HEADER_OFFSET:
        {
#if DBG
            PTRANSPORT_HEADER_OFFSET pOffset =
                ( PTRANSPORT_HEADER_OFFSET ) pBuffer;
#endif

#if DBG
            if ( NDIS_PROTOCOL_ID_TCP_IP == pOffset->ProtocolType )
            {
                DBG_PRINT( "IP header offset: %d, %d"NEWLINE,
                    pOffset->ProtocolType, pOffset->HeaderOffset );
            }
#endif
            *pulBytesRead = ulBufferLength;
            break;
        }

        case OID_DEVICE_MODE:
            nsStatus = SetDeviceMode( pAdapter, Oid, pBuffer, ulBufferLength,
                pulBytesRead, pulBytesNeeded );
            break;

        default:

#ifdef DBG
            DBG_PRINT( "set oid: %08x"NEWLINE, Oid );
#endif
            return NDIS_STATUS_FAILURE;
    }
    return( nsStatus );
}  // MiniportSetInformation
