/* ---------------------------------------------------------------------------
          Copyright (c) 2003-2008 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    NdisDevice.c - NDIS driver device functions.

    Author      Date        Description
    THa         12/01/03    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         02/23/06    Updated for WinCE 5.0.
    THa         04/13/06    Add EEPROM access support.
    THa         06/02/06    Report statistics from MIB counters.
    THa         07/27/06    Version 1.4 supports NDIS 5.
    THa         10/04/06    Version 1.6 delays reporting driver reset status.
    THa         12/21/06    Check LED setting for hardware hang if
                            SET_DEFAULT_LED is set to a non-default setting.
    THa         12/18/07    Update for 64-bit systems.
    THa         02/15/08    Add force link support.
    THa         12/15/08    Fix getting SpeedDuplex parameter problem.
   ---------------------------------------------------------------------------
*/


#include "target.h"
#include "NdisDevice.h"


PHARDWARE phw;

/* -------------------------------------------------------------------------- */

/*
    GetConfigEntryUlong

    Description:
        This function retrieves a double-word value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PUINT pulValue
            Buffer to store the value.

    Return (BOOLEAN):
        TRUE if successful; otherwise FALSE.
*/

BOOLEAN GetConfigEntryUlong (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PUINT        pulValue )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    NDIS_PARAMETER_TYPE           parameter = NdisParameterHexInteger;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        parameter );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        *pulValue = pReturnedValue->ParameterData.IntegerData;
        return TRUE;
    }
    return FALSE;
}  // GetConfigEntryUlong


/*
    GetConfigEntryUshort

    Description:
        This function retrieves a word value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PUSHORT pusValue
            Buffer to store the value.

    Return (BOOLEAN):
        TRUE if successful; otherwise FALSE.
*/

BOOLEAN GetConfigEntryUshort (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PUSHORT      pusValue )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    NDIS_PARAMETER_TYPE           parameter = NdisParameterHexInteger;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        parameter );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        *pusValue = ( USHORT ) pReturnedValue->ParameterData.IntegerData;
        return TRUE;
    }
    return FALSE;
}  // GetConfigEntryUshort


/*
    GetConfigEntryUchar

    Description:
        This function retrieves a byte value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PUCHAR pbValue
            Buffer to store the value.

    Return (BOOLEAN):
        TRUE if successful; otherwise FALSE.
*/

BOOLEAN GetConfigEntryUchar (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PUCHAR       pbValue )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    NDIS_PARAMETER_TYPE           parameter = NdisParameterInteger;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        parameter );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        *pbValue = ( UCHAR ) pReturnedValue->ParameterData.IntegerData;
        return TRUE;
    }
    return FALSE;
}  // GetConfigEntryUchar


/*
    GetConfigEntryString

    Description:
        This function retrieves a string value from the registry.

    Parameters:
        NDIS_HANDLE hRegistry
            Handle to registry key.

        NDIS_STRING* EntryString
            Pointer to NDIS string indicating the registry value.

        PCHAR szValue
            Buffer to store the string value.

        USHORT wLength
            The length of the buffer.

    Return (int):
        The length of the string.
*/

int GetConfigEntryString (
    NDIS_HANDLE  hRegistry,
    NDIS_STRING* EntryString,
    PCHAR        szValue,
    USHORT       wLength )
{
    PNDIS_CONFIGURATION_PARAMETER pReturnedValue;
    NDIS_STATUS                   nsStatus;
    ANSI_STRING                   anszString;

    anszString.MaximumLength = wLength;
    anszString.Buffer = szValue;

    NdisReadConfiguration( &nsStatus, &pReturnedValue, hRegistry, EntryString,
        NdisParameterString );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {
        NdisUnicodeStringToAnsiString( &anszString,
            &pReturnedValue->ParameterData.StringData );
        return( anszString.Length );
    }
    return( 0 );
}  // GetConfigEntryString

/* -------------------------------------------------------------------------- */

#define nszcBusType                 NDIS_STRING_CONST( "BusType" )
#define nszcIoBaseAddress           NDIS_STRING_CONST( "IoBaseAddress" )
#define nszcInterruptNumber         NDIS_STRING_CONST( "InterruptNumber" )
#define nszcIoBaseLength            NDIS_STRING_CONST( "IoLen" )
#define nszcMemoryAddress           NDIS_STRING_CONST( "MemoryBaseAddress" )
#define nszcMemorySize              NDIS_STRING_CONST( "MemorySize" )

#define nszcNetworkAddress          NDIS_STRING_CONST( "NetworkAddress" )
#define nszcSpeedDuplex             NDIS_STRING_CONST( "SpeedDuplex" )

#define DEFAULT_IO_BASE_ADDRESS     0x300
#define DEFAULT_IO_BASE_LENGTH      0x10
#define DEFAULT_MEMORY_SIZE         0x1000
#define DEFAULT_MEMORY_WINDOW       0xd4000
#define DEFAULT_INTERRUPT_NUMBER    3
#define MAX_IOBASEADDR              0xFFFFFFFF
#define MIN_IOBASEADDR              0x0100
#define MAX_IRQ                     15
#define MIN_IRQ                     2


/*
    GetResources

    Description:
        This function gets adapter resources from the operating system and the
        registry.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        NDIS_HANDLE hConfiguration
            Handle to adapter configuration.

        NDIS_HANDLE hRegistry
            Handle to registry key.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS GetResources (
        PNDIS_ADAPTER pAdapter,
    IN  NDIS_HANDLE   hConfiguration,
    IN  NDIS_HANDLE   hRegistry )
{
    UCHAR               bBuffer[ sizeof( NDIS_RESOURCE_LIST ) +
        ( sizeof( CM_PARTIAL_RESOURCE_DESCRIPTOR ) * 10 )];
    PNDIS_RESOURCE_LIST pResource = ( PNDIS_RESOURCE_LIST ) bBuffer;
    UINT                uiBufferSize = sizeof( bBuffer );
    NDIS_STATUS         nsStatus;
    UINT                ulIndex;
    int                 NextIo = 0;
    PHARDWARE           pHardware = &pAdapter->m_Hardware;

    NdisMQueryAdapterResources( &nsStatus, hConfiguration, pResource,
        &uiBufferSize );
    if ( NDIS_STATUS_SUCCESS == nsStatus )
    {

#if 0
        // Assume Windows 9X
        pAdapter->ucOSEnvironment = WIN9X;
        // if NdisMQueryAdapterResources returns success check the OS environment
        NdisReadConfiguration ( &nsStatus,
                                &pncpReturnValue,
                                hRegistryHandle,
                                &nsEnvironment,
                                NdisParameterInteger);

        if (nsStatus == NDIS_STATUS_SUCCESS)
            // NT = NdisEnvironmentWindowsNt, 95 = NdisEnvironmentWindows.
            if (NdisEnvironmentWindowsNt == (UINT) pncpReturnValue->ParameterData.IntegerData)
                // must be Win2k
                pAdapter->ucOSEnvironment = WIN2K;
#endif

    }
    else
    {
#if 0
        // if NdisMQueryAdapterResources returns fail the OS is WinNT
        pAdapter->ucOSEnvironment = WINNT40;
#endif
    }
    if ( pResource )
    {
        // Parse the NDIS_RESOURCE_LIST (CM_PARTIAL_RESOURCE_LIST/
        // CM_PARTIAL_RESOURCE_DESCRIPTOR).
        for ( ulIndex = 0; ulIndex < pResource->Count; ulIndex++ )
        {
            // We handle 3 types of resources: I/O port, interrupt, and
            // memory base.  Ignore all others.
            switch ( pResource->PartialDescriptors[ ulIndex ].Type )
            {
                case CmResourceTypePort:
                        pAdapter->m_ulIOBaseAddress = ( UINT )
                            pResource->PartialDescriptors[ ulIndex ].u.Port.
                            Start.u.LowPart;
                        pAdapter->m_ulIOLength =
                            pResource->PartialDescriptors[ ulIndex ].u.Port.
                            Length;

#ifdef DBG
DbgPrint( "  io: %x, %x"NEWLINE, pAdapter->m_ulIOBaseAddress,
    pAdapter->m_ulIOLength );
#endif
                    break;

                case CmResourceTypeInterrupt:
                    pAdapter->m_ulInterruptNumber =
                        pResource->PartialDescriptors[ ulIndex ].u.Interrupt.
                        Vector;

#ifdef DBG
DbgPrint( "  intr: %d"NEWLINE, pAdapter->m_ulInterruptNumber );
#endif
                    break;

                case CmResourceTypeMemory:
                    pAdapter->m_ulMemoryAddress = ( ULONG )
                        pResource->PartialDescriptors[ ulIndex ].u.Memory.
                        Start.u.LowPart;
                    pAdapter->m_ulMemorySize =
                        pResource->PartialDescriptors[ ulIndex ].u.Memory.
                        Length;

#ifdef DBG
DbgPrint( "  mem: %lx, %x"NEWLINE, pAdapter->m_ulMemoryAddress,
    pAdapter->m_ulMemorySize );
#endif
                    break;
            }
        }
    }
    else
    {
        NDIS_STRING nszIoBaseAddress = nszcIoBaseAddress;
        NDIS_STRING nszIoBaseLength = nszcIoBaseLength;
        NDIS_STRING nszInterruptNumber = nszcInterruptNumber;
        NDIS_STRING nszMemoryAddress = nszcMemoryAddress;
        NDIS_STRING nszMemorySize = nszcMemorySize;
        UINT        uiAddr;

        pAdapter->m_ulIOBaseAddress = DEFAULT_IO_BASE_ADDRESS;
        GetConfigEntryUlong( hRegistry, &nszIoBaseAddress,
            &pAdapter->m_ulIOBaseAddress );
        if ( pAdapter->m_ulIOBaseAddress < MIN_IOBASEADDR  ||
                pAdapter->m_ulIOBaseAddress > MAX_IOBASEADDR )
        {
            NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
                NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1,
                pAdapter->m_ulIOBaseAddress );
            return( NDIS_STATUS_FAILURE );
        }

        pAdapter->m_ulIOLength = DEFAULT_IO_BASE_LENGTH;
        GetConfigEntryUlong( hRegistry, &nszIoBaseLength,
            &pAdapter->m_ulIOLength );

        pAdapter->m_ulInterruptNumber = DEFAULT_INTERRUPT_NUMBER;
        GetConfigEntryUlong( hRegistry, &nszInterruptNumber,
            &pAdapter->m_ulInterruptNumber );
        if ( pAdapter->m_ulInterruptNumber < MIN_IRQ  ||
                pAdapter->m_ulInterruptNumber > MAX_IRQ )
        {
            NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
                NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1,
                pAdapter->m_ulInterruptNumber );
            return( NDIS_STATUS_FAILURE );
        }

        pAdapter->m_ulMemoryAddress = DEFAULT_MEMORY_WINDOW;
        GetConfigEntryUlong( hRegistry, &nszMemoryAddress, &uiAddr );
        pAdapter->m_ulMemoryAddress = uiAddr;

        pAdapter->m_ulMemorySize = DEFAULT_MEMORY_SIZE;
        GetConfigEntryUlong( hRegistry, &nszMemorySize,
            &pAdapter->m_ulMemorySize );
    }
    return NDIS_STATUS_SUCCESS;
}  // GetResources


/*
    ReadConfiguration

    Description:
        This function gets configuration for the adapter from the operating
        system and the registry.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        NDIS_HANDLE hConfiguration
            Handle to adapter configuration.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS ReadConfiguration (
        PNDIS_ADAPTER pAdapter,
    IN  NDIS_HANDLE   hConfiguration )
{
    // Handle for reading from registry.
    NDIS_HANDLE hRegistry;
    NDIS_STATUS nsStatus;
    BOOLEAN     bStatus;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;
    NDIS_STRING nszBusType = nszcBusType;
    NDIS_STRING nszAddress = nszcNetworkAddress;
    NDIS_STRING nszSpeed = nszcSpeedDuplex;
    CHAR        szAddress[ 16 ];
    int         len;
    ULONG       ulData;
    USHORT      wData;

    // Open the configuration space.
    NdisOpenConfiguration( &nsStatus, &hRegistry, hConfiguration );
    if ( nsStatus != NDIS_STATUS_SUCCESS )
    {
        NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
            NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 0 );
        return( nsStatus );
    }

    // get bus type

#ifdef SH_BUS
    pAdapter->m_BusType = NdisInterfaceInternal;

#else
    pAdapter->m_BusType = NdisInterfaceIsa;
#endif
    bStatus = GetConfigEntryUlong( hRegistry, &nszBusType,
        ( PUINT ) &pAdapter->m_BusType );

#ifdef DBG
    if ( !bStatus )
        DbgPrint( " def" );
DbgPrint( "  bustype: %x"NEWLINE, pAdapter->m_BusType );
#endif

    len = GetConfigEntryString( hRegistry, &nszAddress, szAddress, 16 );
    if ( 12 == len )
    {
        UCHAR bData;
        int   index = 5;

#if DBG
        DBG_PRINT( "MAC: %s"NEWLINE, szAddress );
#endif
        while ( len-- > 0 )
        {
            if ( '0' <= szAddress[ len ]  &&  szAddress[ len ] <= '9' )
            {
                bData = szAddress[ len ] - '0';
                if ( ( len & 1 ) )
                    pHardware->m_bOverrideAddress[ index ] = bData;
                else
                    pHardware->m_bOverrideAddress[ index-- ] |= ( bData << 4 );
            }
            else if ( 'A' <= szAddress[ len ]  &&  szAddress[ len ] <= 'F' )
            {
                bData = szAddress[ len ] - 'A' + 10;
                if ( ( len & 1 ) )
                    pHardware->m_bOverrideAddress[ index ] = bData;
                else
                    pHardware->m_bOverrideAddress[ index-- ] |= ( bData << 4 );
            }
            else
                break;
        }
        if ( len < 0 )
        {
#if DBG
            PrintMacAddress( pHardware->m_bOverrideAddress );
            DBG_PRINT( NEWLINE );
#endif
            pHardware->m_bMacOverrideAddr = TRUE;
        }
    }

    bStatus = GetConfigEntryUlong( hRegistry, &nszSpeed, &ulData );
    if ( bStatus )
    {
        if ( ulData )
        {
            pHardware->m_bDuplex = ( UCHAR )( ulData & 3 );
            if ( ulData >= 0x10000 )
            {
                pHardware->m_bForceLink = 1;
                ulData -= 0x10000;
            }
            if ( ulData >= 0x100 )
            {
                pHardware->m_bSpeed = 100;
            }
            else if ( ulData >= 0x10 )
            {
                pHardware->m_bSpeed = 10;
            }
        }
#if DBG
        DBG_PRINT( "SpeedDuplex = %x; %u %u %u"NEWLINE, ulData,
            pHardware->m_bSpeed, pHardware->m_bDuplex,
            pHardware->m_bForceLink );
#endif
    }

    pHardware->m_pDevice = pAdapter;

    if ( ( nsStatus = GetResources( pAdapter, hConfiguration, hRegistry )) !=
            NDIS_STATUS_SUCCESS )
    {
        goto ReadConfigurationDone;
    }

ReadConfigurationDone:
    NdisCloseConfiguration( hRegistry );

    return( nsStatus );
}  // ReadConfiguration


/*
    DeRegisterAdapter

    Description:
        This routine de-registers the adapter from the operating system.  It
        releases the I/O address range used by the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID DeRegisterAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

#ifdef DBG
DbgPrint( "DeRegister"NEWLINE );
#endif

    if ( pHardware->m_pVirtualMemory )
    {
#ifdef UNDER_CE
        VirtualFree( pHardware->m_pVirtualMemory, 0, MEM_RELEASE );
        pHardware->m_ulVIoAddr = 0;

#else
        NdisMUnmapIoSpace( pAdapter->m_hAdapter,
            pHardware->m_pVirtualMemory,
            pAdapter->m_ulMemorySize );
#endif
        pHardware->m_pVirtualMemory = NULL;
    }
    if ( pHardware->m_ulVIoAddr )
    {
        NdisMDeregisterIoPortRange( pAdapter->m_hAdapter,
            pAdapter->m_ulIOBaseAddress, pAdapter->m_ulIOLength,
            ( PVOID ) pHardware->m_ulVIoAddr );
        pHardware->m_ulVIoAddr = 0;
    }
}  // DeRegisterAdapter


/*
    RegisterAdapter

    Description:
        This function registers the adapter to the NDIS wrapper.  It acquires
        the I/O address range to run the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS RegisterAdapter (
    PNDIS_ADAPTER pAdapter )
{
    NDIS_STATUS nsStatus = NDIS_STATUS_SUCCESS;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;

#ifdef UNDER_CE
    BOOL        bResult;
#endif

#ifdef DBG
DbgPrint( "Register"NEWLINE );
#endif

#ifndef NDIS51_MINIPORT
    NdisMSetAttributes( pAdapter->m_hAdapter, ( NDIS_HANDLE ) pAdapter, FALSE,
        pAdapter->m_BusType );

#else
    NdisMSetAttributesEx( pAdapter->m_hAdapter, ( NDIS_HANDLE ) pAdapter, 0,
        NDIS_ATTRIBUTE_SURPRISE_REMOVE_OK, pAdapter->m_BusType );
#endif

    if ( pAdapter->m_ulIOLength )
    {
#ifdef UNDER_CE
        pHardware->m_pVirtualMemory = VirtualAlloc( 0, PAGE_SIZE, MEM_RESERVE,
            PAGE_NOACCESS );
        DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: VirtualAlloc at 0x%x.\r\n" ),
            pHardware->m_pVirtualMemory ));
        if ( !pHardware->m_pVirtualMemory )
            goto RegisterAdapterError;
        bResult = VirtualCopy( pHardware->m_pVirtualMemory,
            ( PVOID )( pAdapter->m_ulIOBaseAddress & ( ~( PAGE_SIZE - 1 ))),
            PAGE_SIZE, PAGE_READWRITE | PAGE_NOCACHE );
        pHardware->m_ulVIoAddr = ( ULONG ) pHardware->m_pVirtualMemory +
            ( pAdapter->m_ulIOBaseAddress & ( PAGE_SIZE - 1 ));
        DEBUGMSG( ZONE_INIT, ( TEXT( "KS884X: VirtualCopy at 0x%x.\r\n" ),
            pHardware->m_ulVIoAddr ));

#else
        nsStatus = NdisMRegisterIoPortRange(( PVOID* ) &pHardware->m_ulVIoAddr,
            pAdapter->m_hAdapter, pAdapter->m_ulIOBaseAddress,
            pAdapter->m_ulIOLength );
#endif
        if ( nsStatus != NDIS_STATUS_SUCCESS )
        {
#ifndef UNDER_CE
            NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
                NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 3, 0x10560010, 0x1,
                pAdapter->m_ulIOBaseAddress );
#endif
            goto RegisterAdapterError;
        }
    }

    return( nsStatus );

RegisterAdapterError:
    DeRegisterAdapter( pAdapter );
    return( nsStatus );
}  // RegisterAdapter


/*
    InitAdapter

    Description:
        This function initializes the adapter when the adapter is started.  It
        calls ReadConfiguration() to get all information required to start the
        adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        NDIS_HANDLE hAdapter
            Handle to adapter context containing adapter information.

        NDIS_HANDLE hConfiguration
            Handle to adapter configuration.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS InitAdapter (
        PNDIS_ADAPTER pAdapter,
    IN  NDIS_HANDLE   hAdapter,
    IN  NDIS_HANDLE   hConfiguration )
{
    NDIS_STATUS nsStatus;

    // Save the adapter handle furnished by NDIS.
    pAdapter->m_hAdapter = hAdapter;

    nsStatus = ReadConfiguration( pAdapter, hConfiguration );
    if ( !NT_SUCCESS( nsStatus ) )
    {
        return( nsStatus );
    }
    return NDIS_STATUS_SUCCESS;
}  // InitAdapter


/*
    SetupAdapter

    Description:
        This function setups the adapter when the adapter is started.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise an error code indicating
        failure.
*/

NDIS_STATUS SetupAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;
    UCHAR     bPromiscuous;

    if ( !AcquireAdapter( pAdapter, FALSE ) )
    {
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }
    if ( !HardwareInitialize( pHardware )  ||  !HardwareReset( pHardware ) )
    {
        NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
            NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1, 0x10560100 );
        ReleaseAdapter( pAdapter );
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }

    HardwareDisable( pHardware );
    HardwareSetPowerSaving( pHardware, FALSE );

#if 0
    HardwareGetVersion( pHardware );
#endif

    // Read the Ethernet address.
    if ( !HardwareReadAddress( pHardware ) )
    {
        NdisWriteErrorLogEntry( pAdapter->m_hAdapter,
            NDIS_ERROR_CODE_UNSUPPORTED_CONFIGURATION, 1, 0x10560101 );
        ReleaseAdapter( pAdapter );
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }

    pHardware->m_ulHardwareState = MediaStateDisconnected;

    /* Initialize to invalid value so that link detection is done. */
    pHardware->m_PortInfo[ MAIN_PORT ].bLinkPartner = 0xFF;
    pHardware->m_PortInfo[ MAIN_PORT ].ulHardwareState =
        MediaStateDisconnected;

#ifdef DEF_KS8842
    pHardware->m_PortInfo[ OTHER_PORT ].bLinkPartner = 0xFF;
    pHardware->m_PortInfo[ OTHER_PORT ].ulHardwareState =
        MediaStateDisconnected;
#endif

    HardwareSetup( pHardware );
    HardwareSwitchSetup( pHardware );

    if ( ( pAdapter->m_ulPacketFilter &
            ( NDIS_PACKET_TYPE_PROMISCUOUS | NDIS_PACKET_TYPE_ALL_LOCAL )) )
        bPromiscuous = TRUE;
    else
        bPromiscuous = FALSE;
    if ( !HardwareSetPromiscuous( pHardware, bPromiscuous ) )
    {
        ReleaseAdapter( pAdapter );
        return NDIS_STATUS_ADAPTER_NOT_FOUND;
    }

    pAdapter->m_ulNdisMediaState = pAdapter->m_Hardware.m_ulHardwareState;

    PortInitCounters( pHardware, MAIN_PORT );

#if defined( DEF_KS8842 )  &&  !defined( TWO_NETWORK_INTERFACE )
    PortInitCounters( pHardware, OTHER_PORT );
#endif
    PortInitCounters( pHardware, HOST_PORT );

    HardwareSetupInterrupt( pHardware );

#ifdef DBG
DbgPrint( "Setup completed"NEWLINE );
#endif

#if 0
    HardwareGetInfo( pHardware );
#endif

    ReleaseAdapter( pAdapter );

#if (NDISVER >= 50)
    /* Specifies the lowest device power state from which the
       miniport's NIC can signal a wake-up on receipt of a Magic
       Packet.  (A Magic Packet is a packet that contains 16 contiguous
       copies of the receiving NIC's Ethernet address.)
    */
#ifdef DEF_KS8841
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinMagicPacketWakeUp =
        NdisDeviceStateD3;
#else
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinMagicPacketWakeUp =
        NdisDeviceStateUnspecified;
#endif

    /* Specifies the lowest device power state from which the
       miniport's NIC can signal a wake-up event on receipt of a
       network frame that contains a pattern specified by the protocol
       driver.
    */
#ifdef DEF_KS8841
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinPatternWakeUp =
        NdisDeviceStateD3;
#else
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinPatternWakeUp =
        NdisDeviceStateUnspecified;
#endif

    /* Specifies the lowest device power state from which the
       miniport's NIC can signal a wake-up event in response to a link
       change (the connection or disconnection of the NIC's network
       cable).
    */
#ifdef DEF_KS8841
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinLinkChangeWakeUp =
        NdisDeviceStateD3;
#else
    pAdapter->PNP_Capabilities.WakeUpCapabilities.MinLinkChangeWakeUp =
        NdisDeviceStateUnspecified;
#endif

    pAdapter->CurrentPowerState = pAdapter->NextPowerState =
        NdisDeviceStateD0;
#endif

    phw = pHardware;

    return NDIS_STATUS_SUCCESS;
}  // SetupAdapter


/*
    ShutdownAdapter

    Description:
        This routine shutdowns the adapter when the driver is unloaded.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void ShutdownAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

#ifndef UNDER_CE
#ifdef DBG_
    int       i;
#endif
#endif

    AcquireAdapter( pAdapter, FALSE );
    HardwareDisableInterrupt( pHardware );
    HardwareDisable( pHardware );
    ReleaseAdapter( pAdapter );

#ifndef UNDER_CE
#ifdef DBG_
    DbgPrint( "counters:\n" );
    for ( i = OID_COUNTER_FIRST; i < OID_COUNTER_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_cnCounter[ 0 ][ i ]);
    }
    DbgPrint( "wait delays:\n" );
    for ( i = WAIT_DELAY_FIRST; i < WAIT_DELAY_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_nWaitDelay[ i ]);
    }
    DbgPrint( "bad:\n" );
    for ( i = COUNT_BAD_FIRST; i < COUNT_BAD_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_nBad[ i ]);
    }
    DbgPrint( "good:\n" );
    for ( i = COUNT_GOOD_FIRST; i < COUNT_GOOD_LAST; i++ )
    {
        DbgPrint( "%u = %u\n", i, pHardware->m_nGood[ i ]);
    }
#endif
#endif
}  // ShutdownAdapter


/*
    StartAdapter

    Description:
        This routine starts the adapter when the driver is woke up.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void StartAdapter (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    AcquireAdapter( pAdapter, FALSE );
    HardwareEnable( pHardware );
    HardwareEnableInterrupt( pHardware );
    ReleaseAdapter( pAdapter );
}  // StartAdapter

/* -------------------------------------------------------------------------- */

/*
    MiniportSynchronizeISR

    Description:
        This function is used to do things not interruptable by software
        interrupt.

    Parameters:
        PVOID Context
            Pointer to hardware information structure.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN MiniportSynchronizeISR (
    IN  PVOID Context )
{
    PHARDWARE pHardware = ( PHARDWARE ) Context;

    HardwareDisableInterrupt( pHardware );
    return TRUE;
}  // MiniportSynchronizeISR


/*
    HardwareDisableInterruptSync

    Description:
        This function is used to disable hardware interrupt atomically so that
        the operation is not interrupted.

    Parameters:
        void* pHardware
            Pointer to hardware information structure.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

void HardwareDisableInterruptSync (
    void* pHardware )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER )(( PHARDWARE ) pHardware )->
        m_pDevice;

    NdisMSynchronizeWithInterrupt( &pAdapter->m_Interrupt,
        MiniportSynchronizeISR, pHardware );
}  /* HardwareDisableInterruptSync */

/* -------------------------------------------------------------------------- */

#define ADAPTER_LOCK_TIMEOUT  100

/*
    AcquireAdapter

    Description:
        This function is used to acquire the adapter so that only one process
        has access to the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        BOOLEAN fWait
            Indicate whether to wait or not.  User functions should wait, while
            periodic timer routine should not.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN AcquireAdapter (
    IN  PNDIS_ADAPTER pAdapter,
    IN  BOOLEAN       fWait )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;
    int       cnTimeOut = ADAPTER_LOCK_TIMEOUT;

    while ( --cnTimeOut )
    {
        NdisAcquireSpinLock( &pAdapter->m_lockHardware );

        if ( !pHardware->m_bAcquire )
        {
            pHardware->m_bAcquire = TRUE;
            NdisReleaseSpinLock( &pAdapter->m_lockHardware );
            return TRUE;
        }
        NdisReleaseSpinLock( &pAdapter->m_lockHardware );
        NdisMSleep( 10 );
    }
    if ( fWait )
    {

    }
    return FALSE;
}  // AcquireAdapter


/*
    ReleaseAdapter

    Description:
        This routine is used to release the adapater so that other process can
        have access to the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void ReleaseAdapter (
    IN  PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    NdisAcquireSpinLock( &pAdapter->m_lockHardware );
    pHardware->m_bAcquire = FALSE;
    NdisReleaseSpinLock( &pAdapter->m_lockHardware );
}  // ReleaseAdapter

/* -------------------------------------------------------------------------- */

/*
    AdapterCheckForHang

    Description:
        This function checks the adapter is working properly.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (BOOLEAN):
        TRUE if adapter is not working; FALSE if working properly.
*/

BOOLEAN AdapterCheckForHang (
    IN  PNDIS_ADAPTER pAdapter )
{
#if SET_DEFAULT_LED
    PHARDWARE pHardware = &pAdapter->m_Hardware;
    USHORT    wData;

    HW_READ_WORD( pHardware, REG_SWITCH_CTRL_5_OFFSET, &wData );
    if ( ( TO_LO_BYTE( wData ) & SET_DEFAULT_LED ) != SET_DEFAULT_LED )
    {
        return TRUE;
    }
#endif

    return FALSE;
}  // AdapterCheckForHang


/*
    AdapterGetDescription

    Description:
        This routine returns the adapter description.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

VOID AdapterGetDescription (
    PVOID* ppSource,
    PUINT  puiLength )
{
#ifdef DEF_KS8841
    static char gszDescription[] = "KSZ8841 Ethernet";

#else
    static char gszDescription[] = "KSZ8842 Ethernet";
#endif

    *ppSource = gszDescription;
    *puiLength = strlen( gszDescription );
}  // AdapterGetDescription


/*
    AdapterReadCounter

    Description:
        This function retrieves the statistics counter value.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

        EOidCounter OidCounter
            The counter index.

    Return (ULONGLONG):
        The counter value.
*/

ULONGLONG AdapterReadCounter (
    PNDIS_ADAPTER pAdapter,
    EOidCounter   OidCounter )
{
    PHARDWARE    pHardware = &pAdapter->m_Hardware;
    PPORT_CONFIG pPort = &pHardware->m_Port[ MAIN_PORT ];
    ULONGLONG    qCounter = pHardware->m_cnCounter[ MAIN_PORT ][ OidCounter ];

    switch ( OidCounter )
    {
        case OID_COUNTER_XMIT_OK:

#if 0
#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_RX_MULTICAST ] +
                pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_TX_MULTICAST ] +
                pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];
#endif
#endif
#if DBG
            DBG_PRINT( "xmit ok: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_RCV_OK:

#if 0
#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ] +
                pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];
#endif
#endif
#if DBG
            DBG_PRINT( "rcv ok: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_DIRECTED_FRAMES_XMIT:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];
#endif
#if DBG
            DBG_PRINT( "xmit frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_BROADCAST_FRAME_XMIT:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ];
#endif
#if DBG
            DBG_PRINT( "broadcast frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_DIRECTED_FRAMES_RCV:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_UNICAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_UNICAST ];
#endif
#if DBG
            DBG_PRINT( "rcv frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_BROADCAST_FRAMES_RCV:

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ HOST_PORT ];
            qCounter = pPort->cnCounter[ MIB_COUNTER_TX_BROADCAST ];

#else
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_BROADCAST ];
#endif
#if DBG
            DBG_PRINT( "rcv broadcast frames: %u"NEWLINE, ( UINT ) qCounter );
#endif
            break;
        case OID_COUNTER_RCV_ERROR_CRC:
            qCounter = pPort->cnCounter[ MIB_COUNTER_RX_CRC_ERR ];

#if defined( DEF_KS8842 )
            pPort = &pHardware->m_Port[ OTHER_PORT ];
            qCounter += pPort->cnCounter[ MIB_COUNTER_RX_CRC_ERR ];
#endif
            break;
    }

#ifndef UNDER_CE
#ifdef DBG
    switch ( OidCounter )
    {
        case OID_COUNTER_DIRECTED_BYTES_XMIT:
            DBG_PRINT( "xmit bytes: %u"NEWLINE, ( UINT ) qCounter );
            break;
        case OID_COUNTER_BROADCAST_BYTES_XMIT:
            DBG_PRINT( "broadcast bytes: %u"NEWLINE, ( UINT ) qCounter );
            break;
        case OID_COUNTER_DIRECTED_BYTES_RCV:
            DBG_PRINT( "rcv bytes: %u"NEWLINE, ( UINT ) qCounter );
            break;
        case OID_COUNTER_BROADCAST_BYTES_RCV:
            DBG_PRINT( "rcv broadcast: %u"NEWLINE, ( UINT ) qCounter );
            break;
    }
#endif
#endif

    return( qCounter );
}  // AdapteReadCounter


/*
    AdapterReset

    Description:
        This function resets the adapter.

    Parameters:
        PNDIS_ADAPTER pAdapter
            Pointer to adapter information structure.

    Return (BOOLEAN):
        TRUE if successful; otherwise, FALSE.
*/

BOOLEAN AdapterReset (
    PNDIS_ADAPTER pAdapter )
{
    PHARDWARE pHardware = &pAdapter->m_Hardware;

    AcquireAdapter( pAdapter, FALSE );
    if ( NdisDeviceStateD0 == pAdapter->CurrentPowerState )
    {
        HardwareDisableInterrupt( pHardware );
        HardwareDisable( pHardware );

        HardwareReset( pHardware );
    }
    {
        pHardware->m_ulHardwareState = MediaStateDisconnected;

        /* Initialize to invalid value so that link detection is done. */
        pHardware->m_PortInfo[ MAIN_PORT ].bLinkPartner = 0xFF;
        pHardware->m_PortInfo[ MAIN_PORT ].ulHardwareState =
            MediaStateDisconnected;

#ifdef DEF_KS8842
        pHardware->m_PortInfo[ OTHER_PORT ].bLinkPartner = 0xFF;
        pHardware->m_PortInfo[ OTHER_PORT ].ulHardwareState =
            MediaStateDisconnected;
#endif

        HardwareSetup( pHardware );
        HardwareSwitchSetup( pHardware );

        /* Reset may wipe out these registers. */
        if ( pHardware->m_bMacOverrideAddr )
            HardwareSetAddress( pHardware );
        if ( pHardware->m_bMulticastListSize )
            HardwareSetGroupAddress( pHardware );

#if 0
        memset(( void* ) pHardware->m_cnCounter[ MAIN_PORT ], 0,
            ( sizeof( ULONGLONG ) * OID_COUNTER_LAST ) );
        PortInitCounters( pHardware, MAIN_PORT );

#if defined( DEF_KS8842 )  &&  !defined( TWO_NETWORK_INTERFACE )
        PortInitCounters( pHardware, OTHER_PORT );
#endif
        PortInitCounters( pHardware, HOST_PORT );
#endif

        HardwareEnable( pHardware );
        HardwareEnableInterrupt( pHardware );
        ReleaseAdapter( pAdapter );
        return TRUE;
    }
    ReleaseAdapter( pAdapter );
    return FALSE;
}  // AdapterReset

/* -------------------------------------------------------------------------- */

/*
    MiniportReset

    Description:
        This function is used by NDIS to reset the adapter.

    Parameters:
        PBOOLEAN AddresssingReset
            Flag to indicate adapter needs the addressing information reloaded.

        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (NDIS_STATUS):
        NDIS_STATUS_SUCCESS if successful; otherwise, an error code indicating
        failure.
*/

NDIS_STATUS MiniportReset (
    OUT PBOOLEAN    AddressingReset,
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

    *AddressingReset = FALSE;
    if ( AdapterReset( pAdapter ) )
    {
        pAdapter->m_ulDriverState |= DRIVER_STATE_RESET;
        NdisMSetTimer( &pAdapter->m_ResetTimer, 2000 );
        return( NDIS_STATUS_PENDING );
    }
    return( NDIS_STATUS_FAILURE );
}  // MiniportReset

/* -------------------------------------------------------------------------- */

/*
    MiniportCheckForHang

    Description:
        This function is used by NDIS to check if the adapter is not working.
        The NDIS library calls this function every 2 seconds.  If TRUE is
        returned, it will call MiniportReset to force an adapter reset.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (BOOLEAN):
        TRUE if adapter requires a reset; FALSE if adapter is functioning
        properly.
*/

BOOLEAN MiniportCheckForHang (
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

    return( AdapterCheckForHang( pAdapter ));
}  // MiniportCheckForHang

/* -------------------------------------------------------------------------- */

/*
    MiniportDisableInterrupt

    Description:
        This routine is used by NDIS to disable hardware interrupts.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

VOID MiniportDisableInterrupt (
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

#ifdef DEBUG_INTERRUPT
    DbgPrint( "dis int"NEWLINE );
#endif
    HardwareSaveBank( &pAdapter->m_Hardware );
    HardwareDisableInterrupt( &pAdapter->m_Hardware );
    HardwareRestoreBank( &pAdapter->m_Hardware );
}  // MiniportDisableInterrupt


/*
    MiniportEnableInterrupt

    Description:
        This routine is used by NDIS to enable hardware interrupts.

    Parameters:
        NDIS_HANDLE hAdapaterContext
            Handle to adapter context containing adapter information.

    Return (None):
*/

VOID MiniportEnableInterrupt (
    IN  NDIS_HANDLE hAdapterContext )
{
    PNDIS_ADAPTER pAdapter = ( PNDIS_ADAPTER ) hAdapterContext;

#ifdef DEBUG_INTERRUPT
    DbgPrint( "ena int"NEWLINE );
#endif
    HardwareEnableInterrupt( &pAdapter->m_Hardware );
}  // MiniportEnableInterrupt

/* -------------------------------------------------------------------------- */

#pragma pack(1)
typedef struct
{
    // Total command length
    UINT   dwLength;

    // Driver Extension Command
    USHORT wCommand;

    // Option flags
    USHORT wFlags;

    // Driver Extension Return Code
    UINT   dwReturnCode;

    // Parameters
    UCHAR  bParam[ 4 ];
} TDeviceInfo, *PTDeviceInfo;
#pragma pack()


#define DEVICE_COMMAND_INIT       0
#define DEVICE_COMMAND_GET        1
#define DEVICE_COMMAND_SET        2
#define DEVICE_FLAG_OFF           0
#define DEVICE_FLAG_CABLE_STATUS  1
#define DEVICE_FLAG_LINK_STATUS   2
#define DEVICE_FLAG_EEPROM_REG    3

#define DEVICE_FLAG_CAPABILITIES  1

#define DEVICE_COMMAND_COUNTERS   3

#define DEVICE_INIT_SIZE          ( sizeof( TDeviceInfo ) + 8 )
#define DEVICE_INIT_RETURN_SIZE   ( sizeof( TDeviceInfo ) + 4 )
//#define DEVICE_GET_SIZE          ( sizeof( TDeviceInfo ) + 16 )
#define DEVICE_GET_SIZE           ( sizeof( TDeviceInfo ) + 0 )
#define DEVICE_CABLE_STATUS_SIZE  ( sizeof( TDeviceInfo ) + 10 * sizeof( UINT ))
#define DEVICE_LINK_STATUS_SIZE   ( sizeof( TDeviceInfo ) + 6 * sizeof( UINT ))
#define DEVICE_COUNTERS_SIZE      ( sizeof( TDeviceInfo ) + 4 * sizeof( UINT ))


NDIS_STATUS GetDeviceMode (
    IN  PNDIS_ADAPTER pAdapter,
    IN  NDIS_OID      Oid,
    IN  PVOID         pBuffer,
    IN  ULONG         ulLen,
    OUT PULONG        pulBytesWritten,
    OUT PULONG        pulBytesNeeded )
{
    NDIS_STATUS status = NDIS_STATUS_SUCCESS;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;

    // make sure input buffer is valid
    if ( pBuffer  &&  ulLen >= sizeof( TDeviceInfo ) )
    {
        PTDeviceInfo pInfo = ( PTDeviceInfo ) pBuffer;

        *pulBytesNeeded = 0;
        *pulBytesWritten = sizeof( TDeviceInfo );
        if ( Oid == pInfo->dwReturnCode )
        {
            if ( DEVICE_COMMAND_GET == pInfo->wCommand )
            {
                switch ( pInfo->wFlags )
                {
                    case DEVICE_FLAG_CABLE_STATUS:
                        if ( ulLen >= DEVICE_CABLE_STATUS_SIZE )
                        {
                            AcquireAdapter( pAdapter, TRUE );
                            HardwareGetCableStatus( pHardware, 0,
                                pInfo->bParam );
                            ReleaseAdapter( pAdapter );
                            *pulBytesWritten = DEVICE_CABLE_STATUS_SIZE;
                        }
                        else
                        {
                            *pulBytesNeeded = DEVICE_CABLE_STATUS_SIZE;
                            status = NDIS_STATUS_BUFFER_TOO_SHORT;
                        }
                        break;
                    case DEVICE_FLAG_LINK_STATUS:
                        if ( ulLen >= DEVICE_LINK_STATUS_SIZE )
                        {
                            AcquireAdapter( pAdapter, TRUE );
                            HardwareGetLinkStatus( pHardware, 0,
                                pInfo->bParam );
                            ReleaseAdapter( pAdapter );
                            *pulBytesWritten = DEVICE_LINK_STATUS_SIZE;
                        }
                        else
                        {
                            *pulBytesNeeded = DEVICE_LINK_STATUS_SIZE;
                            status = NDIS_STATUS_BUFFER_TOO_SHORT;
                        }
                        break;
                    case DEVICE_FLAG_EEPROM_REG:
                    {
                        PUSHORT pwParam = ( PUSHORT ) pInfo->bParam;

                        AcquireAdapter( pAdapter, TRUE );
                        pwParam[ 1 ] = EepromReadWord( pHardware,
                            ( UCHAR ) pwParam[ 0 ]);
                        ReleaseAdapter( pAdapter );
                        *pulBytesWritten = sizeof( TDeviceInfo );
                        break;
                    }
                    default:
                        status = NDIS_STATUS_INVALID_DATA;
                }
            }
            else if ( DEVICE_COMMAND_INIT == pInfo->wCommand )
            {
                if ( ulLen >= DEVICE_INIT_SIZE )
                {
                    pInfo->bParam[ 0 ] = 'M';
                    pInfo->bParam[ 1 ] = 'i';
                    pInfo->bParam[ 2 ] = 'c';
                    pInfo->bParam[ 3 ] = 'r';
                    *pulBytesWritten = DEVICE_INIT_RETURN_SIZE;
                }
                else
                {
                    *pulBytesNeeded = DEVICE_INIT_SIZE;
                    status = NDIS_STATUS_BUFFER_TOO_SHORT;
                }
            }
            else
            {
                status = NDIS_STATUS_INVALID_DATA;
            }
        }
        else
        {
            status = NDIS_STATUS_INVALID_DATA;
        }
        pInfo->dwLength = *pulBytesWritten;
        pInfo->dwReturnCode = status;
    }
    else
    {
        *pulBytesNeeded = sizeof( TDeviceInfo );
        *pulBytesWritten = 0;
        status = NDIS_STATUS_BUFFER_TOO_SHORT;
    }
    return( status );
}  // GetDeviceMode


NDIS_STATUS SetDeviceMode (
    IN  PNDIS_ADAPTER pAdapter,
    IN  NDIS_OID      Oid,
    IN  PVOID         pBuffer,
    IN  ULONG         ulLen,
    OUT PULONG        pulBytesRead,
    OUT PULONG        pulBytesNeeded )
{
    NDIS_STATUS status = NDIS_STATUS_SUCCESS;
    PHARDWARE   pHardware = &pAdapter->m_Hardware;

    if ( pBuffer  &&  ulLen >= sizeof( TDeviceInfo ) )
    {
        PTDeviceInfo pInfo = ( PTDeviceInfo ) pBuffer;

        *pulBytesNeeded = 0;
        *pulBytesRead = sizeof( TDeviceInfo );
        if ( Oid == pInfo->dwReturnCode  &&
                DEVICE_COMMAND_SET == pInfo->wCommand )
        {
            switch ( pInfo->wFlags )
            {
                case DEVICE_FLAG_CAPABILITIES:
                {
                    PUINT pulData = ( PUINT ) pInfo->bParam;

                    AcquireAdapter( pAdapter, TRUE );
                    HardwareSetCapabilities( pHardware, 0, *pulData );
                    ReleaseAdapter( pAdapter );
                    break;
                }
                case DEVICE_FLAG_EEPROM_REG:
                {
                    PUSHORT pwParam = ( PUSHORT ) pInfo->bParam;

                    AcquireAdapter( pAdapter, TRUE );
                    EepromWriteWord( pHardware, ( UCHAR ) pwParam[ 0 ],
                        pwParam[ 1 ]);
                    ReleaseAdapter( pAdapter );
                    break;
                }
                default:
                    status = NDIS_STATUS_INVALID_DATA;
            }
        }
        else
            status = NDIS_STATUS_INVALID_DATA;
        pInfo->dwLength = *pulBytesRead;
        pInfo->dwReturnCode = status;
    }
    else
    {
        *pulBytesNeeded = sizeof( TDeviceInfo );
        *pulBytesRead = 0;
        status = NDIS_STATUS_BUFFER_TOO_SHORT;
    }
    return( status );
}  // SetDeviceMode


