/* ---------------------------------------------------------------------------
          Copyright (c) 2004-2008 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    transmit.c - Linux network device driver transmit processing.

    Author      Date        Description
    THa         01/05/04    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         08/15/05    Disable interrupt when setting up transmission.
    THa         10/12/05    Updated for new descriptor structure.
    THa         04/03/06    Use TWO_NETWORK_INTERFACE instead of TWO_PORTS.
    THa         06/05/06    Make sure interrupts are enabled before processing
                            them.
    THa         06/07/06    No need to block interrupts as the transmit
                            interrupt processing is handled in a tasklet.
    THa         06/29/06    Change statistics reporting to provide meaningful
                            data.
    THa         07/14/06    Fix two network interfaces driver issues.
    THa         09/22/06    Fix HardwareAllocPacket issue.
    THa         02/02/07    Implement transmit hardware checksumming correctly.
    THa         05/11/07    Update for 2.6.19 and newer kernels.
    THa         12/05/07    Update for 64-bit Linux kernels.
   ---------------------------------------------------------------------------
*/


#include <linux/pci.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <asm/irq.h>
#include "target.h"
#include "hardware.h"
#include "device.h"


#if 1
#define SKB_WAITING
#endif

/* -------------------------------------------------------------------------- */

static inline PDMA_BUFFER alloc_tx_buf (
    PBUFFER_INFO pBufInfo,
    PTDesc       pCurrent )
{
    PDMA_BUFFER pDma;

#if 0
    pDma = NULL;
    if ( pBufInfo->cnAvail )
#else
    ASSERT( pBufInfo->cnAvail );
#endif
    {
        pDma = pBufInfo->pTop;
        pBufInfo->pTop = pDma->pNext;
        pBufInfo->cnAvail--;
#if 0
        /* Remember current transmit buffer. */
        pAdapter->m_TxBufInfo.pCurrent = pDma;
#endif

        /* Link to current descriptor. */
        pCurrent->pReserved = pDma;
    }
    return( pDma );
}  /* alloc_tx_buf */


static inline void free_tx_buf (
    PBUFFER_INFO pBufInfo,
    PDMA_BUFFER  pDma )
{
    pDma->pNext = pBufInfo->pTop;
    pBufInfo->pTop = pDma;
    pBufInfo->cnAvail++;
}  /* free_tx_buf */

/* -------------------------------------------------------------------------- */

/*
    send_packet

    Description:
        This function is used to send a packet out to the network.

    Parameters:
        struct sk_buff* skb
            Pointer to socket buffer.

        struct net_device* dev
            Pointer to network device.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

static int send_packet (
    struct sk_buff*    skb,
    struct net_device* dev )
{
    PTDesc           pDesc;
    PTDesc           pFirstDesc;
    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PTDescInfo       pInfo = &pHardware->m_TxDescInfo;
    PDMA_BUFFER      pDma;
    int              len;

#ifdef SCATTER_GATHER
    int              last_frag = skb_shinfo( skb )->nr_frags;
#endif
    int              rc = 1;

#ifdef TWO_NETWORK_INTERFACE
    pHardware->m_bPortTX = DEV_TO_HW_PORT( priv->port );
#endif

#if ETH_ZLEN > 60
    len = ETH_ZLEN < skb->len ? skb->len : ETH_ZLEN;
#else
    /* Hardware will pad the length to 60. */
    len = skb->len;
#endif

    /* Remember the very first descriptor. */
    pFirstDesc = pInfo->pCurrent;
    pDesc = pFirstDesc;

    pDma = alloc_tx_buf( &hw_priv->m_TxBufInfo, pDesc );

#ifdef SCATTER_GATHER
    if ( last_frag ) {
        int         frag;
        skb_frag_t* this_frag;

        pDma->len = skb->len - skb->data_len;

#ifdef HIGH_MEM_SUPPORT
        pDma->dma = pci_map_page( hw_priv->pdev, virt_to_page( skb->data ),
            (( unsigned long ) skb->data & ~PAGE_MASK ), pDma->len,
            PCI_DMA_TODEVICE );
#else
        pDma->dma = pci_map_single( hw_priv->pdev, skb->data, pDma->len,
            PCI_DMA_TODEVICE );
#endif
        SetTransmitBuffer( pDesc, pDma->dma );
        SetTransmitLength( pDesc, pDma->len );

        frag = 0;
        do {
            this_frag = &skb_shinfo( skb )->frags[ frag ];

            /* Get a new descriptor. */
            GetTxPacket( pInfo, pDesc );

#ifdef SKIP_TX_INT
            ++pHardware->m_TxIntCnt;
#endif

            pDma = alloc_tx_buf( &hw_priv->m_TxBufInfo, pDesc );
            pDma->len = this_frag->size;

#ifdef HIGH_MEM_SUPPORT
            pDma->dma = pci_map_page( hw_priv->pdev, this_frag->page,
                this_frag->page_offset, pDma->len, PCI_DMA_TODEVICE );
#else
            pDma->dma = pci_map_single( hw_priv->pdev,
                page_address( this_frag->page ) + this_frag->page_offset,
                pDma->len, PCI_DMA_TODEVICE );
#endif
            SetTransmitBuffer( pDesc, pDma->dma );
            SetTransmitLength( pDesc, pDma->len );

            frag++;
            if ( frag == last_frag )
                break;

            /* Do not release the last descriptor here. */
            ReleasePacket( pDesc );
        } while ( 1 );

        /* pCurrent points to the last descriptor. */
        pInfo->pCurrent = pDesc;

        /* Release the first descriptor. */
        ReleasePacket( pFirstDesc );
    }
    else
#endif
    {
        pDma->len = len;

#ifdef HIGH_MEM_SUPPORT
        pDma->dma = pci_map_page( hw_priv->pdev, virt_to_page( skb->data ),
            (( unsigned long ) skb->data & ~PAGE_MASK ), pDma->len,
            PCI_DMA_TODEVICE );

#else
        pDma->dma = pci_map_single( hw_priv->pdev, skb->data, pDma->len,
            PCI_DMA_TODEVICE );
#endif
        SetTransmitBuffer( pDesc, pDma->dma );
        SetTransmitLength( pDesc, pDma->len );
    }

#if TXCHECKSUM_DEFAULT
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 19))
    if ( skb->ip_summed == CHECKSUM_HW ) {
#else
    if ( skb->ip_summed == CHECKSUM_PARTIAL ) {
#endif
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22))
        struct iphdr* iph = skb->nh.iph;
#else
        struct iphdr* iph = ( struct iphdr* ) skb_network_header( skb );
#endif
        if ( iph->protocol == IPPROTO_TCP ) {
            ( pDesc )->sw.BufSize.tx.fCsumGenTCP = TRUE;
        }
        else if ( iph->protocol == IPPROTO_UDP ) {
            ( pDesc )->sw.BufSize.tx.fCsumGenUDP = TRUE;
        }
    }
#endif

    /* The last descriptor holds the packet so that it can be returned to
       network subsystem after all descriptors are transmitted.
    */
    pDma->skb = skb;

#ifdef DEBUG_TX_
    {
        int    nLength;
        UCHAR* bData = ( PUCHAR ) pDma->skb->data;

//        if ( 0xFF != bData[ 0 ] )
        {
            DBG_PRINT( "S: "NEWLINE );
            for ( nLength = 0; nLength < pDma->skb->len; nLength++ )
            {
                DBG_PRINT( "%02x ", bData[ nLength ]);
                if ( ( nLength % 16 ) == 15 )
                {
                    DBG_PRINT( NEWLINE );
                }
            }
            DBG_PRINT( NEWLINE );
        }
    }
#endif

    if ( HardwareSendPacket( pHardware ) ) {
        rc = 0;

        /* Update transmit statistics. */
#ifndef NO_STATS
        pHardware->m_cnCounter[ port ][ OID_COUNTER_DIRECTED_FRAMES_XMIT ]++;
        pHardware->m_cnCounter[ port ][ OID_COUNTER_DIRECTED_BYTES_XMIT ] +=
            len;
#endif
        priv->stats.tx_packets++;
        priv->stats.tx_bytes += len;

#if 0
        dev->trans_start = jiffies;
#endif
    }

#ifdef SKB_WAITING
    priv->skb_waiting = NULL;
#endif

    return( rc );
}  /* send_packet */


#ifdef SKB_WAITING
/*
    send_next_packet

    Description:
        This function is used to send the next packet waiting to be sent.

    Parameters:
        struct net_device* dev
            Pointer to network device.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

static int send_next_packet (
    struct net_device* dev )
{
    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    int              rc = 1;

    if ( priv->skb_waiting ) {
        PHARDWARE pHardware = &hw_priv->hw;
        int       num = 1;

#ifdef SCATTER_GATHER
        num = skb_shinfo( priv->skb_waiting )->nr_frags + 1;
#endif
        if ( HardwareAllocPacket( pHardware, priv->skb_waiting->len, num )
                >= num ) {
            rc = send_packet( priv->skb_waiting, dev );
            netif_wake_queue( dev );
        }
    }
    return( rc );
}  /* send_next_packet */
#endif


/*
    transmit_done

    Description:
        This routine is called when the transmit interrupt is triggered,
        indicating either a packet is sent successfully or there are transmit
        errors.

    Parameters:
        struct net_device* dev
            Pointer to network device.

    Return (None):
*/

void
ks8842p_\
transmit_done (
    struct net_device* dev )
{
    int              iLast;
    TDescStat        status;
    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PTDescInfo       pInfo = &pHardware->m_TxDescInfo;
    PTDesc           pDesc;
    PDMA_BUFFER      pDma;

    spin_lock( &priv->lock );

    iLast = pInfo->iLast;

    while ( pInfo->cnAvail < pInfo->cnAlloc )
    {
        /* Get next descriptor which is not hardware owned. */
        GetTransmittedPacket( pInfo, iLast, pDesc, status.ulData );
        if ( status.tx.fHWOwned )
            break;

        pDma = DMA_BUFFER( pDesc );

        /* This descriptor contains a buffer. */
        if ( likely( pDma ) )
        {

#ifdef HIGH_MEM_SUPPORT
            pci_unmap_page( hw_priv->pdev, pDma->dma, pDma->len,
                PCI_DMA_TODEVICE );
#else
            pci_unmap_single( hw_priv->pdev, pDma->dma, pDma->len,
                PCI_DMA_TODEVICE );
#endif

            /* This descriptor contains the last buffer in the packet. */
            if ( pDma->skb )
            {
                /* Notify the network subsystem that the packet has been sent.
                */

#ifdef DEBUG_TX
    {
        int    nLength;
        UCHAR* bData = ( PUCHAR ) pDma->skb->data;

//        if ( 0xFF != bData[ 0 ] )
        {
            for ( nLength = 0; nLength < pDma->skb->len; nLength++ )
            {
                DBG_PRINT( "%02x ", bData[ nLength ]);
                if ( ( nLength % 16 ) == 15 )
                {
                    DBG_PRINT( NEWLINE );
                }
            }
            DBG_PRINT( NEWLINE );
        }
    }
#endif

#if 1
                pDma->skb->dev->trans_start = jiffies;
#endif

                /* Release the packet back to network subsystem. */
                dev_kfree_skb_irq( pDma->skb );
                pDma->skb = NULL;
            }

            /* Free the transmitted buffer. */
            free_tx_buf( &hw_priv->m_TxBufInfo, pDma );
            pDesc->pReserved = NULL;
        }

        /* Free the transmitted descriptor. */
        FreeTransmittedPacket( pInfo, iLast );
    }
    pInfo->iLast = iLast;

#ifdef TWO_NETWORK_INTERFACE
    dev = hw_priv->pDev[ MAIN_PORT ];
#endif
    if ( netif_queue_stopped( dev ) )

#ifdef SKB_WAITING
        send_next_packet( dev );

#else
        netif_wake_queue( dev );
#endif

#ifdef TWO_NETWORK_INTERFACE
    dev = hw_priv->pDev[ OTHER_PORT ];
    if ( netif_queue_stopped( dev ) )

#ifdef SKB_WAITING
        send_next_packet( dev );

#else
        netif_wake_queue( dev );
#endif
#endif

    spin_unlock( &priv->lock );
}  /* transmit_done */


void transmit_reset (
    struct net_device* dev )
{
    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PTDescInfo       pInfo = &pHardware->m_TxDescInfo;
    int              iLast;
    PTDesc           pDesc;
    PDMA_BUFFER      pDma;
    TDescStat        status;

    spin_lock( &priv->lock );
    iLast = pInfo->iLast;

    while ( pInfo->cnAvail < pInfo->cnAlloc )
    {
        /* Get next descriptor which is hardware owned. */
        GetTransmittedPacket( pInfo, iLast, pDesc, status.ulData );
        if ( status.tx.fHWOwned )
        {
            ReleaseDescriptor( pDesc, status );
        }

        pDma = DMA_BUFFER( pDesc );

        /* This descriptor contains a buffer. */
        if ( pDma )
        {

#ifdef HIGH_MEM_SUPPORT
            pci_unmap_page( hw_priv->pdev, pDma->dma, pDma->len,
                PCI_DMA_TODEVICE );
#else
            pci_unmap_single( hw_priv->pdev, pDma->dma, pDma->len,
                PCI_DMA_TODEVICE );
#endif

            /* This descriptor contains the last buffer in the packet. */
            if ( pDma->skb )
            {
                /* Notify the network subsystem that the packet has not been
                   sent.
                */

                /* Release the packet back to network subsystem. */
                dev_kfree_skb( pDma->skb );
                pDma->skb = NULL;
            }

            /* Free the buffer. */
            free_tx_buf( &hw_priv->m_TxBufInfo, pDma );
            pDesc->pReserved = NULL;
        }

        /* Free the descriptor. */
        FreeTransmittedPacket( pInfo, iLast );
    }
    pInfo->iLast = iLast;
    spin_unlock( &priv->lock );
}  /* transmit_reset */

/* -------------------------------------------------------------------------- */

/*
    dev_transmit

    Description:
        This function is used by the upper network layer to send out a packet.

    Parameters:
        struct sk_buff* skb
            Pointer to socket buffer.

        struct net_device* dev
            Pointer to network device.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

int
ks8842p_\
dev_transmit (
    struct sk_buff*    skb,
    struct net_device* dev )
{
    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    int              len;
    int              num = 1;
    int              rc = 0;

#if 1
    struct sk_buff* org_skb = skb;

    if ( skb->len <= 48 ) {
#ifdef NET_SKBUFF_DATA_USES_OFFSET
        if ( skb_end_pointer( skb ) - skb->data >= 50 ) {
#else
        if ( skb->end - skb->data >= 50 ) {
#endif
            memset( &skb->data[ skb->len ], 0, 50 - skb->len );
            skb->len = 50;
        }
        else {
            skb = dev_alloc_skb( 50 );
            if ( !skb ) {
                return 1;
            }
            memcpy( skb->data, org_skb->data, org_skb->len );
            memset( &skb->data[ org_skb->len ], 0, 50 - org_skb->len );
            skb->len = 50;
            skb->dev = org_skb->dev;
            skb->ip_summed = org_skb->ip_summed;
            skb->csum = org_skb->csum;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22))
            skb->nh.raw = skb->data + ETH_HLEN;
#else
            skb_set_network_header( skb, ETH_HLEN );
#endif

            dev_kfree_skb( org_skb );
        }
    }
#endif

#ifdef SKB_WAITING
    /* There is a packet waiting to be sent.  Transmit queue should already be
       stopped.
    */
    if ( priv->skb_waiting ) {

#ifdef DBG
        printk( "waiting to send!\n" );
#endif
        priv->stats.tx_dropped++;
        if ( !netif_queue_stopped( dev ) ) {
            DBG_PRINT( "queue not stopped!\n" );
            netif_stop_queue( dev );
        }
        return 1;
    }
#endif

#if ETH_ZLEN > 60
    len = ETH_ZLEN < skb->len ? skb->len : ETH_ZLEN;
#else
    /* Hardware will pad the length to 60. */
    len = skb->len;
#endif

#if 0
    if ( ( rc = AcquireHardware( hw_priv, FALSE, FALSE )) ) {
        return( rc );
    }
#endif

    spin_lock_irq( &priv->lock );

#ifdef SKB_WAITING
    priv->skb_waiting = skb;
#endif

#ifdef SCATTER_GATHER
    num = skb_shinfo( skb )->nr_frags + 1;
#endif
    if ( HardwareAllocPacket( pHardware, len, num ) >= num ) {
        rc = send_packet( skb, dev );

        if ( !rc ) {

            /* Check whether there are space to hold more socket buffers. */
        }
        else {
            dev_kfree_skb( skb );
        }
    }
    else {

#ifdef DEBUG_TX
        printk( "wait to send\n" );
#endif

        /* Stop the transmit queue until packet is allocated. */
        netif_stop_queue( dev );

#ifndef SKB_WAITING
        rc = -ENOMEM;
#endif
    }

    spin_unlock_irq( &priv->lock );
#if 0
    ReleaseHardware( hw_priv, FALSE );
#endif

    return( rc );
}  /* dev_transmit */

/* -------------------------------------------------------------------------- */

/*
    dev_transmit_timeout

    Description:
        This routine is called when the transmit timer expires.  That indicates
        the hardware is not running correctly because transmit interrupts are
        not triggered to free up resources so that the transmit routine can
        continue sending out packets.  The hardware is reset to correct the
        problem.

    Parameters:
        struct net_device* dev
            Pointer to network device.

    Return (None):
*/

void
ks8842p_\
dev_transmit_timeout (
    struct net_device *dev )
{
    static unsigned long last_reset = 0;

    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    int              port;

#ifdef DBG
    printk( "transmit timeout:%lu\n", jiffies );
#endif

#ifdef TWO_NETWORK_INTERFACE

    /* Only reset the hardware if time between calls is long. */
    if ( jiffies - last_reset <= dev->watchdog_timeo )
        hw_priv = NULL;
#endif
    last_reset = jiffies;
    if ( hw_priv ) {
        AcquireHardware( hw_priv, FALSE, FALSE );

        HardwareDisableInterrupt( pHardware );
        HardwareDisable( pHardware );

        /* Initialize to invalid value so that link detection is done. */
        pHardware->m_PortInfo[ MAIN_PORT ].bLinkPartner = 0xFF;

#ifdef DEF_KS8842
        pHardware->m_PortInfo[ OTHER_PORT ].bLinkPartner = 0xFF;
#endif

        HardwareReset( pHardware );
        HardwareSetup( pHardware );
        HardwareSetDescriptorBase( pHardware, pHardware->m_TxDescInfo.ulRing,
            pHardware->m_RxDescInfo.ulRing );
            
#ifdef TWO_NETWORK_INTERFACE
        PortSet_STP_State( pHardware, MAIN_PORT, STP_STATE_BLOCKED );
        PortSet_STP_State( pHardware, OTHER_PORT, STP_STATE_BLOCKED );
#endif

        /* Reset may wipe out these registers. */
        if ( pHardware->m_bMacOverrideAddr )
            HardwareSetAddress( pHardware );
        if ( pHardware->m_bMulticastListSize )
            HardwareSetGroupAddress( pHardware );

        transmit_reset( dev );
        HardwareResetRxPackets( pHardware );
        HardwareResetTxPackets( pHardware );
        InitBuffers( &hw_priv->m_RxBufInfo );
        InitBuffers( &hw_priv->m_TxBufInfo );
        InitReceiveBuffers( hw_priv );
    }

#ifdef TWO_NETWORK_INTERFACE
    for ( port = MAIN_PORT; port <= OTHER_PORT; port++ ) {
        struct net_device* net_if = priv->pDevInfo->pDev[ port ];

        /* The second net device is not created. */
        if ( !net_if )
            continue;

        priv = ( struct dev_priv* ) net_if->priv;

#else
    for ( port = MAIN_PORT; port < OTHER_PORT; port++ ) {
#endif

        /* This port device is opened for use. */
        if ( priv->opened ) {
        
#ifdef TWO_NETWORK_INTERFACE
            PortSet_STP_State( pHardware, port, STP_STATE_FORWARDING );
#endif

#ifdef SKB_WAITING
            if ( priv->skb_waiting ) {
                priv->stats.tx_dropped++;
                dev_kfree_skb( priv->skb_waiting );
                priv->skb_waiting = NULL;
            }
#endif
        }
    }

    dev->trans_start = jiffies;
    netif_wake_queue( dev );

    if ( hw_priv ) {
        HardwareEnable( pHardware );

        ReleaseHardware( hw_priv, FALSE );
        HardwareEnableInterrupt( pHardware );
    }
}  /* dev_transmit_timeout */
