/* ---------------------------------------------------------------------------
          Copyright (c) 2004-2008 Micrel, Inc.  All rights reserved.
   ---------------------------------------------------------------------------

    interrupt.c - Linux network device driver interrupt processing.

    Author      Date        Description
    THa         01/05/04    Created file.
    THa         06/27/05    Updated for version 0.1.5.
    THa         06/29/05    Updated for Linux 2.6.
    THa         10/12/05    Updated for new descriptor structure.
    THa         04/03/06    Use TWO_NETWORK_INTERFACE instead of TWO_PORTS.
    THa         05/18/06    Implement network NAPI.
    THa         06/05/06    Make sure interrupts are enabled before processing
                            them.
    THa         06/29/06    Change statistics reporting to provide meaningful
                            data.  Add 1912 bytes large frame support.
    THa         07/14/06    Fix two network interfaces driver issues.
    THa         02/02/07    Implement receive hardware checksumming correctly.
    THa         05/11/07    Update for 2.6.19 and newer kernels.
    THa         12/05/07    Update for 64-bit Linux kernels.
    THa         05/27/08    Fix delayed packet receive problem.
   ---------------------------------------------------------------------------
*/


#include <linux/pci.h>
#include "target.h"
#include "hardware.h"
#include "device.h"
#include <linux/etherdevice.h>
#include <linux/in.h>
#include <linux/ip.h>
#include <linux/if_vlan.h>


#if 0
#define RCV_UNTIL_NONE
#endif

#if 1
#define DELAY_TX_INTERRUPT
#endif

#ifdef CONFIG_KS884X_NAPI
#define ks884x_rx_skb                    netif_receive_skb
#define ks884x_rx_quota( count, quota )	 min( count, quota )
#else
#define ks884x_rx_skb                    netif_rx
#define ks884x_rx_quota( count, quota )  count
#endif

/* -------------------------------------------------------------------------- */

static inline int dev_rcv_packets (
    struct net_device* dev )
{
    int              iNext;
    TDescStat        status;
    int              packet_len;
    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PTDescInfo       pInfo = &pHardware->m_RxDescInfo;

#ifndef RCV_UNTIL_NONE
#ifdef CONFIG_KS884X_NAPI
    int              cnLeft = dev->quota;

#else
    int              cnLeft = pInfo->cnAlloc;
#endif
#endif
    PTDesc           pDesc;
    PDMA_BUFFER      pDma;
    struct sk_buff*  skb;
    int              port = MAIN_PORT;
    int              received = 0;
    int              rx_status;

    iNext = pInfo->iNext;

#ifdef RCV_UNTIL_NONE
    while ( 1 ) {

#else
    while ( cnLeft-- ) {
#endif

        /* Get next descriptor which is not hardware owned. */
        GetReceivedPacket( pInfo, iNext, pDesc, status.ulData );
        if ( status.rx.fHWOwned )
            break;

#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_NEXT_PACKET ]++;
#endif

#ifdef CHECK_OVERRUN
        if ( !( pDesc->pCheck->Control.ulData & CPU_TO_LE32( DESC_HW_OWNED )) )
        {

#ifdef DEBUG_OVERRUN
            pHardware->m_ulDropped++;
#endif
            ReleasePacket( pDesc );
            FreeReceivedPacket( pInfo, iNext );

#ifndef RCV_UNTIL_NONE
            cnLeft = pInfo->cnAlloc;
#endif
            continue;
        }
#endif

        /* From hardware.c:HardwareReceive() */
        /* Status valid only when last descriptor bit is set. */
        if ( status.rx.fLastDesc  &&  status.rx.fFirstDesc )
        {
#if defined( CHECK_RCV_ERRORS )  ||  defined( RCV_HUGE_FRAME )
            /* Receive without error.  With receive errors disabled, packets
               with receive errors will be dropped, so no need to check the
               error bit.
            */
            if ( !status.rx.fError

#ifdef RCV_HUGE_FRAME
                    ||  ( status.ulData & (
                    DESC_RX_ERROR_CRC | DESC_RX_ERROR_RUNT |
                    DESC_RX_ERROR_TOO_LONG | DESC_RX_ERROR_PHY )) ==
                    DESC_RX_ERROR_TOO_LONG
#endif
                    )
#endif
            {
                /* Get received port number */
                pHardware->m_bPortRX = ( UCHAR ) status.rx.ulSourePort;

                port = pHardware->m_bPortRX - 1;

#ifdef DEBUG_RX
                DBG_PRINT( "m_bPortRX: %d"NEWLINE, pHardware->m_bPortRX );
#endif

#ifndef NO_STATS
                if ( status.rx.fMulticast )
                {
                    pHardware->m_cnCounter
                        [ port ]
                        [ OID_COUNTER_MULTICAST_FRAMES_RCV ]+=1;
#ifdef DEBUG_RX
                    DBG_PRINT( "M " );
#endif
                }
#endif

                /* received length includes 4-byte CRC */
                packet_len = status.rx.wFrameLen - 4;

        pDma = DMA_BUFFER( pDesc );

#ifdef TWO_NETWORK_INTERFACE
        /* Need to use the other net device if coming from the second port. */
        if ( PORT_2 == pHardware->m_bPortRX ) {
            port = OTHER_PORT;
            dev = hw_priv->pDev[ OTHER_PORT ];

            /* The second net device is not created. */
            if ( !dev )
                goto release_packet;

            priv = ( struct dev_priv* ) dev->priv;

            /* This port device is not opened for use. */
            if ( !priv->opened )
                goto release_packet;
        }
        else {
            port = MAIN_PORT;
            dev = hw_priv->pDev[ MAIN_PORT ];
            priv = ( struct dev_priv* ) dev->priv;
        }
#endif

#if (LINUX_VERSION_CODE > KERNEL_VERSION(2, 6, 9))
        pci_dma_sync_single_for_cpu( hw_priv->pdev, pDma->dma, packet_len,
            PCI_DMA_FROMDEVICE );

#else
        pci_dma_sync_single( hw_priv->pdev, pDma->dma, packet_len,
            PCI_DMA_FROMDEVICE );
#endif

        /* Allocate extra bytes for 32-bit transfer. */
        skb = dev_alloc_skb( packet_len + 6 );
        if ( !skb ) {
            priv->stats.rx_dropped++;
            goto release_packet;
        }

        /* Align socket buffer in 4-byte boundary for better performance. */
        skb_reserve( skb, 2 );
        memcpy( skb_put( skb, packet_len ), pDma->skb->data, packet_len );

#if 0
#ifndef NO_STATS
        /* Descriptor does not have broadcast indication. */
        if ( 0xFF == skb->data[ 0 ] ) {
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_BROADCAST_FRAMES_RCV ]++;
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_BROADCAST_BYTES_RCV ] +=
                    packet_len;
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_MULTICAST_FRAMES_RCV ]--;
        }
        else if ( pBuffer[ 0 ] != 0x01 )
        {
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_UNICAST_FRAMES_RCV ]++;

            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_DIRECTED_FRAMES_RCV ]++;
            pHardware->m_cnCounter[ port ]
                [ OID_COUNTER_DIRECTED_BYTES_RCV ] +=
                    packet_len;
        }
#endif
#endif

#ifdef DEBUG_RX_DATA
    {
        int    nLength;
        UCHAR* bData = ( PUCHAR ) skb->data;

        if ( 0xFF != bData[ 0 ] )
        {
            pHardware->m_nPacketLen = packet_len;
            if ( pHardware->m_nPacketLen > 0x40 )
                pHardware->m_nPacketLen = 0x40;
            for ( nLength = 0; nLength < pHardware->m_nPacketLen; nLength++ )
            {
                DBG_PRINT( "%02X ", bData[ nLength ]);
                if ( ( nLength % 16 ) == 15 )
                {
                    DBG_PRINT( NEWLINE );
                }
            }
            DBG_PRINT( NEWLINE );
        }
    }
#endif

        skb->dev = dev;
        skb->protocol = eth_type_trans( skb, dev );

#if RXCHECKSUM_DEFAULT
if ( pHardware->m_dwReceiveConfig &
        ( DMA_RX_CTRL_CSUM_UDP | DMA_RX_CTRL_CSUM_TCP ) )
{
        unsigned short protocol;
        struct iphdr* iph;

        protocol = skb->protocol;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22))
        skb->nh.raw = skb->data;
        iph = skb->nh.iph;
#else
        skb_reset_network_header( skb );
        iph = ( struct iphdr* ) skb_network_header( skb );
#endif
        if ( protocol == htons( ETH_P_8021Q ) ) {
            protocol = iph->tot_len;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22))
            skb->nh.raw += VLAN_HLEN;
            iph = skb->nh.iph;
#else
            skb_set_network_header( skb, VLAN_HLEN );
            iph = ( struct iphdr* ) skb_network_header( skb );
#endif
        }
        if ( protocol == htons( ETH_P_IP ) ) {
            if ( iph->protocol == IPPROTO_TCP ) {
                skb->ip_summed = CHECKSUM_UNNECESSARY;
            }
        }
}
#endif

        /* Update receive statistics. */
        priv->stats.rx_packets++;
        priv->stats.rx_bytes += packet_len;

        /* Notify upper layer for received packet. */
        dev->last_rx = jiffies;
        rx_status = ks884x_rx_skb( skb );
        received++;

#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_RCV_COMPLETE ]++;
#endif

#ifdef DEBUG_OVERRUN
        pHardware->m_ulReceived++;
#endif

            }

#ifdef CHECK_RCV_ERRORS
            /*
             *  Receive with error.
             */
            else {

                /* Update receive error statistics. */

                pHardware->m_cnCounter
                    [ port ]
                    [ OID_COUNTER_RCV_ERROR ]+=1;

#ifndef NO_STATS
                if ( status.rx.fErrCRC )
                    pHardware->m_cnCounter
                        [ port ]
                        [ OID_COUNTER_RCV_ERROR_CRC ]+=1;

                if ( status.rx.fErrRunt )
                    pHardware->m_cnCounter
                        [ port ]
                        [ OID_COUNTER_RCV_ERROR_RUNT ]+=1;

                if ( status.rx.fErrTooLong )
                    pHardware->m_cnCounter
                        [ port ]
                        [ OID_COUNTER_RCV_ERROR_TOOLONG ]+=1;

                if ( status.rx.fErrPHY )
                    pHardware->m_cnCounter
                        [ port ]
                        [ OID_COUNTER_RCV_ERROR_MII ]+=1;
#endif

#ifdef DEBUG_RX
                DBG_PRINT( "  RX: %08lX"NEWLINE, status.ulData );
#endif

#ifdef DEBUG_COUNTER
                pHardware->m_nBad[ COUNT_BAD_RCV_FRAME ]+=1;
#endif

            }

/* Hardware checksum errors are not associated with receive errors. */
#if RXCHECKSUM_DEFAULT

/* Hardware cannot handle UDP packet in IP fragments. */
#if 0
            if ( status.rx.fCsumErrUDP )
                pHardware->m_cnCounter
                    [ port ]
                    [ OID_COUNTER_RCV_ERROR_UDP ]+=1;
#endif

            if ( status.rx.fCsumErrTCP )
                pHardware->m_cnCounter
                    [ port ]
                    [ OID_COUNTER_RCV_ERROR_TCP ]+=1;

            if ( status.rx.fCsumErrIP )
                pHardware->m_cnCounter
                    [ port ]
                    [ OID_COUNTER_RCV_ERROR_IP ]+=1;
#endif
#endif
        }

release_packet:
        ReleasePacket( pDesc );
        FreeReceivedPacket( pInfo, iNext );
    }
    pInfo->iNext = iNext;

#ifdef DEBUG_COUNTER
    if ( !received )
        pHardware->m_nGood[ COUNT_GOOD_NO_NEXT_PACKET ]++;
    else if ( 1 == received )
        pHardware->m_nGood[ COUNT_GOOD_RCV_CNT_1 ]++;
    else if ( 2 == received )
        pHardware->m_nGood[ COUNT_GOOD_RCV_CNT_2 ]++;
    else
        pHardware->m_nGood[ COUNT_GOOD_RCV_CNT_3 ]++;
#endif
    return( received );
}  /* dev_rcv_packets */

/* -------------------------------------------------------------------------- */

void RxProcessTask (
    unsigned long data )
{
    struct net_device* dev = ( struct net_device* ) data;
    struct dev_priv*   priv = ( struct dev_priv* ) dev->priv;
    struct dev_info*   hw_priv = priv->pDevInfo;
    PHARDWARE          pHardware = &hw_priv->hw;

    if ( !pHardware->m_bEnabled )
        return;
    if ( unlikely( !dev_rcv_packets( dev )) ) {

        /* tasklets are interruptible. */
        spin_lock_irq( &priv->lock );
        HardwareTurnOnInterrupt( pHardware, ( INT_RX | INT_RX_OVERRUN ),
            NULL );
        spin_unlock_irq( &priv->lock );
    }
    else {
        HardwareAcknowledgeInterrupt( pHardware, INT_RX );
        tasklet_schedule( &hw_priv->rx_tasklet );
    }
}  /* RxProcessTask */


void TxProcessTask (
    unsigned long data )
{
    struct net_device* dev = ( struct net_device* ) data;
    struct dev_priv*   priv = ( struct dev_priv* ) dev->priv;
    struct dev_info*   hw_priv = priv->pDevInfo;
    PHARDWARE          pHardware = &hw_priv->hw;

    HardwareAcknowledgeInterrupt( pHardware, INT_TX | INT_TX_EMPTY );

    TRANSMIT_DONE( dev );

    /* tasklets are interruptible. */
    spin_lock_irq( &priv->lock );
    HardwareTurnOnInterrupt( pHardware, INT_TX, NULL );
    spin_unlock_irq( &priv->lock );
}  /* TxProcessTask */


#ifdef CONFIG_KS884X_NAPI
/*
    dev_poll

    Description:
        This function is used to poll the device for received buffers.

    Parameters:
        struct net_device* dev
            Pointer to network device.

        int* budget
            Remaining number of packets allowed to report to the stack.

    Return (int):
        Zero if all packets reported; one to indicate more.
*/

int
ks8842p_\
dev_poll (
    struct net_device* dev,
    int*               budget )
{
    struct dev_priv* priv = ( struct dev_priv* ) dev->priv;
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    int              received;
    int              left = min( *budget, dev->quota );

    HardwareAcknowledgeInterrupt( pHardware, INT_RX );
        
    received = dev_rcv_packets( dev );

    dev->quota -= received;
    *budget -= received;

    if ( received < left ) {
        netif_rx_complete( dev );

        /* tasklets are interruptible. */
        spin_lock_irq( &priv->lock );
        HardwareTurnOnInterrupt( pHardware, ( INT_RX | INT_RX_OVERRUN ),
            NULL );
        spin_unlock_irq( &priv->lock );
    }

    return( received >= left );
}  /* dev_poll */
#endif


/*
    dev_interrupt

    Description:
        This routine is called by upper network layer to signal interrupt.

    Parameters:
        int irq
            Interrupt number.

        void* dev_id
            Pointer to network device.

        struct pt_regs regs
            Pointer to registers.

    Return (None):
*/

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
void

#else
irqreturn_t
#endif
ks8842p_\
dev_interrupt (
    int             irq,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 19))
    void*           dev_id,
    struct pt_regs* regs )
#else
    void*           dev_id )
#endif
{
    UINT               IntEnable = 0;
    struct net_device* dev = ( struct net_device* ) dev_id;
    struct dev_priv*   priv = ( struct dev_priv* ) dev->priv;
    struct dev_info*   hw_priv = priv->pDevInfo;
    PHARDWARE          pHardware = &hw_priv->hw;
    int                port = MAIN_PORT;
    UINT               IntRead;

    HardwareReadInterrupt( pHardware, &IntRead );

    /* Hardware interrupts are disabled. */
    IntEnable = IntRead & pHardware->m_ulInterruptSet;

    /* Not our interrupt! */
    if ( !IntEnable )
    {

#ifdef DBG
        printk( "?%x:%x:%x; ", IntRead, 
            pHardware->m_ulInterruptSet,
            pHardware->m_ulInterruptMask );
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0))
        return IRQ_NONE;

#else
        return;
#endif
    }

#ifdef DEBUG_COUNTER
    pHardware->m_nGood[ COUNT_GOOD_INT ]++;
#endif
    do {

#ifdef DEBUG_COUNTER
        pHardware->m_nGood[ COUNT_GOOD_INT_LOOP ]++;
#endif

#ifdef DEBUG_INTERRUPT
        DBG_PRINT( "I: %08X"NEWLINE, IntEnable );
#endif
        HardwareAcknowledgeInterrupt( pHardware, IntEnable );
        IntEnable &= pHardware->m_ulInterruptMask;

        /* All packets have been sent. */

#ifdef DELAY_TX_INTERRUPT
        if ( unlikely( IntEnable & ( INT_TX | INT_TX_EMPTY )) ) {
            HardwareTurnOffInterrupt( pHardware, ( INT_TX | INT_TX_EMPTY ));
            tasklet_schedule( &hw_priv->tx_tasklet );
        }

#else
        if ( ( IntEnable & INT_TX_EMPTY ) ) {

            /* Also acknowledge transmit complete interrupt. */
            HardwareAcknowledgeTransmit( pHardware );

            TRANSMIT_DONE( dev );
        }

        /* Do not need to process transmit complete as all transmitted
           descriptors are freed.
        */
        else
        if ( ( IntEnable & INT_TX ) ) {
            TRANSMIT_DONE( dev );
        }
#endif

        if ( likely( IntEnable & INT_RX ) ) {

#ifdef DEBUG_COUNTER
            pHardware->m_nGood[ COUNT_GOOD_INT_RX ]++;
#endif

#ifdef CONFIG_KS884X_NAPI
            HardwareTurnOffInterrupt( pHardware,
                ( INT_RX | INT_RX_OVERRUN ));
            if ( likely( netif_rx_schedule_prep( dev )) ) {

                /* tell system we have work to be done. */
                __netif_rx_schedule( dev );
            }

#else
#if 1
            HardwareTurnOffInterrupt( pHardware,
                ( INT_RX | INT_RX_OVERRUN ));
            tasklet_schedule( &hw_priv->rx_tasklet );
#else

            /* Receive the packet. */
            dev_rcv_packets( dev );

#ifdef RCV_UNTIL_NONE
            /* Acknowledge the interrupt. */
            HardwareAcknowledgeReceive( pHardware );
#endif
#endif
#endif
        }

        if ( unlikely( IntEnable & INT_RX_OVERRUN ) ) {

#ifdef DBG
            DBG_PRINT( "Rx overrun"NEWLINE );
#endif

#ifdef TWO_NETWORK_INTERFACE
            /* See which port is being used. */
#endif
            pHardware->m_cnCounter[ port ][ OID_COUNTER_RCV_NO_BUFFER ]++;
            priv->stats.rx_fifo_errors++;

            HardwareAcknowledgeOverrun( pHardware );
            HardwareResumeReceive( pHardware );
        }

        if ( unlikely( IntEnable & INT_PHY ) ) {
            pHardware->m_bLinkIntWorking = TRUE;
            SwitchGetLinkStatus( pHardware );

            /* Acknowledge the interrupt. */
            HardwareAcknowledgeLink( pHardware );
        }

        if ( unlikely( IntEnable & INT_RX_STOPPED ) ) {

#ifdef DBG
            DBG_PRINT( "Rx stopped"NEWLINE );
#endif

            /* Receive just has been stopped. */
            if ( 0 == pHardware->m_bReceiveStop ) {
                pHardware->m_ulInterruptMask &= ~INT_RX_STOPPED;
            }
            else if ( pHardware->m_bReceiveStop > 1 ) {
                if ( pHardware->m_bEnabled  &&
                        ( pHardware->m_dwReceiveConfig &
                        DMA_RX_CTRL_ENABLE ) ) {

#ifdef DBG
                    DBG_PRINT( "Rx disabled"NEWLINE );
#endif
                    HardwareStartReceive( pHardware );
                }
                else {

#ifdef DBG
                    DBG_PRINT( "Rx stop disabled:%d"NEWLINE,
                        pHardware->m_bReceiveStop );
#endif
                    pHardware->m_ulInterruptMask &= ~INT_RX_STOPPED;
                    pHardware->m_bReceiveStop = 0;
                }
            }

            /* Receive just has been started. */
            else {
                pHardware->m_bReceiveStop++;
            }
            HardwareAcknowledgeInterrupt( pHardware, INT_RX_STOPPED );
            break;
        }

#if 0
        if ( ( IntEnable & INT_TX_STOPPED ) ) {
            DWORD dwData;

            pHardware->m_ulInterruptMask &= ~INT_TX_STOPPED;
            DBG_PRINT( "Tx stopped"NEWLINE );
            HW_READ_DWORD( pHardware, REG_DMA_TX_CTRL, &dwData );
            if ( !( dwData & DMA_TX_CTRL_ENABLE ) ) {
                DBG_PRINT( "Tx disabled"NEWLINE );
            }
            HardwareAcknowledgeInterrupt( pHardware, INT_TX_STOPPED );
            break;
        }
#endif
    } while ( 0 );

    HardwareEnableInterrupt( pHardware );

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0))
    return IRQ_HANDLED;
#endif
}  /* dev_interrupt */
