/* ---------------------------------------------------------------------------
          Copyright (c) 2004-2009 Micrel, Inc.
   ---------------------------------------------------------------------------

    device.c - Linux network device driver.

    Author      Date        Description
    THa         01/05/04    Created file.
    THa         10/14/04    Updated with latest specs.
    THa         05/10/05    Updated for two port devices.
    THa         06/27/05    Updated for version 0.1.5.
    THa         06/29/05    Updated for Linux 2.6.
    THa         07/07/05    Changed /proc entries.
    THa         08/15/05    Added PCI configuration I/O.
    THa         10/12/05    Updated for new descriptor structure.
    THa         10/18/05    Provide proc entries to change duplex and speed.
    THa         03/14/06    Release 1.0.0.
    THa         04/03/06    Use TWO_NETWORK_INTERFACE instead of TWO_PORTS.
    THa         04/06/06    Implement MII IO calls.
    THa         05/18/06    Implement network NAPI.
    THa         05/31/06    Add ethtool support.
    THa         06/05/06    Make sure interrupts are enabled before processing
                            them.
    THa         06/07/06    Hardware interrupts may not be disabled
                            immediately after the register is written to.
    THa         06/29/06    Change statistics reporting to provide meaningful
                            data.
    THa         07/14/06    Fix two network interfaces driver issues.
    THa         09/22/06    Fix HardwareAllocPacket issue.
    Tha         03/09/07    Add ethtool hardware checksum functions.
    THa         05/04/07    Add big-endian support.
    THa         05/11/07    Update for 2.6.19 and newer kernels.
    THa         12/03/07    Update jumbo frame support.
    THa         12/05/07    Update for 64-bit Linux kernels.
    THa         12/21/07    Add power management support.
    THa         01/18/08    Update for old 2.4.20 kernel.
    THa         02/15/08    Add force link support.
    THa         03/28/08    Update for Linux 2.6.24.
    THa         05/27/08    Update for 64-bit Linux 2.6.24.
    THa         07/02/08    Fix continual backpressure problem in KSZ8841P.
    THa         03/12/09    Support using two MAC addresses in two network
                            interfaces mode.
    THa         04/15/09    Update for Linux 2.6.29.
    THa         05/07/09    Fix transmit descriptor allocation failed problem.
    THa         06/12/09    STP blocked state still needs to listen.
    THa         06/26/09    Support STP in hardware switch.
    THa         08/28/09    Update to use netdev_ops structure.
   ---------------------------------------------------------------------------
*/


#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioport.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <asm/uaccess.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Micrel, Inc. <www.micrel.com>");
#if defined( DEF_KS8842 )
MODULE_DESCRIPTION("Micrel KSZ8842 PCI Ethernet driver");
#else
MODULE_DESCRIPTION("Micrel KSZ8841 PCI Ethernet driver");
#endif

#include "target.h"
#include "hardware.h"
#include "device.h"

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 21))
#include <linux/ethtool.h>
#endif
#include <linux/etherdevice.h>
#include <asm/irq.h>
#if 1
#include "ks_ioctl.h"


int hwread
(
   unsigned char   BankNum,
   unsigned long   RegAddr,
   unsigned char   Width
);
#endif


#ifdef CONFIG_STP_SUPPORT
#ifdef KBUILD_BASENAME
#include <../net/bridge/br_private.h>

#else
/* From Linux 2.4 */

/* net/bridge/br_private_timer.h */
struct br_timer
{
        int running;
        unsigned long expires;
};

/* net/bridge/br_private.h */
typedef struct bridge_id bridge_id;
typedef struct mac_addr mac_addr;
typedef __u16 port_id;

struct bridge_id
{
        unsigned char   prio[2];
        unsigned char   addr[6];
};

struct mac_addr
{
        unsigned char   addr[6];
        unsigned char   pad[2];
};

struct net_bridge;

struct net_bridge_port
{
        struct net_bridge_port  *next;
        struct net_bridge       *br;
        struct net_device       *dev;
        int                     port_no;

        /* STP */
        port_id                 port_id;
        int                     state;
        int                     path_cost;
        bridge_id               designated_root;
        int                     designated_cost;
        bridge_id               designated_bridge;
        port_id                 designated_port;
        unsigned                topology_change_ack:1;
        unsigned                config_pending:1;
        int                     priority;

        struct br_timer         forward_delay_timer;
        struct br_timer         hold_timer;
        struct br_timer         message_age_timer;
};
#endif
#endif


#ifdef DBG
#define NET_MSG_ENABLE  ( NETIF_MSG_DRV | NETIF_MSG_PROBE | NETIF_MSG_LINK )
#else
#define NET_MSG_ENABLE  0
#endif


#define PCI_VENDOR_ID_KS884X  0x16C6
#define PCI_DEVICE_ID_KS8841  0x8841
#define PCI_DEVICE_ID_KS8842  0x8842


#if defined( DEF_KS8842 )
#define DRV_NAME     "KSZ8842"
#elif defined( DEF_KS8841 )
#define DRV_NAME     "KSZ8841"
#else
#error Either DEF_KS8841 or DEF_KS8842 has to be defined.
#endif
#define BUS_NAME     "PCI"
#define DRV_VERSION  "1.0.19"
#define DRV_RELDATE  "Aug 28, 2009"

static char version[] __devinitdata =
    "Micrel " DRV_NAME " " BUS_NAME " " DRV_VERSION " (" DRV_RELDATE ")";

#ifndef KS_ISA
PHARDWARE phw;
#else
extern PHARDWARE phw;
#endif

#ifdef DEBUG_COUNTER
static unsigned long query_jiffies = 0;
#endif

void RxProcessTask ( unsigned long data );
void TxProcessTask ( unsigned long data );

/* -------------------------------------------------------------------------- */

/*
    StartTimer

    Description:
        This routine starts the kernel timer after the specified time tick.

    Parameters:
        PTTimerInfo pInfo
            Pointer to kernel timer information.

        int nTime
            The time tick.

    Return (None):
*/

static void StartTimer (
    PTTimerInfo pInfo,
    int         nTime )
{
    pInfo->cnCount = 0;
    pInfo->pTimer->expires = jiffies + nTime;
    add_timer( pInfo->pTimer );

    /* infinity */
    pInfo->nCount = -1;
}  /* StartTimer */


/*
    StopTimer

    Description:
        This routine stops the kernel timer.

    Parameters:
        PTTimerInfo pInfo
            Pointer to timer information.

    Return (None):
*/

static void StopTimer (
    PTTimerInfo pInfo )
{
    pInfo->nCount = 0;
    del_timer_sync( pInfo->pTimer );
}  /* StopTimer */

/* -------------------------------------------------------------------------- */

/*
    InitHardware

    Description:
        This routine initializes system variables required to acquire the
        hardware.

    Parameters:
        struct dev_info* priv
            Pointer to real hardware device private data.

    Return (None):
*/

static void InitHardware (
    struct dev_info* priv )
{
    ASSERT( priv->hw.m_ulVIoAddr );

    spin_lock_init( &priv->lockHardware );
    init_waitqueue_head( &priv->wqhHardware );
}  /* InitHardware */


#ifndef KS_ISA
/*
    AcquireHardware

    Description:
        This function is used to acquire the hardware so that only one process
        has access to the hardware.

    Parameters:
        struct dev_info* priv
            Pointer to real hardware device private data.

        int in_intr
            Indicate calling from interrupt routines.

        int wait
            Indicate whether to wait or not.  User functions should wait, while
            periodic timer routine should not.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

int AcquireHardware (
    struct dev_info* priv,
    int              in_intr,
    int              wait )
{
    int rc;

    ASSERT( priv->hw.m_ulVIoAddr );
    do {

        /* Acquire access to the hardware acquire count variable. */
        if ( in_intr )
            spin_lock( &priv->lockHardware );
        else
            spin_lock_irq( &priv->lockHardware );
        if ( 0 == priv->hw.m_bAcquire ) {

            /* Increase hardware acquire count. */
            priv->hw.m_bAcquire++;

            /* Release access to the hardware acquire count variable. */
            if ( in_intr )
                spin_unlock( &priv->lockHardware );
            else
                spin_unlock_irq( &priv->lockHardware );

            /* Hardware acquired. */
            return 0;
        }

        /* Release access to the hardware acquire count variable. */
        if ( in_intr )
            spin_unlock( &priv->lockHardware );
        else
            spin_unlock_irq( &priv->lockHardware );

        /* Willing to wait. */
        if ( wait ) {
            if ( ( rc = wait_event_interruptible( priv->wqhHardware,
                    !priv->hw.m_bAcquire )) ) {

                /* System failure. */
                return( rc );
            }
        }
    } while ( wait );

    /* Hardware not acquired. */
    return -EBUSY;
}  /* AcquireHardware */


/*
    ReleaseHardware

    Description:
        This routine is used to release the hardware so that other process can
        have access to the hardware.

    Parameters:
        struct dev_info* priv
            Pointer to real hardware device private data.

        int in_intr
            Indicate calling from interrupt routines.

    Return (None):
*/

void ReleaseHardware (
    struct dev_info* priv,
    int              in_intr )
{
    ASSERT( priv->hw.m_ulVIoAddr );

    /* Acquire access to the hardware acquire count variable. */
    if ( in_intr )
        spin_lock( &priv->lockHardware );
    else
        spin_lock_irq( &priv->lockHardware );

    /* Decrease hardware acquire count. */
    priv->hw.m_bAcquire--;

    /* Wake up other processes waiting in the queue. */
    if ( 0 == priv->hw.m_bAcquire )
        wake_up_interruptible( &priv->wqhHardware );

    /* Release access to the hardware acquire count variable. */
    if ( in_intr )
        spin_unlock( &priv->lockHardware );
    else
        spin_unlock_irq( &priv->lockHardware );
}  /* ReleaseHardware */
#endif

/* -------------------------------------------------------------------------- */

#ifndef HardwareDisableInterruptSync
/*
    HardwareDisableInterruptSync

    Description:
        This procedure makes sure the interrupt routine is not entered after
        the hardware interrupts are disabled.

    Parameters:
        void* ptr
            Pointer to hardware information structure.

    Return (None):
*/

void HardwareDisableInterruptSync (
    void* ptr )
{
    PHARDWARE          pHardware = ( PHARDWARE ) ptr;
    struct net_device* dev = ( struct net_device* ) pHardware->m_pDevice;
    struct dev_priv*   priv = netdev_priv( dev );
    unsigned long      flags;

    spin_lock_irqsave( &priv->lock, flags );
    HardwareDisableInterrupt( pHardware );
    spin_unlock_irqrestore( &priv->lock, flags );
}  /* HardwareDisableInterruptSync */
#endif

/* -------------------------------------------------------------------------- */

/*
    AllocateSoftDescriptors

    Description:
        This local function allocates software descriptors for manipulation in
        memory.

    Parameters:
        PTDescInfo pDescInfo
            Pointer to descriptor information structure.

        int fTransmit
            Indication that descriptors are for transmit.

    Return (int):
        Zero if successful; otherwise 1.
*/

static int AllocateSoftDescriptors (
    PTDescInfo pDescInfo,
    int        fTransmit )
{
    pDescInfo->pRing = kmalloc( sizeof( TDesc ) * pDescInfo->cnAlloc,
        GFP_KERNEL );
    if ( !pDescInfo->pRing ) {
        return 1;
    }
    memset(( void* ) pDescInfo->pRing, 0,
        sizeof( TDesc ) * pDescInfo->cnAlloc );
    HardwareInitDescriptors( pDescInfo, fTransmit );
    return 0;
}  /* AllocateSoftDescriptors */


/*
    AllocateDescriptors

    Description:
        This local function allocates hardware descriptors for receiving and
        transmitting.

    Parameters:
        struct dev_info* pAdapter
            Pointer to adapter information structure.

    Return (int):
        Zero if successful; otherwise 1.
*/

static int AllocateDescriptors (
    struct dev_info* pAdapter )
{
    PHARDWARE pHardware = &pAdapter->hw;
    int       nOffset;

    /* Allocate memory for RX & TX descriptors. */
    pAdapter->m_DescPool.ulAllocSize =
        pHardware->m_RxDescInfo.nSize * pHardware->m_RxDescInfo.cnAlloc +
        pHardware->m_TxDescInfo.nSize * pHardware->m_TxDescInfo.cnAlloc +
        DESC_ALIGNMENT;

    pAdapter->m_DescPool.pAllocVirtual = pci_alloc_consistent(
        pAdapter->pdev, pAdapter->m_DescPool.ulAllocSize,
        &( pAdapter->m_DescPool.dma_addr ));
    if ( pAdapter->m_DescPool.pAllocVirtual == NULL )
    {
        pAdapter->m_DescPool.ulAllocSize = 0;
        return 1;
    }
    memset( pAdapter->m_DescPool.pAllocVirtual, 0,
        pAdapter->m_DescPool.ulAllocSize );

    /* Align to the next cache line boundary. */
    nOffset = ((( ULONG ) pAdapter->m_DescPool.pAllocVirtual % DESC_ALIGNMENT )
        ? ( DESC_ALIGNMENT -
        (( ULONG ) pAdapter->m_DescPool.pAllocVirtual % DESC_ALIGNMENT )) : 0 );
    pAdapter->m_DescPool.pVirtual = pAdapter->m_DescPool.pAllocVirtual +
        nOffset;
    pAdapter->m_DescPool.ulPhysical = pAdapter->m_DescPool.dma_addr +
        nOffset;

    /* Allocate receive/transmit descriptors. */
    pHardware->m_RxDescInfo.phwRing = ( PTHw_Desc )
        pAdapter->m_DescPool.pVirtual;
    pHardware->m_RxDescInfo.ulRing = pAdapter->m_DescPool.ulPhysical;
    nOffset = pHardware->m_RxDescInfo.cnAlloc * pHardware->m_RxDescInfo.nSize;
    pHardware->m_TxDescInfo.phwRing = ( PTHw_Desc )
        ( pAdapter->m_DescPool.pVirtual + nOffset );
    pHardware->m_TxDescInfo.ulRing = pAdapter->m_DescPool.ulPhysical + nOffset;

    if ( AllocateSoftDescriptors( &pHardware->m_RxDescInfo, FALSE ) ) {
        return 1;
    }
    if ( AllocateSoftDescriptors( &pHardware->m_TxDescInfo, TRUE ) ) {
        return 1;
    }

    return 0;
}  /* AllocateDescriptors */


/*
    InitBuffers

    Description:
        This local routine initializes the DMA buffer information structure.

    Parameters:
        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

    Return (None):
*/

void InitBuffers (
    PBUFFER_INFO pBufInfo )
{
    PDMA_BUFFER pDma;
    PDMA_BUFFER pPrevious = NULL;
    int         i;

    pBufInfo->pTop = pBufInfo->BufArray;
    pBufInfo->cnAvail = pBufInfo->cnAlloc;

    pDma = pBufInfo->BufArray;
    for ( i = 0; i < pBufInfo->cnAlloc; i++ )
    {
        pPrevious = pDma;
        pDma++;
        pPrevious->pNext = pDma;
    }
    pPrevious->pNext = NULL;
}  /* InitBuffers */


/*
    AllocateBuffers

    Description:
        This local function allocates DMA buffers for use in receiving and
        transmitting.

    Parameters:
        struct dev_info* pAdapter
            Pointer to adapter information structure.

        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

    Return (int):
        Zero if successful; otherwise 1.
*/

static int AllocateBuffers (
    struct dev_info* pAdapter,
    PBUFFER_INFO     pBufInfo )
{
    PDMA_BUFFER pDma;
    int         i;

    pBufInfo->BufArray = kmalloc( sizeof( DMA_BUFFER ) * pBufInfo->cnAlloc,
        GFP_KERNEL );
    if ( !pBufInfo->BufArray )
        return 1;
    memset(( void* ) pBufInfo->BufArray, 0,
        sizeof( DMA_BUFFER ) * pBufInfo->cnAlloc );

    /* Set up each buffer. */
    pDma = pBufInfo->BufArray;
    for ( i = 0; i < pBufInfo->cnAlloc; i++ )
    {
        pDma->len = RX_BUF_SIZE;

        pDma++;
    }
    InitBuffers( pBufInfo );

    return 0;
}  /* AllocateBuffers */


/*
    alloc_rx_buf

    Description:
        This function allocates a DMA buffer information block for either
        transmitting or receiving.

    Parameters:
        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

        PTDesc pCurrent
            Pointer to current descriptor.

    Return (PDMA_BUFFER):
        Pointer to the DMA buffer information block if successful; NULL
        otherwise.
*/

PDMA_BUFFER alloc_rx_buf (
    PBUFFER_INFO pBufInfo,
    PTDesc       pCurrent )
{
    PDMA_BUFFER pDma;

#if 0
    pDma = NULL;
    if ( pBufInfo->cnAvail )
#else
    ASSERT( pBufInfo->cnAvail );
#endif
    {
        pDma = pBufInfo->pTop;
        pBufInfo->pTop = pDma->pNext;
        pBufInfo->cnAvail--;
#if 0
        /* Remember current receive buffer. */
        pAdapter->m_RxBufInfo.pCurrent = pDma;
#endif

        /* Link to current descriptor. */
        pCurrent->pReserved = pDma;
    }
    return( pDma );
}  /* alloc_rx_buf */


/*
    free_rx_buf

    Description:
        This routine frees the DMA buffer information block.

    Parameters:
        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

        PDMA_BUFFER pDma
            Pointer to DMA buffer information block.

    Return (None):
*/

void free_rx_buf (
    PBUFFER_INFO pBufInfo,
    PDMA_BUFFER  pDma )
{
    pDma->pNext = pBufInfo->pTop;
    pBufInfo->pTop = pDma;
    pBufInfo->cnAvail++;
}  /* free_rx_buf */


/*
    InitReceiveBuffers

    Description:
        This routine initializes DMA buffers for receiving.

    Parameters:
        struct dev_info* pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void InitReceiveBuffers (
    struct dev_info* pAdapter )
{
    int         i;
    PTDesc      pDesc;
    PDMA_BUFFER pDma;
    PHARDWARE   pHardware = &pAdapter->hw;
    PTDescInfo  pInfo = &pHardware->m_RxDescInfo;

    for ( i = 0; i < pHardware->m_RxDescInfo.cnAlloc; i++ )
    {
        GetRxPacket( pInfo, pDesc );

        /* Get a new buffer for this descriptor. */
        pDma = alloc_rx_buf( &pAdapter->m_RxBufInfo, pDesc );
        if ( !pDma->skb ) {

            pDma->skb = alloc_skb( pDma->len, GFP_ATOMIC );
        }
        if ( pDma->skb  &&  !pDma->dma ) {
            pDma->skb->dev = pAdapter->dev;

#ifdef HIGH_MEM_SUPPORT
            pDma->dma = pci_map_page( pAdapter->pdev,
                virt_to_page( pDma->skb->tail ),
                (( unsigned long ) pDma->skb->tail & ~PAGE_MASK ), pDma->len,
                PCI_DMA_FROMDEVICE );

#else
            pDma->dma = pci_map_single( pAdapter->pdev,
#ifdef NET_SKBUFF_DATA_USES_OFFSET
                skb_tail_pointer( pDma->skb ),
#else
                pDma->skb->tail,
#endif
                pDma->len, PCI_DMA_FROMDEVICE );
#endif
        }

        /* Set descriptor. */
        SetReceiveBuffer( pDesc, pDma->dma );
        SetReceiveLength( pDesc, pDma->len );
        ReleasePacket( pDesc );
    }
}  /* InitReceiveBuffers */


/*
    AllocateMemory

    Description:
        This function allocates memory for use by hardware descriptors for
        receiving and transmitting.

    Parameters:
        struct dev_info* pAdapter
            Pointer to adapter information structure.

    Return (int):
        Zero if successful; otherwise 1.
*/

int AllocateMemory (
    struct dev_info* pAdapter )
{
    PHARDWARE pHardware = &pAdapter->hw;

    /* Determine the number of receive and transmit descriptors. */
    pHardware->m_RxDescInfo.cnAlloc = NUM_OF_RX_DESC;
    pHardware->m_TxDescInfo.cnAlloc = NUM_OF_TX_DESC;

#ifdef SKIP_TX_INT
    pHardware->m_TxIntCnt = 0;
    pHardware->m_TxIntMask = NUM_OF_TX_DESC / 4;
    if ( pHardware->m_TxIntMask > 8 )
    {
        pHardware->m_TxIntMask = 8;
    }
    while ( pHardware->m_TxIntMask ) {
        pHardware->m_TxIntCnt++;
        pHardware->m_TxIntMask >>= 1;
    }
    if ( pHardware->m_TxIntCnt ) {
        pHardware->m_TxIntMask = ( 1 << ( pHardware->m_TxIntCnt - 1 )) - 1;
        pHardware->m_TxIntCnt = 0;
    }
#endif

    /* Determine the descriptor size. */
    pHardware->m_RxDescInfo.nSize =
        ((( sizeof( THw_Desc ) + DESC_ALIGNMENT - 1 ) / DESC_ALIGNMENT ) *
        DESC_ALIGNMENT );
    pHardware->m_TxDescInfo.nSize =
        ((( sizeof( THw_Desc ) + DESC_ALIGNMENT - 1 ) / DESC_ALIGNMENT ) *
        DESC_ALIGNMENT );
    if ( pHardware->m_RxDescInfo.nSize != sizeof( THw_Desc ) )
    {
        DBG_PRINT( "Hardware descriptor size not right!"NEWLINE );
    }
    CheckDescriptorNum( &pHardware->m_RxDescInfo );
    CheckDescriptorNum( &pHardware->m_TxDescInfo );

    /* TX/RX buffers we need. */
    pAdapter->m_RxBufInfo.cnAlloc = pHardware->m_RxDescInfo.cnAlloc;
    pAdapter->m_TxBufInfo.cnAlloc = pHardware->m_TxDescInfo.cnAlloc;

    /* Allocate descriptors */
    if ( AllocateDescriptors( pAdapter ) )
        return 1;

    /* Allocate receive buffers. */
    if ( AllocateBuffers( pAdapter, &pAdapter->m_RxBufInfo ) )
        return 1;

    /* Allocate transmit buffers. */
    if ( AllocateBuffers( pAdapter, &pAdapter->m_TxBufInfo ) )
        return 1;

    InitReceiveBuffers( pAdapter );

    return 0;
}  /* AllocateMemory */


/*
    FreeDescriptors

    Description:
        This local routine frees the software and hardware descriptors
        allocated by AllocateDescriptors().

    Parameters:
        struct dev_info* pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

static void FreeDescriptors (
    struct dev_info* pAdapter )
{
    PHARDWARE pHardware = &pAdapter->hw;

    /* reset descriptor */
    pHardware->m_RxDescInfo.phwRing = NULL;
    pHardware->m_TxDescInfo.phwRing = NULL;
    pHardware->m_RxDescInfo.ulRing = 0;
    pHardware->m_TxDescInfo.ulRing = 0;

    /* Free memory */
    if ( pAdapter->m_DescPool.pAllocVirtual )
        pci_free_consistent( pAdapter->pdev,
            pAdapter->m_DescPool.ulAllocSize,
            pAdapter->m_DescPool.pAllocVirtual,
            pAdapter->m_DescPool.dma_addr );

    /* reset resource pool */
    pAdapter->m_DescPool.ulAllocSize = 0;
    pAdapter->m_DescPool.pAllocVirtual = NULL;

    if ( pHardware->m_RxDescInfo.pRing ) {
        kfree( pHardware->m_RxDescInfo.pRing );
        pHardware->m_RxDescInfo.pRing = NULL;
    }
    if ( pHardware->m_TxDescInfo.pRing ) {
        kfree( pHardware->m_TxDescInfo.pRing );
        pHardware->m_TxDescInfo.pRing = NULL;
    }
}  /* FreeDescriptors */


/*
    FreeBuffers

    Description:
        This local routine frees the DMA buffers allocated by
        AllocateBuffers().

    Parameters:
        struct dev_info* pAdapter
            Pointer to adapter information structure.

        PBUFFER_INFO pBufInfo
            Pointer to DMA buffer information structure.

    Return (None):
*/

static void FreeBuffers (
    struct dev_info* pAdapter,
    PBUFFER_INFO     pBufInfo )
{
    int         i;
    PDMA_BUFFER pDma;

    if ( !pBufInfo->BufArray )
        return;

    pDma = pBufInfo->BufArray;
    for ( i = 0; i < pBufInfo->cnAlloc; i++ )
    {
        if ( pDma->skb )
        {

#ifdef HIGH_MEM_SUPPORT
            pci_unmap_page( pAdapter->pdev, pDma->dma, pDma->len,
                PCI_DMA_FROMDEVICE );

#else
            pci_unmap_single( pAdapter->pdev, pDma->dma, pDma->len,
                PCI_DMA_FROMDEVICE );
#endif
            dev_kfree_skb( pDma->skb );
            pDma->skb = NULL;
            pDma->dma = 0;
        }
        pDma++;
    }
    kfree( pBufInfo->BufArray );
    pBufInfo->BufArray = NULL;
}  /* FreeBuffers */


/*
    FreeMemory

    Description:
        This local routine frees all the resources allocated by
        AllocateMemory().

    Parameters:
        struct dev_info* pAdapter
            Pointer to adapter information structure.

    Return (None):
*/

void FreeMemory (
    struct dev_info* pAdapter )
{
    /* Free transmit buffers. */
    FreeBuffers( pAdapter, &pAdapter->m_TxBufInfo );

    /* Free receive buffers */
    FreeBuffers( pAdapter, &pAdapter->m_RxBufInfo );

    /* Free descriptors. */
    FreeDescriptors( pAdapter );
}  /* FreeMemory */

/* -------------------------------------------------------------------------- */

static void get_mib_counters (
    PHARDWARE  pHardware,
    int        first,
    int        count,
    ULONGLONG* cnCounter )
{
    PPORT_CONFIG pPortCfg;
    int          index;
    int          mib;
    int          port;

    memset( cnCounter, 0, sizeof( ULONGLONG ) * TOTAL_PORT_COUNTER_NUM );
    for ( index = 0, port = first; index < count; index++, port++ ) {
        pPortCfg = &pHardware->m_Port[ port ];
        for ( mib = 0; mib < TOTAL_PORT_COUNTER_NUM; mib++ )
            cnCounter[ mib ] += pPortCfg->cnCounter[ mib ];
    }
}  /* get_mib_counters */


static int display_mib_counters (
    PHARDWARE pHardware,
    int       first,
    int       count,
    char*     buf )
{
    int       mib;
    int       len = 0;
    ULONGLONG cnCounter[ TOTAL_PORT_COUNTER_NUM ];

    get_mib_counters( pHardware, first, count, cnCounter );
    for ( mib = 0; mib < TOTAL_PORT_COUNTER_NUM; mib++ ) {
        if ( buf )
            len += sprintf( buf + len, "%2x = %-10lu  ", mib,
                ( unsigned long ) cnCounter[ mib ]);
        else
            printk( "%2x = %-10lu  ", mib,
                ( unsigned long ) cnCounter[ mib ]);
        if ( ( mib % 4 ) == 3 ) {
            if ( buf )
                len += sprintf( buf + len, "\n" );
            else
                printk( "\n" );
        }
    }
    if ( ( mib % 4 ) != 0 ) {
        if ( buf )
            len += sprintf( buf + len, "\n" );
        else
            printk( "\n" );
    }
    return len;
}  /* display_mib_counter */

/* -------------------------------------------------------------------------- */

#define proc_dir_name  "driver/ksz884x_pci"


static TProcInfo ProcInfo[ PROC_LAST ] = {

    /* Read-only entries. */
    { PROC_INFO,                    "info",                 },

#ifdef DEF_KS8842
    { PROC_DYNAMIC,                 "dynamic",              },
    { PROC_STATIC,                  "static",               },
#endif
    { PROC_CLEAR_PME_STATUS,        "pme",                  },

    /* Read-write entries. */
    { PROC_SET_DUPLEX,              "duplex",               },
    { PROC_SET_SPEED,               "speed",                },
    { PROC_SET_FORCE,               "force",                },

#ifdef DEF_KS8842
    { PROC_SET_MIB,                 "mib",                  },
    { PROC_SET_BROADCAST_STORM,     "broadcast_percent",    },
    { PROC_SET_DIFFSERV,            "v_diffserv",           },
    { PROC_SET_802_1P,              "v_802_1p",             },
#endif

    /* Port specific read-only entries. */
    { PROC_GET_CABLE_STATUS,        "cable_status",         },
    { PROC_GET_LINK_STATUS,         "link_status",          },

    { PROC_GET_PORT_DUPLEX,         "duplex",               },
    { PROC_GET_PORT_SPEED,          "speed",                },
    { PROC_GET_PORT_MIB,            "mib",                  },

    /* Port specific read-write entries. */
    { PROC_SET_CAPABILITIES,        "set_capabilities",     },

#ifdef DEF_KS8842
    { PROC_ENABLE_BROADCAST_STORM,  "broadcast_storm",      },
    { PROC_ENABLE_DIFFSERV,         "diffserv",             },
    { PROC_ENABLE_802_1P,           "p_802_1p",             },
    { PROC_ENABLE_PRIORITY_QUEUE,   "priority_queue",       },

    { PROC_SET_PORT_BASED,          "port_based",           },

    { PROC_ENABLE_PRIORITY_RATE,    "priority_rate",        },
    { PROC_SET_RX_P0_RATE,          "rx_p0_rate",           },
    { PROC_SET_RX_P1_RATE,          "rx_p1_rate",           },
    { PROC_SET_RX_P2_RATE,          "rx_p2_rate",           },
    { PROC_SET_RX_P3_RATE,          "rx_p3_rate",           },
    { PROC_SET_TX_P0_RATE,          "tx_p0_rate",           },
    { PROC_SET_TX_P1_RATE,          "tx_p1_rate",           },
    { PROC_SET_TX_P2_RATE,          "tx_p2_rate",           },
    { PROC_SET_TX_P3_RATE,          "tx_p3_rate",           },
#endif
};


/*
    read_proc

    Description:
        This function process the read operation of /proc files.

    Parameters:

    Return (int):
        The length of buffer data returned.
*/

static
int read_proc (
    char*  buf,
    char** start,
    off_t  offset,
    int    count,
    int*   eof,
    void*  data )
{
#ifdef DEF_KS8842
    int              i;
#endif
    int              len = 0;
    int              processed = TRUE;
    int              rc;
    PTProcPortInfo   proc = ( PTProcPortInfo ) data;
    PTProcInfo       info = proc->info;
    struct dev_info* priv = proc->priv;
    PHARDWARE        pHardware = &priv->hw;
    PPORT_INFO       pInfo;
    DWORD            dwData[ 5 * 2 ];

    /* Why want more? */
    if ( offset ) {
        *eof = 1;
        return( 0 );
    }

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_INC_USE_COUNT;
#endif

    if ( down_interruptible( &priv->proc_sem ) ) {

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
        MOD_DEC_USE_COUNT;
#endif
        return -ERESTARTSYS;
    }

#if 0
    len += sprintf( buf + len, "%s:%d %d\n", info->proc_name, info->proc_num,
        proc->port );
#endif
    switch ( info->proc_num ) {
        case PROC_SET_DUPLEX:
            len += sprintf( buf + len, "%u; ",
                pHardware->m_bDuplex );
            if ( MediaStateConnected == pHardware->m_ulHardwareState ) {
                if ( 1 == pHardware->m_ulDuplex )
                    len += sprintf( buf + len, "half-duplex\n" );
                else if ( 2 == pHardware->m_ulDuplex )
                    len += sprintf( buf + len, "full-duplex\n" );
            }
            else
                len += sprintf( buf + len, "unlinked\n" );
            break;
        case PROC_SET_SPEED:
            len += sprintf( buf + len, "%u; ",
                pHardware->m_bSpeed );
            if ( MediaStateConnected == pHardware->m_ulHardwareState ) {
                len += sprintf( buf + len, "%u\n",
                    pHardware->m_ulTransmitRate / 10000 );
            }
            else
                len += sprintf( buf + len, "unlinked\n" );
            break;
        case PROC_SET_FORCE:
            len += sprintf( buf + len, "%u\n",
                pHardware->m_bForceLink );
            break;
        case PROC_GET_PORT_DUPLEX:
            pInfo = &pHardware->m_PortInfo[ proc->port ];
            if ( MediaStateConnected == pInfo->ulHardwareState ) {
                if ( 1 == pInfo->bDuplex )
                    len += sprintf( buf + len, "half-duplex\n" );
                else if ( 2 == pInfo->bDuplex )
                    len += sprintf( buf + len, "full-duplex\n" );
            }
            else
                len += sprintf( buf + len, "unlinked\n" );
            break;
        case PROC_GET_PORT_SPEED:
            pInfo = &pHardware->m_PortInfo[ proc->port ];
            if ( MediaStateConnected == pInfo->ulHardwareState ) {
                len += sprintf( buf + len, "%u\n",
                    pInfo->ulSpeed / 10000 );
            }
            else
                len += sprintf( buf + len, "unlinked\n" );
            break;

#ifdef DEF_KS8842
        case PROC_SET_BROADCAST_STORM:
            len += sprintf( buf + len, "%u%%\n",
                pHardware->m_bBroadcastPercent );
            break;
        case PROC_SET_DIFFSERV:
            for ( i = 0; i < 64; i++ ) {
                len += sprintf( buf + len, "%2u=0x%02X ", i, ( int )
                    pHardware->m_wDiffServ[ i ]);
                if ( ( i % 8 ) == 7 )
                    len += sprintf( buf + len, "\n" );
            }
            break;
        case PROC_SET_802_1P:
            for ( i = 0; i < 8; i++ )
                len += sprintf( buf + len, "%u=%u ", i,
                    pHardware->m_b802_1P_Priority[ i ]);
            len += sprintf( buf + len, "\n" );
            break;
        case PROC_SET_PORT_BASED:
            len += sprintf( buf + len, "%u\n",
                pHardware->m_Port[ proc->port ].bPortPriority );
            break;
        case PROC_SET_RX_P0_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwRxRate[ 0 ]);
            break;
        case PROC_SET_RX_P1_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwRxRate[ 1 ]);
            break;
        case PROC_SET_RX_P2_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwRxRate[ 2 ]);
            break;
        case PROC_SET_RX_P3_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwRxRate[ 3 ]);
            break;
        case PROC_SET_TX_P0_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwTxRate[ 0 ]);
            break;
        case PROC_SET_TX_P1_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwTxRate[ 1 ]);
            break;
        case PROC_SET_TX_P2_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwTxRate[ 2 ]);
            break;
        case PROC_SET_TX_P3_RATE:
            len += sprintf( buf + len, "%u\n", ( int )
                pHardware->m_Port[ proc->port ].dwTxRate[ 3 ]);
            break;
#endif
        default:
            processed = FALSE;
    }

    /* Require hardware to be acquired first. */
    if ( !processed ) {
        if ( ( rc = AcquireHardware( priv, FALSE, TRUE )) ) {
            up( &priv->proc_sem );
            return( rc );
        }
        switch ( info->proc_num ) {
#ifdef DEF_KS8842
            case PROC_ENABLE_BROADCAST_STORM:
                len += sprintf( buf + len, "%d\n",
                    PortGetBroadcastStorm( pHardware, proc->port ));
                break;
            case PROC_ENABLE_DIFFSERV:
                len += sprintf( buf + len, "%d\n",
                    PortGetDiffServ( pHardware, proc->port ));
                break;
            case PROC_ENABLE_802_1P:
                len += sprintf( buf + len, "%d\n",
                    PortGet802_1P( pHardware, proc->port ));
                break;
            case PROC_ENABLE_PRIORITY_QUEUE:
                len += sprintf( buf + len, "%d\n",
                    PortGetPriority( pHardware, proc->port ));
                break;
            case PROC_ENABLE_PRIORITY_RATE:
                len += sprintf( buf + len, "%d\n",
                    SwitchGetPriorityRate( pHardware, proc->port ));
                break;
#endif
            case PROC_GET_CABLE_STATUS:
                HardwareGetCableStatus( pHardware, proc->port, dwData );
                len += sprintf( buf + len,
                    "^[%u:%u, %u:%u, %u:%u, %u:%u, %u:%u]$\n",
                    dwData[ 0 ], dwData[ 1 ],
                    dwData[ 2 ], dwData[ 3 ],
                    dwData[ 4 ], dwData[ 5 ],
                    dwData[ 6 ], dwData[ 7 ],
                    dwData[ 8 ], dwData[ 9 ]);
                break;
            case PROC_GET_LINK_STATUS:
                HardwareGetLinkStatus( pHardware, proc->port, dwData );
                len += sprintf( buf + len,
                    "^[%08X, %X, %08X, %08X, %08X, %08X]$\n",
                    dwData[ 0 ], dwData[ 1 ], dwData[ 2 ],
                    dwData[ 3 ], dwData[ 4 ], dwData[ 5 ]);
                break;
            case PROC_INFO:
                break;

#ifdef DEF_KS8842
            case PROC_SET_MIB:
                len += display_mib_counters( pHardware, HOST_PORT, 1,
                    buf + len );
                break;
            case PROC_DYNAMIC:
                SwitchDumpDynMacTable( pHardware );
                break;
            case PROC_STATIC:
                SwitchDumpStaticMacTable( pHardware );
                break;
#endif
            case PROC_CLEAR_PME_STATUS:
                HardwareClearWolPMEStatus( pHardware );
                break;
            case PROC_GET_PORT_MIB:
                len += display_mib_counters( pHardware, proc->port, 1,
                    buf + len );
            default:
                break;
        }
        ReleaseHardware( priv, FALSE );
    }
    up( &priv->proc_sem );
    *eof = 1;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_DEC_USE_COUNT;
#endif

    return len;
}  /* read_proc */


/*
    write_proc

    Description:
        This function process the write operation of /proc files.

    Parameters:

    Return (int):
        The length of buffer data accepted.
*/

static
int write_proc (
    struct file*  file,
    const char*   buffer,
    unsigned long count,
    void*         data )
{
    int              len;
    int              num;
    int              rc;
    char             str[ 22 ];
    PTProcPortInfo   proc = ( PTProcPortInfo ) data;
    PTProcInfo       info = proc->info;
    struct dev_info* priv = proc->priv;
    PHARDWARE        pHardware = &priv->hw;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_INC_USE_COUNT;
#endif

    if ( count > 20 )
        len = 20;
    else
        len = count;
    if ( copy_from_user( str, buffer, len ) ) {

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
        MOD_DEC_USE_COUNT;
#endif
        return -EFAULT;
    }
    if ( down_interruptible( &priv->proc_sem ) ) {

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
        MOD_DEC_USE_COUNT;
#endif
        return -ERESTARTSYS;
    }
    if ( ( rc = AcquireHardware( priv, FALSE, TRUE )) ) {
        up( &priv->proc_sem );
        return( rc );
    }
    if ( '0' == str[ 0 ]  &&  'x' == str[ 1 ] ) {
        sscanf( &str[ 2 ], "%x", ( unsigned int* ) &num );
    }
    else if ( '0' == str[ 0 ]  &&  'b' == str[ 1 ] ) {
        int i = 2;

        num = 0;
        while ( str[ i ] ) {
            num <<= 1;
            num |= str[ i ] - '0';
            i++;
        }
    }
    else if ( '0' == str[ 0 ]  &&  'd' == str[ 1 ] ) {
        sscanf( &str[ 2 ], "%u", &num );
    }
    else
        sscanf( str, "%d", &num );
    switch ( info->proc_num ) {
        case PROC_SET_DUPLEX:
            if ( num <= 2 )
                pHardware->m_bDuplex = ( BYTE ) num;
            break;
        case PROC_SET_SPEED:
            if ( 0 == num  ||  10 == num  ||  100 == num )
                pHardware->m_bSpeed = ( BYTE ) num;
            break;
        case PROC_SET_FORCE:
            pHardware->m_bForceLink = ( BYTE ) num;
            break;
        case PROC_SET_CAPABILITIES:
            HardwareSetCapabilities( pHardware, proc->port, num );
            break;
#ifdef DEF_KS8842
        case PROC_SET_BROADCAST_STORM:
            HardwareConfigBroadcastStorm( pHardware, num );
            break;
        case PROC_ENABLE_BROADCAST_STORM:
            if ( !num )
                SwitchDisableBroadcastStorm( pHardware, proc->port );
            else
                SwitchEnableBroadcastStorm( pHardware, proc->port );
            break;
        case PROC_ENABLE_DIFFSERV:
            if ( !num )
                SwitchDisableDiffServ( pHardware, proc->port );
            else
                SwitchEnableDiffServ( pHardware, proc->port );
            break;
        case PROC_ENABLE_802_1P:
            if ( !num )
                SwitchDisable802_1P( pHardware, proc->port );
            else
                SwitchEnable802_1P( pHardware, proc->port );
            break;
        case PROC_ENABLE_PRIORITY_QUEUE:
            if ( !num )
                SwitchDisableMultiQueue( pHardware, proc->port );
            else
                SwitchEnableMultiQueue( pHardware, proc->port );
            break;
#if 0
        case PROC_SET_DIFFSERV:
            pHardware->m_wDiffServ[ 0 ] = ( USHORT ) num;
            break;
        case PROC_SET_802_1P:
            HardwareConfig802_1P_Priority( pHardware, ( UCHAR ) num );
            break;
#endif
        case PROC_SET_PORT_BASED:
            SwitchConfigPortBased( pHardware, proc->port, num );
            break;
        case PROC_ENABLE_PRIORITY_RATE:
            if ( !num )
                SwitchDisablePriorityRate( pHardware, proc->port );
            else
                SwitchEnablePriorityRate( pHardware, proc->port );
            break;
        case PROC_SET_RX_P0_RATE:
            HardwareConfigRxPriorityRate( pHardware, proc->port, 0, num );
            break;
        case PROC_SET_RX_P1_RATE:
            HardwareConfigRxPriorityRate( pHardware, proc->port, 1, num );
            break;
        case PROC_SET_RX_P2_RATE:
            HardwareConfigRxPriorityRate( pHardware, proc->port, 2, num );
            break;
        case PROC_SET_RX_P3_RATE:
            HardwareConfigRxPriorityRate( pHardware, proc->port, 3, num );
            break;
        case PROC_SET_TX_P0_RATE:
            HardwareConfigTxPriorityRate( pHardware, proc->port, 0, num );
            break;
        case PROC_SET_TX_P1_RATE:
            HardwareConfigTxPriorityRate( pHardware, proc->port, 1, num );
            break;
        case PROC_SET_TX_P2_RATE:
            HardwareConfigTxPriorityRate( pHardware, proc->port, 2, num );
            break;
        case PROC_SET_TX_P3_RATE:
            HardwareConfigTxPriorityRate( pHardware, proc->port, 3, num );
            break;
#endif
        default:
            printk( KERN_ALERT "write_proc:%d\n", info->proc_num );
    }
    ReleaseHardware( priv, FALSE );
    up( &priv->proc_sem );

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_DEC_USE_COUNT;
#endif

    return len;
}  /* write_proc */

/* -------------------------------------------------------------------------- */

#ifdef CONFIG_NET_POLL_CONTROLLER
static void netdev_netpoll ( struct net_device* dev )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;

    HardwareDisableInterruptSync( &hw_priv->hw );
    ks8842p_dev_interrupt( dev->irq, dev );
}  /* netdev_netpoll */
#endif


#ifdef CONFIG_STP_SUPPORT
static
void bridge_change (
	PHARDWARE pSwitch )
{
	int port;
	u8  bMember;

	/* No ports in forwarding state. */
	if ( !pSwitch->m_bMember ) {
		PortSet_STP_State( pSwitch, SWITCH_PORT_NUM,
			STP_STATE_SIMPLE );
		SwitchBlockAddress( pSwitch );
	}
        for ( port = 0; port < SWITCH_PORT_NUM; port++ ) {
		if ( STP_STATE_FORWARDING ==
				pSwitch->m_Port[ port ].nSTP_State )
			bMember = HOST_MASK | pSwitch->m_bMember;
		else
			bMember = HOST_MASK | ( 1 << port );
		if ( bMember != pSwitch->m_Port[ port ].bMember ) {
			HardwareConfigPortBaseVlan( pSwitch, port, bMember );
		}
	}
}  /* bridge_change */
#endif


/*
    netdev_close

    Description:
        This function process the close operation of network device.  This is
        caused by the user command "ifconfig ethX down."

    Parameters:
        struct net_device* dev
            Pointer to network device.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

static int netdev_close (
    struct net_device* dev )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    int              rc;

#ifdef TWO_NETWORK_INTERFACE
    PPORT            pPort = priv->port;

#ifdef CONFIG_STP_SUPPORT
    int              port;
#endif
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_DEC_USE_COUNT;
#endif
    priv->opened--;

    if ( !( priv->opened ) ) {
        netif_stop_queue( dev );

        if ( ( rc = AcquireHardware( hw_priv, FALSE, TRUE )) ) {
            return( rc );
        }

#ifdef TWO_NETWORK_INTERFACE
        PortSet_STP_State( pHardware, pPort->FirstPort, STP_STATE_DISABLED );

#ifdef CONFIG_STP_SUPPORT
        port = 1 << pPort->FirstPort;
        if ( ( pHardware->m_bMember & port ) ) {
            pHardware->m_bMember &= ~port;
            bridge_change( pHardware );
        }
#endif
#endif
        if ( priv->multicast )
            --pHardware->m_bAllMulticast;
        if ( priv->promiscuous )
            --pHardware->m_bPromiscuous;

        hw_priv->opened--;
        if ( !( hw_priv->opened ) ) {

            if ( hw_priv->MonitorTimerInfo.nCount ) {
                StopTimer( &hw_priv->MonitorTimerInfo );
            }

            HardwareDisableInterrupt( pHardware );
            HardwareDisable( pHardware );

            /* Delay for receive task to stop scheduling itself. */
            DelayMillisec( 2000 / HZ );

            tasklet_disable( &hw_priv->rx_tasklet );
            tasklet_disable( &hw_priv->tx_tasklet );
            free_irq( dev->irq, hw_priv->dev );

            transmit_reset( hw_priv );
            HardwareResetRxPackets( pHardware );
            HardwareResetTxPackets( pHardware );
            InitBuffers( &hw_priv->m_RxBufInfo );
            InitBuffers( &hw_priv->m_TxBufInfo );
            InitReceiveBuffers( hw_priv );

        }

        ReleaseHardware( hw_priv, FALSE );
    }

#ifdef DBG_
{
    int i;

    DBG_PRINT( "counters:\n" );
    for ( i = OID_COUNTER_FIRST; i < OID_COUNTER_LAST; i++ )
    {
        DBG_PRINT( "%u = %u, %u\n", i,
            ( int ) pHardware->m_cnCounter[ 0 ][ i ],
            ( int ) pHardware->m_cnCounter[ 1 ][ i ]);
    }
    DBG_PRINT( "wait delays:\n" );
    for ( i = WAIT_DELAY_FIRST; i < WAIT_DELAY_LAST; i++ )
    {
        DBG_PRINT( "%u = %u\n", i, pHardware->m_nWaitDelay[ i ]);
    }
    DBG_PRINT( "bad:\n" );
    for ( i = COUNT_BAD_FIRST; i < COUNT_BAD_LAST; i++ )
    {
        DBG_PRINT( "%u = %u\n", i, ( int ) pHardware->m_nBad[ i ]);
    }
    DBG_PRINT( "good:\n" );
    for ( i = COUNT_GOOD_FIRST; i < COUNT_GOOD_LAST; i++ )
    {
        DBG_PRINT( "%u = %u\n", i, ( int ) pHardware->m_nGood[ i ]);
    }
}
#endif

    return 0;
}  /* netdev_close */


static unsigned long next_jiffies = 0;

/*
    netdev_open

    Description:
        This function process the open operation of network device.  This is
        caused by the user command "ifconfig ethX up."

    Parameters:
        struct net_device* dev
            Pointer to network device.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

static int netdev_open (
    struct net_device* dev )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PPORT            pPort = priv->port;
    int              index;
    int              port;
    int              rc = 0;

    if ( !( priv->opened ) ) {
        priv->multicast = 0;
        priv->promiscuous = 0;

        /* Reset device statistics. */
        memset( &priv->stats, 0, sizeof( struct net_device_stats ));
        memset(( void* ) pHardware->m_cnCounter[ pPort->FirstPort ], 0,
            ( sizeof( ULONGLONG ) * OID_COUNTER_LAST ) );

        if ( ( rc = AcquireHardware( hw_priv, FALSE, TRUE )) ) {
            return( rc );
        }

        if ( !( hw_priv->opened ) )
        {
            for ( port = 0; port < pHardware->m_bPortCount; port++ ) {

                /* Initialize to invalid value so that link detection is done.
                */
                pHardware->m_PortInfo[ port ].bLinkPartner = 0xFF;
                pHardware->m_PortInfo[ port ].ulHardwareState =
                    MediaStateDisconnected;
                if ( next_jiffies < jiffies ) {
                    next_jiffies = jiffies + HZ * 4;
                }
                else {
                    next_jiffies += HZ * 2;
                }
                hw_priv->Counter[ port ].time = next_jiffies;
            }
            if ( pHardware->m_bMibPortCount > 1 ) {
                next_jiffies += HZ * 2;
                hw_priv->Counter[ HOST_PORT ].time = next_jiffies;
            }

            /* Remember the network device that requests interrupts. */
            hw_priv->dev = dev;
            if ( ( rc = request_irq( dev->irq, DEV_INTERRUPT_PTR,
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 18))
                    IRQF_SHARED,
#else
                    SA_SHIRQ,
#endif
                    dev->name, dev )) ) {
                ReleaseHardware( hw_priv, FALSE );
                return( rc );
            }
            tasklet_enable( &hw_priv->rx_tasklet );
            tasklet_enable( &hw_priv->tx_tasklet );

            pHardware->m_bPromiscuous = 0;
            pHardware->m_bAllMulticast = 0;
            pHardware->m_bMulticastListSize = 0;

            HardwareReset( pHardware );
            HardwareSetup( pHardware );
            HardwareSetDescriptorBase( pHardware,
                pHardware->m_TxDescInfo.ulRing,
                pHardware->m_RxDescInfo.ulRing );
            HardwareSwitchSetup( pHardware );

            if ( pHardware->m_bMacOverrideAddr )
                HardwareSetAddress( pHardware );
        }

#ifdef TWO_NETWORK_INTERFACE
        PortSet_STP_State( pHardware, pPort->FirstPort, STP_STATE_SIMPLE );
        if ( pPort->FirstPort > 0 )
            HardwareAddrFilterAdd( pHardware, dev->dev_addr );
#endif

        for ( index = 0, port = pPort->FirstPort;
                index < pPort->MibPortCount; index++, port++ ) {
            PortInitCounters( pHardware, port );
        }

        if ( !( hw_priv->opened ) ) {

#ifdef TWO_NETWORK_INTERFACE
            PortInitCounters( pHardware, HOST_PORT );
#endif
            HardwareSetupInterrupt( pHardware );
            HardwareEnable( pHardware );
            HardwareEnableInterrupt( pHardware );

            StartTimer( &hw_priv->MonitorTimerInfo,
                hw_priv->MonitorTimerInfo.nTime );
        }
        ReleaseHardware( hw_priv, FALSE );

        hw_priv->opened++;

        priv->media_state = (( PPORT_INFO ) pPort->pLinked )->ulHardwareState;

        if ( MediaStateConnected == priv->media_state )
            netif_carrier_on( dev );
        else
            netif_carrier_off( dev );
        if ( netif_msg_link( priv ) ) {
            printk(KERN_INFO "%s link %s\n", dev->name,
                ( MediaStateConnected == priv->media_state ? "on" : "off" ));
        }

        netif_start_queue( dev );
    }
    priv->opened++;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_INC_USE_COUNT;
#endif

    return 0;
}  /* netdev_open */


/* RX errors = rx_errors */
/* RX dropped = rx_dropped */
/* RX overruns = rx_fifo_errors */
/* RX frame = rx_crc_errors + rx_frame_errors + rx_length_errors */
/* TX errors = tx_errors */
/* TX dropped = tx_dropped */
/* TX overruns = tx_fifo_errors */
/* TX carrier = tx_aborted_errors + tx_carrier_errors + tx_window_errors */
/* collisions = collisions */

/*
    dev_query_statistics

    Description:
        This function returns the statistics of the network device.  The device
        needs not be opened.

    Parameters:
        struct net_device* dev
            Pointer to network device.

    Return (struct net_device_stat*):
        Network device statistics.
*/

static struct net_device_stats* dev_query_statistics (
    struct net_device* dev )
{
    struct dev_priv* priv = netdev_priv( dev );
    PPORT            pPort = priv->port;
    PHARDWARE        pHardware = &priv->pDevInfo->hw;
    PPORT_CONFIG     pPortCfg;
    int              index;
    int              port;

    /* Reset to zero to add count later. */
    priv->stats.rx_errors =
    priv->stats.tx_errors =
    priv->stats.multicast =
    priv->stats.collisions =
    priv->stats.rx_length_errors =
    priv->stats.rx_crc_errors =
    priv->stats.rx_frame_errors =
    priv->stats.tx_window_errors = 0;

    for ( index = 0, port = pPort->FirstPort; index < pPort->PortCount;
            index++, port++ ) {
        pPortCfg = &pHardware->m_Port[ port ];

        priv->stats.rx_errors += pHardware->m_cnCounter[ port ]
            [ OID_COUNTER_RCV_ERROR ];
        priv->stats.tx_errors += pHardware->m_cnCounter[ port ]
            [ OID_COUNTER_XMIT_ERROR ];

        priv->stats.multicast += ( unsigned long )
            pPortCfg->cnCounter[ MIB_COUNTER_RX_MULTICAST ];

        priv->stats.collisions += ( unsigned long )
            pPortCfg->cnCounter[ MIB_COUNTER_TX_TOTAL_COLLISION ];

        priv->stats.rx_length_errors += ( unsigned long )(
            pPortCfg->cnCounter[ MIB_COUNTER_RX_UNDERSIZE ] +
            pPortCfg->cnCounter[ MIB_COUNTER_RX_FRAGMENT ] +
            pPortCfg->cnCounter[ MIB_COUNTER_RX_OVERSIZE ] +
            pPortCfg->cnCounter[ MIB_COUNTER_RX_JABBER ]);
        priv->stats.rx_crc_errors += ( unsigned long )
            pPortCfg->cnCounter[ MIB_COUNTER_RX_CRC_ERR ];
        priv->stats.rx_frame_errors += ( unsigned long )(
            pPortCfg->cnCounter[ MIB_COUNTER_RX_ALIGNMENT_ERR ] +
            pPortCfg->cnCounter[ MIB_COUNTER_RX_SYMBOL_ERR ]);

        priv->stats.tx_window_errors += ( unsigned long )
            pPortCfg->cnCounter[ MIB_COUNTER_TX_LATE_COLLISION ];
    }

#ifdef DEBUG_COUNTER
    if ( jiffies - query_jiffies >= 1000 * HZ / 1000 ) {
        printk( "I=%u L=%u R=%u\n",
            pHardware->m_nGood[ COUNT_GOOD_INT ],
            pHardware->m_nGood[ COUNT_GOOD_INT_LOOP ],
            pHardware->m_nGood[ COUNT_GOOD_INT_RX ]);
        pHardware->m_nGood[ COUNT_GOOD_INT ] =
        pHardware->m_nGood[ COUNT_GOOD_INT_LOOP ] =
        pHardware->m_nGood[ COUNT_GOOD_INT_RX ] =
            0;
    }
    query_jiffies = jiffies;
#endif

    return( &priv->stats );
}  /* dev_query_statistics */


/*
    device_set_mac_address

    Description:
        This function is used to set the MAC address of the network device.

    Parameters:
        struct net_device* dev
            Pointer to network device.

        void* addr
            Buffer of MAC address.

    Return (int):
        Zero to indicate success.
*/

static int device_set_mac_address (
    struct net_device* dev,
    void*              addr )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    struct sockaddr* mac = addr;
    int              rc;
    UINT             InterruptMask;

#ifdef DBG
    printk( "set mac address\n" );
#endif

    if ( ( rc = AcquireHardware( hw_priv, FALSE, TRUE )) ) {
        return( rc );
    }

#ifdef TWO_NETWORK_INTERFACE
    if ( priv->port->FirstPort > 0 ) {
        HardwareAddrFilterDel( pHardware, dev->dev_addr );
    }
    else
#endif
    {
        pHardware->m_bMacOverrideAddr = TRUE;
        memcpy( pHardware->m_bOverrideAddress, mac->sa_data,
            MAC_ADDRESS_LENGTH );
    }

    memcpy( dev->dev_addr, mac->sa_data, MAX_ADDR_LEN );

    InterruptMask = HardwareBlockInterrupt( pHardware );

#ifdef TWO_NETWORK_INTERFACE
    if ( priv->port->FirstPort > 0 )
        HardwareAddrFilterAdd( pHardware, dev->dev_addr );
    else
#endif
        HardwareSetAddress( pHardware );
    HardwareRestoreInterrupt( pHardware, InterruptMask );

    ReleaseHardware( hw_priv, FALSE );

    return 0;
}  /* device_set_mac_address */


/*
    dev_set_multicast_list

    Description:
        This routine is used to set multicast addresses or put the network
        device into promiscuous mode.

    Parameters:
        struct net_device* dev
            Pointer to network device.

        void* addr
            Buffer of MAC address.

    Return (None):
*/

static void dev_set_multicast_list (
    struct net_device* dev )
{
    struct dev_priv*    priv = netdev_priv( dev );
    struct dev_info*    hw_priv = priv->pDevInfo;
    PHARDWARE           pHardware = &hw_priv->hw;
    int                 rc;
    int                 promiscuous = ( dev->flags & IFF_PROMISC );

#ifdef TWO_NETWORK_INTERFACE
    int                 multicast = ( dev->flags &
        ( IFF_ALLMULTI | IFF_MULTICAST ));

#else
    struct dev_mc_list* mc_ptr;
    int                 multicast = ( dev->flags & IFF_ALLMULTI );
#endif

    if ( ( rc = AcquireHardware( hw_priv, FALSE, FALSE )) ) {
        return;
    }

    if ( promiscuous != priv->promiscuous ) {
        u8 bPromiscuous = pHardware->m_bPromiscuous;

        if ( promiscuous )
            ++pHardware->m_bPromiscuous;
        else
            --pHardware->m_bPromiscuous;
        priv->promiscuous = promiscuous;

        /* Turn on/off promiscuous mode. */
        if ( pHardware->m_bPromiscuous <= 1  &&
                bPromiscuous <= 1 )
            HardwareSetPromiscuous( pHardware, pHardware->m_bPromiscuous );

#ifdef CONFIG_STP_SUPPORT
        if ( !promiscuous  &&  dev->br_port ) {
            int port = priv->port->FirstPort;

            PortSet_STP_State( pHardware, port, STP_STATE_DISABLED );
            port = 1 << port;
            if ( ( pHardware->m_bMember & port ) ) {
                pHardware->m_bMember &= ~port;
                bridge_change( pHardware );
            }
        }
#endif
    }

    if ( multicast != priv->multicast ) {
        u8 bAllMulticast = pHardware->m_bAllMulticast;

        if ( multicast )
            ++pHardware->m_bAllMulticast;
        else
            --pHardware->m_bAllMulticast;
        priv->multicast = multicast;

        /* Turn on/off all multicast mode. */
        if ( pHardware->m_bAllMulticast <= 1  &&
                bAllMulticast <= 1 )
            HardwareSetMulticast( pHardware, pHardware->m_bAllMulticast );
    }

#ifndef TWO_NETWORK_INTERFACE
    if ( ( dev->flags & IFF_MULTICAST )  &&  dev->mc_count ) {
        if ( dev->mc_count <= MAX_MULTICAST_LIST ) {
            int i = 0;

            for ( mc_ptr = dev->mc_list; mc_ptr; mc_ptr = mc_ptr->next ) {
                if ( !( *mc_ptr->dmi_addr & 1 ) )
                    continue;
                if ( i >= MAX_MULTICAST_LIST )
                    break;
                memcpy( pHardware->m_bMulticastList[ i++ ], mc_ptr->dmi_addr,
                    MAC_ADDRESS_LENGTH );
            }
            pHardware->m_bMulticastListSize = ( BYTE ) i;
            HardwareSetGroupAddress( pHardware );
        }

        /* List too big to support so turn on all multicast mode. */
        else {
            pHardware->m_bMulticastListSize = 255;
            if ( !multicast ) {
                pHardware->m_bAllMulticast = TRUE;
                HardwareSetMulticast( pHardware, TRUE );
            }
        }
    }
    else {
        pHardware->m_bMulticastListSize = 0;
        HardwareClearMulticast( pHardware );
        if ( !multicast  &&  pHardware->m_bAllMulticast ) {
            pHardware->m_bAllMulticast = FALSE;
            HardwareSetMulticast( pHardware, FALSE );
        }
    }
#endif

    ReleaseHardware( hw_priv, FALSE );
}  /* dev_set_multicast_list */


#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 21))
/*
    mdio_read

    Description:
        This function returns the PHY register value.

    Parameters:
        struct net_device* dev
            Pointer to network device.

        int phy_id
            The PHY id.

        int reg_num
            The register number.

    Return (int):
        The register value.
*/

static int mdio_read (
    struct net_device *dev,
    int               phy_id,
    int               reg_num )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PPORT_INFO       pLinked = priv->port->pLinked;
    PPORT            pPort = &pLinked->port;
    int              port = pPort->FirstPort;
    u16              val_out;

    HardwareReadPhy( pHardware, port, reg_num << 1, &val_out );

    return val_out;
}  /* mdio_read */


/*
    mdio_write

    Description:
        This procedure sets the PHY register value.

    Parameters:
        struct net_device* dev
            Pointer to network device.

        int phy_id
            The PHY id.

        int reg_num
            The register number.

        int val
            The register value.

    Return (None):
*/

static void mdio_write (
    struct net_device *dev,
    int               phy_id,
    int               reg_num,
    int               val )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PPORT            pPort = priv->port;
    int              index;
    int              port;

    for ( index = 0, port = pPort->FirstPort; index < pPort->PortCount;
            index++, port++ )
        HardwareWritePhy( pHardware, port, reg_num << 1, val );
}  /* mdio_write */


#include "device_ethtool.c"
#endif


/*
    netdev_ioctl

    Description:
        This function is used to process I/O control calls.

    Parameters:
        struct net_device* dev
            Pointer to network device.

        struct ifreq* ifr
            Pointer to interface request structure.

        int cmd
            I/O control code.

    Return (int):
        Zero to indicate success.
*/

#define SIOCDEVDEBUG  ( SIOCDEVPRIVATE + 10 )

static int netdev_ioctl (
    struct net_device* dev,
    struct ifreq*      ifr,
    int                cmd )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;
    PHARDWARE        pHardware = &hw_priv->hw;
    PPORT            pPort = priv->port;
    PTRequest        req = ( PTRequest ) ifr->ifr_data;
    int              nResult;
    int              rc;
    int              result = 0;
    int              port;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    struct mii_ioctl_data *data = ( struct mii_ioctl_data* ) &ifr->ifr_data;

#else
    struct mii_ioctl_data *data = if_mii( ifr );
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_INC_USE_COUNT;
#endif

    if ( down_interruptible( &hw_priv->proc_sem ) ) {

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
        MOD_DEC_USE_COUNT;
#endif
        return -ERESTARTSYS;
    }

    // assume success
    nResult = DEV_IOC_OK;
    switch ( cmd ) {
    	case SIOCGMIIPHY:               /* Get address of MII PHY in use. */
        case SIOCDEVPRIVATE:            /* for binary compat, remove in 2.5 */
            data->phy_id = priv->id;

            /* Fallthrough... */

	case SIOCGMIIREG:               /* Read MII PHY register. */
        case SIOCDEVPRIVATE+1:          /* for binary compat, remove in 2.5 */
            if ( data->phy_id != priv->id  ||
                    data->reg_num >= 6 )
                result = -EIO;
            else {
                PPORT_INFO pPortInfo = pPort->pLinked;

                port = pPortInfo->port.FirstPort;
                HardwareReadPhy( pHardware, port, data->reg_num << 1,
                    &data->val_out );
            }
            break;

        case SIOCSMIIREG:               /* Write MII PHY register. */
        case SIOCDEVPRIVATE+2:          /* for binary compat, remove in 2.5 */
            if ( !capable( CAP_NET_ADMIN ) )
                result = -EPERM;
            else if ( data->phy_id != priv->id  ||
                    data->reg_num >= 6 )
                result = -EIO;
            else {
                int index;

                for ( index = 0, port = pPort->FirstPort;
                        index < pPort->PortCount;
                        index++, port++ )
                    HardwareWritePhy( pHardware, port, data->reg_num << 1,
                        data->val_in );
            }
            break;

        case SIOCDEVDEBUG:
            if ( req ) {
                switch ( req->nCmd ) {
                    case DEV_CMD_INIT:
                        req->param.bData[ 0 ] = 'M';
                        req->param.bData[ 1 ] = 'i';
                        req->param.bData[ 2 ] = 'c';
                        req->param.bData[ 3 ] = 'r';
                        req->nSize = 8 + 4;
                        break;
                    case DEV_CMD_GET:
                        switch ( req->nSubCmd ) {
                            case DEV_READ_REG:
                                hwread( req->param.nData[ 0 ],
                                    req->param.nData[ 1 ],
                                    req->param.nData[ 2 ]);
                                break;
                            case DEV_LINK_STATUS:
                                HardwareGetLinkStatus( pHardware, 0,
                                    &req->param.LinkStatus );
                                break;
                        }
                        break;
                    case DEV_CMD_PUT:
                        if ( ( rc = AcquireHardware( hw_priv, FALSE, TRUE )) ) {
                            result = -EAGAIN;
                            break;
                        }
                        switch ( req->nSubCmd ) {
                            case DEV_CAPABILITIES:
                                HardwareSetCapabilities( pHardware, 0,
                                    req->param.nData[ 0 ]);
                                break;
                            case DEV_CABLE_STATUS:
                                HardwareGetCableStatus( pHardware, 0,
                                    req->param.CableStatus );
                                break;
                        }
                        ReleaseHardware( hw_priv, FALSE );
                        break;
                    default:
                        nResult = DEV_IOC_INVALID_CMD;
                        break;
                }
            }
            if ( req ) {
                req->nResult = nResult;
            }
            else if ( !result )
                result = -EFAULT;
            break;
        default:
            result = -EOPNOTSUPP;
    }

    up( &hw_priv->proc_sem );

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 0))
    MOD_DEC_USE_COUNT;
#endif

    return( result );
}  /* netdev_ioctl */


void update_link (
    struct net_device* dev,
    struct dev_priv*   priv,
    PPORT              pPort )
{
    PPORT_INFO pPortInfo = pPort->pLinked;

    if ( priv->media_state != pPortInfo->ulHardwareState ) {
        priv->media_state = pPortInfo->ulHardwareState;
        if ( priv->opened ) {
            if ( MediaStateConnected == priv->media_state )
                netif_carrier_on( dev );
            else
                netif_carrier_off( dev );
            if ( netif_msg_link( priv ) ) {
                printk(KERN_INFO "%s link %s\n", dev->name,
                    ( MediaStateConnected == priv->media_state ?
                    "on" : "off" ));
            }
        }
    }
}  /* update_link */


#ifdef CONFIG_STP_SUPPORT
static
void monitor_ports (
	PHARDWARE pHardware )
{
	int    port;
	struct net_device* bridge_dev = NULL;
	u8     bMember = pHardware->m_bMember;

	for ( port = 0; port < SWITCH_PORT_NUM; port++ ) {
		struct net_device* dev =
			pHardware->m_PortInfo[ port ].pDevice;

		if ( !dev->br_port )
			continue;

		/* Port is under bridge. */
		bridge_dev = dev->br_port->br->dev;
		if ( pHardware->m_Port[ port ].nSTP_State !=
				dev->br_port->state ) {
			if ( STP_STATE_FORWARDING ==
					pHardware->m_Port[ port ].nSTP_State )
				bMember &= ~( 1 << port );
			if ( STP_STATE_FORWARDING ==
					dev->br_port->state )
				bMember |= ( 1 << port );
			PortSet_STP_State( pHardware, port,
				dev->br_port->state );
			if ( STP_STATE_LEARNING == dev->br_port->state ) {
				SwitchFlushDynMacTable( pHardware );
			}
		}
	}

	if ( bMember != pHardware->m_bMember ) {
		if ( !pHardware->m_bMember ) {
			PortSet_STP_State( pHardware, SWITCH_PORT_NUM,
				STP_STATE_BLOCKED );
			pHardware->m_bBridgeAddress[ 0 ] = 0xFF;
		}
		pHardware->m_bMember = bMember;
		bridge_change( pHardware );
	}

	/* At least one port in forwarding state. */
	if ( pHardware->m_bMember  &&  bridge_dev  &&
			memcmp( bridge_dev->dev_addr,
			pHardware->m_bBridgeAddress, MAC_ADDRESS_LENGTH ) ) {
		memcpy( pHardware->m_bBridgeAddress, bridge_dev->dev_addr,
			MAC_ADDRESS_LENGTH );
		SwitchPassAddress( pHardware );
	}
}  /* monitor_ports */
#endif


/*
    dev_monitor

    Description:
        This routine is run in a kernel timer to monitor the network device.

    Parameters:
        unsigned long ptr
            Pointer value to supplied data.

    Return (None):
*/

static void dev_monitor (
    unsigned long ptr )
{
    struct dev_info*   hw_priv = ( struct dev_info* ) ptr;
    PHARDWARE          pHardware = &hw_priv->hw;
    PTTimerInfo        pInfo = &hw_priv->MonitorTimerInfo;
    struct net_device* dev;
    struct dev_priv*   priv;
    PPORT              pPort;
    int                port;
    int                rc;

    if ( !( rc = AcquireHardware( hw_priv, FALSE, FALSE )) ) {

/* PHY change interrupt is working for KS8841. */
#ifdef DEF_KS8842
        if ( !pHardware->m_bLinkIntWorking )
            SwitchGetLinkStatus( pHardware );
#endif
        if ( hw_priv->pme_wait ) {
            if ( hw_priv->pme_wait <= jiffies ) {
                HardwareClearWolPMEStatus( pHardware );
                hw_priv->pme_wait = 0;

#ifdef DBG
                DBG_PRINT( "clear PME."NEWLINE );
#endif
            }
        }
        else if ( HardwareCheckWolPMEStatus( pHardware ) ) {

            /* PME is asserted.  Wait 2 seconds to clear it. */
            hw_priv->pme_wait = jiffies + HZ * 2;

#ifdef DBG
            DBG_PRINT( "PME asserted."NEWLINE );
#endif

#ifdef DEBUG_OVERRUN
            DBG_PRINT( "%u + %u = %u"NEWLINE, pHardware->m_ulReceived,
                pHardware->m_ulDropped,
                pHardware->m_ulReceived + pHardware->m_ulDropped );
            pHardware->m_ulReceived = pHardware->m_ulDropped = 0;
#endif
        }

        next_jiffies = jiffies;
        for ( port = 0; port < pHardware->m_bMibPortCount; port++ ) {

            /* Reading MIB counters or requested to read. */
            if ( pHardware->m_Port[ port ].bCurrentCounter  ||
                    1 == hw_priv->Counter[ port ].fRead ) {

                /* Need to process receive interrupt. */
                if ( PortReadCounters( pHardware, port ) )
                    goto dev_monitor_release;
                hw_priv->Counter[ port ].fRead = 0;

                /* Finish reading counters. */
                if ( 0 == pHardware->m_Port[ port ].bCurrentCounter ) {
                    hw_priv->Counter[ port ].fRead = 2;
                    wake_up_interruptible(
                        &hw_priv->Counter[ port ].wqhCounter );
                }
            }
            else if ( jiffies >= hw_priv->Counter[ port ].time ) {
                /* Only read MIB counters when the port is connected. */
                if ( HOST_PORT == port  ||  MediaStateConnected ==
                        pHardware->m_PortInfo[ port ].ulHardwareState )
                    hw_priv->Counter[ port ].fRead = 1;
                next_jiffies += HZ * 2 * pHardware->m_bMibPortCount;
                hw_priv->Counter[ port ].time = next_jiffies;
            }
            else if ( port != HOST_PORT  &&
                    pHardware->m_PortInfo[ port ].bLinkDown ) {
                pHardware->m_PortInfo[ port ].bLinkDown = 0;

                /* Read MIB counters the last time after link is lost. */
                hw_priv->Counter[ port ].fRead = 1;
            }
        }

        dev = pHardware->m_PortInfo[ MAIN_PORT ].pDevice;
        priv = netdev_priv( dev );
        pPort = priv->port;
        update_link( dev, priv, pPort );

#ifdef TWO_NETWORK_INTERFACE
        dev = pHardware->m_PortInfo[ OTHER_PORT ].pDevice;
        priv = netdev_priv( dev );
        pPort = priv->port;
        update_link( dev, priv, pPort );
#endif

dev_monitor_release:

#ifdef CONFIG_STP_SUPPORT
        monitor_ports( pHardware );
#endif
        ReleaseHardware( hw_priv, FALSE );
    }

    ++pInfo->cnCount;
    if ( pInfo->nCount > 0 ) {
        if ( pInfo->cnCount < pInfo->nCount ) {
            pInfo->pTimer->expires = jiffies + pInfo->nTime;
            add_timer( pInfo->pTimer );
        }
        else
            pInfo->nCount = 0;
    }
    else if ( pInfo->nCount < 0 ) {
        pInfo->pTimer->expires = jiffies + pInfo->nTime;
        add_timer( pInfo->pTimer );
    }
}  /* dev_monitor */


static struct hw_fn* ks8842_fn = NULL;
static int device_present = 0;
static int net_device_present = 0;


static void init_proc (
    struct dev_info* priv )
{
    PHARDWARE              pHardware = &priv->hw;
    struct proc_dir_entry* proc_info;
    struct proc_dir_entry* proc_set;
    int                    i;
    int                    port;
    char                   proc_name[ 40 ];

    sprintf( proc_name, "%s-%d", proc_dir_name, priv->id );
    sema_init( &priv->proc_sem, 1 );
    priv->proc_main = proc_mkdir( proc_name, NULL );
    if ( priv->proc_main ) {

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 30))
        priv->proc_main->owner = THIS_MODULE;
#endif
        port = HOST_PORT;

        /* Read-only entries. */
        for ( i = 0; i < PROC_SET_DUPLEX; i++ ) {
            priv->ProcPortInfo[ port ][ i ].priv = priv;
            priv->ProcPortInfo[ port ][ i ].info = &ProcInfo[ i ];
            priv->ProcPortInfo[ port ][ i ].port = port;
            proc_info = create_proc_read_entry( ProcInfo[ i ].proc_name,
                0444, priv->proc_main, read_proc,
                &priv->ProcPortInfo[ port ][ i ]);

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 30))
            if ( proc_info ) {
                proc_info->owner = THIS_MODULE;
            }
#endif
        }

        /* Can be written to. */
        for ( i = PROC_SET_DUPLEX; i < PROC_GET_CABLE_STATUS;
                i++ ) {
            priv->ProcPortInfo[ port ][ i ].priv = priv;
            priv->ProcPortInfo[ port ][ i ].info = &ProcInfo[ i ];
            priv->ProcPortInfo[ port ][ i ].port = port;
            proc_set = create_proc_entry( ProcInfo[ i ].proc_name, 0644,
                priv->proc_main );
            if ( proc_set ) {
                proc_set->data = &priv->ProcPortInfo[ port ][ i ];
                proc_set->read_proc = read_proc;
                proc_set->write_proc = write_proc;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 30))
                proc_set->owner = THIS_MODULE;
#endif
            }
        }
#ifdef DEF_KS8842
        for ( i = PROC_ENABLE_PRIORITY_RATE; i <= PROC_SET_TX_P3_RATE; i++ ) {
            priv->ProcPortInfo[ port ][ i ].priv = priv;
            priv->ProcPortInfo[ port ][ i ].info = &ProcInfo[ i ];
            priv->ProcPortInfo[ port ][ i ].port = port;
            proc_set = create_proc_entry( ProcInfo[ i ].proc_name, 0644,
                priv->proc_main );
            if ( proc_set ) {
                proc_set->data = &priv->ProcPortInfo[ port ][ i ];
                proc_set->read_proc = read_proc;
                proc_set->write_proc = write_proc;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 30))
                proc_set->owner = THIS_MODULE;
#endif
            }
        }
#endif
        for ( port = 0; port < pHardware->m_bPortCount; port++ ) {
            sprintf( proc_name, "%d", port );
            priv->proc_port[ port ] = proc_mkdir( proc_name, priv->proc_main );

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 30))
            priv->proc_port[ port ]->owner = THIS_MODULE;
#endif
            for ( i = PROC_GET_CABLE_STATUS; i < PROC_SET_CAPABILITIES;
                    i++ ) {
                priv->ProcPortInfo[ port ][ i ].priv = priv;
                priv->ProcPortInfo[ port ][ i ].info = &ProcInfo[ i ];
                priv->ProcPortInfo[ port ][ i ].port = port;
                proc_info = create_proc_read_entry( ProcInfo[ i ].proc_name,
                    0444, priv->proc_port[ port ], read_proc,
                    &priv->ProcPortInfo[ port ][ i ]);

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 30))
                if ( proc_info ) {
                    proc_info->owner = THIS_MODULE;
                }
#endif
            }
            for ( i = PROC_SET_CAPABILITIES; i < PROC_LAST; i++ ) {
                priv->ProcPortInfo[ port ][ i ].priv = priv;
                priv->ProcPortInfo[ port ][ i ].info = &ProcInfo[ i ];
                priv->ProcPortInfo[ port ][ i ].port = port;
                proc_set = create_proc_entry( ProcInfo[ i ].proc_name, 0644,
                    priv->proc_port[ port ]);
                if ( proc_set ) {
                    proc_set->data = &priv->ProcPortInfo[ port ][ i ];
                    proc_set->read_proc = read_proc;
                    proc_set->write_proc = write_proc;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 30))
                    proc_set->owner = THIS_MODULE;
#endif
                }
            }
        }
    }
}  /* init_proc */


#ifdef DEF_KS8841
static int   pause_time = 0;
#endif
static char* macaddr = ":";

#ifdef TWO_NETWORK_INTERFACE
static char* mac1addr = ":";
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 9))
#ifdef DEF_KS8841
module_param( pause_time, int, 0 );
#endif
module_param( macaddr, charp, 0 );

#ifdef TWO_NETWORK_INTERFACE
module_param( mac1addr, charp, 0 );
#endif

#else
#ifdef DEF_KS8841
MODULE_PARM( pause_time, "i" );
#endif
MODULE_PARM( macaddr, "s" );

#ifdef TWO_NETWORK_INTERFACE
MODULE_PARM( mac1addr, "s" );
#endif
#endif
#ifdef DEF_KS8841
MODULE_PARM_DESC( pause_time, "PAUSE time" );
#endif
MODULE_PARM_DESC( macaddr, "MAC address" );

#ifdef TWO_NETWORK_INTERFACE
MODULE_PARM_DESC( mac1addr, "Second MAC address" );
#endif


#ifdef DEF_KS8841
static UCHAR PauseFrame[] =
{
/*  0 */    0x01, 0x80, 0xC2, 0x00, 0x00, 0x01,     /* 01:80:c2:00:00:01 (DA) */
/*  6 */    0x08, 0x00, 0x70, 0x22, 0x44, 0x55,     /* 08:00:70:22:44:55 (SA) */
/* 12 */    0x88, 0x08,                             /* MAC control */
/* 14 */    0x00, 0x01,                             /* PAUSE */
/* 16 */    0x00, 0x00,                             /* time */
/* 18 */    0x00, 0x00,
};
#endif

int netdev_change_mtu (
    struct net_device* dev,
    int                new_mtu )
{
    struct dev_priv* priv = netdev_priv( dev );
    struct dev_info* hw_priv = priv->pDevInfo;

#ifdef DEF_KS8841
    PHARDWARE        pHardware = &hw_priv->hw;
    struct sk_buff*  skb;
    unsigned short*  wptr;

    if ( 0 == new_mtu ) {
        skb = dev_alloc_skb( 64 );
        if ( skb ) {
            skb->dev = dev;
            memcpy( skb->data, PauseFrame, 18 );
            memcpy( &skb->data[ 6 ], pHardware->m_bOverrideAddress, 6 );
            wptr = ( unsigned short* ) &skb->data[ 16 ];
            *wptr = htons( pause_time );
            skb->len = 18;
            dev->hard_start_xmit( skb, dev );
        }
        return 0;
    }
#endif
    if ( new_mtu < 60 )
        return -EINVAL;
    if ( new_mtu > hw_priv->m_nMTU - ETHERNET_HEADER_SIZE - 4 )
        return -EINVAL;
    if ( dev->mtu != new_mtu ) {
        dev->mtu = new_mtu;
    }
    return 0;
}  /* netdev_change_mtu */


static
void get_mac_addr (
    struct dev_info* hw_priv,
    u8*              macaddr,
    int              port )
{
    int i;
    int j;
    int got_num;
    int num;

    i = j = num = got_num = 0;
    while ( j < MAC_ADDRESS_LENGTH ) {
        if ( macaddr[ i ] ) {
            got_num = 1;
            if ( '0' <= macaddr[ i ]  &&  macaddr[ i ] <= '9' )
                num = num * 16 + macaddr[ i ] - '0';
            else if ( 'A' <= macaddr[ i ]  &&  macaddr[ i ] <= 'F' )
                num = num * 16 + 10 + macaddr[ i ] - 'A';
            else if ( 'a' <= macaddr[ i ]  &&  macaddr[ i ] <= 'f' )
                num = num * 16 + 10 + macaddr[ i ] - 'a';
            else if ( ':' == macaddr[ i ] )
                got_num = 2;
            else
                break;
        }
        else if ( got_num )
            got_num = 2;
        else
            break;
        if ( 2 == got_num ) {
            if ( MAIN_PORT == port )
                hw_priv->hw.m_bOverrideAddress[ j++ ] = ( BYTE ) num;

#ifdef TWO_NETWORK_INTERFACE
            else
                hw_priv->hw.m_bOtherAddress[ j++ ] = ( BYTE ) num;
#endif
            num = got_num = 0;
        }
        i++;
    }
    if ( MAC_ADDRESS_LENGTH == j ) {
        if ( MAIN_PORT == port )
            hw_priv->hw.m_bMacOverrideAddr = TRUE;
    }
}  /* get_mac_addr */


/*
    netdev_init

    Description:
        This function initializes the network device.

    Parameters:
        struct net_device* dev
            Pointer to network device.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

int __devinit
netdev_init (
    struct net_device* dev )
{
    struct dev_priv* priv = netdev_priv( dev );

    spin_lock_init( &priv->lock );

#ifndef HAVE_NET_DEVICE_OPS
    /* Fill in the fields of the device structure with default values. */
    ether_setup( dev );

    dev->open               = netdev_open;
    dev->stop               = netdev_close;
    dev->hard_start_xmit    = DEV_TRANSMIT;
    dev->tx_timeout         = DEV_TRANSMIT_TIMEOUT;
    dev->get_stats          = dev_query_statistics;
    dev->set_mac_address    = device_set_mac_address;
    dev->set_multicast_list = dev_set_multicast_list;
    dev->do_ioctl           = netdev_ioctl;
    dev->change_mtu         = netdev_change_mtu;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 21))
    dev->ethtool_ops        = &netdev_ethtool_ops;
#endif

#ifdef CONFIG_KS884X_NAPI
    dev->poll               = DEV_POLL;
    dev->weight             = hw_priv->hw.m_RxDescInfo.cnAlloc;
#endif
#endif

    dev->watchdog_timeo     = HZ / 2;

#if TXCHECKSUM_DEFAULT
    dev->features |= NETIF_F_IP_CSUM;
#endif
#ifdef SCATTER_GATHER
    dev->features |= NETIF_F_SG;
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 23))
    SET_MODULE_OWNER( dev );
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 4, 21))
    priv->mii_if.phy_id_mask = 0x1;
    priv->mii_if.reg_num_mask = 0x7;
    priv->mii_if.dev = dev;
    priv->mii_if.mdio_read = mdio_read;
    priv->mii_if.mdio_write = mdio_write;
    priv->mii_if.phy_id = priv->id;
#endif
    priv->msg_enable = NET_MSG_ENABLE;

    return 0;
}  /* netdev_init */


#ifdef HAVE_NET_DEVICE_OPS
static const struct net_device_ops netdev_ops = {
	.ndo_init		= netdev_init,
	.ndo_open		= netdev_open,
	.ndo_stop		= netdev_close,
	.ndo_get_stats		= dev_query_statistics,
	.ndo_start_xmit		= DEV_TRANSMIT,
	.ndo_tx_timeout		= DEV_TRANSMIT_TIMEOUT,
	.ndo_change_mtu		= netdev_change_mtu,
	.ndo_set_mac_address	= device_set_mac_address,
	.ndo_do_ioctl		= netdev_ioctl,
	.ndo_set_multicast_list	= dev_set_multicast_list,

#ifdef CONFIG_NET_POLL_CONTROLLER
	.ndo_poll_controller	= netdev_netpoll,
#endif
};
#endif


typedef struct {
	struct dev_info    dev_info;
	struct net_device* NetDevice[ SWITCH_PORT_NUM ];
	int                dev_count;
} PLATFORM_INFO, *PPLATFORM_INFO;


#define KS884X_DMA_MASK  ( ~0x0UL )


static int __devinit pcidev_init (
    struct pci_dev*             pdev,
    const struct pci_device_id* id )
{
    struct net_device* dev;
    struct dev_priv*   priv;
    struct dev_info*   hw_priv;
    PHARDWARE          pHardware;
    PPLATFORM_INFO     pInfo;
    PPORT              pPort;
    unsigned long      reg_base;
    unsigned long      reg_len;
    int                count;
    int                index;
    int                mib_port_count;
    int                port;
    int                port_count;
    int                result;

#ifdef DBG
    pci_read_config_dword( pdev, 0x04, &result );
    printk( "%08x ", result );
    pci_read_config_dword( pdev, 0x34, &result );
    printk( "%08x ", result );
    reg_base = result;
    if ( reg_base ) {
        USHORT wData;

        printk( "PM: " );
        pci_read_config_word( pdev, reg_base, &wData );
        printk( "%04x ", wData );
        pci_read_config_word( pdev, reg_base + 2, &wData );
        printk( "%04x ", wData );
        pci_read_config_word( pdev, reg_base + 4, &wData );
        printk( "%04x ", wData );
    }
    printk( "\n" );
    for ( reg_len = 0; reg_len < 0x20; reg_len += 4 ) {
        pci_read_config_dword( pdev, reg_len, &result );
        printk( "%08x ", result );
    }
    printk( "\n" );
    for ( ; reg_len < 0x40; reg_len += 4 ) {
        pci_read_config_dword( pdev, reg_len, &result );
        printk( "%08x ", result );
    }
    printk( "\n" );
#endif
    if ( PCI_VENDOR_ID_KS884X == pdev->vendor ) {
        u32 dev_rev;

        pci_read_config_dword( pdev, PCI_REVISION_ID, &dev_rev );
        if ( PCI_DEVICE_ID_KS8842 == pdev->device ) {
#if 0
            if ( dev_rec < 0x10 ) {
                return -ENODEV;
            }
#endif
        }
        else if ( PCI_DEVICE_ID_KS8841 == pdev->device ) {
        }
        else
            return -ENODEV;
    }
    else
        return -ENODEV;

    result = pci_enable_device( pdev );
    if ( result ) {
        return( result );
    }

    result = -ENODEV;

#if 0
    if ( !pci_dma_supported( pdev, KS884X_DMA_MASK ) ) {
        return( result );
    }
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 0))
    if ( pci_set_dma_mask( pdev, DMA_32BIT_MASK )  ||
            pci_set_consistent_dma_mask( pdev, DMA_32BIT_MASK ) ) {
        return( result );
    }
#endif

    reg_base = pci_resource_start( pdev, 0 );
    reg_len = pci_resource_len( pdev, 0 );
    if ( ( pci_resource_flags( pdev, 0 ) & IORESOURCE_IO ) != 0 ) {
        return( result );
    }

    if ( !request_mem_region( reg_base, reg_len, DRV_NAME ) ) {
        return( result );
    }
    pci_set_master( pdev );

    result = -ENOMEM;

    pInfo = kmalloc( sizeof( PLATFORM_INFO ), GFP_KERNEL );
    if ( !pInfo ) {
        goto pcidev_init_dev_err;
    }
    memset( pInfo, 0, sizeof( PLATFORM_INFO ));

    hw_priv = &pInfo->dev_info;
    pHardware = &hw_priv->hw;

    hw_priv->pdev = pdev;

    phw = &hw_priv->hw;

#ifndef CONFIG_SH_7751_SOLUTION_ENGINE
    pHardware->m_pVirtualMemory = ioremap( reg_base, reg_len );

#else
    pHardware->m_pVirtualMemory = ( void* ) reg_base;
#endif
    if ( !pHardware->m_pVirtualMemory ) {
        goto pcidev_init_io_err;
    }
    pHardware->m_pPciCfg = pdev;

    printk(KERN_INFO "%s Mem = %p; IRQ = %d\n", version,
        pHardware->m_pVirtualMemory, pdev->irq );

    if ( !HardwareInitialize( pHardware ) ) {
        DBG_PRINT( "chip not detected"NEWLINE );
        result = -ENODEV;
        goto pcidev_init_alloc_err;
    }

#ifdef RCV_HUGE_FRAME
    hw_priv->m_nMTU = 1912 + 4;
#else
    hw_priv->m_nMTU = MAXIMUM_ETHERNET_PACKET_SIZE + 4;
#endif
    if ( AllocateMemory( hw_priv ) ) {
        FreeMemory( hw_priv );
        goto pcidev_init_alloc_err;
    }

    ks8842_fn = kmalloc( sizeof( struct hw_fn ), GFP_KERNEL );
    if ( ks8842_fn == NULL ) {
        goto pcidev_init_func_err;
    }

    HardwareSetupFunc( ks8842_fn );
    hw_priv->hw.m_hwfn = ks8842_fn;

    InitHardware( hw_priv );

    hw_priv->MonitorTimerInfo.pTimer = &hw_priv->timerMonitor;
    hw_priv->MonitorTimerInfo.nCount = 0;

    /* 500 ms timeout */
    hw_priv->MonitorTimerInfo.nTime = 500 * HZ / 1000;
    init_timer( &hw_priv->timerMonitor );
    hw_priv->timerMonitor.function = dev_monitor;
    hw_priv->timerMonitor.data = ( unsigned long ) hw_priv;

    /* tasklet is enabled. */
    tasklet_init( &hw_priv->rx_tasklet, RxProcessTask,
        ( unsigned long ) hw_priv );
    tasklet_init( &hw_priv->tx_tasklet, TxProcessTask,
        ( unsigned long ) hw_priv );

    /* tasklet_enable will decrement the atomic counter. */
    tasklet_disable( &hw_priv->rx_tasklet );
    tasklet_disable( &hw_priv->tx_tasklet );

    for ( index = 0; index < TOTAL_PORT_NUM; index++ )
        init_waitqueue_head( &hw_priv->Counter[ index ].wqhCounter );

    if ( macaddr[ 0 ] != ':' )
        get_mac_addr( hw_priv, macaddr, MAIN_PORT );

#ifdef TWO_NETWORK_INTERFACE
    memcpy( pHardware->m_bOtherAddress, pHardware->m_bOverrideAddress,
        MAC_ADDRESS_LENGTH );
    if ( mac1addr[ 0 ] != ':' )
        get_mac_addr( hw_priv, mac1addr, OTHER_PORT );
#endif

    AcquireHardware( hw_priv, FALSE, TRUE );
    HardwareReadAddress( pHardware );
    ReleaseHardware( hw_priv, FALSE );

#ifdef DEF_KS8841
    port_count = 1;
    mib_port_count = 1;
    pInfo->dev_count = 1;

#elif defined( TWO_NETWORK_INTERFACE )
    port_count = 1;
    mib_port_count = 1;
    pInfo->dev_count = SWITCH_PORT_NUM;

#else
    port_count = SWITCH_PORT_NUM;
    mib_port_count = TOTAL_PORT_NUM;
    pInfo->dev_count = 1;
#endif

    for ( index = 0; index < pInfo->dev_count; index++ ) {
        dev = alloc_etherdev( sizeof( struct dev_priv ));
        if ( !dev ) {
            goto pcidev_init_netdev_err;
        }
        pInfo->NetDevice[ index ] = dev;

        priv = netdev_priv( dev );
        priv->pDevInfo = hw_priv;
        priv->id = net_device_present++;

        for ( count = 0, port = index; count < port_count; count++, port++ ) {
            pHardware->m_PortInfo[ port ].pDevice = dev;
            pHardware->m_PortInfo[ port ].ulHardwareState =
                MediaStateDisconnected;

            pPort = &pHardware->m_PortInfo[ port ].port;
            pPort->PortCount = port_count;
            pPort->MibPortCount = mib_port_count;
            pPort->FirstPort = index;
            pPort->pLinked = &pHardware->m_PortInfo[ port ];
        }

        pPort = &pHardware->m_PortInfo[ index ].port;
        priv->port = pPort;

        dev->mem_start = ( unsigned long ) pHardware->m_pVirtualMemory;
        dev->mem_end = dev->mem_start + reg_len - 1;
        dev->irq = pdev->irq;
        if ( MAIN_PORT == index )
            memcpy( dev->dev_addr, hw_priv->hw.m_bOverrideAddress,
                MAC_ADDRESS_LENGTH );

#ifdef TWO_NETWORK_INTERFACE
        else {
            memcpy( dev->dev_addr, hw_priv->hw.m_bOtherAddress,
                MAC_ADDRESS_LENGTH );
            if ( !memcmp( hw_priv->hw.m_bOtherAddress,
                    hw_priv->hw.m_bOverrideAddress,
                    MAC_ADDRESS_LENGTH ) )
                dev->dev_addr[ 5 ] += pPort->FirstPort;
        }
#endif

#ifdef HAVE_NET_DEVICE_OPS
        dev->netdev_ops = &netdev_ops;
        SET_ETHTOOL_OPS( dev, &netdev_ethtool_ops );
#else
        dev->init = netdev_init;
#endif
        if ( register_netdev( dev ) ) {
            goto pcidev_init_reg_err;
        }
    }

    pci_set_drvdata( pdev, pInfo );

    hw_priv->id = device_present++;
    init_proc( hw_priv );

    return 0;

pcidev_init_reg_err:
    for ( index = 0; index < pInfo->dev_count; index++ ) {
        dev = pInfo->NetDevice[ index ];
        if ( dev ) {
            if ( dev->watchdog_timeo )
                unregister_netdev( dev );

            free_netdev( dev );
        }
    }

pcidev_init_netdev_err:
    kfree( pHardware->m_hwfn );

pcidev_init_func_err:
    FreeMemory( hw_priv );

pcidev_init_alloc_err:

#ifndef CONFIG_SH_7751_SOLUTION_ENGINE
    iounmap(( void* ) pHardware->m_pVirtualMemory );
#endif

pcidev_init_io_err:
    kfree( pInfo );

pcidev_init_dev_err:
    release_mem_region( reg_base, reg_len );

    return( result );
}  /* pcidev_init */


static void pcidev_exit (
    struct pci_dev* pdev )
{
    PPLATFORM_INFO     pInfo = pci_get_drvdata( pdev );
    struct net_device* dev;
    struct dev_info*   hw_priv;
    PHARDWARE          pHardware;
    int                i;
    char               proc_name[ 40 ];

    for ( i = 0; i < pInfo->dev_count; i++ ) {
        dev = pInfo->NetDevice[ i ];
        if ( dev ) {
            if ( dev->watchdog_timeo )
                unregister_netdev( dev );

            free_netdev( dev );
        }
    }

    hw_priv = &pInfo->dev_info;
    FreeMemory( hw_priv );

    pHardware = &hw_priv->hw;
    kfree( pHardware->m_hwfn );

#ifndef CONFIG_SH_7751_SOLUTION_ENGINE
    if ( pHardware->m_pVirtualMemory ) {
        iounmap(( void* ) pHardware->m_pVirtualMemory );
    }
#endif

    if ( hw_priv->proc_main ) {
        int port = HOST_PORT;

        for ( i = 0; i < PROC_LAST; i++ ) {
            if ( hw_priv->ProcPortInfo[ port ][ i ].info ) {
                remove_proc_entry( ProcInfo[ i ].proc_name,
                    hw_priv->proc_main );
            }
        }
        for ( port = 0; port < pHardware->m_bPortCount; port++ ) {
            if ( hw_priv->proc_port[ port ] ) {
                for ( i = 0; i < PROC_LAST; i++ ) {
                    if ( hw_priv->ProcPortInfo[ port ][ i ].info ) {
                        remove_proc_entry( ProcInfo[ i ].proc_name,
                            hw_priv->proc_port[ port ]);
                    }
                }
            }
            sprintf( proc_name, "%d", port );
            remove_proc_entry( proc_name, hw_priv->proc_main );
        }
    }
    sprintf( proc_name, "%s-%d", proc_dir_name, hw_priv->id );
    remove_proc_entry( proc_name, NULL );

    pci_set_drvdata( pdev, NULL );
    release_mem_region( pci_resource_start( pdev, 0 ),
        pci_resource_len( pdev, 0 ));
    kfree( pInfo );
}  /* pcidev_exit */


#ifdef CONFIG_PM
static int pcidev_resume (
    struct pci_dev* pdev )
{
    PPLATFORM_INFO     pInfo = pci_get_drvdata( pdev );
    struct net_device* dev;
    int                i;

#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 9))
    pci_set_power_state( pdev, 0 );
    pci_restore_state( pdev, pInfo->dev_info.pci_cfg_state );
    pci_enable_wake( pdev, 0, 0 );

#else
    pci_set_power_state( pdev, PCI_D0 );
    pci_restore_state( pdev );
    pci_enable_wake( pdev, PCI_D0, 0 );
#endif

    for ( i = 0; i < pInfo->dev_count; i++ ) {
        dev = pInfo->NetDevice[ i ];
        if ( netif_running( dev ) ) {
            netdev_open( dev );
            netif_device_attach( dev );
        }
    }
    return 0;
}  /* pcidev_resume */


static int pcidev_suspend (
    struct pci_dev* pdev,

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 14))
    pm_message_t state )

#else
    u32 state )
#endif
{
    PPLATFORM_INFO     pInfo = pci_get_drvdata( pdev );
    struct net_device* dev;
    int                i;

    for ( i = 0; i < pInfo->dev_count; i++ ) {
        dev = pInfo->NetDevice[ i ];
        if ( netif_running( dev ) ) {
            netif_device_detach( dev );
            netdev_close( dev );
        }
    }

#if (LINUX_VERSION_CODE <= KERNEL_VERSION(2, 6, 9))
    pci_save_state( pdev, pInfo->dev_info.pci_cfg_state );
    pci_enable_wake( pdev, 3, 1 );
    pci_set_power_state( pdev, 3 );

#else
    pci_save_state( pdev );
    pci_enable_wake( pdev, pci_choose_state( pdev, state ), 1 );
    pci_set_power_state( pdev, pci_choose_state( pdev, state ));
#endif
    return 0;
}  /* pcidev_suspend */
#endif


#ifdef DEF_KS8841
static char pcidev_name[] = "ksz8841p";
#else
static char pcidev_name[] = "ksz8842p";
#endif

static struct pci_device_id pcidev_table[] = {
#ifdef DEF_KS8841
    { PCI_VENDOR_ID_KS884X, PCI_DEVICE_ID_KS8841, 0x9020, 0x9720, 0, 0, 0 },
    { PCI_VENDOR_ID_KS884X, PCI_DEVICE_ID_KS8841, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
#else
    { PCI_VENDOR_ID_KS884X, PCI_DEVICE_ID_KS8842, 0x9020, 0x9720, 0, 0, 0 },
    { PCI_VENDOR_ID_KS884X, PCI_DEVICE_ID_KS8842, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0 },
#endif
    { 0 }
};

MODULE_DEVICE_TABLE(pci, pcidev_table);

static struct pci_driver PciDevice_driver = {
#ifdef CONFIG_PM
    suspend  : pcidev_suspend,
    resume   : pcidev_resume,
#endif
    name     : pcidev_name,
    id_table : pcidev_table,
    probe    : pcidev_init,
    remove   : pcidev_exit
};


/*
    init_module

    Description:
        This function starts the device driver.

    Parameters:
        None.

    Return (int):
        Zero if successful; otherwise an error code indicating failure.
*/

#ifdef MODULE
int __init init_module ( void )
#else
static int __init netdev_init_module ( void )
#endif
{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22))
    return( pci_module_init( &PciDevice_driver ));
#else
    return( pci_register_driver( &PciDevice_driver ));
#endif
}  /* init_module */


/*
    cleanup_module

    Description:
        This routine unloads the device driver.

    Parameters:
        None.

    Return (None):
*/

#ifdef MODULE
void __exit cleanup_module ( void )
#else
static void __exit netdev_cleanup_module ( void )
#endif
{
    pci_unregister_driver( &PciDevice_driver );
}  /* cleanup_module */


#ifndef MODULE
module_init(netdev_init_module);
module_exit(netdev_cleanup_module);
#endif
