<?php
/*
	SMG WebGUI

	Software distributed under the License is distributed on an "AS IS" basis,
	WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
	for the specific language governing rights and limitations under the
	License.
	
	Licence information can be found within the file LICENCE.php
	
	The Original Code is SMG WebGUI

	The Initial Developer of the Original Code is
	Jonathan Maurice <jmaurice@sangoma.com>

	Contributor(s):
*/


///////////////////////////////////////////////////////////////////////////////
//
// TestSetup class
//
///////////////////////////////////////////////////////////////////////////////

class TestSetup
{

  const WANROUTER_CMD = 'wanrouter';
  const IFCONFIG_CMD = 'ifconfig -a';
  const DIGITAL_CARD_KEY = 'digital';
  const MEDIA_CARD_KEY = 'media';

  private $installedCards;
  private $digCardSpecs;
  private $mediaCardSpecs;
  
  public function __construct()
  {
    $this->scanCards = false;
    $this->installedCards = null;
    $this->digCardSpecs = new DigitalCardSpecs();
    $this->mediaCardSpecs = new MediaCardSpecs();

    //Try to find all installed cards.
    $this->findDigitalCards();
    $this->findMediaCards();
  }

  public function hasCards()
  {
    if ( $this->hasDigitalCards() || $this->hasMediaCards() ){
      return true;
    }
    else {
      return false;
    }
  }

  public function hasDigitalCards()
  {
    if ($this->installedCards[self::DIGITAL_CARD_KEY] === null){
      return false;
    }
    else {
      return true;
    }
  }

  public function hasMediaCards()
  {
    if ($this->installedCards[self::MEDIA_CARD_KEY] === null){
      return false;
    }
    else {
      return true;
    }
  }

  public function getCards()
  {
    return $this->installedCards;
  }

  public function getDigitalCards()
  {
    return $this->installedCards[self::DIGITAL_CARD_KEY];
  }

  public function getNumDigitalCards()
  {
    $numDigitalCard = 0;
    foreach ($this->installedCards[self::DIGITAL_CARD_KEY] as $cardName => $numCard) {

      $numDigitalCard += $numCard;
    }

    return $numDigitalCard;
  }

  public function getNumSpans()
  {

    $numSpan = 0;
    foreach ($this->installedCards[self::DIGITAL_CARD_KEY] as $cardName => $numCard) {

      $numSpan += $this->digCardSpecs->getNumSpan($cardName) * $numCard;
    }

    return $numSpan;
  }

  public function getMediaCards()
  {
    return $this->installedCards[self::MEDIA_CARD_KEY];
  }

  public function getNumMediaCards()
  {
    return count($this->installedCards[self::MEDIA_CARD_KEY]);
  }

  //Find digital cards, this set $this->installedCards[self::DIGITAL_CARD_KEY] 
  //to an array of digital cards with the key being the card name and the value 
  //the number of that card. If there's no card found this is set to null.
  private function findDigitalCards()
  {
    $cmd = self::WANROUTER_CMD . " hwprobe";
    $output = array();
    exec($cmd, $output);

    //Retrieve Card Cnt: line
    $cardCnt = array_values(preg_grep("/Card Count:.*/", $output));
    //There should be only one Card Cnt: line
    if ( count($cardCnt) != 1){
      $this->installedCards[self::DIGITAL_CARD_KEY] = null;
      return false;
    }
    $cardCnt = end($cardCnt);

    $matchArray = array();
    if( preg_match("/Card Count:\s+?(.*)/", $cardCnt, $matchArray) == 0 ) {
      $this->installedCards[self::DIGITAL_CARD_KEY] = null;
      return false;
    }
    $cardStr = $matchArray[1];
    //Split card string on spaces. This mean that $cardArray array element will 
    //correspond to one card count like this: AXXX=Number
    $cardArray = preg_split("/\s+/", $cardStr);

    //Iterate over each cards and split on the '=' to seperate the card name 
    //from the number of card.
    foreach ($cardArray as $card) {
      $cardNameAndNum = preg_split("/=/", $card);

      if ( count($cardNameAndNum) != 2){
        //This shouldn't happen, unless card output is wrong...
        continue;
      }
      $name = $cardNameAndNum[0];
      $number = $cardNameAndNum[1];
      
      //Make sure this is a valid digital card
      if ($this->digCardSpecs->isDigitalCard($name)) {
        $this->installedCards[self::DIGITAL_CARD_KEY][$name] = $number;
      }
	  
	  
    }

    //Check that we found a digital cards
    if ( count($this->installedCards[self::DIGITAL_CARD_KEY]) ){
      return true;
    }
    else {
      $this->installedCards[self::DIGITAL_CARD_KEY] = null;
      return false;
    }
  }

  //Find media cards, this set $this->installedCards[self::MEDIA_CARD_KEY] 
  //to an array of media cards with the key being the ethernet interface name
  //and the value the MAC address of that card.
  //If there's no card found this is set to null.
  private function findMediaCards()
  {
    $cmd = self::IFCONFIG_CMD;
    $output = array();
    exec($cmd, $output);

    //Retrieve HWaddr line
    $ethIfArray = array_values(preg_grep("/.*?HWaddr.*/", $output));

    //For each ethX line check if the MAC address is a Sangoma MAC
    foreach ($ethIfArray as $ethLine) {
      $ethName = "";
      $ethMac = "";
      $matchArray = array();

      //Retrieve the ethernet interface name and it's MAC address
      if( preg_match("/^(eth\d+?|__tmp\d+?)\s.*?HWaddr\s([0-9A-Fa-f:]+)/", $ethLine, $matchArray) == 0 ) {
        continue;
      }
      $ethName = $matchArray[1];
      $ethMac = $matchArray[2];

      //Check if this is a Sangoma MAC
      if ($this->mediaCardSpecs->isMediaCard($ethMac) ){
        $this->installedCards[self::MEDIA_CARD_KEY][$ethName] = $ethMac;
      }
    }

    //Check that we found a media card
    if ( count($this->installedCards[self::MEDIA_CARD_KEY]) ){
      return true;
    }
    else {
      $this->installedCards[self::MEDIA_CARD_KEY] = null;
      return false;
    }
  }

}

///////////////////////////////////////////////////////////////////////////////
//
// DigitalCardSpecs class
//
///////////////////////////////////////////////////////////////////////////////

final class DigitalCardSpecs
{
  const A101_NUM_PORTS = 1;
  const A102_NUM_PORTS = 2;
  const A104_NUM_PORTS = 4;
  const A108_NUM_PORTS = 8;
  const A116_NUM_PORTS = 16;

  private $cardSpecs;

  public function __construct()
  {
    $this->cardSpecs = array(
                              "A101" => self::A101_NUM_PORTS,
                              "A102" => self::A102_NUM_PORTS,
                              "A104" => self::A104_NUM_PORTS,
                              "A108" => self::A108_NUM_PORTS,
                              "A116" => self::A116_NUM_PORTS
                            );

  }

  public function getNumSpan($in_cardName)
  {
    if( array_key_exists($in_cardName, $this->cardSpecs) ){
      return $this->cardSpecs[$in_cardName];
    }
    else {
      return null;
    }
  }

  public function getCardTypeStr($in_cardName)
  {
    if( array_key_exists($in_cardName, $this->cardSpecs) ){
      $numPort = $this->cardSpecs[$in_cardName];
      return $numPort . " Port T1/E1";
    }
    else {
      return null;
    }
  }

  public function isDigitalCard($in_cardName)
  {
    return array_key_exists($in_cardName, $this->cardSpecs);
  }

}

///////////////////////////////////////////////////////////////////////////////
//
// MediaCardSpecs class
//
///////////////////////////////////////////////////////////////////////////////


final class MediaCardSpecs
{
  const MEDIA_CARD_MAC_BEGIN = '02:19:23';

  public function isMediaCard($in_cardMac)
  {
    $macBegin = self::MEDIA_CARD_MAC_BEGIN;

    if( preg_match("/^$macBegin/", $in_cardMac) ){
      return true;
    }
    else {
      return false;
    }
  }
}

class firmware_check
{
	
	function vocallo_firmware($ethname, $data)
	{
		$count++;
		unset($output);
		$command = "/sbin/ifconfig " . $ethname . " | grep 'inet addr:' | cut -d: -f2 | awk '{ print $1}'";
		exec ($command,$output);
		//echo "output = " . $output[0];
		//Check for the IP Address to See if firmware version can be found
		if ($output[0] == "")
		{
			$command = "/sbin/ifconfig " . $ethname . " 10.$count.1.1 netmask 255.255.255.0";
			//echo $command;
			exec($command);
			$command = "/sbin/ifconfig " . $ethname . " up";
			exec($command);
			$takedown = "yes";
		}
		
		
		$cmd = "/usr/sbin/sngtc_tool -dev " . $ethname . " -list_modules | grep Ver | cut -d\"=\" -f3 | cut -d\" \" -f1";
		exec ($cmd,$firmware);
		
		//Place firmware value in database		
		
		if ($takedown == "yes")
		{
			$command = "/sbin/ifconfig " . $ethname . " down";
			exec($command);
		}
		
		return $firmware;
	}	
	
}

?>
