<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * Module class
*/
require_once ('application/helpers/safe_helper.php');
require_once ('safe_object_class.php');
require_once ('system/libraries/Zip.php');
require_once ('application/libraries/safe_upload_class.php');

class Safe_module_class extends Safe_object_class
{
    private $_node = null;
    private $_config = null;
    private $_modules = array();
    protected $_audit_parameters = array();
    public function __construct($software, $name, $type = "module")
    {
        parent::__construct($software->object_name() , $name);
        if (is_subclass_of($software, "Safe_node")) {
            $this->_node = $software;
        } else {
            $this->_node = $software->node();
        }
    }
    public function set_node($node)
    {
        $this->_node = $node;
    }
    public function node()
    {
        return $this->_node;
    }
    public function config()
    {
        return $this->_config;
    }

    public function documentation()
    {
        $doc_dir = $this->node()->configuration_manager()->directory('doc', null, false);
        $doc_file = $doc_dir.'/'.$this->name().'.txt';
        $doc = array();
        if($this->node()->os()->read_file($doc_file, $doc, true)){
            return $doc;
        }
        return parent::documentation();
    }
    public function set_config($config, $controller_url=null)
    {
        unset($this->_config);
        if($config) {
            // Get data to expose parameters
            $data = $config->get_data(false);
            $cfg_def =
                array(
                    'name' => 'Configuration',
                    'singleton' => true,
                    'methods' => array(
                        'retrieve' => array(
                            'name' => 'Retrieve',
                            'request' => 'GET',
                        ),
                        'update' => array(
                            'name' => 'Update',
                            'request' => 'POST',
                        )
                    ),
                    'class' => $data,
                );
            if($controller_url){
                $cfg_def['controller_url'] = $controller_url;
            }
            // expose config as aggregate object
            $this->register_aggregate_object('configuration', $cfg_def);
        }
        $this->_config = $config;
        return true;
    }
    /**
     * @brief
     *
     * @return
     */
    public function api_retrieve_configuration($obj_name=null, $params=null, &$output=null)
    {
        return $this->_config;
    }
    public function api_update_configuration($obj_name=null, $params=null, &$output=null, $validation_check = true)
    {
        if($this->_config) {
            if($validation_check){
                $this->_config->prepare_edit();
                $params = $this->_config->data_filter($params);
                if(!$this->_config->validate($params, $output)) {
                    return false;
                }else{
                    return $this->_config->save();
                }
            }else{
                return $this->_config->serialize();
            }
        }
        return false;
    }
    public function find_module_by_name($name)
    {
        if($name == $this->name()) {
            return $this;
        }
        $mod = null;
        foreach($this->modules() as $k => $v) {
            if($name == $k) {
                return $v;
            }else{
                $mod = $v->find_module_by_name($name);
                if($mod)
                    return $mod;
            }
        }
        return null;
    }
    /**
     * @brief Overridable generate config for reload
     *
     * @param[in out] $config_manager
     * @param[in out] $obj_type
     *
     * @return
     */
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->reload_generate_config($config_manager, $obj_type)) return false;
        }
        return true;
    }
    /**
     * @brief Generate configuration
     *
     * @param[in out] $config_manager
     *
     * @return
     */
    public function generate_config(&$config_manager)
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->generate_config($config_manager)) return false;
        }
        return true;
    }
    /**
     * @brief Invoked after successfull generate and write config in reload
     * mode
     *
     * @return
     */
    public function post_write_config($obj_type=null)
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_write_config($obj_type)) return false;
        }
        return true;
    }
    public function reload_clear_modified($obj_type=null)
    {
    }
    public function clear_configuration_modified($include_childs = true)
    {
        // Clear childs ?
        if ($include_childs) Safe_object_serializer_class::get_serializer()->clear_modified($this->object_name() , true);
        //clear check_list,modified_object_list cache
        $this->clear_list_cache();
        // Clear ourself
        if ($this->config()) $this->config()->clear_modified();
    }
    public function is_configuration_modified($include_childs = true)
    {
        if ($this->config() && $this->config()->is_modified()) return true;
        return parent::is_configuration_modified($include_childs);
    }
    /**
     * @brief Module support reload ?
     *
     * @return boolean
     */
    public function support_reload(&$need_reload = NULL)
    {
        if (isset($need_reload)) {
            $need_reload = $this->is_configuration_modified(false);
        }
        return false;
    }
    /**
     * @brief Reload module
     *         Write config + post_write_config
     *
     * @return
     */
    public function reload()
    {
        // support reload ?
        if ($this->support_reload() && $this->is_configuration_modified()) {
            $cfg_mgr = $this->node()->configuration_manager();
            // Take care of generate configuration
            if ($this->generate_config($cfg_mgr) && !$cfg_mgr->errors()
            && $cfg_mgr->write_config($this->node())
            // invoke post_write_config
            && $this->post_write_config()) {
                // clear modified flags
                $this->clear_configuration_modified(true);
                return true;
            }
            return false;
        }
        return true;
    }
    public function post_reload()
    {
        return true;
    }
    public function register_module($module)
    {
        $this->_modules[$module->name() ] = $module;
    }
    public function modules()
    {
        return $this->_modules;
    }
    public function module($name)
    {
        if (isset($this->_modules[$name])) return $this->_modules[$name];
        else return null;
    }
    public function configure()
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->configure()) return false;
        }
        return parent::configure();
    }
    public function post_configure()
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_configure()){
                return false;
            }
        }
        return true;
    }
    /**
     * @brief Return array of variables that need to be defined globally for
     *         application (key=>value)
     *
     * @return array
     */
    public function get_global_variables()
    {
        $vars = array();
        foreach ($this->_modules as $module) {
            array_merge($vars, $module->get_global_variables());
        }
        return $vars;
    }
    /**
     * @brief Return module category
     *         Should be implemented by module instance
     *
     * @return string
     */
    private $_category = 'Other';
    public function set_category($category)
    {
        $this->_category = $category;
    }
    public function category()
    {
        return $this->_category;
    }
    /**
     * @brief Return module data settings
     *         If module can expose data(s) it should return an array formatted as follow:
     *         $array= <data_name> (
     *         ['description'] => Data description
     *         ['filesystem']  => true/false - Are data stored in filesystem
     *         ['singleton']  => true/false - Only one data present at a time
     *         ['pattern']        => Defined the data path pattern for filesystem data (to use with glob)
     *         ['global_capabilities'] See below but globaly data MUST exist (ie. download all,
     *         delete all...)
     *         ['global_capabilities_no_data'] See below but globaly even if no
     *         data (ie. upload, ...)
     *         ['capabilities'] =array() possible entries are:
     *         'download'    => Data can be downloaded
     *         'delete'      => Data can be deleted (cleanup)
     *         'view'        => Data can be viewed
     *         'rotate'      => Data can be viewed
     *
     *         If data are not in filesystem, for every entries in capabilities the module MUST implement the corresponding
     *         data_<capability>($data_name) function
     * @return array as describe above
     */
    public function get_data_settings()
    {
        return array();
    }
    /**
     * @brief Return module action settings
     *         If module can expose action(s) it should return an array formatted as follow:
     *         $array= <action_name> (
     *         ['description'] => action description)
     *
     * @param[in] $action - (optional) action name to retrieve
     * @return array as describe above
     */
    public function get_action_settings($action=null)
    {
        return array();
    }
    /**
     *
     * @param $action action name
     * @param $type for ui or for rest
     * @return boolean if the url not set, or return the url
     */
    public function get_action_url($action, $type='ui')
    {
        return false;
    }
    public function get_aggregate_object_method_params($type, $method)
    {
        $param = parent::get_aggregate_object_method_params($type, $method);
        if( $param === false){
            $definition_all = $this->get_data_settings();
            $definition = $definition_all[$type];
            if($definition['methods'][$method]){
                $method_name = 'api_'.$method.'_'.$type.'_parameters';
                if(method_exists($this, $method_name)) {
                    $param =  call_user_func_array( array( $this, $method_name), array());
                    return $param;
                }
            }
            return false;
        }else{
            return $param;
        }

    }
    /**
     * @brief Return count of specific data name
     *
     * @param[in out] $data_name
     *
     * @return
     */
    public function get_data_count($data_name)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            return count($this->node()->os()->search_files(
                $data_cfg[$data_name]['pattern'],$data_cfg[$data_name]['recursive']));
        }
        return 0;
    }
    /**
     * @brief Return count of specific data name
     *
     * @param[in out] $data_name
     *
     * @return array( 'key' => 'file path' ) for each data entries
     */
    private $_data_files_cache = array();
    public function get_data_files($data_name)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            if(!$this->_data_files_cache[$data_name]){
                $data_files = array();
                $files = $this->node()->os()->search_files($data_cfg[$data_name]['pattern'], $data_cfg[$data_name]['recursive']);
                foreach ($files as $file) {
                    $data_files[$this->get_data_file_key($data_name, $file) ] = $file;
                }
                $this->_data_files_cache[$data_name] = $data_files;
            }
            return $this->_data_files_cache[$data_name];
        }
        throw new Exception('Module ' . $this->description() . ' get_data_files error for ' . $data_name);
    }
    public function data_summary($data_name){
        $data_files = $this->get_data_files($data_name);
        $data_summary = array();
        foreach ($data_files as $key => $file) {
            $data_summary[] = array('Name' => $key);
        }
        return $data_summary;
    }
    /**
     * @brief Retrieve data file key
     *           Default impletmentation if returning file name (basename)
     *
     * @param[in out] $data_name
     * @param[in out] $file
     *
     * @return
     */
    public function get_data_file_key($data_name, $file)
    {
        return basename($file);
    }
    /**
     * @brief Retrieve data file content
     *
     * @param[in out] $data_name
     * @param[in out] $key
     *
     * @return
     */
    public function get_data_file_content_by_key($data_name, $key, $line_count = NULL, $max_size = NULL)
    {
        $data_cfg = $this->get_data_settings();
        $content = "";
        $tmp_file_name = pathinfo($key, PATHINFO_FILENAME);
        if($data_cfg[$data_name]['file_ext']){
            $tmp_ext = $data_cfg[$data_name]['file_ext'];
        }else{
            //handle the log file name like rtcpmon.log.2
            if(strpos($tmp_file_name , '.') === false){
                $tmp_ext = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_EXTENSION);
            }else{
                $tmp_ext = pathinfo($key, PATHINFO_EXTENSION);
            }
        }
        if($data_cfg[$data_name]['file_dir']){
            $tmp_file_path = $data_cfg[$data_name]['file_dir'];
        }else{
            $tmp_file_path = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_DIRNAME );
        }
        $file = $this->node()->os()->find_file($tmp_file_path, $tmp_file_name.'.'.$tmp_ext, $data_cfg[$data_name]['recursive']);
        if ($file) {
            // Get data file
            $os = $this->node()->os();
            $os->chown_file($file, 'webconfig');
            // Read content
            if (isset($line_count)) {
                $output = array();
                $rc = $os->execute('sudo tail -n ' . $line_count . ' ' . $file, $output);
                $content = implode('\n', $output);
            } elseif (isset($max_size)) {
                $content = file_get_contents($file, false, NULL, -1, $max_size);
            } else {
                $content = file_get_contents($file);
            }
        }
        return $content;
    }
    /**
     * @brief Validate a data key exists for a specific data set
     *
     * @param[in out] $data_name
     * @param[in out] $key
     *
     * @return
     */
    public function data_key_exists($data_name, $key)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            $tmp_file_name = pathinfo($key, PATHINFO_FILENAME);
            if($data_cfg[$data_name]['file_ext']){
                $tmp_ext = $data_cfg[$data_name]['file_ext'];
            }else{
                $tmp_ext = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_EXTENSION);
            }
            if($data_cfg[$data_name]['file_dir']){
                $tmp_file_path = $data_cfg[$data_name]['file_dir'];
            }else{
                $tmp_file_path = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_DIRNAME );
            }
            return  $this->node()->os()->check_file_exists($tmp_file_path .'/' . $tmp_file_name.'.'.$tmp_ext, $data_cfg[$data_name]['recursive']);
        }
        throw new Exception('Module ' . $this->description() . ' data_key_exists error for ' . $data_name);
    }
    /**
     * @brief Delete all data
     *
     * @param[in out] $data_name
     *
     * @return
     */
    public function delete_data($data_name, $data_key = NULL)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            // Locate single key ?
            if (isset($data_key)) {
                $tmp_file_name = pathinfo($data_key, PATHINFO_FILENAME);
                if($data_cfg[$data_name]['file_ext']){
                    $tmp_ext =
                        '.{'.str_replace('|',',',$data_cfg[$data_name]['file_ext']).'}';
                }else{
                    $tmp_ext = '.'.pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_EXTENSION);
                }
                if($data_cfg[$data_name]['file_dir']){
                    $tmp_file_path = $data_cfg[$data_name]['file_dir'];
                }else{
                    $tmp_file_path = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_DIRNAME );
                }
                if($data_name == 'archive'){
                    $tmp_ext = "*";//remove the .info file
                    $files = $this->node()->os()->search_files($tmp_file_path . '/' . $tmp_file_name.$tmp_ext, $data_cfg[$data_name]['recursive']);
                }else{
                    $file = $this->node()->os()->find_file($tmp_file_path, $tmp_file_name.$tmp_ext, $data_cfg[$data_name]['recursive']);
                    if($file){
                        $files[] = $file;
                    }
                }
            } else {
                $data_files = array();
                $files = $this->node()->os()->search_files($data_cfg[$data_name]['pattern'], $data_cfg[$data_name]['recursive']);
            }
            // loop to delete selected file(s)
            if($files){
                foreach ($files as $file) {
                    $os = $this->node()->os();
                    $os->chown_file($file, 'webconfig');
                    $os->delete_file($file);
                } 
            }
            return true;
        }
        throw new Exception('Module ' . $this->description() . ' delete_data error for ' . $data_name);
    }
    /**
     * @brief download all data
     *
     * @param[in out] $data_name
     *
     * @return
     */
    public function download_data($data_name, $data_key = NULL)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            $data_files = $this->get_data_files($data_name);
            if (isset($data_key)) {
                if (isset($data_files[$data_key])) {
                    if ($data_name == 'license') {
                        $data_files['License.txt.sig'] = $data_files['License.txt.sig'];
                        $data_files['License.txt'] = $data_files['License.txt'];
                    }
                    else {
                        $data_files = array($data_key => $data_files[$data_key]);
                    }
                }else{
                    $data_files = array();
                }
            }
            $os = $this->node()->os();
            foreach ($data_files as $key => $file) {
                $os->chown_file($file, 'webconfig');
            }
            if (count($data_files) > 1) {
                // All of them, zip and dowload
                ini_set('memory_limit', '300M');
                // Ready to ZIIIIP....
                $zip_obj =  new CI_Zip();
                $name = php_uname('n') . '-' . $data_name . '-' . date('Y-m-d-H-i-s') . '.zip';
                foreach ($data_files as $key => $file) {
                    $rc = $zip_obj->read_file($file);
                }
                // Proceed to download
                $rc = $zip_obj->download($name);
            } else if (count($data_files)) {
                // Download the file
                if($data_key){
                    safe_download($data_files[$data_key], NULL, $data_key);
                }else{
                    $data_key = key($data_files);
                    safe_download($data_files[$data_key], NULL, $data_key);
                }

            }
        }
        header('HTTP/1.0 404 Not Found');
        exit();
    }
    
    public function upload_file($data_name, $data_cfg, &$output , $from = 'POST')
    {
        //update load a file and saved to a file path
        //save some file info into resturn parameter
        $_POST['Upload'] = 'Upload';
        $upload_lib = new Safe_upload_class();
        if ($upload_lib->is_uploading()) {
            if($data_cfg['singleton']){
                //only allow one file
                $this->delete_data($data_name);
            }
            ini_set('memory_limit', '500M');
            // path and ext are based data_cfg pattern
            $ext = pathinfo($data_cfg['pattern']);
            $upload_lib->set_upload_path($ext['dirname']);
            if($data_cfg['file_ext']){
                $upload_lib->set_allowed_types($data_cfg['file_ext']);
            }else{
                $upload_lib->set_allowed_types($ext['extension']);
            }
            $upload_lib->set_max_size(500 * 1024);
            $rc = $upload_lib->upload();
            $up_data = $upload_lib->get_data();
            if ($rc) {
                if($from == 'PUT'){
                    @unlink($tmpfname);
                }
                $output = $up_data;
                $output['size'] = (int)ceil(1024 * $up_data['file_size']);
                return true;
            }else{
                $output['message'] = 'Failed to upload ' . $this->description() . ' ' . $data_cfg['description'] . '.';
                $output['reason'] = $upload_lib->get_errors();
            }
        }else{
            $output['message'] = 'Failed to upload ' . $this->description() . ' ' . $data_cfg['description'];
        }
        return false;
    }
    /**
     * @brief upload data
     *
     * @param[in out] $data_name
     *
     * @return
     */
    public function upload_data($data_name, $data_key = NULL, &$output , $from = 'POST')
    {
        $data_cfg_setting = $this->get_data_settings();
        if (isset($data_cfg_setting[$data_name]) && true == $data_cfg_setting[$data_name]['filesystem'] && isset($data_cfg_setting[$data_name]['pattern'])) {
            $data_cfg = $data_cfg_setting[$data_name];
            if($from == 'PUT'){
                //convert put upload file to post upload file
                if (!isset($_SERVER['HTTP_UPLOAD_FILE_NAME'])) {
                    $output['notif_error'][] = 'Failed to upload ' . $this->description() . ' ' . $data_cfg['description'];
                    return false;
                }else{
                    $putdata = fopen("php://input", "r");
                    $tmpfname = tempnam(sys_get_temp_dir(), "put_upload_");
                    $fp = fopen($tmpfname, "w");
                    while ($data = fread($putdata, 1024)){
                        fwrite($fp, $data);
                    }
                    fclose($fp);
                    fclose($putdata);
                    $_FILES['archive']['name'] = $_SERVER['HTTP_UPLOAD_FILE_NAME'];
                    $_FILES['archive']['type'] = 'application/octet-stream';
                    $_FILES['archive']['tmp_name'] = $tmpfname;
                    $_FILES['archive']['error'] = 0;
                    $_FILES['archive']['size'] = filesize($tmpfname);
                    $_FILES['archive']['from'] = 'PUT';
                }
            }
            if($this->upload_file($data_name, $data_cfg, $output, $from)){
                if($data_cfg['singleton']){
                    //after upload ini
                    $output_msg = array();
                    if($this->initialize_data($data_name, $output['file_name'], $output_msg, $data_key)){
                        return true;
                    }else{
                        $output['message'] = 'Fail to initialize '.$data_cfg['description'].'! ';
                        if($output_msg){
                            $output['reason'] = $output_msg;
                        }
                    }
                }else{
                    return true;
                }
            }
        }
        return false;
    }
    public function initialize_data($data_name, $data_key, &$output=null)
    {
        return true;
    }
    public function describe_data($data_name, $data_key, &$output=null)
    {
        $output[] = $data_key;
        return false;
    }
    public function install_data($data_name, $data_key, &$output=null)
    {
        return false;
    }
    /**
     * @brief Check if update package can be install
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function can_install_data($info, &$reason)
    {
        // Default is true
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->can_install_data($info, $reason)) return false;
        }
        return true;
    }
    /**
     * @brief
     *
     * @param[in out] $data_name
     *
     * @return
     */
    public function rotate_data($data_name)
    {
        // For now return false
        // May use logrotate feature as default
        return false;
    }
    /**
     * @brief Check if backup can be performed
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function can_backup($type, &$reason)
    {
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->can_backup($type, $reason)) return false;
        }
        return true;
    }
    /**
     * @brief Pre backup hook
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function pre_backup($type, &$reason)
    {
        safe_log(SAFE_LOG_INFO, 'pre_backup - '.$this->name());
        $script = $this->get_hook_script('pre', 'backup');
        if($script) {
            safe_log(SAFE_LOG_INFO, 'pre_backup - Invoking ' . $script . ' ' . $type);
            $this->node()->execute('nohup '.$script . ' ' . $type, $retCode, true, true, $rc);
            //db forcing re-opening db, table struct may be altered by this hooks
            Safe_object_serializer_class::get_serializer(true);
        }
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->pre_backup($type, $reason)) return false;
        }
        return true;
    }
    /**
     * @brief Post backup hook
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function post_backup($type, &$reason)
    {
        safe_log(SAFE_LOG_INFO, 'post_backup - '.$this->name());
        $script = $this->get_hook_script('post', 'backup');
        if($script) {
            safe_log(SAFE_LOG_INFO, 'post_backup - Invoking ' . $script);
            $this->node()->execute('nohup '.$script.' '.$type, $retCode, true, true, $rc);
            //db forcing re-opening db, table struct may be altered by this hooks
            Safe_object_serializer_class::get_serializer(true);
        }
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_backup($type, $reason)) return false;
        }
        return true;
    }
    /**
     * @brief Check if archive restore can be done
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function can_restore($info, &$reason)
    {
        // Default is true
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->can_restore($info, $reason)) return false;
        }
        return true;
    }
    /**
     * @brief Pre restore hook
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function pre_restore($info, &$reason)
    {
        safe_log(SAFE_LOG_INFO, 'pre_restore - '.$this->name());
        $script = $this->get_hook_script('pre', 'restore');
        if($script) {
            safe_log(SAFE_LOG_INFO, 'pre_restore - Invoking ' . $script . ' ' . $archive);
            $skip = $info['skip-rest-api'] && $this->name()=='rest' ? ' --skip-rest-api ' : '';
            if(!$this->node()->execute('nohup '.$script . ' ' . $info['type'] . $skip, $retCode, true, true, $rc)){
                $reason = $retCode;
                return false;
            }
            //db forcing re-opening db, table struct may be altered by this hooks
            Safe_object_serializer_class::get_serializer(true);
        }
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->pre_restore($info, $reason)) return false;
        }
        return true;
    }
    /**
     * @brief Post restore hook
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function post_restore($info, &$reason)
    {
        safe_log(SAFE_LOG_INFO, 'post_restore - '.$this->name());
        // Check if product script exists
        $script = $this->get_hook_script('post', 'restore');
        if($script) {
            safe_log(SAFE_LOG_INFO, 'post_restore - Invoking ' . $script);

            // skip network configuration
            $skip = $info['skip_network'] ? ' --mode=update ' : '';
            $skip.= $info['skip-rest-api'] && $this->name()=='rest' ? ' --skip-rest-api ' : '';
            $this->node()->execute('nohup '.$script. ' ' . $info['type'] . $skip, $retCode, true, true, $rc);

            //db forcing re-opening db, table struct may be altered by this hooks
            Safe_object_serializer_class::get_serializer(true);
        }
        // Make sure config is synched from DB
        if($this->config()){
            $this->config()->synch();
        }
        $this->clear_cache();
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_restore($info, $reason)) return false;
        }
        return true;
    }
    protected function get_hook_script($step , $type)
    {
        $base = $this->node()->configuration_manager()->directory('base');
        $script = $base.'/hooks/'.$this->name().'.' . $type .'.'.$step;
        if(file_exists($script)) {
            return $script;
        }else{
            return '';
        }
    }
    /**
     * @brief Retrieve list of modified obj/conf
     * Optionnaly return all or only reloadable
     *
     * @param[in out] $obj_list
     * @param[in out] $reloadable_only
     *
     * @return
     */
    public function modified_object_list(&$obj_list)
    {
        if(true == $this->is_configuration_modified(true)){
            $mod_desc = $this->description();
            // Do we support reload and need relaod?
            $need_reload = false;
            $support_reload = false;
            $support_reload = $this->support_reload($need_reload);
            if($need_reload){
                $key = ($support_reload)?"reload":"restart";
                $obj_list[$key][] =
                    array(
                        'module' => $this->name(),
                        'status' => 'M',
                        'description' => $mod_desc,
                    );
            }else if($this->config() && $this->config()->status() != Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE){
                // Check for config modified
                //$obj_list[$this->name()]['configuration'] = $this->config();
                $key = ($support_reload)?"reload":"restart";
                $obj_list[$key][] =
                    array(
                        'module' => $this->name(),
                        'obj_type' => 'configuration',
                        'status' => $this->config()->status(),
                        'description' => $mod_desc.'/Configuration',
                    );
            }

            // Check aggregate objects
            foreach($this->has_aggregate_objects(true) as $k => $v ) {
                // Check if this object is dynamic or if any child (which can be
                // dyanmic)
                if($v['configurable'] === false){
                    continue;
                }else{
                    $obj_desc = $v['name'];
                    // Get those objects and check for modified ones
                    $objs = $this->get_aggregate_objects($k, true);
                    // Check if array returned, if not array most likely a
                    // singleton, convert to array so foreach is happy
                    // TODO: revisit this modif later
                    if($objs and !is_array($objs)){
                        $objs=array($k => $objs);
                    }
                    foreach($objs as $obj_name => $obj) {
                        if($obj->status() != Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE) {
                            //$obj_list[$this->name()][$k][$obj_name] = $obj;
                            $key = ($v['dynamic'])?"reload":"restart";
                            $obj_list[$key][] =
                                array(
                                    'module' => $this->name(),
                                    'obj_type' => $k,
                                    'obj_name' => $obj_name,
                                    'status' => $obj->status(),
                                    'description' => $mod_desc.'/'.$obj_desc.'/'.$obj_name,
                                );
                        }
                        if($v['has_child'] && $obj->is_configuration_modified(true)) {
                            $child_objs_def = $obj->has_aggregate_objects(true);
                            foreach($child_objs_def as $child_obj_type => $child_obj_def) {
                                if($child_obj_def['configurable'] === false) continue;
                                $child_desc = $child_obj_def['name'];
                                $child_objs = $obj->get_aggregate_objects($child_obj_type, true);
                                foreach($child_objs as $child_obj_name => $child_obj) {
                                    if($child_obj->is_modified()) {
                                        //$obj_list[$this->name()][$k.'/'.$child_obj_type][$obj_name.'/'.$child_obj_name] = $child_obj;
                                        $key = ($v['dynamic'])?"reload":"restart";
                                        $obj_list[$key][] =
                                            array(
                                                'module' => $this->name(),
                                                'obj_type' => $k,
                                                'obj_name' => $obj_name,
                                                'sub_type' => $child_obj_type,
                                                'sub_name' => $child_obj_name,
                                                'status' =>  $child_obj->status(),
                                                'description' => $mod_desc.'/'.$obj_desc.'/'.$obj_name.'/'.$child_desc.'/'.$child_obj_name,
                                            );
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // Loop around modules
        foreach($this->modules() as $mod) {
            $mod->modified_object_list($obj_list);
        }
    }

    /**
     * @brief Check conflict for a particular resource
     *
     *  For example TCP port in use by another module cfg/obj...
     *  Implementation is done per module.
     *
     *  Return an array of detected conflict, following format:
     *  'name'       => Module reporting conflict
     *  'obj_type'   => Object type
     *  'obj_name'   => (opt) Object name (except for configuration)
     *  'child_type' => (opt) Child type
     *  'child_name' => (opt) Object name
     *
     * @param[in out] $src_module_name - (string) Module asking for conflict check
     * @param[in out] $resource_type   = (string) Resource type to check, for ex: "IP", "PORT"...
     * @param[in out] $resource_data   = (array)  Resource data
     *
     * @return
     */
    public function check_resource_conflict($src_module_name, $resource_type, $resource_data)
    {
        $conflicts = array();
        // Loop around modules
        foreach($this->modules() as $mod) {
            // Skip ourself :)
            if($src_module_name == $mod->name()){
                continue;
            }
            $sub_conflicts = array();
            $sub_conflicts = $mod->check_resource_conflict($src_module_name, $resource_type, $resource_data);
            if($sub_conflicts){
                $conflicts = array_merge($conflicts, $sub_conflicts);
            }
        }
        return $conflicts;
    }
    public function check_display_name_conflict($src_module_name, $src_obj_type, $disp_name, $obj_name)
    {
        $conflict = false;
        $skip = true;

        $module = $this->find_module_by_name($src_module_name);
        $objects = $module->object_enum($src_obj_type, ADD_NONE_NEVER, false, null, true);

        while ($skip) {
            $list = array_search($disp_name, $objects);
            if ($list) {
                if ($list == $obj_name) {
                    unset($objects[$list]);
                }
                else {
                    $skip = false;
                    $conflict = true;
                }
            } 
            else {
                $skip = false;
            }
        }
        return $conflict;
    }
    public function check_object_usage($src_module_name, $obj_type, $obj_name, $sub_typ=null, $sub_name=null)
    {
        $usages = array();
        foreach($this->modules() as $mod) {
            // Skip ourself :)
            if($src_module_name == $mod->name()){
                continue;
            }
            $sub_usages = array();
            $sub_usages = $mod->check_object_usage($src_module_name, $obj_type, $obj_name, $sub_typ, $sub_name);
            if($sub_usages){
                $usages = array_merge($usages, $sub_usages);
            }
        }
        return $usages;
    }
    public function check_object_modified($obj, $src_module_name, $obj_type, $obj_name, $sub_typ=null, $sub_name=null)
    {
        $modifies = array();
        foreach($this->modules() as $mod) {
            // Skip ourself :)
            if($src_module_name == $mod->name()){
                continue;
            }
            $sub_modifies = array();
            $sub_modifies = $mod->check_object_modified($obj, $src_module_name, $obj_type, $obj_name, $sub_typ, $sub_name);
            if($sub_modifies){
                $modifies = array_merge($modifies, $sub_modifies);
            }
        }
        return $modifies;
    }
    /**
     * @brief Module checklist
     *
     * Return list of item not complying to check list
     *
     * @return
     */
    public function checklist(&$list)
    {
        // Loop around modules
        foreach($this->modules() as $mod) {
            $mod->checklist($list);
        }
        return count($list);
    }
    public function handle_global_messages(&$msg_list, $direct_call = false)
    {
        // Loop around modules
        foreach($this->modules() as $mod) {
            $mod->handle_global_messages($msg_list, $direct_call);
        }
        return count($msg_list);
    }
    public function set_audit_params($params)
    {
        $this->_audit_parameters = $params;
    }
    public function get_audit_params()
    {
        return $this->_audit_parameters;
    }
    public function get_audit_points(&$audit_points)
    {
        // Create our auditpoints
        foreach($this->_audit_parameters as $name => $point){
            if($point['type']){
                $audit_class = 'Safe_audit_point_'.$point['type'];
                if(class_exists($audit_class)){
                    $obj = new $audit_class($this, $point['name']);
                    $obj->set_parameters($point);
                    $audit_points[] = $obj;
                }else{
                    safe_log(SAFE_LOG_ERROR, "WARNING - ".$this->name().": Cannot instantiate audit point <${audit_class}>");
                }
            }
        }
        // Loop around modules
        foreach ($this->modules() as $module) {
            $module->get_audit_points($audit_points);
        }
    }
    public function get_object_enum($type)
    {
        return $this->get_aggregate_object_name_list($type);
    }
    public function object_enum($type, $add_none = ADD_NONE_ALWAYS, $optional = false, $none_label = null)
    {
        $object_enum = $this->get_object_enum($type);
        return safe_create_enum_list($object_enum, $add_none, $optional, $none_label);
    }
    public function process_name()
    {
        return $this->name();
    }
    /**
     * @brief Check conflict for a particular resource
     *  Return an array of object description:
     *  'module'   => Module description
     *  'obj_type' => Object type description
     *  'obj_name' => Object name
     *  'sub_type' => (opt) Child type description
     *  'sub_name' => (opt) Child Object name
     *  'controller_url' => (opt) controller url set in module
     *  'url' =>  Object url for edit
     *
     * @param[in out] $obj_data - (array) "obj_type", "obj_name", "sub_type", "sub_name"
     *
     * @return
     */
    public function get_object_description($obj_data){
        $description = array();
        $url = '';
        $controller_url = '';
        $description['module'] = $this->description();
        if(isset($obj_data['obj_type'])){
            $obj_type_def = $this->get_aggregate_object_definition($obj_data['obj_type']);
            $description['obj_type'] = $obj_type_def['name'];
            if(isset($obj_type_def['controller_url'])){
                $description['controller_url'] = $obj_type_def['controller_url'];
            }
            if(isset($obj_data['obj_name'])){
                $url = "/{$obj_data['name']}/{$obj_data['obj_type']}/{$obj_data['obj_name']}";
                $description['obj_name'] = $obj_data['obj_name'];
                if(isset($obj_data['sub_type']) && isset($obj_data['sub_name'])){
                    $aggr_obj = $this->get_aggregate_objects($obj_data['obj_type']);
                    $obj = $aggr_obj[$obj_data['obj_name']];
                    if($obj){
                        $sub_obj_type_def = $obj->get_aggregate_object_definition($obj_data['sub_type']);
                        $description['sub_type'] = $sub_obj_type_def['name'];
                        $description['sub_name'] = $obj_data['sub_name'];
                        $url = $url . "/{$obj_data['sub_type']}/{$obj_data['sub_name']}";
                    }
                }
            }
        }
        if($url){
            $description['url'] = $url;
        }
        return $description;
    }
    public function pre_reset($info, &$reason)
    {
        safe_log(SAFE_LOG_INFO, 'pre_reset - '.$this->name().' - '.$info['type']);
        // Check if product script exists
        $script = $this->get_hook_script('pre', 'reset.'.$info['type']);
        if($script) {
            safe_log(SAFE_LOG_INFO, 'pre_reset - Invoking ' . $script);
            $this->node()->execute('nohup '.$script. ' ' . $info['type'], $retCode, true, true, $rc);
        }
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->pre_reset($info, $reason)) return false;
        }
        return true;
    }
    public function do_reset($info, &$reason)
    {
        safe_log(SAFE_LOG_INFO, 'do_reset - '.$this->name().' - '.$info['type']);
        if($info['type'] == 'config' || $info['type'] == 'factory' ){
            // All modules have performed their cleanup, now remove all objects:
            // 1- Remove all (N) record from object table
            // 2- Change all records to (D)
            // 3- Disable error checking in configuration manager
            // 4- Generate config (this will proceed to all delete)
            $ser = Safe_object_serializer_class::get_serializer(TRUE);
            $ser->delete_object_with_status($this->object_name() , TRUE, Safe_object_serializer_class::OBJ_STATUS_NEW);
            $ser->update_object_with_status($this->object_name() , TRUE, NULL, Safe_object_serializer_class::OBJ_STATUS_DELETED);
            //clear the checklist and modify list cache
            $this->clear_list_cache();
        }
        $this->clear_cache();
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->do_reset($info, $reason)) return false;
        }
        return true;
    }
    public function post_reset($info, &$reason)
    {
        safe_log(SAFE_LOG_INFO, 'post_reset - '.$this->name().' - '.$info['type']);
        // Check if product script exists
        $script = $this->get_hook_script('post', 'reset.'.$info['type']);
        if($script) {
            safe_log(SAFE_LOG_INFO, 'post_reset - Invoking ' . $script);
            $this->node()->execute('nohup '.$script. ' ' . $info['type'], $retCode, true, true, $rc);
        }
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_reset($info, $reason)) return false;
        }
        return true;
    }
    /*---------------------------------------------------------------------
     * Troubleshooter section
     *---------------------------------------------------------------------*/
    public function troubleshooter_enabled()
    {
        return false;
    }
    /**
     * @brief Populate troubleshooter configuration parameters (configurable
     * object)
     *
     * @param[in out] $configurable
     *
     * @return
     */
    public function troubleshooter_parameters(&$configurable)
    {
        return true;
    }
    /**
     * @brief Start troubleshooting process according to $parameters
     *
     * @param[in out] $parameters
     * @param[in out] $output
     *
     * @return boolean to indicate success/failure to start
     */
    public function troubleshooter_start($parameters, &$output=null)
    {
        return true;
    }
    /**
     * @brief Stop troubleshooting process
     *
     * @param[in out] $parameters
     * @param[in out] $output
     *
     * @return boolean to indicate success/failure to stop (stopping a non
     * started process should be considered as success)
     */
    public function troubleshooter_stop($parameters, &$output=null)
    {
        return true;
    }
    /**
     * @brief Retrieve troubleshooting process status
     *
     * @param[in out] $parameters
     * @param[in out] $output
     *
     * @return process status according to STATUS_* const in Safe_service_class
     */
    public function troubleshooter_status($parameters, &$output=null)
    {
        return Safe_service_class::STATUS_STOPPED;
    }
    /**
     * @brief Populate $artifacts array with file pathname of produced artifacts
     *
     * @param[in out] $parameters
     * @param[in out] $artifacts
     * @param[in out] $output
     *
     * @return
     */
    public function troubleshooter_artifacts($parameters, &$artifacts, &$output=null)
    {
        return true;
    }
    public function can_create_aggregate_object($obj_type=null, $name, &$output = null){
        //if in configuration rest update mode, pass by can create check
        if($this->node()->get_config_status('update') == 'rest'){
            return true;
        }else{ 
            return parent::can_create_aggregate_object($obj_type, $name, $output);
        }
    }
}
/* End of file safe_module_class.php */
