<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * Application class
*/
require_once ('application/helpers/safe_helper.php');
require_once ('safe_service_class.php');
require_once ('safe_application_settings_class.php');
class Safe_application_class extends Safe_service_class
{
    protected $_preferences = NULL;
    public function __construct($node, $name, $process_name = null)
    {
        parent::__construct($node, $name, $process_name);
    }
    /**
     * @brief Default category getter for application
     *
     * @param[in out] $name
     *
     * @return
     */
    public function category()
    {
        return 'Application';
    }

    public function register_modules()
    {
        // Nothing to register, just base class method
    }

    public function configure()
    {
        // Preferences
        $this->_preferences = new Safe_application_settings_class($this->path() . '/application',$this->node());
        $this->_preferences->configure();
        $this->_preferences->synch();
        $license = $this->node()->software()->license();
        if ($license->status()) {
            // Check for product name change in license
            // - Brand
            // - Product
            // - or user defined object description
            $prod_name = trim($license->info('Brand'));
            if (!$prod_name) $prod_name = trim($license->info('Product'));
            if ($prod_name) $this->set_description($prod_name);
        }
        $this->set_documentation( array(
            'The Application module manages the system startup mode, configuration and licensing.',
            )
        );
        // expose configuratio as aggregate object to allow methods: apply,
        // reload...
        $this->register_aggregate_object( 'configuration',
            array(
                'name' => 'Configuration',
                'description' => array(
                    'Manages overall Application configuration.',
                    ),
                'singleton' => true,
                'dynamic' => true,
                'controller_url' => '/SAFe/sng_config_manager',
                'methods' => array(
                    'reload' => array(
                        'name' => 'Reload',
                        'description' => array( 'Apply and reload all dynamic configuration.' ),
                        'request' => 'POST',
                    ),
                    'apply' => array(
                        'name' => 'Apply',
                        'description' => array( 'Apply Application configuration.' ),
                        'request' => 'POST',
                    ),
                    'smartapply' => array(
                            'name' => 'Smart Apply',
                            'description' => array( 'Smart Apply Application configuration.' ),
                            'request' => 'POST',
                    ),
                    'status' => array(
                        'name' => 'Get Status',
                        'description' => array( 'Get Application configuration status.' ),
                        'request' => 'GET',
                    ),
                )
            )
        );
        $this->register_aggregate_object( 'license',
            array(
                'name' => 'License',
                'singleton' => true,
                'base_path' => false,
                'description' => array(
                    'Manages Application licensed Features.',
                    ),
                'controller_url' => '/SAFe/sng_license',
                'methods' => array(
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => array( 'Returns Application currently installed license and status.' ),
                        'request' => 'GET',
                    ),
                    'info' => array(
                        'name' => 'Info',
                        'description' => array(
                            'Returns System information required to produce a new License.'
                        ),
                        'request' => 'GET',
                    ),
                ),
            )
        );
        $this->register_aggregate_object( 'version',
            array(
                'name' => 'Version',
                'description' => array(
                    'Manages Application components version.',
                    ),
                'singleton' => true,
                'base_path' => false,
                'methods' => array(
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => array( 'Returns Application version information.' ),
                        'request' => 'GET',
                    ),
                ),
            )
        );
        $this->register_aggregate_object( 'preferences',
            array(
                'name' => 'Preferences',
                'description' => array(
                    'Manages Application preferences.',
                    ),
                'singleton' => true,
                'base_path' => false,
                'methods' => array(
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => array( 'Returns Application preferences.' ),
                        'request' => 'GET',
                    ),
                    'update' => array(
                        'name' => 'Update',
                        'description' => array( 'Modifies Application preferences.' ),
                        'request' => 'POST',
                    ),
                ),
            )
        );
        return parent::configure();
    }
    /**
     * @brief Find module by name (either in modules or services)
     *
     * @param[in out] $name
     *
     * @return
     */
    public function find_module_by_name($name)
    {
        if($name == $this->name() || 'application' == $name) {
            return $this;
        }
        if($name == 'hardware') {
            return $this->node()->hardware();
        }
        $mod = null;
        foreach($this->node()->software()->services() as $k => $v) {
            // skip ourself :)
            if($k != $this->name()) {
                if($name == $k) {
                    return $v;
                }else{
                    $mod = $v->find_module_by_name($name);
                    if($mod)
                        return $mod;
                }
            }
        }
        return parent::find_module_by_name($name);
    }
    /**
     * @brief find obj by module name, type, obj name (either in modules or
     * services)
     *
     * @param
     *            [in out] $name
     *
     * @return
     *
     */
    public function find_obj_by_name($module_name, $type, $obj_name, $sub_type = null, $sub_obj_name = null) {
        $module = $this->find_module_by_name ( $module_name );
        if($module){
            $obj = $module->get_aggregate_object($type, $obj_name);
            if($obj) {
                if ($sub_type != null && $sub_obj_name != null) {
                    $sub_obj = $obj->get_aggregate_object($sub_type,
                            $sub_obj_name);
                    if($sub_obj){
                        return $sub_obj;
                    }
                }else{
                    return $obj;
                }
            }
        }
        return false;
    }
    public function find_obj_by_path($path) {
        $path =  str_replace('/NSC/local/software/', '', $path);
        $path =  str_replace('/NSC/local/hardware/', '', $path);
        $module = null;
        $type = null;
        $name = null;
        $sub_type = null;
        $sub_name = null;
        list ( $module, $type, $name, $sub_type, $sub_name ) = explode ( '/', $path );
        return $this->find_obj_by_name ( $module, $type, $name, $sub_type, $sub_name );
    }
    public function find_module_by_path($path) {
        $path =  str_replace('/NSC/local/software/', '', $path);
        $path =  str_replace('/NSC/local/hardware/', '', $path);
        $module = null;
        $type = null;
        $name = null;
        $sub_type = null;
        $sub_name = null;
        list ( $module, $type, $name, $sub_type, $sub_name ) = explode ( '/', $path );
        return $this->find_module_by_name ( $module );
    }

    /**
     * @brief
     *
     * @return
     */
    public function get_data_settings()
    {
        $cfg_manager = $this->node()->configuration_manager();
        $archive_manager = $cfg_manager->archive_manager();
        $data = array(
            'log' => array(
                'description' => 'Log File',
                'filesystem' => TRUE,
                'pattern' => $cfg_manager->directory('log') . '/' . $this->name() . '.log*',
                'global_capabilities' => array(
                    'download',
                    //'rotate'
                ) ,
                'capabilities' => array(
                    'download',
                    'delete',
                    'view'
                ),
                'methods' => array(
                        'download' => array(
                                'name' => 'Download',
                                'description' => array( 'Download log files.' ),
                                'request' => 'GET',
                                'icon' => 'download',
                                'button' => true,
                                'action_url' => '/SAFe/sng_data_manager/download/' . $this->name() . '/log'
                        ),
                        'delete' => array(
                                'name' => 'Delete',
                                'description' => array( 'Delete a  Core Dump files.' ),
                                'request' => 'POST',
                        ),
                ),
            ) ,
            'core' => array(
                'description' => 'Core Dump',
                'filesystem' => TRUE,
                'pattern' => $cfg_manager->directory('core-dump') . '/core.' . $this->name() . '*',
                'global_capabilities' => array(
                    'delete'
                ) ,
                'capabilities' => array(
                    'download',
                    'delete'
                ),
                'methods' => array(
                        'download' => array(
                                'name' => 'Download',
                                'description' => array( 'Download a Core Dump files.' ),
                                'request' => 'GET',
                                'icon' => 'download',
                                'button' => true,
                                'action_url' => '/SAFe/sng_data_manager/download/' . $this->name() . '/core'
                        ),
                        'delete' => array(
                                'name' => 'Delete',
                                'description' => array( 'Delete a  Core Dump files.' ),
                                'request' => 'POST',
                                'icon' => 'times',
                                'confirm' => true,
                                'action_url' => '/SAFe/sng_data_manager/delete_confirm/' . $this->name() . '/core'
                        ),
                ),
            ) ,
            'corebt' => array(
                'description' => 'Core Dump Back Trace',
                'filesystem' => TRUE,
                'pattern' => $cfg_manager->directory('core-dump') . '/core.' . $this->name() . '*.bt',
                'global_capabilities' => array(
                    'download',
                    'delete'
                ) ,
                'capabilities' => array(
                    'download',
                    'delete',
                    'view'
                ),
                'methods' => array(
                        'download' => array(
                                'name' => 'Download',
                                'description' => array( 'Download a Core Dump files.' ),
                                'request' => 'GET',
                                'icon' => 'download',
                                'button' => true,
                                'action_url' => '/SAFe/sng_data_manager/download/' . $this->name() . '/corebt'
                        ),
                        'delete' => array(
                                'name' => 'Delete',
                                'description' => array( 'Delete a  Core Dump files.' ),
                                'request' => 'POST',
                                'icon' => 'times',
                                'confirm' => true,
                                'action_url' => '/SAFe/sng_data_manager/delete_confirm/' . $this->name() . '/corebt'
                        ),
                ),
            ) ,
        );
        // Get license info
        $license = $cfg_manager->license();
        if($license){
            // Ensure we're having a
            $this->node()->os()->mkdir($cfg_manager->directory('base') . '/upload');
            $data['license'] = array(
                        'description' => 'License',
                        'filesystem' => true,
                        'singleton' => true,
                        'pattern' =>  $cfg_manager->directory('base') . '/upload/license.tgz',
                        'file_ext' => 'zip|tgz|gz|tar.gz',
                        'global_capabilities_no_data' => array(
                            'upload',
                        ) ,
                        'capabilities' => array(
                            'download'
                        ),
                        'methods' => array(
                            'upload' => array(
                                    'icon' => 'upload',
                                    'name' => 'Upload',
                                    'description' => array( 'Upload License.' ),
                                    'request' => 'POST',
                            )
                        )
                );
        }
        if (isset($archive_manager)) $data['archive'] = array(
            'description' => 'Archive',
            'filesystem' => true,
            'pattern' => $archive_manager->directory(true) ,
            'file_ext' => 'tgz|gz|tar.gz',
            'global_capabilities_no_data' => array(
                'backup',
                'upload'
            ) ,
            'capabilities' => array(
                'download',
                'delete',
                'restore'
            ),
            'global_methods' => true,
            'methods' => array(
                    'backup' => array(
                            'icon' => 'archive',
                            'scope' => 'global',
                            'name' => 'Backup',
                            'description' => array( 'Backup system settings.' ),
                            'request' => 'POST',
                            'class' => $this->api_backup_archive_parameters()->get_data(false),
                    ),
                    'download' => array(
                            'name' => 'Download',
                            'description' => array( 'Download a backup files.' ),
                            'request' => 'GET',
                    ),
                    'upload' => array(
                            'icon' => 'upload',
                            'scope' => 'global',
                            'name' => 'Upload',
                            'description' => array( 'Upload a backup files.' ),
                            'request' => 'POST',
                    ),
                    'delete' => array(
                            'name' => 'Delete',
                            'description' => array( 'Delete a backup files.' ),
                            'request' => 'POST',
                    ),
                    'restore' => array(
                            'name' => 'Restore',
                            'description' => array( 'Restore a backup files.' ),
                            'request' => 'POST',
                    )
            ),
        );
        return $data;
    }
    public function get_data_file_content_by_key($data_name, $key, $line_count = NULL)
    {
        // Do not return core content too large
        if ('core' == $data_name) return '';
        return parent::get_data_file_content_by_key($data_name, $key, $line_count);
    }
    public function get_data_count($data_name)
    {
        if ('license' == $data_name){
            $license = $this->node()->software()->license();
            return (true == $license->status())?1:0;
        }
        return parent::get_data_count($data_name);
    }
    public function get_data_files($data_name)
    {
        if ('license' == $data_name){
            $license = $this->node()->software()->license();
            if(true == $license->status()){
                $info = $this->node()->configuration_manager()->license();
                return array(basename($info['file']) => $info['file'], 'License.txt.sig' => $info['file'] . '.sig');
            }
        }
        return parent::get_data_files($data_name);
    }
    public function data_summary($data_name){
        if('archive' == $data_name){
            $the_product = Safe_get_product();
            $cfg_mgr = $the_product->local_node()->configuration_manager();
            $archive_mgr = $cfg_mgr->archive_manager();
            $data_files = $this->get_data_files($data_name);
            $data_summary = array();
            foreach ($data_files as $key => $file) {
                if($key == 'sng-factory-reset-image.tgz'){
                    continue;
                }
                $info = $archive_mgr->info($key);
                $data_summary[] = array('Name' => $key , 'Type' => $info['type']);
            }
            return $data_summary;
        }else{
            return parent::data_summary($data_name);
        }
    }
    /**
     * @brief Return data description
     * Special hook for license
     *
     * @param[in out] $data_name
     * @param[in out] $data_key
     * @param[in out] $output
     *
     * @return
     */
    public function describe_data($data_name = null, $data_key = null, &$output = null)
    {
        if('license' == $data_name){
            $license = $this->node()->software()->license();
            $output = $license->describe();
            // Add the system info in it
            /* TODO: Refine to create section in license table output
            $sys_info = $this->system_info();
            $output = array_merge($output, $sys_info);
             */

            return true;
        }
        return parent::describe_data($data_name, $data_key, $output);
    }
    public function initialize_data($data_name = null, $data_key = null, &$output = null)
    {
        if('license' == $data_name){
            $info = $this->node()->configuration_manager()->license();
            $data_cfg = $this->get_data_settings();
            $data_cfg = $data_cfg[$data_name];
            $file = $data_cfg['pattern'];
            $ext = pathinfo($file);
            $file = $ext['dirname'].'/'.$data_key;
            $script_path = $this->node()->configuration_manager()->directory('sng-base') . '/scripts';
            $rc = $this->node()->os()->execute($script_path . '/update-license.sh '.$file.' '.$info['file'].' '.$info['pubkey'], $output, true, true);
            // In any case remove the file
            $this->node()->os()->delete_file($file);
            //reload the new license info to obj
            //send out the license obj modified event
            $license_obj = $this->node()->software()->license(true);
            $path_info = $this->path_info();
            $receivers = $this->node()->software()->application()->check_object_modified($this, $path_info['module'], 'license', $license_obj->name());
            return $rc;
        }
        return parent::initialize_data($data_name, $data_key, $output);
    }

    public function modified_object_list(&$obj_list, $split_type=true)
    {
        //check if have modified_object_list in cache
        $data = array();
        if($this->get_data_cache('modified_object_list', $data)){
            $obj_list = $data;
        }else{
            // Parent function
            parent::modified_object_list($obj_list);
            // Take care of services
            foreach($this->node()->software()->services() as $service) {
                if($service->name() != $this->name()) {
                    $service->modified_object_list($obj_list);
                }
            }
            $this->node()->hardware()->modified_object_list($obj_list);
            //save modified_object_list to cache
            $this->set_data_cache('modified_object_list', $obj_list);
        }
    }
    public function modified_module_list(&$module_list)
    {
        $modified_list = array();
        $this->modified_object_list($modified_list);

        $_modified_list = array();

        if($modified_list['reload']){
            $_modified_list = array_merge($_modified_list, $modified_list['reload']);
        }
        if($modified_list['restart']){
            $_modified_list = array_merge($_modified_list, $modified_list['restart']);
        }

        $module_list = array();
        foreach($_modified_list as $item){
            if(!in_array($item['module'], $module_list)){
                $module_list[] = $item['module'];
            }
        }
    }
    /**
     * @brief Application conflict check - Add services in the loop
     *
     * @param[in out] $src_module_name
     * @param[in out] $resource_type
     * @param[in out] $resource_data
     *
     * @return
     */
    public function check_resource_conflict($src_module_name, $resource_type, $resource_data)
    {
        // Parent function
        $conflicts = parent::check_resource_conflict($src_module_name, $resource_type, $resource_data) ;
        // Take care of services
        foreach($this->node()->software()->services() as $service) {
            if($service->name() != $this->name() && $src_module_name != $service->name()) {
                $sub_conflicts = array();
                $sub_conflicts = $service->check_resource_conflict($src_module_name, $resource_type, $resource_data) ;
                if($sub_conflicts){
                    $conflicts = array_merge($conflicts, $sub_conflicts);
                }
            }
        }
        return $conflicts;
    }
    public function check_object_usage($src_module_name, $obj_type, $obj_name, $sub_typ=null, $sub_name=null)
    {
        // Parent function
        $usages = parent::check_object_usage($src_module_name, $obj_type, $obj_name, $sub_typ, $sub_name) ;
        // Take care of services
        foreach($this->node()->software()->services() as $service) {
            if($service->name() != $this->name() && $src_module_name != $service->name()) {
                $sub_usages = array();
                $sub_usages = $service->check_object_usage($src_module_name, $obj_type, $obj_name, $sub_typ, $sub_name) ;
                if($sub_usages){
                    $usages = array_merge($usages, $sub_usages);
                }
            }
        }
        return $usages;
    }
    public function check_object_modified($obj, $src_module_name, $obj_type, $obj_name, $sub_typ=null, $sub_name=null)
    {
        // Parent function
        $modifies = parent::check_object_modified($obj, $src_module_name, $obj_type, $obj_name, $sub_typ, $sub_name) ;
        // Take care of services
        foreach($this->node()->software()->services() as $service) {
            if($service->name() != $this->name() && $src_module_name != $service->name()) {
                $sub_modifies = array();
                $sub_modifies = $service->check_object_modified($obj, $src_module_name, $obj_type, $obj_name, $sub_typ, $sub_name) ;
                if($sub_modifies){
                    $modifies = array_merge($modifies, $sub_modifies);
                }
            }
        }
        return $modifies;
    }
    /**
     * @brief Check if update package can be install
     *
     * @param[in out] $reason
     *
     * @return true/false
     */
    public function can_install_data($info, &$reason)
    {
        if('update' == $info['type']){
            if (Safe_service_class::STATUS_RUNNING == $this->status()) {
                $reason[] = $this->description() . ' is Running';
                return false;
            }
        }
        return parent::can_install_data($info, $reason);
    }

    public function restart($force = false, &$result = null)
    {
        $cluster_monitor = $this->find_module_by_name('clustermon');

        if ($cluster_monitor !== NULL) {
            return $cluster_monitor->restart_service($this, $force, $result);
        } else {
            return $this->restart_service($force, $result);
        }
    }

    public function restart_service($force = false, &$result = null)
    {
        return parent::restart($force, $result);
    }

    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_status_configuration($name = null, $data = null, &$output = null)
    {
        $result = array();
        $result['modified'] = false;
        $result['status'] = false;
        $result['can_apply'] = false;
        $result['can_reload'] = false;
        $result['reboot_request'] = false;
        $result['troubleshooter_running'] = false;

        // Checklist
        $checklist = array();
        $this->node()->checklist($checklist, false);
        // Checklist ?
        if($checklist){
            $result['status_text']  = 'error';
            $result['description']  = 'Configuration not completed.';
            $result['checklist'] = array(
                'status_text' => 'error',
                'description' => 'Configuration not completed.',
            );
            foreach($checklist as $item){
                $module = $this->find_module_by_name($item['module']);
                if($module){
                    $mod_name = $module->description();
                    $obj_def = $module->get_aggregate_object_definition($item['obj_type']);
                    if($obj_def){
                        $obj_type = $obj_def['name'];
                        if ($item['url']) {
                            $fix_url = $item['url'];
                        } else {
                            $fix_url = $obj_def['controller_url'];
                        }
                    }else{
                        $obj_type = $item['obj_type'];
                    }
                    $obj_name = $item['obj_name'];
                }else{
                    $mod_name = $item['module'];
                    $obj_type = $item['obj_type'];
                    $obj_name = $item['obj_name'];
                }
                $item['url'] = $fix_url;
                if($item['action']){
                    $item['url'] .= '/'.$item['action'];
                }
                $item['description'] = $mod_name . ' '. $obj_type.' '.$obj_name.' '.$item['description'];
                $result['checklist']['items'][] = $item;
            }
        } else if ($this->node()->is_configuration_modified()) {
            if (!$this->node()->is_configuration_generated()){
                $result['status_text']  = 'error';
                $result['description']  = 'Configuration not generated.';
            }else{
                $result['status_text']  = 'warning';
                $result['description']  = 'Configuration modified.';
            }
            $result['modified'] = true;
            $modified_obj_list = array();
            $this->modified_object_list($modified_obj_list);
            // Check application status
            if (Safe_service_class::STATUS_RUNNING == $this->status()){
                // Modified and running
                $result['can_apply'] = true;
                $result['can_reload'] = false;

                if($modified_obj_list['restart']){
                    $result['restart'] = array(
                        'status_text' => 'warning',
                        'url' => '/SAFe/sng_config_manager',
                        'action' => 'restart');
                    $result['restart']['items'] = $modified_obj_list['restart'];
                }
                if($modified_obj_list['reload']){
                    $result['can_reload'] = true;
                    $result['reload'] = array(
                        'status_text' => 'info',
                        'url' => '/SAFe/sng_config_manager',
                        'action' => 'reload');
                    $result['reload']['items'] = $modified_obj_list['reload'];
                }
            }else{
                //modified and not running
                $result['can_apply'] = true;
                $result['can_reload'] = false;
                $result['apply'] = array(
                    'status_text' => 'info',
                    'url' => '/SAFe/sng_config_manager',
                    'action' => 'apply');
                $result['apply']['items'] = array();
                if($modified_obj_list['reload']){
                    $result['apply']['items'] = array_merge($result['apply']['items'], $modified_obj_list['reload']);
                }
                if($modified_obj_list['restart']){
                    $result['apply']['items'] = array_merge($result['apply']['items'],$modified_obj_list['restart']);
                }
            }
            $clustermon_service = $this->node()->software()->application()->find_module_by_name('clustermon');
            $res = $clustermon_service->check_master();
            if($res === false){
                $result['display'] = false;
            }
        }else{
            $result['status'] = true;
            $result['status_text']  = 'ok';
            $result['description']  = 'Configuration is up to date.';
        }
        // Add configuration generate status: count, last generate date
        $result['generate'] = $this->node()->configuration_manager()->generate_status();

        // Check if system reboot has been requested
        $reboot_requests = $this->get_reboot_requests();
        if(!empty($reboot_requests)){
            $result['reboot'] = $reboot_requests;
            $result['reboot_request'] = true;
        }

        return $result;
    }
    public function api_smartapply_configuration($name = null, $data = null, &$output = null){
        $status_result = $this->api_status_configuration();
        if($status_result){
            if(isset($status_result['status']) && $status_result['status']){
                $rs = true;//Configuration is up to date.
            }else{
                if(isset($status_result['checklist']) && $status_result['checklist']){
                        $output['status'] =  $status_result;
                        $output['message'] = 'Apply configuration failed.';
                        return false;//check list have errors
                }else{
                    if(isset($status_result['modified']) && $status_result['modified']){
                        $action = 'apply';
                        if (Safe_service_class::STATUS_RUNNING == $this->status()){
                            // Modified and running
                            if(isset($status_result['restart']) && $status_result['restart']){
                                $action = 'restart';
                            }else{
                                if(isset($status_result['reload']) && $status_result['reload']){
                                    $action = 'reload';
                                }
                            }
                        }
                        if($action == 'reload'){
                            $rs = $this->api_reload_configuration(null, null, $output);
                        }else{
                            $param = array();
                            if($action == 'restart'){
                                $param['restart'] = true;
                                $param['call_from'] = 'view_controller';//auto restart nsc
                            }
                            $rs = $this->api_apply_configuration(null, $param, $output);
                        }
                    }else{
                        $rs = false;
                        $output['message'] = 'Apply configuration failed.';
                    }
                }
            }
        }else{
            $rs = false;
            $output['message'] = 'Get configuration status error.';
        }
        //clear cache 
        $this->clear_list_cache();
        return $rs;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_apply_configuration($name = null, $data = null, &$output = null){
        $rc = true;
        if(isset($data['restart']) && $data['restart']==true ){
            $restart = true;
        }
        if(isset($data['force']) && $data['force']==true ){
            $this->node()->configuration_manager()->forget_error(TRUE);
            $force = true;
        }
        if(isset($data['call_from']) && $data['call_from']=='view_controller' ){
            $call_from = 'view_controller';
        }else{
            $call_from = 'rest';
        }
        //check if have enough space to apply
        $need_space = 500;//500 mb
        $disk_space = disk_free_space ( '/' )/1024/1024; 
        if($disk_space <= $need_space){
            $output['message'] = 'Apply configuration failed.';
            $output['reason'][] = array(
                    'module' =>'storage',
                    'obj_type' =>'/',
                    'description' => "No enough space(less than {$need_space}MB).",
                    'url' => '/SAFe/sng_storage_status',
                );
            return false;
        }
        if ($force || $this->node()->is_configuration_modified()) {
            if ($call_from == 'rest' && Safe_service_class::STATUS_RUNNING == $this->status()){
                //if call from rest api, and system is running return error
                $output['sys_running'] ='System is Running';
                $rc = false;
            }else{
                // Checklist ?
                $checklist = array();
                $this->node()->checklist($checklist, false);
                if($checklist){
                    $output['message'] = 'Apply configuration failed.';
                    $output['reason'] = 'Configuration not completed.';
                    $output['checklist']['items'][] = $checklist;
                    $rc = false;
                }else{
                    $reload_list = array();
                    $this->modified_module_list($reload_list);
                    Safe_enter_critical_section();
                    if (!$this->node()->generate_config()) {
                        $output['message'] = 'Apply configuration failed.';
                        $output['reason'] = $this->node()->configuration_manager()->progress();
                        $rc = false;
                    } else {
                        if ($restart) {
                            $reason = array();
                            $rc_restart = $this->restart($force, $reason);
                            if (0 == $rc_restart) {
                                $output['restart_succeeded'] = 'System apply configuration and restart succeeded.';
                                $rc = true;
                            } else {
                                $output['restart_failed'] = 'System apply configuration and restart failed.';
                                $output['restart_failed_data'] = $reason;
                                $rc = false;
                            }
                        }
                        // Check if some service with modified config and
                        // supporting reload and running -> post_write
                        foreach($reload_list as $module_name) {
                            $module = $this->find_module_by_name($module_name);
                            // TODO - need to loop around obj_type
                            if(is_subclass_of($module, "Safe_service_class")
                                    &&
                                    $module->status == Safe_service_class::STATUS_RUNNING ) {
                                if(!$module->post_write_config()) {
                                    $output[] = 'Post Write Failed for '.$module->name() . '.';
                                }
                            }
                        }
                    }
                    Safe_leave_critical_section();
                }
            }
        }
        return $rc;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_reload_configuration($name = null, $data = null, &$output = null){
        if(isset($data['call_from']) && $data['call_from'] == 'view_controller' ){
            $call_from = 'view_controller';
        }else{
            $call_from = 'rest';
        }
        //check if have enough space to apply
        $need_space = 500;//500 mb
        $disk_space = disk_free_space ( '/' )/1024/1024;
        if($disk_space <= $need_space){
            $output['message'] = 'Apply configuration failed.';
            $output['reason'][] = array(
                    'module' =>'storage',
                    'obj_type' =>'/',
                    'description' => "No enough space(less than {$need_space}MB).",
                    'url' => '/SAFe/sng_storage_status',
            );
            return false;
        }
        Safe_enter_critical_section();
        $reload_list = array();
        $this->modified_module_list($reload_list);
        $cfg_mgr = $this->node()->configuration_manager();
        $rc = true;
        if ($call_from == 'rest' && Safe_service_class::STATUS_RUNNING != $this->status()){
            //if call from rest api, and system is not running return error
            $output['sys_running'] ='System is not Running';
            $rc = false;
        }
        if($rc) {
            foreach($reload_list as $module_name) {
                $module = $this->find_module_by_name($module_name);
                // TODO - need to loop around obj_type
                if(!$module->reload_generate_config($cfg_mgr)) {
                    $output[] = 'Generate configuration failed for ' . $module->name() . '.';
                    $rc = false;
                }
            }
            // Ensure application reload_generate is called
            if(!$this->reload_generate_config($cfg_mgr)) {
                $output[] = 'Generate configuration failed for ' . $this->name() . '.';
                $rc = false;
            }
        }
        if($rc) {
            // Step 2
            if($cfg_mgr->errors()) {
                $output['cfg_mgr'] = 'Apply configuration failed.';
                $output['cfg_mgr_data'] = $cfg_mgr->progress();
                $rc = false;
            }
            if($rc) {
                // Step 3
                if(!$cfg_mgr->write_config($this->node())) {
                    $output[] = 'Write configuration failed.';
                    $rc = false;
                }
                // Step 4
                foreach($reload_list as $module_name) {
                    $module = $this->find_module_by_name($module_name);
                    // TODO - need to loop around obj_type
                    if(!$module->post_write_config()) {
                        $output[] = 'Post Write Failed for '.$module->name() . '.';
                    }
                }
                // Step 5
                foreach($reload_list as $module_name) {
                    $module = $this->find_module_by_name($module_name);
                    // TODO - need to loop around obj_type
                    $module->reload_clear_modified(true);
                }
                // Application post_reload
                if(!$this->post_reload()) {
                    $output[] = 'Post Reload Failed for '.$this->name() . '.';
                }
            }
        }
        Safe_leave_critical_section();
        return $rc;
    }


    public function api_create_preferences($name=null, $data = null, &$output = null){
        //for documentation list use
        return $this->_preferences;
    }
    public function api_retrieve_preferences($name=null, $data = null, &$output = null){
        return $this->_preferences;
    }
    public function api_update_preferences($name=null, $data=null, &$output = null) {
        $preferences = $this->api_retrieve_preferences($name);
        if($preferences) {
            if ($preferences->validate($data,$output)) {
                if (true == $preferences->save()) {
                    return true;
                }
            }
        }
        return false;
    }

    public function api_create_license($name=null, $data = null, &$output = null){
        // Cannot create one
        return $this->api_retrieve_license();
    }
    public function api_retrieve_license($name=null, $data = null, &$output = null){
        return $this->node()->software()->license();
    }
    public function api_info_license($name=null, $data = null, &$output = null){
        return $this->api_retrieve_system_info();
    }
    public function api_upload_license($name=null, $data = null, &$output = null){
        $data_def = $this->get_data_settings();
        $upload_api_method  = $data_def['license']['methods']['upload']['request'];
        if($upload_api_method){
            return $this->upload_data('license', $name , $output , $upload_api_method);
        }else{
            $output['message'] = 'Upload request method error.';
            return false;
        }
    }

    public function api_create_version($name=null, $data = null, &$output = null){
        // Cannot create one
        return $this->api_retrieve_version();
    }
    public function api_retrieve_version($name=null, $data = null, &$output = null){
        return $this->node()->software()->version();
    }

    public function api_retrieve_system_info($name=null, $data = null, &$output = null){
        $sys_info = $this->system_info();

        $obj = new Safe_configurable_object_class('','sys_info');
        foreach($sys_info as $k => $v){
            $obj->add_field($k, $k, 'text', $v);
        }
        return $obj;
    }
    /**
     * @brief Return System information used to generate license
     *
     * @return array of key value pairs
     */
    public function system_info()
    {
        $info = array();
        // Check for hw serial numbers
        $result = array();
        $script_path = $this->node()->configuration_manager()->directory('sng-base') . '/bin';
        $this->node()->os()->execute($script_path . '/sng-system-info', $result, true, true);
            foreach($result as $res){
                $item = explode('=',$res);
                $info[$item[0]] = $item[1];
            }

        return $info;
    }
    public function api_download_log($name=null, $data = null, &$output = null){
        $this->download_data('log', $name);
        exit();
    }
    public function api_delete_log($name=null, $data = null, &$output = null){
        if ($name){
            return $this->delete_data('log', $name);
        }else{
            //not allow download all files
            return false;
        }
    }
    public function api_download_core($name=null, $data = null, &$output = null){
        if ($name){
            $this->download_data('core', $name);
            exit();
        }else{
            //not allow download all files
            return false;
        }

    }
    public function api_delete_core($name=null, $data = null, &$output = null){
        return $this->delete_data('core', $name);
    }
    public function api_download_corebt($name=null, $data = null, &$output = null){
        $this->download_data('corebt', $name);
        exit();
    }
    public function api_delete_corebt($name=null, $data = null, &$output = null){
        return $this->delete_data('corebt', $name);
    }

    public function api_backup_archive($name=null, $data = null, &$output = null){
        $the_product = Safe_get_product();
        $cfg_mgr = $the_product->local_node()->configuration_manager();
        $archive_mgr = $cfg_mgr->archive_manager();
        $reason = array();
        // Validate submitted data
        $cfg = $this->api_backup_archive_parameters();

        if($cfg->validate($data, $reason)) {
            $cfg->set_data_values($data);
            $type = $cfg->get_data_value('backup_type', false);
            if($cfg->get_data_value('name', false)){
                $name = $cfg->get_data_value('name', false);
            }
            // prevent creating a backup (configuration or system) if the checklist is not empty
            if($type == 'configuration' || $type == 'system'){
                $checklist = array();
                $this->node()->checklist($checklist, false);
                if($type == 'system' && $name == 'sng-factory-reset-image.tgz'){
                    //allow system factory-reset backup
                }else{
                    // Checklist ?
                    if($checklist){
                        // Display error
                        $output = array('Fail to create archive: configuration not completed.');
                        return false;
                    }
                }
            }
            if ($the_product->can_backup($type, $reason)) {
                $the_product->pre_backup($type, $reason);
                // Create archive
                $archive = $archive_mgr->create_archive($type, $name);
                // Post backup hook
                $the_product->post_backup($type, $reason);
                if (FALSE !== $archive) {
                    return array('name' => $archive);
                } else {
                    // Display error
                    $output = array('Fail to create archive');
                    return false;
                }
            }
        }
        $output = $reason;
        return false;
    }
    public function api_restore_archive_parameters()
    {
        $the_product = Safe_get_product();
        $cfg_mgr = $the_product->local_node()->configuration_manager();
        $archive_mgr = $cfg_mgr->archive_manager();
        return $archive_mgr->get_archive_restore_configurable_object();
    }
    public function api_backup_archive_parameters()
    {
        $the_product = Safe_get_product();
        $cfg_mgr = $the_product->local_node()->configuration_manager();
        $archive_mgr = $cfg_mgr->archive_manager();
        return $archive_mgr->get_archive_configurable_object();
    }
    public function api_download_archive($name=null, $data = null, &$output = null){
        if ($name){
            $this->download_data('archive', $name);
        }else{
            return false;
        }

    }
    public function api_upload_archive($name=null, $data = null, &$output = null){
        $data_def = $this->get_data_settings();
        $upload_api_method  = $data_def['archive']['methods']['upload']['request'];
        if($upload_api_method){
            return $this->upload_data('archive', $name , $output , $upload_api_method);
        }else{
            $output['message'] = 'Upload request method error.';
            return false;
        }
    }
    public function api_delete_archive($name=null, $data = null, &$output = null){
        if ($name){
            return $this->delete_data('archive', $name);
        }else{
            return false;
        }
    }
    public function api_restore_archive($name=null, $data = null, &$output = null, $skip_network = false, $skip_license = false,$skip_rest_api = false){
        $the_product = Safe_get_product();
        $cfg_mgr = $the_product->local_node()->configuration_manager();
        $archive_mgr = $cfg_mgr->archive_manager();
        $reasons = array();
        $info = $archive_mgr->info($name);
        if(empty($info) || !in_array($info['type'], array('configuration','system'))){
            $output[] = array(
                    'module' => 'application',
                    'obj_type' => 'archive',
                    'obj_name' => $name,
                    'description' => "Backup type error, cannot restore."
            ); 
            return false;
        }
        $auto_reboot = false;
        //allow the rest api pass these paramters
        if(isset($data['backup_exclude_opts'])){
            foreach($data['backup_exclude_opts'] as $tmp_item){
                if(strtolower($tmp_item) == 'network'){
                    $skip_network = true;
                    continue;
                }
                if(strtolower($tmp_item) == 'license'){
                    $skip_license = true;
                    continue;
                }
                if(strtolower($tmp_item) == 'rest_api'){
                    $skip_rest_api = true;
                    continue;
                }
            }
            $auto_reboot=true;
        }

        $info['skip_network'] = $skip_network;
        $info['skip-rest-api'] = true;//$skip_rest_api;
        if ($archive_mgr->can_restore($name, $reasons) && $the_product->can_restore($name, $reasons)) {
            // Loop through all service/modules for pre-restore
            if (!($rc = $the_product->pre_restore($info, $arch, $reasons))
                // Restore
                || !($rc = $archive_mgr->restore_archive($name, $skip_license))
                || !($rc = $the_product->post_restore($info, $reasons))
                    ) {
                if($auto_reboot)
                    $this->node()->os()->shutdown(array('mode'=>'reboot'),$output);
                $output = $reasons;
                return false;
            }
        }else{
            if($auto_reboot)
                $this->node()->os()->shutdown(array('mode'=>'reboot'),$output);
            $output = $reasons;
            return false;
        }
        if($auto_reboot)
            $this->node()->os()->shutdown(array('mode'=>'reboot'),$output);
        return true;
    }
    protected function get_hook_script($step , $type)
    {
        $script = parent::get_hook_script($step, $type);
        if(!$script){
            $base = $this->node()->configuration_manager()->directory('base');
            $_script = $base.'/hooks/application.' . $type .'.'.$step;
            if(file_exists($_script)) {
                return $_script;
            }
        }
        return $script;
    }
    /**
     * @brief Add system reboot request
     *
     * This function will update/create a reboot request file in
     * /var/sng/ui/reboot.json
     *
     * This file will be removed on boot by bootstrap mechanism so reboot
     * request is cleared
     *
     * @param[in out] $module  => Module adding reboot request
     * @param[in out] $reason  => Reason for reboot
     *
     * @return
     */
    const REBOOT_REQUEST = "/var/sng/ui/reboot.json";
    public function add_reboot_request($module, $reason)
    {
        $reboot_data = array();

        // Get reboot data if any
        if($this->node()->os()->read_file(self::REBOOT_REQUEST, $reboot_data)){
            // convert json to array
            $reboot_data = json_decode($reboot_data, true);
        }
        // Append new reboot request data
        $reboot_data[] = array($module => $reason);

        return $this->node()->os()->write_file(self::REBOOT_REQUEST, json_encode($reboot_data));
    }

    /**
     * @brief Return reboot requests data
     *
     * @return array, if empty no pending reboot request
     */
    public function get_reboot_requests()
    {
        $reboot_data = array();

        // Get reboot data if any
        if($this->node()->os()->read_file(self::REBOOT_REQUEST, $reboot_data)){
            // convert json to array
            $reboot_data = json_decode($reboot_data, true);
        }

        return $reboot_data;
    }
}
/* End of file safe_product_class.php */
