<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/**
 * SNG Monitor service wrapper
 *
 * @author William Adam
 * @version
 */
if (!defined('BASEPATH')) exit('No direct script access allowed');
require_once ('application/helpers/safe_helper.php');
safe_require_class('service');

class Sng_monitor_db_class
{
    private $db = null; // the codeigniter db object
    private $db_groupname = 'monitor'; // the CodeIgniter groupname set in
    private $db_tables = array( 'status', 'events' );

    public function configure()
    {
        try {
            $CI = & get_instance();
            // if the files for the db doesn't it will be created.
            $this->db = $CI->load->database($this->db_groupname, TRUE);
            foreach($this->db_tables as $table){
                if (!$this->db->table_exists($table)){
                    $func = 'create_table_'.$table;
                    if(!call_user_func(array($this, $func)) ){
                        error_log('ERROR create <'.$table.'> table');
                        return false;
                    }
                }
            }
        }
        catch(Exception $e) {
            // TODO: catch CodeIgniter DB_Driver exception.
            // we will never go here since CodeIgniter DB_Driver will fail
            // with a PHP error on the user browser
            error_log(__FUNCTION__ . ": can't open databases");
            return false;
        }
    }
    public function __destruct()
    {
        try {
            if ($this->db) $this->db->close();
        }
        catch(Exception $e) {
        }
    }
    private function create_table_status()
    {
        try {
            error_log("Safe_object_serializer: Trying to create Monitor Status DB");
            $sqlq = "
        CREATE TABLE 'status' (
          'id' INTEGER PRIMARY KEY AUTOINCREMENT,
          'modified_date' DATE DEFAULT (datetime('now','localtime')),
          'type' TINYTEXT NOT NULL,
          'module' TINYTEXT NOT NULL,
          'obj_type' TINYTEXT DEFAULT NULL,
          'status' TINYTEXT NOT NULL,
          'data' TEXT,
          CONSTRAINT unique_object UNIQUE ('type', 'module') ) ;
      ";
            if (!$this->db->query($sqlq)) throw new Exception("Cannot create DB!");
        }
        catch(Exception $e) {
            error_log(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
        return true;
    }
    private function create_table_events()
    {
        try {
            error_log("Safe_object_serializer: Trying to create Monitor Events DB");
            $sqlq = "
        CREATE TABLE 'events' (
          'id' INTEGER PRIMARY KEY AUTOINCREMENT,
          'date' DATE DEFAULT (datetime('now','localtime')),
          'local_date' DATE DEFAULT (datetime('now','localtime')),
          'module' TINYTEXT NOT NULL,
          'level' TINYTEXT NOT NULL,
          'data' TEXT
        );
      ";
            if (!$this->db->query($sqlq)) throw new Exception("Cannot create DB!");
        }
        catch(Exception $e) {
            error_log(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
        return true;
    }
    public function add_events($level, $events)
    {
        foreach($events as $event){
            //if($event->is_enabled()){
            {
                $insert['module'] = $event->module()->name();
                $insert['local_date'] =  date('Y-m-d H:i:s', $event->date());
                $def_tz = date_default_timezone_get();
                date_default_timezone_set("UTC");
                $insert['date'] =  date('Y-m-d H:i:s', $event->date());
                date_default_timezone_set($def_tz);
                $insert['level'] = $event->level(false);
                $data = $event->to_array();
                // Add some data to help report
                $data['category'] = $event->service_desc();
                $data['description'] = $event->event_desc();
                $data['status'] = $event->status();
                $data['severity'] = $event->level(true);
                $insert['data'] = json_encode($data);
                if(!$this->db->insert('events', $insert)){
                    error_log('Fail to add event');
                }
            }
        }
    }
    public function events_count($module=null)
    {
        if($module){
            $this->db->where(array('module'=> $module));
        }
        $this->db->select('module, level, count(*) AS count');
        $this->db->group_by('module, level');
        $query = $this->db->get('events');
        $result = array();
        if($query){
            foreach($query->result() as $row){
                $result[$row->module][$row->level] = $row->count;
            }
        }
        return $result;
    }

    public function events_list($module=null, $level=null, $hour_period=null)
    {
        if($module){
            $this->db->where(array('module'=> $module));
        }
        if($level){
            $this->db->where(array('level'=> $level));
        }
        if($hour_period){
            $this->db->where("date >= datetime('now', '-{$hour_period} hours')");
        }
        $this->db->order_by('date', 'desc');
        $query = $this->db->get('events');
        $result = array();
        if($query){
            foreach($query->result_array() as $row){
                $row['data'] = json_decode($row['data'],true);
                $result[] = $row;
            }
        }
        return $result;
    }
    public function events_cleanup($life)
    {
        $this->db->where("date < datetime('now', '-{$life} days')");
        $query = $this->db->delete('events');
    }

    /**
     * @brief Retrieve events time summary
     * SQL = select date(date) as _date, level, count(*) as count from events where 
     *        date >= datetime('now', '-24 hours')  group by _date, level;
     *
     * @param[in out] $hour_period
     *
     * @return 
     */
    public function events_time_summary($hour_period=24)
    {
        $this->db->select("date, level, strftime('%Y-%m-%d %H:00:00', local_date) as _date, count(*) as count");
        $this->db->where("date >= datetime('now', '-{$hour_period} hours')");
        $this->db->order_by("date");
        $this->db->group_by(array("_date", "level"));

        $query = $this->db->get('events');
        $result = array();
        if($query){
            foreach($query->result() as $row){
                $timestamp = strtotime($row->_date);
                $result[$row->level][] =array((int)$timestamp*1000, (int)$row->count);
            }
        }
        return $result;
    }
    public function update_status($path, $status, $data)
    {
        $path_parts = explode('/', $path);
        $module = $path_parts[0];
        $type = $module; 
        $obj_type = ''; 
        if($path_parts[1]){
            $type = $path_parts[1];
        }
        if($path_parts[2]){
            $obj_type = $path_parts[2];
        }
        // Build data
        $insert['status'] = $status;
        $insert['data'] = json_encode($data);
        $insert['module'] = $module;
        $insert['type'] = $type;
        $insert['obj_type'] = $obj_type;
        $insert['modified_date'] = date('Y-m-d H:i:s');
        // Build where
        $where['module'] = $module;
        $where['type'] = $type;
        //$where['obj_name'] = $path_parts[2];
        // try to get
        $q = $this->db->get_where('status', $where);
        if($q->num_rows()){
            $this->db->where($where);
            $this->db->update('status', $insert);
        }else{
            $this->db->insert('status', $insert);
        }
    }
    public function get_status($type=null)
    {
        if($type){
            $this->db->where(array('type'=>$type));
        }
        $this->db->order_by('id');
        $query = $this->db->get('status');
        if($query){
            return $query->result_array();
        }else{
            return array();
        }
    }


}
class Sng_monitor_config_class extends Safe_configurable_object_class
{
    public function __construct($node, $parent_name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, "configuration", $node);
    }
    public function configure()
    {
        // General section
        $this->add_field('general/event_life', 'Days to Keep Events', 'text', '7','5');
        $this->set_field_rules('general/event_life', 'required|integer|greater_than[0]');
        $this->set_field_category('general/event_life', 'General');
        return parent::configure();
    }
}

class Sng_monitor_service_class extends Safe_service_class
{
    private $_db = null;
    private function _db(){
        if(!$$this->_db){
            $this->_db = new Sng_monitor_db_class();
            $this->_db->configure();
        }
        return $this->_db;
    }
    public function get_status($type=null)
    {
        return $this->_db()->get_status($type);
    }
    public function update_status($status)
    {
        return $this->_db()->update_status($status['name'], $status['status'], $status['data']);
    }
    public function add_events($level, $events)
    {
        return $this->_db()->add_events($level, $events);
    }
    public function events_count($module=null)
    {
        return $this->_db()->events_count($module);
    }
    public function events_time_summary($hour_period=24)
    {
        return $this->_db()->events_time_summary($hour_period);
    }
    public function events_list($module=null, $level=null, $hour_period=null)
    {
        return $this->_db()->events_list($module, $level, $hour_period);
    }
    public function cleanup()
    {
        // Get event life time
        $life = $this->config()->get_data_value('general/event_life', false);
        if($life){
            // DB cleanup
            $this->_db()->events_cleanup($life);
        }
    }
    public function __construct($software)
    {
        parent::__construct($software, 'monitor', 'sngmonit');
    }
    /**
     * @brief
     *
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("Monitor");
        // Create the config object
        $cfg = new Sng_monitor_config_class($this->node(), $this->path() . '/' .$this->name());
        // Synch with DB
        $cfg->configure();
        $cfg->synch();
        // Attach config to module
        $this->set_config($cfg);
        return parent::configure();
    }

    public function can_restore($info, &$reason)
    {
        // Prevent calling parent as service status will be checked
        return true;
    }
    public function allow_user_ctl()
    {
        return true;
    }
    /**
     * @brief Generate sshd configuration
     *
     * @param[in out] $config_manager
     *
     * @return true on success
     */
    public function generate_config(&$config_manager)
    {
        //Generates service configuration files
        if (!$this->_generate_service_config($config_manager)) return false;
        // Invoke parent generate
        return parent::generate_config($config_manager);
    }
    public function _generate_service_config(&$config_manager)
    {
        $auditor = $this->node()->software()->application()->find_module_by_name('auditor');
        $auditor->generate_service_configuration($config_manager);
        $notifier = $this->node()->software()->application()->find_module_by_name('notifier');

        return true;
    }
    /**
     * @brief Invoked after a successfull write_config
     *
     * @return
     */
    public function post_write_config($obj_type=null)
    {
        // restart service
        if($this->status() == Safe_service_class::STATUS_RUNNING){
            $this->restart();
        }
        return parent::post_write_config($obj_type);
    }
    public function support_reload(&$need_reload = null) {
        parent::support_reload($need_reload);
        return true;
    }
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        return $this->generate_config($config_manager);
    }
    public function reload_clear_modified($obj_type=null)
    {
        $auditor = $this->node()->software()->application()->find_module_by_name('auditor');
        $auditor->clear_configuration_modified();
        $notifier = $this->node()->software()->application()->find_module_by_name('notifier');
        $notifier->clear_configuration_modified();
        return $this->clear_configuration_modified();
    }

    public function is_configuration_modified($include_childs = true)
    {
        $auditor = $this->node()->software()->application()->find_module_by_name('auditor');
        $notifier = $this->node()->software()->application()->find_module_by_name('notifier');
        if(true == $auditor->is_configuration_modified($include_childs) ||
            true == $notifier->is_configuration_modified($include_childs))
        {
            return true;
        }
        return parent::is_configuration_modified($include_childs);

    }
    public function severity_level_enum()
    {
        return Safe_audit_point::severity_enum();
    }
}
/* End of file sng_sshd_service_class.php */
