<?php

///////////////////////////////////////////////////////////////////////////////
//
// Copyright 2010 ClearFoundation.
//
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//
///////////////////////////////////////////////////////////////////////////////

require_once("../../gui/Webconfig.inc.php");
require_once("../../api/ShellExec.class.php");
require_once("../../api/File.class.php");
require_once(GlobalGetLanguageTemplate(__FILE__));
require_once("../include/SngProduct.class.php");
require_once("../include/DynamicDiv.class.php");

///////////////////////////////////////////////////////////////////////////////
//
// Variables
//
///////////////////////////////////////////////////////////////////////////////
$productName = "";
$prodDefFilePath = "/usr/local/sng/conf/prod-def.xml";
$packageList = array();
$updateProperties = null;

///////////////////////////////////////////////////////////////////////////////
//
// Main
//
///////////////////////////////////////////////////////////////////////////////


//Create a dynamic div element to be refreshed with ajax once the update is 
//intalling.
$installDynDivName = "UpdateLogs";
$installDynDiv = new DynamicDiv($installDynDivName, $_SERVER['PHP_SELF'], "", 2000);

//Check if the request is a refresh.
if ($installDynDiv->IsDynamicDivRefreshRequest()){
  //This is an ajax request.
  DisplayUpdateLogs(true);
}
else {
  //This is not an ajax request.
  // Set the product name according to the parameter "product" given in the 
  // request
  if( isset($_REQUEST['product']) && !isset($_SESSION['product']) ) {
    registerProduct($_REQUEST['product']);
  }

  $productName = getProductName();

  /*if( $productName == "" ) {
    WebDialogWarning("Product Name was not registered properly");
    exit();
  }*/

  WebAuthenticate();
  WebHeader(WEB_LANG_PAGE_TITLE);
  WebDialogIntro(WEB_LANG_PAGE_TITLE, "/images/icon-diskusage.png", WEB_LANG_PAGE_INTRO);

  $updateProperties = GetUpdateProperties($prodDefFilePath, $productName);
  $rpmsUpdate = new RpmsUpdate($productName, $updateProperties->repo, 
                               $updateProperties->conf);

  // Handle Update
  if ( isset($_POST['InstallUpdate']) ) {
    if ( $rpmsUpdate->InstallUpdate() ){

      $installDynDiv->Enable();
      $installDynDiv->DynamicDivBegin();
      DisplayUpdateLogs();
      $installDynDiv->DynamicDivEnd();
    }
    else {
      WebDialogWarning("Update failed!");
    }
  }
  else {

    //Check if there is an update available.
    if( !$rpmsUpdate->hasUpdate($packageList) ) {
      WebDialogInfo("No update available.");
    }
    else {

      WebDialogInfo("An update is available.");
      DisplayUpdates($packageList);
    }
  }

  WebFooter();
}




///////////////////////////////////////////////////////////////////////////////
// F U N C T I O N S
///////////////////////////////////////////////////////////////////////////////

//Parse product configuration file and return the necessary update 
//informations.
function GetUpdateProperties($in_prodDefFilePath, $in_productName)
{

  $updateElements = "";
  $xml = parseProdDefXML($in_prodDefFilePath);

  if( !getProductValue($xml, $in_productName, "update", $updateElements) ) {
    WebDialogWarning("Unable to find update properties.");
    exit();
  }

  return $updateElements;
}

//Display update informations with the changelog.
function DisplayUpdates($in_packageList)
{
  global $productName;

  WebTableOpen(WEB_LANG_PACKAGE_LIST);

  $appProdVersRpmName = GetAppProductVersionRpmName($in_packageList);

  if($appProdVersRpmName === NULL){
    WebDialogWarning("Failed to retrieve app-$productName-version rpm.");
    WebTableClose();
    exit();
  }

  $appProdVersRpm = DownloadPatchRpm($appProdVersRpmName);

  if($appProdVersRpm === NULL){
    WebDialogWarning("Failed to download $appProdVersRpmName.");
    WebTableClose();
    exit();
  }

  //Rpm is now downloaded, display details.
  if( !DisplayUpdateDetails($appProdVersRpm) ){
    WebDialogWarning("Failed to display update details.");
    WebTableClose();
    exit();
  }
  
  DisplayInstallToolbar($productName);
  WebTableClose();
}

//Display update rpm version, date and changelog.
function DisplayUpdateDetails($in_rpmFilePath)
{
  $cmd = "rpm -qip $in_rpmFilePath";
  $output = array();
  $rc = null;

  exec($cmd, $output, $rc);

  if($rc != 0){
    return false;
  }

  //Parse the output
  $version = null;
  $date = null;
  $matchArray = array();
  foreach ($output as $key => $line) {
    //Find rpm version
    if( preg_match("/^Version[\s:]+?([\d.]+)/", $line, $matchArray) ){
      $version = $matchArray[1];
    }
    
    //Find rpm build date
    if( preg_match("/Build Date[\s:]+?([\w :]+)/", $line, $matchArray) ){
      $date = $matchArray[1];
    }
  }

  echo "<tr>";
  echo "<td class='mytablesubheader' valign='top' width='200'>";
  echo "Version";
  echo "</td>";
  echo "<td>";
  if($version === NULL){
    echo "Not found";
  }
  else {
    echo $version;
  }
  echo "</td>";
  echo "</tr>";
  echo "<tr>";
  echo "<td class='mytablesubheader' valign='top'>";
  echo "Date";
  echo "</td>";
  echo "<td>";
  if($date === NULL){
    echo "Not found";
  }
  else {
    echo $date;
  }
  echo "</td>";
  echo "</tr>";

  DisplayChangelog($in_rpmFilePath);

  return true;
}

function DisplayChangelog($in_rpmFilePath)
{
  $cmd = "rpm -qp --changelog $in_rpmFilePath";
  $output = array();
  $rc = null;

  exec($cmd, $output, $rc);

  if($rc != 0){
    return false;
  }

  echo "<tr>";
  echo "<td class='mytablesubheader' valign='top'>";
  echo "Changelog";
  echo "</td>";
  echo "<td>";
  if($rc != 0){
    echo "Not found";
  }
  else {
    foreach ($output as $key => $value) {
      echo $value."</br>";
    }
  }
  echo "</td>";
  echo "</tr>";
}

//Download rpm to COMMON_TEMP_DIR
function DownloadPatchRpm($in_rpmName)
{
  global $productName;
  global $updateProperties;

  $tempDir = COMMON_TEMP_DIR;
  $repoFile = $updateProperties->repo;
  $repoName = $productName . "-patches";

  if ( !file_exists($repoFile) ) {
    WebDialogWarning("Can't find repo file $repoFile");
    return NULL;
  }

  $repoInfo = parse_ini_file($repoFile, true);
  if ($repoInfo == FALSE){
    WebDialogWarning("Invalid repo file");
  }

  if ( !array_key_exists($repoName, $repoInfo) ) {
    WebDialogWarning("Can't find repo info");
    return NULL;
  }
  
  if ( !array_key_exists("baseurl", $repoInfo[$repoName]) ) {
    WebDialogWarning("Can't find repo baseurl");
    return NULL;
  }

  $repoBaseUrl = $repoInfo[$repoName]["baseurl"];
  $rpmUrl = $repoBaseUrl. "/" . $in_rpmName;

  $cmd = "wget";
  $args = "-N --directory-prefix " . $tempDir . " " . $rpmUrl;
  $output = array();
  $rc = null;

  //wget tend to send output to stderr, $output is ignored...
  exec($cmd. " " . $args, $output, $rc);

  if( $rc != 0 ) {
    return NULL;
  }
  else {
    return $tempDir . "/" . $in_rpmName;
  }
}

//Return name of the update rpm, should be app-$productName-version
function GetAppProductVersionRpmName ($in_packageList)
{
  global $productName;

  $appName = "app-$productName-version";
  $appVersion = preg_grep("/^app-$productName-version/", $in_packageList);

  //There should only have one app-productName-version rpm
  if ( count($appVersion) != 1 ){

    return NULL;
  }
  else {
    $rpmName = end($appVersion);

    //Line returned by yum check-update is divided into 3 parts like:
    //rpm-name.baseArch     version    repo-name
    //Split the line on white space
    $split_rpmName = preg_split("/\s+/", $rpmName);
    $baseArch = preg_replace("/$appName.(\w+)/", "$1", $split_rpmName[0]);
    $rpmVersion = $split_rpmName[1];

    $rpmName = $appName ."-".$rpmVersion .".". $baseArch. ".rpm";
    return $rpmName;
  }
}

//Display install update button
function DisplayInstallToolbar()
{
  global $productName;

  WebFormOpen($_SERVER['PHP_SELF'] . "?product=" . $productName);

  echo "<tr>";
  echo "<td class='mytablesubheader' valign='top'>";
  echo "</td>";
  echo "<td> ".
    WebButton("InstallUpdate", "Update", WEBCONFIG_ICON_UPDATE) .
  "</td></tr>";
  WebFormClose();
}

//Display output of update being installed.
function DisplayUpdateLogs($in_isRefresh = false)
{
  $isBusy = RpmsUpdate::IsBusy();
  $logContent = RpmsUpdate::GetLogs();

  $output = "";

  foreach ($logContent as $line) {
    $output = $output . "<tr><td>$line</br></td></tr>";
  }

  if($in_isRefresh === false || $isBusy){
    //Update is still running show please wait...
    $output = $output . "<tr><td>" .WEBCONFIG_ICON_LOADING. "&nbsp;"
                      . "Installing update... please wait". "</br></td></tr>";
  }
  else {
    //Update is finish
    WebDialogInfo("Update installed successfully!");
  }

  WebTableOpen(WEB_LANG_PACKAGE_LIST);
  echo $output;
  WebTableClose();
}


///////////////////////////////////////////////////////////////////////////////
//
// Inner class
//
///////////////////////////////////////////////////////////////////////////////

//Handle update installation.
class RpmsUpdate
{
  const COMMAND_PID = "/sbin/pidof";
  const COMMAND_YUM = "/usr/bin/yum";
  const FILE_INSTALL_LOG = "install-update.log";
  const CHECK_UPDATE_SCRIPT = "/usr/local/sng/scripts/check-update.sh";
  const INSTALL_UPDATE_SCRIPT = "/usr/local/sng/scripts/install-update.sh";

  private $productName = "";
  private $repoFile = "";
  private $confFile = "";
  private $updateLogsPath = "";

  public function __construct($in_productName, $in_repoFile, $in_confFile)
  {
    $this->productName = $in_productName;
    $this->repoFile = $in_repoFile;
    $this->confFile = $in_confFile;
    $this->updateLogsPath = COMMON_TEMP_DIR . "/" . self::FILE_INSTALL_LOG;
  }

  //Check if a new app-PRODUCT-version is available. If so return true and all the 
  //package that will be updated in $out_packageList.
  public function hasUpdate(&$out_packageList)
  {

    $yumConfFile = $this->confFile;
    $repoFile = $this->repoFile;

    try {
      $shell = new ShellExec();
      $args = escapeshellarg($yumConfFile)." ".escapeshellarg($this->productName)." ".escapeshellarg("patches");

      $rc = $shell->Execute(escapeshellcmd(self::CHECK_UPDATE_SCRIPT), $args);
      if( $rc == 1 ) {
        $errorStr = "";
        foreach ($shell->GetOutput() as $key => $value) {
          $errorStr = $errorStr . "\n" . $value;
        }
        WebDialogWarning($errorStr);
        return false;
      } else if( $rc == 2 ) {
        array_push($out_packageList, "No Updates Available");
      } else if( $rc == 0 ) {
        $out_packageList = $shell->GetOutput();
      } else {
        WebDialogWarning("Unknown return code.");
        return false;
      }
    } catch (Exception $e) {
      WebDialogWarning($e->GetMessage());
      return false;
    }

    //Check that package list as app-PRODUCT-version
    if ( count(preg_grep("/^app-".$this->productName."-version/", $out_packageList)) ){
      return true;
    }
    else{
      return false;
    }
  }

  public function InstallUpdate()
  {

    $yumConfFile = $this->confFile;
    $repoFile = $this->repoFile;

    $shell = new ShellExec();
    $args = escapeshellarg($yumConfFile)." ".escapeshellarg($this->productName)." ".escapeshellarg("patches");
    $options = array();
    $options['background'] = true;
    $options['log'] = self::FILE_INSTALL_LOG;

    //Reset log file
    $outFile = new File($this->updateLogsPath);
    try {
      if ($outFile->Exists()){
        $outFile->Delete();
      }
      $outFile->Create("webconfig", "webconfig", "0644");
    } catch (Exception $e) {
      WebDialogWarning($e->GetMessage());
      return false;
    }

    //Launch update installation.
    try {

      $rc = $shell->Execute(escapeshellcmd(self::INSTALL_UPDATE_SCRIPT), $args, false, $options ); 
      if( $rc != 0 ) {

        $errorStr = "";
        foreach ($shell->GetOutput() as $key => $value) {
          $errorStr = $errorStr . "\n" . $value;
        }
        WebDialogWarning($errorStr);
        return false;
      }

    } catch (Exception $e) {
      WebDialogWarning($e->GetMessage());
      return false;
    }

    return true;
  }

  //Return true is an update is already running, false if not.
  public static function IsBusy()
  {
    try {
      $shell = new ShellExec();
      $exitcode = $shell->Execute(self::COMMAND_PID, "-s -x " . self::COMMAND_YUM, false);
    } catch (Exception $e) {
      //TODO check if null when called
      return null;
    }
        
    if ($exitcode == 0)
      return true;
    else
      return false;
  } 

  //Return update log as an array.
  public static function GetLogs() 
  {
    try {
      $log = new File(COMMON_TEMP_DIR . "/" . self::FILE_INSTALL_LOG);
      $lines = $log->GetContentsAsArray();
    } catch (FileNotFoundException $e) {
      $lines = array();
    } catch (Exception $e) {
      $lines = array();
    }
    
    return $lines;
  }

}

?>
