<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * Node class
*/
require_once ('application/helpers/safe_helper.php');
safe_require_class('safe_object_class');
safe_require_class('safe_module_class');

class Safe_audit_point extends Safe_object_class
{
    const TYPE_SYSTEM='system';
    const TYPE_FILESYSTEM='filesystem';
    const TYPE_PROCESS='process';
    const TYPE_SERVICE='service';
    const TYPE_FILE='file';
    const TYPE_DIRECTORY='directory';
    const TYPE_DEVICE='device';
    const TYPE_LOG='log';

    const RESOURCE_CPU='cpu';
    const RESOURCE_TOTALCPU='totalcpu';
    const RESOURCE_MEM='memory';
    const RESOURCE_LOADAVG='loadavg';
    const RESOURCE_SPACE='space';
    const RESOURCE_CONTENT='content';
    const RESOURCE_STATUS='status';
    const RESOURCE_TIMESTAMP='timestamp';

    const ID_1MIN='1min';
    const ID_5MIN='5min';
    const ID_15MIN='15min';
    const ID_USER='user';
    const ID_SYSTEM='system';
    const ID_WAIT='wait';

    // Audit point field definition
    protected $_level_label = 'Severity';
    protected $_action_label = 'Then ';
    protected $_category='';
    // Value field definition
    protected $_value_label = 'Limit';
    protected $_field_type = 'text';
    protected $_field_default = '0';
    protected $_field_size = '5';
    protected $_field_rules = 'integer|greater_or_equal[0]|less_or_equal[100]';

    protected $_value_fields = array(
        'value' => array());
    /*
        'field_name => array(
                        'label' => 'If Greated Than',
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => 0,
                        'rules' => ,
                        'value' => 'the value'
                    ),
    */

    // Audit point data
    public $_module = null;
    public $_name = null;
    public $_id = null;
    public $_obj_type='';
    public $_obj_name='';
    public $_sub_type=null;
    public $_sub_name=null;
    public $_action = 'false';
    public $_level = 'critical';
    public $_value = '0';

    public function __construct($module, $obj_type, $obj_name=null, $sub_type=null, $sub_name=null)
    {
        $name = $obj_type;
        if($obj_name){
            $name .= '/'.$obj_name;
        }
        if($sub_type){
            $name .= '/'.$sub_type;
        }
        if($sub_name){
            $name .= '/'.$sub_name;
        }

        parent::__construct('audit', $module->name().'/'.$name);
        $this->_name = $name;
        $this->_module = $module;
        $this->_obj_type = $obj_type;
        $this->_obj_name = $obj_name;
        $this->_sub_type = $sub_type;
        $this->_sub_name = $sub_name;
    }
    public function module()
    {
        return $this->_module;
    }
    public function category()
    {
        return $this->_category;
    }
    public function set_category($category)
    {
        $this->_category = $category;
    }
    public function set_parameters($parameters)
    {
        // Check common parameters
        if($parameters['description']){
            $this->set_description($parameters['description']);
        }
        if($parameters['level']){
            $this->_level = $parameters['level'];
        }
        if($parameters['action']){
            $this->_action = $parameters['action'];
        }
        if($parameters['category']){
            $this->set_category($parameters['category']);
        }
        // Fields parameters
        $params = $parameters['param'];
        // Strip value_fields
        $default_value = $this->_value_fields;
        $this->_value_fields = array();
        if($params){
            foreach($params as $param => $def){
                foreach($def as $k => $v){
                    $this->_value_fields[$param][$k] = $v;
                }
                if(!$this->_value_fields[$param]['label']){
                    $this->_value_fields[$param]['label'] = ucfirst($param);
                }
                $help_msg = array();
                if($default_value[$param]['help']){
                    $help_msg = to_index_array($default_value[$param]['help']);
                }
                if($this->_value_fields[$param]['help']){
                   $help_msg = array_merge( $help_msg ,to_index_array($this->_value_fields[$param]['help']));
                }
                if($help_msg){
                    $this->_value_fields[$param]['help'] = $help_msg;
                }
            }
        }
        // Fields help
        $help = $parameters['help'];
        if($help){
            $help = to_index_array($help);
            $this->set_documentation($help);
        }
        
    }
    protected function _field_layout($configurable)
    {
        // Create composite layout
        // Default is to include level and value in the action parameter row
        $row_fields = array($this->name().'/level');

        foreach($this->_value_fields as $k => $v){
            $row_fields[] = $this->name().'/'.$k;
        }

        $configurable->composite_layout(
            $this->name().'/action',
            $row_fields);
    }
    protected function _action_enum($configurable)
    {
        $action = array(
            'false' => 'Disable',
            'alert' => 'Enable',
            //'stop' => 'Stop',
            //'restart' => 'Restart',
        );
        return $action;
    }
    protected function _field_action_def($configurable)
    {
        $action = $this->_action_enum($configurable);
        $configurable->add_enum_field(
            $this->name().'/action', 
            $this->description(), 
            'dropdown',
            $this->_action,
            //$configurable->enable_disable_enum());
            $action);
        $help = array('Enable/Disable ' . $this->description() . ' audit point.');
        $configurable->set_field_help($this->name().'/action', $help);
    }
    static public function severity_enum()
    {
        $level = array(
            'info' => 'Information',
            'error' => 'Error',
            'critical' => 'Critical',
        );
        return $level;
    }
    protected function _level_enum()
    {
        return Safe_audit_point::severity_enum();
    }
    protected function _field_level_def($configurable)
    {
        $configurable->add_enum_field(
            $this->name().'/level', 
            $this->_level_label, 
            'dropdown',
            $this->_level,
            $this->_level_enum());
        $configurable->set_field_help($this->name().'/level','Select the severity level.');
    }

    protected function _field_value_def($configurable)
    {
        foreach($this->_value_fields as $k => $v){
            if('flag' == $v['type']){
                $configurable->add_enum_field(
                    $this->name().'/'.$k, 
                    (isset($v['label']))?$v['label']:$this->_value_label, 
                    'dropdown', 
                    (isset($v['default']))?$v['default']:$this->_value, 
                    (isset($v['value']))?$v['value']:$configurable->enable_disable_enum()); 
            }else{
                $configurable->add_field(
                    $this->name().'/'.$k, 
                    (isset($v['label']))?$v['label']:$this->_value_label, 
                    (isset($v['type']))?$v['type']:$this->_field_type, 
                    (isset($v['default']))?$v['default']:$this->_value, 
                    (isset($v['size']))?$v['size']:$this->_field_size); 
                if(isset($v['rules'])){
                    $configurable->set_field_rules($this->name().'/'.$k, $v['rules']);
                }elseif($this->_field_rules){
                    $configurable->set_field_rules($this->name().'/'.$k, $this->_field_rules);
                }
            }
            if(isset($v['help'])){
                $configurable->set_field_help($this->name().'/'.$k, $v['help']);
            }
        }
    }
    public function create_field($configurable)
    {
        $def = $this->_module->get_audit_params();
        if($def[$this->_name]){
            $this->set_parameters($def[$this->_name]);
        }

        // Create the field
        $this->_field_action_def($configurable);
        $this->_field_level_def($configurable);
        $this->_field_value_def($configurable);
        
        //generate help messages before composite_layout, composite_layout will process the help msg
        $fields = array_merge(array('action','level'), array_keys($this->_value_fields));
        $all_help_msg =  array();
        foreach($fields as $field_name){
            $field_name = $this->name() . '/' . $field_name;
            $tmp_help_msg = $configurable->get_field_help($field_name);
            if($tmp_help_msg){
                $all_help_msg[][$configurable->get_data_label($field_name)] = $tmp_help_msg;
                $configurable->set_field_help($field_name, array());
            }
        }
        
        $this->_field_layout($configurable);
        // Create fields in group
        $row_fields = array($this->name().'/level');

        foreach($this->_value_fields as $k => $v){
            $row_fields[] = $this->name().'/'.$k;
        }

        $configurable->create_group(
            $this->name().'/action',
            $row_fields);

        $configurable->conditional_control($this->name().'/action', $this->name().'/action');
        
        // Help specified ?
        if($this->documentation()){
            //add xml help msg at the top
            $all_help_msg =  array_merge($this->documentation(), $all_help_msg);
        }
        if($all_help_msg){
            $configurable->set_field_help($this->name().'/action', $all_help_msg);
        }
        // Associated category ?
        if($this->category()){
            $configurable->set_field_category($this->name().'/action', $this->category());
        }

    }
    public function set_field_value($configurable)
    {
        $this->_action = $configurable->get_data_value($this->name().'/action', false);
        $this->_level = $configurable->get_data_value($this->name().'/level', false);
        // Keep _value for backward compatibility
        $this->_value = $configurable->get_data_value($this->name().'/value', false);
        if($this->_value_fields['limit']){
            $this->_value = $configurable->get_data_value($this->name().'/limit', false);
        }
        // Populate multiple value fields
        foreach($this->_value_fields as $k => $v){
            $this->_value_fields[$k]['value'] = $configurable->get_data_value($this->name().'/'.$k, false);
        }
    }
    public function action($label=false)
    {
        if($label){
            $action_enum = $this->_action_enum();
            return $action_enum[$this->_action];
        }else{
            return $this->_action;
        }
    }
    public function level($label=false)
    {
        if($label){
            $level_enum = $this->_level_enum();
            return $level_enum[$this->_level];
        }else{
            return $this->_level;
        }
    }
    public function values()
    {
        $values = array();
        foreach($this->_value_fields as $k => $v){
            if(isset($v['value'])){
                $values[$k] = $v['value'];
            }else{
                $values[$k] = $this->_value;
            }
        }
        return $values;
    }
    public function value()
    {
        return $this->_value;
    }
    public function value_description()
    {
        return $this->_value_label.' '.$this->_value;
    }
    public function id()
    {
        return $this->name();
    }
    public function obj_type()
    {
        return $this->_obj_type;
    }
    public function obj_name()
    {
        return $this->_obj_name;
    }
    public function sub_type()
    {
        return $this->_sub_type;
    }
    public function sub_name()
    {
        return $this->_sub_name;
    }
    public function check_enabled()
    {
        return true;
    }
    public function is_enabled()
    {
        return 'false' !== $this->action();
    }
    protected function _monit_condition()
    {
        $cfg = array();

        return $cfg;
    }
    protected function _monit_check()
    {
        $cfg = array();

        return $cfg;
    }
    protected function _monit_config_header()
    {
        $monit_cfg = array();
        $monit_cfg[] = '# Audit point';
        $monit_cfg[] = '# -----------';
        $monit_cfg[] = '# Module   : '.$this->_module->name();
        $monit_cfg[] = '# Obj_type : '.$this->obj_type();

        return $monit_cfg;
    }

    public function monit_config()
    {
        $monit_cfg = array();
        if($this->check_enabled()){
            $monit_cfg[$this->id()]['header'] = $this->_monit_config_header();
            $monit_cfg[$this->id()]['check'] = $this->_monit_check();
            // Enabled ?
            if($this->is_enabled()){
                $monit_cfg[$this->id()]['condition'] = $this->_monit_condition();
            }else{
                $monit_cfg[$this->id()]['condition'] = array();
            }
        }

        return $monit_cfg;
    }
}
/******************************************************************************
 *   System Audit
 ******************************************************************************/
class Safe_audit_point_system extends Safe_audit_point
{
    public function id()
    {
        $hostname = $this->_module->node()->hostname();
        //return $this->_module->name();
        return $hostname;
    }
    protected function _monit_check()
    {
        return 'CHECK SYSTEM '.$this->id();
    }
}

class Safe_audit_point_system_load extends Safe_audit_point_system
{
    public function __construct($module)
    {
        parent::__construct($module, Safe_audit_point::RESOURCE_LOADAVG);
        $this->set_description('Load Average');
        $this->set_category('System');
        
        // Create process check fields
        $this->_value_fields = array(
            Safe_audit_point::ID_1MIN => array(
                        'label' => Safe_audit_point::ID_1MIN,
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'System load average threshold for ' . Safe_audit_point::ID_1MIN . '.'
                    ),
            Safe_audit_point::ID_5MIN => array(
                        'label' => Safe_audit_point::ID_5MIN,
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'System load average threshold for ' . Safe_audit_point::ID_5MIN . '.'
                    ),
            Safe_audit_point::ID_15MIN => array(
                        'label' => Safe_audit_point::ID_15MIN,
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'System load average threshold for  ' . Safe_audit_point::ID_15MIN . '.'
                    ),
                );
    }
    public function set_documentation($documentation)
    {
        $cpu_number = $this->_module->node()->os()->cpu_count();
        $msg = " Your system have {$cpu_number} cpu.";
        if(is_array($documentation)){
            $documentation[] = $msg;
        }else{
            $documentation = $documentation . $msg;
        }
        parent::set_documentation($documentation);
    }
    protected function _monit_condition()
    {
        $cfg = parent::_monit_condition();
        $allowed_key = array( Safe_audit_point::ID_1MIN, Safe_audit_point::ID_5MIN, Safe_audit_point::ID_15MIN);
        foreach($this->values() as $k => $v){
            if(in_array($k, $allowed_key)){
               if($v){
                    $cfg[] = '  IF loadavg('.$k.') > '.$v.' THEN alert';
                }
            }else{
                error_log('WARNING - Invalid Audit point parameter '.$k.' for '.$this->name().' skipping.');
            }
        }
        return $cfg;
    }
}

class Safe_audit_point_system_cpu extends Safe_audit_point_system
{
    public function __construct($module)
    {
        parent::__construct($module, Safe_audit_point::RESOURCE_CPU);
        $this->set_description('CPU Usage %');
        $this->set_category('System');
        // Create process check fields
        $this->_value_fields = array(
            Safe_audit_point::ID_USER => array(
                        'label' => ucfirst(Safe_audit_point::ID_USER),
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'CPU Usage threshold for ' . Safe_audit_point::ID_USER . '.'
                    ),
            Safe_audit_point::ID_SYSTEM => array(
                        'label' => ucfirst(Safe_audit_point::ID_SYSTEM),
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'CPU Usage threshold for ' . Safe_audit_point::ID_SYSTEM . '.'
                    ),
            Safe_audit_point::ID_WAIT => array(
                        'label' => ucfirst(Safe_audit_point::ID_WAIT),
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'CPU Usage threshold for ' . Safe_audit_point::ID_WAIT . '.'
                    ),
                );
    }
    protected function _monit_condition()
    {
        $cfg = parent::_monit_condition();
        $allowed_key = array(Safe_audit_point::ID_USER, Safe_audit_point::ID_SYSTEM, Safe_audit_point::ID_WAIT);
        foreach($this->values() as $k => $v){
            if(in_array($k, $allowed_key)){
               if($v){
                    $cfg[] = '  IF cpu('.$k.') > '.$v.'% THEN alert';
                }
            }else{
                error_log('WARNING - Invalid Audit point parameter '.$k.' for '.$this->name().' skipping.');
            }
        }
        return $cfg;
    }
}

class Safe_audit_point_system_memory extends Safe_audit_point_system
{
    public function __construct($module)
    {
        parent::__construct($module,Safe_audit_point::RESOURCE_MEM);
        $this->set_description('Memory Usage %');
        $this->set_category('System');
    }
    protected function _monit_condition()
    {
        $cfg = array();
        if($this->_value){
            $cfg[] = '  IF memory usage > '.$this->_value.'% THEN alert';
        }
        return $cfg;
    }
}

/******************************************************************************
 *   File Audit
 ******************************************************************************/
class Safe_audit_point_file_content extends Safe_audit_point
{
    protected $_regex='';
    protected $_file='';
    protected function _field_value_def($configurable)
    {
        // No value
    }
    public function __construct($module, $type, $file, $regex, $id, $description='Regular Expression', $category=null )
    {
        parent::__construct($module, $type, $id);
        $this->set_description($description);
        if($category){
            $this->set_category($category);
        }
        $this->_regex = $regex;
        $this->_file = $file;
    }
    public function value()
    {
        return $this->_regex;
    }
    protected function _monit_check()
    {
        return 'CHECK FILE '.$this->id().' WITH PATH '.$this->_file;
    }
    protected function _monit_condition()
    {
        $cfg[] = '  IF match "'.$this->_regex.'" THEN alert';
        return $cfg;
    }
    public function set_parameters($parameters)
    {
        if($parameters['param']['regex']){
            if($parameters['param']['regex']['value']){
                $this->_regex=$parameters['param']['regex']['value'];
            }
            unset($parameters['param']['regex']);
        }
        if($parameters['param']['file']){
            if($parameters['param']['file']['value']){
                $this->_regex=$parameters['param']['file']['value'];
            }
            unset($parameters['param']['file']);
        }
        parent::set_parameters($parameters);
    }
}

class Safe_audit_point_log_content extends Safe_audit_point_file_content
{
    public function __construct($module, $file, $regex, $id, $description=null, $category=null )
    {
        if(!$description){
            $description = ucfirst($id) . ' Messages';
        }
        if(!$category){
            $category = $module->description();
        }
        parent::__construct($module, Safe_audit_point::TYPE_LOG, $file, $regex, $id, $description, $category);
    }
}

class Safe_audit_point_log extends Safe_audit_point_log_content
{
    public function __construct($module, $id)
    {
        $cfg_mgr = $module->node()->configuration_manager();
        $file = '/var/log/'.$module->name().'.log';
        parent::__construct($module, $file, 'ERROR', $id);
    }
}
class Safe_audit_point_application_log extends Safe_audit_point_log_content
{
    public function __construct($module, $id)
    {
        $cfg_mgr = $module->node()->configuration_manager();
        $log_dir = $cfg_mgr->directory('log');
        $file = $log_dir.'/'.$module->name().'.log';
        parent::__construct($module, $file, 'ERROR', $id);
    }
}
class Safe_audit_point_system_log extends Safe_audit_point_log_content
{
    public function __construct($module, $id)
    {
        $cfg_mgr = $module->node()->configuration_manager();
        $file = '/var/log/messages';
        parent::__construct($module, $file, 'ERROR', $id);
    }
}
/******************************************************************************
 *   Directory Audit
 ******************************************************************************/
class Safe_audit_point_directory_timestamp extends Safe_audit_point
{
    protected $_directory='';
    protected function _field_value_def($configurable)
    {
        // No parameter
    }
    public function __construct($module, $type, $directory, $description=null, $category=null)
    {
        parent::__construct($module, Safe_audit_point::TYPE_DIRECTORY, $type);
        if($description){
            $this->set_description($description);
        }else{
            $this->set_description(ucfirst($type).' Directory');
        }
        if($category){
            $this->set_category($category);
        }
        $this->_directory = $directory;
    }
    protected function _monit_check()
    {
        return 'CHECK DIRECTORY '.$this->id().' WITH PATH '.$this->_directory;
    }
    protected function _monit_condition()
    {
        $cfg[] = '  IF CHANGED TIMESTAMP THEN alert';
        return $cfg;
    }
    public function set_parameters($parameters)
    {
        if($parameters['arg']['directory']){
            if($parameters['arg']['directory']['value']){
                $this->_directory=$parameters['arg']['directory']['value'];
            }
        }
        parent::set_parameters($parameters);
    }
}
class Safe_audit_point_directory extends Safe_audit_point_directory_timestamp
{
    public function __construct($module, $id)
    {
        $dir = '/';
        parent::__construct($module, $id, $dir);
    }
}
class Safe_audit_point_application_directory extends Safe_audit_point_directory_timestamp
{
    public function __construct($module, $id)
    {
        $cfg_mgr = $module->node()->configuration_manager();
        $dir = $cfg_mgr->directory($id);
        parent::__construct($module, $id, $dir);
    }
}
/******************************************************************************
 *   File System Audit
 ******************************************************************************/
class Safe_audit_point_filesystem extends Safe_audit_point
{
    private $_path;
    public function __construct($module, $id='root', $path='/')
    {
        $type = Safe_audit_point::TYPE_FILESYSTEM;
        parent::__construct($module, $type, $id);
        $this->set_category('System');
        $this->set_description('File System Usage % ('.$id.')');
        $this->_path=$path;
    }
    public function set_parameters($parameters)
    {
        if($parameters['arg']['path']){
            if($parameters['arg']['path']['value']){
                $this->_path=$parameters['arg']['path']['value'];
            }
        }
        parent::set_parameters($parameters);
    }
    protected function _monit_check()
    {
        return 'CHECK FILESYSTEM '.$this->id().' WITH PATH '.$this->_path;
    }
    protected function _monit_condition()
    {
        if($this->_value){
            $cfg[] = '  IF space usage > '.$this->_value.'% THEN alert';
        }
        return $cfg;
    }
}

/******************************************************************************
 *   Process Audit
 ******************************************************************************/
class Safe_audit_point_process extends Safe_audit_point
{
    protected $_pid_file = '';

    public function __construct($module, $type=Safe_audit_point::TYPE_PROCESS)
    {
        parent::__construct($module, $type);
        $this->set_description($module->description());
        $this->set_category(ucfirst($type));
        $this->_pid_file = $module->pid_file();
        // Create process check fields
        $this->_value_fields = array(
            'memory' => array(
                        'label' => 'Memory %',
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'Memory Usage threshold for ' . $this->description() . '.'
                    ),
            'cpu' => array(
                        'label' => 'CPU %',
                        'type'  => 'numeric',
                        'size'  => '5',
                        'default' => '50',
                        'help' => 'CPU Usage threshold for ' . $this->description() . '.'
                    ),
            'restart' => array(
                        'label' => 'Auto Restart',
                        'type'  => 'flag',
                        'default' => 'false',
                    ),
                );
    }
    protected function _monit_check()
    {
        return 'CHECK PROCESS '.$this->id().' WITH PIDFILE '.$this->_pid_file;
    }
    protected function _monit_condition()
    {
        $cfg = parent::_monit_condition();
        if($this->is_enabled()){
                $cfg[] = '  IF CHANGED PID THEN alert';
        }

        $allowed_key = array('memory', 'cpu');
        foreach($this->values() as $k => $v){
            if(in_array($k, $allowed_key)){
               if($v){
                    $cfg[] = '  IF total'.$k.' > '.$v.'% THEN alert';
               }
            }else{
                error_log('WARNING - Invalid Audit point parameter '.$k.' for '.$this->name().' skipping.');
            }
        }
        return $cfg;
    }
}
class Safe_audit_point_service extends Safe_audit_point_process
{
    public function __construct($module)
    {
        parent::__construct($module, Safe_audit_point::TYPE_SERVICE);
    }
    protected function _monit_condition()
    {
        $cfg = parent::_monit_condition();
        if($this->is_enabled()){
            $values = $this->values();
            if($values['restart'] === 'true'){
                $ctl_cfg[] = '  start = "/sbin/service '.$this->_module->process_name().' start"';
                $ctl_cfg[] = '  stop = "/sbin/service '.$this->_module->process_name().' stop"';
                $cfg = array_merge($ctl_cfg, $cfg);
            }
        }
        return $cfg;
    }
}

/******************************************************************************
 *   Program Audit
 ******************************************************************************/
class Safe_audit_point_program extends Safe_audit_point
{
    protected $_program;
    protected $_args = array();
    protected $_field_rules = 'numeric|greater_or_equal[0]|less_or_equal[100]';
    public function __construct($module, $type, $id='', $program='', $description=null, $category=null )
    {
        parent::__construct($module, $type, $id);
        if($description){
            $this->set_description($description);
        }
        if($category){
            $this->set_category($category);
        }
        $this->_program = $program;
    }
    public function is_enabled()
    {
        // Always enable so we gather stats
        return true;
    }
    protected function _program_args()
    {
        $args = '';
        foreach($this->values() as $k => $v){
            if('' != $v){
                $args .= ' --'.$k.'='.$v;
            }
        }
        foreach($this->_args as $k => $v){
            if('' != $v){
                $args .= ' --'.$k.'='.$v;
            }
        }
        if($args){
            return $args;
        }else{
            return $this->_value;
        }
    }
    public function set_parameters($parameters)
    {
        foreach($parameters['arg'] as $k => $v){
            if('program' == $k){
                if($v['value']){
                    $this->_program=$v['value'];
                }
            }else{
                $this->_args[$k] = $v['value'];
            }
        }
        parent::set_parameters($parameters);
    }
    protected function _monit_check()
    {
        // All program MUST live in /usr/local/sng/monitor/programs directory 
        $program = "/usr/local/sng/monitor/programs/".$this->_program;
        $check = 'CHECK PROGRAM '.$this->id().' WITH PATH '.$program;
        $args = $this->_program_args();
        if($args){
            $check .= ' ARGS "'.$args.'"';
        }
        return $check;
    }
    protected function _monit_condition()
    {
        $action = ('false' === $this->_action)?'alert':$this->_action;
        $cfg[] = '  IF status != 0 THEN alert';
        return $cfg;
    }
}

/* End of file safe_auditor_class.php */
