#!/usr/bin/python2.7
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab
"""
* Copyright (C) 2013  Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* Tyler Goodlet <tgoodlet@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.

Configurator ADT utils with parts borrowed from pycopia.jsonconfig
as well as the 'bunch' pipy (2.7) module

TODO:
    - context manager for writing multiple values to a component in
      a single block
    - consider using a collections.namedtuple instead to save aux space?
    - change __repr__ on HygenicBunch to be what dict has
"""
import sys, copy, os
from _abcoll import Mapping
from collections import namedtuple

from bunch import Bunch, json, OrderedDict, yaml
from yaml_loaders import OrderedDictYAMLLoader

from backends import logger

# for python 2.x and 3.x compatibility
if sys.version_info.major == 3:
    unicode = str

def dirtify(inst):
    '''
    mark us as dirty
    alert our ancestors that we are dirty
    '''
    inst._dirty = True
    try:
        # if parent already marked dirty then stop
        if not inst._parent._dirty:
            dirtify(inst._parent)
    except AttributeError:
        pass

def trim_empties(inst, stop_key=None):
    if inst._key == stop_key:
        return
    try:
        parent = inst._parent
        if len(inst) is 0:
            parent._del_item(inst._key)
            trim_empties(parent)
        else:
            return
    except AttributeError:
        pass

def key_path(inst, path=None):
    '''
    return a tuple of keys to
    this node starting from the
    root of the tree
    '''
    if path is None:
        path = [inst.key]
    else:
        path.append(inst.key)
    try:
        parent = getattr(inst, 'parent')
        return key_path(parent, path=path)
    except KeyError:
        # path.reverse()
        # return path[:-1] # don't include the root
        return tuple(path[:-1]) # don't include the root

def get_from_path(tree, path):
    if not isinstance(path, list):
        path = list(path)
    try:
        key = path.pop()
    except IndexError:
        # nothing left to pop...
        return tree
    return get_from_path(tree._get_item(key), path)
    # return get_from_path(tree[key], path)

def sanitize(inst):
    '''
    mark all dirty hygenic bunches from here down as clean
    '''
    inst.__dict__['_dirty']    = False
    inst.__dict__['_modified'] = False
    for item in inst.iterdirty():
        sanitize(item)

def is_dirty(inst):
    return inst._dirty

def depth_first_traversal(tree, iter_attr='itertrees'):
    '''generate tree elements using depth-first traversal'''
    if isinstance(iter_attr, str):
        itr = getattr(tree, iter_attr)
        try: itr = itr()
        except TypeError: pass
    # elif isinstance(iter_attr, list):
    #     itr = lambda : tree._get_item(iter_attr.pop())#[-1:])
    # else:
    #     raise StopIteration

    yield tree
    # for element in getattr(tree, iter_attr)():
    for element in itr:
        # yield element
        for child in depth_first_traversal(element, iter_attr):
            yield child
dft = depth_first_traversal

def gen_dirty(tree):
    '''
    traverse to the tree(s) with dirty nodes only
    tree nodes can be checked for modification by
    looking at the tree._modified attribute
    '''
    return dft(tree, iter_attr='iterdirty')

def zipped_traversal(target, follower, iter_attr='itertrees'):
    '''
    perform a dft over two similar trees and provide
    the output pairs at each iteration

    If the follower does not contain an element None will
    be yielded instead and no more depth steps will be taken
    if it is still none after re-entry to the generator.
    (useful for generating follower nodes in the caller)
    '''
    if isinstance(iter_attr, tuple):
        target_attr, follower_attr = iter_attr

    for element in getattr(target, iter_attr)():
        try:
            efollower = follower._get_item(element._key)
        except KeyError:
            logger.debug("got key error in traversor for key = "+element.key)
            efollower = element.__class__()
        yield element, efollower
        # if efollower and element:
        for child, child_follower in zipped_traversal(element, efollower, iter_attr):
            yield child, child_follower

def zip_dirty(tree, follower):
    return zipped_traversal(tree, follower, iter_attr='iterdirty')

# TODO: what happens if we have
def diff_traversal(current, target, keys_attr='treekeys'):
    '''
    traverse to the tree(s) with differing nodes only,
    using the target tree to determine which nodes must
    be yielded
    '''
    # if target == current: return ()
    # here added/removed are lists of treekeys wrt the
    # current tree node and leavesdiff is a bool indicating
    # a difference in leaf nodes
    Diff = namedtuple('Diff', 'added removed leavesdiff')
    def key_gather(tree):
        return set(getattr(tree, keys_attr))

    def differ(current, target):
        # try:
        # build sets of tree-like node keys
        target_keys  = set(target.treekeys)
        try:
            current_keys = set(current.treekeys)
        except AttributeError:
            current_keys = set()

        # current_keys = key_gather(current)
        # union        = target_keys | current_keys
        intersect    = target_keys & current_keys
        added        = list(target_keys - intersect)
        removed      = list(current_keys - intersect)

        leavesdiff = []
        # for key in intersect:
        # for key in target_keys:
        for key in target_keys:
            tl = target._get_item(key).leavesDict()
            try:
                cl = current._get_item(key).leavesDict()
            except KeyError:
                continue

            if cl != tl:
                # only deliver the diff
                for name,val in tl.items():
                    try:
                        if tl[name] == cl[name]:
                            tl.pop(name)
                    except KeyError:
                        continue

                # tl = {n:v for n,v in tl.items() if tl[n] != cl[n]}
                if len(tl) > 0:
                    leavesdiff.append((key, tl))

        # gather immediate leaves
        # if current.leavesDict() != target.leavesDict():
        #     leavesdiff.append( (None, target.leavesDict()) )
        # leavesdiff = current.leavesDict() != target.leavesDict()

        yield current, target, Diff(added, removed, leavesdiff)

        # for key in intersect:
        # get keys again since caller may have created some
        # target_keys  = set(target.treekeys)
        # use treekeys so items come in order (i.e. OrderedDict beneath)
        for key in target.treekeys:
        # for key in target_keys:
            ncurrent = current._get_item(key)
            ntarget = target._get_item(key)
            logger.debug('in traverser key = '+key)#+' with ncurrent = '+str(ncurrent))
            # try: ncurrent = current._get_item(key)
            # except KeyError: ncurrent = None
            # try: ntarget = target._get_item(key)
            # except KeyError: ntarget = None

            # if nodes are equal go to the next breadth step
            if ntarget == ncurrent:
                logger.debug("skipping nodes are the same...")
                continue
            else:
                # if ntarget and ncurrent:
                for child_target, child_follower, diff in differ(ncurrent, ntarget):
                    yield child_target, child_follower, diff
        # except AttributeError:
        #     pass
    # return the generator
    return differ(current, target)


class HygenicBunch(Bunch):
    '''
    A bunch who reports when he's dirty and
    when an element has been modified.

    This is a bare bones implementation of an abstract/general tree
    which uses recursive depth-first traversal to acquire dirty
    components and provide copies of subtrees.

    Here an element is any non-HygenicBunch and
    trees are any sub-HygenicBunch of the root.

    Use this class to house a stateful software configuration.

    Operation:
        - containers will be marked dirty if a value is re-set
          (it previously existed and was changed via setitem) with
          all parent containers being marked dirty up to the root
    '''
    def __init__(self, *args, **kwargs):
        rent                       = kwargs.pop('parent', None)
        self.__dict__['_key']      = kwargs.pop('key', None)
        self.__dict__['_value']    = kwargs.pop('value', None)
        self.__dict__['_modified'] = kwargs.pop('modified', False)

        Bunch.__init__(self, *args, **kwargs)
        self.__dict__["_dirty"]    = False
        if isinstance(rent, self.__class__):
            self.__dict__['_parent'] = rent

    def __dir__(self):
        '''include key values in interactive prompt output'''
        # pass
        listing = self.keys() + ['contents']
        # return self.keys()
        return listing

    def __deepcopy__(self, memo):
        cls = self.__class__
        result = cls.__new__(cls)
        memo[id(self)] = result
        for k, v in self.__dict__.items():
            if '_value' in k:
                result.__dict__[k] = self.__dict__['_value']
            else:
                result.__dict__[k] = copy.deepcopy(v, memo)

        for k, v in self.items():
            result._set_item(k, copy.deepcopy(v, memo))
        return result

    def dirty_copy(self):
        '''return a copy of the dirty tree'''
        pass

    # def update(self, *args, **kwargs):
    def update(self, other, **kwds):
        '''if we update mark us dirty and modified'''
        self.__dict__['_modified'] = True
        dirtify(self)
        # super(self.__class__, self).update(*args, **kwargs)
        # Bunch.update(self, *args, **kwargs)

        if isinstance(other, Mapping):
            for key in other:
                self._set_item(key, other[key])
        elif hasattr(other, "keys"):
            for key in other.keys():
                self._set_item(key, other[key])
        else:
            for key, value in other:
                self._set_item(key, value)

        for key, value in kwds.items():
            self._set_item(key, value)

    def check_tree(self):
        '''
        closure which will verify if a provided element
        is a subtree (i.e. the same class) as the initial instance
        '''
        def is_tree(element):
            return isinstance(element, self.__class__)
        return is_tree

    is_tree = property(check_tree)

    def iterleaves(self):
        for name, value in self.iteritems():
            if not self.is_tree(value):
                yield name, value
    # @property
    def leavesDict(self):
        # return [(key,leaf) for key, leaf in self.iterleaves()]
        return {key:leaf for key, leaf in self.iterleaves()}

    def itertrees(self):
        # return (node for node in filter(self.is_subtree, self.itervalues()))
        for tree in filter(self.is_tree, self.itervalues()):
            yield tree
        # return filter(lambda item : isinstance(item, self.__class__), self.itervalues())

    @property
    def treekeys(self):
        return [e.key for e in self.itertrees()]

    @property
    def subtrees(self):
        return [e for e in self.itertrees()]

    def iterdirty(self):
        return (node for node in filter(is_dirty, self.itertrees()))
        # for node in filter(is_dirty, self.subtrees)
        #     yield node
        # return filter(HygenicBunch.is_dirty, self.subtrees)

    @property
    def dirtyvalues(self):
        return [value for value in self.iterdirty()]

    @property
    def dirtykeys(self):
        return [value.key for value in self.iterdirty()]

    def set_value(self, value):
        self.__dict__['_value'] = value

    dirty    = property(lambda self : self._dirty)
    modified = property(lambda self : self.__dict__['_modified'])
    parent   = property(lambda self : self.__dict__['_parent'])
    key      = property(lambda self : self.__dict__['_key'])
    value    = property(lambda self : self.__dict__['_value'], set_value)

    @property
    def contents(self, key=None):
        '''
        show our tree contents in a pleasant way
        using a PyYaml representer
        '''
        print(self.toYAML(explicit_start=True))

    def copy(self):
        return bunchify(Bunch.copy(self))

    def leavescopy(self):
        return HygenicBunch(self.iterleaves())

    def __getitem__(self, key):
        return self._get_item(key)

    def _get_item(self, key):
        return Bunch.__getitem__(self, key)

    def __setitem__(self, key, value):
        return self._set_item(key, value)

    def _set_item(self, key, value):
        if '_' in key[:1]:
            return super(self.__class__, self).__setitem__(key, value)
        # if not self.has_key(key):
        try:
            # print('calling Bunch __getitem__ from _set_item')
            Bunch.__getitem__(self, key)
        except KeyError:
            # if key dne then don't mark us dirty
            # print('set item on new item : '+str(key))
            # if inserting sub-bunch, assign the parent
            if isinstance(value, self.__class__):
                value._parent = self
            return Bunch.__setitem__(self, key, value)

        # if we're actually rewriting a value then we're now dirty
        # and must indicate this to our parents
        self.__dict__['_modified'] = True
        dirtify(self)
        # print('set item on old item : '+str(key))
        return Bunch.__setitem__(self, key, value)

    def __delitem__(self, name):
        self._del_item(name)

    def _del_item(self, name):
        # return super(self.__class__, self).__delitem__(name)
        Bunch.__delitem__(self, name)
        self.__dict__['_modified'] = True
        dirtify(self)


class AutoHBunch(HygenicBunch):
    '''
    a hygenic bunch with automatic container node creation

    Operation:
        - use add_subcontainer to build a new sub-tree (or bunchify)
    '''
    def add_subcontainer(self, key, value=None, parent=None):
        d = type(self)(parent=self, key=key)
        Bunch.__setitem__(self, key, d)
        return d

    def __getitem__(self, key):
        if '_' in key:
            return super(self.__class__, self).__getitem__(key)
        try:
            return super(self.__class__, self).__getitem__(key)
        except KeyError:
            # print('get item error : '+str(key))
            hb = self.add_subcontainer(key)
            hb.__dict__["_dirty"] = False
            # hb._dirty = False
            return hb
#try:
# Attempt to register the HygenicBunch with PyYAML as a representer
# import yaml
# from yaml.representer import Representer, SafeRepresenter

# def hygenic_display(dumper, data):
#     """
#     Converts HygenicBunch to a representation node for user visualization.
#     Tree nodes who report dirty are displayed with a '(modified)' flag-suffix.
#     """
#     suffix = 'TREE'
#     if data._modified:
#         suffix = 'TREE-MODIFIED'
#     elif data._dirty:
#         suffix = 'TREE-DIRTY'
#     return dumper.represent_mapping(u'!'+suffix, data)

# SafeRepresenter.add_representer(HygenicBunch, hygenic_display)
# SafeRepresenter.add_multi_representer(HygenicBunch, hygenic_display)
# Representer.add_representer(HygenicBunch, hygenic_display)
# Representer.add_multi_representer(HygenicBunch, hygenic_display)

# except ImportError:
#     pass

def bunchify(x, parent=None, key=None, value=None, auto=False, cls=HygenicBunch):
    '''
    Turn sequences/maps into hygenic bunches
    taken and extended from the bunch module

    Inputs 'key' and 'value' are only assigned on the root
    '''
    def bunchifier(x, parent=None, key=None, value=None, auto=False):
        if isinstance(x, dict):
            hb = cls(parent=parent, key=key, value=value)
            for k,v in x.iteritems():
                # hb[k] = bunchifier(v, parent=hb, key=k, value=None)
                # print("setting k = "+str(k)+" v = "+str(v))
                # pdb.set_trace()
                # Bunch.__setitem__(hb, k, bunchifier(v, parent=hb, key=k, value=None))
                HygenicBunch._set_item(hb, k, bunchifier(v, parent=hb, key=k, value=None))
            return hb
        elif isinstance(x, (list, tuple)):
            return type(x)( bunchifier(v) for v in x )
        else:
            return x
    return bunchifier(x, parent=parent, key=key, value=value, auto=auto)

def get_component(obj, name):
    '''
    try both getattr and getitem methods
    to acquire a component from obj
    '''
    try:
        curr_eng_ref = getattr(obj, name)
    except AttributeError:
        curr_eng_ref = obj[name]
    return curr_eng_ref

def read_config(path_or_file):
    """
    Read a config file.
    taken from pycopia.jsonconfig
    """
    if isinstance(path_or_file, (str, unicode)):
        fp = open(path_or_file, "r")
        doclose = True
    else:
        fp = path_or_file
        doclose = False

    # load by format
    root, ext = os.path.splitext(path_or_file)
    if 'json' in ext:
        d = json.load(fp, encoding="utf-8")

    elif 'yaml' in ext:
        d = yaml.load(fp, Loader=OrderedDictYAMLLoader)

    if doclose:
        fp.close()
    return bunchify(d)
    # return d

def write_config(conf, path_or_file, format='yaml'):
    """
    Write a JSON config file.
    adapted from from pycopia.jsonconfig
    """
    if isinstance(path_or_file, (str, unicode)):
        fp = open(path_or_file, "w+")
        doclose = True
    else:
        fp = path_or_file
        doclose = False
    # dump to file using the format of choice
    if 'json' in format:
        json.dump(conf, fp, indent=2)
    elif 'yaml' in format:
        conf.toYAML(stream=fp)
    if doclose:
        fp.close()

def get_config(filename=None, init=None, auto=False):
    """
    Get an existing or new json config object.
    Optionally initialize from another dictionary.

    taken and modified from pycopia.jsonconfig
    """
    if init is not None:
        return bunchify(init)
    if filename is None:
        if auto:
            return AutoHBunch()
        else:
            return HygenicBunch()

    if os.path.exists(filename):
        return read_config(filename)
    else:
        d = HygenicBunch()
        write_config(d, filename)
        return d

if __name__ == '__main__':
    #test basic modifications
    cf = get_config(auto=True)
    cf.parts.program.flags.flagname = 2
    cf.parts.program.path = "$BASE/program"
    cf.parts.BASE = "bin"
    assert cf.parts.program.flags.flagname == 2
    assert cf.parts.program.path == "$BASE/program"
    print('writing config...')
    write_config(cf, "/tmp/testjson.json")
    del cf
    print('reading config...')
    cf = read_config("/tmp/testjson.json")
    print('finished reading...')
    assert type(cf) is HygenicBunch
    assert cf.parts.program.flags.flagname == 2
    assert cf.parts.program.path == "$BASE/program"
    assert cf._dirty == False
    cf.parts.program.flags.flagname = 3
    assert cf.parts.program.flags.flagname == 3
    assert cf._dirty == True
    write_config(cf, "/tmp/testjson.json")
    sanitize(cf)
    assert cf._dirty == False
    del cf
    cf = read_config("/tmp/testjson.json")
    assert cf.parts.program.flags.flagname == 3
    assert cf._dirty == False
    del cf.parts.program.flags.flagname
    assert len(cf.parts.program.flags) == 0
    assert len(cf.parts.program["flags"]) == 0
    assert cf._dirty == True
    assert cf.parts.program.flags is cf.parts.program["flags"]
    del cf

    # safepy testing
    import nsc_master
    s = nsc_master.Product()
    s.connect('nsc-atom-master', 80)
    cf = bunchify(s.config(), key=s.name)
    cf.system.user.root.email = 'doggy'
    # cf.sip.profile['NSC_SIP_PROFILE1']['sip-port'] = 234098
    assert cf._dirty
