#!/usr/bin/env python
# vim: tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab
"""
* Copyright (C) 2011   Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* William Adam <william.adam@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.
"""

#Include this directory to  python sys.path
import paths

import sys
import os
import subprocess
import re
import xml.dom.minidom as xml
import fs_helper

from optparse import OptionParser
import logging
import logging.handlers

# Usage:
#   fs_voice_quality --jitter=[Jitter limit] --loss=[Packet Loss Limit]
#   --rtt=[Max RTT]
#   All are optional, if none specified no check performed
# 
# Output format: (This tool is for monit processing)- NO pretty XML print (use
# inline)
"""
<program>  
    <name>Voice Quality</name>
    <message>Output message string </message>
    <status></status>
    <data>
        <count>xx</count>
        <jitter>xx</jitter>
        <loss>xx</loss>
        <rtt>xx</rtt>
    </data>
</program>
"""
# Upon success, rc=0, data section populated, message set to:
#    OK
#
# Upon limit in percent exceeded, returns -1 and message set to:
#    Active sessions amount of xx% matches resource limit [active sessions>yy%]
#  Where xx is limit specified as argument and yy is current active sessions in %
#
# Upon error, rc=0 (yes I know :) - NO data section - and message set to (fs_cli
# output) example:
#    [ERROR] fs_cli.c:1305 main() Error Connecting [Socket Connection Error]
"""
mysql> 
SELECT COUNT(*) AS total, 
       COUNT(CASE WHEN average_jitter > 0 THEN 1 ELSE NULL END) AS jitter, 
       COUNT(CASE WHEN pkt_loss_percent > 0 THEN 1 ELSE NULL END) AS loss, 
       COUNT(CASE WHEN average_round_trip_time>0 THEN 1 ELSE NULL END) AS rtt 
  FROM (
    SELECT average_inter_arrival_jitter * 1000 / sampling_rate AS average_jitter,
           (cumulative_lost_cnt/(sent_pkt_cnt + 0.000001)) * 100 AS pkt_loss_percent,
           average_round_trip_time  
        FROM rtcpmon_reports 
          WHERE event_time >= DATE_SUB( NOW(), INTERVAL 60 SECOND)
          HAVING 
             average_jitter > 0 
          OR pkt_loss_percent > 0 
          OR average_round_trip_time > 0
       ) AS report;

+-------+--------+------+-----+
| total | jitter | loss | rtt |
+-------+--------+------+-----+
|   307 |    131 |    0 | 176 |
+-------+--------+------+-----+

"""    
def do_check(logger, options):
    # Output variables
    error=None
    stats = {}
    rc = 0

    # Build SQL query
    select = ["COUNT(*) AS total"]
    subselect = []
    having = []
    limits = {}
    where = []

    if options.jitter:
        select.append("COUNT(CASE WHEN average_jitter > 0 THEN 1 ELSE NULL END) AS jitter")
        subselect.append("average_inter_arrival_jitter * 1000 / sampling_rate AS average_jitter")
        having.append("average_jitter > " + str(options.jitter))
        limits['jitter'] = str(options.jitter)

    if options.loss:
        select.append("COUNT(CASE WHEN pkt_loss_percent > 0 THEN 1 ELSE NULL END) AS loss")
        subselect.append("(cumulative_lost_cnt/(sent_pkt_cnt + 0.000001)) * 100 AS pkt_loss_percent")
        having.append("pkt_loss_percent > " + str(options.loss))
        limits['loss'] = str(options.loss)

    if options.rtt:
        select.append("COUNT(CASE WHEN average_round_trip_time>0 THEN 1 ELSE NULL END) AS rtt")
        subselect.append("average_round_trip_time")
        having.append("average_round_trip_time > " + str(options.rtt))
        limits['rtt'] = str(options.rtt)

    if options.time:
        where.append(" WHERE event_time >= DATE_SUB( NOW(), INTERVAL " + \
                str(options.time) + " SECOND)" )

    # Valid query ?
    if not len(having):
        out = ['No limit to check']
        error = 2
    else:
        query = "SELECT " + ','.join(select) + " FROM (SELECT " + ','.join(subselect) + \
            " FROM rtcpmon_reports " + ''.join(where) + " HAVING " + ' OR '.join(having) + ") AS report;"

        # Excute command
        cmd = ['mysql', 'nsc', '--raw', '-e', query]

        error, out = fs_helper.os_execute(cmd)

    # Create output
    doc = xml.Document()
    # inner root
    root = doc.createElement('program');
    s = doc.createElement('name')
    st = doc.createTextNode('Call Quality')
    s.appendChild(st)
    root.appendChild(s)


    if not error:
        # resuffle output to a dictionary
        col = out[0].split('\t')
        res = out[1].split('\t')
        for i, r in enumerate(col):
            stats[r] = res[i]
            if int(res[i]):
                rc = -1
    else:
        error = ' '.join(out)

    data = doc.createElement('data')

    # process resource(s) status
    if stats:

        error = 'Call Quality '
        if rc:
            error = error + 'limit match '

        for k, v in stats.iteritems():
            if 'total' != k:
                error = error + k + "[count=" + v + ", limit=" + limits[k] + "] "
            s = doc.createElement(k)
            st = doc.createTextNode(v)
            s.appendChild(st)
            data.appendChild(s)

    # Populate message element
    s = doc.createElement('message')
    st = doc.createTextNode(str(error))
    s.appendChild(st)
    root.appendChild(s)
    if data:
        root.appendChild(data)

    # output result
    return rc, root


def main():
    #Parse the options
    usage = "usage: %prog [options] arg"
    optParser = OptionParser(usage)

    optParser.add_option('-j', '--jitter', action='store', type='int',
                        dest='jitter', metavar='JITTER',
                        help="Max Average Jitter (in milliseconds)")

    optParser.add_option('-l', '--loss', action='store', type='float',
                        dest='loss', metavar='LOSS',
                        help="Max Packet Lost Percentage")

    optParser.add_option('-r', '--rtt', action='store', type='int',
                        dest='rtt', metavar='RTT',
                        help="Max Average Round Trip Time (in milliseconds)")

    optParser.add_option('-t', '--time', action='store', type='int',
                        dest='time', metavar='TIME',
                        help="Time Period in seconds")

    optParser.add_option('-p', '--pretty', action='store_true',
                        dest='pretty', metavar='PRETTY',
                        help="Pretty print output")

    (options, args) = optParser.parse_args()

    # Create logger
    logger = logging.getLogger(os.path.basename(os.path.abspath(sys.argv[0])))
    logger.setLevel(logging.INFO)
    handler = logging.StreamHandler()
    logger.addHandler(handler)

    # Execute Check
    rc, root = do_check(logger, options)
    # Check output
    if options.pretty:
        logger.info(root.toprettyxml())
    else:
        logger.info(root.toxml())
    # Report error/success
    sys.exit(rc)


if __name__ == '__main__':
  main()
