<?php
if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
 * CodeIgniter
 *
 * An open source application development framework for PHP 5.1.6 or newer
 *
 * @package		CodeIgniter
 * @author		ExpressionEngine Dev Team
 * @copyright	Copyright (c) 2008 - 2011, EllisLab, Inc.
 * @license		http://codeigniter.com/user_guide/license.html
 * @link		http://codeigniter.com
 * @since		Version 1.0
 * @filesource
 */

// ------------------------------------------------------------------------

/**
 * Form Validation Class
 *
 * @package		CodeIgniter
 * @subpackage	Libraries
 * @category	Validation
 * @author		ExpressionEngine Dev Team
 * @link		http://codeigniter.com/user_guide/libraries/form_validation.html
 */
class CI_Form_validation {

	protected $CI;
	protected $_field_data			= array();
	protected $_config_rules		= array();
	protected $_error_array			= array();
	protected $_error_messages		= array();
	protected $_error_prefix		= '<p>';
	protected $_error_suffix		= '</p>';
	protected $error_string			= '';
	protected $_safe_form_data		= FALSE;
	protected $_validation_obj      = null;// for validation  callback 
    
	public function set_validation_obj($obj){
	    //if the validation have a call back rule, check the callback function in this obj
	    $this->_validation_obj = $obj;
	}
	public function run_rule_callback($method_name, $field_name, $param, $postdata) {
	    if(method_exists($this -> _validation_obj,$method_name)){
	        $result = $this -> _validation_obj -> $method_name($field_name, $param, $postdata);
	        if($result === true){
	            return $result;
	        }else{
	            //$this->safe_form_class->set_message($result);
	            foreach($result as $key => $message){
	                $this->set_message($key, $message);
	            }
	            return false;
	        }
	    }else{
	        $this->set_message($method_name , $method_name . ' not found! ');
	        return false;
	    }
	}
	/**
	 * Constructor
	 */
	public function __construct($rules = array())
	{
		$this->CI =& get_instance();

		// Validation rules can be stored in a config file.
		$this->_config_rules = $rules;

		// Automatically load the form helper
		$this->CI->load->helper('form');

		// Set the character encoding in MB.
		if (function_exists('mb_internal_encoding'))
		{
			mb_internal_encoding($this->CI->config->item('charset'));
		}

		log_message('debug', "Form Validation Class Initialized");
	}

	// --------------------------------------------------------------------
	public function clear_rules()
	{
	    $this->_field_data = null;
	}
	/**
	 * Set Rules
	 *
	 * This function takes an array of field names and validation
	 * rules as input, validates the info, and stores it
	 *
	 * @access	public
	 * @param	mixed
	 * @param	string
	 * @return	void
	 */
	public function set_rules($field, $label = '', $rules = '')
	{
		// No reason to set rules if we have no POST data
		if (count($_POST) == 0)
		{
			return $this;
		}

		// If an array was passed via the first parameter instead of indidual string
		// values we cycle through it and recursively call this function.
		if (is_array($field))
		{
			foreach ($field as $row)
			{
				// Houston, we have a problem...
				if ( ! isset($row['field']) OR ! isset($row['rules']))
				{
					continue;
				}

				// If the field label wasn't passed we use the field name
				$label = ( ! isset($row['label'])) ? $row['field'] : $row['label'];

				// Here we go!
				$this->set_rules($row['field'], $label, $row['rules']);
			}
			return $this;
		}

		// No fields? Nothing to do...
		if ( ! is_string($field) OR  ! is_string($rules) OR $field == '')
		{
			return $this;
		}

		// If the field label wasn't passed we use the field name
		$label = ($label == '') ? $field : $label;

		// Is the field name an array?  We test for the existence of a bracket "[" in
		// the field name to determine this.  If it is an array, we break it apart
		// into its components so that we can fetch the corresponding POST data later
		if (strpos($field, '[') !== FALSE AND preg_match_all('/\[(.*?)\]/', $field, $matches))
		{
			// Note: Due to a bug in current() that affects some versions
			// of PHP we can not pass function call directly into it
			$x = explode('[', $field);
			$indexes[] = current($x);

			for ($i = 0; $i < count($matches['0']); $i++)
			{
				if ($matches['1'][$i] != '')
				{
					$indexes[] = $matches['1'][$i];
				}
			}

			$is_array = TRUE;
		}
		else
		{
			$indexes	= array();
			$is_array	= FALSE;
		}

		// Build our master array
		$this->_field_data[$field] = array(
			'field'				=> $field,
			'label'				=> $label,
			'rules'				=> $rules,
			'is_array'			=> $is_array,
			'keys'				=> $indexes,
			'postdata'			=> NULL,
			'error'				=> ''
		);

		return $this;
	}

	// --------------------------------------------------------------------

	/**
	 * Set Error Message
	 *
	 * Lets users set their own error messages on the fly.  Note:  The key
	 * name has to match the  function name that it corresponds to.
	 *
	 * @access	public
	 * @param	string
	 * @param	string
	 * @return	string
	 */
	public function set_message($lang, $val = '')
	{
	    if ( ! is_array($lang))
		{
			$lang = array($lang => $val);
		}

		$this->_error_messages = array_merge($this->_error_messages, $lang);

		return $this;
	}

	// --------------------------------------------------------------------

	/**
	 * Set The Error Delimiter
	 *
	 * Permits a prefix/suffix to be added to each error message
	 *
	 * @access	public
	 * @param	string
	 * @param	string
	 * @return	void
	 */
	public function set_error_delimiters($prefix = '<p>', $suffix = '</p>')
	{
		$this->_error_prefix = $prefix;
		$this->_error_suffix = $suffix;

		return $this;
	}

	// --------------------------------------------------------------------

	/**
	 * Get Error Message
	 *
	 * Gets the error message associated with a particular field
	 *
	 * @access	public
	 * @param	string	the field name
	 * @return	void
	 */
	public function error($field = '', $prefix = '', $suffix = '')
	{
		if ( ! isset($this->_field_data[$field]['error']) OR $this->_field_data[$field]['error'] == '')
		{
			return '';
		}

		if ($prefix == '')
		{
			$prefix = $this->_error_prefix;
		}

		if ($suffix == '')
		{
			$suffix = $this->_error_suffix;
		}

		return $prefix.$this->_field_data[$field]['error'].$suffix;
	}
	/**
	 * Set Error Message
	 *
	 * Set the error message associated with a particular field
	 *
	 * @access	public
	 * @param	string	the field name
	 * @param	string	the error message
	 * @return	void
	 */
	public function set_error($field = '', $message = '')
	{
	    if (isset($this->_field_data[$field])){
	        $this->_field_data[$field]['error'] = sprintf($message, $this->_field_data[$field]['label']);
	    }
	}

	// --------------------------------------------------------------------

	/**
	 * Error String
	 *
	 * Returns the error messages as a string, wrapped in the error delimiters
	 *
	 * @access	public
	 * @param	string
	 * @param	string
	 * @return	str
	 */
	public function error_string($prefix = '', $suffix = '')
	{
		// No errrors, validation passes!
		if (count($this->_error_array) === 0)
		{
			return '';
		}

		if ($prefix == '')
		{
			$prefix = $this->_error_prefix;
		}

		if ($suffix == '')
		{
			$suffix = $this->_error_suffix;
		}

		// Generate the error string
		$str = '';
		foreach ($this->_error_array as $val)
		{
			if ($val != '')
			{
				$str .= $prefix.$val.$suffix."\n";
			}
		}

		return $str;
	}

	// --------------------------------------------------------------------

	/**
	 * Run the Validator
	 *
	 * This function does all the work.
	 *
	 * @access	public
	 * @return	bool
	 */
	public function run($group = '')
	{
	    //clear form errors before run again
	    $this->_error_array = array();
	    // Do we even have any data to process?  Mm?
		if (count($_POST) == 0)
		{
			return FALSE;
		}

		// Does the _field_data array containing the validation rules exist?
		// If not, we look to see if they were assigned via a config file
		if (count($this->_field_data) == 0)
		{
			// No validation rules?  We're done...
			if (count($this->_config_rules) == 0)
			{
				return FALSE;
			}

			// Is there a validation rule for the particular URI being accessed?
			$uri = ($group == '') ? trim($this->CI->uri->ruri_string(), '/') : $group;

			if ($uri != '' AND isset($this->_config_rules[$uri]))
			{
				$this->set_rules($this->_config_rules[$uri]);
			}
			else
			{
				$this->set_rules($this->_config_rules);
			}

			// We're we able to set the rules correctly?
			if (count($this->_field_data) == 0)
			{
				log_message('debug', "Unable to find validation rules");
				return FALSE;
			}
		}

		// Load the language file containing error messages
		$this->CI->lang->load('form_validation');

		// Cycle through the rules for each field, match the
		// corresponding $_POST item and test for errors
		foreach ($this->_field_data as $field => $row)
		{
			// Fetch the data from the corresponding $_POST array and cache it in the _field_data array.
			// Depending on whether the field name is an array or a string will determine where we get it from.

			if ($row['is_array'] == TRUE)
			{
				$this->_field_data[$field]['postdata'] = $this->_reduce_array($_POST, $row['keys']);
			}
			else
			{
				if (isset($_POST[$field]) AND $_POST[$field] != "")
				{
					$this->_field_data[$field]['postdata'] = $_POST[$field];
				}else{
				    if (isset($_FILES[$field]) AND $_FILES[$field] != "")
				    {
				        $this->_field_data[$field]['postdata'] = $_FILES[$field];
				    }
				}
			}

      // Only invoke rule exectution if some rules are defined
      // Otherwise no rule will empty the field value
      if($row['rules']) {
        $this->_execute($row, explode('|', $row['rules']), $this->_field_data[$field]['postdata']);
      }
		}

		// Did we end up with any errors?
		$total_errors = count($this->_error_array);

		if ($total_errors > 0)
		{
			$this->_safe_form_data = TRUE;
		}

		// Now we need to re-set the POST data with the new, processed data
		$this->_reset_post_array();

		// No errors, validation passes!
		if ($total_errors == 0)
		{
			return TRUE;
		}

		// Validation fails
		return FALSE;
	}

	// --------------------------------------------------------------------

	/**
	 * Traverse a multidimensional $_POST array index until the data is found
	 *
	 * @access	private
	 * @param	array
	 * @param	array
	 * @param	integer
	 * @return	mixed
	 */
	protected function _reduce_array($array, $keys, $i = 0)
	{
		if (is_array($array))
		{
			if (isset($keys[$i]))
			{
				if (isset($array[$keys[$i]]))
				{
					$array = $this->_reduce_array($array[$keys[$i]], $keys, ($i+1));
				}
				else
				{
					return NULL;
				}
			}
			else
			{
				return $array;
			}
		}

		return $array;
	}

	// --------------------------------------------------------------------

	/**
	 * Re-populate the _POST array with our finalized and processed data
	 *
	 * @access	private
	 * @return	null
	 */
	protected function _reset_post_array()
	{
		foreach ($this->_field_data as $field => $row)
		{
			if ( ! is_null($row['postdata']))
			{
				if ($row['is_array'] == FALSE)
				{
					if (isset($_POST[$row['field']]))
					{
						$_POST[$row['field']] = $this->prep_for_form($row['postdata']);
					}
				}
				else
				{
					// start with a reference
					$post_ref =& $_POST;

					// before we assign values, make a reference to the right POST key
					if (count($row['keys']) == 1)
					{
						$post_ref =& $post_ref[current($row['keys'])];
					}
					else
					{
						foreach ($row['keys'] as $val)
						{
							$post_ref =& $post_ref[$val];
						}
					}

					if (is_array($row['postdata']))
					{
						$array = array();
						foreach ($row['postdata'] as $k => $v)
						{
							$array[$k] = $this->prep_for_form($v);
						}

						$post_ref = $array;
					}
					else
					{
						$post_ref = $this->prep_for_form($row['postdata']);
					}
				}
			}
		}
	}

	// --------------------------------------------------------------------

	/**
	 * Executes the Validation routines
	 *
	 * @access	private
	 * @param	array
	 * @param	array
	 * @param	mixed
	 * @param	integer
	 * @return	mixed
	 */
	protected function _execute($row, $rules, $postdata = NULL, $cycles = 0)
	{
	    // If the $_POST data is an array we will run a recursive call
		if (is_array($postdata))
		{
			foreach ($postdata as $key => $val)
			{
				$this->_execute($row, $rules, $val, $cycles);
				$cycles++;
			}

			return;
		}

		// --------------------------------------------------------------------

		// If the field is blank, but NOT required, no further tests are necessary
		$callback = FALSE;
    $conditionnal_required = FALSE;
    // Walk array to find required[field] (i.e. conditionnal required rule.
    foreach($rules as $k)
    {
      if(preg_match("/required\[/", $k)) {
        $conditionnal_required = TRUE;
        break;
      }
    }
		if ( ! in_array('required', $rules) AND is_null($postdata))
		{
			// Before we bail out, does the rule contain a callback?
			if (preg_match("/(callback_\w+(\[.*?\])?)/", implode(' ', $rules), $match))
			{
				$callback = TRUE;
				$rules = (array('1' => $match[1]));
			}
			elseif(!$conditionnal_required)
        // bail out if no conditionnal required
          return;
		}

		// --------------------------------------------------------------------

		// Isset Test. Typically this rule will only apply to checkboxes.
		if (is_null($postdata) AND $callback == FALSE AND !$conditionnal_required)
		{
			if (in_array('isset', $rules, TRUE) OR in_array('required', $rules))
			{
				// Set the message type
				$type = (in_array('required', $rules)) ? 'required' : 'isset';

				if ( ! isset($this->_error_messages[$type]))
				{
					if (FALSE === ($line = $this->CI->lang->line($type)))
					{
						$line = 'The field was not set';
					}
				}
				else
				{
					$line = $this->_error_messages[$type];
				}

				// Build the error message
				$message = sprintf($line, $this->_translate_fieldname($row['label']));

				// Save the error message
				$this->_field_data[$row['field']]['error'] = $message;

				if ( ! isset($this->_error_array[$row['field']]))
				{
					$this->_error_array[$row['field']] = $message;
				}
			}

			return;
		}
		// --------------------------------------------------------------------

		// Cycle through each rule and run it
		foreach ($rules As $rule)
		{
			$_in_array = FALSE;

			// We set the $postdata variable with the current data in our master array so that
			// each cycle of the loop is dealing with the processed data from the last cycle
			if ($row['is_array'] == TRUE AND is_array($this->_field_data[$row['field']]['postdata']))
			{
				// We shouldn't need this safety, but just in case there isn't an array index
				// associated with this cycle we'll bail out
				if ( ! isset($this->_field_data[$row['field']]['postdata'][$cycles]))
				{
					continue;
				}

				$postdata = $this->_field_data[$row['field']]['postdata'][$cycles];
				$_in_array = TRUE;
			}
			else
			{
				$postdata = $this->_field_data[$row['field']]['postdata'];
			}

			// --------------------------------------------------------------------

			// Is the rule a callback?
			$callback = FALSE;
			if (substr($rule, 0, 9) == 'callback_')
			{
			    $rule = substr($rule, 9);
				$callback = TRUE;
			}

			// Strip the parameter (if exists) from the rule
			// Rules can contain a parameter: max_length[5]
			$param = FALSE;
			if (preg_match("/(.*?)\[(.*)\]/", $rule, $match))
			{
				$rule	= $match[1];
				$param	= $match[2];
			}
			// Call the function that corresponds to the rule
			if ($callback === TRUE)
			{
			    if ( ! method_exists($this->CI, $rule))
				{
					//run call back rule in the obj
				    if( $this->_validation_obj != null && method_exists($this->_validation_obj, $rule) ){
					    $result = $this->run_rule_callback($rule,$row['field'], $param, $postdata);
					}else{
				        continue;
					}
				}else{
    				// Run the function and grab the result
	    			$result = $this->CI->$rule($postdata, $param);
				}

				// Re-assign the result to the master data array
				if ($_in_array == TRUE)
				{
					$this->_field_data[$row['field']]['postdata'][$cycles] = (is_bool($result)) ? $postdata : $result;
				}
				else
				{
					$this->_field_data[$row['field']]['postdata'] = (is_bool($result)) ? $postdata : $result;
				}

				// If the field isn't required and we just processed a callback we'll move on...
				if ( ! in_array('required', $rules, TRUE) AND $result !== FALSE)
				{
					continue;
				}
			}
			else
			{
				if ( ! method_exists($this, $rule))
				{
					// If our own wrapper function doesn't exist we see if a native PHP function does.
					// Users can use any native PHP function call that has one param.
					if (function_exists($rule))
					{
						$result = $rule($postdata);

						if ($_in_array == TRUE)
						{
							$this->_field_data[$row['field']]['postdata'][$cycles] = (is_bool($result)) ? $postdata : $result;
						}
						else
						{
							$this->_field_data[$row['field']]['postdata'] = (is_bool($result)) ? $postdata : $result;
						}
					}
					else
					{
						log_message('debug', "Unable to find validation rule: ".$rule);
					}

					//continue;
				}else{
                  $result = $this->$rule($postdata, $param);
                }
        // If conditionnal required is present and executed rule is required and
        // result is "SKIP" then bail out as condition not meet
        if($rule === "required" AND $conditionnal_required AND $result === "SKIP")
          return;
        elseif ($result === "SKIP")
          $result = TRUE;

				if ($_in_array == TRUE)
				{
					$this->_field_data[$row['field']]['postdata'][$cycles] = (is_bool($result)) ? $postdata : $result;
				}
				else
				{
					$this->_field_data[$row['field']]['postdata'] = (is_bool($result)) ? $postdata : $result;
				}
			}

			// Did the rule test negatively?  If so, grab the error.
			if ($result === FALSE)
			{
			    if ( ! isset($this->_error_messages[$rule]))
				{
					if (FALSE === ($line = $this->CI->lang->line($rule)))
					{
						$line = 'Unable to access an error message corresponding to your field name.';
					}
				}
				else
				{
					$line = $this->_error_messages[$rule];
				}

				// Is the parameter we are inserting into the error message the name
				// of another field?  If so we need to grab its "field label"
				if (isset($this->_field_data[$param]) AND isset($this->_field_data[$param]['label']))
				{
					$param = $this->_translate_fieldname($this->_field_data[$param]['label']);
				}

				// Build the error message
				$message = sprintf($line, $this->_translate_fieldname($row['label']), $param);

				// Save the error message
				$this->_field_data[$row['field']]['error'] = $message;

				if ( ! isset($this->_error_array[$row['field']]))
				{
					$this->_error_array[$row['field']] = $message;
				}

				return;
			}
		}
	}

	// --------------------------------------------------------------------

	/**
	 * Translate a field name
	 *
	 * @access	private
	 * @param	string	the field name
	 * @return	string
	 */
	protected function _translate_fieldname($fieldname)
	{
		// Do we need to translate the field name?
		// We look for the prefix lang: to determine this
		if (substr($fieldname, 0, 5) == 'lang:')
		{
			// Grab the variable
			$line = substr($fieldname, 5);

			// Were we able to translate the field name?  If not we use $line
			if (FALSE === ($fieldname = $this->CI->lang->line($line)))
			{
				return $line;
			}
		}

		return $fieldname;
	}

	// --------------------------------------------------------------------

	/**
	 * Get the value from a form
	 *
	 * Permits you to repopulate a form field with the value it was submitted
	 * with, or, if that value doesn't exist, with the default
	 *
	 * @access	public
	 * @param	string	the field name
	 * @param	string
	 * @return	void
	 */
	public function set_value($field = '', $default = '')
	{
		if ( ! isset($this->_field_data[$field]))
		{
			return $default;
		}

		// If the data is an array output them one at a time.
		//     E.g: form_input('name[]', set_value('name[]');
		if (is_array($this->_field_data[$field]['postdata']))
		{
			return array_shift($this->_field_data[$field]['postdata']);
		}
    if (NULL === $this->_field_data[$field]['postdata']){
      return $default;
    }

		return $this->_field_data[$field]['postdata'];
	}

	// --------------------------------------------------------------------

	/**
	 * Set Select
	 *
	 * Enables pull-down lists to be set to the value the user
	 * selected in the event of an error
	 *
	 * @access	public
	 * @param	string
	 * @param	string
	 * @return	string
	 */
	public function set_select($field = '', $value = '', $default = FALSE)
	{
		if ( ! isset($this->_field_data[$field]) OR ! isset($this->_field_data[$field]['postdata']))
		{
			if ($default === TRUE AND count($this->_field_data) === 0)
			{
				return ' selected="selected"';
			}
			return '';
		}

		$field = $this->_field_data[$field]['postdata'];

		if (is_array($field))
		{
			if ( ! in_array($value, $field))
			{
				return '';
			}
		}
		else
		{
			if (($field == '' OR $value == '') OR ($field != $value))
			{
				return '';
			}
		}

		return ' selected="selected"';
	}

	// --------------------------------------------------------------------

	/**
	 * Set Radio
	 *
	 * Enables radio buttons to be set to the value the user
	 * selected in the event of an error
	 *
	 * @access	public
	 * @param	string
	 * @param	string
	 * @return	string
	 */
	public function set_radio($field = '', $value = '', $default = FALSE)
	{
		if ( ! isset($this->_field_data[$field]) OR ! isset($this->_field_data[$field]['postdata']))
		{
			if ($default === TRUE AND count($this->_field_data) === 0)
			{
				return ' checked="checked"';
			}
			return '';
		}

		$field = $this->_field_data[$field]['postdata'];

		if (is_array($field))
		{
			if ( ! in_array($value, $field))
			{
				return '';
			}
		}
		else
		{
			if (($field == '' OR $value == '') OR ($field != $value))
			{
				return '';
			}
		}

		return ' checked="checked"';
	}

	// --------------------------------------------------------------------

	/**
	 * Set Checkbox
	 *
	 * Enables checkboxes to be set to the value the user
	 * selected in the event of an error
	 *
	 * @access	public
	 * @param	string
	 * @param	string
	 * @return	string
	 */
	public function set_checkbox($field = '', $value = '', $default = FALSE)
	{
		if ( ! isset($this->_field_data[$field]) OR ! isset($this->_field_data[$field]['postdata']))
		{
			if ($default === TRUE AND count($this->_field_data) === 0)
			{
				return ' checked="checked"';
			}
			return '';
		}

		$field = $this->_field_data[$field]['postdata'];

		if (is_array($field))
		{
			if ( ! in_array($value, $field))
			{
				return '';
			}
		}
		else
		{
			if (($field == '' OR $value == '') OR ($field != $value))
			{
				return '';
			}
		}

		return ' checked="checked"';
	}

	// --------------------------------------------------------------------

	/**
	 * Required
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 * @param  field = field_name,[in|not_in],aaa,bbb,ccc], if $_POST[field_name] in array(aaa,bbb,ccc) field was required
	 *                 if field='', check it's self value in array(aaa,bbb,ccc) or not
	 */
	public function required($str,$field)
	{
	    $validate = true;
	    if(isset($field)){
	        if(strpos($field,',') !== false){
	            $pieces = explode(",", $field);
	            if(count($pieces) > 2){
	                $field_name = array_shift($pieces);
	                $operator = array_shift($pieces);
	                $data_array = $pieces;
	                $check_data = '';
	                if($field_name == ''){
	                    $check_data = $str;//check self value 
	                }elseif(isset($_POST[$field_name])){
	                    $check_data = $_POST[$field_name];
	                }
	                $check_result = false;
	                switch ($operator){
	                    case 'in':
	                        $check_result = in_array($check_data,$data_array);
	                        break;
	                    case 'not_in':
	                        $check_result = !in_array($check_data,$data_array);
	                        break;
	                }
	                if($check_result){
    	                if($field_name == ''){
    	                    //self value check 
    	                    return false;
    	                }elseif(isset($_POST[$field_name])){
    	                    //need to continue check other rules
    	                   $validate = 'true';
    	                }
	                }else{
	                    if($field_name == ''){
	                        //self value not empty, but need to continue check other rules
	                        $validate = 'true';
	                    }else{
	                        //pass by other rules
	                        $validate = 'false';
	                    }
	                }
	            }else{
	                $validate = 'false';
	            }
	        }else{
	            if(isset($_POST[$field])){
	                $validate = $_POST[$field];
	            }else{
	                $validate = 'false';
	            }
	        }
	    }
	    
    
        if(!$validate || 'false' == $validate)
          return "SKIP";
    
    		if ( ! is_array($str))
    		{
    			return (trim($str) == '') ? FALSE : TRUE;
    		}
    		else
    		{
    			return ( ! empty($str));
    		}
	}
	public function disabled($str)
	{
	    return false;
	}

	// --------------------------------------------------------------------

	/**
	 * Performs a Regular Expression match test.
	 *
	 * @access	public
	 * @param	string
	 * @param	regex
	 * @return	bool
	 */
	public function regex_match($str, $regex)
	{
		if ( ! preg_match($regex, $str))
		{
			return FALSE;
		}

		return  TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Match one field to another
	 *
	 * @access	public
	 * @param	string
	 * @param	field
	 * @return	bool
	 */
	public function matches($str, $field)
	{
		if ( ! isset($_POST[$field]))
		{
			return FALSE;
		}

		$field = $_POST[$field];

		return ($str !== $field) ? FALSE : TRUE;
	}
	// --------------------------------------------------------------------

	/**
	 * NOT Match one field to another
	 *
	 * @access	public
	 * @param	string
	 * @param	field
	 * @return	bool
	 */
	public function not_matches($str, $field)
	{
		if ( ! isset($_POST[$field]) OR 0 == strlen($str) OR 'none' == $str )
		{
			return TRUE;
		}

		$field = $_POST[$field];

		return ($str !== $field) ? TRUE : FALSE;
	}
	
	// --------------------------------------------------------------------

	/**
	 * Match one field to another
	 *
	 * @access	public
	 * @param	string
	 * @param	field
	 * @return	bool
	 */
	public function is_unique($str, $field)
	{
		list($table, $field)=explode('.', $field);
		$query = $this->CI->db->limit(1)->get_where($table, array($field => $str));
		
		return $query->num_rows() === 0;
    }

	// --------------------------------------------------------------------

	/**
	 * Minimum Length
	 *
	 * @access	public
	 * @param	string
	 * @param	value
	 * @return	bool
	 */
	public function min_length($str, $val)
	{
		if (preg_match("/[^0-9]/", $val))
		{
			return FALSE;
		}

		if (function_exists('mb_strlen'))
		{
			return (mb_strlen($str) < $val) ? FALSE : TRUE;
		}

		return (strlen($str) < $val) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Max Length
	 *
	 * @access	public
	 * @param	string
	 * @param	value
	 * @return	bool
	 */
	public function max_length($str, $val)
	{
		if (preg_match("/[^0-9]/", $val))
		{
			return FALSE;
		}

		if (function_exists('mb_strlen'))
		{
			return (mb_strlen($str) > $val) ? FALSE : TRUE;
		}

		return (strlen($str) > $val) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Exact Length
	 *
	 * @access	public
	 * @param	string
	 * @param	value
	 * @return	bool
	 */
	public function exact_length($str, $val)
	{
		if (preg_match("/[^0-9]/", $val))
		{
			return FALSE;
		}

		if (function_exists('mb_strlen'))
		{
			return (mb_strlen($str) != $val) ? FALSE : TRUE;
		}

		return (strlen($str) != $val) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Valid Email
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function valid_email($str)
	{
		return ( ! preg_match("/^([a-z0-9\+_\-]+)(\.[a-z0-9\+_\-]+)*@([a-z0-9\-]+\.)+[a-z]{2,6}$/ix", $str)) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Valid Emails
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function valid_emails($str)
	{
		if (strpos($str, ',') === FALSE)
		{
			return $this->valid_email(trim($str));
		}

		foreach (explode(',', $str) as $email)
		{
			if (trim($email) != '' && $this->valid_email(trim($email)) === FALSE)
			{
				return FALSE;
			}
		}

		return TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Validate IP Address
	 *
	 * @access	public
	 * @param	string
	 * @return	string
	 */
	public function valid_ip($ip)
	{
		return $this->CI->input->valid_ip($ip);
	}
	// --------------------------------------------------------------------
	
	/**
	 * Validate IP Address and network prefix
	 *
	 * @access	public
	 * @param	string
	 * @return	string
	 */
	public function valid_ip_or_prefix($ip)
	{
	    if(strpos($ip,'/') === false){
	        return $this->valid_ip($ip);
	    }else{
            $tmp = explode( '/', $ip );
            if(count($tmp) > 2){
                return false;
            }else{
                if($this->valid_ip($tmp[0]) && $this->integer($tmp[1]) && $tmp[1]>0 && $tmp[1]<=32){
                    return true;
                }
            }	        
	    }
	    return false;
	}

	// --------------------------------------------------------------------

	/**
	 * Alpha
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function alpha($str)
	{
		return ( ! preg_match("/^([a-z])+$/i", $str)) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Alpha-numeric
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function alpha_numeric($str)
	{
		return ( ! preg_match("/^([a-z0-9])+$/i", $str)) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Alpha-numeric with underscores and dashes
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function alpha_dash($str)
	{
	    return ( ! preg_match("/^([-a-z.0-9_-])+$/i", $str)) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Numeric
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function numeric($str)
	{
		return (bool)preg_match( '/^[\-+]?[0-9]*\.?[0-9]+$/', $str);

	}

	// --------------------------------------------------------------------

	/**
	 * Is Numeric
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function is_numeric($str)
	{
		return ( ! is_numeric($str)) ? FALSE : TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Integer
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function integer($str)
	{
		return (bool) preg_match('/^[\-+]?[0-9]+$/', $str);
	}

	// --------------------------------------------------------------------

	/**
	 * Decimal number
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function decimal($str)
	{
    return ((bool) preg_match('/^[\-+]?[0-9]+\.[0-9]+$/', $str)
      || $this->integer($str));
	}

	// --------------------------------------------------------------------

	/**
	 * Greather than
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function greater_than($str, $min)
	{
    // Check if val is either a field or special string (today...)
    if(isset($_POST[$min]))
      $min = $_POST[$min];
		if ( ! is_numeric($str))
		{
			return FALSE;
		}
		return $str > $min;
	}
	/**
	 * Greather or equal
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function greater_or_equal($str, $min)
	{
    // Check if val is either a field or special string (today...)
    if(isset($_POST[$min]))
      $min = $_POST[$min];
		if ( ! is_numeric($str))
		{
			return FALSE;
		}
		return $str >= $min;
	}

	// --------------------------------------------------------------------

	/**
	 * Less or equal
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function less_or_equal($str, $max)
	{
		if ( ! is_numeric($str))
		{
			return FALSE;
		}
		return $str <= $max;
	}
	// --------------------------------------------------------------------

	/**
	 * Less than
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function less_than($str, $max)
	{
	    if ( ! is_numeric($str))
		{
			return FALSE;
		}
		return $str < $max;
	}

	// --------------------------------------------------------------------

	/**
	 * Different than
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function different_than($str, $token)
	{
		return ($str != $token);
	}

	// --------------------------------------------------------------------

	/**
	 * Is a Natural number  (0,1,2,3, etc.)
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function is_natural($str)
	{
		return (bool) preg_match( '/^[0-9]+$/', $str);
	}

	// --------------------------------------------------------------------

	/**
	 * Is a Natural number, but not a zero  (1,2,3, etc.)
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function is_natural_no_zero($str)
	{
		if ( ! preg_match( '/^[0-9]+$/', $str))
		{
			return FALSE;
		}

		if ($str == 0)
		{
			return FALSE;
		}

		return TRUE;
	}

	// --------------------------------------------------------------------

	/**
	 * Valid Base64
	 *
	 * Tests a string for characters outside of the Base64 alphabet
	 * as defined by RFC 2045 http://www.faqs.org/rfcs/rfc2045
	 *
	 * @access	public
	 * @param	string
	 * @return	bool
	 */
	public function valid_base64($str)
	{
		return (bool) ! preg_match('/[^a-zA-Z0-9\/\+=]/', $str);
	}

	// --------------------------------------------------------------------
  /**
    * @brief Test if string is a valid URL
    *
	  * @access	public
	  * @param	string
	  * @return	bool
   */
	public function valid_url($str)
	{
    if (strlen($str))
      return (bool) filter_var($str, FILTER_VALIDATE_URL);
    else
      return FALSE;
	}

  /**
    * @brief Validate SIP account
    *
    * @param[in out] $str
    *
    * @return 
   */
  public function  valid_sip_account($str)
  {
      // get user and domain
	    $tokens = explode('@',$str);
      if( count($tokens) == 2
        && $this->valid_sip_user($tokens[0])
        && $this->valid_ip_or_domain($tokens[1]) )
      {
        return true;
      }
      return false;

  }
  /*
   * SIP SecMon specific validation rules
  */
  function validate_account_filter($str)
  {
      // Rules are:
      // 1/ Empty string
      // 2/ 'unknown' string
      // 3/ comma separated account list (email)
      // 4/ 'regex=xxxxx' string
      $rc = (bool)FALSE;
      $str = trim($str);
      if ('' == $str || 'unknown' == strtolower($str) || !strncmp($str, 'regex=', strlen('regex='))) {
          return (bool)TRUE;
      } else {
          // Coma separeted account list
          // default is all good :)
          $rc = (bool)TRUE;
          $account_list = explode(',', $str);
          foreach ($account_list as $account) {
              // check for exclamation mark
              $account = trim($account);
              if ($account) {
                  if (!strncmp($account, '!', 1)) $account = substr($account, 1);
                  if (false == ($this->valid_sip_account($account) || preg_match('/@.+/', $account))) {
                      $rc = (bool)false;
                      break;
                  }
              }
          }
      }
      return $rc;
  }
  
  function validate_user_agent_filter($str)
  {
      // Rules are:
      // 1/ Empty string
      // 3/ Full string
      // 4/ 'regex=xxxxx' string
      $str = trim($str);
      if ('' == $str || $str || !strncmp($str, 'regex=', strlen('regex='))) {
          return (bool)TRUE;
      }
      return (bool)FALSE;
  }
	// --------------------------------------------------------------------
  /**
    * @brief Test if string is a valid IP or domain
    *
	  * @access	public
	  * @param	string
	  * @return	bool
   */
	public function valid_ip_or_domain($str)
	{
	    // first try to validate ip
	    $tokens = explode('.',$str);
	    if ( count($tokens) == 4
	          AND ctype_digit($tokens[0])
	          AND ctype_digit($tokens[1])
	          AND ctype_digit($tokens[2])
	          AND ctype_digit($tokens[3]) )
	    {
	      // most probably an IP
	      return (bool)$this->valid_ip($str);
	    }else{
	      // Maybe a domain,
	      // Try to see if all parts are valid
	      foreach($tokens as $piece)
	      {
	        if (!preg_match('/^[a-z\d][a-z\d-]{0,62}$/i', $piece)
	          || preg_match('/-$/', $piece) )
	        {
	          return FALSE;
	        }
	      }
	      return TRUE;
	    }
	    return FALSE;
	}
	public function ssl_ip_or_domain($str)
	{
	    // first try to validate ip
	    if(strtolower(substr($str, 0, 6)) == 'ssl://'){
	        $token = substr($str, 7);
        }else{
            $token = $str;
        }
	    return $this->valid_ip_or_domain($token);
	}
  public function valid_sip_user($str){
    return ( ! preg_match("/^([-a-z.0-9_-i\+\&=\!~\$\?\(\)\*\'\,\;\/])+$/i", $str)) ? FALSE : TRUE;
  }

	
	// --------------------------------------------------------------------
    /**
    * Test if string is valid PORT number.
    * @param string $str
    */
    public  function valid_port_number($str) {
          if(ctype_digit($str) && intval($str) > 0 && intval($str) <= 65535){
              return true;
          }else{
              return false;
          }
    }
	/**
	 * Test if string has a valid ip, domain name or IP:PORT notation.
	 * @param string $str
	 */
	public function valid_ip_port_domain($str) {
		$pos = strpos($str,':');
		if ($pos !== false) {
			$ip_port = explode(':',$str);
			if (count($ip_port) == 2 &&  $this->valid_ip_or_domain($ip_port[0]) &&  $this->valid_port_number($ip_port[1])){
			    return true;
			}else{
			    return false;
			}
		}else{
		    return $this->valid_ip_or_domain($str);
		}
	}
	
	// --------------------------------------------------------------------
	
	/**
	 * Test if the source ip validation rule in sip security monitor is correct
	 * @param str $str
	 */
	public function src_ip_filter_rule($str) {
		
		$success_count = 0;
		$fail_count = 0;
		
		if (trim($str) == "")
			return true;
		else {
			if (strpos($str,',') !== false) {
				$ip_list = explode (',',trim($str));
				for ($a = 0; $a < count($ip_list); $a++) {
					if (strpos($ip_list[$a],'/') !== false ) {
						//Valid Network
						if (strpos($ip_list[$a],'!') !== false) {
							$network = preg_split('/[!\/]/',trim($ip_list[$a]));
							//Check for exclamation mark then validate network
							if ($network[0] == "" && $this->valid_ip($network[1]) && $network[2] != "" && ($network[2] >= 1 && $network[2] <= 24))
								$success_count++;
							else
								$fail_count++;
						}
						else {
							//No exclamation mark
							//Validate network
							$network = explode('/',trim($ip_list[$a]));
							if ($this->valid_ip($network[0]) && $network[1] != "" && ($network[1] >= 1 && $network[1] <= 24))
								$success_count++;
							else
								$fail_count++;
						}
						unset($network);
					}
					else {
						//Check IP for a !
						if (strpos($ip_list[$a],'!') !== false) {
							$ip_addr = explode('!',trim($ip_list[$a]));
							if ($ip_addr[0] == "" && $this->valid_ip($ip_addr[1]))
								$success_count++;
							else 
								$fail_count++;
						}
						else { 
							if ($this->valid_ip(trim($ip_list[$a]))) 
								$success_count++;
							else
								$fail_count++;
						}
						unset($ip_addr);
					}
					//echo "FAIL = $fail_count & PASS = $success_count<br />";
				}
				if ($fail_count > 0)
					return false;
				elseif ($fail_count == 0 && $success_count > 0)
					return true;
			}
			else {
				//No commas entered. Validate standalone network or ip
				if (strpos($str,'/') !== false ) {
					if (strpos($str,'!') !== false) {
						$network = preg_split('/[!\/]/',trim($str));
						if ($network[0] == "" && $this->valid_ip($network[1]) && $network[2] != "" && ($network[2] >= 1 && $network[2] <= 24))
							return true;
					}
					else {
						$network = explode('/',trim($str));
						if ($this->valid_ip($network[0]) && $network[1] != "" && ($network[1] >= 1 && $network[1] <= 24))
							return true;
					}
				}
				else {
					if (strpos($str,'!') !== false) {
						$ip = explode('!',trim($str));
						if ($ip[0] == "" && $this->valid_ip($ip[1]))
							return true;
					}
					else {
						if ($this->valid_ip(trim($str)))
							return true;
					}
				}
			}
		}
		
		return false;
	}
	
	// --------------------------------------------------------------------
	
	/**
	 * Validates if the external sip ip entered is valid or not
	 *
	 * @param string $str
	 * @return boolean
	 */
	public function valid_ext_sip_ip ($str) {
		
		if ($str == "auto-nat")
			return true;
		elseif ($this->valid_ip($str))
			return true;
		elseif (preg_match('"\bstun\b"',$str) || preg_match('"\bhost\b"',$str)) {
			$pos = strpos($str,':');
			if ($pos !== false) {
				$string = explode(':',$str);
				if (($this->valid_ip_or_domain($string[1]) && $string[0] == "host") || ($this->valid_ip_or_domain($string[1]) && $string[0] == "stun"))
					return true;
			}
		}
		
		return false;
	}
		
	// --------------------------------------------------------------------
	
	/**
	 * Validates SIP Faliure Response Codes
	 * @param string $str
	 * @return boolean
	 */
	
	public function sip_fail_response_code ($str) {
		if (is_numeric($str) && preg_match('/^[56][0-9][0-9]$/i',$str))
			return true;
		
		return false;
	}
	
	// --------------------------------------------------------------------
		
	/**
	 * Test if string is a valid IP, Blank line or contains the word ANY
	 * @param $str string
	 * 				String which will be validated.
	 */
	public function valid_ip_or_blank_or_any($str) {
		// first try to validate ip
		$tokens = explode('.',$str);
		if ( count($tokens) == 4
				AND ctype_digit($tokens[0])
				AND ctype_digit($tokens[1])
				AND ctype_digit($tokens[2])
				AND ctype_digit($tokens[3]) )
		{
			// most probably an IP
			return (bool)$this->valid_ip($str);
		} else {
			if ($str == "")
				return true;
			elseif (strtoupper($str) == "ANY")
				return true;
		}
		return false;
	}
	
	// --------------------------------------------------------------------
  /**
    * @brief Test if string is a valid time HH:MM:SS format
    *
	  * @access	public
	  * @param	string
	  * @return	bool
   */
	public function valid_time($str)
	{
		//return (bool) preg_match('/([01]?[0-9]|2[0-3]):[0-5][0-9]:[0-5][0-9]/', $str);
    $tokens = explode(':',$str);
    // Validate ranges (required as mktime can accept hour greater than 24 to
    // represent hours in the next day. Similar behavior for min and sec)
    if(isset($tokens[0]) && ($tokens[0] >= 24 || $tokens[0] <0))
      return FALSE;
    if(isset($tokens[1]) && ($tokens[1] >= 60 || $tokens[1] <0))
      return FALSE;
    if(isset($tokens[2]) && ($tokens[2] >= 60 || $tokens[2] <0))
      return FALSE;

    // Try to create a time
    if(count($tokens)==2)
      return (FALSE != mktime($tokens[0], $tokens[1]));
    elseif(count($tokens)==3)
      return (FALSE != mktime($tokens[0], $tokens[1], $tokens[2]));

    return FALSE;
	}
	
	// --------------------------------------------------------------------
  /**
    * @brief Test if string is a valid date YYYY-MM-DD format
    *
	  * @access	public
	  * @param	string
	  * @return	bool
   */
	public function valid_date($str)
	{
    $tokens = explode('-',$str);
    return ( count($tokens)==3
          AND checkdate($tokens[1], $tokens[2], $tokens[0]) );
  }
  public function date_greater_than($str, $val){
    // Check if val is either a field or special string (today...)
    if(isset($_POST[$val]))
      $val = $_POST[$val];
    if($val == 'today')
      $val = date('Y-m-d');
    // Check both date valid
    if($this->valid_date($str) && $this->valid_date($val)) {
      // Compare dates
      if(strtotime($str) > strtotime($val))
        return TRUE;
    }

    return FALSE;
  }

  /**
   * Test if a string is a valid date and time format YYYY-MM-DD HH:MM
   * @param string $str
   * @return boolean
   */
  public function valid_date_time($str) {
  	$tokens = explode(' ',$str);
    return ((2 == count($tokens)) && $this->valid_date($tokens[0]) && $this->valid_time($tokens[1]));
  }


  /**
   * Test if the string is a valid UUID
   * @param string $str
   */
  public function valid_uuid($str) {
    if (eregi("^[0-9a-f]{8}(-[0-9a-f]{4}){3}-[0-9a-f]{12}$", $str))
      return true;

    return false;
  }
  /**
    * @brief Test $str in supplied list
    *
    * @param[in out] $str
    * @param[in out] $list
    *
    * @return 
   */
  public function in_list($str, $list) {
      //if $str is array, need to check each item of $str is in $list
      if(!is_array($str)){
          $str = array($str);
      }
      $options = explode(',',$list);
      foreach($str as $key => $tmp_item){
          if(0 == strlen($tmp_item) || !in_array($tmp_item,$options)) {
              return false;
          }
      }
      return true;
  }
  
  public function not_in_list($str, $list) {
      if(0 != strlen($str)){
          $options = explode(',',$list);
          return (!in_array($str,$options));
      }
      return true;
  }
  
	// --------------------------------------------------------------------

	/**
	 * Prep data for form
	 *
	 * This function allows HTML to be safely shown in a form.
	 * Special characters are converted.
	 *
	 * @access	public
	 * @param	string
	 * @return	string
	 */
	public function prep_for_form($data = '')
	{
		if (is_array($data))
		{
			foreach ($data as $key => $val)
			{
				$data[$key] = $this->prep_for_form($val);
			}

			return $data;
		}

		if ($this->_safe_form_data == FALSE OR $data === '')
		{
			return $data;
		}

		return str_replace(array("'", '"', '<', '>'), array("&#39;", "&quot;", '&lt;', '&gt;'), stripslashes($data));
	}

	// --------------------------------------------------------------------

	/**
	 * Prep URL
	 *
	 * @access	public
	 * @param	string
	 * @return	string
	 */
	public function prep_url($str = '')
	{
		if ($str == 'http://' OR $str == '')
		{
			return '';
		}

		if (substr($str, 0, 7) != 'http://' && substr($str, 0, 8) != 'https://')
		{
			$str = 'http://'.$str;
		}

		return $str;
	}

	// --------------------------------------------------------------------

	/**
	 * Strip Image Tags
	 *
	 * @access	public
	 * @param	string
	 * @return	string
	 */
	public function strip_image_tags($str)
	{
		return $this->CI->input->strip_image_tags($str);
	}

	// --------------------------------------------------------------------

	/**
	 * XSS Clean
	 *
	 * @access	public
	 * @param	string
	 * @return	string
	 */
	public function xss_clean($str)
	{
		return $this->CI->security->xss_clean($str);
	}

	// --------------------------------------------------------------------

	/**
	 * Convert PHP tags to entities
	 *
	 * @access	public
	 * @param	string
	 * @return	string
	 */
	public function encode_php_tags($str)
	{
		return str_replace(array('<?php', '<?PHP', '<?', '?>'),  array('&lt;?php', '&lt;?PHP', '&lt;?', '?&gt;'), $str);
	}

}
// END Form Validation Class

/* End of file Form_validation.php */
/* Location: ./system/libraries/Form_validation.php */
