<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/**
 * SNG Network service configuration wrapper
 *
 * @author William Adam
 * @version
 */
if (!defined('BASEPATH')) exit('No direct script access allowed');
require_once ('application/helpers/safe_helper.php');
safe_require_class('service');
/*
 * Some IP helper functions
 * from  https://mebsd.com/coding-snipits/php-ipcalc-coding-subnets-ip-addresses.html
 */

// convert cidr to netmask
// e.g. 21 = 255.255.248.0
function cidr2netmask($cidr)
{
    for( $i = 1; $i <= 32; $i++ )
        $bin .= $cidr >= $i ? '1' : '0';

    $netmask = long2ip(bindec($bin));

    if ( $netmask == "0.0.0.0")
        return false;

    return $netmask;
}

// get network address from cidr subnet
// e.g. 10.0.2.56/21 = 10.0.0.0
function cidr2network($ip, $cidr)
{
    $network = long2ip((ip2long($ip)) & ((-1 << (32 - (int)$cidr))));

    return $network;
}

// convert netmask to cidr
// e.g. 255.255.255.128 = 25
function netmask2cidr($netmask)
{
    $bits = 0;
    $netmask = explode(".", $netmask);

    foreach($netmask as $octect)
        $bits += strlen(str_replace("0", "", decbin($octect)));

    return $bits;
}

// is ip in subnet
// e.g. is 10.5.21.30 in 10.5.16.0/20 == true
//      is 192.168.50.2 in 192.168.30.0/23 == false 
function cidr_match($ip, $network, $cidr)
{
    if ((ip2long($ip) & ~((1 << (32 - $cidr)) - 1) ) == ip2long($network))
    {
        return true;
    }

    return false;
}


class Sng_network_route_add_config extends Safe_configuration_user_class 
{
    protected $_operator = 'add';

    public function __construct($route_name, $data = array(), $action=Safe_configuration_class::CFG_CREATE)
    {
        parent::__construct($route_name, $action, $data);
    }
    public function run($node)
    {
        $data = $this->data();
        $cidr = $data['prefix'];
        $network = cidr2network($data['address'], $cidr);
        // Prepare command:
        //  ip route add <network-addr>/<network-mask-bits> [via <gateway-addr>] dev <ethernet-device-name>
        $cmd = '/sbin/ip route '.$this->_operator.' ';
        $cmd .= $network;
        // Nework route or host route ?
        if($cidr) {
            $cmd .= '/'.$cidr;
        }
        if (strlen(trim($data['gateway']))){
            $cmd .= ' via '.$data['gateway'];
        }
        $cmd .= ' dev '.$data['interface'];

        $node->execute('sudo nohup ' . $cmd); 
        // For now always return true
        // Known valid usecase when ip route command fail:
        //   * add a route on a down interface
        //
        return true;
    }
}

class Sng_network_route_delete_config extends Sng_network_route_add_config
{
    protected $_operator = 'del';
    public function __construct($route_name, $data = array(), $action=Safe_configuration_class::CFG_DELETE)
    {
        parent::__construct($route_name, $data, $action);
    }
}

class Sng_network_route_class extends Safe_configurable_object_class
{
    public function __construct($node, $parent_name, $name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, $name, $node);
        // Do NOT re-use records on update/delete
        $this->_set_reuse(false);
    }
    public function configure()
    {
        // General settings
        $adapters = $this->_node->hardware()->adapters();
        $interfaces=array();
        foreach($adapters as $k => $v){
            $if_ip = $v->ip_address();
            $if_prefix = netmask2cidr($v->mask());
            if($if_ip){
                $label = $k . ' - ' . $if_ip . '/' . $if_prefix;
                $interfaces[$k] = $label;
            }
        }
        $this->add_field('address', 'Destination', 'text', '',25);
        $this->set_field_help('address', 'IP address.');
        $this->set_field_rules('address', 'required|valid_ip');

        $this->add_field('prefix', 'Prefix', 'text', '',5);
        $this->set_field_help('prefix', 'The network prefix for the destination IP address.');
        $this->set_field_rules('prefix', 'required|greater_than[1]|less_or_equal[32]');
        $this->composite_layout('address', array('prefix'));

        $this->add_enum_field('interface', 'Network Interface', 'dropdown', '', $interfaces);
        $this->set_field_help('interface', 'The network interface used to reach destination IP address.' );

        $this->add_field('gateway', 'Gateway', 'text', '',25);
        $this->set_field_help('gateway', 'The default gateway, or an IP address that can be used to reach the destination IP address.');
        $this->set_field_rules('gateway', 'valid_ip');

        return parent::configure();
    }

    public function description()
    {
        $data = $this->get_data_values();
        return $data['interface'] . ' / ' . $data['prefix'] . ' - ' . $data['gateway'];
    }
    public function summary($type , $long, $min)
    {
        return parent::summary($type, $long, 4);
    }

    public function validate($data, &$output)
    {
        // Run regular validation first
        $validate = parent::validate($data, $output);
        if (!$validate) {
            return $validate;
        }
        // Now take care of extra rules:
        // 1/ destination must be unique
        // 2/ if a gateway is specified, it must be on same subnet as interface
        $network_module = $this->node()->find_module_by_name('network');
        $routes = $network_module->api_retrieve_route();
        $address = $this->get_data_value('address', false);
        foreach($routes as $route){
            // Skip ourself
            if($this->name() == $route->name()){
                continue;
            }
            $route_data = $route->get_data_values(false);
            // 1/ check unique route
            if($address == $route_data['address']){
                $validate = false;
                $output['address']= 'Destination already used by route '.$route->name();
                $this->set_form_error('address', $output['address']);
                break;
            }
        }
        // 2/ check gateway
        if($data['gateway']){
            $interface = $this->get_data_value('interface', false);
            // Get adapter and associated IP address
            $if = $this->node()->hardware()->api_retrieve_adapter($interface);
            if($if){
                $if_ip = $if->ip_address();
                $if_prefix = netmask2cidr($if->mask());
                $subnet = cidr_match($data['gateway'], cidr2network($if_ip, $if_prefix), $if_prefix);
                if(!$subnet){
                    $validate = false;
                    $output['gateway']= 'Gateway is not on '.$data['interface'].'subnet ('.$if_ip.'/'.$if_prefix.')';
                    $this->set_form_error('gateway', $output['gateway']);
                }
            }
        }

        return $validate;
    }
}


class Sng_network_service_class extends Safe_service_class
{
    public function __construct($software)
    {
        parent::__construct($software->node()->software(), "network", $process_name);
    }

    /**
     * @brief
     *         
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("Network");

        // Register objects
        $this->register_aggregate_object('route', 
            array(
                'name' => 'Static Route',
                'base_path' => $this->object_name() . '/route',
                'dynamic' => true,
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Create a Static Route',
                        'request' => 'POST'
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a Static Route',
                        'request' => 'GET'
                        ),
                    'update' => array(
                            'name' => 'Update',
                            'description' => 'Update a Static Route',
                            'request' => 'POST'
                    ),
                    'delete' => array(
                            'name' => 'Delete',
                            'description' => 'Delete a Static Route',
                            'request' => 'POST'
                    )
               )
            )
        );

        return parent::configure();
    }

    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_route($name, $data=null, &$output = null) {
        $route = new Sng_network_route_class($this->node(), $this->object_name().'/route', $name);
        $route->configure();
        return $route;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_route($name=null, $data=null, &$output = null) {
        $objs = $this->get_aggregate_objects('route');
        if($name){
            return $objs[$name];
        }else{
            return $objs;
        }
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_route($name, $data=null, &$output = null) {
        // We cannot update static route object, must delete old one and create 
        // a new one so serialization can occur correctly (ie. delete using ip 
        // route del with EXACT same parameters as create.
        // Only exception is if route is not yet serialized, then we can just 
        // update as usual (ie. status == NEW)
        $route = $this->api_retrieve_route($name);
        if($route) {
            if ($route->validate($data,$output)) {
                return $route->save($output);
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_route($name, $data=null, &$output = null) {
        $route = $this->api_retrieve_route($name);
        if($route) {
            if($route->can_dispose($output)) {
                return $route->dispose();
            }
        }
        return false;
    }

    public function can_restore($info, &$reason)
    {
        // Prevent calling parent as service status will be checked
        return true;
    }
    /**
     * @brief Create config jobs for dialplan module
     *           
     * @param[in out] $config_manager
     *           
     * @return
     */
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        if(!$this->_generate_config($config_manager, $obj_type)) return false;
        return parent::reload_generate_config($config_manager, $obj_type);
    }
    /**
     * @brief 
     *
     * @param[in out] $obj_type
     *
     * @return 
     */
    public function reload_clear_modified($obj_type=null)
    {
        return $this->clear_configuration_modified();
    }

    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    public function generate_config(&$config_manager)
    {
        if(!$this->_generate_config($config_manager)){
            return false;
        }
        return parent::generate_config($config_manager);
    }
    /**
     * @brief Invoked after a successfull write_config
     *
     * @return
     */
    public function post_write_config($obj_type=null)
    {
        // restart service
        //$this->restart();
        return parent::post_write_config($obj_type);
    }
    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    private function _generate_config(&$config_manager, $obj_type=null)
    {
        // Populate persistent routes
        // 1st remove existing
        $adapters = $this->node()->hardware()->adapters();
        foreach($adapters as $if){
            $file = "/etc/sysconfig/network-scripts/route-".$if->name();
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
        }

        // Loop around static routes to create interface route def
        $interfaces = array();
        foreach($this->get_aggregate_objects('route') as $route) {
            $data = $route->get_data_values(false);
            $if = $data['interface'];
            unset($data['interface']);
            $id = count($interfaces[$if]);
            // For host route, need to add a /32 netmask
            if(strlen(trim($data['prefix']))) {
                $data['netmask'] = cidr2netmask($data['prefix']);
                $data['address'] = cidr2network($data['address'], $data['prefix']);
            }else{
                $data['netmask'] = '255.255.255.255';
            }
            unset($data['prefix']);

            $route_def = array();
            foreach($data as $k => $v) {
                $route_def[strtoupper($k).$id] = $v;
            }
            $interfaces[$if][] = $route_def;
        }

        // Loop around interface route def to create file
        foreach($interfaces as $if => $routes) {
            $content = '';

            foreach($routes as $route){
                // Write all entries
                foreach($route as $k => $v){
                    $content[] = $k.'='.$v;
                }
            }

            $file = "/etc/sysconfig/network-scripts/route-".$if;
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, $content) );
        }
        // we MUST handled deleted routes before new ones
        // Loop around deleted static routes to create ip route del job
        $obj_path = $this->aggregate_object_base_path('route');
        $deleted_routes = Safe_object_serializer_class::get_serializer()->find_deleted_objects($obj_path);
        foreach($deleted_routes as $route => $route_data) {
            $config_manager->add_config(
                new Sng_network_route_delete_config(
                    $route, $route_data['data']) );
        }

        // Loop around static routes to create ip route add job
        foreach($this->get_aggregate_objects('route') as $route) {
            // Only on new route
            if($route->status() == Safe_object_serializer_class::OBJ_STATUS_NEW) {
                $route_data = $route->get_data_values(false);
                $config_manager->add_config(
                    new Sng_network_route_add_config(
                        $route->name(), $route_data) );
            }
        }

        return true;
    }
}
/* End of file sng_firewall_service_class.php */
