<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/**
 * SNG httpd(webconfig) service wrapper
 *
 * @author William Adam
 * @version
 */
if (!defined('BASEPATH')) exit('No direct script access allowed');
require_once ('application/helpers/safe_helper.php');
safe_require_class('service');

class Sng_httpd_config_class extends Safe_configurable_object_class
{
    private $_adapters = array();
    public function __construct($node, $parent_name, $name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, 'configuration', $node);
    }
    public function configure()
    {
        // Default interface name
	    $def_if = 'all';
	    $mgmt_if = $this->_adapters();
	    $if_enum = array();
	    $if_enum['all'] = 'All interfaces';
        foreach($mgmt_if as $if_name => $if_ip) {
            $if_enum[$if_name] = $if_name . ' - ' . $if_ip;
        }
        $this->add_enum_field('interface', 'Network Interface', 'dropdown', '', $if_enum);
        $this->set_field_help('interface', 'Select interface that web server listens on.');

        $this->add_field('http_port', 'HTTP port', 'text', '80',10);
        $this->set_field_rules('http_port', 'required|valid_port_number|less_than[61000]|not_matches[https_port]|different_than[81]|callback_port_check[http_port]');
        $this->set_field_help('http_port', 'Port number for HTTP protocol.');
        
        $this->add_field('https_port', 'HTTPS port', 'text', '443',10);
        $this->set_field_rules('https_port', 'required|valid_port_number|less_than[61000]|not_matches[http_port]|different_than[81]|callback_port_check[https_port]');
        $this->set_field_help('https_port', 'Port number for HTTPS protocol.');

        return parent::configure();
    }
    public function description()
    {
        return parent::description();
    }
    private  function _adapters(){
        if(!$this->_adapters){
            // Populate IP enumeration with all management interface IP
            $mgmt_if = $this->_node->hardware()->adapters(Safe_hardware_adapter_class::ETHERNET_ADAPTER);
            foreach($mgmt_if as $if_name=>$if) {
                if(strlen($if->ip_address()))
                {
                    $this->_adapters[$if->name()] = $if->ip_address();
                }
            }
        }
        return $this->_adapters;
    }
    public function port_check($field_name = null){
        if($field_name){
            $port_name = $this->get_data_label($field_name);
            $port_value = $this->get_data_value($field_name);
            $port_type = 'TCP';
        }else{
            return true;
        }
        //adapters interfaces name to ip address
        $tmp_adapters = $this->_adapters();
        $if_name = $this->get_data_value('interface',false);
        if(isset($tmp_adapters[$if_name])){
            $ip =  $tmp_adapters[$if_name];
        }else{
            $ip = 'all';
        }
        //ip port
        $ports[] = array('name' =>$port_name,'port'=>$port_value, 'ip'=>$ip, 'type' => $port_type);
    
        //check $conflicts
        $conflicts = $this->node()->software()->application()->check_resource_conflict('webconfig','IP_PORT', $ports);
    
        foreach($conflicts as $conflict){
            $mod = $this->node()->software()->application()->find_module_by_name($conflict['name']);
            $name = $mod->description();
            $error_msg = $error_msg . " with {$name} at {$conflict['desc']}, " ;
        }
    
        $error_msg_2 ='';
        //check system port
        foreach($ports as $port_item){
            if(!$this->node()->os()->valid_ip_port($port_item['ip'], $port_item['port'], $port_item['type'], 'webconfig', $prog_name)){
                $error_msg_2 = " Port used by ".$prog_name;
            }
        }
        if($error_msg == '' && $error_msg_2 == ''){
            return true;
        }else{
            if ($error_msg!=''){
                $error['port_check'] = 'Port conflict'.substr($error_msg, 0, -2).'. ';
            }
            if ($error_msg_2!=''){
                $error['port_check'] = $error['port_check'] . $error_msg_2 .' ';
            }
            return $error;
        }
    }
}
class Sng_httpd_service_class extends Safe_service_class
{
    public function __construct($software)
    {
        parent::__construct($software, 'webconfig');
    }
    /**
     * @brief
     *
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("Web Server");
        // Create the config object
        $cfg = new Sng_httpd_config_class($this->node(), $this->object_name(),$this->name());
        // Synch with DB
        $cfg->configure();
        $cfg->synch();
        // Attach config to module
        $this->set_config($cfg);
        
        return parent::configure();
    }
    public function can_restore($info, &$reason)
    {
        // Prevent calling parent as service status will be checked
        return true;
    }
    public function allow_user_ctl()
    {
        // User not allowed to control service
        return 'hide';
    }
    /**
     * @brief Generate httpd configuration
     *
     * @param[in out] $config_manager
     *
     * @return true on success
     */
    public function generate_config(&$config_manager)
    {
        //Generates service configuration files
        if (!$this->_generate_service_config($config_manager)) return false;
        // Invoke parent generate
        return parent::generate_config($config_manager);
    }

    private function interface_to_ip($interface){
        $mgmt_if = $this->node()->hardware()->adapters(Safe_hardware_adapter_class::ETHERNET_ADAPTER);
        $tmp_adapters = array();
        foreach($mgmt_if as $if_name=>$if) {
            if(strlen($if->ip_address()))
            {
                $tmp_adapters[$if->name()] = $if->ip_address();
            }
        }
        if(isset($tmp_adapters[$interface])){
            $ip =  $tmp_adapters[$interface];
        }else{
            $ip = 'all';
        }
        return $ip;
    }
    public function check_resource_conflict($src_module_name, $resource_type, $resource_data)
    {
        // IP_PORT conflicts ?
        $conflicts = array();
        if('IP_PORT' == $resource_type){
            // get our configuration
            $data = $this->config()->get_data_values(false);
            $ip = $this->interface_to_ip($data['interface']);
            $ports[] = array('name' =>$this->name(),'port'=>$data['http_port'], 'ip'=>$ip, 'type' => 'TCP');
            $ports[] = array('name' =>$this->name(),'port'=>$data['https_port'], 'ip'=>$ip, 'type' => 'TCP');
            // Loop around resource data
            foreach($resource_data as $resource){
                foreach($ports as $port_item){
                    if(safe_check_port_conflict($port_item, $resource)){
                        $conflicts[] = array('name' => $this->name(), 'obj_type' =>'Configuration', 'desc' =>'port '.$port_item['port']);
                    }
                }
            }
        }
        return $conflicts;
    }

    public function _generate_service_config(&$config_manager)
    {
        $data = $this->config()->get_data_values(false);
        $ip = $this->interface_to_ip($data['interface']); 
        
        $ports[] = array('name' =>$this->name(),'port'=>$data['http_port'], 'ip'=>$ip, 'type' => 'TCP');
        $ports[] = array('name' =>$this->name(),'port'=>$data['https_port'], 'ip'=>$ip, 'type' => 'TCP');
        
        //check system port
        foreach($ports as $port_item){
            if(!$this->node()->os()->valid_ip_port($port_item['ip'], $port_item['port'], $port_item['type'], 'webconfig')){
                $config_manager->add_error(array(
                        'module' => 'webconfig',
                        'obj_type' => '',
                        'description' => "{$this->description()} port {$port_item['port']} is used by other program.",
                        'url' => '/SAFe/sng_httpd_config'
                    )
                );
            }
        }
        //check $conflicts
        $conflicts = $this->node()->software()->application()->check_resource_conflict($this->name(), 'IP_PORT', $ports);
        foreach($conflicts as $conflict){
            $mod = $this->node()->software()->application()->find_module_by_name($conflict['name']);
            $name = $mod->description();
            $config_manager->add_error(array(
                    'module' => 'webconfig',
                    'obj_type' => '',
                    'description' => "{$this->description()} has IP_PORT conflict with {$name} at {$conflict['desc']}.",
                    'url' => '/SAFe/sng_httpd_config'
                )
            );
            
        }
        
        // Get our configuration data (unresolved)
        $http_conf_file  = '/usr/webconfig/conf/httpd.d/server_http.conf';
        $https_conf_file = '/usr/webconfig/conf/httpd.d/server_https.conf';
        if($ip == 'all'){
            $http_conf_string = 'Listen '.$data['http_port'];
        }else{
            $http_conf_string = 'Listen '.$ip.':'.$data['http_port'];
        }
        $config_manager->add_config(new Safe_configuration_class($http_conf_file, 'webconfig', Safe_configuration_class::CFG_UPDATE,Safe_configuration_class::CFG_CHOWN));
        $sed_data[] = "/^\s*listen.*/Id";
        $sed_data[] = "1i\\$http_conf_string\\";
        $sed_data[] = "s/^\s*<VirtualHost.*/<VirtualHost \_default_:".$data['http_port'].">/";

        $config_manager->add_config(new Safe_configuration_class($http_conf_file, $sed_data, Safe_configuration_class::CFG_UPDATE,Safe_configuration_class::CFG_SED));
        $config_manager->add_config(new Safe_configuration_class($http_conf_file, 'root', Safe_configuration_class::CFG_UPDATE,Safe_configuration_class::CFG_CHOWN));
        unset($sed_data);
        
        if($ip == 'all'){
            $https_conf_string = 'Listen '.$data['https_port'];
        }else{
            $https_conf_string = 'Listen '.$ip.':'.$data['https_port'];
        }
        $config_manager->add_config(new Safe_configuration_class($https_conf_file, 'webconfig', Safe_configuration_class::CFG_UPDATE,Safe_configuration_class::CFG_CHOWN));
        $sed_data[] = "/^\s*listen.*/Id";
        $sed_data[] = "1i\\$https_conf_string\\";
        $sed_data[] = "s/^\s*<VirtualHost.*/<VirtualHost \_default_:".$data['https_port'].">/";
        $config_manager->add_config(new Safe_configuration_class($https_conf_file, $sed_data, Safe_configuration_class::CFG_UPDATE,Safe_configuration_class::CFG_SED));
        $config_manager->add_config(new Safe_configuration_class($https_conf_file, 'root', Safe_configuration_class::CFG_UPDATE,Safe_configuration_class::CFG_CHOWN));
        return true;
    }
    /**
     * @brief Override httpd restart to use reload
     *
     * @param[in out] $force
     * @param[in out] $result
     *
     * @return 
     */
    public function reload($force = false, &$result = null)
    {
        return $this->node()->os()->execute('nohup /usr/local/sng/scripts/restart-webconfig.sh', $result, true, true);
    }

    public function restart($force = false, &$result = null)
    {
       return $this->reload($force, $result);
    }

    /**
     * @brief Invoked after a successfull write_config
     *
     * @return
     */
    public function post_write_config($obj_type=null)
    {
        //clear before httpd restart else it will not clear
        if($this->is_configuration_modified()){
            $this->clear_configuration_modified();
            // restart service (in fact reload)
            $this->reload();
        }
        return parent::post_write_config($obj_type);
    }
    public function support_reload(&$need_reload = null) {
        parent::support_reload($need_reload);
        return true;
    }
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        return $this->generate_config($config_manager);
    }
    public function reload_clear_modified($obj_type=null)
    {
        return $this->clear_configuration_modified();
    }
    /**
     * @brief Create httpd server URI
     *
     * @param[in out] $hostname
     *
     * @return 
     */
    public function server_uri($hostname, $protocol='http')
    {
        $uri = 'http://'.$hostname.':'.$this->config()->get_data_value($protocol.'_port',false);
        return $uri;
    }
}
/* End of file sng_httpd_service_class.php */
