<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * SNG REST module class
*/
require_once ('application/helpers/safe_helper.php');
safe_require_class('safe_module_class');
class Sng_rest_key_config_class extends Safe_configurable_object_class
{
    /**
     * @brief Constructor
     *
     * @param[in out] $parent_name
     * @param[in out] $name
     *
     * @return
     */
    public function __construct($node, $parent_name, $name)
    {
        parent::__construct($parent_name, $name, $node);
    }
    /**
     * @brief Configure the config class object
     *
     * @return
     */
    public function configure()
    {
        // General settings
        $default_key = strtoupper(md5(rand(0,9999)+time()));
        $this->add_field('key', 'API key', 'text', $default_key,40);
        $this->set_field_help('key','REST API Key for  authentication/access control. This option is read only,generated by system.');
        $this->set_field_readonly('key');
        $this->set_field_rules('key', 'required');
        $this->add_field('description', 'Description', 'text', '',50);
        return parent::configure();
    }
    public function description()
    {
        return $this->get_data_value('key') . ' - ' . $this->get_data_value('description');
    }
}
class Sng_rest_allowed_ip_config_class extends Safe_configurable_object_class
{
    /**
     * @brief Constructor
     *           
     * @param[in out] $parent_name
     * @param[in out] $name
     *           
     * @return
     */
    public function __construct($node, $parent_name, $name)
    {
        parent::__construct($parent_name, $name, $node);
    }
    /**
     * @brief Configure the config class object
     *         
     * @return
     */
    public function configure()
    {
        // General settings
        $this->add_field('ipaddr', 'IP Address', 'text', '',25);
        $this->set_field_help('ipaddr', 
                array('Valid IPv4 address to allow RESTful API access from.',
                        'Also support for network prefix notation: ',
                        'For example: "10.20.1.0/24" -> Allow all 10.20.1.* hosts'
                ));
        $this->set_field_rules('ipaddr', 'required|valid_ip_or_prefix');
        $this->add_field('description', 'Description', 'text', '',50);
        return parent::configure();
    }
    public function description()
    {
        return $this->get_data_value('ipaddr') . ' - ' . $this->get_data_value('description');
    }
}

class Sng_rest_config_class extends Safe_configurable_object_class
{
    /**
     * @brief Constructor
     *           
     * @param[in out] $parent_name
     * @param[in out] $name
     *           
     * @return
     */
    public function __construct($node, $parent_name)
    {
        parent::__construct($parent_name, "configuration");
    }
    /**
     * @brief Configure the config class object
     *         
     * @return
     */
    public function configure()
    {
        // General settings
        $this->add_enum_field('enable', 'RESTful API', 'dropdown', 'false', $this->enable_disable_enum());
        $this->set_field_help('enable', 'Option to enable/disable RESTful API');
        $this->set_field_category('enable', 'General');
        // Access control
        $this->add_enum_field('ip-whitelist', 'IP White List', 'dropdown', 'true', $this->enable_disable_enum());
        $this->set_field_help('ip-whitelist', 'Any client connecting to RESTful API will be checked against the list of allowed IP addresses.');
        
        $this->add_enum_field('api-key', 'API Key', 'dropdown', 'false', $this->enable_disable_enum());
        $this->set_field_help('api-key', 
            array(
                'Any client connecting to RESTful API will be checked against the api key.',
                'Key is send using X-API-KEY header.'
                )
            );

        $this->set_field_category('ip-whitelist', 'Access Control');
        $this->set_field_category('api-key', 'Access Control');

        return parent::configure();
    }
    public function description()
    {
        return $this->get_data_value('enable');
    }
}



class Sng_rest_module_class extends Safe_module_class
{
    protected $_obj_base_path = '';
    /**
     * @brief
     *           
     * @param[in out] $fs_app
     *           
     * @return
     */
    public function __construct($app)
    {
        parent::__construct($app->node()->software(), "rest");
    }
    /**
     * @brief
     *         
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("REST");

        // Create the config object
        $cfg = new Sng_rest_config_class($this->node(), $this->object_name());
        // Synch with DB
        $cfg->configure();
        $cfg->synch();
        // Attach config to module
        $this->set_config($cfg);
        // Register objects
        $this->register_aggregate_object('ipaddr', 
            array(
                'name' => 'Allowed IP Address',
                'base_path' => $this->object_name() . '/ipaddr',
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Create a IP Address Condition',
                        'request' => 'POST',
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a IP Address Condition',
                        'request' => 'GET',
                        ),
                    'update' => array(
                        'name' => 'Update',
                        'description' => 'Update a  IP Address Condition',
                        'request' => 'POST',
                        ),
                    'delete' => array(
                        'name' => 'Delete',
                        'description' => 'Delete a  IP Address Condition',
                        'request' => 'POST',
                        ),
                    ),
            )
        );
        $this->register_aggregate_object('apikey',
                array(
                        'name' => 'API Key',
                        'base_path' => $this->object_name() . '/apikey',
                        'methods' => array(
                                'create' => array(
                                        'name' => 'Create',
                                        'description' => 'Create a API Key',
                                        'request' => 'POST',
                                ),
                                'retrieve' => array(
                                        'name' => 'Retrieve',
                                        'description' => 'Retrieve a API Key',
                                        'request' => 'GET',
                                ),
                                'update' => array(
                                        'name' => 'Update',
                                        'description' => 'Update a  API Key',
                                        'request' => 'POST',
                                ),
                                'delete' => array(
                                        'name' => 'Delete',
                                        'description' => 'Delete a API Key',
                                        'request' => 'POST',
                                ),
                        ),
                )
        );
        return parent::configure();
    }
    public function is_enabled()
    {
       $cfg  = $this->config();
       return ('true' == $cfg->get_data_value('enable', false));
    }

    public function check_ip_whitelist($ipaddr) {
       $cfg  = $this->config();
       if ('true' == $cfg->get_data_value('ip-whitelist', false)) {
           // Pull out the ip whitelist objects
           $ipaddrs = $this->get_aggregate_objects('ipaddr');
           foreach($ipaddrs as $ip) {
               $w_ip = $ip->get_data_value('ipaddr', false);
               if( $w_ip === $ipaddr){
                   return true;
               }else{
                   if(strpos($w_ip, '/') !== false && safe_cidr_match($ipaddr, $w_ip)){
                       return true;
                   }
               }
           }
           return false;
       }
       return true;
    }
    public function is_api_key_enabled()
    {
        $cfg  = $this->config();
        return ('true' == $cfg->get_data_value('api-key', false));
    }
    public function check_api_key($user_key) {
        // Pull out the key objects
        $keys = $this->get_aggregate_objects('apikey');
        foreach($keys as $key) {
            $key_string = $key->get_data_value('key', false);
            if( $key_string === $user_key){
                return $key;
            }
        }
        return false;
    } 
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_create_ipaddr($name, $data=null) {
        $ipaddr = new Sng_rest_allowed_ip_config_class($this->node(), $this->object_name() . '/ipaddr', $name);
        $ipaddr->configure();
        if($data) {
            $ipaddr->set_data_values($data);
        }
        return $ipaddr;
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_retrieve_ipaddr($name, $data=null) {
        $ipaddrs = $this->get_aggregate_objects('ipaddr');
        return $ipaddrs[$name];
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_update_ipaddr($name, $data=null, &$output = null) {
        $ipaddr = $this->api_retrieve_ipaddr($name);
        if($ipaddr) {
            if ($ipaddr->validate($data,$output)) {
                if (true == $ipaddr->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_delete_ipaddr($name, $data=null, &$output = null) {
        $ipaddr = $this->api_retrieve_ipaddr($name);
        if($ipaddr) {
            if($ipaddr->can_dispose($output)) {
                return $ipaddr->dispose();
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_create_apikey($name, $data=null) {
        $apikey = new Sng_rest_key_config_class($this->node(), $this->object_name() . '/apikey', $name);
        $apikey->configure();
        if($data) {
            $apikey->set_data_values($data);
        }
        return $apikey;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_retrieve_apikey($name, $data=null) {
        $apikeys = $this->get_aggregate_objects('apikey');
        return $apikeys[$name];
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_update_apikey($name, $data=null, &$output = null) {
        $apikey = $this->api_retrieve_apikey($name);
        if($apikey) {
            if ($apikey->validate($data,$output)) {
                if (true == $apikey->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_delete_apikey($name, $data=null, &$output = null) {
        $apikey = $this->api_retrieve_apikey($name);
        if($apikey) {
            if($apikey->can_dispose($output)) {
                return $apikey->dispose();
            }
        }
        return false;
    }
}
/* End of file sng_rest_module_class.php */

