#!/usr/bin/env python
# vim: tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab
"""
* Copyright (C) 2011   Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* William Adam <william.adam@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.
"""

#Include this directory to  python sys.path
import paths

import sys
import os
import subprocess
import re
import xml.dom.minidom as xml
import fs_helper

from optparse import OptionParser
import logging
import logging.handlers

# Usage:
#   fs_trunk_status [Trunk Name]
#      [Trunk Name] = Optional trunk name to check status for. If not specified
#      global status will be checked (ie. all trunks)
# 
# Output format: (This tool is for monit processing)- NO pretty XML print (use
# inline)
"""
<program>  
    <name>Session Statistics</name>
    <message>Output message string </message>
    <status></status>
    <data>
        <count>xx</count>
        <percentage>xx</percentage>
    </data>
</program>
"""
# Upon success, rc=0, data section populated, message set to:
#    OK
#
# Upon limit in percent exceeded, returns -1 and message set to:
#    Active sessions amount of xx% matches resource limit [active sessions>yy%]
#  Where xx is limit specified as argument and yy is current active sessions in %
#
# Upon error, rc=0 (yes I know :) - NO data section - and message set to (fs_cli
# output) example:
#    [ERROR] fs_cli.c:1305 main() Error Connecting [Socket Connection Error]
"""
Sample fs_cli output:
  <gateway>
    <name>Trunk1</name>
    <profile>NSC_SIP_Profile1</profile>
    <scheme>Digest</scheme>
    <realm>sangoma.co</realm>
    <username>w</username>
    <password>yes</password>
    <from>&lt;sip:w@mtl-nsc-devel-will.sangoma.local;transport=udp&gt;</from>
    <contact>&lt;sip:gw+Trunk1@10.20.1.42:5060;transport=udp;gw=Trunk1&gt;</contact>
    <exten>w</exten>
    <to>sip:w@mtl-nsc-devel-will.sangoma.local</to>
    <proxy>sip:mtl-nsc-devel-will.sangoma.local</proxy>
    <context>sip-trunk-Trunk1</context>
    <expires>3600</expires>
    <freq>3600</freq>
    <ping>1381339432</ping>
    <pingfreq>5</pingfreq>
    <state>FAIL_WAIT</state>
    <status>DOWN</status>
    <calls-in>0</calls-in>
    <calls-out>0</calls-out>
    <failed-calls-in>0</failed-calls-in>
    <failed-calls-out>0</failed-calls-out>
  </gateway>
"""    
def do_check(logger, options):
    # Output variables
    total=0
    active=0
    inactive=0
    percent=0.0
    error=None
    limitpercent=None
    status=None
    data=None

    # Process arguments - TODO: use cmd line parsing
    if options.limit:
        limitpercent = float(options.limit)

    # Excute command
    cmd = 'sofia xmlstatus gateway'

    # Process arguments - TODO: use cmd line parsing
    if options.name:
        cmd += ' ' + options.name

    error, out = fs_helper.execute_cmd(cmd)

    # Create output
    doc = xml.Document()
    # inner root
    root = doc.createElement('program');
    s = doc.createElement('name')
    st = doc.createTextNode('Trunk Status')
    s.appendChild(st)
    root.appendChild(s)

    rc = 0
    resources = None
    # Prepare statistics data if no fs_cli error
    if not error:

        # Create xml object
        xml_data = xml.parseString(''.join(out))
        # Convert to dictionary
        _resources = fs_helper.xml_to_dict(xml_data.getElementsByTagName('gateway'))
        if 'gateway' in _resources:
            # Make sure that gateways/gateway is a list (not the case if only one gateway)
            if not isinstance(_resources['gateway'], list):
                resources = [_resources['gateway']]
            else:
                resources = _resources['gateway']

    # process resource(s) status
    if resources:
        total = len(resources)
        active_list =[] 
        inactive_list = []
        # Loop around to count the one UP or down
        for g in resources:
            if g['status'][:2] == 'UP':
                active_list.append(g['name'])
            else:
                inactive_list.append(g['name'])

        active = len(active_list)
        inactive = len(inactive_list)

        # Compute usage %
        if total > 0:
            percent=float((float(active)/float(total))*100)
        else:
            percent=0.0

        inactive_list_str = ','.join(inactive_list)
        active_list_str = ','.join(active_list)
        # Check limit
        if limitpercent:
            error = '%.1f%% Trunk active limit=[%.1f%%] active=[%s] inactive[%s]' \
                % (percent, limitpercent, active_list_str, inactive_list_str)
            if percent <= limitpercent:
                rc = -1
        else:
            error = '%.1f%% Trunk active active=[%s] inactive[%s]' \
                % (percent, active_list_str, inactive_list_str)

        data = doc.createElement('data')
        # Active sessions
        s = doc.createElement('count')
        st = doc.createTextNode(str(active))
        s.appendChild(st)
        data.appendChild(s)
        # Max sessions
        s = doc.createElement('max')
        st = doc.createTextNode(str(total))
        s.appendChild(st)
        data.appendChild(s)
        # Session usage %
        s = doc.createElement('percentage')
        st = doc.createTextNode(str(percent))
        s.appendChild(st)
        data.appendChild(s)

    # Populate message element
    s = doc.createElement('message')
    st = doc.createTextNode(str(error))
    s.appendChild(st)
    root.appendChild(s)
    if data:
        root.appendChild(data)

    # output result
    return rc, root


def main():
    #Parse the options
    usage = "usage: %prog [options] arg"
    optParser = OptionParser(usage)
    optParser.add_option('-n', '--name', action='store', type='string',
                        dest='name', metavar='NAME',
                        help="Resource name")

    optParser.add_option('-l', '--limit', action='store', type='float',
                        dest='limit', metavar='LIMIT',
                        help="Status limit in %")

    optParser.add_option('-p', '--pretty', action='store_true',
                        dest='pretty', metavar='PRETTY',
                        help="Pretty print output")

    (options, args) = optParser.parse_args()

    # Create logger
    logger = logging.getLogger(os.path.basename(os.path.abspath(sys.argv[0])))
    logger.setLevel(logging.INFO)
    handler = logging.StreamHandler()
    logger.addHandler(handler)

    # Execute Check
    rc, root = do_check(logger, options)
    # Check output
    if options.pretty:
        logger.info(root.toprettyxml())
    else:
        logger.info(root.toxml())
    # Report error/success
    sys.exit(rc)


if __name__ == '__main__':
  main()
