#!/bin/bash
# Appliance Update script

# ./update.sh action [base path]
#    action:
#        help        = Display help
#        initialize  = Initialize update archive/directory
#        version     = Get package version info
#        changelog   = Get changelog content
#        check       = Return packages to update
#        start       = Start udpate process (background)
#        status      = Return current update status
#        log         = Return current step log
#        clean       = Clean update archive/directory
#        kill        = Kill update process :)

PERSISTENT_UPDATE_DIR="/tmp/update"

UPDATE_BASE="$(dirname `dirname ${0}`)"
UPDATE_ROOT="$(dirname ${UPDATE_BASE})"

UPDATE_SCRIPTS_DIR=${UPDATE_BASE}/scripts
UPDATE_PACKAGE_DIR=${UPDATE_BASE}/RPMS

if [ -z "$2" ]; then
  UPDATE_WORKING_DIR="${PERSISTENT_UPDATE_DIR}"
else
  UPDATE_WORKING_DIR="$2"
fi

mkdir -p ${UPDATE_WORKING_DIR}

ACTION=$1

case "${ACTION}" in
  'help') 
    echo "help        = Display help"
    echo "initialize  = Initialize update archive/directory"
    echo "version     = Get package version info"
    echo "changelog   = Get changelog content"
    echo "check       = Return packages to update"
    echo "start       = Start udpate process (background)"
    echo "status      = Return current update status"
    echo "log         = Return current step log"
    echo "clean       = Clean update archive/directory"
    echo "kill        = Kill update process"
    ;;
  'version')
    cat ${UPDATE_BASE}/version.txt
    exit 0
    ;;
  'changelog')
    cat ${UPDATE_BASE}/changelog.txt
    exit 0
    ;;
  'initialize')
    # Check not yet initialized
    if [ ! -f "${UPDATE_ROOT}/init" ]; then
      echo "Initializing archive..." > ${UPDATE_ROOT}/init
      # Not yet initilialized, check md5
      if ! ${UPDATE_SCRIPTS_DIR}/md5check.sh ${UPDATE_BASE} >> ${UPDATE_ROOT}/init
      then
        echo "ERROR: MD5 validation failed." >> ${UPDATE_ROOT}/init
        echo "-1" > ${UPDATE_ROOT}/init.rc
      else
        sed -i "s|XXX_SNG_UPDATE_RPMS_DIR_XXX|${UPDATE_PACKAGE_DIR}|g" ${UPDATE_BASE}/conf/*.{repo,conf}
        sed -i "s|XXX_SNG_UPDATE_DIR_XXX|${UPDATE_BASE}|g" ${UPDATE_BASE}/conf/*.{repo,conf}
        echo "Archive successfuly initialized" >> ${UPDATE_ROOT}/init
        echo "0" > ${UPDATE_ROOT}/init.rc
        rm -rf ${UPDATE_WORKING_DIR}
        mkdir -p ${UPDATE_WORKING_DIR}
      fi
    fi
    cat ${UPDATE_ROOT}/init
    rc=`cat ${UPDATE_ROOT}/init.rc`
    exit $rc
    ;;

  'check')
    if [ ! -f "${UPDATE_ROOT}/init" ]; then
      echo "Not initialized"
      rc=-1
    else
      yum -c ${UPDATE_BASE}/conf/yum.conf --enablerepo=update check-update -q > ${UPDATE_WORKING_DIR}/check.txt
      rc=0
      while read i; do
        i=$(echo $i)
        if [ "${i}x" != "x" ]; then
          echo $i | cut -d" " -f1,2
        fi
      done < ${UPDATE_WORKING_DIR}/check.txt
    fi

    exit $rc
    ;;

  'install' | 'start')
    if [ ! -f "${UPDATE_ROOT}/init" ]; then
      echo "Not initialized"
      rc=-1
    else
      # Already done
      if [ -f "${UPDATE_WORKING_DIR}/update.done" ]; then
        rc=`cat ${UPDATE_WORKING_DIR}/update.done`
        cat ${UPDATE_WORKING_DIR}/update.status
      else
        # Started, check running or stuck
        if [ -f "${UPDATE_WORKING_DIR}/update.pid" ]; then
          echo "Running"
          rc=-1
        else
          cmd="${UPDATE_BASE}/scripts/run-update.sh ${UPDATE_BASE} ${UPDATE_WORKING_DIR}"
          if [ "install" == "${ACTION}" ]; then
            ${cmd} | tee ${UPDATE_WORKING_DIR}/update.log
          else
            nohup ${cmd} 1> ${UPDATE_WORKING_DIR}/update.log 2>&1 &
          fi
          rc=$?
        fi
      fi
    fi
    exit $rc
    ;;

  'logs')
    # Return one of the following status
    # 0  - Completed success
    # 1  - Running
    # -1 - Error
    if [ ! -f "${UPDATE_ROOT}/init" ]; then
      echo "Not initialized"
      rc=-1
    else
      cat ${UPDATE_WORKING_DIR}/update.log
      rc=$?
    fi
    exit $rc
    ;;

  'status')
    # Return one of the following status
    # 0  - Completed success
    # 1  - Running
    # -1 - Not Initialized/Dead
    # -2 - Not started
    if [ ! -f "${UPDATE_ROOT}/init" ]; then
      echo "Not initialized"
      rc=-1
    else
      # Already done
      if [ -f "${UPDATE_WORKING_DIR}/update.done" ]; then
        rc=`cat ${UPDATE_WORKING_DIR}/update.done`
        if [ "1" -eq $rc ]; then
          # Need to change (cause 1 means update running)
          rc=-3
        fi
        cat ${UPDATE_WORKING_DIR}/update.status
      else
        # Started, check running or stuck
        if [ -f "${UPDATE_WORKING_DIR}/update.pid" ]; then
          pid=`cat ${UPDATE_WORKING_DIR}/update.pid`
          ps -p ${pid} &> /dev/null
          rc=$?
          if [ 0 == ${rc} ]; then
            cat ${UPDATE_WORKING_DIR}/update.status
            rc=1
          else
            echo "Update process dead"
            rc=-1
          fi
        else
          echo "Not started"
          rc=-2
        fi
      fi
    fi
    exit $rc
    ;;
  'clean' )
    rm -rf  ${UPDATE_WORKING_DIR}
    exit 0
    ;;

  *)
    echo "Invalid action ${ACTION}"
    exit -1
    ;;
esac
