#!/usr/bin/python2.7
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab
"""
* Copyright (C) 2013  Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* Tyler Goodlet <tgoodlet@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.

Backend abstract classes

TODO:
    - DON'T use .toDict() as it is unecessary since HygenicBunch is
      dictlike for all comparison purposes.

DONE- SafepyEngine._set_hook
      needs to be modified to track and push changes in nested subelements.

    - implement pickling methods?
    - should we hook dict.update so instead of
      s.sip.profile.PROF1 = {} we do s.sip.profile.PROF1.update({})

    - can we use a context manager intf to update mutiple subtrees??
    - use dictdiffer to show a diff output from the last state
    - can optimizations still be done with _set_hook regarding the
      number of requests but for now seemingly superfluous calls
      to safepy guarantee correct recursive tree insertion in _push_tree?
      (i.e. 3 gets per creation)
"""
import os, types
from collections import deque, OrderedDict
from itertools import ifilter
from copy import deepcopy

# set up a local logger
import logging
logger = logging.getLogger(__name__)
logger.propagate = False
logger.setLevel(logging.INFO)
console_handler = logging.StreamHandler()
fmt_str = '%(levelname)s %(filename)s %(funcName)s %(lineno)d: %(message)s'
formatter = logging.Formatter(fmt_str)
console_handler.setFormatter(formatter)
logger.addHandler(console_handler)

# package specific
import utils
# import ipdb

class BackendError(Exception): pass

# FIXME: this will most likely need to be implemented with a metaclass
#        in order to reuse the  hygenic bunch again on another backend!!!!
# Now that I think of it this could have been done with simple child
# class simply redefining the methods of interest on the child...

def accessor_decorated_type(cls, proxy_obj,
                            getter,
                            setter,
                            deleter,
                            caller,
                            dirlister=None):

    def new_getitem(self, name):
        return getter(proxy_obj, self, name)
        # return old_getitem(self, name)

    def new_setitem(self, name, value):
        return setter(proxy_obj, self, name, value)
        # return old_setitem(self, name, value)

    def new_delitem(self, name):
        return deleter(proxy_obj, self, name)
        # return old_delitem(self, name)

    def new_call(self, *args, **kwargs):
        return caller(proxy_obj, self, *args, **kwargs)

    # define a metaclass which hooks accessors
    class HookedMeta(type):

        def __init__(cls, name, bases, dct):

            # reassign accessor methods
            cls.__getitem__ = new_getitem
            cls.__setitem__ = new_setitem
            cls.__delitem__ = new_delitem
            cls.__call__    = new_call
            if dirlister:
                def new_dir(self):
                    return dirlister(proxy_obj, self)
                cls.__dir__ = new_dir

            super(HookedMeta, cls).__init__(name, bases, dct)

    # define a hooked bunch type
    class HookedBunch(cls):
        __metaclass__ = HookedMeta

    return HookedBunch

# decorate a hygenic bunch with backend interface hooks
# only useful if all nodes will delegate to the config engine
# in a similar way...
def items_hook_decorate(cls, proxy,
                        getter,
                        setter,
                        deleter,
                        caller,
                        dirlister=None):
    """
    decorate a class with get/set/del[/dir] hooks
    """
    old_getitem = cls.__getitem__
    old_setitem = cls.__setitem__
    old_delitem = cls.__delitem__
    old_dir     = cls.__dir__

    def new_getitem(self, name):
        return getter(proxy, self, name)
        # return old_getitem(self, name)

    def new_setitem(self, name, value):
        return setter(proxy, self, name, value)
        # return old_setitem(self, name, value)

    def new_delitem(self, name):
        return deleter(proxy, self, name)
        # return old_delitem(self, name)

    def new_call(self, *args, **kwargs):
        return caller(proxy, self, *args, **kwargs)

    cls.__getitem__ = new_getitem
    cls.__setitem__ = new_setitem
    cls.__delitem__ = new_delitem
    cls.__call__    = new_call

    if dirlister:
        def new_dir(self):
            return dirlister(proxy, self)
        cls.__dir__ = new_dir

    return cls

'''
TODO:
    - add a safe mode where you can't override modified elements
      without first pushing a new state??
'''
class StateSaver(object):
    '''
    simple 'saver' type to hold sequential changes
    to a state tree

    creates a memoization map of change iterators
    keyed by the 'key_path' (see utils.key_path)
    '''
    def __init__(self, state_tree):
        self.prev_tree = state_tree
        self.kps = []
        self._changes = OrderedDict()
        self._itr_cache = {}
        self.lock = False

    @property
    def changes(self):
        return self._changes

    def reset(self):
        self.kps = []
        self._changes.clear()
        self._itr_cache.clear()
        self.lock = False

    def add_change(self, tree):
        '''
        store a change to the tree
        keyed by the changed node's
        "key path"
        '''
        if self.lock:
            return
        else:
            key_path = utils.key_path(tree)
            self.kps.append(key_path)
            try:
                self.changes[key_path].append(deepcopy(tree))
            except KeyError:
                self.changes[key_path] = [deepcopy(tree)]

    def pop_last_change(self):
        # return the last changed node data
        kp = self.kps.pop()
        tree = self._changes[kp].pop()
        return kp, tree

    def _next(self, kp):
        '''
        memoize iterators for a given
        node using its 'key path'
        '''
        try:
            return next(self._itr_cache[kp])
        except KeyError:
        # no iterator for this kp exists yet...
            logger.debug('inserting new iterator...')
            self._itr_cache[kp] = iter(reversed(self._changes[kp]))
            # discard the most recent change
            next(self._itr_cache[kp])
            return next(self._itr_cache[kp])

    def __iter__(self):
        '''
        iterate changes in lifo order
        without popping them
        '''
        # self._itr_cache.clear()
        for kp in reversed(self.kps):
            try:
                yield kp, self._next(kp)
            except StopIteration as si:
                # yield self.get_first_parent(kp, self.prev_tree)
                try:
                    yield kp, utils.get_from_path(self.prev_tree, kp)
                except KeyError:
                    key = kp[-1:]
                    yield key, self.prev_tree[key[0]]
                    # NOTE: getting here means the last object tree
                    # does not have a path to this node!

                    # fill requested node with empty bunches
                    # until most we reach the child-most element
                    # root = self.prev_tree
                    # tree = root
                    # path = []
                    # for i, key in enumerate(reversed(kp[1:])):
                    # # key = kp[:-1]
                    #     try:
                    #         tree = tree._get_item(key)
                    #     except KeyError:
                    #         logger.debug("inserting bunch to prev_tree...")
                    #         tree._set_item(key, utils.HygenicBunch(key=key))
                    #         index = kp.index(key)
                    #         yield kp[index:] , tree[key] #root[key]
                    #         # yield kp[index:] , root[kp[-1]]
                    #         break


                        # tree = tree._get_item(key)

                    # for i, node in enumerate(reversed(kp)):
                    #     try:
                    #         # should fill out as per engine
                    #         tree = tree[node]
                    #     except KeyError:
                    #         yield kp[i:], tree
                    # provide 'None' to indicate that NO sub-tree
                    # exists for this key path on the last state
                    # ( most likely means it will be deleted shortly )
                    # yield kp, None
                #     continue
                    # this tree dne in the last state
                    # get up the
                    # kp = kp[:-1]
                    # yield 

    def get_first_parent(self, kp, tree):
        # import ipdb
        # ipdb.set_trace()
        ppath = list(kp)
        for node in kp:
            try:
                parent = utils.get_from_path(tree, tuple(ppath))
                return tuple(ppath), parent
            except KeyError:
                ppath.pop()
                continue
        raise ValueError("path to a parent of "+str(kp)+" was found!?")

class BackendInterface(object):
    '''
    An abstract base class which wraps a configuration
    engine and manages its state for easy get/set/restore
    operations by a configurator.

    This class manages nested configuration states and
    restoring after some execution. See a concrete backend
    for specific usage documenation.

    Concrete backends *must* implement a number of mechanisms which provide
        1) a reference to a configuration engine (usually a python lib or module)
        2) which can return a mapping describing an initial configuration state
        3) routines for updating the remote software which can be called
           by get/set hook methods

        The following methods *must* be implemented in the backend:

        * :meth:`retrieve_initial_state`
        * :meth:`apply_changes`
        * :meth:`_bind_engine`

        The following methods *should* be implemented in the backend
        if order for it update the configured device or software

        * :meth:`_get_hook`  which will hook `obj[key]`
        * :meth:`_set_hook`  which will hook `obj[key] = value`
        * :meth:`_del_hook`  which will hook `del obj[key]`
        * :meth:`_call_hook` which will hook `obj.func()`

    '''
    # determines whether to save state after every 'engine action'
    _incremental_saves = False

    def __init__(self, equipment, controller,
                logfile=None,
                logger=logger,
                eng_inst=None,
                tree_type=utils.HygenicBunch,
                *args):

        # set up and acquire initial state
        # decorate our tree container class with hooks
        self._tree_type = accessor_decorated_type(tree_type, self,
                                              self.__class__._get_hook,
                                              self.__class__._set_hook,
                                              self.__class__._del_hook,
                                              self.__class__._call_hook,
                                              dirlister=self.__class__._dir_hook)
        self._ctl           = controller
        self._equipment     = equipment
        self._states        = deque()
        self._initial_state = None
        self._name          = None
        self.log            = logger

        # acquire the engine reference
        if isinstance(eng_inst, type):
            self._engine = eng_inst(equipment, *args)
        elif eng_inst:
            # will be set to None if not passed in init
            self._engine = eng_inst

        # self.writelog("running initialization...")
        self.log.info("entering initialize...")
        self.initialize(version=None)
        if not self.engine:
            raise BackendError("failed to acquire backend engine!?")

        # if not self._initial_state: # if initialize didn't do it...
        try:
            self.state = self._bind_engine(self._get_initial_state())
            self.post_bind()
            self.new_state()
        except NotImplementedError:
            # bind engine may be called in initialize
            self.log.debug('failed to bind')
            # self.writelog("no initial state was acquired...betting on "
            # "initialize having done this...")

    def __dir__(self):
        listing = dir(self.state)
        listing.extend(dir(self.__class__))
        return listing

    def __len__(self):
        return len(self._states)

    def __del__(self):
        self.finalize()

    def _bind_engine(self, root, value=None):
        '''
        Bind the engine to the state tree
        (relies on a pre-provided conf template)

        Here, nodes in the tree-like container should have:
        node.key   = <name of engine root reference>
        node.value = <reference on conf engine corresponding to node.key>
        '''
        for node in utils.dft(root):
            # first iteration
            if node is root:
                if not value:
                    assert root.value == self.engine
                else:
                    root.value = value
            else: # all other iters
                try:
                    node.value = utils.get_component(node.parent.value, node.key)
                except TypeError:
                    continue
        return root

    def _get_name(self):
        '''return the name of this engine'''
        if not self._name:
            self._name = getattr(self._engine, 'name', self.__class__.__name__)
        return self._name

    engine = property(lambda s : s._engine)
    name   = property(_get_name)

    def _get_initial_state(self):
        '''
        Get the initial state from the concrete backend.

        The child classe's method 'retrieve_initial_state'
        MUST return a mapping or other type compatible
        with our 'tree_type' constructor
        '''
        init = self.retrieve_initial_state()
        hb = utils.bunchify(init,
                            key=self._engine.name,
                            cls=self._tree_type,
                            value=self.engine)
        return hb

    # REQUIRED methods for the concrete backend
    def retrieve_initial_state(self):
        '''
        Return the intial server configuration state
        in a mapping container
        '''
        raise NotImplementedError("child's job!")

    def apply_changes(self):
        '''
        push configuration changes to the configured device
        '''
        raise NotImplementedError("child's job!")

    def new_state(self):
        '''
        Save a new state tree and sanitize
        the current tree of all dirtyness

        Returns a reference to the just previous tree
        '''
        curr_state = deepcopy(self.state)
        self._states.append(StateSaver(curr_state))
        utils.sanitize(self.state)
        return curr_state

    @property
    def saver(self):
        return self._states[-1]

    # TODO: options for disabling saver
    def restore(self, state_tree=None):
        '''
        Restore to the previous configuration state

        TODO:
            - implement a mechanism to print out the dirty tree history
        '''
        # if state_tree is None:
        if len(self._states) > 0:
            prev_state = self.saver.prev_tree
        else:
        # sanity check for concrete backends
            raise BackendError("You must set a new state before you can "
                               "restore to it!")

        self.log.debug("RESTORING previous state")

        if not self._incremental_saves:
            self._push_tree(prev_state, absolute=True)
        else:
            # lock out the recording of any changes
            # made by the restore process...
            self.saver.lock = True
            # changes = iter(self.saver)
            # ipdb.set_trace()
            for kp, node in self.saver:#iter(self.saver):
                # if kp == (u'profile', u'sip'):

                # if no tree exists in previous state
                # then delete the heck out of it
                if node is None:
                    key = kp[-1]
                    # del_node = 
                    try:
                        self.log.debug("trying to delete explicit "+key)
                        del self.state[key]
                        self.state._del_item(key)
                    except AttributeError:
                        # no delete method on engine
                        self.log.debug("failed to delete explicit "+key
                                       +" going to recursive del")
                        self._delete_recursive(self.state[key])

                    # delete the parent most node from cache
                    self.state._del_item(key)

                else:
                    self.pre_restore(node, kp)
                    self.log.debug("in restore key path = "+str(kp))
                    self._push_tree(node, path=kp, absolute=True)

        # if there is a stack of states (more then init state)
        # delete (via gc) our current state-tree and point
        # current 'state' to the previous state-tree
        if len(self._states) > 1:
            # last_state = self.state
            last_saver = self._states.pop()
            if not self.state.toDict() == last_saver.prev_tree.toDict():
                for key in last_saver.prev_tree.keys():
                    curr = self.state[key]
                    last = last_saver.prev_tree[key]
                    if curr != last:
                        self.log.warning("branch '"+curr.key+"' restored not exactly "
                                         "the same as in previous state")

                # raise BackendError("failed to restore exactly last obj tree")
            # self.state = self._states.pop().prev_tree

            # bind the engine to the copy if not already
            if not self.state._value:
                self._bind_engine(self.state, self.engine)
        else:
            # only init (server) state is left
            last_saver = self._states[0]
            if not self.state.toDict() == last_saver.prev_tree.toDict():
                for key in last_saver.prev_tree.keys():
                    curr = self.state[key]
                    last = last_saver.prev_tree[key]
                    if curr != last:
                        self.log.warning("branch "+curr.key+" restored not exactly "
                                         " the same as in previous state")
                # raise BackendError("failed to restore exactly last obj tree")
            self.saver.reset()
            # if stack is 'empty' (init state is left)
            # don't ever pop the init state
            utils.sanitize(self.state)

    def pre_restore(self, node, keypath):
        '''
        hook for operations just prior
        to pushing a node to restore
        '''
        pass

    def hard_reset(self):
        '''
        wipe all saved states and
        restore to the initial server state
        '''
        init_state = self._states[0]
        for istate in range(len(self._states)):
            self.restore()

        assert self.state == init_state.prev_tree
        assert len(self._states) == 1
        # self._states.clear()
        # self._states.append(init_state)
        # self.restore()

    def save_to_file(self, path, state=None):
        '''
        save a copy of the current state to file
        '''
        if not state:
            state = self.state
        root, ext = os.path.splitext(path)
        utils.write_config(state, path)

    def _insert_subtree(self, name, tree, eng_obj):
        '''
        insert a new tree-like node assigning
        a representative name for its key and
        an underlying 'engine' component for value

        return a tuple containing the new child tree
        and the corresponding child engine element
        '''
        self.log.debug("in insert subtree "+name+" on eng_obj :"+str(eng_obj))
        eng_chld = utils.get_component(eng_obj, name)
        if eng_chld is None:
            raise ValueError("no child engine element found!")
        # create/instance a new state tree node
        tree._set_item(name,
                       self._tree_type(parent=tree,
                                       key=name,
                                       value=eng_chld))
        return tree._get_item(name), eng_chld


    def _push_tree(self, tree, path=None, absolute=False):
        '''
        Push a new tree as the current config
        This will iterate the diff between trees
        and add/remove nodes as needed

        If 'path' is provided it is assumed that 'tree'
        corresponds to the data that should be pushed
        to the corresponding node in the current state-tree
        '''

        if path:
            # recurse the key path and return current
            # version of that node
            cur_tree = utils.get_from_path(self.state, path)
        else:
            cur_tree = self.state

        if not cur_tree == tree:
            for orig_node, target_node, diff in utils.diff_traversal(cur_tree, tree):
                self.log.debug("performing diff traversal interation...")
                self.log.debug("orig_node = "+str(orig_node.key))
                self.log.debug("target_node = "+str(target_node.key))
                self.log.debug(str(diff)+"\n")

                if absolute:
                    # delete any old branches
                    for key in diff.removed:
                        try:
                            self.log.debug("trying to delete explicit "+key)
                            del orig_node[key]
                        except AttributeError:
                            # no delete method on engine
                            self.log.debug("failed to delete explicit "+key
                                           +" going to recursive del")
                            self._delete_recursive(orig_node[key])

                            # remove form the cache as well...
                            orig_node._del_item(key)
                            if path:
                                utils.trim_empties(orig_node, path[0])

                # insert any new nodes
                for key in diff.added:
                    # if key not in orig_node.treekeys:
                    self.log.debug("adding key "+key+" to "+orig_node.key)
                    try:
                        orig_node[key] = target_node[key].toDict()
                    except SafepyError:
                        # if we can't push the entire sub-tree
                        # try just the leaves
                        orig_node[key] = target_node[key].leavesDict()
                    except AttributeError:
                        # safepy obj node doesn't support push...
                        self.log.debug("update failed for engine node!")
                        self.log.debug("unable to write full dictionary to node!")
                        # val = self._get_hook(orig_node, key)

                # update any sub-trees with differing leaves
                for key, diff_dict in diff.leavesdiff:
                    # if orig_node[key].toDict() != diff_dict:
                    try:
                        # first try assigning the entire sub-tree
                        orig_node[key] = target_node._get_item(key).toDict()
                    except:
                        # settle for the leaves diff
                        orig_node[key] = diff_dict

                # try: # update any immediate leaves
                #     if orig_node != target_node:
                #         orig_node._parent[orig_node._key] = utils.bunchify(target_node.toDict())
                # except AttributeError: # no parent on orig_node
                #     pass

    def _delete_recursive(self, root, del_keys='treekeys'):
        '''
        attempt to delete an entire tree from
        the root downwards
        '''
        for node in utils.dft(root):
            self.log.info("DELETE Recursive node '"+node._key+"'")
            for key in node.treekeys:
                self.log.debug("trying to delete key '"+key+"'")
                try:
                    del node[key]
                    self.log.debug("deleted "+key)
                except (AttributeError, SafepyError):
                    self.log.debug("failed to delete "+key+" skipping...")
                    continue

    def load_config(self, path, abs=False):
        '''
        load a previously saved config and apply it to the device
        '''
        # push current state to the stack
        self.new_state()
        conf = utils.read_config(path)
        self._push_tree(conf, absolute=abs)

    # OPTIONAL methods for the concreate backend
    def initialize(self, version=None):
        '''setup and assign self._initial_state'''
        pass

    def post_bind(self):
        '''
        run some execution just after
        binding the engine to the state tree
        '''
        pass

    def finalize(self):
        '''
        Final teardown...
        feel free to overload this
        '''
        self.hard_reset()

    # optional interceptor methods
    def _get_hook(self, tree, name):
        # return tree._get_item(name)
        return tree[name]

    def _set_hook(self, tree, name, value):
        tree[name] = value

    def _del_hook(self, tree, name):
        del tree[name]

    def _call_hook(self, tree, *args, **kwargs):
        return tree.value(*args, **kwargs)

    def _dir_hook(self, tree):
        return tree._get_item('values')

    # std accessor interfaces
    def get(self, key):
        '''
        get item from engine
        must throw key error if not found
        '''
        try:
            return getattr(self, key)# self.__dict__[key]
        except AttributeError:
            # self.log.debug("caught attr error in get key = "+key)
            try:
                return getattr(self.state, key)
            except AttributeError as ke:
                return self.state[key]

    def set(self, key, value):
        '''
        set item giving priority to the underlying engine
        '''
        try:
            return setattr(self, key, value)
        except AttributeError:
            # self.log.debug("caught attr error in set")
            self.state[key] = value

'''
TODO:
    - how to restore method states? (ex. start/stop status of services)
'''
def get_safepy_mod(prod_name, ver_tuple):
    modname = prod_name+'_'+'_'.join(str(e) for e in ver_tuple)
    if '9_9' in modname:
        modname = prod_name+'_master'
    return __import__(modname)

safepy_type_map = {
        True : 'true',
        False: 'false',
        }

class SafepyError(Exception): pass

# method name enums
creater = 'create'
lister  = 'list'
deleter = 'delete'
getter  = 'retrieve'
setter  = 'update'
childs  = 'children'

# meths that modify
modifiers = {creater, deleter, setter}
# union of all meth names
util_methods = modifiers | {lister, getter}

# stateful method antonyms
service_antonyms = {'start' : 'stop',
                    'stop'  : 'start'}

'''
TODO:
    - figure out to handle method states
    - add convenience routines for adding sip profiles, dialplans, etc
'''
type_map = {False : 'false',
            True  : 'true'}

def convert_types(item):
    '''
    convert python types
    to safepy types
    '''
    if isinstance(item, dict):
    # if issubclass(item.__class__, dict):
        for k,v in item.items():
            try:
                item._set_item(k, convert_types(item._get_item(k)))
            except AttributeError:
                try:
                    item[k] = convert_types(item[k])
                except (KeyError, TypeError):
                    return item
        return item
    else:
        try:
            return type_map[item]
        except (KeyError, TypeError):
            return item

class SafepyEngine(BackendInterface):
    '''
    REST api engine wrapper for products which use the
    safepy interface
    '''
    _rest_port = '80'
    _rest_host = None
    _safepymod = None
    _incremental_saves = True

    def initialize(self, version=None):
        '''
        setup and assign self.engine
        '''
        self.version   = self._ctl.get_sw_version()[:2]
        self.log.info('product version : '+str(self.version))

        # load the safepy module
        if self._safepymod is None:
            prod_name      = self._ctl.name.lower()
            self.safepymod = get_safepy_mod(prod_name, self.version)
        else:
            # take class settings as specified
            self.safepymod = self._safepymod

        # connect safepy
        self._engine   = self.safepymod.Product()

        if self._rest_host is not None:
            host = self._rest_host
        else:
            host = self._equipment.get('hostname')

        # connect safepy
        self._engine.connect(host, self._rest_port)

    def retrieve_initial_state(self):
        '''
        return the intitial server config in a dict
        safepy does this at the root module
        '''
        # FIXME safepy : really shouldn't exist in the api!!!
        # if hasattr(self.engine, 'config'):
        # don't call .config() except for > 2.1
        #if int(self.version[1]) < 1:
        #    conf = {}
        #else:
        conf = self.engine.config()
        if conf is False:
            self._handle_error()
        return conf

    def post_bind(self):
        '''
        gather all children of the root and fill out
        with empty bunches if possible
        '''
        engine = self.state._value
        for key in engine.children.keys():
            if key not in self.state:
                # side effects are that tree nodes
                # are inserted if engine supports it
                self.state[key]

    def _handle_error(self):
        err_dict = self.engine.last_error
        if len(err_dict):
            try:
                error = str(err_dict['error'])
            except KeyError:
                error = ''
            status = str(err_dict['status'])
            raise SafepyError(error+' error with status '+status)

    def _list(self, eng_node):
        """
        handle a call to 'list()' method of a safepy node
        note the hacking around inconsistent behaviour
        of returned values.

        A notable example of this is p.directory.domain.list()
        """
        l = getattr(eng_node, lister)()
        # if not isinstance(l, list):
        if l is False:
            self._handle_error()
            return []
        elif isinstance(l, dict):
            return getattr(eng_node, childs).keys()
        else:
            return l

    def _call_hook(self, tree, *args, **kwargs):
        '''
        if the called engine component is a service
        call make sure to store the status state
        '''
        ret = tree.value(*args, **kwargs)
        if not ret:
            self._handle_error()
        # tree.update(ret)
        return ret

    def _dir_hook(self, tree):
        # self.log.debug("in dir hook..")
        # self.log.debug("tree name is "+tree.key)
        # l = tree.treekeys
        # print("\nsafepy attr listing:\n"+str(l))
        # return l
        eng_obj = tree._value
        if eng_obj is not None:
            l = self._list(eng_obj)
            if l:
                l.sort()
                self.log.info("\nsafepy attr listing:\n"+str(l))
        # return [e for e in tree.keys() if e not in dir(self._tree_type)]
        return tree.keys()

    # item interceptor methods
    def _get_hook(self, tree, name):
        '''
        Try each of these in order until exception...
        1) try to get from state tree, return
        2) check if in .list() then gen new tree-node, return
        3) check if item can be 'retrieved', gen new leaf-node(s), return
        4) else: raise KeyError
        '''
        # self.log.debug("in get hook..name = "+name)
        # ipython related...
        if '_getAttributeNames' in name:
            return None
        # this is somehow magically required (seriously no idea why...)
        elif 'trait_names' in name:
            return None
        # always be transparent with magic/private attrs
        elif '__' in name[:2] or '_' in name[:2]:
            # self.log.debug("caught '_' in name...")
            return tree._get_item(name)

        # we store safepy eng-object in 'value' attribute of tree-node
        # must be explicit here to avoid recursion errors
        eng_obj = tree.__dict__['_value']

        # don't ever track utility methods
        # simply return the corresponding eng node
        if name in util_methods:
            eng_chld = utils.get_component(eng_obj, name)
            # the 'update' module forced my hand on this check
            if isinstance(eng_chld, types.MethodType):
                return eng_chld

        try: # to get from state tree first
            return tree._get_item(name)

        # name wasn't found in the stateful tree
        except KeyError, AttributeError:
            self.log.debug("caught key error in get_hook")
            self.log.debug("'"+name+"' not found in current tree")

            child, eng_chld = self._gather_children(eng_obj, tree, name)

            if child is not None:
                # instead of waiting, populate any immediate children now...
                # Currently this is 100% REQUIRED since safepy
                # does not return the true total config at startup
                # if isinstance(child, self._tree_type):
                    # self._gather_children(eng_chld, child)
                self._do_retrieve(child, eng_chld)

                return tree._get_item(name)
            else:
                self.log.debug("raised key error in get_hook!")
                raise KeyError("no key '"+name+"' defined for tree "+tree.key)


    def _gather_children(self, eng_obj, tree, name=None):
        '''
        gather all relevant children of the current
        node returning the value/node corresponding
        to 'name'
        '''
        # if name is None: name = "None"
        self.log.debug("in gather_children : name is "+str(name)+" tree is "+tree.key+
             " eng obj is "+str(eng_obj))
        target_child = None
        eng_chld     = None
        try:
            children = eng_obj.children
        except AttributeError:
            return None, None

        if name in children:
            if name not in util_methods:
                target_child, eng_chld = self._insert_subtree(name, tree, eng_obj)

        # FIXME: move (if target_child is None) here?
        if lister in children:
            self.log.debug("do GET (i.e. list()) in get_children...")
            entries = self._list(eng_obj)
            new = [e for e in entries if e not in children.keys()]
            if target_child is None and name in new:
                target_child, eng_chld = self._insert_subtree(name, tree, eng_obj)


        # gather leaf nodes using safepy 'retrieve()'
        self.log.debug("entering _retrieve...")
        self._do_retrieve(tree, eng_obj)

        # if child is not None:
        self.log.debug("end of gather_children, targetchild is "
                       +str(target_child)+" eng_chld is "+str(eng_chld))
        return target_child, eng_chld

    def _do_retrieve(self, tree, eng_obj):
        try:
            children = eng_obj.children
            rd = self._retrieve(tree, eng_obj)
            hb = utils.bunchify(rd, parent=tree._parent,
                                cls=self._tree_type)
            tree.update(hb)

        except AttributeError:
            self.log.debug("retrieve not supported!")

    def _retrieve(self, tree, eng_obj):
        '''
        retrieve any leaf nodes that exist
        on the server and insert them in our tree
        '''
        # eng_obj = tree.__dict__['_value']
        # if getter in eng_obj.children:
        self.log.debug("in _retrieve...calling")
        rd = getattr(eng_obj, getter)()
        if not rd: # safepy list() method returns False on failure
            self._handle_error()
        return rd

    def _set_hook(self, tree, name, value):
        '''
        safepy setitem call handler

        will be called on each setitem/setattr
        on the config tree

        Usage cases:
        1) creation or update by dict:
            s.sip.profile['new_profile'] = {'auth-calls' : True,
                                            'max-sessions': '14'}

        2) modification of leaf element: (profile already extant)
            s.sip.profile['new_profile']['auth-calls']  =  True

        TODO:
            - shouldn't we only move forward if the current eng_node
              supports one of the modification methods???
            - most of the subsections here should be factored into
              their own methods. The pythonic eafp isn't so beneficial
              here as REST requests cost us time and money!
              (i.e. only call certain sections if setter in eng_obj.children)
        '''
        self.log.debug("in set_hook name = "+str(name)+" value = "+str(value))
        self.log.debug("in set_hook tree.key = "+str(tree.key))

        eng_obj  = tree.__dict__['_value']
        if eng_obj is None:
        # most likely a nested subtree leaf
        # except KeyError:
            self.log.debug("eng_obj is None for key "+str(name)
                          +" propagating SET to parent...")
            # update a copy and pass to parent
            copy = deepcopy(tree)
            # copy.key = tree.key # FIXME: put in utils??
            copy._set_item(name, value)
            # call the parent to set and then return
            self._set_hook(tree.parent, tree.key, copy.toDict())
            return

        # convert to safepy types
        value = convert_types(value)
        # self.log.debug("coverted value = "+str(value))

        try: # to update
            # NOTE: raises KeyError if dne!
            self._update(tree, name, value)

        # if dne or unable to 'update' try to create...
        # KeyError -> _get_hook raises this if not found
        # AttributeError -> if the method call using getattr fails
        # SafepyError -> if the REST far end reports a 'not found error'
        except (KeyError, AttributeError):

            if creater in eng_obj.children:
                # try to create the value
                self._create(eng_obj, name, value)

                # _get_hook should, implicitly, update our state tree
                # but costs another 2 request...
                self.log.debug("entering get_hook from set_hook creator section")
                val = self._get_hook(tree, name)

                # this MUST save the entire created sub tree
                # FIXME: do we have to save the entire subtree?
                self.saver.add_change(tree)
                # save the node itself with whatever
                # 'out-of-the-box' settings
                # self.saver.add_change(val)
                return

            # there is no 'create' method
            else:
                raise

    def _create(self, eng_obj, name, value):
        '''do a safepy create'''
        # here, 'value' should normally be a dict
        self.log.debug("CREATE object '"+name+"'")
        rd = getattr(eng_obj, creater)(name, value)
        if not rd:
            self._handle_error()

    def _update(self, tree, name, value):
        # assert that the element is in our state tree
        eng_obj = tree._value
        self.log.debug("entering get_hook from set_hook/update")

        #TODO: factor this into _set_hook
        # check with getitem first
        # NOTE: raises KeyError to be caught by caller if dne!
        val = self._get_hook(tree, name)

        # FIXME:
        # if value is {} AND target node is simple type (i.e. bool)
        # then no exception is thrown... ex. conf.sngms.configuration['individual-ip'] = {}
        if isinstance(val, dict):
            if value == {}:
                return

        # try to do an 'update()' if diff exists
        if convert_types(value) != val and value != {}:

            # if returned node is tree-like then update the entire tree
            # ex. usage : module.obj = {'key-element' : 9999}
            if isinstance(val, self._tree_type) and val.value is not None:
                self.log.debug("calling update form 1...")
                rd = getattr(val.value, setter)(value)
                if not rd:
                    self._handle_error()
                # update our state tree as well
                val.update(value)
                self.saver.add_change(tree)
                return

            # if returned node is a single leaf then update ONLY that entry
            # ex. usage : obj['key-element'] = 9999
            else: # val.value is None
                self.log.debug("calling update form 2...")
                rd = getattr(eng_obj, setter)({name: value})
                if not rd:
                    self._handle_error()
                # update our state tree's element
                if isinstance(val, dict):
                    # update the nested dict or bunch
                    val.update(value)
                else:
                    tree._set_item(name, value)

                self.saver.add_change(tree._parent)
                return

        # FIXME: just remove this block?
        # nothing to change
        else:
            self.log.debug("nothing to change since cached tree matches")
            return

    def _del_hook(self, tree, name):
        '''
        You can delete like this...
        del cf.sip.profile.sip1 -> tree=profile, name='sip1'

        NOT like this...
        del cf.sip.profile.sip1['sip-port'] -> tree=sip1, name='sip-port'
        '''
        self.log.debug("in del hook...name = "+name)
        item = tree._get_item(name)
        eng_obj = item.__dict__['_value']
        rd = getattr(eng_obj, deleter)()
        if not rd:
            self._handle_error()

        # delete from local cache
        tree._del_item(name)
        #utils.trim_empties(tree)
        self.saver.add_change(tree)

    def apply_changes(self):
        '''
        push changes to the safepy device either by
        1) reloading or,
        2) stopping, applying, then starting
           (otherwise known as 'apply & restart' in the ui)

        Warning: hardcoded safepy api elements are found here!
        '''
        name = self.state.value.name
        # get controller for the primary service (i.e. 'nsc')
        ctl = self.state[name]
        status = ctl.configuration.status()

        if status['modified']:
            if status['can_reload']:
                ctl.configuration.reload()
                status = ctl.configuration.status()

            if status['modified']:
                self.log.info("Restarting Primary Services!")
                ctl.service.stop()
                ctl.configuration.apply()
                ctl.service.start()
                status = ctl.configuration.status()

            if status['modified']:
                raise BackendError("Failed to apply config changes!?!?")


if __name__ == '__main__':
    pass
