#!/usr/bin/env python2.7
# vim:ts=4:sw=4:softtabstop=4:smarttab:expandtab
"""
* Copyright (C) 2013  Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* Tyler Goodlet <tgoodlet@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.

Push rest configurations via safepy + yaml

"""
import sys
import os
import re
import subprocess
import socket
from optparse import OptionParser

# Create OUR libs path
base = os.path.dirname(os.path.realpath(__file__))
lib_path = os.path.normpath(base + '/../libs')
# Append OUR lib to path
sys.path.append(lib_path)

from configurator import backends
import ipaddr

class SafepyWrapper(backends.SafepyEngine):
    '''
    REST api engine wrapper for products which use the
    safepy interface
    '''
    # connect on 'lo'
    _rest_port = '81'
    _rest_host = '127.0.0.1'

    # this must be assigned to a safepy module instance
    _safepymod = None

# emulating objects expected by backends.SafepyEngine
# pertaining to the QA framework libraries
class FakeEquipment(object):
    data = {}
    data['hostname'] = socket.gethostname()

    def get(self, key):
        return self.data[key]

class FakeController(object):
    name = 'nsc'
    conf = {}
    conf['version_file'] = '/usr/local/sng/conf/nsc-version.xml'

    def shell_command(self, cmd):
        if isinstance(cmd, str):
            command = cmd.split()
        return subprocess.check_output(command)

    def get_sw_version(self):
        '''
        Returns the current software/product version
        in tuple of form:
        (maj, minor, patch, build, version_string)

        Where version_str is of the form 'maj.minor.patch-build'
        '''
        resp = self.shell_command("cat "+self.conf['version_file'])
        match = re.search('.*build_version=\"(.*?)\".*major_version=\"(.*?)\"'
                          '.*minor_version=\"(.*?)\".*patch_version=\"(.*?)\"'
                          '.*product_version=\"(.*?)\"', resp)
        build, maj, minor, patch, version_str = match.groups()
        return maj, minor, patch, build, version_str

def verify_socket_addr(value):
    '''
    Split and return strings of type <xxxx:yy> into tuple
    (xxxx, yy) with value checking
    '''
    if value is None or value is '':
        return None, None
    elif not isinstance(value, str):
        raise TypeError('Must pass a string!')

    if ':' in value:
        ip, port = value.split(':')
        if len(port) == 0:
            port = None
    else:  # either an ip OR port
        try: # value may be just port int
            int(value)
            port = value
            ip = None
        except ValueError:
            ip = value
            port = None

    if ip is not None:
        if ip == '0.0.0.0' or '*' in ip:
            raise ValueError("'" + ip + "' is not a valid ip address!")
        # run the address through ipaddr as a sanity check
        # and to catch any incorrect addresses
        ipaddr.IPv4Address(ip)

    if port == '*':
        raise ValueError("'" + port + "' is not a valid port address!")

    return ip, port

def main(argv):

    parser = OptionParser()
    parser.add_option("-f", "--config-file", dest="conf_path",
                      help="config file (normally YAML doc)")
    parser.add_option("-s", "--server", dest="server_socket",
                      help="Socket address serving the rest connection")
    # parser.add_option("-m", "--module", dest="safe_mod",
    #                   help="safepy version specfic module name")
    parser.add_option("--package", dest="safe_package",
                      help="path to safepy python package")

    (options, args) = parser.parse_args()
    path = options.conf_path
    if path is None:
        raise ValueError("a config file path must be provided!")

    # import the requested safepy module
    package_name = options.safe_package

    # default to product_release
    if package_name is None:
        package_name = 'product_release'

    # import the requested safepy module
    module = __import__(package_name)

    # Do we have the server argument ?
    server_socket = options.server_socket
    if server_socket is None:
        # No server address specify, most likely trying to use local server
        # Try to find out the httpd listen directive
        if os.path.isfile('/usr/webconfig/conf/httpd.d/server_local.conf'):
            # Extract Listen argument
            r = re.compile("Listen\s+?([0-9.:].*)")
            m = r.search( 
                    open(
                        '/usr/webconfig/conf/httpd.d/server_local.conf',
                        "r").read())
            if m:
                server_socket = m.group(1)

    ip, port = verify_socket_addr(server_socket)

    # override connection info if specified
    if ip is not None:
        SafepyWrapper._rest_host = ip
    if port is not None:
        SafepyWrapper._rest_port = port

    # override backend to use a specific safepy module
    SafepyWrapper._safepymod = module

    # instance engine and fakes
    equip = FakeEquipment()
    ctl   = FakeController()
    conf  = SafepyWrapper(equip, ctl)

    # push config using backend
    conf.load_config(path)

    return 0

if __name__ == '__main__':
    try:
        sys.exit(main(sys.argv))
    except SystemExit:
        raise
    except Exception, e:
        sys.stderr.write("Exception caught: %s\n" % (e))
        raise
