<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * FS distributor module class
*/
require_once ('application/helpers/safe_helper.php');
safe_require_class('safe_module_class');

class Fs_distributor_resource_config_class extends Safe_configurable_object_class
{
    /**
     * @brief Constructor
     *           
     * @param[in out] $parent_name
     * @param[in out] $name
     *           
     * @return
     */
    public function __construct($node, $parent_name, $name)
    {
        parent::__construct($parent_name, $name, $node);
    }
    /**
     * @brief Configure the config class object
     *         
     * @return
     */
    public function configure()
    {
        // General settings
        $this->add_field('resource', 'Resource', 'text', '',50);
        $this->set_field_rules('resource', 'required');
        $this->set_field_help('resource', 
            array( 
                'Resource Identifier.'
            )
        );
        $this->add_field('weight', 'Weight', 'text', '1',10);
        $this->set_field_rules('weight', 'required|integer|greater_than[0]');
        $this->set_field_help('weight', 
            array( 
                'Weight of the resource.',
                'Defines the number of time this Resource will be returned before balancing to next resource in list.'
            )
        );
        $this->add_field('description', 'Description', 'text', '',50);
        return parent::configure();
    }

    public function weight()
    {
        return $this->get_data_value('weight', false);
    }
    public function resource()
    {
        return $this->get_data_value('resource', false);
    }
    public function summary($type = 'horizontal' , $long = false){
        $table_line = parent::summary($type , $long);
        array_shift($table_line['data']);
        return $table_line;
    }
    
}
class Fs_distributor_generic_resource_config_class  extends Fs_distributor_resource_config_class
{

}
class Fs_distributor_trunk_resource_config_class extends Fs_distributor_resource_config_class
{
    public function configure()
    {
        // General settings
        $names = array();
        //check which trunk already bind to carrier
        $sip_trunks = $this->_node->software()->application()->sip_trunks();
        foreach ($sip_trunks as $trunk) {
            $names[$trunk->name()] = $trunk->name();
        }
        // SIP trunk section
        $this->add_enum_field('trunk', 'SIP Trunk', 'dropdown', '', $names);
        $this->set_field_rules('trunk', 'required[,in,none]');
        $this->set_field_help('trunk', array(
                'Select a SIP Trunk to bind to this Load Balancing list.',
                'Selecting None to not bind to any.'));
        $this->add_field('weight', 'Weight', 'text', '1',10);
        $this->set_field_rules('weight', 'required|integer|greater_than[0]');
        $this->set_field_help('weight',
                array(
                        'Weight of the resource.',
                        'Defines the number of time this Resource will be returned before balancing to next resource in list.'
                )
        );
        $this->add_field('description', 'Description', 'text', '',50);
    }
    public function resource()
    {
        return $this->get_data_value('trunk', false);
    }
    
}
class Fs_distributor_list_config_class extends Safe_configurable_object_class
{
    /**
     * @brief Constructor
     *
     * @param[in out] $parent_name
     * @param[in out] $name
     *
     * @return
     */
    public function __construct($node, $parent_name, $name)
    {
        parent::__construct($parent_name, $name, $node);
    }
    /**
     * @brief Configure the config class object
     *
     * @return
     */
    public function configure()
    {
        // General settings
        $this->add_enum_field('enable', 'Enable List', 'dropdown', 'true', $this->enable_disable_enum());
        $this->set_field_help('enable', 
            array( 
                'Enable/Disable List from being used by Load Balancing module.'
            )
        );
        $this->add_field('description', 'Description', 'text', '',50);

        // Register objects
        $this->register_aggregate_object('resource', 
            array(
                'name' => 'Resource',
                'base_path' => $this->object_name() . '/resource',
                'dynamic' => true,
                /* Add some extra infos for config_manager create object */
                'autoname' => true, 
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Create a Resource',
                        'request' => 'POST',
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a Resource',
                        'request' => 'GET',
                        ),
                    'update' => array(
                        'name' => 'Update',
                        'description' => 'Update a Resource',
                        'request' => 'POST',
                        ),
                    'delete' => array(
                        'name' => 'Delete',
                        'description' => 'Delete a Resource',
                        'request' => 'POST',
                        ),
                    ),
            )
        );
        return parent::configure();
    }
    public function description()
    {
        $desc = $this->get_data_value('description', false);
        if($desc){
            return $desc;
        }
        return parent::description();
    }
    public function resources()
    {
        $nodes = $this->get_aggregate_objects('resource');
        return $nodes;
    }

    public function is_enabled()
    {
        $data = $this->get_data_values(false);
        return ('true' == $data['enable']);
    }
    public function weight()
    {
        $weight = 0;
        foreach($this->resources() as $node){
            $weight += $node->weight();
        }
        return $weight;
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_create_resource($name, $data=null) {
        $condition = new Fs_distributor_resource_config_class($this->node(), $this->object_name() . '/resource', $name);
        $condition->configure();
        if($data) {
            $condition->set_data_values($data);
        }
        return $condition;
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_retrieve_resource($name, $data=null) {
        $nodes = $this->resources();
        return $nodes[$name];
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_update_resource($name, $data=null, &$output = null) {
        $resource = $this->api_retrieve_resource($name);
        if($resource) {
            if ($resource->validate($data,$output)) {
                if(true == $resource->save()){
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_delete_resource($name, $data=null, &$output = null) {
        $nodes = $this->resources();
        if($nodes[$name]) {
            if($nodes[$name]->can_dispose($output)) {
                return $nodes[$name]->dispose();
            }
        }
        return false;
    } 
    /**
     * @brief Delete an object
     *         Mark the object as deleted
     *         
     * @return
     *         true on success
     */
    public function dispose()
    {
        foreach($this->resources() as $node){
            $node->dispose();
        }
        return parent::dispose();
    }
    public function get_prefix_string(){
        return '${distributor '.$this->name().'}/';
    }
    public function get_resource_type()
    {
        return '';
    }
}
class Fs_distributor_config_class extends Safe_configurable_object_class
{
    /**
     * @brief Constructor
     *           
     * @param[in out] $parent_name
     * @param[in out] $name
     *           
     * @return
     */
    public function __construct($node, $parent_name)
    {
        parent::__construct($parent_name, "configuration");
    }
    /**
     * @brief Configure the config class object
     *         
     * @return
     */
    public function configure()
    {
        // General settings
        $this->add_enum_field('enable', 'Enable Load Balancing', 'dropdown', 'false', $this->enable_disable_enum());
        $this->set_field_help('enable', 
            array( 
                'Enable/Disable Balancing module.'
            )
        );

        return parent::configure();
    }
}

class Fs_distributor_list_generic_class  extends Fs_distributor_list_config_class
{
    public function api_create_resource($name, $data=null) {
        $condition = new Fs_distributor_generic_resource_config_class($this->node(), $this->object_name() . '/resource', $name);
        $condition->configure();
        if($data) {
            $condition->set_data_values($data);
        }
        return $condition;
    }
    public function can_dispose(&$reason = null)
    {
        $usages = $this->node()->software()->application()->check_object_usage('distributor','generic', $this->name());
        if($usages){
            $reason['msg'] = "  {$this->name()} is used by ";
            $reason['obj'] = $usages;
            return false;
        }
        return parent::can_dispose($reason);
    }
    public function get_resource_type()
    {
        return 'generic';
    }
}
class Fs_distributor_list_trunk_class  extends Fs_distributor_list_config_class
{
    public function api_create_resource($name, $data=null) {
        $condition = new Fs_distributor_trunk_resource_config_class($this->node(), $this->object_name() . '/resource', $name);
        $condition->configure();
        if($data) {
            $condition->set_data_values($data);
        }
        return $condition;
    }
    public function get_resource_type()
    {
        return 'trunk';
    }
    public function get_prefix_string()
    {
        return 'sofia/gateway/${distributor '.$this->name().' ${sofia gwlist down}}/';
    }
    public function can_dispose(&$reason = null)
    {
        $usages = $this->node()->software()->application()->check_object_usage('distributor','trunk', $this->name());
        if($usages){
            $reason['msg'] = "  {$this->name()} is used by ";
            $reason['obj'] = $usages;
            return false;
        }
        return parent::can_dispose($reason);
    }
    public function can_create_aggregate_object($obj_type=null, $name, &$output = null){
        if($obj_type == 'resource'){
            $err_msg = array();
            $media_module = $this->node()->software()->application()->find_module_by_name('sip');
            if(count($media_module->get_aggregate_objects('trunk')) <= 0){
                $err_msg[] = 'No trunk defined.';
            }
            if($err_msg){
                $output =  $err_msg;
                return false;
            }
        }
        return parent::can_create_aggregate_object($obj_type, $name, $output);
    }
}


class Fs_distributor_module_class extends Fs_module_class
{
    protected $_obj_base_path = '';
    /**
     * @brief
     *           
     * @param[in out] $fs_app
     *           
     * @return
     */
    public function __construct($fs_app)
    {
        parent::__construct($fs_app, "distributor");
        $this->_obj_base_path = $this->path().'/distributor';
    }
    /**
     * @brief
     *         
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("Load Balancing");

        // Create the config object
        $cfg = new Fs_distributor_config_class($this->node() , $this->_obj_base_path, $this->name());
        // Synch with DB
        $cfg->configure();
        $cfg->synch();
        // Attach config to module
        $this->set_config($cfg);
        // Register objects
        $this->register_aggregate_object('generic', 
            array(
                'name' => 'Generic list',
                'base_path' => $this->_obj_base_path . '/generic',
                'dynamic' => true,
                'has_child' => true,
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Create a Generic List',
                        'request' => 'POST',
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a Generic List',
                        'request' => 'GET',
                        ),
                    'update' => array(
                        'name' => 'Update',
                        'description' => 'Update a Generic List',
                        'request' => 'POST',
                        ),
                    'delete' => array(
                        'name' => 'Delete',
                        'description' => 'Delete a Generic List',
                        'request' => 'POST',
                        ),
                    ),
            )
        );
        $this->register_aggregate_object('trunk',
                array(
                        'name' => 'SIP Trunk List',
                        'base_path' => $this->_obj_base_path . '/trunk',
                        'dynamic' => true,
                        'has_child' => true,
                        'methods' => array(
                                'create' => array(
                                        'name' => 'Create',
                                        'description' => 'Create a Trunk List',
                                        'request' => 'POST',
                                ),
                                'retrieve' => array(
                                        'name' => 'Retrieve',
                                        'description' => 'Retrieve a Trunk List',
                                        'request' => 'GET',
                                ),
                                'update' => array(
                                        'name' => 'Update',
                                        'description' => 'Update a Trunk List',
                                        'request' => 'POST',
                                ),
                                'delete' => array(
                                        'name' => 'Delete',
                                        'description' => 'Delete a Trunk List',
                                        'request' => 'POST',
                                ),
                        ),
                )
        );

        return parent::configure();
    }
    public function generics()
    {
        return $this->get_aggregate_objects('generic');
    }
    public function trunks()
    {
        return $this->get_aggregate_objects('trunk');
    }
     public function lists($type = null)
     {
         if(isset($type) && $type!=''){
             return $this->get_aggregate_objects($type);
         }else{
             return array_merge($this->get_aggregate_objects('generic') , $this->get_aggregate_objects('trunk'));
         }
     }
    /**
     * Decides whether the module is enabled or disabled
     * @see Fs_module_class::is_enabled()
     */
    public function is_enabled($mod_name = "")
    {
        // Always enable mod_distributor, so it can support reload
        return true;
    }

    public function can_create_aggregate_object($type, $name, &$reason="")
    {
        // Default implementation - Check object name exists (but check in 
        // merged list)
        $objs = $this->lists();
        if($objs[$name]){
            $reason = "Already exists";
            return false;
        }
        return parent::can_create_aggregate_object($type, $name, $reason);
    }

    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_create_generic($name, $data=null) {
        $generic = new Fs_distributor_list_generic_class($this->node(), $this->_obj_base_path . '/generic', $name);
        $generic->configure();
        if($data) {
            $generic->set_data_values($data);
        }
        return $generic;
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_retrieve_generic($name, $data=null) {
        $generics = $this->generics();
        return $generics[$name];
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_update_generic($name, $data=null, &$output = null) {
        $generic = $this->api_retrieve_generic($name);
        if($generic) {
            if ($generic->validate($data,$output)) {
                if (true == $generic->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief 
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return 
     */
    public function api_delete_generic($name, $data=null, &$output = null) {
        $generics = $this->generics();
        if($generics[$name]) {
            if($generics[$name]->can_dispose($output)) {
                foreach ($generics[$name]->has_aggregate_objects(false, true) as $_type => $_definition){
                    if($_definition['configurable']){
                        foreach ($generics[$name]->get_aggregate_objects($_type) as $sub_obj_name => $sub_object) {
                            $sub_object->dispose();
                        }    
                    }
                }
                return $generics[$name]->dispose();
            }
        }
        return false;
    }
    
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_create_trunk($name, $data=null) {
        $trunk = new Fs_distributor_list_trunk_class($this->node(), $this->_obj_base_path . '/trunk', $name);
        $trunk->configure();
        if($data) {
            $trunk->set_data_values($data);
        }
        return $trunk;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_retrieve_trunk($name, $data=null) {
        $trunks = $this->trunks();
        return $trunks[$name];
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_update_trunk($name, $data=null, &$output = null) {
        $trunk = $this->api_retrieve_trunk($name);
        if($trunk) {
            if ($trunk->validate($data,$output)) {
                if (true == $trunk->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     *
     * @return
     */
    public function api_delete_trunk($name, $data=null, &$output = null) {
        $trunks = $this->trunks();
        if($trunks[$name]) {
            if($trunks[$name]->can_dispose($output)) {
                foreach ($trunks[$name]->has_aggregate_objects(false, true) as $_type => $_definition){
                    if($_definition['configurable']){
                        foreach ($trunks[$name]->get_aggregate_objects($_type) as $sub_obj_name => $sub_object) {
                            $sub_object->dispose();
                        }    
                    }
                }
                return $trunks[$name]->dispose();
            }
        }
        return false;
    }
    /**
     * @brief Create config jobs for dialplan module
     *           
     * @param[in out] $config_manager
     *           
     * @return
     */
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        if(!$this->_generate_config($config_manager)) return false;
        return parent::reload_generate_config($config_manager, $obj_type);
    }

    /**
     * @brief 
     *
     * @param[in out] $obj_type
     *
     * @return 
     */
    public function reload_clear_modified($obj_type=null)
    {
        return $this->clear_configuration_modified();
    }

    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    public function generate_config(&$config_manager)
    {
        if(!$this->_generate_config($config_manager)) return false;
        return parent::generate_config($config_manager);
    }
    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    private function _generate_config(&$config_manager)
    {
        // Get our configuration data (unresolved)
        $cfg = $this->config()->get_data_values(false);
        // Create xml document
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0', 'UTF-8');
        $xml_writer->startElement('configuration');
        $xml_writer->writeAttribute('name', 'distributor.conf');
        $xml_writer->writeAttribute('description', 'Distributor Configuration');
        $xml_writer->startElement('lists');
        if('true' == $cfg['enable']){
            // Process all lists
            $lists = array_merge($this->generics() , $this->trunks());
            foreach($lists as $list){
                if($list->is_enabled()){
                    $xml_writer->startElement('list');
                    $xml_writer->writeAttribute('name', $list->name());
                    $xml_writer->writeAttribute('total-weight', $list->weight());
                    // Add nodes
                    foreach($list->resources() as $node){
                        $xml_writer->startElement('node');
                        $xml_writer->writeAttribute('name', $node->resource());
                        $xml_writer->writeAttribute('weight', $node->weight());
                        $xml_writer->endElement(); // Close node
                    }

                    $xml_writer->fullEndElement(); // Close list
                }
            }
        }

        $xml_writer->fullEndElement(); // Close lists
        $xml_writer->fullEndElement(); // Close Configuration
        // Content
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = 'autoload_configs/distributor.conf.xml';
        $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));
        return true;
    }
    /**
     * @brief Invoked after a successfull write_config
     *         
     * @return
     */
    public function post_write_config()
    {
        // Reload configuration
        if(!$this->fs_app()->reloadxml()) return false;
        $output = array();
        $rc = $this->eslapi_execute("distributor_ctl reload", '', $output);
        if (!$rc) return $rc;
        return parent::post_write_config();
    }
    public function support_reload(&$cfg_modified = null)
    {
        parent::support_reload($cfg_modified);
        return true;
    }
    public function check_object_usage($src_module_name, $obj_type, $obj_name, $sub_type = null, $sub_name = null) {
        // check usages ?
        $usages = array();
        $trunks = $this->trunks ();
        foreach ( $trunks as $trunk ) {
            foreach ( $trunk->resources () as $resource ) {
                $data = $resource->get_data_value ( 'trunk', false );
                if ($data == 'none') {
                    continue;
                } else {
                    if ($data == $obj_name && $src_module_name == 'sip' && $obj_type == 'trunk') {
                        $usages[] = array (
                                'name' => $this->name(),
                                'obj_type' => 'trunk',
                                'obj_name' => $trunk->name(),
                                'sub_type' => 'resource',
                                'sub_name' => $resource->name() 
                        );
                    }
                }
            }
        }
        return $usages;
    }
    public function get_object_enum($type)
    {
        if($type == 'lists'){
            return array_merge(parent::get_object_enum('generic') , parent::get_object_enum('trunk'));
        }else{
            return parent::get_object_enum($type);
        }
    }
    public function new_fields(&$config_manager, $type)
    {
        $obj_def = $this->get_aggregate_object_definition($type);
        $label = 'New ' . $obj_def['name'];
        $number = count($this->get_aggregate_objects($type))+1;
        if($type == 'trunk'){
            $new_name = 'SIP_Trunk_List_' . $number;
        }elseif($type == 'generic'){
            $new_name = 'Generic_List_' . $number;
        }
        $config_manager->add_field('profile-name', $label, 'string', $new_name, 50);
        $config_manager->set_field_rules('profile-name', 'required|alpha_dash');
    }
}

