<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/**
 * SNG SSH Linux service wrapper
 *
 * @author William Adam
 * @version
 */
if (!defined('BASEPATH')) exit('No direct script access allowed');
require_once ('application/helpers/safe_helper.php');
safe_require_class('service');
safe_require_class('auditor');

class Sng_notifier_config_class extends Safe_configurable_object_class
{
    public function __construct($node, $parent_name, $name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, "configuration", $node);
    }
    public function configure()
    {
        // Panel section
        /* Not available for now
        $this->add_enum_field('panel/enable', 'Notification Panel', 'dropdown', 'false',$this->enable_disable_enum());
        $this->add_field('panel/max_msg', 'Maximum Messages', 'text', '120','5');
        $this->add_field('panel/refresh', 'Refresh Frequency (sec)', 'text', '120','5');

        $this->create_group('panel',
            array(
                'panel/max_msg',
                'panel/refresh',
            )
        );

        $this->conditional_control('panel/enable', 'panel');
        $this->set_field_category('panel/enable', 'Panel');
        */
        // Email section
        $this->add_enum_field('email/enable', 'Email Notification', 'dropdown', 'false',$this->enable_disable_enum());
        $this->set_field_help('email/enable', 'Enable/Disable Email notification.');
        $this->add_field('email/smtp/server', 'SMTP Server Address', 'text', '','70');
        $this->set_field_rules('email/smtp/server', 'required|ssl_ip_or_domain');
        $this->set_field_help('email/smtp/server',
            array('SMTP server address.',
                'Use IP address or FQDN.',
                'To enable secure(SSL) connection please add ssl:// prefix.') 
            );
        $this->add_field('email/smtp/port', 'SMTP Server Port', 'text', '25','10');
        $this->set_field_rules('email/smtp/port', 'valid_port_number');
        $this->set_field_help('email/smtp/port',
            array('SMTP server port.',
                'Non-encrypted connection usually uses port 25 or 587.',
                'Secure (SSL) connection usually uses port 465.')
            );
        $this->add_field('email/smtp/user', 'SMTP User', 'text', '','30');
        $this->set_field_help('email/smtp/user','SMTP user account identifier.');
        $this->add_field('email/smtp/password', 'SMTP Password', 'password', '','30');
        $this->set_field_help('email/smtp/password','SMTP user account password.');

        $this->create_group('email',
            array(
                'email/smtp/server',
                'email/smtp/port',
                'email/smtp/user',
                'email/smtp/password',
            )
        );

        $this->conditional_control('email/enable', 'email');
        $this->set_field_category('email/enable', 'Email');
         
        // Cloud section
        /* Not available now
        $this->add_enum_field('cloud/enable', 'Cloud Notification', 'dropdown', 'false',$this->enable_disable_enum());
        $this->add_field('cloud/address', 'Server Address', 'text', 'https://samms.sangoma.com ;)','70');

        $this->create_group('cloud',
            array(
                'cloud/address',
            )
        );

        $this->conditional_control('cloud/enable', 'cloud');
        $this->set_field_category('cloud/enable', 'Cloud');
        */
        return parent::configure();
    }
    /**
        * @brief Retrieve email configuration
        *
        * @return array with smtp info if enable or empty array if email 
        * disabled
     */
    public function email_config()
    {
        $cfg = $this->get_data_values_exploded(false);
        if ($cfg['email']['enable'] === 'true'){
            return $cfg['email'];
        }else{
            return array();
        }
    }
    public function summary($type = 'horizontal' , $long = false)
    {
        // TODO: Override to display only notification enabled flags
        if(!$this->email_config()){
            return parent::summary($type, $long, 1);
        }else{
            return parent::summary($type, $long);
        }
    }
}

class Sng_notifier_rule_class extends Safe_configurable_object_class
{
    public function __construct($node, $parent_name, $name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, $name, $node);
    }
    public function configure()
    {
        // Get Module list with audit points
        $service_list['__any__'] = '( All Services )';
        $modules = $this->node()->modules();
        foreach($modules as $module){
            $audit_points = array();
            $module->get_audit_points($audit_points);
            if($audit_points){
                $service_list[$module->name()] = $module->description();
            }
        }

        $this->add_enum_field('service', 'Service', 'dropdown', '__any__',$service_list);
        $this->set_field_help('service','Select the service that need to send notifications.');
        $severity = array(
            'info' => 'Information',
            'error' => 'Error',
            'critical' => 'Critical',
        );

        $this->add_multiple_field('severity', 'Severity', 'inline_checkbox', array(),$severity);
        $this->set_field_help('severity','Select the severity level need to send notifications.');
        $this->set_field_rules('severity', 'required[,in,__empty__]');
        $action = array(
            'email' => 'Email',
            'hourly' => 'Hourly Report',
            'daily' => 'Daily Report',
        );
        //$this->composite_layout('service', array('severity'));

        $this->add_enum_field('action', 'Action', 'dropdown', '',$action);
        $this->set_field_help('action','Select the action type of the notifications.');
        // Get user list from system
        $system = $this->node()->find_module_by_name('system');
        $users = $system->api_retrieve_user(null);
        $user_list = array('__all__' => '( All Users )');
        foreach($users as $user){
            $user_list[$user->name()] = $user->description();
        }
        $this->add_enum_field('user', 'User', 'dropdown', '__all__',$user_list);
        $this->set_field_help('user','Select the users to receive the notifications.');
        //$this->composite_layout('action', array('user'));

        return parent::configure();
    }
    public function summary($type = 'horizontal' , $long = false){
        $table_line = parent::summary($type , $long, 5);
        array_shift($table_line['data']);
        return $table_line;
    }
    public function save()
    {
        parent::save();
        $this->clear_modified();
        return true;
    }
    public function validate($data, &$output){
        if(!isset($data['severity'])){
            //if unchecked all web does not post 'severity' to server
            $data['severity'] = '';
            $_POST['severity'] = '';
        }
        return parent::validate($data, $output);
    }
}

class Sng_notifier_service_class extends Safe_service_class
{
    public function __construct($software)
    {
        parent::__construct($software, 'notifier');
    }
    /**
     * @brief
     *
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("Notifier");
        
        // Create the config object
        $cfg = new Sng_notifier_config_class($this->node(), $this->path() . '/' .$this->name());
        // Synch with DB
        $cfg->configure();
        $cfg->synch();
        // Attach config to module
        $this->set_config($cfg);

        // Register objects
        $this->register_aggregate_object('rule', 
            array(
                'name' => 'Notification Rule',
                'base_path' => $this->object_name() . '/rule',
                'dynamic' => true,
                'autoname' => true,
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Create a Rule',
                        'request' => 'POST'
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a Rule',
                        'request' => 'GET'
                        ),
                    'update' => array(
                            'name' => 'Update',
                            'description' => 'Update a Rule',
                            'request' => 'POST'
                    ),
                    'delete' => array(
                            'name' => 'Delete',
                            'description' => 'Delete a Rule',
                            'request' => 'POST'
                    )
               )
            )
        );
        return parent::configure();
    }
    public function get_action_settings()
    {
        $email_cfg = $this->config()->email_config();
        $_app = $this->node()->software()->application();
        if($email_cfg)
        {
            //check if smtp is enabled
            $smtp_test_param =  new Safe_configurable_object_class('autofill', 'smtp_test_param');
            $smtp_test_param->add_field('email', 'Email Address', 'text', '','50');
            $smtp_test_param->set_field_rules('email','required|valid_email');
            $data = array(
                    'test' => array(
                            'name' => 'Test',
                            'description' => 'Notifier test',
                            'request' => 'POST',
                            'param' => $smtp_test_param
                           )
            );
            return $data;
        }else{
            return null;
        }
    }
    public function api_test_action($data=null, &$output = null) {
        $action_settings = $this->get_action_settings();
        $action_cfg = $action_settings['test'];
        $param = $action_cfg['param'];
        if ($param->validate($data, $output)) {
            $rs =  $this->test($data, $output);
            return $rs;
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_rule($name, $data=null, &$output = null) {
        $rule = new Sng_notifier_rule_class($this->node(), $this->object_name().'/rule', $name);
        $rule->configure();
        if($data) {
            $rule->set_data_values($data);
        }
        return $rule;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_rule($name, $data=null, &$output = null) {
        $objs = $this->get_aggregate_objects('rule');
        if(!$name){
            return $objs;
        }else{
            return $objs[$name];
        }
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_rule($name, $data=null, &$output = null) {
        $rule = $this->api_retrieve_rule($name);
        if($rule) {
            if ($rule->validate($data,$output)) {
                if (true == $rule->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_rule($name, $data=null, &$output = null) {
        $rule = $this->api_retrieve_rule($name);
        if($rule) {
            if($rule->can_dispose($output)) {
                return $rule->dispose();
            }
        }
        return false;
    }

    public function allow_user_ctl()
    {
        return 'hide';
    }
    public function allow_dashboard()
    {
        return false;
    }
    /**
     * @brief Sends notification according to configuration
     *
     * @param[in out] $level
     * @param[in out] $events
     *
     * @return 
     */
    public function notify($level, $events)
    {
        $email_cfg = $this->config()->email_config();
        $_app = $this->node()->software()->application();
        if($email_cfg)
        {
            // Filter out not enabled events (action is ignore)
            $_events = array();
            foreach($events as $event){
                if($event->is_enabled()){
                    $_events[] = $event;
                }
            }
            if(!$_events){
                return;
            }
            $CI = & get_instance();
            // Get system module
            $system = $_app->find_module_by_name('system');
            // Get root user
            $root = $system->api_retrieve_user('root');
            // Get httpd module to retrieve server URI
            $host = $this->node()->hostname();
            $httpd = $_app->find_module_by_name('webconfig');
            $uri = $httpd->server_uri($host);
            $email_lib = $CI->load->library('email');
            // Protocol settings
            $config['protocol'] = 'smtp';
            $config['smtp_host'] = $email_cfg['smtp']['server'];
            $config['smtp_port'] = $email_cfg['smtp']['port'];
            $config['smtp_user'] = $email_cfg['smtp']['user'];
            $config['smtp_pass'] = $email_cfg['smtp']['password'];
            // Layout settings
            $config['mailtype'] = 'html';
            $config['wordwrap'] = false;
            $config['crlf'] = "\r\n";
            $config['newline'] = "\r\n";
            $email_lib->initialize($config);
            // Get notifier rules
            $rules = $this->api_retrieve_rule(null);
            if($rules){
                // Get notifier users
                $users = $system->api_retrieve_user();
                foreach($rules as $rule){
                    // Check rule is email ?
                    $action = $rule->get_data_value('action', false);
                    if('email' !== $action){
                        continue;
                    }
                    // Match severity ?
                    $severity = $rule->get_data_value('severity', false);
                    if(false !== array_search($level, $severity)){
                        // Check if service filter
                        $service = $rule->get_data_value('service', false);
                        if('__any__' != $service){
                            $filtered_events = array();
                            foreach($_events as $event){
                                if($event->module()->name() == $service){
                                    $filtered_events[] = $event;
                                }
                            }
                            if(!$filtered_events){
                                continue;
                            }
                            $_events = $filtered_events;
                        }
                        // Build recipient list
                        $user = $rule->get_data_value('user', false);
                        if('__all__' != $user){
                            $recipients[] = $users[$user];
                        }else{
                            $recipients = $users;
                        }
                        foreach($recipients as $user){
                            //  prepare and send email
                            $email_lib->from($root->email(), $root->description());
                            $email_lib->reply_to($root->email(), $root->description());
                            $email_lib->to($user->email()); 
                            
                            // Eamil body
                            $email_lib->subject($_app->description().' - '.ucfirst($level).' Notification');
                            $data['title'] = $_app->description().' - '.ucfirst($level).' Notification';
                            $data['product'] = $_app->description();
                            $data['level'] = ucfirst($level);
                            $data['events'] = $_events;
                            $data['host'] = $host;
                            $data['server_uri'] = $uri;
                            error_log('INFO - Sending '.$level.' notification to '.$user->email());
                            $email_lib->message($CI->load->view('sng_notifier_email', $data, true));
                            if(!$email_lib->send()){
                                error_log('ERROR - SMTP Error - '.implode(', ', $email_lib->get_error()));
                                file_put_contents('/tmp/email.html', $email_lib->print_debugger());
                            }
                        }
                    }
                }
            }

        }
    }
    public function report($type, $events)
    {
        $email_cfg = $this->config()->email_config();
        $_app = $this->node()->software()->application();
        if($email_cfg)
        {
            $CI = & get_instance();
            // Get system module
            $system = $_app->find_module_by_name('system');
            // Get root user
            $root = $system->api_retrieve_user('root');
            // Get httpd module to retrieve server URI
            $host = $this->node()->hostname();
            $httpd = $_app->find_module_by_name('webconfig');
            $uri = $httpd->server_uri($host);
            $email_lib = $CI->load->library('email');
            // Protocol settings
            $config['protocol'] = 'smtp';
            $config['smtp_host'] = $email_cfg['smtp']['server'];
            $config['smtp_port'] = $email_cfg['smtp']['port'];
            $config['smtp_user'] = $email_cfg['smtp']['user'];
            $config['smtp_pass'] = $email_cfg['smtp']['password'];
            // Layout settings
            $config['mailtype'] = 'html';
            $config['wordwrap'] = false;
            $email_lib->initialize($config);
            // Get notifier rules
            $rules = $this->api_retrieve_rule(null);
            if($rules){
                // Get notifier users
                $users = $system->api_retrieve_user();
                foreach($rules as $rule){
                    // Check rule is email ?
                    $action = $rule->get_data_value('action', false);
                    if($type !== $action){
                        continue;
                    }
                    // Filter events
                    $severity = $rule->get_data_value('severity', false);
                    $service = $rule->get_data_value('service', false);
                    $filtered_events = array();
                    foreach($events as $event){
                        // Event for specified service (or any)
                        if(('__any__' == $service || $event['module'] == $service)
                            && false !== array_search($event['level'], $severity)){
                            $filtered_events[] = $event;
                            }else{
                            }
                    }
                    // Something to report ?
                    if(!$filtered_events){
                        // TODO: consider sending an email with everything's ok ;)
                        continue;
                    }

                    // Build recipient list
                    $user = $rule->get_data_value('user', false);
                    if('__all__' != $user){
                        $recipients[] = $users[$user];
                    }else{
                        $recipients = $users;
                    }
                    foreach($recipients as $user){
                        //  prepare and send email
                        $email_lib->from($root->email(), $root->description());
                        $email_lib->reply_to($root->email(), $root->description());
                        $email_lib->to($user->email()); 
                        
                        // Eamil body
                        $email_lib->subject($_app->description().' - '.ucfirst($type).' Report');
                        $data['title'] = $_app->description().' - '.ucfirst($type).' Report';
                        $data['product'] = $_app->description();
                        $data['type'] = ucfirst($type);
                        $data['events'] = $filtered_events;
                        $data['host'] = $host;
                        $data['server_uri'] = $uri;
                        $email_lib->message($CI->load->view('sng_notifier_email_report', $data, true));
                        error_log('INFO - Sending '.$type.' report to '.$user->email());
                        if(!$email_lib->send()){
                            error_log( $email_lib->print_debugger() );
                            file_put_contents('/tmp/email.html', $email_lib->print_debugger());
                        }
                    }
                }
            }

        }
    }
    public function test($data = null, &$output)
    {
        if(!$data['email']){
            $output['reason'] = 'Email address is required';
        }else{
            $email = $data['email'];
            $email_cfg = $this->config()->email_config();
            $_app = $this->node()->software()->application();
            if($email_cfg){
                $CI = & get_instance();
                // Get system module
                $system = $_app->find_module_by_name('system');
                // Get root user
                $root = $system->api_retrieve_user('root');
                // Get httpd module to retrieve server URI
                $host = $this->node()->hostname();
                $httpd = $_app->find_module_by_name('webconfig');
                $uri = $httpd->server_uri($host);
                $email_lib = $CI->load->library('email');
                // Protocol settings
                $config['protocol'] = 'smtp';
                $config['smtp_host'] = $email_cfg['smtp']['server'];
                $config['smtp_port'] = $email_cfg['smtp']['port'];
                $config['smtp_user'] = $email_cfg['smtp']['user'];
                $config['smtp_pass'] = $email_cfg['smtp']['password'];
                // Layout settings
                $config['mailtype'] = 'html';
                $config['wordwrap'] = false;
                $config['crlf'] = "\r\n";
                $config['newline'] = "\r\n";
                $email_lib->initialize($config);
            
                $email_lib->from($root->email(), $root->description());
                $email_lib->reply_to($root->email(), $root->description());
                $email_lib->to($email);
                // Eamil body
                $email_lib->subject($_app->description().' - '.ucfirst($level).' Email Test');
                $data['title'] = $_app->description().' - '.ucfirst($level).' Email Test';
                $data['product'] = $_app->description();
                $data['host'] = $host;
                $data['server_uri'] = $uri;
                error_log('INFO - Sending test email notification to '.$email);
                $email_lib->message($CI->load->view('sng_notifier_email_test', $data, true));
                if(!$email_lib->send()){
                    error_log('ERROR - SMTP Error - '.implode(', ', $email_lib->get_error()));
                    file_put_contents('/tmp/email.html', $email_lib->print_debugger());
                    $output['reason'] = $email_lib->get_error();
                }
            }else{
                $output['reason'] = 'Email address is required';
            }
        }
        if($output['reason'] != ''){
            $output['message'] = 'Send test email failed.';
            return false;
        }else{
            $output['message'] = 'Send test email successfully.';
            return true;
        }
    }
    public function check_object_usage($src_module_name, $obj_type, $obj_name, $sub_typ=null, $sub_name=null)
    {
        if($src_module_name == 'system' && $obj_type == 'user'){
            $usages = array();
            foreach($this->api_retrieve_rule(null) as $rule){
                if($obj_name == $rule->get_data_value('user', false)) {
                    $usages[] = array (
                            'name' => $this->name(),
                            'obj_type' => 'rule',
                            'obj_name' => $rule->name(),
                    );
                }
            }
            if($usages){
                return $usages;
            }
        }
        return parent::check_object_usage($src_module_name, $obj_type, $obj_name, $sub_typ, $sub_name);
    }
}
/* End of file sng_sshd_service_class.php */

