#!/usr/bin/env python
# vim: tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab
"""
* Copyright (C) 2011   Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* William Adam <william.adam@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.
"""

#Include this directory to  python sys.path
import paths

import sys
import os
import subprocess
import re
import xml.dom.minidom as xml
import fs_helper

from optparse import OptionParser
import logging
import logging.handlers

# Usage:
#   fs_capacity_session [percent limit]
#       Where [percent limit] defines the limit in percent of active sessions
#       if not specified, no check performed
# 
# Output format: (This tool is for monit processing)- NO pretty XML print (use
# inline)
"""
<program>  
    <name>Session Statistics</name>
    <message>Output message string </message>
    <status></status>
    <data>
        <count>xx</count>
        <percentage>xx</percentage>
    </data>
</program>
"""
# Upon success, rc=0, data section populated, message set to:
#    OK
#
# Upon limit in percent exceeded, returns -1 and message set to:
#    Active sessions amount of xx% matches resource limit [active sessions>yy%]
#  Where xx is limit specified as argument and yy is current active sessions in %
#
# Upon error, rc=0 (yes I know :) - NO data section - and message set to (fs_cli
# output) example:
#    [ERROR] fs_cli.c:1305 main() Error Connecting [Socket Connection Error]

def do_check(logger, options):
    # Output variables
    session=None
    sessionmax=None
    sessionpercent=None
    error=None
    limitpercent=None
    status=None
    data=None

    # Process arguments - TODO: use cmd line parsing
    if options.limit:
        limitpercent = float(options.limit)

    # Excute command
    cmd = 'status'
    error, out = fs_helper.execute_cmd(cmd)

    # Iterate output
    for line in out:
        # check for active sessions
        match = re.match(r'(\d{1,}) session\(s\) (\d{1,})\/(\d{1,}).*', line)
        if match:
            session=int(match.group(1))

        # check for max sessions
        match = re.match(r'(\d{1,}) session\(s\) max.*', line)
        if match:
            sessionmax=int(match.group(1))

    # Create output
    doc = xml.Document()
    # inner root
    root = doc.createElement('program');
    s = doc.createElement('name')
    st = doc.createTextNode('Session Statistics')
    s.appendChild(st)
    root.appendChild(s)

    rc = 0
    # Prepare statistics data if no fs_cli error
    if not error:
        # Compute usage %
        if session > 0:
            sessionpercent=float((float(session)/float(sessionmax))*100)
        else:
            sessionpercent=0.0

        # Check limit
        if limitpercent:
            error = '%.1f%% session active limit=[%.1f%%]' \
                % (sessionpercent, limitpercent)
            if limitpercent and sessionpercent > limitpercent:
                rc = -1
        else:
            error = '%.1f%% session active' \
                % (sessionpercent)

        data = doc.createElement('data')
        # Active sessions
        s = doc.createElement('count')
        st = doc.createTextNode(str(session))
        s.appendChild(st)
        data.appendChild(s)
        # Max sessions
        s = doc.createElement('max')
        st = doc.createTextNode(str(sessionmax))
        s.appendChild(st)
        data.appendChild(s)
        # Session usage %
        s = doc.createElement('percentage')
        st = doc.createTextNode(str(sessionpercent))
        s.appendChild(st)
        data.appendChild(s)

    # Populate message element
    s = doc.createElement('message')
    st = doc.createTextNode(str(error))
    s.appendChild(st)
    root.appendChild(s)
    if data:
        root.appendChild(data)

    return rc, root


def main():
    #Parse the options
    usage = "usage: %prog [options] arg"
    optParser = OptionParser(usage)
    optParser.add_option('-l', '--limit', action='store', type='float',
                        dest='limit', metavar='LIMIT',
                        help="Session usage capacity limit in %")

    optParser.add_option('-s', '--severity', action='store', type='choice',
                        dest='severity', metavar='SEVERITY',
                        choices=['warning', 'error', 'critical'],
                        help="Severity level")

    optParser.add_option('-p', '--pretty', action='store_true',
                        dest='pretty', metavar='PRETTY',
                        help="Pretty print output")

    (options, args) = optParser.parse_args()

    # Create logger
    logger = logging.getLogger(os.path.basename(os.path.abspath(sys.argv[0])))
    logger.setLevel(logging.INFO)
    handler = logging.StreamHandler()
    logger.addHandler(handler)

    rc, root = do_check(logger, options)
    # output result
    if options.pretty:
        logger.info(root.toprettyxml())
    else:
        logger.info(root.toxml())
    sys.exit(rc)


if __name__ == '__main__':
  main()
