<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/**
 * Config file libs
 * Read/write differrant config file,
 * Kept track of modification in a SQLite DB (data in JSON format)
 *
 *
 * @author Michel Buczynski
 * @version
 */
// We presentely using the PEAR::Config libs
// patConfiguration from http://www.php-tools.de/ may be a alternative
// since we storing data in a temp db, we have no choice of do it ours self for
// writing to config files.
require_once 'Config.php';
/**
 *
 * @author boutch
 *         
 */
class Safe_config_class
{
    public $data = array();
    // note thats in PEAR::Config the Datasource Can be a file url, a dsn, an
    // object...
    public $filename = ""; // the config file pathfilename or object name
    public $encoding_type_is_json = true;
    public $type = 'xml'; // This are file encoding container type from PEAR::Config
    public $loading_timestamp = 0; // the modify timestamp of the config file on
    // disk at th loading time.
    public $timestamp = 0; // the timestamp lastest change
    //public $saved_timestamp = 0; // 0 (false) or timestamp of safe on files.
    public $modified = 0; // counter modification (treated as a boolean.
    
    /**
     * @var CI_DB_pdo_driver
     */
    private $db = null; // the codeigniter db object
    private $db_groupname = 'safe'; // the CodeIgniter groupname set in
    // application/config/database.php
    private $_empty_array = array();
    // TODO: handling CodeIgniter database errors
    // using this $this->db->_error_message(); will return empty if no error.
    // and in ./config/database.php: $db['default']['db_debug'] = TRUE; Set it
    // to false.
    // or force it by $this->db->db_debug= flase;
    // TODO: put some method private and proteceted.
    public function __construct()
    {
        try {
            $CI = & get_instance();
            // if the files for the db doesn't it will be created.
            $this->db = $CI->load->database($this->db_groupname, TRUE);
            if (!$this->db->table_exists("safe_config")) if (!$this->create_db()) return false;
        }
        catch(Exception $e) {
            // TODO: catch CodeIgniter DB_Driver exception.
            // we will never go here since CodeIgniter DB_Driver will fail
            // with a PHP error on the user browser
            show_error(__FUNCTION__ . ": can't open databases");
            return false;
        }
    }
    /**
     * Load config file to memory
     *
     * @return boolean
     */
    public function load_file()
    {
        try {
            if (!$this->encoding_type_is_json) return false;
            $pear_conf = new Config();
            if (!file_exists($this->filename)) {
                safe_add_error(__FUNCTION__ . ":{$this->filename}: doesn't exist!");
                return false;
            }
            $root = & $pear_conf->parseConfig($this->filename, $this->type);
            if (PEAR::isError($root)) {
                safe_add_error(__FUNCTION__ . ":{$this->filename}:Error while reading configuration: " . $root->getMessage() , true);
                return false;
            }
            // creating the real array
            $this->data = $root->toArray();
            // putting the root at the root!
            $this->data = $this->data['root'];
            // free up memory
            unset($root);
            unset($pear_conf);
            // setting up the modify timestamp of the config file
            $this->timestamp = $this->loading_timestamp = filemtime($this->filename);
            //error_log ( __FUNCTION__ . $this->filename . " " . $this->loading_timestamp );
            
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
        return true;
    }
    private function get_latest_loading_timestamp()
    {
        $q = $this->db->select_max('loading_timestamp')->get_where('safe_config', array(
            'filename' => $this->filename
        ));
        if ($q->num_rows() > 0) {
            $r = $q->row();
            return $r->loading_timestamp;
        }
        return 0;
    }
    private function set_loading_timestamp($timestamp = 0)
    {
    }
    private function num_rows($timestamp = 0)
    {
        if (!$timestamp) $timestamp = $this->loading_timestamp;
        $q = $this->db->get_where('safe_config', array(
            'filename' => $this->filename,
            'loading_timestamp' => $timestamp
        ));
        return $q->num_rows();
    }
    private function get_row($timestamp = 0)
    {
        if (!$timestamp) $timestamp = $this->loading_timestamp;
        else $this->loading_timestamp = $timestamp;
        $q = $this->db->get_where('safe_config', array(
            'filename' => $this->filename,
            'loading_timestamp' => $this->loading_timestamp
        ));
        if ($q->num_rows() > 0) {
            $r = $q->result();
            if (count($r) == 1 || $r[0]->modified) $r = $r[0];
            else if (count($r) == 2) $r = $r[1];
            else {
                throw new Exception("Too mush rows!");
                return false;
            }
            $this->filename = $r->filename;
            $this->loading_timestamp = $r->loading_timestamp;
            $this->timestamp = $r->timestamp;
            $this->modified = $r->modified;
            $this->data = $this->decode_data($r->data);
            //error_log ( "GET_ROW LST:" . $this->loading_timestamp );
            return true;
        }
        return false;
    }
    /**
     * Load config from a file or form the db, in memory
     *
     * @param
     *         	 $load_file
     * @return boolean true if data is loaded;
     */
    public function load_data($load_file = true)
    {
        try {
            $db_ts_max = $this->get_latest_loading_timestamp();
            if ($load_file && file_exists($this->filename)) {
                $file_ts = filemtime($this->filename);
                if ($db_ts_max < $file_ts) if ($this->num_rows($file_ts) == 0) {
                    if ($this->load_file()) return $this->insert_db();
                } else return $this->get_row($file_ts);
            }
            // Normally if we are here the data doesn't come from a file so the
            // latest from the db
            if ($db_ts_max) {
                return $this->get_row($db_ts_max);
            } else return false;
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}:" . $e->getMessage() , true);
            return false;
        }
        return false;
    }
    /**
     * Get data structure from memory
     *
     * @param
     *         	 array specify the path to a specific section
     * @return array null
     */
    public function &get_data($section = array())
    {
        if (count($section) == 0) return $this->data;
        $data = & $this->data;
        foreach ($section as $k) {
            if (array_key_exists($k, $data)) $data = & $data[$k];
            else {
                safe_add_error(__FUNCTION__ . ":{$this->filename}: " . "Can't get data: section '$k' don't exist!");
                return $this->_empty_array;
            }
        }
        return $data;
    }
    /**
     * Set data struture to the memory and DB.
     *
     * @param
     *         	 array specify the path to a specific section
     * @param $new_data    array  
     * @param $update_only boolean
     * @return boolean     
     */
    public function set_data($section, $new_data, $update_only = TRUE)
    {
        if (count($section) == 0 || $section == null) {
            // if no section specify put it directly
            $this->data = $new_data;
        } else {
            // find the section
            try {
                $data = & $this->data;
                foreach ($section as $k) {
                    if ($update_only == FALSE || array_key_exists($k, $data)) {
                        $data = & $data[$k];
                    } else {
                        safe_add_error(__FUNCTION__ . ":{$this->filename}: " . "Can't set data: section '$k' don't exist!");
                        return false;
                    }
                }
                $data = $new_data;
            }
            catch(Exception $e) {
                safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
                return false;
            }
        }
        return $this->save_data();
    }
    public function save_new_data($new_data)
    {
        try {
            if (empty($new_data)) throw new Exception("New data is empty");
            if (is_string($new_data)) {
                $new_data = $this->decode_data($new_data);
                if (empty($new_data)) throw new Exception("Error in JSON the New data");
            }
            $this->conf->loading_timestamp = 0;
            return $this->save_data();
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
        }
    }
    /**
     * The modified config data ($this->$data) into the database.
     * Since the data must be modified, the method is protected it sould be
     * called only from set_data
     *
     * @return boolean
     */
    public function save_data()
    {
        try {
            if (empty($this->loading_timestamp)) {
                $db_ts_max = $this->get_latest_loading_timestamp();
                if ($db_ts_max > 0) {
                    $this->loading_timestamp = $db_ts_max;
                    $this->modified = $this->is_modified();
                } else {
                    $this->loading_timestamp = time();
                    $this->modified = - 1;
                }
            }
            //error_log ( 'SAVE_DATA LST:' . $this->loading_timestamp );
            $this->timestamp = time();
            $this->modified++;
            switch ($this->num_rows()) {
            case 2:
                return $this->update_db();
            case 1:
                return $this->insert_db();
            case 0:
                $this->modified = 0;
                return $this->insert_db();
            }
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
        }
    }
    /**
     * This is call back function for array_walk for converting the data in
     * memory
     * ($this->$data) into a PEAR::Config structure.
     *
     * @param $v mixed       
     *                        	 value
     * @param $k string|mixed
     *                        	 key
     * @param
     *        	 Config_container &$conf Config tree
     */
    private function save_file_call_back($v, $k, &$conf)
    {
        try {
            if (gettype($v) == 'array') {
                if (!is_indexed_array($v)) $this->save_file_section_callback($v, $k, $conf);
                else foreach ($v as $v0) $this->save_file_section_callback($v0, $k, $conf);
            } else $conf->createDirective($k, $v);
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return null;
        }
    }
    /**
     * Create Section and walk threw is element.
     *
     *
     * @param $v mixed       
     *                        	 value
     * @param $k string|mixed
     *                        	 key
     * @param
     *        	 Config_container &$conf Config tree
     */
    private function save_file_section_callback($v, $k, &$conf)
    {
        if (empty($v['__SAFE_DISABLE_SECTION__'])) {
            $section_conf = & new Config_Container('section', $k);
            $conf->addItem($section_conf);
            array_walk($v, array(
                $this,
                'save_file_call_back'
            ) , $section_conf);
        }
    }
    /**
     * Save memory data structure to file.
     *
     * @return boolean
     */
    public function save_file($update_db = true)
    {
        // TODO : other format than XML....
        try {
            if (!$this->encoding_type_is_json) return false;
            $pear_conf = new Config();
            $conf = $pear_conf->getRoot();
            switch ($this->type) {
            case 'xml':
                // $section_conf=& new Config_Container_XML();
                // all Config_Container_XXXX dont have the same set of
                // member function
                // methode.
                $section_conf = & new Config_Container($this->type);
                break;

            case 'apache':
                $section_conf = & new Config_Container_Apache();
                break;

            case 'genericconf':
                $section_conf = & new Config_Container_GenericConf();
                break;

            case 'inicommented':
                $section_conf = & new Config_Container_IniCommented();
                break;

            case 'inifile':
                $section_conf = & new Config_Container_IniFile();
                break;

            case 'phparray':
                $section_conf = & new Config_Container_PHPArray();
                break;

            case 'phpconstants':
                $section_conf = & new Config_Container_PHPConstants();
                break;

            default:
                echo "format not supported";
                return;
            }
            $conf->addItem($section_conf);
            array_walk($this->data, array(
                $this,
                'save_file_call_back'
            ) , $conf);
            safe_write_file($this->filename, $conf->toString($this->type));
            unset($conf);
            unset($pear_conf);
            if ($update_db) {
                $old_ts = $this->loading_timestamp;
                $this->timestamp = $this->loading_timestamp = filemtime($this->filename);
                $this->modified = 0;
                if ($this->num_rows($old_ts) > 1) return $this->update_db($old_ts);
                if ($this->num_rows() > 0) return $this->update_db(); //should'nt be here: but i can happen on double save_file() call
                else return $this->insert_db();
            }
            return true;
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
    }
    /**
     *
     * @return boolean
     */
    private function create_db()
    {
        try {
            //error_log ( "SAFe_Config: Trying to creating safe_config DB" );
            $sqlq = <<<EOF
			  CREATE TABLE 'safe_config' ( 'id' INTEGER PRIMARY KEY
			 		 AUTOINCREMENT, 'filename' VARCHAR(255) NOT NULL,
			 		 'loading_timestamp' INTEGER NOT NULL, 'timestamp' INTEGER,
			 		 'modified' BOOLEAN ,  'data' TEXT, CONSTRAINT
			 		 unique_file UNIQUE ('filename','loading_timestamp','modified') ) ;
EOF;
            if (!$this->db->query($sqlq)) throw new Exception("Can't not create DB!");
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
        return true;
    }
    public function update_db($where_loading_timestamp = 0)
    {
        try {
            $r = array();
            $r['timestamp'] = $this->timestamp;
            $r['modified'] = $this->modified;
            $r['data'] = $this->encode_data();
            $r['loading_timestamp'] = $this->loading_timestamp;
            //// I really dont know why?
            // 			$is_mod = $this->is_modified ( $ts );
            // 			if ($is_mod || $r ['modified'])
            // 				$this->modified = $r ['modified'] = $is_mod;
            $where = array(
                'filename' => $this->filename,
                'loading_timestamp' => (!empty($where_loading_timestamp) ? $where_loading_timestamp : $this->loading_timestamp) ,
                'modified' => $this->is_modified($where_loading_timestamp)
            );
            $this->db->where($where);
            //error_log ( "UPDATE LST:" . $this->loading_timestamp . ' WHERE:' . implode ( ':', $where ) );
            if (!$this->db->update('safe_config', $r)) throw new Exception("{$this->filename}: " . "Can't insert into BD!");
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
        return true;
    }
    /**
     *
     * @return boolean
     */
    private function insert_db()
    {
        try {
            $r = array();
            $r['filename'] = $this->filename;
            $r['loading_timestamp'] = $this->loading_timestamp;
            $r['timestamp'] = $this->timestamp;
            $r['modified'] = $this->modified;
            $r['data'] = $this->encode_data();
            //error_log ( "INSERT LST:" . $this->loading_timestamp );
            if (!$this->db->insert('safe_config', $r)) throw new Exception("{$this->filename}: " . "Can't insert into BD!");
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
        return true;
    }
    /**
     *
     * @return number boolean
     */
    public function is_modified($timestamp = null)
    {
        try {
            if (empty($timestamp)) $timestamp = $this->loading_timestamp;
            $q = $this->db->get_where('safe_config', array(
                'filename' => $this->filename,
                'loading_timestamp' => $timestamp,
                'modified !=' => 0
            ));
            $r = $q->result();
            if ($q->num_rows() == 1) return $r[0]->modified;
            if ($q->num_rows() == 0) return 0;
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}:" . $e->getMessage() , true);
            return false;
        }
        return false;
    }
    /**
     *
     * @return boolean
     */
    public function is_modified_and_not_saved()
    {
        try {
            if ($this->is_modified()) {
                $q = $this->db->get_where('safe_config', array(
                    'filename' => $this->filename,
                    'loading_timestamp' => $this->loading_timestamp,
                    'modified !=' => 0
                ));
                $r = $q->result();
                if ($r[0]->saved == 0) {
                    return true;
                }
            }
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            return false;
        }
    }
    /**
     *
     * @return boolean
     */
    public function reset_modification()
    {
        try {
            if (empty($this->data)) {
                $this->load_data();
            }
            if ($this->is_modified()) {
                $this->db->where(array(
                    'filename' => $this->filename,
                    'loading_timestamp' => $this->loading_timestamp,
                    'modified !=' => 0
                ));
                $this->db->delete('safe_config');
                $this->loading_timestamp = 0; // the modify timestamp of the config
                // file o
                $this->timestamp = 0; // the timestamp lastest change
                //$this->saved_timestamp = 0; // 0 (false) or timestamp of safe on
                // files.
                $this->modified = 0; // counter modification (treated as a boolean.
                $this->data = array();
                $this->load_data();
            }
        }
        catch(Exception $e) {
            safe_add_error(__FUNCTION__ . ":{$this->filename}:" . $e->getMessage() , true);
            return false;
        }
    }
    private function encode_data()
    {
        if ($this->encoding_type_is_json) return json_encode($this->data);
        else return serialize($this->data);
    }
    private function decode_data($serialize_data)
    {
        if ($this->encoding_type_is_json) return json_decode($serialize_data, true);
        else {
            echo $serialize_data;
            return unserialize($serialize_data);
        }
    }
}
/* End of file safe_config_class.php */
