#!/usr/bin/env python
# vim: tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab
"""
* Copyright (C) 2011   Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* William Adam <william.adam@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.
"""

#Include this directory to  python sys.path
import paths

import sys
import os
import subprocess
import re
import xml.dom.minidom as xml
import fs_helper
import simplejson as json

from optparse import OptionParser
import logging
import logging.handlers

# Usage:
#   sng_test_check.py --limit=x
#     Check for file /tmp/sng_test_check.json, containing following json
#     Something similar to:
"""
{
  "msg": "Test Check", 
  "name": "This is a test check", 
  "rc": -1,
  "value": 50,
  "data":{
    "count": 10,
    "max": 1
  }
}
"""
#     name: "name of the service"
#     value: y // checked against limit
#     msg: "bla bla" // message returned <message> element
#     data: { count: ... // Data array returned in <data> element
# 
# Output format: (This tool is for monit processing)- NO pretty XML print (use
# inline)
"""
<program>  
    <name>Session Statistics</name>
    <message>Output message string </message>
    <status></status>
    <data>
        <count>xx</count>
        <percentage>xx</percentage>
    </data>
</program>
"""
# Upon success, rc=0, data section populated, message set to:
#    OK
#
# Upon limit in percent exceeded, returns -1 and message set to:
#    Active sessions amount of xx% matches resource limit [active sessions>yy%]
#  Where xx is limit specified as argument and yy is current active sessions in %
#
# Upon error, rc=0 (yes I know :) - NO data section - and message set to (fs_cli
# output) example:
#    [ERROR] fs_cli.c:1305 main() Error Connecting [Socket Connection Error]
def do_check(logger, options):
    # Output variables
    rc=0
    data=None
    limit=None
    error="All good ;)"
    error_prefix = ""
    config={"name":"name of the service", "value":0, "msg":"bla, bla..."}
    config_file = "/tmp/sng_test_check.json"

    # Process arguments - TODO: use cmd line parsing
    if options.limit:
        limit = int(options.limit)
    
    # Read the cfg file
    try:
        config = json.loads(open(config_file).read())
    except:
        pass

    if 'msg' in config:
        error_prefix = config['msg'] + ' - '

    # Check limit
    if limit and 'value' in config:
        if config['value'] > limit:
            rc = -1
            error = 'Limit:' + str(limit) + ' Value:' + str(config['value'])
    elif 'rc' in config:
        rc = int(config['rc'])
        if rc:
            error = "Check fail (" + str(config['rc']) + ")"

    # Create output
    doc = xml.Document()
    # inner root
    root = doc.createElement('program');
    s = doc.createElement('name')
    st = doc.createTextNode(config['name'])
    s.appendChild(st)
    root.appendChild(s)

    if 'data' in config:
        data = doc.createElement('data')
        for k, v in config['data'].items():
            # Active sessions
            s = doc.createElement(k)
            st = doc.createTextNode(str(v))
            s.appendChild(st)
            data.appendChild(s)

    # Populate message element
    s = doc.createElement('message')
    st = doc.createTextNode(error_prefix + str(error))
    s.appendChild(st)
    root.appendChild(s)
    if data:
        root.appendChild(data)

    # output result
    return rc, root


def main():
    #Parse the options
    usage = "usage: %prog [options] arg"
    optParser = OptionParser(usage)
    optParser.add_option('-n', '--name', action='store', type='string',
                        dest='name', metavar='NAME',
                        help="Resource name")

    optParser.add_option('-l', '--limit', action='store', type='float',
                        dest='limit', metavar='LIMIT',
                        help="Status limit in %")

    optParser.add_option('-p', '--pretty', action='store_true',
                        dest='pretty', metavar='PRETTY',
                        help="Pretty print output")

    (options, args) = optParser.parse_args()

    # Create logger
    logger = logging.getLogger(os.path.basename(os.path.abspath(sys.argv[0])))
    logger.setLevel(logging.INFO)
    handler = logging.StreamHandler()
    logger.addHandler(handler)

    # Execute Check
    rc, root = do_check(logger, options)
    # Check output
    if options.pretty:
        logger.info(root.toprettyxml())
    else:
        logger.info(root.toxml())
    # Report error/success
    sys.exit(rc)


if __name__ == '__main__':
  main()
