<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * William Adam <wadam@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * SNG System module class
*/
require_once ('application/helpers/safe_helper.php');
safe_require_class('safe_module_class');
safe_require_class('safe_auditor_class');


class Sng_system_user_class extends Safe_configurable_object_class
{
    public function __construct($node, $parent_name, $name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, $name, $node);
    }
    public function configure()
    {
        // Contact fields
        $this->add_field('name', 'Name', 'text', '','50');
        $host = $this->node()->hostname();
        $email_def = $this->name().'@'.$host;
        $this->add_field('email', 'Email Address', 'text', $email_def,'50');
        $this->set_field_category('name', 'Contact');
        // Credentials fields
        // TODO(wadam) - Add callback to valiate psw not placeholder
        // TODO(wadam) - Figure out how to avoid sending psw to client page
        $this->add_field('secure/password', 'Password', 'password', '','30');
        $this->set_field_rules('secure/password', 'required');
        $this->add_field('secure/verify', 'Verify', 'password', '','30');
        $this->set_field_rules('secure/verify', 'matches[secure/password]');
        $this->set_field_category('secure/password', 'Credentials');
        return parent::configure();
    }
    public function summary($type = 'horizontal' , $long = false){
        $desc_custom = array('name', 'email');
        $table_line = array("ID" => $this->name());
        foreach ($desc_custom as $desc_key){
            $table_line[$this->get_data_label($desc_key)] = $this->get_data_value($desc_key);
        }
        return array('data' => $table_line);
    }
    public function dispose()  
    {
        $prog = '/usr/local/sng/bin/sng-user-mgmt';
        // Complete cmd
        $cmd .= ' --action=remove --user='.$this->name();
        // Ready to executeo
        $output = array();
        $rc = $this->node()->os()->execute($prog.' '.$cmd, $output, true, true);
        error_log('output='.print_r($output, true));

        if($rc){
            return parent::dispose();
        }else{
            return false;
        }
    }

    public function save()
    {
        $update_psw = true;
        // Get the object status
        // For new user, then backend must be invoked using add
        // For modified user, then backend must be invoked using password
        $status = $this->status();
        switch($status){
        case Safe_object_serializer_class::OBJ_STATUS_NEW:
            if($this->name() != 'root'){
                $cmd = '--action=add';
                break;
            }else{
                // Have to skip password for root as it already has one
                $update_psw = false;
            }
        case Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE:
        case Safe_object_serializer_class::OBJ_STATUS_MODIFIED:
            $cmd = '--action=update';
            break;
        default:
            return false;
        }

        $prog = 'nohup /usr/local/sng/bin/sng-user-mgmt';
        // Complete cmd
        $cmd .= ' --user='.$this->name();
        // Update psw ?
        if($update_psw){
            $cmd .= ' --password='.$this->get_data_value('secure/password', false);
        }
        $cmd .= ' --name="'.$this->description().'"';
        // Ready to executeo
        $output = array();
        $rc = $this->node()->os()->execute($prog.' '.$cmd, $output, true, true);

        if($rc){
            parent::save();
            $this->clear_modified();
            return true;
        }else{
            return false;
        }
    }

    public function can_dispose(&$reason=null)
    {
        // Check root user
        if('root' == $this->name()){
            $reason['msg'] = 'User root cannot be removed !';
            return false;
        }
        // Check if a module uses this one
        $usages = $this->node()->check_object_usage('system', 'user', $this->name());
        if($usages){
            $reason['msg'] = "  {$this->name()} is used by ";
            $reason['obj'] = $usages;
            return false;
        }
        // Check user not currently logged in
        $prog = '/usr/local/sng/bin/sng-user-mgmt';
        // Complete cmd
        $cmd .= ' --action=logged --user='.$this->name();
        // Ready to executeo
        $output = array();
        $rc = $this->node()->os()->execute($prog.' '.$cmd, $output, true, true);

        if(!$rc){
            $reason['msg'] = $output;
            return false;
        }
        return true;
    }
    public function email()
    {
        return $this->get_data_value('email', false);
    }
    public function description()
    {
        $name = $this->get_data_value('name');
        if($name){
            return $name;
        }else{
            return $this->name();
        }
    }
}

/******************************************************************************
 *   System Module
 ******************************************************************************/
class Sng_system_module_class extends Safe_module_class
{
    public function __construct($app)
    {
        parent::__construct($app->node(), "system");
    }
    /**
     * @brief
     *         
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("System");
        // Register objects
        $this->register_aggregate_object('user', 
            array(
                'name' => 'User',
                'dynamic' => true,
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Add a User',
                        'request' => 'POST'
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a User',
                        'request' => 'GET'
                        ),
                    'update' => array(
                            'name' => 'Update',
                            'description' => 'Update a User',
                            'request' => 'POST'
                    ),
                    'delete' => array(
                            'name' => 'Delete',
                            'description' => 'Delete a User',
                            'request' => 'POST'
                    )
               )
            )
        );
        // Check if 'root' user defined
        // if not, create it
        $root = $this->api_retrieve_user('root');
        if(!$root){
            $data = array(
                'name' => 'System Administrator',
                'secure/password' => 'sangoma',
                'secure/verify' => 'sangoma',
            );
            $root = $this->api_create_user('root', $data);
            $root->save();
        }
        return parent::configure();
    }
    public function new_fields(&$config_manager, $type)
    {
        $label = 'User ID';
        $config_manager->add_field('profile-name', $label, 'string', '', 50);
        $config_manager->set_field_rules('profile-name', 'required|alpha_dash');
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_user($name, $data=null, &$output = null) {
        $view = new sng_system_user_class($this->node(), $this->aggregate_object_base_path('user'), $name);
        $view->configure();
        if($data) {
            $view->set_data_values($data);
        }
        return $view;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_user($name=null, $data=null, &$output = null) {
        $objs = $this->get_aggregate_objects('user');
        if($name){
            return $objs[$name];
        }else{
            return $objs;
        }
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_user($name, $data=null, &$output = null) {
        $user = $this->api_retrieve_user($name);
        if($user) {
            if ($user->validate($data,$output)) {
                if (true == $user->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_user($name, $data=null, &$output = null) {
        $user = $this->api_retrieve_user($name);
        if($user) {
            if($user->can_dispose($output)) {
                return $user->dispose();
            }
        }
        return false;
    }
    /**
     * @brief Check if we can create agg obj
     *
     * @param[in out] $type
     * @param[in out] $name
     * @param[in out] $reason
     *
     * @return 
     */
    public function can_create_aggregate_object($type, $name, &$reason="")
    {
        // base class check
        if(!parent::can_create_aggregate_object($type, $name, $reason)){
            return false;
        }
        // User object ?
        if('user' == $type){
            // Ask backend if user exists ?
            $prog = '/usr/local/sng/bin/sng-user-mgmt';
            // Complete cmd
            $cmd .= ' --action=exist --user='.$name;
            // Ready to execute
            $output = array();
            $rc = $this->node()->os()->execute($prog.' '.$cmd, $output, true, true);

            if(!$rc){
                $reason = $output[0];
                return false;
            }
        }
        return true;
    }
    public function statistics()
    {
        return $this->node()->os()->statistics();
    }
}
/* End of file sng_update_module_class.php */
