<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/**
 * SNG SSH Linux service wrapper
 *
 * @author William Adam
 * @version
 */
if (!defined('BASEPATH')) exit('No direct script access allowed');
require_once ('application/helpers/safe_helper.php');
safe_require_class('service');
class Sng_snmpd_config_class extends Safe_configurable_object_class
{
    public function __construct($node, $parent_name, $name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, $name, $node);
    }
    public function configure()
    {
        // Manual configuration ?
        $this->add_enum_field('assisted', 'Assisted Configuration', 'dropdown', 'false', $this->enable_disable_enum());
        // SNMP parameters
        $snmp_version = array( 'false' => 'SNMP v1/2c', 'true'=>'SNMP v3');
        $this->add_enum_field('version3', 'Version', 'dropdown', 'false', $snmp_version);

        $this->set_field_category('assisted', 'General');
        // Default interface name
        $def_if = 'all';
        // Populate IP enumeration with all management interface IP
        $mgmt_if = $this->_node->hardware()->adapters(Safe_hardware_adapter_class::ETHERNET_ADAPTER);
        $if_enum = array();
        $if_enum['all'] = 'All interfaces';
        foreach($mgmt_if as $if_name=>$if) {
            if(strlen($if->ip_address()))
            {
                $if_enum[$if->ip_address()] = $if->name() . ' - ' . $if->ip_address();
            }
        }
        $this->add_enum_field('ip', 'Network Interface', 'dropdown', '', $if_enum);
        $this->set_field_help('ip', 'Select interface that snmpd listens on.');
        // protocol
        $transport = array(
            'udp' => 'UDP',
            'tcp' => 'TCP' );
        $this->add_enum_field('transport', 'Transport', 'dropdown', 'udp', $transport);

        $this->add_field('port', 'Port', 'text', '161',10);
        $this->set_field_help('port', 'Port number that snmpd listens on.');
        $this->set_field_rules('port', 'required|integer|greater_than[0]|less_than[61000]|callback_port_check[port]');
        $this->set_field_category('ip', 'Network Configuration');

        // User/psw
        $this->add_field('user', 'User Name', 'text', '',20);
        $this->set_field_rules('user', 'required[version3]|alpha_dash');
        $this->add_field('password', 'Password', 'password', '',20);
        $this->set_field_rules('password', 'required[version3]|alpha_dash|min_length[8]');
        $this->set_field_category('user', 'SNMPv3 User');

        // Create conditional group
        $this->create_group('options', 
            array('ip', 'transport', 'port', 'version', 'community', 'user', 'password') );
        $this->conditional_control('assisted', 'options');

        return parent::configure();
    }

    public function description()
    {
        return parent::description();
    }
    public function is_custom(){
        $assisted = $this->get_data_value('assisted', false);
        return 'true' === $assisted;
    }
    public function is_version3(){
        $version3 = $this->get_data_value('version3', false);
        return 'true' === $version3;
    }

    public function summary($type = 'horizontal' , $long = false){
        if($this->is_custom()){
            if($this->is_version3()){
                $table_line = parent::summary($type , true);
            }else{
                $table_line = parent::summary($type , false, 5);
            }
        }else{
            $table_line = parent::summary($type , false, 1);
        }
        return $table_line;
    }
    
    public function port_check($field_name = null){
        if($field_name){
            $port_name = $this->get_data_label($field_name);
            $port_value = $this->get_data_value($field_name);
            $port_type =  $this->get_data_value('transport');
        }else{
            return true;
        }
        $ip = $this->get_data_value('ip',false);
        //ip port
        $ports[] = array('name' =>$port_name,'port'=>$port_value, 'ip'=>$ip, 'type' => $port_type);
    
        //check $conflicts
        $conflicts = $this->node()->software()->application()->check_resource_conflict('snmpd','IP_PORT', $ports);
    
        foreach($conflicts as $conflict){
            $mod = $this->node()->software()->application()->find_module_by_name($conflict['name']);
            $name = $mod->description();
            $error_msg = $error_msg . " with {$name} at {$conflict['desc']}, " ;
        }
    
        $error_msg_2 ='';
        //check system port
        foreach($ports as $port_item){
            if(!$this->node()->os()->valid_ip_port($port_item['ip'], $port_item['port'], $port_item['type'], 'snmpd')){
                $error_msg_2 = " This port already using by other program now. " ;
            }
        }
        if($error_msg == '' && $error_msg_2 == ''){
            return true;
        }else{
            if ($error_msg!=''){
                $error['port_check'] = 'This port has conflict'.substr($error_msg, 0, -2).'. ';
            }
            if ($error_msg_2!=''){
                $error['port_check'] = $error['port_check'] . $error_msg_2 .' ';
            }
            return $error;
        }
    }
}
class Sng_snmpd_view_class extends Safe_configurable_object_class
{
    public function __construct($node, $parent_name, $name)
    {
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, $name, $node);
    }
    public function configure()
    {
        // Specify OID
        $this->add_field('oid', 'OID', 'text', '',25);
        $this->set_field_help('oid', 'OID Tree');
        $this->set_field_rules('oid', 'required|alpha_dash');
        // TYPE: included or excluded
        $type = array(
            'included' => 'Included',
            'excluded' => 'Excluded'
            );
        $this->add_enum_field('type', 'Type', 'dropdown', 'included', $type);
        $this->set_field_help('type', 'Include or exclude OID tree.');
        // Access: RW or RO
        $access = array(
            'ro' => 'Read Only',
            'rw' => 'Read/Write',
        );
        $this->add_enum_field('access', 'Access', 'dropdown', 'roview', $access);
        $this->set_field_help('access', 'OID tree access.');

        return parent::configure();
    }
    public function summary($type = 'horizontal' , $long = false){
        $table_line = parent::summary($type , $long);
        array_shift($table_line['data']);
        return $table_line;
    }
}

class Sng_snmpd_service_class extends Safe_service_class
{
    public function __construct($software)
    {
        parent::__construct($software, 'snmpd');
    }
    /**
     * @brief
     *
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("SNMP");
    
        // Create the config object
        $cfg = new Sng_snmpd_config_class($this->node(), parent::path(),$this->name());
        // Synch with DB
        $cfg->configure();
        $cfg->synch();
        // Attach config to module
        $this->set_config($cfg);

        // Register objects
        $this->register_aggregate_object('view', 
            array(
                'name' => 'View',
                'dynamic' => true,
                'autoname' => true,
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Add a View',
                        'request' => 'POST'
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a View',
                        'request' => 'GET'
                        ),
                    'update' => array(
                            'name' => 'Update',
                            'description' => 'Update a View',
                            'request' => 'POST'
                    ),
                    'delete' => array(
                            'name' => 'Delete',
                            'description' => 'Delete a View',
                            'request' => 'POST'
                    )
               )
            )
        );
    
        return parent::configure();
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_view($name, $data=null, &$output = null) {
        $view = new sng_snmpd_view_class($this->node(), $this->aggregate_object_base_path('view'), $name);
        $view->configure();
        if($data) {
            $view->set_data_values($data);
        }
        return $view;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_view($name=null, $data=null, &$output = null) {
        $objs = $this->get_aggregate_objects('view');
        if($name){
            return $objs[$name];
        }else{
            return $objs;
        }
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_view($name, $data=null, &$output = null) {
        $view = $this->api_retrieve_view($name);
        if($view) {
            if ($view->validate($data,$output)) {
                if (true == $view->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_view($name, $data=null, &$output = null) {
        $route = $this->api_retrieve_view($name);
        if($route) {
            if($route->can_dispose($output)) {
                return $route->dispose();
            }
        }
        return false;
    }

    public function can_restore($info, &$reason)
    {
        // Prevent calling parent as service status will be checked
        return true;
    }
    public function allow_user_ctl()
    {
        return true;
    }
    /**
     * @brief Check for IP_PORT conflicts
     *
     * @param[in out] $src_module_name
     * @param[in out] $resource_type
     * @param[in out] $resource_data
     **  'name'       => Module reporting conflict
     *  'obj_type'   => Object type 
     *  'obj_name'   => (opt) Object name (except for configuration) 
     *  'child_type' => (opt) Child type 
     *  'child_name' => (opt) Object name
     * @return 
     */
    public function check_resource_conflict($src_module_name, $resource_type, $resource_data)
    {
        // IP_PORT conflicts ?
        $conflicts = array();
        if('IP_PORT' == $resource_type){
            // get our configuration
            $data = $this->config()->get_data_values(false);
            $ports[] = array('name' =>$this->name(),'port'=>$data['port'], 'ip'=>$data['ip'], 'type' => 'TCP');
            // Loop around resource data
            foreach($resource_data as $resource){
                // TODO - Check
                foreach($ports as $port_item){
                    if(safe_check_port_conflict($port_item, $resource)){
                        $conflicts[] = array('name' => $this->name(), 'obj_type' =>'Configuration', 'desc' =>'port '.$port_item['port']);
                    }    
                }
            }
        }
        return $conflicts;
    }
    /**
     * @brief Generate sshd configuration
     *
     * @param[in out] $config_manager
     *
     * @return true on success
     */
    public function generate_config(&$config_manager)
    {
        //Generates service configuration files
        if (!$this->_generate_service_config($config_manager)) return false;
        // Invoke parent generate
        return parent::generate_config($config_manager);
    }
    public function _generate_service_config(&$config_manager)
    {
        if(!$this->is_custom_configuration()){
            return true;
        }
        // If assisted configuration then create /etc/snmp/snmpd.conf file
        $cfg = $this->config();

        $cfg_data = $cfg->get_data_values(false);
        $cfg_file = '/etc/snmp/snmpd.conf';
        $snmpd_conf = array();

        if($cfg->is_version3()){
            // Loop around views
            foreach($this->api_retrieve_view() as $view){
                $view_data = $view->get_data_values(false);
                $snmpd_conf[] = $view_data['access'].'user '.$cfg_data['user'].' auth '.$view_data['oid']; 
            }
        }else{
            // Map comunity to security
            $snmpd_conf[] = '# First, map the community name "public" into a "security name"';
            $snmpd_conf[] = '#       sec.name  source          community';
            $snmpd_conf[] = 'com2sec notConfigUser  default       sangoma';
            $snmpd_conf[] = '';
            $snmpd_conf[] = '# Second, map the security name into a group name:';
            $snmpd_conf[] = '#       groupName      securityModel securityName';
            $snmpd_conf[] = 'group   notConfigGroup v1           notConfigUser';
            $snmpd_conf[] = 'group   notConfigGroup v2c          notConfigUser';
            $snmpd_conf[] = ''; 
            $snmpd_conf[] = '# Third, create a view for us to let the group have rights to:'; 
            $snmpd_conf[] = '#       name           incl/excl     subtree         mask(optional)';
            // Loop around views
            foreach($this->api_retrieve_view() as $view){
                $view_data = $view->get_data_values(false);
                $snmpd_conf[] = 'view '. $view_data['access']. 'view ' . $view_data['type'] .' '.$view_data['oid'];
            }

            $snmpd_conf[] = ''; 
            $snmpd_conf[] = '# Finally, grant the group read-only access to the systemview view.'; 
            $snmpd_conf[] = '#       group          context sec.model sec.level prefix read   write  notif'; 
            $snmpd_conf[] = 'access  notConfigGroup ""      any       noauth    exact  roview rwview none'; 
        }

        $snmpd_conf[] = ''; 
        $snmpd_conf[] = '#syslocation Unknown (edit /etc/snmp/snmpd.conf)'; 
        // TODO- Get root user
        $snmpd_conf[] = 'syscontact Root <root@localhost> (configure /etc/snmp/snmp.local.conf)'; 
        // Write config job
        $config_manager->add_config(
            new Safe_configuration_class(
                $cfg_file, 
                'webconfig', 
                Safe_configuration_class::CFG_UPDATE,
                Safe_configuration_class::CFG_CHOWN));
        $config_manager->add_config(
            new Safe_configuration_class(
                $cfg_file,
                $snmpd_conf));
        $config_manager->add_config(
            new Safe_configuration_class(
                $cfg_file, 
                'root', 
                Safe_configuration_class::CFG_UPDATE,
                Safe_configuration_class::CFG_CHOWN));
        
        // snmpd.options.conf
        $options_file = '/etc/sysconfig/snmpd.options';
        $snmpd_options_conf = array();
        $snmpd_options_conf[] = '# snmpd command line options';
        $listen = strtoupper($cfg_data['transport']).':';
        if('all' != $cfg_data['ip']){
            $listen .= $cfg_data['ip'].':';
        }
        $listen .= $cfg_data['port'];

        // less syslog verbosity
        $snmpd_options_conf[] = 'OPTIONS="-LS 0-4 d -Lf /dev/null -p /var/run/snmpd.pid -a '.$listen.'"';

        // Write config job
        $config_manager->add_config(
            new Safe_configuration_class(
                $options_file, 
                'webconfig', 
                Safe_configuration_class::CFG_UPDATE,
                Safe_configuration_class::CFG_CHOWN));
        $config_manager->add_config(
            new Safe_configuration_class(
                $options_file,
                $snmpd_options_conf));
        $config_manager->add_config(
            new Safe_configuration_class(
                $options_file, 
                'root', 
                Safe_configuration_class::CFG_UPDATE,
                Safe_configuration_class::CFG_CHOWN));

        // Check v3
        if($cfg->is_version3()){
            // Create /var/net-snmp/snmpd.conf
            // createUser snmpuser MD5 "test12345" DES
            $user_file = '/var/net-snmp/snmpd.conf';
            $user_conf = array();
            $user_conf[] =  'createUser '.$cfg_data['user'].' MD5 "'.$cfg_data['password'].'" DES';
            // Write config job
            $config_manager->add_config(
                new Safe_configuration_class(
                    $user_file, 
                    'webconfig', 
                    Safe_configuration_class::CFG_UPDATE,
                    Safe_configuration_class::CFG_CHOWN));
            $config_manager->add_config(
                new Safe_configuration_class(
                    $user_file,
                    $user_conf));
            $config_manager->add_config(
                new Safe_configuration_class(
                    $user_file, 
                    'root', 
                    Safe_configuration_class::CFG_UPDATE,
                    Safe_configuration_class::CFG_CHOWN));

        }
        
        return true;
    }
    /**
     * @brief Invoked after a successfull write_config
     *
     * @return
     */
    public function post_write_config($obj_type=null)
    {
        // restart service
        if($this->status() == Safe_service_class::STATUS_RUNNING){
            $this->restart();
        }
        return parent::post_write_config($obj_type);
    }
    public function support_reload(&$need_reload = null) {
        parent::support_reload($need_reload);
        return true;
    }
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        return $this->generate_config($config_manager);
    }
    public function reload_clear_modified($obj_type=null)
    {
        return $this->clear_configuration_modified();
    }
    public function is_custom_configuration()
    {
        return $this->config()->is_custom();
    }
}
/* End of file sng_sshd_service_class.php */
