<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/**
 * SNG Auditor Class
 *
 * @author William Adam
 * @version
 */
if (!defined('BASEPATH')) exit('No direct script access allowed');
require_once ('application/helpers/safe_helper.php');
safe_require_class('service');
safe_require_class('auditor');

class Sng_auditor_config_class extends Safe_configurable_object_class
{
    private $_audit_points = array();
    public function __construct($node, $parent_name, $name, $audit_points)
    {
        // Re-shuffle categories
        $_audits['system'] = array();
        $_audits['application'] = array();
        $_audits['process'] = array();
        $_audits['service'] = array();
        $_audits['module'] = array();
        foreach($audit_points as $pname => $point){
            if(is_subclass_of($point->module(), 'Safe_application_class') && !array_key_exists(strtolower($point->category()), $_audits)){
                $_audits['application'][$pname] = $point;
            }else if(array_key_exists(strtolower($point->category()), $_audits)){
                $_audits[strtolower($point->category())][$pname] = $point;
            }else{
                $_audits['module'][$pname] = $point;
            }
        }
        $this->_audit_points = array_merge($_audits['system'], $_audits['application'], $_audits['module'], $_audits['service'], $_audits['process']);
        // Parent constructor to invoke unserialize if needed
        parent::__construct($parent_name, $name, $node);
    }
    public function configure()
    {
        foreach($this->_audit_points as $point)
        {
            $point->create_field($this);
        }

        return parent::configure();
    }
    public function generate_service_configuration(&$config_manager)
    {
        // Loop around audit points
        $service_cfg = array();
        foreach($this->_audit_points as $point)
        {
            // Update fields in audit point
            $point->set_field_value($this);
            // Get the configuration data
            $cfg = $point->monit_config();
            if ($cfg){
                // Merge service id conditions with header and check
                foreach($cfg as $k => $v){
                    $service_cfg[$k]['header'] = $v['header'];
                    $service_cfg[$k]['check'] = $v['check'];
                    if(!$service_cfg[$k]['condition']){
                        $service_cfg[$k]['condition'] = array();
                    }
                    $service_cfg[$k]['condition'] = array_merge($service_cfg[$k]['condition'], $v['condition']);
                }
            }
        }

        // Create the audit.conf array
        $audit_conf = array();
        foreach($service_cfg as $cfg){
            $audit_conf = array_merge($audit_conf, $cfg['header']);
            $audit_conf[] = $cfg['check'];
            if($cfg['condition']){
                $audit_conf = array_merge($audit_conf, $cfg['condition']);
            }
            $audit_conf[] = '';
        }

        // Got something to configure ?
        if($audit_conf){
            $config_manager->add_config(
                new Safe_configuration_class(
                    '/usr/local/sng/monitor/conf/audit.conf',
                    $audit_conf));
        }else{
            // Delete cfg
        }
    }
    public function audit_points_data()
    {
        foreach($this->_audit_points as $point)
        {
            // Update fields in audit point
            $point->set_field_value($this);
        }
        return $this->_audit_points;
    }
}
class Sng_auditor_event
{
    private $_audit_point = null;
    private $_event_data = array();
    private $_service_desc = '';
    private $_event_desc = '';
    public function __construct($audit_point, $event_data)
    {
        $this->_audit_point = $audit_point;
        $this->_event_data = $event_data;
        $this->_service_desc = $audit_point->module()->description();
        $this->_event_desc = $audit_point->description();
        if($this->_event_desc == $this->_service_desc){
            $this->_event_desc = 'Service';
        }
    }

    public function is_enabled()
    {
        $action = $this->_audit_point->action();
        return ('false' !== $action);
    }
    public function level($resolve=false)
    {
        return $this->_audit_point->level($resolve);
    }
    public function action($resolve=true)
    {
        return $this->_audit_point->action($resolve);
    }
    public function category()
    {
        return $this->_audit_point->category();
    }
    public function description()
    {
        return $this->_audit_point->description();
    }
    public function service_desc()
    {
        return $this->_service_desc;
    }
    public function event_desc()
    {
        return $this->_event_desc;
    }
    public function value()
    {
        return $this->_audit_point->value();
    }
    public function data()
    {
        $data = array();
        $messages = (string)$this->_event_data->message; 
        // Check if XML content
        $xml = simplexml_load_string($messages);
        if($xml){
            // Try to locate program > message
            $msg = $xml->xpath('/program/message');
            if(FALSE !== $msg){
                foreach($msg as $m){
                    $data[] = (string)$m;
                }
            }
        }else{
            $data = explode(PHP_EOL, $messages);
        }

        return $data;
    }
    public function date()
    {
        return (int)$this->_event_data->collected_sec;
    }
    public function to_array()
    {
        $event = Safe_xml_to_array($this->_event_data->asXml());
        // Override message
        $event['message'] = $this->data();
        return $event;
    }
    public function module()
    {
        return $this->_audit_point->_module;
    }
    public function status()
    {
        $state = (string)$this->_event_data->state;
        if('0' == $state){
            return 'SUCCESS';
        }elseif('2' == $state){
            // state change, figure out it START -> SUCCESS otherwise ERROR
            $action = (string)$this->_event_data->action;
            if('6' == $action){
                return 'SUCCESS';
            }
        }

        return 'ERROR';
    }

}
/*
Monit resource ID defines
-------------------------
#define RESOURCE_ID_CPU_PERCENT       1
#define RESOURCE_ID_MEM_PERCENT       2
#define RESOURCE_ID_MEM_KBYTE         3
#define RESOURCE_ID_LOAD1             4
#define RESOURCE_ID_LOAD5             5
#define RESOURCE_ID_LOAD15            6
#define RESOURCE_ID_CHILDREN          7
#define RESOURCE_ID_TOTAL_MEM_KBYTE   8
#define RESOURCE_ID_TOTAL_MEM_PERCENT 9
#define RESOURCE_ID_INODE             10
#define RESOURCE_ID_SPACE             11
#define RESOURCE_ID_CPUUSER           12
#define RESOURCE_ID_CPUSYSTEM         13
#define RESOURCE_ID_CPUWAIT           14
#define RESOURCE_ID_TOTAL_CPU_PERCENT 15
#define RESOURCE_ID_SWAP_PERCENT      16
#define RESOURCE_ID_SWAP_KBYTE        17

Monit state ID defines
----------------------
#define STATE_SUCCEEDED    0
#define STATE_FAILED       1
#define STATE_CHANGED      2
#define STATE_CHANGEDNOT   3
#define STATE_INIT         4

Monit action used with state change
#define ACTION_IGNORE      0
#define ACTION_ALERT       1
#define ACTION_RESTART     2
#define ACTION_STOP        3
#define ACTION_EXEC        4
#define ACTION_UNMONITOR   5
#define ACTION_START       6
#define ACTION_MONITOR     7

*/

define ("RESOURCE_ID_CPU_PERCENT",       1);
define ("RESOURCE_ID_MEM_PERCENT",       2);
define ("RESOURCE_ID_MEM_KBYTE",         3);
define ("RESOURCE_ID_LOAD1",             4);
define ("RESOURCE_ID_LOAD5",             5);
define ("RESOURCE_ID_LOAD15",            6);
define ("RESOURCE_ID_CHILDREN",          7);
define ("RESOURCE_ID_TOTAL_MEM_KBYTE",   8);
define ("RESOURCE_ID_TOTAL_MEM_PERCENT", 9);
define ("RESOURCE_ID_INODE",             10);
define ("RESOURCE_ID_SPACE",             11);
define ("RESOURCE_ID_CPUUSER",           12);
define ("RESOURCE_ID_CPUSYSTEM",         13);
define ("RESOURCE_ID_CPUWAIT",           14);
define ("RESOURCE_ID_TOTAL_CPU_PERCENT", 15);
define ("RESOURCE_ID_SWAP_PERCENT",      16);
define ("RESOURCE_ID_SWAP_KBYTE",        17);

class Sng_auditor_service_class extends Safe_service_class
{
    // Auditor Monit helpers
    /**
     * @brief Helper to convert Monit service name + type + id to Audit point 
     * name
     *
     * @param[in out] $name
     * @param[in out] $type
     * @param[in out] $id
     *
     * @return 
     */
    private function _monit_service_type_id_to_name($service, $type, $id, $msg)
    {
        if($service == 'Monit'){
            $service = 'monitor/service';
        }
        $name = $service;
        switch($type){
            case '5':{ // TYPE_SYSTEM
                $service = 'system';
                // Monit doesn't send (as expected) the resource_id field in system resource 
                // event report (id always 2 for any resource), but something similar to:
                //  msg=cpu user usage of 1.7% matches resource limit [cpu user usage>1.0%]
                // From code inspection resource name is 1st word in sentence, 
                // only exception being loadvg which contains the sampling ie. 
                // loadavg(1min), loadavg(5min) ...
                // So extract 1 word (ie. resource name) before space or "(" 
                // gives the proper resource name
                // Monit other way to report success:
                // 'mtl-nsc-review-will.sangoma.local' cpu user usage check succeeded [current cpu user usage=0.7%]
                // meaning we need to skip the system name
                switch($id){
                    case '2':{ // this is a system resource event, go figure which
                        $_matches = preg_split("/[(\s]+/", $msg);
                        $_tokens = array('cpu', 'memory', 'loadavg');
                        if(in_array($_matches[0], $_tokens)){
                            $name = $service.'/'.$_matches[0];
                        }else if(in_array($_matches[1], $_tokens)){
                            $name = $service.'/'.$_matches[1];
                        }else{
                            error_log('pattern mismatch on <'.$msg.'>');
                        }
                        break;
                    }
                    break;
                }
            }
            break;
        }
        return $name;
    }
    private $_audit_points = array();
    public function __construct($software)
    {
        parent::__construct($software, 'auditor');
    }
    /**
     * @brief
     *
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("Auditor");

        return parent::configure();
    }
    public function post_configure()
    {
        // query audit points
        $this->node()->get_audit_points(&$this->_audit_points);
        //safe_var_dump($audit_points);
        // Create the config object
        $cfg = new Sng_auditor_config_class($this->node(), parent::path(),$this->name(), $this->_audit_points);
        // Synch with DB
        $cfg->configure();
        $cfg->synch();
        // Attach config to module
        $this->set_config($cfg);
        return parent::post_configure();
    }
    public function allow_user_ctl()
    {
        return 'hide';
    }
    public function allow_dashboard()
    {
        return false;
    }
    public function generate_service_configuration(&$config_manager)
    {
        return $this->config()->generate_service_configuration(&$config_manager);
    }
    public function audit_points()
    {
        return $this->config()->audit_points_data();
    }

    private function _create_event_obj($event_xml)
    {
        $audit_point = null;
        $event = null;
        // Get the service name (id)
        $service = $event_xml->service;
        // Get type and id so we can adjust service(id) to audit name
        $type =  $event_xml->type;
        $id =  $event_xml->id;
        try{
            $msg = (string)$event_xml->message;
        }catch(Exception $e){
            $msg = "";
        }

        $name = $this->_monit_service_type_id_to_name($service, $type, $id, $msg);

        //$audit_point = $this->_audit_points[$name];
        foreach($this->audit_points() as $k => $v){
            if(!strcmp($name, $k)){
                $audit_point = $v;
                break;
            }
        }
        if(null != $audit_point){
            $event = new Sng_auditor_event(
                $audit_point,
                $event_xml);
        }else{
            error_log('ERROR - Fail to locate '.$name.' audit point for service <'.$name.'> Type <'.$type.'> Id <'.$id.'>');
        }
        return $event;
    }

    public function collector($data)
    {
        $events = array();
        $status = array();
        // Create XML doc
        try{
            $xml_data = simplexml_load_string($data);
        }catch(Exception $e){
            error_log('Invalid Monit XML data');
            return array();
        }

        // Event array
        $events = array();
        $dbg = false;

        // Process events
        $monit_events = $xml_data->xpath('/monit/event');
        foreach($monit_events as $monit_event){
            $msg = 'Event '.$monit_event->service;
            if($dbg)
                error_log($msg);
            $event = $this->_create_event_obj($monit_event);
            if($event){
                if($dbg)
                    error_log('Audit='.$event->description());
                // Add to event array using level for indexing
                $events[$event->level()][] = $event;
            }
        }
        if($dbg)
            file_put_contents('/tmp/monit/monitor.event', Safe_format_xml($data), FILE_APPEND);
        // If no /monit/services/service[@name='system']/system/load then just 
        // heart-beat - skip
        if($xml_data->xpath('/monit/services/service/system/load')){
            // Process services status
            $monit_services = $xml_data->xpath('/monit/services/service');
            foreach($monit_services as $service){
                $attr = $service->attributes();
                $name = (string)$attr->name;
                $service_status = array();
                // Check System, service or program
                $service_status = array();
                if('5' == $service->type){
                    $type = 'system';
                    $name = 'system';
                    $service_status = safe_xml_to_array($service->system->asXML());
                }
                if('3' == $service->type){
                    $type = 'service';
                    $xml_service_data = safe_xml_to_array($service->asXML());

                    $service_status['memory'] = $xml_service_data['memory'];
                    $service_status['cpu'] = $xml_service_data['cpu'];

                    $res_limit_mask = (int)$xml_service_data['resource_hint'];
                    if($service_status['memory']){
                        if( $res_limit_mask & (1<<RESOURCE_ID_TOTAL_MEM_PERCENT)){
                            $service_status['memory']['limit'] = true;
                        }else{
                            $service_status['memory']['limit'] = false;
                        }
                    }

                    if($service_status['cpu']){
                        if( $res_limit_mask & (1<<RESOURCE_ID_TOTAL_CPU_PERCENT)){
                            $service_status['cpu']['limit'] = true;
                        }else{
                            $service_status['cpu']['limit'] = false;
                        }
                    }
                }
                if('7' == $service->type){
                    $type = 'statistics';
                    $xml_service_data = safe_xml_to_array($service->program->asXML());
                    $service_status['output'] = $xml_service_data['output'];
                }
                    // has status ?
                if($service_status){
                    if($dbg)
                        error_log('Report '.$type.':'.$name);
                    $status[] = array(
                        'name' => $name, 
                        'type' => $type,
                        'status' => (string)$service->status,
                        'data' => $service_status);
                }
            }
        }

        if($dbg)
            file_put_contents('/tmp/monit/monitor.report', Safe_format_xml($data), FILE_APPEND);

        // Return processed data
        return array(
            'events' => $events,
            'status' => $status);
    }
}
/* End of file sng_sshd_service_class.php */
