#!/usr/bin/env python2.7
# vim: tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab
"""
* Copyright (C) 2011   Sangoma Technologies Corp.
* All Rights Reserved.
*
* Author(s)
* William Adam <william.adam@sangoma.com>
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.
"""

import requests
import json
import collections
import copy

from .object import Object
from .logger import logger as defaultlogger

DEFAULT_HOST = '127.0.0.1'
DEFAULT_PORT = 81
DEFAULT_PROTOCOL = 'http'

_SAFE_REST_URI = '/SAFe/sng_rest/'

class Server(Object):
    def __init__(self, name, description=None, logger=None):
        self._uri = None
        self.logger = logger or defaultlogger
        super(Server, self).__init__(None, name, description)

    def connect(self,
            host=DEFAULT_HOST,
            port=DEFAULT_PORT,
            protocol=DEFAULT_PROTOCOL,
            **kwargs):
        self.host = host
        self.port = str(port)
        self.protocol = protocol
        # Build requests uri
        self._uri = self.protocol + '://' + self.host + ':' + self.port + _SAFE_REST_URI
        self._api_uri = self._uri + 'api/'

        # Create a session
        self._session = requests.Session()
        self._adapter = requests.adapters.HTTPAdapter()
        self._session.mount(self.protocol + '://', self._adapter)

        # Custom headers
        self._session.headers = {'content-type': 'application/json'}

        self.logger.info('connect: ' + self._uri)

    def discconnect(self):
        self.logger.info('disconnect:' + self._uri)

    def _data_to_safe(self, data, sub_data=None, base_name=None):
        if not sub_data:
            sub_data = data

        if isinstance(sub_data, collections.Mapping):
            for k, v in sub_data.items():
                if base_name:
                    _name = base_name + '/' + k
                else:
                    _name = k

                if isinstance(v, collections.Mapping):
                    self._data_to_safe(data, v, _name)
                    del sub_data[k]
                else:
                    data[_name] = v

        return data

    # Create nested dict from list and adding value to last dict level
    def _nested_dict(self, d, l, v):
        _l = l.pop(0)
        if len(l):
            d[_l] = dict()
            self._nested_dict(d[_l], l, v)
        else:
            d[_l] = v

    # From http://www.xormedia.com/recursively-merge-dictionaries-in-python/
    def _merge_dict(self, a, b):
        '''recursively merges dict's. not just simple a['key'] = b['key'], if
        both a and bhave a key who's value is a dict then dict_merge is called
        on both values and the result stored in the returned dictionary.'''
        if not isinstance(b, dict):
            return b
        result = copy.deepcopy(a)
        for k, v in b.iteritems():
            if k in result and isinstance(result[k], dict):
                    result[k] = self._merge_dict(result[k], v)
            else:
                result[k] = copy.deepcopy(v)
        return result

    def _data_from_safe(self, data):
        if isinstance(data, collections.Mapping):
            for k, v in data.items():
                tokens = k.split('/')
                if len(tokens) > 1:
                    _data = dict()
                    self._nested_dict(_data, tokens, v)
                    data = self._merge_dict(data, _data)
                    del data[k]

        return data


    _last_error = {}
    @property
    def last_error(self):
        return self._last_error

    _last_response = {}
    @property
    def last_response(self):
        return self._last_response

    def _wrap_result(self, r):
        self._last_error = {}
        self._last_response = r.text
        response = json.loads(r.text)
        data = []
        if 200 == r.status_code:
            # Any data returned ?
            if 'data' in response:
                data = self._data_from_safe(response['data'])
            else:
                data = self._data_from_safe(response)
            return data
        else:
            # Any error returned ?
            self._last_error['status'] = r.status_code
            if 'error' in response:
              self._last_error['error'] = self._data_from_safe(response['error'])

        return False

    def do_post(self, uri, params=None):
        self.logger.info('POST: ' + uri)
        if params:
          p = params.copy()
          p = self._data_to_safe(p)
          r = self._session.post(uri, data=json.dumps(p))
        else:
          r = self._session.post(uri)
        return self._wrap_result(r)
        
    def do_get(self, uri, params=None):
        self.logger.info('GET: ' + uri)
        if params:
          p = params.copy()
          p = self._data_to_safe(p)
          r = self._session.get(uri, data=json.dumps(p))
        else:
          r = self._session.get(uri)
        return self._wrap_result(r)
        

    def delegate(self, request, method, path=None, params=None, category='api'):
        if not self._uri:
            self.connect()

        _uri = self._uri + category
        if method:
            _uri = _uri + '/'  + method 
        if path:
            _uri = _uri + '/' + path
        # Call proper do_<request>
        action = getattr(self, 'do_' + request.lower())

        return action(_uri, params)

