<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * FS SIP module class
*/
require_once ('application/helpers/safe_helper.php');
safe_require_class('safe_service_class');
safe_module_require_class('fs', 'application');

class Fs_sip_module_class extends Fs_module_class
{
    protected $_obj_base_path = '';
    private $_sip_profiles = array();
    private $_sip_trunk = array();
    private $_db = "";
    protected $_channel_data_cache = array();
    protected $_channel_list_cache = array();
    
    private $_sip_fields_enum = array();
    /**
     * @brief
     *           
     * @param[in out] $fs_app
     *           
     * @return
     */
    public function __construct($fs_app)
    {
        parent::__construct($fs_app, "sip");
        $this->_obj_base_path = $this->path().'/sip';
        $this->_db = $this->db();
    }
    /**
     * @brief
     *         
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("SIP");
        // Expose aggregate obj
        $this->register_aggregate_object('trunk', 
            array(
                'name' => 'Trunk',
                'dynamic' => true,
                'base_path' => $this->_obj_base_path . '/trunk',
                'controller_url' => '/SAFe/fs_sip_trunk_config',
                    'methods' => array(
                            'create' => array(
                                    'name' => 'Create',
                                    'description' => 'Create a SIP Trunk',
                                    'request' => 'POST',
                            ),
                            'retrieve' => array(
                                    'name' => 'Retrieve',
                                    'description' => 'Retrieve a SIP Trunk',
                                    'request' => 'GET',
                            ),
                            'update' => array(
                                    'name' => 'Update',
                                    'description' => 'Update a SIP Trunk',
                                    'request' => 'POST',
                            ),
                            'delete' => array(
                                    'name' => 'Delete',
                                    'description' => 'Delete a SIP Trunk',
                                    'request' => 'POST',
                            ),
                            'status' => array(
                                    'name' => 'Retrieve Status',
                                    'descripton' => 'Retrieve SIP Trunk Status',
                                    'request' => 'GET',
                            ),
                    ),
            )
        );
        $this->register_aggregate_object('profile', 
            array(
                'name' => 'Profile',
                'dynamic' => true,
                'base_path' => $this->_obj_base_path . '/profile',
                'controller_url' => '/SAFe/fs_sip_profile_config',
                'has_child' => true,
                'methods' => array(
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Create a SIP Profile',
                        'request' => 'POST',
                        ),
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a SIP Profile',
                        'request' => 'GET',
                        ),
                    'update' => array(
                                'name' => 'Update',
                                'description' => 'Update a SIP Profile',
                                'request' => 'POST',
                        ),
                    'delete' => array(
                                'name' => 'Delete',
                                'description' => 'Delete a SIP Profile',
                                'request' => 'POST',
                        ),
                    'start' => array(
                                'name' => 'Start',
                                'descripton' => 'Start a SIP Profile',
                                'request' => 'POST',
                      ),
                    'stop' => array(
                                'name' => 'Stop',
                                'descripton' => 'Stop a SIP Profile',
                                'request' => 'POST',
                      ),
                    'status' => array(
                                'name' => 'Retrieve Status',
                                'descripton' => 'Retrieve SIP Profile Status',
                                'request' => 'GET',
                     ),
                    ),
                 )
             );
        $this->register_aggregate_object('channel',
                array(
                        'name' => 'Channel',
                        'configurable' => false,// filter out from sng_config_controller
                        'pagination' => true,
                        'dynamic' => true,
                        'base_path' => false,
                        'global_methods' => true, //some methods may be used without obj_name 
                        'methods' => array(
                                'retrieve' => array(
                                        'name' => 'Retrieve',
                                        'description' => 'Retrieve a Channel',
                                        'request' => 'GET',
                                        'scope' => 'both' //Can be executed either on obj and globally 
                                ),
                        ),
                )
        );
        return parent::configure();
    }
    /**
     * Returns sip profiles in an array
     */
    public function profiles()
    {
        $this->_sip_profiles = $this->get_aggregate_objects('profile');
        return $this->_sip_profiles;
    }
    /**
     * Returns sip trunk profiles in an array
     */
    public function trunks()
    {
        $this->_sip_trunks = $this->get_aggregate_objects('trunk');
        return $this->_sip_trunks;
    }
    /**
     * @brief Module support reload ?
     *
     * @param[in out] $need_reload
     *
     * @return 
     */
    public function support_reload(&$need_reload = null) {
        parent::support_reload($need_reload);
        return true;
    }
    /**
     * @brief Create config jobs for dialplan module
     *           
     * @param[in out] $config_manager
     *           
     * @return
     */
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        if(!$this->_generate_config($config_manager, $obj_type)) return false;
        return parent::reload_generate_config($config_manager, $obj_type);
    }

    /**
     * @brief 
     *
     * @param[in out] $obj_type
     *
     * @return 
     */
    public function reload_clear_modified($obj_type=null)
    {
        return $this->clear_configuration_modified();
    }

    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    public function generate_config(&$config_manager)
    {
        if(!$this->_generate_config($config_manager)) return false;
        return parent::generate_config($config_manager);
    }
    public function check_resource_conflict($src_module_name, $resource_type, $resource_data)
    {
        // IP_PORT conflicts ?
        $conflicts = array();
        if('IP_PORT' == $resource_type){
            // get our configuration
            $if_enum = $this->node()->hardware()->adapters_ip();
            $profiles = $this->profiles();
            foreach($profiles as $profile){
                $data = $profile->get_data(false);
                //sip port
                $ports[] = array(   'name' =>$profile->get_data_label('sip-port'),
                                    'port'=>$profile->get_data_value('sip-port'), 
                                    'ip'=>$if_enum[$profile->get_data_value('sip-ip',false)], 
                                    'type' => $profile->get_data_value('transport',false)
                        );
                if(strpos(strtoupper($profile->get_data_value('transport',false)),'TLS') !== false){
                    //tls port
                    $ports[] = array(    'name' =>$profile->get_data_label('TLS/tls-sip-port'),
                                         'port'=>$profile->get_data_value('TLS/tls-sip-port'), 
                                         'ip'=>$if_enum[$profile->get_data_value('sip-ip',false)], 
                                         'type' => 'TCP'
                            );
                }
            }
            // Loop around resource data
            foreach($resource_data as $resource){
                foreach($ports as $port_item){
                    if(safe_check_port_conflict($port_item, $resource)){
                        $conflicts[] = array('name' => $this->name(), 'obj_type' =>'Configuration', 'desc' =>'port '.$port_item['port']);
                    }
                }
            }
        }
        return $conflicts;
    }
    /**
     * @brief Invoked after a successfull write_config
     *
     * @return
     */
    public function post_write_config($obj_type=null)
    {
        // Loop around profiles (ALL)
        $profiles = $this->get_aggregate_objects('profile', true);
        foreach($profiles as $profile) {
            // Proceed based on status
            switch($profile->status()) {
            case Safe_object_serializer_class::OBJ_STATUS_NEW:
                // stop the profile (in case the user created a new profile 
                // with the same name of an old profile), this command will fail 
                // most of the times
                $this->eslapi_execute( 'sofia profile '. $profile->name().' stop', '', $output);
                // start the new profile reading the new values from the XML
                $this->eslapi_execute( 'sofia profile '. $profile->name().' start', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_MODIFIED:
                // as simple as rescanning the new values from the XML
                $this->eslapi_execute( 'sofia profile '. $profile->name().' rescan', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE:
                break;
            case Safe_object_serializer_class::OBJ_STATUS_DELETED:
                // stop the profile is equivalent to delete it from memory 
                $this->eslapi_execute( 'sofia profile '. $profile->name().' stop', '', $output);
                break;
            }
        }
    
        // Loop around trunks (ALL)
        $trunks = $this->get_aggregate_objects('trunk', true);
        foreach($trunks as $trunk) {
            $profile_name = $trunk->get_data_value('sip_profile', false);
            // Proceed based on status
            switch($trunk->status()) {
            case Safe_object_serializer_class::OBJ_STATUS_NEW:
                $this->eslapi_execute( 'sofia profile '. $profile_name.' rescan', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_MODIFIED:
                $this->eslapi_execute( 'sofia profile '. $profile_name.' killgw '.$trunk->name(), '', $output);
                $this->eslapi_execute( 'sofia profile '. $profile_name.' rescan', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_DELETED:
                $this->eslapi_execute( 'sofia profile '. $profile_name.' killgw '.$trunk->name(), '', $output);
                break;
            }
        }
    
        return parent::post_write_config($obj_type);
    }
    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    private function _generate_config(&$config_manager, $obj_type=null)
    {
        // Validate
        // - Trunks
        if(!$obj_type || 'trunk' == $obj_type) {
        }
        // - Profiles
        if(!$obj_type || 'profile' == $obj_type) {
            if (!count($this->profiles())) {
                // At least one SIP profile is required
                $config_manager->add_error(array(
                    'module' => 'sip',
                    'obj_type' => 'profile',
                    'description' => 'Not defined',
                    ));
                //return false;
            }
        }

        // Cleanup
        // - Trunks
        if(!$obj_type || 'trunk' == $obj_type) {
        }
        // - Profiles
        if(!$obj_type || 'profile' == $obj_type) {
            // 1st, clean up SIP profile directory and xml
            foreach ($this->profiles() as $profile) {
                $tmp_call_routing = $profile->get_data_value('call-routing', false);
                if( $tmp_call_routing== '' || $tmp_call_routing == '__none__'){
                    $config_manager->add_error(array(
                            'module' => 'sip',
                            'obj_type' => 'profile',
                            'obj_name' => $profile->name(),
                            'description' => 'No routing defined.',
                    ));
                }
                $directory = "sip_profiles/" . $profile->name();
                $config_manager->add_config(
                    new Safe_configuration_class(
                        $directory, '', 
                        Safe_configuration_class::CFG_DELETE, 
                        Safe_configuration_class::CFG_DIR));
                $config_manager->add_config(
                    new Safe_configuration_class(
                        'sip_profiles/'.$profile->name().'.xml', '', 
                        Safe_configuration_class::CFG_DELETE, 
                        Safe_configuration_class::CFG_FILE));
            }
        }

        // Generate
        // - Trunks
        if(!$obj_type || 'trunk' == $obj_type) {
            // SIP Trunk configuration
            foreach ($this->trunks() as $profile) {
                if (!$this->_write_trunk($config_manager, $profile)) return false;
            }
        }
        // - Profiles
        if(!$obj_type || 'profile' == $obj_type) {
            //Serialize SIP Profiles
            foreach ($this->profiles() as $profile) {
                if (!$this->_write_profile($config_manager, $profile)) return false;
            }
            //Serialize SIP Profile Certificates
            if (!$this->_serialize_sip_certificates($config_manager)) return false;
            //Serialize SIP Profile Core Dialplan with CAC Rules
            $this->_create_core_dir($config_manager);
            foreach ($this->profiles() as $profile) {
                if (!$this->_write_sip_core($config_manager, $profile)) return false;
            }
        }

        // Handle deleted
        $this->_handle_deleted_trunks($config_manager);
        $this->_handle_deleted_profiles($config_manager);

        return true;
    }
    protected function _handle_deleted_trunks(&$config_manager)
    {
        //Handle Deleted SIP Trunk
        $deleted_sip_trunk = Safe_object_serializer_class::get_serializer()->find_deleted_objects(
            $this->_obj_base_path . '/trunk');
        foreach ($deleted_sip_trunk as $sip_trunk => $value) {
            $file = 'sip_profiles/' . $value['data']['sip_profile'] . '/' . $sip_trunk . '.xml';
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
            $file = 'dialplan/core/sip_trunks/'.$sip_trunk.'.xml';
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
        }
    }
    protected function _handle_deleted_profiles(&$config_manager)
    {
        // Handle deleted SIP profiles
        $deleted_sip_profiles = Safe_object_serializer_class::get_serializer()->find_deleted_objects(
            $this->_obj_base_path . '/profile');
        foreach ($deleted_sip_profiles as $sip_profile => $value) {
            // Delete configuration file
            $file = 'sip_profiles/' . $sip_profile . '.xml';
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
            //Remove Core Dialplan File
            $file = "dialplan/core/profiles/profile." . $sip_profile . ".xml";
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
            //Remove certificate folder on delete
            $directory = 'ssl/profiles/' . $sip_profile;
            $config_manager->add_config(
                new Safe_configuration_class(
                    $directory, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_DIR));
        }
        return true;
    }
    /**
     * Creates the Core dialplan folder
     * @param object  $config_manager
     * @return boolean
     */
    protected function _create_core_dir(&$config_manager)
    {
        // Create directory
        $directory = "dialplan/core/profiles";
        $config_manager->add_config(
            new Safe_configuration_class(
                $directory, '', 
                Safe_configuration_class::CFG_CREATE, 
                Safe_configuration_class::CFG_DIR));
        return true;
    }

    /**
     * Write Dialplan for sip profiles with cac rules
     * @param object  $config_manager
     * @param object  $profile       
     *                                 
     * @return boolean
     */
    protected function _write_sip_core(&$config_manager, $profile)
    {
        //Create core dialplan file for the sip profile
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0', 'UTF8');
        $xml_writer->startElement("include");
            //<!-- Ingress profile context -->
            $xml_writer->writeComment('Ingress profile context  ');
            $xml_writer->startElement("context");
            $xml_writer->writeAttribute('name', 'profile.' . $profile->name() . '.ingress');
                //Secure Media Element
                $xml_writer->startElement('extension');
                $xml_writer->writeAttribute('name', 'add_media_secure');
                $xml_writer->writeAttribute('continue', 'true');
                    $xml_writer->startElement('condition');
                    $xml_writer->writeAttribute('field', '${sip_has_crypto}');
                    $xml_writer->writeAttribute('expression', '^(AES_CM_128_HMAC_SHA1_32|AES_CM_128_HMAC_SHA1_80)$');
                        $xml_writer->StartElement('action');
                        $xml_writer->writeAttribute('application', 'set');
                        $xml_writer->writeAttribute('data', 'sip_secure_media=true');
                        $xml_writer->endElement();//action
                    $xml_writer->fullEndElement();//condition
                $xml_writer->fullEndElement();//extension
                //Limit Exceeded Element
                $xml_writer->startElement('extension');
                $xml_writer->writeAttribute('name', 'limit_exceeded');
                    $xml_writer->startElement('condition');
                    $xml_writer->writeAttribute('field', 'destination_number');
                    $xml_writer->writeAttribute('expression', '^limit_exceeded$');
                    
                        $xml_writer->startElement('action');
                        $xml_writer->writeAttribute('application', 'log');
                        $xml_writer->writeAttribute('data', 'info limit exceeded for IP: ${sip_received_ip} User:${sip_auth_username}@${sip_auth_realm}');
                        $xml_writer->endElement();
                    
                        $xml_writer->startElement('action');
                        $xml_writer->writeAttribute('application', 'hangup');
                        $xml_writer->writeAttribute('data', 'CALL_REJECTED');
                        $xml_writer->endElement();
                    $xml_writer->fullEndElement();//condition
                $xml_writer->fullEndElement();//extension
           
                //CAC Rules binding
                $xml_writer->startElement('X-PRE-PROCESS');
                $xml_writer->writeAttribute('cmd', 'optional_include');
                $xml_writer->writeAttribute('data', '../../cac-rules/rules-' . $profile->name() . '.xml');
                $xml_writer->endElement();
        
            //Transfer to User Element
            $data = $profile->get_data_values(false);
            $xml_writer->startElement('extension');
            $xml_writer->writeAttribute('name', 'transfer-to-user');
                // Set routing context
                $xml_writer->startElement('condition');
                $xml_writer->writeAttribute('field', '${sangoma_routing_context}');
                $xml_writer->writeAttribute('expression', '^$');
                $xml_writer->writeAttribute('break', 'never');
                    $xml_writer->writeComment('Set Routing context');
                    $xml_writer->StartElement('action');
                    $xml_writer->writeAttribute('application', 'set');
                    $xml_writer->writeAttribute('data', 'sangoma_routing_context=routing.' . $data['call-routing']);
                    $xml_writer->endElement();
                $xml_writer->endElement(); //condition
                // Set default destination
                $xml_writer->startElement('condition');
                $xml_writer->writeAttribute('field', '${sangoma_routing_destination}');
                $xml_writer->writeAttribute('expression', '^$');
                $xml_writer->writeAttribute('break', 'never');
                    $xml_writer->writeComment('Set default destination');
                    $xml_writer->StartElement('action');
                    $xml_writer->writeAttribute('application', 'set');
                    $xml_writer->writeAttribute('data', 'sangoma_routing_destination=${destination_number}');
                    $xml_writer->endElement();
                $xml_writer->endElement();//condition
                // Set hmr context
                $xml_writer->startElement('condition');
                $xml_writer->writeAttribute('field', '${sangoma_ingress_hmr_context}');
                $xml_writer->writeAttribute('expression', '^$');
                $xml_writer->writeAttribute('break', 'never');
                    $xml_writer->writeComment('Set HMR context');
                    $xml_writer->StartElement('action');
                    $xml_writer->writeAttribute('application', 'set');
                    $xml_writer->writeAttribute('inline', 'true');
                    if ('__none__' != $data['ingress']){
                        $xml_writer->writeAttribute('data', 'sangoma_ingress_hmr_context=hmr.' . $data['ingress']);
                    }else{
                        $xml_writer->writeAttribute('data', 'sangoma_ingress_hmr_context=');
                    }
                    $xml_writer->endElement();
                $xml_writer->endElement(); //condition
                // Set post hmr context
                $xml_writer->startElement('condition');
                $xml_writer->writeAttribute('field', '${sangoma_ingress_hmr_context}');
                $xml_writer->writeAttribute('expression', '^$');
                $xml_writer->writeAttribute('break', 'never');
                    $xml_writer->writeComment('Transfer to HMR context');
                    $xml_writer->StartElement('anti-action');
                    $xml_writer->writeAttribute('application', 'set');
                    $xml_writer->writeAttribute('data', 'sangoma_post_hmr_context=${sangoma_routing_context}');
                    $xml_writer->endElement();
                    $xml_writer->StartElement('anti-action');
                    $xml_writer->writeAttribute('application', 'transfer');
                    $xml_writer->writeAttribute('data', '${sangoma_routing_destination} XML ${sangoma_ingress_hmr_context}');
                    $xml_writer->endElement();
                $xml_writer->endElement(); //condition

                // Transfer to routing context
                $xml_writer->startElement('condition');
                $xml_writer->writeAttribute('field', 'destination_number');
                $xml_writer->writeAttribute('expression', '(.*)');
                    $xml_writer->writeComment('Transfer to routing context');
                    $xml_writer->StartElement('action');
                    $xml_writer->writeAttribute('application', 'transfer');
                    $xml_writer->writeAttribute('data', '$1 XML ${sangoma_routing_context}');
                    $xml_writer->endElement();
                $xml_writer->endElement();//condition
                
                
            $xml_writer->endElement();//extension
        $xml_writer->endElement();//context
        //<!-- Egress profile context -->
        $xml_writer->writeComment('Egress profile context  ');
        $xml_writer->startElement("context");
        $xml_writer->writeAttribute('name', 'profile.' . $profile->name() . '.egress');
            $xml_writer->startElement('extension');
            $xml_writer->writeAttribute('name', 'sangoma_profile_' . $profile->name() . '_egress');//need to get the name
                // <!-- Set routing stage variable so we can figure out where to go next in advanced hmr -->                                      
                $xml_writer->startElement('condition');
                    $xml_writer->writeComment('Set routing bridge');
                    $xml_writer->StartElement('action');
                    $xml_writer->writeAttribute('application', 'set');
                    $xml_writer->writeAttribute('data', 'sangoma_routing_bridge=sofia/' . $profile->name() );
                    $xml_writer->endElement();
                $xml_writer->endElement();//condition

                // Set hmr context
                $xml_writer->startElement('condition');
                $xml_writer->writeAttribute('field', '${sangoma_egress_hmr_context}');
                $xml_writer->writeAttribute('expression', '^$');
                $xml_writer->writeAttribute('break', 'never');
                    $xml_writer->writeComment('Set HMR context');
                    $xml_writer->StartElement('action');
                    $xml_writer->writeAttribute('application', 'set');
                    $xml_writer->writeAttribute('inline', 'true');
                    if ('__none__' != $data['egress']){
                        $xml_writer->writeAttribute('data', 'sangoma_egress_hmr_context=hmr.' . $data['egress']);
                    }else{
                        $xml_writer->writeAttribute('data', 'sangoma_egress_hmr_context=');
                    }
                    $xml_writer->endElement();
                $xml_writer->endElement(); //condition
                // Set post hmr context
                $xml_writer->startElement('condition');
                $xml_writer->writeAttribute('field', '${sangoma_egress_hmr_context}');
                $xml_writer->writeAttribute('expression', '^$');
                $xml_writer->writeAttribute('break', 'never');
                    $xml_writer->writeComment('Transfer to HMR context');
                    $xml_writer->StartElement('anti-action');
                    $xml_writer->writeAttribute('application', 'set');
                    $xml_writer->writeAttribute('data', 'sangoma_post_hmr_context=profile.' . $profile->name() . '.outbound');
                    $xml_writer->endElement();
                    $xml_writer->StartElement('anti-action');
                    $xml_writer->writeAttribute('application', 'transfer');
                    $xml_writer->writeAttribute('data', '${sangoma_routing_destination} XML ${sangoma_egress_hmr_context}');
                    $xml_writer->endElement();
                $xml_writer->endElement(); //condition

                // Bridge
                $xml_writer->startElement('condition');
                    $xml_writer->writeComment('Bridge');
                    $xml_writer->StartElement('action');
                    $xml_writer->writeAttribute('application', 'bridge');
                    $xml_writer->writeAttribute('data', '${sangoma_routing_bridge}/${sangoma_routing_destination}');
                    $xml_writer->endElement();
                $xml_writer->endElement();//condition
            $xml_writer->endElement();//extension
        $xml_writer->endElement();//context                
        
        //<!-- Egress outbound context -->
        $xml_writer->writeComment('Outbound profile context ');
        $xml_writer->startElement('context');
        $xml_writer->writeAttribute('name', 'profile.' . $profile->name() . '.outbound');
            $xml_writer->startElement('extension');
            $xml_writer->writeAttribute('name', 'sangoma_profile_' . $profile->name() . '_outbound');//need to get the name
                    $xml_writer->startElement('condition');
                        $xml_writer->writeComment('Bridge');
                        $xml_writer->StartElement('action');
                        $xml_writer->writeAttribute('application', 'bridge');
                        $xml_writer->writeAttribute('data', '${sangoma_routing_bridge}/${sangoma_routing_destination}');
                        $xml_writer->endElement();
                    $xml_writer->endElement();//condition
            $xml_writer->endElement();//extension
        $xml_writer->endElement();//context
        // Close all opened elements
        while ($xml_writer->endElement() !== false) {
            continue;
        }
        // Close document
        $xml_writer->endDocument();
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = "dialplan/core/profiles/profile." . $profile->name() . ".xml";
        $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));
        return true;
    }
    /**
             * Serializes all SIP Certificates placed in a SIP Profile
             * @param $config_manager
             */
    protected function _serialize_sip_certificates(&$config_manager)
    {
        // Let certificate module do its job :)
        return true;
    }
    protected function _write_profile(&$cfg_mgr, $sip_profile)
    {
        $profile_name = $sip_profile->name();
        $no_limit = false;
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0', 'UTF-8');
        $xml_writer->startElement("profile");
        $xml_writer->writeAttribute("name", $profile_name);
        $this->_write_profile_default_tags($xml_writer, $profile_name);

        $sip_profile_cfg = $sip_profile->get_data_values_exploded(false);
        foreach ($sip_profile_cfg['__all__'] as $k => $v) {
            if ($k == "sip-ip") {
                // Get back ip from interface list
                $transcoding_mode = new Sng_transcoding_config_class($this->node());
                $mgmt_if = $this->node()->hardware()->adapters();
                if (isset($mgmt_if) && isset($mgmt_if[$v]) && strlen($mgmt_if[$v]->ip_address())) {
                    if (!$transcoding_mode->require_transcoding_interface() && Safe_hardware_adapter_class::TRANSCODING_ADAPTER == $mgmt_if[$v]->type()) {
                        $cfg_mgr->add_error(array(
                                'module' => 'sip',
                                'obj_type' => 'profile',
                                'description' => 'SIP Profile cannot use ' . $v . ' interface when in Hidden Mode.',
                                'url' => '/SAFe/fs_sip_profile_config/modify/sip/profile/'.$profile_name
                        ));
                    }
                    $v = $mgmt_if[$v]->ip_address();
                } else {
                    // TODO(wadam) - We're in trouble...
                    // TODO(wadam) - Redirect to Managment IF config page
                    //'/SAFe/ntg_sip_profiles/edit/'.$profile_name,
                    $cfg_mgr->add_error(array(
                            'module' => 'sip',
                            'obj_type' => 'profile',
                            'description' => 'SIP profile ' . $profile_name . ' bind to interface ' . $v . ' without IP address.',
                            'url' => '/admin/network.php'
                    ));
                    $v = '0.0.0.0';
                    //return false;
                    
                }
            }
            // Do some filtering on param
            switch ($k) {
            case 'domains':
            case 'call-routing':
            case 'ingress':
            case 'egress':
                // Stuff to skip
                break;

            case 'transport':
                $this->_write_transport_param($xml_writer, $v);
                $transport_value = $v;
                break;

            case 'inbound-media-profile':
                if($v && $v != '__none__') {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../media_profiles/" . $v . "/inbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case 'outbound-media-profile':
                if($v && $v != '__none__') {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../media_profiles/" . $v . "/outbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case 'sip-tos-value':
                if ($v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'sip-tos-value');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'rtp-tos-value':
                if ($v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'rtp-tos-value');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'enable-load-limit':
                if ($v == "false") $no_limit = true;
                elseif ($v == "true") $no_limit = false;
                $xml_writer->startElement('param');
                $xml_writer->writeAttribute('name', 'enable-load-limit');
                $xml_writer->writeAttribute('value', $v);
                $xml_writer->EndElement();
                break;

            case 'max-sessions':
                if (!$no_limit && $v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'max-sessions');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-high-threshold':
                if (!$no_limit) {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-high-threshold');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-low-threshold':
                if (!$no_limit) {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-low-threshold');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-reject-cause':
                if (!$no_limit) {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-reject-cause');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-cause-string':
                if (!$no_limit && $v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-cause-string');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'rtp-ip':
            case 'ext-rtp-ip':
            case 'ext-sip-ip':
            case 'outbound-proxy':
            case 'apply-nat-acl':
            case 'fqdn-in-contact':
            case 'max-sip-request-length':
                if (strlen(trim($v))) {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;
            case 'sip-dialplan-handler':
                if ('__none__' != trim($v)) {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;
            default:
                if('__disable__' != $v) {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;
            }
        }
        //Add TLS Parameters
        if (sizeof($sip_profile_cfg['TLS'])) {
            $sip_tls = $sip_profile_cfg['TLS'];
            if ($transport_value == "udptcptls" || $transport_value == "tls") {
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", 'tls');
                $xml_writer->writeAttribute("value", 'true');
                $xml_writer->EndElement();
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", "tls-cert-dir");
                $xml_writer->writeAttribute("value", "/usr/local/nsc/conf/ssl/profiles/" . $profile_name);
                $xml_writer->EndElement();
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", "tls-bind-params");
                $xml_writer->writeAttribute("value", 'transport=tls');
                $xml_writer->EndElement();
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", "tls-verify-depth");
                $xml_writer->writeAttribute("value", '10');
                $xml_writer->EndElement();
                if ($transport_value == "tls") {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", 'tls-only');
                    $xml_writer->writeAttribute("value", 'true');
                    $xml_writer->EndElement();
                }
                foreach ($sip_tls as $k => $v) {
                    if ($k != 'certificate' && '__disable__' != $v){
                        //Print TLS params
                        $xml_writer->startElement("param");
                        $xml_writer->writeAttribute("name", $k);
                        $xml_writer->writeAttribute("value", $v);
                        $xml_writer->EndElement();
                    }
                }
                // Check srtp parameters
                if ($sip_tls['enable-secure-media'] == 'true'){
                    $srtp = $sip_profile_cfg['srtp'];
                    foreach ($srtp as $k => $v) {
                        if ('__disable__' != $v){
                            //Print SRTP params
                            $xml_writer->startElement("param");
                            $xml_writer->writeAttribute("name", $k);
                            $xml_writer->writeAttribute("value", $v);
                            $xml_writer->EndElement();
                        }
                    }
                }
            }
        }
        // Include custom xml
        $xml_writer->startElement("X-PRE-PROCESS");
        $xml_writer->writeAttribute("cmd", "optional_include");
        $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/sip_profiles/' . $profile_name . '.profile.settings.xml');
        $xml_writer->EndElement();
        
        $xml_writer->fullEndElement(); //end settings tag
        // Add the domains
        if (sizeof($sip_profile_cfg['__all__']['domains'])) {
            $sip_domains = $sip_profile_cfg['__all__']['domains'];
            $domains = $this->fs_app()->domain_profiles();
            $xml_writer->startElement("domains");
            foreach ($sip_domains as $sip_domain) {
                // Check domain exists
                if (isset($domains[$sip_domain])) {
                    $domain = $domains[$sip_domain];
                    $xml_writer->startElement("domain");
                    $xml_writer->writeAttribute("name", $domain->name());
                    $xml_writer->writeAttribute("alias", "false");
                    $xml_writer->writeAttribute("parse", "false");
                    $xml_writer->EndElement(); //end domain tag
                    
                } else {
                    // TODO(wadam) - Log and error
                    // Should not occurs
                    
                }
            }
            $xml_writer->fullEndElement(); //end domains tag
            
        }
        //Serialize SIP Limits
        $limits = $sip_profile->request_limits();
        $xml_writer->startElement("limits");
        if (!empty($limits)) {
            foreach ($limits as $limit) {
                $limit_data = $limit->get_data_values();
                $xml_writer->startElement('request-rate');
                if ($limit_data['host'] == "") $xml_writer->writeAttribute("host", "ANY");
                else $xml_writer->writeAttribute("host", $limit_data['host']);
                $xml_writer->writeAttribute("method", $limit_data['method']);
                $xml_writer->writeAttribute("rate", $limit_data['limit'] . '/' . $limit_data['period']);
                $xml_writer->EndElement();
            }
        }
        $xml_writer->fullEndElement();
        $xml_writer->fullEndElement(); //end profile tag
        $xml_writer->endDocument();
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = "sip_profiles/" . $profile_name . ".xml";
        $cfg_mgr->add_config(new Safe_configuration_class($file, $xmlstr));
        return true;
    }
    protected function _write_profile_default_tags(&$xml_writer, $profile_name)
    {
        // Check if this profile is bound to some SIP trunks
        $trunks = $this->trunks();
        foreach ($trunks as $trunk) {
            $cfg = $trunk->get_data_values();
            if ($cfg['sip_profile'] == $profile_name) {
                // Include proxy information
                $xml_writer->startElement("gateways");
                $xml_writer->startElement("X-PRE-PROCESS");
                $xml_writer->writeAttribute("cmd", "include");
                $xml_writer->writeAttribute("data", $profile_name . "/*.xml");
                $xml_writer->EndElement();
                $xml_writer->fullEndElement();
                // break we just want to know if at least one gateway is bound
                break;
            }
        }
        // Open settings tag
        $xml_writer->startElement("settings");
        // Include parameters common to all sip profile
            $xml_writer->startElement("X-PRE-PROCESS");
            $xml_writer->writeAttribute("cmd", "include");
            $xml_writer->writeAttribute("data", "common/common.xml");
        $xml_writer->EndElement();
        // Start param list
        $xml_writer->startElement("param");
        $xml_writer->writeAttribute("name", "context");
        //$xml_writer->writeAttribute("value", self::NSC_CORE_ROUTING_CONTEXT);
        $xml_writer->writeAttribute('value', 'profile.' . $profile_name . '.ingress');
        $xml_writer->EndElement();
        
        $xml_writer->startElement("param");
        $xml_writer->writeAttribute("name", "dialplan");
        $xml_writer->writeAttribute("value", "XML");
        $xml_writer->EndElement();
        
        $xml_writer->startElement("param");
        $xml_writer->writeAttribute("name", "t38-passthru");
        $xml_writer->writeAttribute("value", "true");
        $xml_writer->EndElement();
    }
    protected function _write_transport_param(&$xml_writer, $transport_value)
    {
        if ($transport_value != "udptcp" && $transport_value != "udptcptls" && $transport_value != "tls") {
            $xml_writer->startElement("param");
            $xml_writer->writeAttribute("name", "bind-params");
            $xml_writer->writeAttribute("value", "transport=" . $transport_value);
            $xml_writer->EndElement();
        }
    }
    protected function _write_trunk(&$config_manager, $sip_trunk)
    {
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0', 'UTF-8');
        $sip_trunk_cfg = $sip_trunk->get_data_values(false, false);
        $profile_name = $sip_trunk->name();
        $xml_writer->startElement("include");
        $xml_writer->startElement("gateway");
        $xml_writer->writeAttribute("name", $profile_name);
        // Latch some values
        $sip_profile = $sip_trunk_cfg['sip_profile'];
        // Perform some validation with profile
        $profile_obj = $this->api_retrieve_profile($sip_profile);
        if($profile_obj){
            $sip_profile_cfg = $profile_obj->get_data_values(false);
            if(!$profile_obj->check_transport($sip_trunk->get_transport())){
                $trunk_obj_def = $this->get_aggregate_object_definition('trunk');
                if($trunk_obj_def['controller_url']){
                    $fix_url = $trunk_obj_def['controller_url'];
                    // TODO: redirect to edit trunk doesn't work...
                    // .'/edit/'.$profile_name;
                }
                $config_manager->add_error(array(
                        'module' => 'sip',
                        'obj_type' => 'trunk',
                        'description' => 'Trunk ' .$profile_name .' transport '. $sip_trunk->get_transport(true).' incompatible with Profile '.$profile_obj->name().' transport.',
                        'url' => '/SAFe/fs_sip_trunk_config/edit/sip/trunk/'.$profile_name
                ));
            }
        }
        // Remove from cfg list
        unset($sip_trunk_cfg['enabled']);
        unset($sip_trunk_cfg['sip_profile']);

        // Check if registration enabled
        if('true' != $sip_trunk_cfg['register']){
            $sip_trunk_cfg['register-proxy'] = "";
        }

        foreach ($sip_trunk_cfg as $k => $v) {
            switch ($k) {
            case "ingress":
            case "egress":
                // stuff to skip
                break;

            case 'inbound-media-profile':
                if($v && $v != '__none__') {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../../media_profiles/" . $v . "/inbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case 'outbound-media-profile':
                if($v && $v != '__none__') {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../../media_profiles/" . $v . "/outbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case "call-routing":
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", 'context');
                $xml_writer->writeAttribute("value", "trunk.".$profile_name.'.ingress');
                $xml_writer->EndElement();
                break;
            default:
                if ($v == ""){
                    // Write the empty attribute but commented out
                    $xml_writer->startComment();
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                    $xml_writer->endComment();
                    break;
                }else{
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;
            }
        }
        // Include custom xml
        $xml_writer->startElement("X-PRE-PROCESS");
        $xml_writer->writeAttribute("cmd", "optional_include");
        $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/sip_profiles/' . $sip_profile.'/'.$profile_name . '.gateway.xml');
        $xml_writer->EndElement();

        $xml_writer->fullEndElement(); //end gateway tag
        $xml_writer->fullEndElement(); //end include tag
        $xml_writer->endDocument();
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = "sip_profiles/" . $sip_profile . "/" . $profile_name . ".xml";
        $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));

        // Generates user dialplan
        $cfg = $sip_trunk->get_data_values(false);
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0','UTF-8');

        $xml_writer->startElement("include");
        $xml_writer->writeComment('Ingress trunk context');
        $xml_writer->startElement("context");
        $xml_writer->writeAttribute("name", "trunk.".$sip_trunk->name().'.ingress');

        $xml_writer->startElement("extension");
        $xml_writer->writeAttribute("name", "transfer-to-profile");
        // Set routing context
        $xml_writer->startElement('condition');
            $xml_writer->writeAttribute('field', 'destination_number');
            $xml_writer->writeAttribute('expression', '(.*)');
            // Override profile routing ?
            if('__none__' != $cfg['call-routing']){
                $xml_writer->writeComment('Set Routing context');
                $xml_writer->StartElement('action');
                $xml_writer->writeAttribute('application', 'set');
                $xml_writer->writeAttribute('data', 'sangoma_routing_context=routing.' . $cfg['call-routing']);
                $xml_writer->endElement();
            }
            // Override profile ingress ?
            if ('__none__' != $cfg['ingress']){
                $xml_writer->writeComment('Set ingress HMR context');
                $xml_writer->StartElement('action');
                $xml_writer->writeAttribute('application', 'set');
                $xml_writer->writeAttribute('data', 'sangoma_ingress_hmr_context=hmr.' . $cfg['ingress']);
                $xml_writer->endElement();
            }

            $xml_writer->writeComment('Transfer to profile ingress');
            $xml_writer->startElement("action");
            $xml_writer->writeAttribute("application", "transfer");
            $xml_writer->writeAttribute("data", '$1 XML profile.${sofia_profile_name}.ingress');
            $xml_writer->endElement(); //end action tag

        $xml_writer->endElement(); //end condition tag
        $xml_writer->endElement(); //end extension tag
        $xml_writer->endElement(); //end context tag
        
        // Egress trunk
        $xml_writer->writeComment('Egress trunk context');
        $xml_writer->startElement("context");
        $xml_writer->writeAttribute("name", "trunk.".$sip_trunk->name().".egress");
            $xml_writer->startElement("extension");
            $xml_writer->writeAttribute("name", "sangoma_trunk_".$sip_trunk->name()."_egress");
            // <!-- Set routing stage variable so we can figure out where to go next in advanced hmr -->                                      
            $xml_writer->startElement('condition');
                $xml_writer->writeComment('Set routing bridge');
                $xml_writer->StartElement('action');
                $xml_writer->writeAttribute('application', 'set');
                $xml_writer->writeAttribute('data', 'sangoma_routing_bridge=sofia/gateway/' . $sip_trunk->name() );
                $xml_writer->endElement();
            $xml_writer->endElement();//condition

            // Set hmr context
            $xml_writer->startElement('condition');
                $xml_writer->writeComment('Set HMR context');
                $xml_writer->StartElement('action');
                $xml_writer->writeAttribute('application', 'set');
                $xml_writer->writeAttribute('inline', 'true');
                if ('__none__' == $cfg['egress']){
                    // Check for profile egress hmr
                    if('__none__' == $sip_profile_cfg['egress']){
                        $xml_writer->writeAttribute('data', 'sangoma_egress_hmr_context=');
                    }else{
                        $xml_writer->writeAttribute('data', 'sangoma_egress_hmr_context=hmr.' . $sip_profile_cfg['egress']);
                    }
                }else{
                    $xml_writer->writeAttribute('data', 'sangoma_egress_hmr_context=hmr.' . $cfg['egress']);
                }
                $xml_writer->endElement();
            $xml_writer->endElement(); //condition
            // Set post hmr context
            $xml_writer->startElement('condition');
            $xml_writer->writeAttribute('field', '${sangoma_egress_hmr_context}');
            $xml_writer->writeAttribute('expression', '^$');
            $xml_writer->writeAttribute('break', 'never');
                $xml_writer->writeComment('Transfer to HMR context');
                $xml_writer->StartElement('anti-action');
                $xml_writer->writeAttribute('application', 'set');
                $xml_writer->writeAttribute('data', 'sangoma_post_hmr_context=trunk.' . $sip_trunk->name() . '.outbound');
                $xml_writer->endElement();
                $xml_writer->StartElement('anti-action');
                $xml_writer->writeAttribute('application', 'transfer');
                $xml_writer->writeAttribute('data', '${sangoma_routing_destination} XML ${sangoma_egress_hmr_context}');
                $xml_writer->endElement();
            $xml_writer->endElement(); //condition

            // Bridge
            $xml_writer->startElement('condition');
                $xml_writer->writeComment('Bridge');
                $xml_writer->StartElement('action');
                $xml_writer->writeAttribute('application', 'bridge');
                $xml_writer->writeAttribute('data', '${sangoma_routing_bridge}/${sangoma_routing_destination}');
                $xml_writer->endElement();
            $xml_writer->endElement();//condition
            $xml_writer->endElement();//extension
        $xml_writer->endElement();//context                

        //<!-- Egress outbound context -->
        $xml_writer->writeComment('Outbound trunk context ');
        $xml_writer->startElement('context');
        $xml_writer->writeAttribute('name', 'trunk.' . $sip_trunk->name() . '.outbound');
            $xml_writer->startElement('extension');
            $xml_writer->writeAttribute('name', 'sangoma_trunk_' . $sip_trunk->name() . '_outbound');//need to get the name
                    $xml_writer->startElement('condition');
                        $xml_writer->writeComment('Bridge');
                        $xml_writer->StartElement('action');
                        $xml_writer->writeAttribute('application', 'bridge');
                        $xml_writer->writeAttribute('data', '${sangoma_routing_bridge}/${sangoma_routing_destination}');
                        $xml_writer->endElement();
                    $xml_writer->endElement();//condition
            $xml_writer->endElement();//extension
        $xml_writer->endElement();//context
        // Close all opened elements
        while ($xml_writer->endElement() !== false) {
            continue;
        }

        $xml_writer->endDocument();
        // Create xml string
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = 'dialplan/core/sip_trunks/trunk.'.$sip_trunk->name().'.xml';
        // Instruct configuration manager to create the file
        $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));

        return true;
    }
    /**
     * start a  SIP Profile
     * @param string $profile_name (SIP profile name to be start)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_start_profile($profile_name, $params = null, &$out_message = null)
    {
        //need to check $this->_sip_profiles loaded or not 
        if (isset($this->_sip_profiles[$profile_name])) {
            $this->eslapi_execute('sofia profile '.$profile_name.' start', '', $out_message);
            if (preg_match("/\bsuccessfully\b/i", implode(' ', $out_message))){
                //Reload XML [Success]
                //NSC_SIP_Profile1 started successfully
                return true;
            }else{
                //Invalid Profile [NSC_SIP_Profile2]
                return false;
            }
        }else{
            $out_message[0] = 'No sip profile found.';
            return false;
        }
    }
    /**
     * stop a  SIP Profile
     * @param string $profile_name (SIP profile name to be stop)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_stop_profile($profile_name, $params = null, &$out_message = null)
    {
        // Set the basic data
        if (isset($this->_sip_profiles[$profile_name])) {
            $this->eslapi_execute('sofia profile '.$profile_name.' stop', '', $out_message);
            if (preg_match("/\bstopping\b/i", implode(' ', $out_message))){
                //Reload XML [Success]
                //stopping: NSC_SIP_Profile1
                return true;
            }else{
                //Invalid Profile [NSC_SIP_Profile2]
                return false;
            }
        }else{
            $out_message['message'] = 'No sip profile found.';
            return false;
        }
    }
    /**
     * flush  SIP Profile registrations
     * @param string $profile_name (SIP profile name to be stop)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_flush_registration_profile($profile_name, $params = null, &$out_message=null)
    {
        $call_id = safe_set_default($params,'call_id','');//reg call id
        $this->profiles();
        if (isset($this->_sip_profiles[$profile_name])) {
            $this->eslapi_execute('sofia profile ' . $profile_name . ' flush_inbound_reg '.$call_id, '', $out_message);
            //+OK flushing all registrations matching specified call_id
            return true;
        }else{
            $out_message['message'] = 'No sip profile found.';
            return false;
        }
    }
    /**
     * Returns the SIP Channel Status
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_channels_profile($profile_name, $params = null, &$out_message=null)
    {
        $uuid = safe_set_default($params,'uuid','');//channel uuid
        $limit = safe_set_default($params,'limit',100);//max return record number
        $offset = safe_set_default($params,'offset',0);//start from record number
        $select = safe_set_default($params,'select','');//set select 

        $db = $this->_db;
        $db->connect();
        $conditions = "";
        if($uuid!=''){
            $conditions = "uuid = '$uuid'";
        }
        $order_by = 'created';
        $result = $db->select('channels', $conditions, $limit, $offset, $order_by, '', $select, '');
        $db->disconnect();
        if(count($result)>0){
            if($select != ''){
                return $result;
            }else{
                $output = null;
                $i=0;
                foreach($result as $item){
                    $output[$i] = (array) $item;
                    $i++;
                }
            }
            return $output;
        }else{
            return false;
        }
    }
    public function api_retrieve_channel($name, $data = null, &$output = null){
        
        if($name){
            //return one record with uniqe uuid
            if(!isset($this->_channel_data_cache[$name])){
                $uuid = safe_set_default($data, 'uuid', urldecode($name));
                $channel_output = $this->api_channels_profile($this->name(), array('uuid' => $uuid), $output);
                if($channel_output === false){// no record found
                    $output['message'] = 'Not Found';
                    return false;
                }else{
                    //save to channel_data_cache, key using  urlencode(uuid)
                    $this->_channel_data_cache[urlencode($name)] = $channel_output[0];
                }
            }
            return $this->_channel_data_cache[$name];
        }else{
            $cache_key = md5(implode('', $data));
            if(!isset($this->_channel_list_cache[$cache_key])){
                $params_total = array( 'select' => 'count(*) as total', 'limit' => 1, 'offset' => 0);
                $rs_total = $this->api_channels_profile($this->name(), $params_total, $output);
                $this->_channel_list_cache[$cache_key]['total'] = $rs_total[0]->total;
                $limit = safe_set_default($data,'pagination/limit',100);//max return record number
                $offset = safe_set_default($data,'pagination/offset',0);//start from record number
                $params = array( 'limit' => $limit, 'offset' => $offset );
                $this->_channel_list_cache[$cache_key]['param'] = $params;
                $this->_channel_list_cache[$cache_key]['data'] = array();
                if($rs_total[0]->total > 0){
                    $channel_output = $this->api_channels_profile($this->name(), $params, $output);
                    foreach($channel_output as $channel_item){
                        $this->_channel_data_cache[urlencode($channel_item['uuid'])] = $channel_item;
                        $this->_channel_list_cache[$cache_key]['data'][urlencode($channel_item['uuid'])] =  $channel_item;
                    }
                }
            }
            $output['total'] = $this->_channel_list_cache[$cache_key]['total'];
            $output['limit'] = $this->_channel_list_cache[$cache_key]['param']['limit'];
            $output['offset'] = $this->_channel_list_cache[$cache_key]['param']['offset'];
            $output['data'] = $this->_channel_list_cache[$cache_key]['data'];
            return $output;
        }
    }
    /**
     * Returns the SIP Profile Registrations
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_registration_profile($profile_name, $params = null, &$out_message = null)
    {
        $call_id = safe_set_default($params,'call_id','');//call_id
        $limit = safe_set_default($params,'limit',100);//max return record number
        $offset = safe_set_default($params,'offset',0);//start from record number
        $select =    safe_set_default($params,'select','');//set select 
        
        $db = $this->_db;
        $db->connect();
        $conditions = "";
        if($profile_name!=''){
            $conditions = "profile_name = '{$profile_name}'";
            if($call_id != ''){
                $conditions = $conditions ." AND  call_id = '{$call_id}' ";
            }
        }
        $order_by = '';
        $result = $db->select('sip_registrations', $conditions, $limit, $offset, $order_by, '', $select, '');
        $db->disconnect();
        if(count($result) > 0){
            if($select != ''){
                return $result;
            }else{
                $output = null;
                $i=0;
                foreach($result as $item){
                    $output[$i]['call-id']        = $item->call_id;
                    $output[$i]['user']           = $item->sip_user.'@'.$item->sip_host;
                    $output[$i]['contact']        = $item->contact;
                    $output[$i]['thru-contact']   = $item->thru_contact;
                    $output[$i]['agent']          = $item->user_agent;
                    //the $item->expires is different with the sofia return
                    $output[$i]['status']         = $item->status."($item->rpid) exp(1969-12-31 19:00:00) expsecs(-$item->expires)";
                    $output[$i]['host']           = $item->hostname;
                    $output[$i]['network-ip']     = $item->network_ip;
                    $output[$i]['network-port']   = $item->network_port;
                    $output[$i]['sip-auth-user']  = $item->sip_username;
                    $output[$i]['sip-auth-realm'] = $item->sip_realm;
                    $output[$i]['mwi-account']    = $item->mwi_user.'@'.$item->mwi_host;
                    $i++;
                }
            }
            return $output;
        }else{
            return false;
        }
    }
    /**
     * Returns the SIP Profile status
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_status_profile($profile_name = '', $params = null, &$out_message = null)
    {
        if (Safe_service_class::STATUS_RUNNING == $this->fs_app()->status()){
            $output = '';
            if($profile_name==''){
                $this->eslapi_execute('sofia xmlstatus', '', $output);
            }else{
                $this->eslapi_execute('sofia xmlstatus profile ' . $profile_name, '', $output);
            }
            try{
                $xml_output = Safe_xml_to_array($output);
            }catch(exception $e){
                $out_message = $output;
                $xml_output = array();
            }
            return $xml_output;
        }
        return array();
    }
    /**
     * Returns the SIP Trunk Status
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param string &$out_message (Array output msg)
     */
    public function api_gateway_status_profile($profile_name = '', $params = null, &$out_message = null)
    {
        if (Safe_service_class::STATUS_RUNNING == $this->fs_app()->status()){
            $output = '';
            $this->eslapi_execute('sofia xmlstatus gateway ' . $profile_name , '', $output);
            try{
                $xml_output = Safe_xml_to_array($output);
            }catch(exception $e){
                $out_message = $output;
                $xml_output = array();
            }
            return $xml_output;
        }
        return array();
    }
    public function api_status_trunk($name = '', $params = null, &$output = null)
    {
        if ($this->api_retrieve_trunk($name)) {
            return $this->api_gateway_status_profile($name, $params, $output);
        }else{
            $output['message'] = 'No Trunk found.';
            return false;
        }
    }
    

    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_profile($name, $data=null, &$output = null) {
        $profile = new Fs_sip_profile_class($this->node(), $this->_obj_base_path .'/profile' , $name);
        $profile->configure();
        if($data) {
            $profile->set_data_values($data);
        }
        return $profile;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_profile($name, $data=null, &$output = null) {
        $profiles = $this->profiles();
        return $profiles[$name];
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_profile($name, $data=null, &$output = null) {
        $profile = $this->api_retrieve_profile($name);
        if($profile) {
            if ($profile->validate($data,$output)) {
                if (true == $profile->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_profile($name, $data=null, &$output = null) {
        $profile = $this->api_retrieve_profile($name);
        if($profile) {
            if($profile->can_dispose($output)) {
                foreach ($profile->has_aggregate_objects(false, true) as $_type => $_definition){
                    if($_definition['configurable']){
                        foreach ($profile->get_aggregate_objects($_type) as $sub_obj_name => $sub_object) {
                            $sub_object->dispose();
                        }    
                    }
                }
                return $profile->dispose();
            }
        }
        return false;
    }
    
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_trunk($name, $data=null, &$output = null) {
        $trunk = new Fs_gateway_class($this->node(), $this->_obj_base_path . '/trunk', $name);
        $trunk->configure();
        if($data) {
            $trunk->set_data_values($data);
        }
        return $profile;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_trunk($name, $data=null, &$output = null) {
        $trunk = $this->trunks();
        return $trunk[$name];
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_trunk($name, $data = null, &$output = null){
        $trunk = $this->api_retrieve_trunk($name);
        if($trunk) {
            if ($trunk->validate($data,$output)) {
                if (true == $trunk->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_trunk($name, $data=null, &$output = null) {
        $trunk = $this->api_retrieve_trunk($name);
        if($trunk) {
            if($trunk->can_dispose($output)) {
                return $trunk->dispose();
            }
        }
        return false;
    }
    public function can_create_aggregate_object($obj_type=null, $name, &$output = null){
        if($obj_type == 'trunk'){
            if(count($this->get_aggregate_objects('profile')) <= 0){
                $output[] = 'No SIP profile defined.';
                return false;
            }
        }
        if($obj_type == 'profile'){
            $err_msg = array();
            $media_module = $this->node()->software()->application()->find_module_by_name('media');
            if(count($media_module->get_aggregate_objects('profile')) <= 0){
                $err_msg[] = 'No media profile defined.';
            }
            $routing_module = $this->node()->software()->application()->find_module_by_name('routing');
            if((count($routing_module->get_aggregate_objects('basic')) + count($routing_module->get_aggregate_objects('advanced'))) <= 0){
                $err_msg[] = 'No routing defined.';
            }
            if($err_msg){
                $output =  $err_msg;
                return false;
            }
        }
        return parent::can_create_aggregate_object($type, $name, $reason);
    }
    
    public function checklist(&$list)
    {
        if (!$this->profiles()){
            $list[] = array(
                'module' => 'sip',
                'obj_type' => 'profile',
                'description' => 'Not defined',
            );
        }
    }
    public function new_fields(&$config_manager, $type)
    {
        if($type == 'profile'){
              $sip_profiles = $this->profiles();
              $c = count($sip_profiles)+1;
              $config_manager->add_field('profile-name', 'Profile Name', 'string', 'NSC_SIP_Profile' . $c, 50);
              $config_manager->set_field_rules('profile-name', 'required|alpha_dash');
              if($sip_profiles) {
                  // Create enum list of existing profiles
                  $profile_list['__none__'] = "(None)";
                  foreach($sip_profiles as $k=>$v){
                      $profile_list[$k] = $v->name();
                  }
                  $config_manager->add_enum_field('src', 'Copy From', 'dropdown', '__none__', $profile_list);
              }
        }else{
            parent::new_fields($config_manager, $type);
        }
    } 
    
    public function check_object_usage($src_module_name, $obj_type, $obj_name, $sub_type = null, $sub_name = null) 
    {
        // check usages ?
        $usages = array();
        if(!in_array($src_module_name, array('routing', 'hmr','certificate')))
            return $usages;

        // Check for profiles
        $profiles = $this->profiles();

        foreach($profiles as $profile){
            $_routing = array($profile->get_data_value('call-routing', false));
            $_hmr = array($profile->get_data_value('ingress', false), $profile->get_data_value('egress', false));
            $_cert = array($profile->get_data_value('TLS/certificate', false));
            
            if( ('routing' == $src_module_name && in_array($obj_name, $_routing))
                || ('hmr' == $src_module_name && in_array($obj_name, $_hmr))
                || ('certificate' == $src_module_name && in_array($obj_name, $_cert) && $obj_type == 'server') ){
                    $tmp = array (
                            'name' => $this->name(),
                            'obj_type' => 'profile',
                            'obj_name' => $profile->name()
                    );
                if(!in_array($tmp, $usages)){
                    $usages[] = $tmp;
                }
            }
        }
        $profiles = $this->trunks();
        foreach($profiles as $profile){
            $_routing = array($profile->get_data_value('call-routing', false));
            $_hmr = array($profile->get_data_value('ingress', false), $profile->get_data_value('egress', false));

            if( ('routing' == $src_module_name && in_array($obj_name, $_routing)
                || ('hmr' == $src_module_name && in_array($obj_name, $_hmr)))){
                    $tmp = array (
                            'name' => $this->name(),
                            'obj_type' => 'trunk',
                            'obj_name' => $profile->name()
                    );
                if(!in_array($tmp, $usages)){
                    $usages[] = $tmp;
                }
            }
        }
        return $usages;
    }
    public function check_object_modified($src_module_name, $obj_type, $obj_name, $sub_type = null, $sub_name = null)
    {
        // check usages ?
        $usages = array();
        if(!in_array($src_module_name, array('media')))
            return $usages;

        // Check for profiles
        $profiles = $this->profiles();

        foreach($profiles as $profile){
            $_media = array($profile->get_data_value('inbound-media-profile', false), $profile->get_data_value('outbound-media-profile', false));

            if('media' == $src_module_name && in_array($obj_name, $_media)){
                $profile->mark_modified();
                $tmp = array (
                        'name' => $this->name(),
                        'obj_type' => 'profile',
                        'obj_name' => $profile->name()
                );
                if(!in_array($tmp, $usages)){
                    $usages[] = $tmp;
                }
            }
        }
        $profiles = $this->trunks();
        foreach($profiles as $profile){
            $_media = array($profile->get_data_value('inbound-media-profile', false), $profile->get_data_value('outbound-media-profile', false));

            if('media' == $src_module_name && in_array($obj_name, $_media)){
                $profile->mark_modified();
                $tmp = array (
                        'name' => $this->name(),
                        'obj_type' => 'trunk',
                        'obj_name' => $profile->name()
                );
                if(!in_array($tmp, $usages)){
                    $usages[] = $tmp;
                }
            }
        }
        return $usages;
    }
    public function sip_response_enum($filter=null){
        $sip_response = array(
                '100' => '100 Trying',
                '180' => '180 Ringing',
                '181' => '181 Call is Being Forwarded',
                '182' => '182 Queued',
                '183' => '183 Session in Progress',
                '199' => '199 Early Dialog Terminated',
                '200' => '200 OK',
                '202' => '202 Accepted',
                '204' => '204 No Notification',
                '300' => '300 Multiple Choices',
                '301' => '301 Moved Permanently',
                '302' => '302 Moved Temporarily',
                '305' => '305 Use Proxy',
                '380' => '380 Alternative Service',
                '400' => '400 Bad Request',
                '401' => '401 Unauthorized',
                '402' => '402 Payment Required',
                '403' => '403 Forbidden',
                '404' => '404 Not Found',
                '405' => '405 Method Not Allowed',
                '406' => '406 Not Acceptable',
                '407' => '407 Proxy Authentication Required',
                '408' => '408 Request Timeout',
                '409' => '409 Conflict',
                '410' => '410 Gone',
                '411' => '411 Length Required',
                '412' => '412 Conditional Request Failed',
                '413' => '413 Request Entity Too Large',
                '414' => '414 Request-URI Too Long',
                '415' => '415 Unsupported Media Type',
                '416' => '416 Unsupported URI Scheme',
                '417' => '417 Unknown Resource-Priority',
                '420' => '420 Bad Extension',
                '421' => '421 Extension Required',
                '422' => '422 Session Interval Too Small',
                '423' => '423 Interval Too Brief',
                '424' => '424 Bad Location Information',
                '428' => '428 Use Identity Header',
                '429' => '429 Provide Referrer Identity',
                '430' => '430 Flow Failed',
                '433' => '433 Anonymity Disallowed',
                '436' => '436 Bad Identity-Info',
                '437' => '437 Unsupported Certificate',
                '438' => '438 Invalid Identity Header',
                '439' => '439 First Hop Lacks Outbound Support',
                '470' => '470 Consent Needed',
                '480' => '480 Temporarily Unavailable',
                '481' => '481 Call/Transaction Does Not Exist',
                '482' => '482 Loop Detected',
                '483' => '483 Too Many Hops',
                '484' => '484 Address Incomplete',
                '485' => '485 Ambiguous',
                '486' => '486 Busy Here',
                '487' => '487 Request Terminated',
                '488' => '488 Not Acceptable Here',
                '489' => '489 Bad Event',
                '491' => '491 Request Pending',
                '493' => '493 Undecipherable',
                '494' => '494 Security Agreement Required',
                '500' => '500 Server Internal Error',
                '501' => '501 Not Implemented',
                '502' => '502 Bad Gateway',
                '503' => '503 Service Unavailable',
                '504' => '504 Server Time-out',
                '505' => '505 Version Not Supported',
                '513' => '513 Message Too Large',
                '580' => '580 Precondition Failure',
                '600' => '600 Busy Everywhere',
                '603' => '603 Decline',
                '604' => '604 Does Not Exist Anywhere',
                '606' => '606 Not Acceptable'
        );
        return $sip_response;
    }

    public function sip_hangup_enum()
    {
        if(!$this->_hangup_causes){
            $this->_hangup_causes = array(
                'NORMAL_CLEARING' => 'NORMAL_CLEARING', // Set 1st to be default
                'UNSPECIFIED' => 'UNSPECIFIED',
                'UNALLOCATED_NUMBER' => 'UNALLOCATED_NUMBER',
                'NO_ROUTE_TRANSIT_NET' => 'NO_ROUTE_TRANSIT_NET',
                'NO_ROUTE_DESTINATION' => 'NO_ROUTE_DESTINATION',
                'CHANNEL_UNACCEPTABLE' => 'CHANNEL_UNACCEPTABLE',
                'CALL_AWARDED_DELIVERED' => 'CALL_AWARDED_DELIVERED',
                'USER_BUSY' => 'USER_BUSY',
                'NO_USER_RESPONSE' => 'NO_USER_RESPONSE',
                'NO_ANSWER' => 'NO_ANSWER',
                'SUBSCRIBER_ABSENT' => 'SUBSCRIBER_ABSENT',
                'CALL_REJECTED' => 'CALL_REJECTED',
                'NUMBER_CHANGED' => 'NUMBER_CHANGED',
                'REDIRECTION_TO_NEW_DESTINATION' => 'REDIRECTION_TO_NEW_DESTINATION',
                'EXCHANGE_ROUTING_ERROR' => 'EXCHANGE_ROUTING_ERROR',
                'DESTINATION_OUT_OF_ORDER' => 'DESTINATION_OUT_OF_ORDER',
                'INVALID_NUMBER_FORMAT' => 'INVALID_NUMBER_FORMAT',
                'FACILITY_REJECTED' => 'FACILITY_REJECTED',
                'RESPONSE_TO_STATUS_ENQUIRY' => 'RESPONSE_TO_STATUS_ENQUIRY',
                'NORMAL_UNSPECIFIED' => 'NORMAL_UNSPECIFIED',
                'NORMAL_CIRCUIT_CONGESTION' => 'NORMAL_CIRCUIT_CONGESTION',
                'NETWORK_OUT_OF_ORDER' => 'NETWORK_OUT_OF_ORDER',
                'NORMAL_TEMPORARY_FAILURE' => 'NORMAL_TEMPORARY_FAILURE',
                'SWITCH_CONGESTION' => 'SWITCH_CONGESTION',
                'ACCESS_INFO_DISCARDED' => 'ACCESS_INFO_DISCARDED',
                'REQUESTED_CHAN_UNAVAIL' => 'REQUESTED_CHAN_UNAVAIL',
                'PRE_EMPTED' => 'PRE_EMPTED',
                'FACILITY_NOT_SUBSCRIBED' => 'FACILITY_NOT_SUBSCRIBED',
                'OUTGOING_CALL_BARRED' => 'OUTGOING_CALL_BARRED',
                'INCOMING_CALL_BARRED' => 'INCOMING_CALL_BARRED',
                'BEARERCAPABILITY_NOTAUTH' => 'BEARERCAPABILITY_NOTAUTH',
                'BEARERCAPABILITY_NOTAVAIL' => 'BEARERCAPABILITY_NOTAVAIL',
                'SERVICE_UNAVAILABLE' => 'SERVICE_UNAVAILABLE',
                'BEARERCAPABILITY_NOTIMPL' => 'BEARERCAPABILITY_NOTIMPL',
                'CHAN_NOT_IMPLEMENTED' => 'CHAN_NOT_IMPLEMENTED',
                'FACILITY_NOT_IMPLEMENTED' => 'FACILITY_NOT_IMPLEMENTED',
                'SERVICE_NOT_IMPLEMENTED' => 'SERVICE_NOT_IMPLEMENTED',
                'INVALID_CALL_REFERENCE' => 'INVALID_CALL_REFERENCE',
                'INCOMPATIBLE_DESTINATION' => 'INCOMPATIBLE_DESTINATION',
                'INVALID_MSG_UNSPECIFIED' => 'INVALID_MSG_UNSPECIFIED',
                'MANDATORY_IE_MISSING' => 'MANDATORY_IE_MISSING',
                'MESSAGE_TYPE_NONEXIST' => 'MESSAGE_TYPE_NONEXIST',
                'WRONG_MESSAGE' => 'WRONG_MESSAGE',
                'IE_NONEXIST' => 'IE_NONEXIST',
                'INVALID_IE_CONTENTS' => 'INVALID_IE_CONTENTS',
                'WRONG_CALL_STATE' => 'WRONG_CALL_STATE',
                'RECOVERY_ON_TIMER_EXPIRE' => 'RECOVERY_ON_TIMER_EXPIRE',
                'MANDATORY_IE_LENGTH_ERROR' => 'MANDATORY_IE_LENGTH_ERROR',
                'PROTOCOL_ERROR' => 'PROTOCOL_ERROR',
                'INTERWORKING' => 'INTERWORKING',
                'ORIGINATOR_CANCEL' => 'ORIGINATOR_CANCEL',
                'CRASH' => 'CRASH',
                'SYSTEM_SHUTDOWN' => 'SYSTEM_SHUTDOWN',
                'LOSE_RACE' => 'LOSE_RACE',
                'MANAGER_REQUEST' => 'MANAGER_REQUEST',
                'BLIND_TRANSFER' => 'BLIND_TRANSFER',
                'ATTENDED_TRANSFER' => 'ATTENDED_TRANSFER',
                'ALLOTTED_TIMEOUT' => 'ALLOTTED_TIMEOUT',
                'USER_CHALLENGE' => 'USER_CHALLENGE',
                'MEDIA_TIMEOUT' => 'MEDIA_TIMEOUT',
                'PICKED_OFF' => 'PICKED_OFF',
                'USER_NOT_REGISTERED' => 'USER_NOT_REGISTERED',
                'PROGRESS_TIMEOUT' => 'PROGRESS_TIMEOUT',
                'GATEWAY_DOWN' => 'GATEWAY_DOWN',
            );
        }
        return $this->_hangup_causes;
    }
    public function sip_direction_enum($resource = null){
        if($resource != null ){
            $sip_fields['sip_to_'.$resource] = 'SIP To:'.ucfirst($resource);
            $sip_fields['sip_from_'.$resource] = 'SIP From:'.ucfirst($resource);
        }
        return $sip_fields;
    }
    public function sip_log_level(){
        $log_level = array(
                'DEBUG' => 'Debug',
                'INFO' => 'Info',
                'NOTICE' => 'Notice',
                'WARNING' => 'Warning',
                'ERR' => 'Error',
                'CRIT' => 'Critical',
                'ALERT' => 'Alert'
        );
        return $log_level;
    }
    /**
        * @brief 
        *
        * @param[in out] $filter  (optional) enum type (string) filter array
        * @param[in out] $include_filter (optional) boolean indicating if filter 
        * is list of enum to include (true) or to exclude (false)
        *
        * @return 
     */
    public function sip_fields_enums($filter=array(), $include_filter=true)
    {
        if (!$this->_sip_fields){
             $domain_module = $this->node()->software()->application()->find_module_by_name('directory');
             
             $enum_resource = array();
             $enum_resource['domain']     = array('type' => 'domain', 'module' => $domain_module);
             $enum_resource['from_trunk'] = array('type' => 'trunk', 'module' => $this);
             
             foreach($enum_resource as $rs_key => $rs_value){
                 $enum_resource[$rs_key]['data'] = $rs_value['module']->object_enum($rs_value['type'], ADD_NONE_IF_EMPTY);
             }
            
             $this->_sip_fields = array(
                 'standard' => array(
                        'label' => 'Standard Information',
                        'type'  => 'enum',
                        'enum'  => array(
                                'destination_number' => 'Destination Address',
                                'caller_id_name' => 'Caller ID Name',
                                'caller_id_number' => 'Caller ID Number',
                                'callee_id_name' => 'Callee ID Name',
                                'callee_id_number' => 'Callee ID Number',
                            )
                        ),
                 'header' => array(
                        'label' => 'SIP Header Information',
                        'type'  => 'enum',
                        'enum'  => array(
                                '${sip_req_uri}' => 'R-URI',
                                '${sip_req_user}' => 'R-URI User',
                                '${sip_req_host}' => 'R-URI Host',
                                '${sip_req_port}' => 'R-URI Host Network Port',
                                '${sip_req_params}' => 'R-URI Parameters',
                                
                                '${sip_full_to}' => 'To',//add
                                
                                '${sip_to_uri}' => 'To: URI',
                                '${sip_to_user}' => 'To: User',
                                '${sip_to_host}' => 'To: Host',
                                '${sip_to_port}' => 'To: Host Network Port',
                                '${sip_to_params}' => 'To: Parameters',
                                
                                '${sip_to_tag}' => 'To: tag',//add
                                '${sip_full_from}' => 'From',//add
                                '${sip_from_display}' => 'From: Display',//add
                                
                                '${sip_from_uri}' => 'From: URI',
                                '${sip_from_user}' => 'From: User',
                                '${sip_from_host}' => 'From: Host',
                                '${sip_from_port}' => 'From: Host Network Port',
                                '${sip_from_params}' => 'From: Parameters',
                                
                                '${sip_from_tag}' => 'From: tag',//add
                                
                                '${sip_contact_uri}' => 'Contact: URI',
                                '${sip_contact_user}' => 'Contact: User',
                                '${sip_contact_host}' => 'Contact: Host',
                                '${sip_contact_port}' => 'Contact: Host Network Port',
                                '${sip_contact_params}' => 'Contact: Parameters',
                                '${sip_referred_by_uri}' => 'Referred-By: URI',
                                '${sip_referred_by_user}' => 'Referred-By: User',
                                '${sip_referred_by_host}' => 'Referred-By: Host',
                                '${sip_referred_by_port}' => 'Referred-By: Host Network Port',
                                '${sip_referred_by_params}' => 'Referred-By: Parameters',
                                
                                '${sip_full_via}' => 'Via',//add
                                
                                '${sip_via_protocol}' => 'Via Protocol',
                                '${sip_via_host}' => 'Via Host',
                                '${sip_via_port}' => 'Via Port',
                                '${sip_via_rport}' => 'Via Remote Port',
                                
                                '${sip_full_route}' => 'Route',//add
                                
                                '${sip_call_id}' => 'Call-ID',
                                
                                '${sip_cseq}' => 'CSeq',//add
                                
                                '${sip_subject}' => 'Subject',
                                
                                '${sip_user_agent}' => 'User-Agent',//update
                                '${alert_info}' => 'Alert-Info',//update
                                '${sip_call_info}' => 'Call-Info',//update
                                '${sip_history_info}' => 'History-Info',//update
                                '${max_forwards}' => 'Max-Forwards',//update
                            )
                        ),
                 'call' => array(
                        'label' => 'SIP Call Information',
                        'type'  => 'enum',
                        'enum'  => array(
                                '${sip_authorized}' => 'Authorized',
                                '${sip_looped_call}' => 'Looped Call',
                                '${sip_cid_type}' => 'Caller ID Type',
                                '${sip_trunk}' => 'Trunk Name',
                                '${sip_local_network_addr}' => 'SIP Local Network Address',
                                '${sip_network_ip}' => 'Remote Network IP',
                                '${sip_network_port}' => 'Remote Network Port',
                            )
                        ),
                 'sdp' => array(
                        'label' => 'SDP Information',
                        'type'  => 'enum',
                        'enum'  => array(
                                '${ep_codec_string}' => 'SDP Codec',
                                '${media_audio_mode}' => 'Media Mode',
                                '${remote_media_ip}' => 'SDP RTP IP',
                                '${remote_media_port}' => 'SDP RTP Port',
                                '${sip_remote_audio_rtcp_port}' => 'RTCP Port',
                                '${sip_has_crypto}' => 'SDP Crypto Suite',
                                '${sip_secure_media}' => 'Secure Media',
                                '${dtmf_type}' => 'DTMF type',
                                '${switch_r_sdp}' => 'Remote SDP',
                            )
                        ),
                 'invite_header' => array(
                        'label' => 'SIP INVITE Header',
                        'type'  => 'enum',
                        'enum'  => array(
                                '${sip_invite_req_uri}' => 'Request URI',
                                '${sip_invite_params}' => 'R-URI Parameters',
                                '${sip_invite_full_via}' => 'Via',
                                '${sip_invite_route_uri}' => 'Route',
                                '${sip_invite_full_from}' => 'From',
                                '${sip_from_display}' => 'From: Display',
                                '${sip_invite_from_uri}' => 'From: URI',
                                '${sip_invite_from_params}' => 'From: Parameters',
                                '${sip_invite_full_to}' => 'To',
                                '${sip_invite_to_uri}' => 'To: URI',
                                '${sip_invite_to_params}' => 'To: Parameters',
                                '${sip_contact_user}' => 'Contact: User',
                                '${sip_invite_contact_params}' => 'Contact: Parameters',
                                '${alert_info}' => 'Alert-Info',
                                '${sip_invite_call_id}' => 'Call-ID',
                                '${sip_invite_cseq}' => 'CSeq',
                                '${max_forwards}' => 'Max-Forwards',
                            )
                        ),
                 'custom_header' => array(
                        'label' => 'SIP Custom Header',
                        'pattern' => '${sip_h_xxx}',
                        ),
                 'custom_from' => array(
                        'label' => 'SIP From: Parameter',
                        'pattern' => '${sip_from_xxx}',
                    ),
                 'custom_to' => array(
                        'label' => 'SIP To: Parameter',
                        'pattern' => '${sip_to_xxx}',
                    ),
                 'custom_referred_by' => array(
                        'label' => 'SIP Referred-By:Parameter',
                        'pattern' => '${sip_referred_by_xxx}',
                    ),
                 'variable' => array(
                        'label' => 'Variable',
                        'pattern' => '${xxx}',
                    ),
                 'domain' => array(
                         'label' => 'Known Domain',
                         'type'  => 'enum',
                         'enum'  => $this->sip_direction_enum('host'),
                         'name_label' => 'Direction',
                         'value_label' => 'Domain',
                         'value' => array(
                                  'type' =>'enum',
                                  'enum'  => $enum_resource['domain']['data'],
                                 )
                 ),
                 'user' => array(
                         'label' => 'Known User',
                         'type'  => 'enum',
                         'enum'  => $this->sip_direction_enum('user'),
                         'name_label' => 'Direction',
                         'value' => array(
                         )
                 ),
                 'auth' => array(
                         'label' => 'Authorized User',
                         'value' => array(
                         )
                 ),
                 'from_trunk' => array(
                         'label' => 'From SIP Trunk',
                         'type'  => 'enum',
                         'enum'  => $enum_resource['from_trunk']['data'],
                         'name_label' => 'Trunk',
                         'value' => array(
                         )
                 ),

                );
             // Sort the enum
             foreach($this->_sip_fields as &$enum){
                 if($enum['enum']){
                     asort($enum['enum']);
                 }
             }
         }
         if($filter){
             $_sip_fields = array();
             if($include_filter){
                 // include filter
                 foreach($filter as $f){
                     if($this->_sip_fields[$f]){
                         $_sip_fields[$f] = $this->_sip_fields[$f];
                     }
                 }
             }else{
                 foreach($this->_sip_fields as $k => $v){
                     if(!in_array($k, $filter)){
                         $_sip_fields[$k] = $v;
                     }
                 }
             }
             return $_sip_fields;
         }else{
             return $this->_sip_fields;
         }
    }

}
/* End of file fs_sip_profile_module_class.php */

