<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * SAFe object classes
*/
require_once ('application/helpers/safe_helper.php');
class Safe_object_serializer_class
{
    const OBJ_STATUS_NEW = 'N';
    const OBJ_STATUS_MODIFIED = 'M';
    const OBJ_STATUS_DELETED = 'D';
    const OBJ_STATUS_UP_TO_DATE = 'U';
    const OBJ_TYPE_CONFIGURABLE = 'configurable';
    const OBJ_TYPE_AUTOFILL = 'autofill';
    private static $_serializer = null;
    public static function get_serializer($force = FALSE)
    {
        if ($force) Safe_object_serializer_class::$_serializer = null;
        if (null == Safe_object_serializer_class::$_serializer) {
            Safe_object_serializer_class::$_serializer = new Safe_object_serializer_class();
        }
        return Safe_object_serializer_class::$_serializer;
    }
    /**
     * @brief Return next unique id
     *         
     * @return unique id (must be used before new record added)
     */
    public static function get_next_unique_id()
    {
        $ser = Safe_object_serializer_class::get_serializer();
        return $ser->next_insert_id();
    }
    public static function build_unique_name($prefix)
    {
        $prefix =  preg_replace('/[^0-9a-zA-Z\-_]/', '', $prefix);
        $id = Safe_object_serializer_class::get_next_unique_id();
        return $prefix.'_'.$id;   
    }
    /**
     * @var CI_DB_pdo_driver
     */
    private $db = null; // the codeigniter db object
    private $db_groupname = 'safe'; // the CodeIgniter groupname set in
    // application/config/database.php
    private $db_table = 'safe_object';
    // TODO: handling CodeIgniter database errors
    // using this $this->db->_error_message(); will return empty if no error.
    // and in ./config/database.php: $db['default']['db_debug'] = TRUE; Set it
    // to false.
    // or force it by $this->db->db_debug= flase;
    // TODO: put some method private and proteceted.
    public function __construct()
    {
        safe_enter_critical_section();
        try {
            $CI = & get_instance();
            // if the files for the db doesn't it will be created.
            $this->db = $CI->load->database($this->db_groupname, TRUE);
            if (!$this->db->table_exists($this->db_table)) if (!$this->create_db()) return false;
        }
        catch(Exception $e) {
            // TODO: catch CodeIgniter DB_Driver exception.
            // we will never go here since CodeIgniter DB_Driver will fail
            // with a PHP error on the user browser
            error_log("ERROR - Cannot open database ".$e->getMessage());
            show_error(__FUNCTION__ . ": can't open databases");
        }
        safe_leave_critical_section();
    }
    public function __destruct()
    {
        try {
            if ($this->db) $this->db->close();
        }
        catch(Exception $e) {
        }
    }
    /**
     *
     * @return boolean
     */
    private function create_db()
    {
        try {
            error_log("Safe_object_serializer: Trying to create safe_object DB");
            $sqlq = "
        CREATE TABLE '" . $this->db_table . "' (
          'id' INTEGER PRIMARY KEY AUTOINCREMENT,
          'path' TINYTEXT NOT NULL,
          'name' VARCHAR(128) NOT NULL,
          'type' VARCHAR(32) NOT NULL DEFAULT '" . Safe_object_serializer_class::OBJ_TYPE_CONFIGURABLE . "',
          'class' VARCHAR(32) NOT NULL,
          'modified' TINYINT NOT NULL DEFAULT 1,
          'status' CHAR(1) NOT NULL DEFAULT '" . Safe_object_serializer_class::OBJ_STATUS_NEW . "',
          'data' TEXT,
          'modified_date' DATE DEFAULT (datetime('now','localtime')),
          'modified_by'   VARCHAR(128),
          'data_ref' TEXT,
          CONSTRAINT unique_object UNIQUE ('type', 'path', 'name') ) ;
      ";
            if (!$this->db->query($sqlq)) throw new Exception("Cannot create DB!");
        }
        catch(Exception $e) {
            //safe_add_error(__FUNCTION__ . ":{$this->filename}: " . $e->getMessage() , true);
            error_log("ERROR - Cannot create_db() ".$e->getMessage());
            return false;
        }
        return true;
    }
    public function store_object($path, $name, $class, $object_data, $object_type = NULL, &$object_status = NULL, &$object_id = NULL)
    {
        $insert['path'] = $path;
        $insert['name'] = $name;
        $insert['class'] = $class;
        $insert['data'] = json_encode($object_data);
        $insert['modified_date'] = date('Y-m-d H:i:s');
        if (isset($object_type)) $insert['type'] = $object_type;
        // Object exists ?
        $old_data = array();
        $obj_status = '';
        if (TRUE == $this->retrieve_object($path, $name, $old_data, $obj_status, $object_id)) {
            // Yes, need to update
            if (json_encode($old_data) != $insert['data']) {
                // Data modified, update data and status to modified
                $insert['modified'] = 1;
                // Keep the NEW status
                if ($obj_status != Safe_object_serializer_class::OBJ_STATUS_NEW) {
                    $obj_status = Safe_object_serializer_class::OBJ_STATUS_MODIFIED;
                }
                error_log('store_object - ' . $obj_status . ' - ' . $path . '/' . $name);
            } else {
                // data not modified, check if need to revert Deleted
                if ($obj_status != Safe_object_serializer_class::OBJ_STATUS_DELETED) {
                    // Not a deleted object, skip update
                    error_log('Skipping store_object with same data for ' . $path . '/' . $name);
                    return TRUE;
                }
                // Deleted object, revert
                error_log('store_object with same data (reverting DELETED) for ' . $path . '/' . $name);
                $obj_status = Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE;
                $insert['modified'] = 0;
            }
            if(isset($_SESSION['user_login'])){
                $insert['modified_by'] = $_SESSION['user_login'];
            }
            $insert['status'] = $obj_status;
            $where = array(
                'path' => $path,
                'name' => $name,
            );
            $this->db->where($where);
            if (!$this->db->update($this->db_table, $insert)) {
                error_log("Fail to UPDATE serialize object " . $name());
                return FALSE;
            }
            if (isset($object_status)) $object_status = $obj_status;
        } else {
            // No, insert it
            if (!$this->db->insert($this->db_table, $insert)) {
                error_log("Fail to INSERT serialize object " . $name());
                return FALSE;
            }
            if (isset($object_id)) $object_id = $this->db->insert_id();
            if (isset($object_status)) $object_status = Safe_object_serializer_class::OBJ_STATUS_NEW;
        }
        return TRUE;
    }
    /**
     * @brief Mark an object as deleted
     *           
     * @param[in out] $path
     * @param[in out] $name
     *           
     * @return
     */
    public function dispose_object($path, $name)
    {
        // If NEW object then delete it
        // else mark as deleted (and modified)
        $dummy = array();
        $status = Safe_object_serializer_class::OBJ_STATUS_MODIFIED;
        $id = 0;
        // Retrieve object status and id
        if ($this->retrieve_object($path, $name, $dummy, $status, $id)) {
            if ($status == Safe_object_serializer_class::OBJ_STATUS_NEW) {
                // Not yet fetched, delete
                $this->db->where('id', $id);
                $this->db->delete($this->db_table);
            } else {
                // Existing and fetched object, mark as deleted
                $insert['modified'] = 1;
                $insert['status'] = Safe_object_serializer_class::OBJ_STATUS_DELETED;
                $this->db->where('id', $id);
                if (!$this->db->update($this->db_table, $insert)) {
                    error_log("Fail to UPDATE serialize object " . $name());
                    return FALSE;
                }
            }
            return TRUE;
        }
        return FALSE;
    }
    /**
     * @brief Mark an object as up to date
     *           
     * @param[in out] $path
     * @param[in out] $name
     *           
     * @return
     */
    public function mark_object_uptodate($path, $name)
    {
        $dummy = array();
        $status = Safe_object_serializer_class::OBJ_STATUS_MODIFIED;
        $id = 0;
        // Retrieve object status and id
        if ($this->retrieve_object($path, $name, $dummy, $status, $id)) {
            // Existing and fetched object, mark as deleted
            $insert['modified'] = 0;
            $insert['status'] = Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE;
            $this->db->where('id', $id);
            if (!$this->db->update($this->db_table, $insert)) {
                error_log("Fail to UPDATE serialize object " . $name());
                return FALSE;
            }
            return TRUE;
        }
        return FALSE;
    }
    /**
     * @brief Mark an object as modified
     *
     * @param[in out] $path
     * @param[in out] $name
     *
     * @return
     */
    public function mark_object_modified($path, $name)
    {
        $dummy = array();
        $status = Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE;
        $id = 0;
        // Retrieve object status and id
        if ($this->retrieve_object($path, $name, $dummy, $status, $id)) {
            // Existing and fetched object, mark as deleted
            $insert['modified'] = 1;
            $insert['status'] = Safe_object_serializer_class::OBJ_STATUS_MODIFIED;
            $this->db->where('id', $id);
            if (!$this->db->update($this->db_table, $insert)) {
                error_log("Fail to UPDATE serialize object " . $name());
                return FALSE;
            }
            return TRUE;
        }
        return FALSE;
    }
    /**
     * @brief
     *           
     * @param[in out] $path
     * @param[in out] $name
     * @param[in out] $object_data
     *           
     * @return
     */
    public function retrieve_object($path, $name, &$object_data = NULL, &$object_status = NULL, &$object_id = NULL, &$class = NULL, &$type = NULL)
    {
        $object_data = array();
        $q = $this->db->get_where($this->db_table, array(
            'path' => $path,
            'name' => $name
        ));
        if ($q->num_rows()) {
            $r = $q->row();
            $object_data = json_decode($r->data, true);
            if (isset($class)) $class = $r->class;
            if (isset($object_status)) $object_status = $r->status;
            if (isset($object_id)) $object_id = $r->id;
            if (isset($type)) $type = $r->type;
            return TRUE;
        }
        return FALSE;
    }
    /**
     * @brief
     *           
     * @param[in out] $path
     * @param[in out] $like
     *           
     * @return
     */
    public function find_objects($path, $like = null)
    {
        $obj_list = array();
        $this->db->where('path', $path);
        $this->db->where('status !=', Safe_object_serializer_class::OBJ_STATUS_DELETED);
        if (isset($like)) $this->db->like('name', $like, 'after');
        $query = $this->db->get($this->db_table);
        foreach ($query->result() as $row) {
            $obj_list[$row->name] = array(
                'class' => $row->class,
                'data' => json_decode($row->data, true)
            );
        }
        //echo $this->db->last_query() . "<br />";
        return $obj_list;
    }
    /**
     * @brief
     *           
     * @param[in out] $path
     * @param[in out] $like
     *           
     * @return
     */
    public function find_deleted_objects($path, $like = null)
    {
        $obj_list = array();
        $this->db->where('path', $path);
        $this->db->where('status', Safe_object_serializer_class::OBJ_STATUS_DELETED);
        if (isset($like)) $this->db->like('name', $like, 'after');
        $query = $this->db->get($this->db_table);
        foreach ($query->result() as $row) {
            $obj_list[$row->name] = array(
                'class' => $row->class,
                'data' => json_decode($row->data, true)
            );
        }
        return $obj_list;
    }
    /**
     * @brief
     *           
     * @param[in out] $path
     * @param[in out] $name
     *           
     * @return
     */
    public function delete_object($path, $name = null)
    {
        $this->db->where('path', $path);
        if (isset($name)) $this->db->where('name', $name);
        $this->db->delete($this->db_table);
    }
    /**
     * @brief
     *           
     * @param[in out] $path
     * @param[in out] $like_path
     * @param[in out] $type
     * @param[in out] $object_list
     *           
     * @return
     */
    public function find_modified_objects($path = null, $like_path = true, $type = Safe_object_serializer_class::OBJ_TYPE_CONFIGURABLE, &$object_list = null)
    {
        $this->db->where('modified !=', 0);
        if (isset($path)) {
            if (true == $like_path) $this->db->like('path', $path, 'after');
            else $this->db->where(array(
                'path' => $path
            ));
        }
        if (isset($type)) {
            $this->db->where(array(
                'type' => $type
            ));
        }
        $q = $this->db->get($this->db_table);
        if ($q->num_rows()) {
            if (isset($object_list)) {
                $object_list = array();
                foreach ($q->rows() as $row) $object_list[] = $row->path . $row->name;
            }
        }
        return $q->num_rows();
    }
    /**
     * @brief Clear all modified objects
     *           
     * @param[in out] $path
     * @param[in out] $recurse
     *           
     * @return
     */
    public function clear_modified($path, $recurse, $type = Safe_object_serializer_class::OBJ_TYPE_CONFIGURABLE)
    {
        // Delete disposed objects
        $this->delete_object_with_status($path, $recurse, Safe_object_serializer_class::OBJ_STATUS_DELETED, $type);
        // update status and modified flags for other modified objects
        $this->update_object_with_status($path, $recurse, NULL, Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE, $type);
    }
    public function update_object_with_status($path, $recurse, $old_status, $new_status, $type = Safe_object_serializer_class::OBJ_TYPE_CONFIGURABLE)
    {
        // update status and modified flags for other modified objects
        //$this->db->where (array('modified !=' => 0));
        // If old_status not set, assume all
        if (isset($old_status)) $this->db->where('status', $old_status);
        if (isset($type)) $this->db->where('type', $type);
        if ($recurse) $this->db->like('path', $this->db->escape_like_str($path) , 'after');
        else $this->db->where('path', $path);
        if (Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE == $new_status) $update['modified'] = 0;
        else $update['modified'] = 1;
        $update['status'] = $new_status;
        $query = $this->db->update($this->db_table, $update);
    }
    /**
     * @brief Clear all object in path with specific status
     *           
     * @param[in out] $path
     * @param[in out] $recurse
     * @param[in out] $status
     *           
     * @return
     */
    public function delete_object_with_status($path, $recurse, $status, $type = Safe_object_serializer_class::OBJ_TYPE_CONFIGURABLE)
    {
        // Delete disposed objects
        $this->db->where('status', $status);
        if (isset($type)) $this->db->where('type', $type);
        if ($recurse) $this->db->like('path', $this->db->escape_like_str($path) , 'after');
        else $this->db->where('path', $path);
        $this->db->delete($this->db_table);
    }
    public function get_objects()
    {
        $objects = array();
        $q = $this->db->get($this->db_table);
        $objects = $q->result_array();
        return $objects;
    }
    public function get_object($id)
    {
        $object = array();
        $this->db->where('id', $id);
        $q = $this->db->get($this->db_table);
        if ($q->num_rows()) {
            $object = $q->row_array();
        }
        return $object;
    }
    public function remove_object($id)
    {
        $this->db->where('id', $id);
        $this->db->delete($this->db_table);
    }
    /**
     * @brief
     *         
     * @return
     */
    public function next_insert_id()
    {
        $last_row = array();
        $query = $this->db->get('safe_object');
        $last = $query->last_row('array');
        $id = $last['id'];
        return $id + 1;
    }
}
class Safe_object_class
{
    private $_path;
    private $_name;
    private $_description = '';
    private $_documentation = '';
    private $_class = "";
    private $_aggr_obj_type = array();
    protected $_properties = array();
    public function __construct($path, $name)
    {
        $this->_path = $path;
        $this->_name = $name;
        $this->_description = $name;
        $this->_class = get_class($this);
    }
    public function object_name()
    {
        return $this->_path . "/" . $this->_name;
    }
    public function class_name()
    {
        return $this->_class;
    }
    public function path()
    {
        return $this->_path;
    }
    public function name()
    {
        return $this->_name;
    }
    public function is_configuration_modified($include_childs = true)
    {
        $ser = Safe_object_serializer_class::get_serializer();
        $modified_cnt = $ser->find_modified_objects($this->object_name() , $include_childs);
        return (0 != $modified_cnt);
    }
    public function configure()
    {
        return TRUE;
    }

    /**
     * @brief Return object description
     *         
     * @return string - object description
     */
    public function description()
    {
        return $this->_description;
    }
    /**
     * @brief Set object decsription
     *            
     * @param[in] $description - string
     *            
     * @return
     */
    public function set_description($description)
    {
        $this->_description = $description;
    }
    public function set_properties($properties)
    {
        $this->_properties = $properties;
    }
    public function property($name, $default)
    {
        if($this->_properties[$name]){
            return $this->_properties[$name];
        }else{
            return $default;
        }
    }
    /**
     * @brief Return object Documentation
     *
     * Some documentation about object
     *         
     * @return string - object summary
     */
    public function documentation()
    {
        return $this->_documentation;
    }
    /**
     * @brief Set object documentation
     *            
     * @param[in] $documentation - string
     *            
     * @return
     */
    public function set_documentation($documentation)
    {
        $this->_documentation = $documentation;
    }
    /**
     * Used to tell application configuration manager that this is an aggregated objects
     * @return array ( 'type' => object definition )
     */
    public function has_aggregate_objects($include_all=false, $internal=false)
    {
        $types = $this->_aggr_obj_type;
        if(!$include_all) {
            // Remove configuration
            unset($types['configuration']);
            // Filter out service 
            unset($types['service']);
        }
        if(!$internal){
            // Filter out controller specific parameters
            unset($types['controller']);
            // as well as for objects
            foreach($types as &$type) {
                unset($type['controller']);
            }
        }
        return $types;
    }
    public function get_aggregate_object_definition($type=null)
    {
        if($type) {
            return $this->_aggr_obj_type[$type];
        } else {
            return $this->has_aggregate_objects();
        }
    }
    /**
     * @brief Register aggregate object definition
     *
     * @param[in out] $type (string) object type (singular)
     * @param[in out] $definition (array)
     *
     * ['name']           => (string) Long name for display
     * ['description']    => (string) Object description
     * ['dynamic']        => (boolean) object can/can't be reloaded dynamically
     * ['base_path']      => (string) Base object path in database
     * ['class']          => (string) Object class
     * ['has_child']      => (boolean) Object can have children objects
     * ['methods']        => (array) Exposed object methods
     *                          4 fixed entries for CRUD ('create', 'retrieve', 
     *                          'update' and 'delete') + control methods 
     *                          entry (ie. anything not a CRUD entry is control). 
     *                          Method name is resolved in the object 
     *                          (reflection) based on following rules:
     *                            For CRUD - <method>_<object type>
     *                               function prototype:
     *                                  create - <obj/array error>  = create_<object type>($name = null, $data  = null);
     *                                  retrieve - <obj/array> = retrieve_<object type>($filter_name = null);
     *                                  update - <obj/array error> = update_<object type>($name, $data);
     *                                  delete - <true/array error> = delete_<object type>($name);
     *                            For control - ctl_<object type>_<method>
     *                               function prototype:
     *                                  <true/array error> = ctl_<object type>_<method>($name, $params = null);
     *                          Method entry format is:
     *                          [method] => array(
     *                             ['name'] => (string) Long name for display
     *                             ['description'] => (string) Method description
     *                             ['request'] => (string) REST request method allowed (GET, POST, PUT or DELETE)
     *                             (optional) ['param class'] => (string) name of the configurable object containing 
     *                                                       parameter descrition (mostly for control methods)
     *
     * @return 
     */
    public function register_aggregate_object($type, $definition = null)
    {
        if(isset($definition)) {
            if(!is_array($definition)) {
                // Old style name
                $definition = array(
                    'name' => $definition);
            }
        }
        // Validate some mandatory entries
        if(!$definition['name']){
            $definition['name'] = $type;
        }
        if(!$definition['description']){
            $definition['description'] = $definition['name'];
        }
        if(!$definition['dynamic']){
            $definition['dynamic'] = false;
        }
        if(!$definition['base_path'] && false !== $definition['base_path']){
            $definition['base_path'] = $this->object_name().'/'.$type;
        }
        if(!isset($definition['configurable'])){
            $definition['configurable'] = true;
        }
        if(!isset($definition['pagination'])){
            $definition['pagination'] = false;
        }
        if(!isset($definition['global_methods'])){
            $definition['global_methods'] = false;
        }
        if(isset($definition['methods']) && is_array($definition['methods'])){
            foreach($definition['methods'] as $method){
                if(isset($method['scope']) && ($method['scope'] == 'global' || $method['scope'] == 'both')){
                    $definition['global_methods'] = true;
                }
            }
        }
        
        // ...
        /*
        echo '<pre>';
        echo 'Obj='.$type;
        print_r($definition);
        echo '</pre>';
        */

        $this->_aggr_obj_type[$type] = $definition;
    }
    /**
     * Returns an array of all aggregated objects for a specific module
     * Optionally return all objects regardless of their status
     * @return $array
     *                			Array of aggregated configs
     */
    private $_agg_objects_cache = array();
    public function get_aggregate_objects($type, $all = false, $params = array())
    {
        $definition = $this->_aggr_obj_type[$type];
        
        // Do we have base path ?
        if($definition['base_path']) {
            // Check objects cache (or requesting all)
            if(!$this->_agg_objects_cache[$type] || $this->_agg_objects_cache[$type]['all'] != $all){
                $objects = array();
                // Re-create object list
                $_objects = Safe_object_serializer_class::get_serializer()->find_objects($definition['base_path']);
                foreach ($_objects as $k => $v) {
                    $obj = new $v['class']($this->node(), $definition['base_path'], $k);
                    $obj->configure();
                    $obj->synch();
                    $objects[$k] = $obj;
                }
                // Check if we want the deleted objects as well
                if($all){
                    // Re-create object list
                    $_deleted = Safe_object_serializer_class::get_serializer()->find_deleted_objects($definition['base_path']);
                    foreach ($_deleted as $k => $v) {
                        $obj = new $v['class']($this->node(), $definition['base_path'], $k);
                        $obj->configure();
                        $obj->unserialize();
                        $objects[$k] = $obj;
                    }
                }
                // Store in cache
                $this->_agg_objects_cache[$type]['objects'] = $objects;
                $this->_agg_objects_cache[$type]['all'] = $all;
            }
            // Use cache
            return $this->_agg_objects_cache[$type]['objects'];
        } else if (method_exists($this, 'api_retrieve_'.$type)) {
            return call_user_func(array(
                $this,
                'api_retrieve_'.$type
            ), null, $params);
        } else if (method_exists($this, $type)) {
            return call_user_func(array(
                $this,
                $type
            ));
        }
        error_log("WARNING - Cannot retrieve aggregate_object ${type} for module ".$this->name());
        return array();
    }
    
    /**
     * @brief clear objs in cache (so it can be regenerate from db )
     *
     * @param[in out] $type
     *
     * @return
     */
    protected function clear_cache($type = null)
    {
        //clear objs in cache
        if ($type){
            if($this->_agg_objects_cache[$type]){
                unset($this->_agg_objects_cache[$type]);
            }
        }else{
            $this->_agg_objects_cache = array();
        }
    }
    /**
     * @brief Check if specified aggregate object can be created
     *
     * @param[in out] $_type
     * @param[in out] $name
     * @param[in out] $reason)
     *
     * @return 
     */
    public function can_create_aggregate_object($type, $name, &$reason="")
    {
        // Default implementation - Check object name exists
        $objs = $this->get_aggregate_objects($type);
        if($objs[$name]){
            $reason = "Already exists";
            return false;
        }else{
            if (preg_match('/^__.*__$/i', $name)) {
                $reason = " '{$name}' is reserved by system.";
                return false;
            }
        }
        return true;
    }
    public function create_aggregate_object($type, $name='default', &$output=null)
    {
        if('configuration'==$type && $this->config()) {
            return $this->config();
        }
        $obj = null;
        $definition = $this->_aggr_obj_type[$type];

        // Create allowed ?
       // if($definition['methods']['create'])
        {
            $data = null;
            $create_name = 'api_create_'.$type;
            if(method_exists($this, $create_name)) {
                $obj =  call_user_func_array(
                    array(
                    $this,
                    $create_name),
                    array($name , $data , &$output)
                );
            }
        }
        return $obj;
    }
    public function delete_aggregate_object($type, $name='default')
    {
        $definition = $this->_aggr_obj_type[$type];
        // delete allowed ?
        if($definition['methods']['delete']){
            $delete_name = 'api_delete_'.$type;
            if(method_exists($this, $delete_name)) {
                return  call_user_func_array(
                        array(
                                $this,
                                $delete_name),
                        array($name)
                );
            }
        }
        return false;
    }
    public function get_aggregate_object_method_params($type, $method)
    {
        $definition = $this->_aggr_obj_type[$type];
        if($definition['methods'][$method]){
            $method_name = 'api_'.$method.'_'.$type.'_parameters';
            if(method_exists($this, $method_name)) {
                $param =  call_user_func_array( array( $this, $method_name), array() );
                return $param;
            }
        }
        return false;
    }
    
    
    /**
     * @brief
     *           
     * @param[in out] $type
     *           
     * @return
     */
    public function aggregate_object_name($type)
    {
        if($this->_aggr_obj_type[$type])
            return $this->_aggr_obj_type[$type]['name'];
        else
            return $type;
    }
    /**
     * @brief Return object type base path
     *
     * @param[in out] $type
     *
     * @return 
     */
    public function aggregate_object_base_path($type)
    {
        if($this->_aggr_obj_type[$type])
            return $this->_aggr_obj_type[$type]['base_path'];
        else
            return null;
    }
    /**
     * @brief new_fields
     *
     * @param[in out] $config_manager
     * @param[in out] $type
     * 
     * Return 
     *
     * @return
     */
    public function new_fields(&$config_manager, $type)
    {
        $obj_def = $this->get_aggregate_object_definition($type);
        $label = 'New ' . $obj_def['name'];
        $number = count($this->get_aggregate_objects($type))+1;
        $new_name = ucfirst($type) . $number;
        $config_manager->add_field('profile-name', $label, 'string', $new_name, 50);
        $config_manager->set_field_rules('profile-name', 'required|alpha_dash');
    }

    /**
     * @brief Populate auditor with audit point for object
     *
     * @param[in out] $auditor
     *
     * @return 
     */
    public function get_audit_points(&$audit_points)
    {
        return true;
    }
}
abstract class Safe_serializable_object_class extends Safe_object_class
{
    const SYNC_DB = 'db';
    const SYNC_MEM = 'mem';
    private $_sync_strategy = Safe_serializable_object_class::SYNC_DB;
    private $_status = Safe_object_serializer_class::OBJ_STATUS_NEW;
    private $_id = 0;
    private $_type = Safe_object_serializer_class::OBJ_TYPE_CONFIGURABLE;
    static private $_serializer = null;
    public function __construct($path, $name, $sync_strategy = Safe_serializable_object_class::SYNC_DB)
    {
        parent::__construct($path, $name);
        $this->_sync_strategy = $sync_strategy;
    }
    protected function _set_sync_strategy($sync_strategy)
    {
        $this->_sync_strategy = $sync_strategy;
    }
    /**
     * @brief Synchronize serializable object according to startegy
     *         
     * @return
     */
    public function synch()
    {
        // Get current data to check if changed
        $data = $this->_get_serialized_data();
        // Try to retrieve object
        if (TRUE != $this->unserialize() || Safe_object_serializer_class::OBJ_STATUS_DELETED == $this->_status) {
            // Not yet stored, store it
            $this->_set_unserialized_data($data);
            $this->serialize();
        } else {
            // Check if retrieved data is same as the one supplied at constructor
            // and sync strategy set to memory then update db with new content
            $db_data = array();
            $this->_serializer()->retrieve_object($this->path() , $this->name() , $db_data);
            if ($this->_sync_strategy == Safe_serializable_object_class::SYNC_MEM && $data != $db_data) {
                error_log('synch - flush - ' . $this->object_name() . ' Status = ' . $this->_status);
                $this->_status = Safe_object_serializer_class::OBJ_STATUS_MODIFIED;
                // Need to update cache
                $this->_set_unserialized_data($data);
                $this->_serializer()->store_object($this->path() , $this->name() , $this->class_name() , $data, $this->_type, $this->_status, $this->_id);
                //$this->serialize();
            }
        }
    }
    private function _serializer()
    {
        Safe_serializable_object_class::$_serializer = Safe_object_serializer_class::get_serializer();
        return Safe_serializable_object_class::$_serializer;
    }
    public function serialize()
    {
        error_log('serialize - flush - ' . $this->object_name() . ' - Status = ' . $this->_status);
        // Get the data to serialize
        $data = $this->_get_serialized_data();
        if (!empty($data)) {
            // Invoke serializer
            return $this->_serializer()->store_object($this->path() , $this->name() , $this->class_name() , $data, $this->_type, $this->_status, $this->_id);
        }
        // No data to serialize
        return FALSE;
    }
    public function unserialize()
    {
        // Invoke serializer
        $data = array();
        $obj_class = '';
        if ($this->_serializer()->retrieve_object($this->path() , $this->name() , $data, $this->_status, $this->_id, $obj_class, $this->_type)) {
            $this->_set_unserialized_data($data);
            return TRUE;
        }
        return FALSE;
    }
    /**
     * @brief Return object status (from DB)
     *         See Safe_object_serializer_class::OBJ_STATUS_*
     *         
     * @return object status
     */
    public function status()
    {
        return $this->_status;
    }
    /**
     * @brief Returns object id
     *         
     * @return Object id (auto inc id from sqlite)
     */
    public function id()
    {
        return $this->_id;
    }
    public function type()
    {
        return $this->_type;
    }
    public function set_type($type)
    {
        $this->_type = $type;
    }
    /**
     * @brief Indicates if cofigurable object is modified
     *         
     * @return true if modified
     */
    public function is_modified()
    {
        return ($this->status() != Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE);
    }
    /**
     * @brief Clear modified object flag and status
     *         
     * @return
     */
    public function clear_modified()
    {
        return $this->_serializer()->mark_object_uptodate($this->path() , $this->name());
    }
    public function mark_modified()
    {
        $this->_status = Safe_object_serializer_class::OBJ_STATUS_MODIFIED;
        return $this->_serializer()->mark_object_modified($this->path() , $this->name());
    }
    /**
     * @brief Check if the object can be deleted
     *         This function should be implemented in extended objects
     *         to validate any dependencies before deleting an object
     *         
     * @return
     *         true - Object can be deleted
     *         false - Object has dependencies
     *         
     */
    public function can_dispose(&$reason = NULL)
    {
        return TRUE;
    }
    /**
     * @brief Delete an object
     *         Mark the object as deleted
     *         
     * @return
     *         true on success
     */
    public function dispose()
    {
        return $this->_serializer()->dispose_object($this->path() , $this->name());
    }
    //
    // Return a php array of the data that need to be serialized
    abstract protected function _get_serialized_data();
    // Set the php array of retrieved data
    abstract protected function _set_unserialized_data($data);
    /**
     * @brief get_data_values
     *
     * for compatibility on objects not configurable
     *
     * @return 
     */
    public function get_data_values()
    {
        return $this->_get_serialized_data();
    }
}
class Safe_configurable_object_class extends Safe_serializable_object_class
{
    public $_fields = array();
    private $_user_rule_msgs = array();
    protected $_node = null;
    private $_form = null;
    /**
     * @var _group array
     */
    private $_group = array();
    private $_group_condition = array();
    private $_default_category = null;
    private $_default_category_setting = 'expanded';
    /**
     * @brief
     *           
     * @param[in out] $path
     * @param[in out] $name
     *           
     * @return
     */
    public function __construct($path, $name, $node=null)
    {
        $this->_node = $node;
        parent::__construct($path, $name);
    }
    /**
     * @brief Configure the object by adding fields
     *         
     * @return
     */
    public function configure()
    {
        return TRUE;
    }
    public function prepare_edit()
    {
    }
    /**
     * @brief Return node object 
     *
     * @return 
     */
    public function node()
    {
        return $this->_node;
    }
    /**
     * @brief
     *         
     * @return
     */
    public function enable_disable_enum($string_value=true)
    {
        if($string_value){
            return array(
                'true' => 'Enable',
                'false' => 'Disable'
            );
        }else{
            return array(
                '1' => 'Enable',
                '0' => 'Disable'
            );
        }
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $label
     * @param[in out] $type
     * @param[in out] $default
     * @param[in out] $size
     *           
     * @return
     */
    public function add_field($name, $label, $type, $default, $size = null)
    {
        $this->_fields[$name] = array(
            'field' => $name,
            'label' => $label,
            'type' => $type,
            'default' => $default,
        );
        if (isset($size)) $this->_fields[$name]['size'] = $size;
        //set defualt category
        if($this->_default_category != null){
            $this->set_field_category($name , $this->_default_category, $this->_default_category_setting, false);
        }
        return TRUE;
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $label
     * @param[in out] $type
     * @param[in out] $default
     * @param[in out] $enum_array
     *           
     * @return
     */
    public function add_enum_field($name, $label, $type, $default, $enum_array)
    {
        if(isset($enum_array[''])) {
        /*
            throw new Exception("Field ".$name." enumeration contains empty key <".implode(',',array_keys($enum_array).">"));
            error_log("Field ".$name." enumeration contains empty key <".implode(',',array_keys($enum_array).">"));
         */
            // Fixing enum
            $fixed = array();
            foreach($enum_array as $k => $v) {
                if('' == $k){
                    $k = '__empty__';
                }
                $fixed[$k] = $v;
            }
            $enum_array = $fixed;
        }
        if('' == $default){
            $default = '__empty__';
            if(!$enum_array[$default]){
                $keys = array_keys($enum_array);
                $default = $keys[0];
            }
        }
        if (strtolower($type) == "dropdown" || strtolower($type) == "radio") {
            if (TRUE == $this->add_field($name, $label, $type, $default)) {
                $this->_fields[$name]['value'] = $enum_array;
                // Add field rule for enumeration
                $this->set_field_rules($name, '');
                $this->_fields[$name]['rules'] = 'required|in_list[' . implode(',', array_keys($this->_fields[$name]['value'])) . ']';
                return TRUE;
            }
        }
    }
    public function add_upload_field($name, $label, $type, $default, $size = null, $no_textbox = false)
    {
        if (strtolower($type) == "upload") {
            $this->_fields[$name] = array(
                'field' => $name,
                'label' => $label,
                'type' => $type,
                'default' => $default
            );
            if (isset($no_textbox)) $this->_fields[$name]['no_box'] = $no_textbox;
            if (isset($size)) $this->_fields[$name]['size'] = $size;
            return TRUE;
        }
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $label
     * @param[in out] $type
     * @param[in out] $default
     * @param[in out] $enum_array
     *           
     * @return
     */
    public function add_multiple_field($name, $label, $type, $default = array() , $enum_array)
    {
        if (strtolower($type) == "inline_checkbox"){
            $type = "checkbox";
            $inline = true;
        }else{
            $inline = false;
        }
        if (strtolower($type) == "checkbox") {
            if (is_array($default) && count($default) > 0) {
                if (TRUE == $this->add_field($name, $label, $type, $default)) {
                    $this->_fields[$name]['value'] = $enum_array;
                    if($inline) $this->_fields[$name]['inline'] = 'true';
                    return TRUE;
                }
            } elseif (count($default) == 0) {
                if (TRUE == $this->add_field($name, $label, $type, "")) {
                    $this->_fields[$name]['value'] = $enum_array;
                    if($inline) $this->_fields[$name]['inline'] = 'true';
                    return TRUE;
                }
            }
        }
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $param_name
     * @param[in out] $param_value
     *           
     * @return
     */
    private function _set_field_param($name, $param_name, $param_value)
    {
        if (isset($this->_fields[$name])) {
            $this->_fields[$name][$param_name] = $param_value;
            return TRUE;
        }
        return FALSE;
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $rules
     *           
     * @return
     */
    public function set_field_rules($name, $rules)
    {
        if ('dropdown' == $this->_fields[$name]['type'] && $rules != 'disabled') {
            // Ensure enum check is added to rule
            $rules.= '|required|in_list[' . implode(',', array_keys($this->_fields[$name]['value'])) . ']';
        }
        return $this->_set_field_param($name, 'rules', $rules);
    }
    public function set_field_disabled($name, $is_disabled = TRUE)
    {
        return $this->_set_field_param($name, 'disabled', (TRUE == $is_disabled) ? "disabled" : "enabled");
    }
    public function set_field_readonly($name)
    {
        return $this->_set_field_param($name, 'readonly', 'readonly');
    }
    public function set_field_style($name,$style='')
    {
        return $this->_set_field_param($name, 'style', $style);
    }
    public function set_field_category($name, $category, $setting = 'expanded', $set_as_default = false)
    {
        $this->_set_field_param($name, 'category_setting', $setting);
        if($set_as_default == true){
            $this->set_default_category($category, $setting);
        }
        return $this->_set_field_param($name, 'category', $category);
    }
    public function set_default_category($category, $setting = 'expanded')
    {
        $this->_default_category = $category;
        $this->_default_category_setting = $setting;
    }
    public function get_default_category()
    {
        return $this->_default_category;
    }
    public function get_default_category_setting()
    {
        return $this->_default_category_setting;
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $value
     *           
     * @return
     */
    public function set_field_value($name, $value)
    {
        return $this->_set_field_param($name, 'value', $value);
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $default
     *           
     * @return
     */
    public function set_field_default($name, $default)
    {
        return $this->_set_field_param($name, 'default', $default);
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $help
     *           
     * @return
     */
    public function set_field_help($name, $help)
    {
        return $this->_set_field_param($name, 'help', $help);
    }
    public function get_field_help($name)
    {
        $help = '';
        if (isset($this->_fields[$name]['help'])) {
            $help = $this->_fields[$name]['help'];
        }
        return $help;
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $events
     *           
     * @return
     */
    public function set_field_events($name, $events)
    {
        return $this->_set_field_param($name, 'events', $events);
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $attributes array()
     *
     * @return
     */
    public function set_field_attributes($name, $attributes = array())
    {
        if(is_array($attributes)){
            foreach($attributes as $key => $value){
                $this->_set_field_param($name, $key, $value);
            }
        }
        return true;
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $flag
     *           
     * @return
     */
    // TODO: remove after the tests
    public function enable_conditional_control($name, $flag)
    {
        return $this->_set_field_param($name, 'enable_cc', $flag);
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $group
     *           
     * @return
     */
    public function conditional_control($name, $group)
    {
        return $this->_set_field_param($name, 'group', $group);
    }
    /**
     * @brief
     *           
     * @param[in out] $groupname
     * @param[in out] $fields (one dimentional array)
     *           
     * @return
     */
    public function create_group($groupname, $fields , $condition = array('default' => array( 'value' => 'false', 'status' => 'disable')))
    {
        if ($groupname != '' && is_array($fields)) {
            $this->_group[$groupname] = array();
            foreach ($fields as $f) {
                $this->_group[$groupname]['fields'][] = $f;
            }
            if(isset($condition['default'])){
                $this->_group[$groupname]['default'] = $condition['default'];
            }
            if(isset($condition['condition'])){
                $this->_group[$groupname]['condition'] = $condition['condition'];
            }
            
        }
    }
    public function composite_layout($name, $fields)
    {
        if($this->_set_field_param($name, 'composite', $fields)){
            $_help_msg = array();
            foreach($fields as $field){
                $this->_set_field_param($field, 'composite_parent', $name);
                if (isset($this->_fields[$field]['help'])) {
                    if(is_array($this->_fields[$field]['help'])){
                        $field_help =  $this->_fields[$field]['help'];
                    }else{
                        $field_help =  array($this->_fields[$field]['help']);
                    }
                    $_help_msg = array_merge($_help_msg , array($this->get_data_label($field)), $field_help, array(' '));
                    unset($this->_fields[$field]['help']);
                }
            }
            if($_help_msg){
                if(isset($this->_fields[$name]['help'])){
                    if(is_array($this->_fields[$name]['help'])){
                       $parent_help =  $this->_fields[$name]['help'];
                    }else{
                       $parent_help =  array($this->_fields[$name]['help']);
                    }
                    $this->_set_field_param($name, 'help', array_merge(array($this->get_data_label($name)), $parent_help, array(' '), $_help_msg));
                }else{
                    $this->_set_field_param($name, 'help',  $_help_msg);
                }
            }
        }
    }

    public function create_group_condition($field_name, $condition)
    {
        if ($field_name != '' && is_array($condition)) {
            $this->_group_condition[$field_name] = $condition;
            return $this->_set_field_param($field_name, 'group_condition', 'yes');
        }
    }
    /**
     * @brief
     *         
     * @return
     */
    public function get_group()
    {
        return $this->_group;
    }
    public function get_group_condition()
    {
        return $this->_group_condition;
    }
    /**
     * @brief Return object data
     *         
     * @return
     */
    public function get_data($include_hidden = true)
    {
        if($include_hidden){
            return $this->_fields;
        } else {
            // filter out hidden fields
            $fields = array();
            foreach($this->_fields as $k => $v) {
                if($v['type'] == 'hidden') continue;
                $fields[$k] = $v;
            }

            return $fields;
        }
    }
    /**
     * @brief
     *         
     * @return
     */
    public function get_data_keys()
    {
        $keys = array();
        // walk data and strip values, just keep keys
        foreach ($this->_fields as $v) $keys[] = $v['field'];
        return $keys;
    }
    /**
     * @brief
     *           
     * @param[in out] $name
     * @param[in out] $resolve_enum
     *           
     * @return
     */
    public function get_data_value($name, $resolve_enum = true)
    {
        if (!isset($this->_fields[$name]['value']) || false == $resolve_enum){
            if('__empty__' == $this->_fields[$name]['default']){
                return '';
            }else{
                return $this->_fields[$name]['default'];
            }
        }else{
            return $this->_fields[$name]['value'][$this->_fields[$name]['default']];
        }
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $resolve_enum
     *
     * @return
     */
    public function get_data_label($name)
    {
        if (!isset($this->_fields[$name]['label']) ){
            return '';
        }else{
            return $this->_fields[$name]['label'];
        }
    }
    /**
     * @brief Explode data values array into array of data vlues
     *           based on following parameter name pattern
     *           xx/yy where xx = configuration category and yy = parameter name
     *           Returned array is as follow:
     *           $values[xx][yy] = value
     *           
     * @param[in out] $resolve_enum
     *           
     * @return
     */
    public function get_data_values_exploded($resolve_enum = true, $include_hidden = true)
    {
        $values = $this->get_data_values($resolve_enum, $include_hidden);
        $exploded_values = array();
        foreach ($values as $k => $v) {
            $exploded_key = explode('/', $k);
            if(sizeof($exploded_key) > 1){
                // loop around exploded key(s)
                $entry = &$exploded_values;
                foreach($exploded_key as $key){
                    $entry = &$entry[$key];
                }
                $entry = $v;
            }else{
                $exploded_values['__all__'][$k] = $v;
            }
        }
        return $exploded_values;
    }
    /**
     * @brief
     *           If resolve_enum is set to true, when the field value is an enum, the
     *           returned value will be the label of the enum. To return the actual value,
     *           set to false.
     *           
     * @param[in out] $resolve_enum
     *           
     * @return
     */
    public function get_data_values($resolve_enum = true, $include_hidden = true)
    {
        $values = array();
        // walk data and strip keys, just keep values
        foreach ($this->_fields as $k => $v) {
            if (!$include_hidden && $v['type'] == 'hidden'){
                continue;
            }
            if (!isset($v['value']) || false == $resolve_enum){
                $values[$k] = $v['default'];
            }else{
                if (isset($v['value'][$v['default']])){
                    $values[$k] = $v['value'][$v['default']];
                } else {
                    if ($resolve_enum){
                        $values[$k] = 'INVALID';
                    }else{
                        $values[$k] = $v['default'];
                    }
                    //safe_add_error( 'Unable to resolve enum value "'.$v['default'].'" for field '.$k);
                }
            }
            // Adjust known special values
            if(!$resolve_enum && '__empty__' == $values[$k]){
                $values[$k] = '';
            }
        }
        return $values;
    }
    public function get_data_values_with_field_desc($resolve_enum = true, $include_hidden = true, $include_password = true)
    {
        $values = array();
        // walk data and strip keys, just keep values
        foreach ($this->_fields as $k => $v) {
            if (!$include_hidden && $v['type'] == 'hidden'){
                continue;
            }
            if (!$include_password && $v['type'] == 'password'){
                continue;
            }
            if (!isset($v['value']) || false == $resolve_enum){
                $values[$v['label']] = $v['default'];
            }else{
                if(is_array($v['default'])){
                    foreach($v['default'] as $v1){
                        if (isset($v['value'][$v1])) {
                            if($values[$v['label']]){
                                $values[$v['label']] .= ', ';
                            }
                            $values[$v['label']] .= $v['value'][$v1];
                        }else{
                            $values[$v['label']] .= 'INVALID - ' . $v1;
                        }
                    }
                }else{
                    if (isset($v['value'][$v['default']])) {
                        $values[$v['label']] = $v['value'][$v['default']];
                    }else {
                        $values[$v['label']] = 'INVALID - ' . $v['default'];
                        //safe_add_error( 'Unable to resolve enum value "'.$v['default'].'" for field '.$k);
                    }
                }
                // Adjust known special values
                if(!$resolve_enum && '__empty__' == $values[$k]){
                    $values[$k] = '';
                }
            }
        }
        return $values;
    }
    public function summary($type = 'horizontal' , $long = false, $min=3)
    {
        $list =  $this->get_data_values_with_field_desc(true, false, false);
        if($long){
            array_unshift($list,array('Name' => $this->name()));
            return array( 'data' => $list);
        }else{
            $result =  array('Name' => $this->name());
            foreach($list as $key => $item){
                $min--;
                $result[$key] = $item;
                if($min == 0) break;
            }
            return array( 'data' => $result);
        } 
    }
    /**
     * @brief
     *           
     * @param[in out] $data
     *           
     * @return
     */
    public function set_data_values($data)
    {
        // Walk data and adjust _fields
        foreach ($data as $k => $v) {
            if (!isset($this->_fields[$k])) {
                // TODO(wadam) - Does not exist in the current object
                // Need to create field
                
            } else {
                // Update the field value or default
                if (isset($this->_fields[$k]['value'])) {
                    if('' == $v){
                        $v = '__empty__';
                    }
                    // Enum value, figure out if we have label or real value
                    if (isset($this->_fields[$k]['value'][$v])){
                        $this->_fields[$k]['default'] = $v;
                    }else {
                        // Default is to use supplied value
                        $this->_fields[$k]['default'] = $v;
                        // Label, walk enum to get the key
                        foreach ($this->_fields[$k]['value'] as $enum_key => $enum_label) {
                            if ($enum_label == $v) {
                                $this->_fields[$k]['default'] = $enum_key;
                                break;
                            }
                        }
                        //safe_add_error( 'Unable to resolve enum value "'.$v.'" for field '.$k);
                    }
                }else{
                    $this->_fields[$k]['default'] = $v;
                }
            }
        }
        return TRUE;
    }
    /**
     * @brief
     *         
     * @return
     */
    public function get_data_labels()
    {
        $values = array();
        // walk data and strip keys, just keep values
        foreach ($this->_fields as $v) $values[$v['field']] = $v['label'];
        return $values;
    }
    /**
     * @brief Serializable object interface implementation
     *         
     * @return
     */
    protected function _get_serialized_data()
    {
        return $this->get_data_values(false);
    }
    /**
     * @brief Set the php array of retrieved data
     *           
     * @param[in out] $data
     *           
     * @return
     */
    protected function _set_unserialized_data($data)
    {
        $this->set_data_values($data);
    }
    /**
     * @brief
     *         
     * @return
     */
    public function save()
    {
        return $this->serialize();
    }
    /**
     * @brief Set user rule messages for error
     *           
     * @param[in out] $user_rule_msgs
     *           format=  array('rule'=>'message',...);
     *           
     * @return
     */
    public function set_user_rule_msgs($user_rule_msgs)
    {
        $this->_user_rule_msgs = $user_rule_msgs;
    }
    /**
     * @brief Return user rule messages
     *         
     * @return array('rule'=>'message',...);
     */
    public function user_rule_msgs()
    {
        return $this->_user_rule_msgs;
    }
    /**
     * @brief get safe from obj
     *
     * @return
     */
    public function get_form(){
        if($this->_form != null){
            $this->_form->set_config($this->get_data());
        }else{
            $this->_form = new Safe_form_class($this->get_data());
        }
        return $this->_form;
    }
    /**
     * @brief validation post data 
     *
     * @param obj $obj (obj to get config)
     * @param array $post (post data to be validate)
     * @param array &$validation_errors (Array output msg)
     *
     * @return
     */
    public function validate($data, &$out_message = null){
        if(!isset($out_message)) {
            $out_message = array();
        }
        // Validate field exists and allowed to be modified
        $fields = $this->get_data(true);
        foreach($data as $key => $value) {
            // Exists
            if($fields[$key]) {
                if($fields[$key]['type'] == 'hidden' || $fields[$key]['readonly'] == 'readonly') {
                    unset($data[$key]);//ignore readonly field
                }
                if($fields[$key]['disabled'] == 'disabled') {
                    $this->set_field_rules($key, 'disabled');
                }
            } else {
                $out_message[$key] = 'Unknown Parameter.';
            }
        }
        // Any error ?
        if($out_message)
            return false;
        // Update data in object
        $this->set_data_values($data);
        $_POST = $this->get_data_values(false);
        $this->get_form();
        //set validation obj (for call back check rule)
        $this->_form->set_validation_obj($this);
        return $this->_form->run_validation($out_message);
    }
    public function set_form_error($field = '', $message = ''){
        $this->_form->set_error($field , $message);
    }
    
}
/* End of file safe_object_class.php */
