#!/usr/bin/env python
# vim: tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab
"""
* Copyright (C) 2013   Sangoma Technologies Corp.
* All Rights Reserved.
*
* This code is Sangoma Technologies Confidential Property.
* Use of and access to this code is covered by a previously executed
* non-disclosure agreement between Sangoma Technologies and the Recipient.
* This code is being supplied for evaluation purposes only and is not to be
* used for any other purpose.
"""

import sys
import os
import subprocess
import xml.dom.minidom as xml

from optparse import OptionParser
import logging
import logging.handlers

# Usage:
#   sng_raid_check.py
# 
# Output format: (This tool is for monit processing)- NO pretty XML print
"""
<program>
    <name>RAID Status</name>
    <message>Output message string </message>
</program>
"""
# Upon success, rc=0, data section populated, message set to:
#    OK
#
# Upon limit in percent exceeded, returns -1 and message set to:
#    RAID degraded ....
#
# Upon error, rc=0 (yes I know :) - ... (TODO)

def read_raid_tool(args):
    args.insert(0, '/usr/local/sng/bin/sng-raid-tool')
    out, err = subprocess.Popen(args, stdout=subprocess.PIPE).communicate()
    return out

def getAttr(node, name):
    return node.getAttribute(name)

def getSubText(node, name):
    try:
        return node.getElementsByTagName(name)[0].childNodes[0].data
    except:
        return None

def getSubListText(node, name):
    res = []

    try:
        for elm in node.getElementsByTagName(name):
            res.append(elm.childNodes[0].data)
    except:
        pass

    return res

def get_changes_disks(failed):
    changes = xml.parseString(read_raid_tool(['--changes']))
    devevts = []

    actions = {
        'device-remove': 'Device %(disk)s removed (%(arrays)s).',
        'device-detach': 'Device %(disk)s excluded from arrays (%(arrays)s)',
        'device-fail':   'Device %(disk)s failed (%(arrays)s).',
    }

    for changenode in changes.documentElement.getElementsByTagName("change"):
        cgtype = getAttr(changenode, "type")
        cgarray = getSubListText(changenode, "array")

        action = actions.get(cgtype)

        if action is None:
            continue

        for array in cgarray:
            if array not in failed:
                continue

            disknode = changenode.getElementsByTagName("disk")[0]

            diskvendor = getSubText(disknode, "vendor")
            diskmodel  = getSubText(disknode, "model")
            diskserial = getSubText(disknode, "serial")
            disksize   = getSubText(disknode, "size")
            diskdevice = getSubText(disknode, "device")

            listextra = []

            if diskserial is not None:
                listextra.append('serial ' + diskserial)

            if disksize is not None:
                listextra.append('size ' + disksize)

            textdata = diskvendor + ' ' + diskmodel
            textextra = ', '.join(listextra)

            if textextra <> '':
                textdata = textdata + ' (' + textextra + ')'

            devevts.append(action % {'disk': textdata, 'arrays': ','.join(failed)})
            break

    return devevts

def check_raid(logger, options):
    rc = 0
    message = 'OK'

    status = xml.parseString(read_raid_tool(['--status']))
    failed = []

    for arraynode in status.documentElement.getElementsByTagName("array"):
        state = getSubText(arraynode, "state")
        if state == "degraded":
            failed.append(getSubText(arraynode, "name"))
            rc = -1

    if len(failed) <> 0:
        changes = get_changes_disks(failed)
        message = 'RAID array(s) degraded'

        if len(changes) <> 0:
            nlspace  = '\n    '

            message += ':' + nlspace
            message += nlspace.join(changes)

        else:
            message += '.'

    doc = xml.Document()

    # inner root
    root = doc.createElement('program');
    s = doc.createElement('name')
    st = doc.createTextNode('RAID Status')
    s.appendChild(st)
    root.appendChild(s)

    # message element
    s = doc.createElement('message')
    st = doc.createTextNode(message)
    s.appendChild(st)
    root.appendChild(s)

    # output result
    if options.pretty:
        logger.info(root.toprettyxml())
    else:
        logger.info(root.toxml())

    return rc

def main():
    #Parse the options
    usage = "usage: %prog [options] arg"
    optParser = OptionParser(usage)
    optParser.add_option('-p', '--pretty', action='store_true',
                        dest='pretty', metavar='PRETTY',
                        help="Pretty print output")

    (options, args) = optParser.parse_args()

    # Create logger
    logger = logging.getLogger(os.path.basename(os.path.abspath(sys.argv[0])))
    logger.setLevel(logging.INFO)
    handler = logging.StreamHandler()
    logger.addHandler(handler)

    rc = check_raid(logger, options)
    sys.exit(rc)

if __name__ == '__main__':
  main()
