<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * Module class
*/
require_once ('application/helpers/safe_helper.php');
require_once ('safe_object_class.php');
class Safe_module_class extends Safe_object_class
{
    private $_node = null;
    private $_config = null;
    private $_modules = array();
    public function __construct($software, $name, $type = "module")
    {
        parent::__construct($software->object_name() , $name);
        if (is_subclass_of($software, "Safe_node")) {
            $this->_node = $software;
        } else {
            $this->_node = $software->node();
        }
    }
    public function set_node($node)
    {
        $this->_node = $node;
    }
    public function node()
    {
        return $this->_node;
    }
    public function config()
    {
        return $this->_config;
    }
    public function allow_rest(){
        return false;
    }
    public function set_config($config)
    {
        unset($this->_config);
        if($config) {
            // Get data to expose parameters
            $data = $config->get_data(false);
            // expose config as aggregate object
            $this->register_aggregate_object( 'configuration',
                array(
                    'name' => 'Configuration',
                    'singleton' => true,
                    'methods' => array(
                        'retrieve' => array(
                            'name' => 'Retrieve',
                            'request' => 'GET',
                        ),
                        'update' => array(
                            'name' => 'Update',
                            'request' => 'POST',
                        )
                    ),
                    'class' => $data,
                )
            );
        }
        $this->_config = $config;
        return true;
    }
    /**
     * @brief 
     *
     * @return 
     */
    public function api_retrieve_configuration($obj_name=null, $params=null, &$output=null)
    {
        return $this->_config;
    }
    public function api_update_configuration($obj_name=null, $params=null, &$output=null)
    {
        if($this->_config) {
            if($this->_config->validate($params, $output)) {
                return $this->_config->save();
            }
        }
        return false;
    }
    /**
     * @brief REST GET data
     *         
     * @return
     */
    public function rest_get_data()
    {
        $data = array();
        // If config present, add it
        if (isset($this->_config)) {
            $data['configuration'] = $this->_config->rest_get_data();
        }
        // Ask parent to add its stuff
        $rest_data = parent::rest_get_data();
        if ($data) $rest_data = array_merge($rest_data, $data);
        // Check for modules
        foreach ($this->modules() as $module) {
            $data = $module->rest_get_data();
            if ($data) $rest_data[$module->name()] = $data;
        }
        return $rest_data;
    }
    public function find_module_by_name($name)
    {
        if($name == $this->name()) {
            return $this;
        }
        $mod = null;
        foreach($this->modules() as $k => $v) {
            if($name == $k) {
                return $v;
            }else{
                $mod = $v->find_module_by_name($name);
                if($mod)
                    return $mod;
            }
        }
        return null;
    }
    /**
     * @brief Overridable generate config for reload
     *
     * @param[in out] $config_manager
     * @param[in out] $obj_type
     *
     * @return 
     */
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->reload_generate_config($config_manager, $obj_type)) return false;
        }
        return true;
    }
    /**
     * @brief Generate configuration
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    public function generate_config(&$config_manager)
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->generate_config($config_manager)) return false;
        }
        return true;
    }
    /**
     * @brief Invoked after successfull generate and write config in reload 
     * mode
     *
     * @return 
     */
    public function post_write_config($obj_type=null)
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_write_config($obj_type)) return false;
        }
        return true;
    }
    public function reload_clear_modified($obj_type=null)
    {
    }
    public function clear_configuration_modified($include_childs = true)
    {
        // Clear childs ?
        if ($include_childs) Safe_object_serializer_class::get_serializer()->clear_modified($this->object_name() , true);
        // Clear ourself
        if ($this->config()) $this->config()->clear_modified();
    }
    public function is_configuration_modified($include_childs = true)
    {
        if ($this->config() && $this->config()->is_modified()) return true;
        return parent::is_configuration_modified($include_childs);
    }
    /**
     * @brief Module support reload ?
     *         
     * @return boolean
     */
    public function support_reload(&$need_reload = NULL)
    {
        if (isset($need_reload)) {
            $need_reload = $this->is_configuration_modified(false);
        }
        return false;
    }
    /**
     * @brief Reload module
     *         Write config + post_write_config
     *         
     * @return
     */
    public function reload()
    {
        // support reload ?
        if ($this->support_reload() && $this->is_configuration_modified()) {
            $cfg_mgr = $this->node()->configuration_manager();
            // Take care of generate configuration
            if ($this->generate_config($cfg_mgr) && !$cfg_mgr->errors()
            && $cfg_mgr->write_config($this->node())
            // invoke post_write_config
            && $this->post_write_config()) {
                // clear modified flags
                $this->clear_configuration_modified(true);
                return true;
            }
            return false;
        }
        return true;
    }
    public function register_module($module)
    {
        $this->_modules[$module->name() ] = $module;
    }
    public function modules()
    {
        return $this->_modules;
    }
    public function module($name)
    {
        if (isset($this->_modules[$name])) return $this->_modules[$name];
        else return null;
    }
    public function configure()
    {
        // Configure all modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->configure()) return false;
        }
        return parent::configure();
    }
    /**
     * @brief Return array of variables that need to be defined globally for
     *         application (key=>value)
     *         
     * @return array
     */
    public function get_global_variables()
    {
        $vars = array();
        foreach ($this->_modules as $module) {
            array_merge($vars, $module->get_global_variables());
        }
        return $vars;
    }
    /**
     * @brief Return module category
     *         Should be implemented by module instance
     *         
     * @return string
     */
    private $_category = 'Other';
    public function set_category($category)
    {
        $this->_category = $category;
    }
    public function category()
    {
        return $this->_category;
    }
    /**
     * @brief Return module data settings
     *         If module can expose data(s) it should return an array formatted as follow:
     *         $array= <data_name> (
     *         ['description'] => Data description
     *         ['filesystem']  => true/false - Are data stored in filesystem
     *         ['singleton']  => true/false - Only one data present at a time
     *         ['pattern']        => Defined the data path pattern for filesystem data (to use with glob)
     *         ['global_capabilities'] See below but globaly data MUST exist (ie. download all,
     *         delete all...)
     *         ['global_capabilities_no_data'] See below but globaly even if no
     *         data (ie. upload, ...)
     *         ['capabilities'] =array() possible entries are:
     *         'download'    => Data can be downloaded
     *         'delete'      => Data can be deleted (cleanup)
     *         'view'        => Data can be viewed
     *         'rotate'      => Data can be viewed
     *         
     *         If data are not in filesystem, for every entries in capabilities the module MUST implement the corresponding
     *         data_<capability>($data_name) function
     * @return array as describe above
     */
    public function get_data_settings()
    {
        return array();
    }
    /**
     * @brief Return count of specific data name
     *           
     * @param[in out] $data_name
     *           
     * @return
     */
    public function get_data_count($data_name)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            return count($this->node()->os()->search_files(
                $data_cfg[$data_name]['pattern'],$data_cfg[$data_name]['recursive']));
        }
        return 0;
    }
    /**
     * @brief Return count of specific data name
     *           
     * @param[in out] $data_name
     *           
     * @return array( 'key' => 'file path' ) for each data entries
     */
    public function get_data_files($data_name)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            $data_files = array();
            $files = $this->node()->os()->search_files($data_cfg[$data_name]['pattern'], $data_cfg[$data_name]['recursive']);
            foreach ($files as $file) {
                $data_files[$this->get_data_file_key($data_name, $file) ] = $file;
            }
            return $data_files;
        }
        throw new Exception('Module ' . $this->description() . ' get_data_files error for ' . $data_name);
    }
    /**
     * @brief Retrieve data file key
     *           Default impletmentation if returning file name (basename)
     *           
     * @param[in out] $data_name
     * @param[in out] $file
     *           
     * @return
     */
    public function get_data_file_key($data_name, $file)
    {
        return basename($file);
    }
    /**
     * @brief Retrieve data file content
     *           
     * @param[in out] $data_name
     * @param[in out] $key
     *           
     * @return
     */
    public function get_data_file_content_by_key($data_name, $key, $line_count = NULL, $max_size = NULL)
    {
        $data_cfg = $this->get_data_settings();
        $content = "";
        $tmp_file_name = pathinfo($key, PATHINFO_FILENAME);
        if($data_cfg[$data_name]['file_ext']){
            $tmp_ext = $data_cfg[$data_name]['file_ext'];
        }else{
            $tmp_ext = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_EXTENSION);
        }
        if($data_cfg[$data_name]['file_dir']){
            $tmp_file_path = $data_cfg[$data_name]['file_dir'];
        }else{
            $tmp_file_path = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_DIRNAME );
        }
        $file = $this->node()->os()->find_file($tmp_file_path, $tmp_file_name.'.'.$tmp_ext, $data_cfg[$data_name]['recursive']);
        if ($file) {
            // Get data file
            $os = $this->node()->os();
            $os->chown_file($file, 'webconfig');
            // Read content
            if (isset($line_count)) {
                $os = $this->node()->os();
                $output = array();
                $rc = $os->execute('sudo tail -n ' . $line_count . ' ' . $file, $output);
                $content = implode('\n', $output);
            } elseif (isset($max_size)) {
                $content = file_get_contents($file, false, NULL, -1, $max_size);
            } else {
                $content = file_get_contents($file);
            }
        }
        return $content;
    }
    /**
     * @brief Validate a data key exists for a specific data set
     *           
     * @param[in out] $data_name
     * @param[in out] $key
     *           
     * @return
     */
    public function data_key_exists($data_name, $key)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            $tmp_file_name = pathinfo($key, PATHINFO_FILENAME);
            if($data_cfg[$data_name]['file_ext']){
                $tmp_ext = $data_cfg[$data_name]['file_ext'];
            }else{
                $tmp_ext = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_EXTENSION);
            }
            if($data_cfg[$data_name]['file_dir']){
                $tmp_file_path = $data_cfg[$data_name]['file_dir'];
            }else{
                $tmp_file_path = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_DIRNAME );
            }
            return  $this->node()->os()->check_file_exists($tmp_file_path .'/' . $tmp_file_name.'.'.$tmp_ext, $data_cfg[$data_name]['recursive']);
        }
        throw new Exception('Module ' . $this->description() . ' data_key_exists error for ' . $data_name);
    }
    /**
     * @brief Delete all data
     *           
     * @param[in out] $data_name
     *           
     * @return
     */
    public function delete_data($data_name, $data_key = NULL)
    {
        $data_cfg = $this->get_data_settings();
        if (isset($data_cfg[$data_name]) && true == $data_cfg[$data_name]['filesystem'] && isset($data_cfg[$data_name]['pattern'])) {
            // Locate single key ?
            if (isset($data_key)) {
                $tmp_file_name = pathinfo($data_key, PATHINFO_FILENAME);
                if($data_cfg[$data_name]['file_ext']){
                    $tmp_ext = $data_cfg[$data_name]['file_ext'];
                }else{
                    $tmp_ext = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_EXTENSION);
                }
                if($data_cfg[$data_name]['file_dir']){
                    $tmp_file_path = $data_cfg[$data_name]['file_dir'];
                }else{
                    $tmp_file_path = pathinfo($data_cfg[$data_name]['pattern'] , PATHINFO_DIRNAME );
                }
                $file = $this->node()->os()->find_file($tmp_file_path, $tmp_file_name.'.'.$tmp_ext, $data_cfg[$data_name]['recursive']);
                if($file){
                    $files[] = $file;
                }
            } else {
                $data_files = array();
                $files = $this->node()->os()->search_files($data_cfg[$data_name]['pattern'], $data_cfg[$data_name]['recursive']);
            }
            // loop to delete selected file(s)
            foreach ($files as $file) {
                $os = $this->node()->os();
                $os->chown_file($file, 'webconfig');
                $os->delete_file($file);
            }
            return true;
        }
        throw new Exception('Module ' . $this->description() . ' delete_data error for ' . $data_name);
    }
    public function initialize_data($data_name, $data_key, &$output=null)
    {
        return false;
    }
    public function describe_data($data_name, $data_key, &$output=null)
    {
        $output[] = $data_key;
        return false;
    }
    public function install_data($data_name, $data_key, &$output=null)
    {
        return false;
    }
    public function can_install_data($data_name, $data_key, &$output=null)
    {
        return false;
    }
    /**
     * @brief
     *           
     * @param[in out] $data_name
     *           
     * @return
     */
    public function rotate_data($data_name)
    {
        // For now return false
        // May use logrotate feature as default
        return false;
    }
    /**
     * @brief Check if backup can be performed
     *           
     * @param[in out] $reason
     *           
     * @return true/false
     */
    public function can_backup(&$reason)
    {
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->can_backup($reason)) return false;
        }
        return true;
    }
    /**
     * @brief Pre backup hook
     *           
     * @param[in out] $reason
     *           
     * @return true/false
     */
    public function pre_backup(&$reason)
    {
        error_log('pre_backup - '.$this->name());
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->pre_backup($reason)) return false;
        }
        return true;
    }
    /**
     * @brief Post backup hook
     *           
     * @param[in out] $reason
     *           
     * @return true/false
     */
    public function post_backup(&$reason)
    {
        error_log('post_backup - '.$this->name());
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_backup($reason)) return false;
        }
        return true;
    }
    /**
     * @brief Check if archive restore can be done
     *           
     * @param[in out] $reason
     *           
     * @return true/false
     */
    public function can_restore(&$reason)
    {
        // Default is true
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->can_restore($reason)) return false;
        }
        return true;
    }
    /**
     * @brief Pre restore hook
     *           
     * @param[in out] $reason
     *           
     * @return true/false
     */
    public function pre_restore(&$reason)
    {
        error_log('pre_restore - '.$this->name());
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->pre_restore($reason)) return false;
        }
        return true;
    }
    /**
     * @brief Post restore hook
     *           
     * @param[in out] $reason
     *           
     * @return true/false
     */
    public function post_restore(&$reason)
    {
        error_log('post_restore - '.$this->name());
        // Make sure config is synched from DB
        if($this->config()){
            $this->config()->synch();
        }
        $this->clear_cache();
        // Loop all sub-modules
        foreach ($this->_modules as $k => $v) {
            if (true != $v->post_restore($reason)) return false;
        }
        return true;
    }
    private function _modified_aggregate_object_list($obj, &$obj_list, $reloadable_only=true)
    {
        if ($agg_obj = $obj->has_aggregate_objects(true)) {
            // Loop around aggregate obj type
            foreach($agg_obj as $obj_type => $obj_type_name) {
                $objs = $obj->get_aggregate_objects($obj_type);
                // loop around obj
                foreach($objs as $k => $v) {
                    if($v->is_modified()) {
                        $obj_list[$obj_type][$k] = 
                            array( 
                                'name' => $k,
                            );
                    }
                    // Recurse...
                    if ($v->has_aggregate_objects(true)) {
                        $agg_list = array();
                        $this->_modified_aggregate_object_list($v, $agg_list, $reloadable_only);
                        if($agg_list){
                            $tmp = $obj_list[$obj_type][$k]['objs'];
                            if($tmp) {
                                $obj_list[$obj_type][$k]['objs'] = array_merge($tmp, $agg_list); 
                            }else{
                                $obj_list[$obj_type][$k]['objs'] = $agg_list; 
                            }
                        }
                    }
                }
            }
        }
    }
    /**
     * @brief Retrieve list of modified obj/conf
     * Optionnaly return all or only reloadable
     *
     * @param[in out] $obj_list
     * @param[in out] $reloadable_only
     *
     * @return 
     */
    public function modified_object_list(&$obj_list, $reloadable_only=true)
    {
        // Do we support reload and need relaod?
        $need_reload = false;
        if($this->support_reload($need_reload) && $need_reload) {
            // Check for config modified
            $obj_list[$this->name()]['configuration'] = $this->config();
        }
        // Check aggregate objects
        foreach($this->has_aggregate_objects(true) as $k => $v ) {
            // Check if this object is dynamic or if any child (which can be 
            // dyanmic)
            if($v['dynamic'] || $v['has_child']) {
                // Get those objects and check for modified ones
                $objs = $this->get_aggregate_objects($k, true);
                foreach($objs as $obj_name => $obj) {
                    if($v['dynamic'] && $obj->status() != Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE) {
                        $obj_list[$this->name()][$k][$obj_name] = $obj;
                    }
                    if($v['has_child'] && $obj->is_configuration_modified(true)) {
                        $child_objs_def = $obj->has_aggregate_objects(true);
                        foreach($child_objs_def as $child_obj_type => $child_obj_def) {
                            if($child_obj_def['dynamic']) {
                                $child_objs = $obj->get_aggregate_objects($child_obj_type, true);
                                foreach($child_objs as $child_obj_name => $child_obj) {
                                    if($child_obj->is_modified()) {
                                        $obj_list[$this->name()][$k.'/'.$child_obj_type][$obj_name.'/'.$child_obj_name] = $child_obj;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        // Loop around modules
        foreach($this->modules() as $mod) {
            $mod_obj_list = array();
            $mod->modified_object_list($obj_list, $reloadable_only);
        }
    }

    /**
     * @brief Check conflict for a particular resource
     *
     *  For example TCP port in use by another module cfg/obj...
     *  Implementation is done per module.
     *
     *  Return an array of detected conflict, following format:
     *  'name'       => Module reporting conflict
     *  'obj_type'   => Object type 
     *  'obj_name'   => (opt) Object name (except for configuration) 
     *  'child_type' => (opt) Child type 
     *  'child_name' => (opt) Object name
     *
     * @param[in out] $src_module_name - (string) Module asking for conflict check 
     * @param[in out] $resource_type   = (string) Resource type to check, for ex: "IP", "PORT"...
     * @param[in out] $resource_data   = (array)  Resource data
     *
     * @return 
     */
    public function check_resource_conflict($src_module_name, $resource_type, $resource_data) 
    {
        $conflicts = array();
        // Loop around modules
        foreach($this->modules() as $mod) {
            // Skip ourself :)
            if($src_module_name == $mod->name()){
                continue;
            }
            $sub_conflicts = array();
            $sub_conflicts = $mod->check_resource_conflict($src_module_name, $resource_type, $resource_data);
            if($sub_conflicts){
                $conflicts = array_merge($conflicts, $sub_conflicts);
            }
        }
        return $conflicts;
    }
    /**
     * @brief Module checklist
     * 
     * Return list of item not complying to check list
     *
     * @return 
     */
    public function checklist(&$list)
    {
        // Loop around modules
        foreach($this->modules() as $mod) {
            $mod->checklist($list);
        }
        return count($list);
    }
}
/* End of file safe_module_class.php */
