<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
*/
/*
 * FS SIP module class
*/
require_once ('application/helpers/safe_helper.php');
safe_require_class('safe_service_class');
safe_module_require_class('fs', 'application');

class Fs_sip_module_class extends Fs_module_class
{
    protected $_obj_base_path = '';
    private $_sip_profiles = array();
    private $_sip_trunk = array();
    private $_db = "";
    /**
     * @brief
     *           
     * @param[in out] $fs_app
     *           
     * @return
     */
    public function __construct($fs_app)
    {
        parent::__construct($fs_app, "sip");
        $this->_obj_base_path = $fs_app->object_name().'/profile';
        $this->_db = $this->db();
    }
    public function allow_rest(){
        return true;
    }
    /**
     * @brief
     *         
     * @return
     */
    public function configure()
    {
        // Set the module description
        $this->set_description("SIP");
        // Expose aggregate obj
        $this->register_aggregate_object('trunk', 
            array(
                'name' => 'Trunk',
                'dynamic' => true,
                'base_path' => $this->_obj_base_path . '/sip_trunk',
                // TODO: Adjust to non NSC related URL
                'controller_url' => '/SAFe/nsc_sip_trunk',
                    'methods' => array(
                            'create' => array(
                                    'name' => 'Create',
                                    'description' => 'Create a SIP Trunk',
                                    'request' => 'POST',
                            ),
                            'retrieve' => array(
                                    'name' => 'Retrieve',
                                    'description' => 'Retrieve a SIP Trunk',
                                    'request' => 'GET',
                            ),
                            'update' => array(
                                    'name' => 'Update',
                                    'description' => 'Update a SIP Trunk',
                                    'request' => 'POST',
                            ),
                            'delete' => array(
                                    'name' => 'Delete',
                                    'description' => 'Delete a SIP Trunk',
                                    'request' => 'POST',
                            ),
                    ),
            )
        );
        $this->register_aggregate_object('profile', 
            array(
                'name' => 'Profile',
                'dynamic' => true,
                'base_path' => $this->_obj_base_path . '/sip',
                // TODO: Adjust to non NSC related URL
                'controller_url' => '/SAFe/nsc_sip_profiles',
                'methods' => array(
                    /*
                    'create' => array(
                        'name' => 'Create',
                        'description' => 'Create a SIP Profile',
                        'request' => 'POST',
                        ),
                     */
                    'retrieve' => array(
                        'name' => 'Retrieve',
                        'description' => 'Retrieve a SIP Profile',
                        'request' => 'GET',
                        ),
                        /*
                    'update' => array(
                        'name' => 'Update',
                        'description' => 'Update a SIP Profile',
                        'request' => 'POST',
                        ),
                    'delete' => array(
                        'name' => 'Delete',
                        'description' => 'Delete a SIP Profile',
                        'request' => 'POST',
                        ),
                  // All other methods
                      'start' => array(
                          'name' => 'Start',
                          'descripton' => 'Start a SIP Profile',
                          'request' => 'POST',
                          ),
                      'stop' => array(
                          'name' => 'Stop',
                          'descripton' => 'Stop a SIP Profile',
                          'request' => 'POST',
                          ),
                      'flush_registration' => array(
                          'name' => 'Flush Registration(s)',
                          'descripton' => 'Flush SIP Registration(s)',
                          'request' => 'POST',
                          'class' => 'Fs_sip_profile_flush_reg_config', // Or anything alike to hold the flush parameters
                          ),
                         // Removed the get_ prefix from all the function above
                     'channels' => array(
                         'name' => 'Get Channels',
                         'descripton' => 'Retrieve Channels...',
                         'request' => 'GET',
                         ),
                     'registrations' => array(
                         'name' => 'Get Channels',
                         'descripton' => 'Retrieve registrations...',
                         'request' => 'GET',
                         ),
                     'status' => array(
                         'name' => 'Get Channels',
                         'descripton' => 'Retrieve status...',
                         'request' => 'GET',
                         ),
                     'gateway_status' => array(
                         'name' => 'Get Channels',
                         'descripton' => 'Retrieve gateway_status...',
                         'request' => 'GET',
                         ),
                         */
                     ),
                 )
             );

        return parent::configure();
    }
    /**
     * Returns sip profiles in an array
     */
    public function profiles()
    {
        $this->_sip_profiles = $this->get_aggregate_objects('profile');
        return $this->_sip_profiles;
    }
    /**
     * Returns sip trunk profiles in an array
     */
    public function trunks()
    {
        $this->_sip_trunks = $this->get_aggregate_objects('trunk');
        return $this->_sip_trunks;
    }
    /**
     * @brief Module support reload ?
     *
     * @param[in out] $need_reload
     *
     * @return 
     */
    public function support_reload(&$need_reload = null) {
        parent::support_reload($need_reload);
        return true;
    }
    /**
     * @brief Create config jobs for dialplan module
     *           
     * @param[in out] $config_manager
     *           
     * @return
     */
    public function reload_generate_config(&$config_manager, $obj_type=null)
    {
        if(!$this->_generate_config($config_manager, $obj_type)) return false;
        return parent::reload_generate_config($config_manager, $obj_type);
    }

    /**
     * @brief 
     *
     * @param[in out] $obj_type
     *
     * @return 
     */
    public function reload_clear_modified($obj_type=null)
    {
        // the hard way :)
        Safe_object_serializer_class::get_serializer()->clear_modified($this->_obj_base_path, true);
    }

    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    public function generate_config(&$config_manager)
    {
        if(!$this->_generate_config($config_manager)) return false;
        return parent::generate_config($config_manager);
    }
    /**
     * @brief Invoked after a successfull write_config
     *
     * @return
     */
    public function post_write_config($obj_type=null)
    {
        // Loop around profiles (ALL)
        $profiles = $this->get_aggregate_objects('profile', true);
        foreach($profiles as $profile) {
            // Proceed based on status
            switch($profile->status()) {
            case Safe_object_serializer_class::OBJ_STATUS_NEW:
                // stop the profile (in case the user created a new profile 
                // with the same name of an old profile), this command will fail 
                // most of the times
                $this->eslapi_execute( 'sofia profile '. $profile->name().' stop', '', $output);
                // start the new profile reading the new values from the XML
                $this->eslapi_execute( 'sofia profile '. $profile->name().' start', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_MODIFIED:
                // as simple as rescanning the new values from the XML
                $this->eslapi_execute( 'sofia profile '. $profile->name().' rescan', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_UP_TO_DATE:
                break;
            case Safe_object_serializer_class::OBJ_STATUS_DELETED:
                // stop the profile is equivalent to delete it from memory 
                $this->eslapi_execute( 'sofia profile '. $profile->name().' stop', '', $output);
                break;
            }
        }
    
        // Loop around trunks (ALL)
        $trunks = $this->get_aggregate_objects('trunk', true);
        foreach($trunks as $trunk) {
            $profile_name = $trunk->get_data_value('sip_profile', false);
            // Proceed based on status
            switch($trunk->status()) {
            case Safe_object_serializer_class::OBJ_STATUS_NEW:
                $this->eslapi_execute( 'sofia profile '. $profile_name.' rescan', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_MODIFIED:
                $this->eslapi_execute( 'sofia profile '. $profile_name.' killgw '.$trunk->name(), '', $output);
                $this->eslapi_execute( 'sofia profile '. $profile_name.' rescan', '', $output);
                break;
            case Safe_object_serializer_class::OBJ_STATUS_DELETED:
                $this->eslapi_execute( 'sofia profile '. $profile_name.' killgw '.$trunk->name(), '', $output);
                break;
            }
        }
    
        return parent::post_write_config($obj_type);
    }
    /**
     * @brief 
     *
     * @param[in out] $config_manager
     *
     * @return 
     */
    private function _generate_config(&$config_manager, $obj_type=null)
    {
        // Validate
        // - Trunks
        if(!$obj_type || 'trunk' == $obj_type) {
        }
        // - Profiles
        if(!$obj_type || 'profile' == $obj_type) {
            if (!count($this->profiles())) {
                // At least one SIP profile is required
                $config_manager->add_error('Validating', 'No SIP profile defined.');
                //return false;
            }
        }

        // Cleanup
        // - Trunks
        if(!$obj_type || 'trunk' == $obj_type) {
        }
        // - Profiles
        if(!$obj_type || 'profile' == $obj_type) {
            // 1st, clean up SIP profile directory and xml
            foreach ($this->profiles() as $profile) {
                $directory = "sip_profiles/" . $profile->name();
                $config_manager->add_config(
                    new Safe_configuration_class(
                        $directory, '', 
                        Safe_configuration_class::CFG_DELETE, 
                        Safe_configuration_class::CFG_DIR));
                $config_manager->add_config(
                    new Safe_configuration_class(
                        'sip_profiles/'.$profile->name().'.xml', '', 
                        Safe_configuration_class::CFG_DELETE, 
                        Safe_configuration_class::CFG_FILE));
            }
        }

        // Generate
        // - Trunks
        if(!$obj_type || 'trunk' == $obj_type) {
            // SIP Trunk configuration
            foreach ($this->trunks() as $profile) {
                if (!$this->_write_trunk($config_manager, $profile)) return false;
            }
        }
        // - Profiles
        if(!$obj_type || 'profile' == $obj_type) {
            //Serialize SIP Profiles
            foreach ($this->profiles() as $profile) {
                if (!$this->_write_profile($config_manager, $profile)) return false;
            }
            //Serialize SIP Profile Certificates
            if (!$this->_serialize_sip_certificates($config_manager)) return false;
            //Serialize SIP Profile Core Dialplan with CAC Rules
            $this->_create_core_dir($config_manager);
            foreach ($this->profiles() as $profile) {
                if (!$this->_write_sip_core($config_manager, $profile)) return false;
            }
        }

        // Handle deleted
        $this->_handle_deleted_trunks($config_manager);
        $this->_handle_deleted_profiles($config_manager);

        return true;
    }
    protected function _handle_deleted_trunks(&$config_manager)
    {
        //Handle Deleted SIP Trunk
        $deleted_sip_trunk = Safe_object_serializer_class::get_serializer()->find_deleted_objects(
            $this->_obj_base_path . '/sip_trunk');
        foreach ($deleted_sip_trunk as $sip_trunk => $value) {
            $file = 'sip_profiles/' . $value['data']['sip_profile'] . '/' . $sip_trunk . '.xml';
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
            $file = 'dialplan/core/sip_trunks/'.$sip_trunk.'.xml';
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
            //Check if carriers_gateway table populated
            if ($value['data']['carrier_name'] != "none") {
                //remove from carrier_gateway table
                //build data array
                unset($data);
                $data['db_class'] = $this->db();
                $data['data'] = $value['data'];
                $data['primary_key'] = array(
                    'id'
                );
                $config_manager->add_config(
                    new Safe_configuration_class('carrier_gateway', 
                    $data, 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_SQL));
            }
        }
    }
    protected function _handle_deleted_profiles(&$config_manager)
    {
        // Handle deleted SIP profiles
        $deleted_sip_profiles = Safe_object_serializer_class::get_serializer()->find_deleted_objects(
            $this->_obj_base_path . '/sip');
        foreach ($deleted_sip_profiles as $sip_profile => $value) {
            // Delete configuration file
            $file = 'sip_profiles/' . $sip_profile . '.xml';
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
            //Remove Core Dialplan File
            $file = "dialplan/core/profiles/profile-" . $sip_profile . ".xml";
            $config_manager->add_config(
                new Safe_configuration_class(
                    $file, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_FILE));
            //Remove certificate folder on delete
            $directory = 'ssl/profiles/' . $sip_profile;
            $config_manager->add_config(
                new Safe_configuration_class(
                    $directory, '', 
                    Safe_configuration_class::CFG_DELETE, 
                    Safe_configuration_class::CFG_DIR));
        }
        return true;
    }
    /**
     * Creates the Core dialplan folder
     * @param object  $config_manager
     * @return boolean
     */
    protected function _create_core_dir(&$config_manager)
    {
        // Create directory
        $directory = "dialplan/core/profiles";
        $config_manager->add_config(
            new Safe_configuration_class(
                $directory, '', 
                Safe_configuration_class::CFG_CREATE, 
                Safe_configuration_class::CFG_DIR));
        return true;
    }
    /**
     * Write Dialplan for sip profiles with cac rules
     * @param object  $config_manager
     * @param object  $profile       
     *                                 
     * @return boolean
     */
    protected function _write_sip_core(&$config_manager, $profile)
    {
        //Create core dialplan file for the sip profile
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0', 'UTF8');
        $xml_writer->startElement("include");
        $xml_writer->startElement("context");
        $xml_writer->writeAttribute('name', 'profile-' . $profile->name());
        //Secure Media Element
        $xml_writer->startElement('extension');
        $xml_writer->writeAttribute('name', 'add_media_secure');
        $xml_writer->writeAttribute('continue', 'true');
        $xml_writer->startElement('condition');
        $xml_writer->writeAttribute('field', '${sip_has_crypto}');
        $xml_writer->writeAttribute('expression', '^(AES_CM_128_HMAC_SHA1_32|AES_CM_128_HMAC_SHA1_80)$');
        $xml_writer->StartElement('action');
        $xml_writer->writeAttribute('application', 'set');
        $xml_writer->writeAttribute('data', 'sip_secure_media=true');
        $xml_writer->endElement();
        $xml_writer->fullEndElement();
        $xml_writer->fullEndElement();
        //Limit Exceeded Element
        $xml_writer->startElement('extension');
        $xml_writer->writeAttribute('name', 'limit_exceeded');
        $xml_writer->startElement('condition');
        $xml_writer->writeAttribute('field', 'destination_number');
        $xml_writer->writeAttribute('expression', '^limit_exceeded$');
        $xml_writer->startElement('action');
        $xml_writer->writeAttribute('application', 'log');
        $xml_writer->writeAttribute('data', 'info limit exceeded for IP: ${sip_received_ip} User:${sip_auth_username}@${sip_auth_realm}');
        $xml_writer->endElement();
        $xml_writer->startElement('action');
        $xml_writer->writeAttribute('application', 'hangup');
        $xml_writer->writeAttribute('data', 'CALL_REJECTED');
        $xml_writer->endElement();
        $xml_writer->fullEndElement();
        $xml_writer->fullEndElement();

        //CAC Rules binding
        $xml_writer->startElement('X-PRE-PROCESS');
        $xml_writer->writeAttribute('cmd', 'optional_include');
        $xml_writer->writeAttribute('data', '../../cac-rules/rules-' . $profile->name() . '.xml');
        $xml_writer->endElement();
        
        //Transfer to User Element
        /*
        <extension name="transfer-to-user">
        <condition field="${sangoma.routing_context}" expression="^$" break="never">
          <action application="set" data="sangoma.routing_context=user-core-routing1"/>
        </condition>
        <condition field="destination_number" expression="(.*)">
          <action application="transfer" data="$1 XML ${sangoma.routing_context}"/>
        </condition>
        </extension>
        */
        $data = $profile->get_data_values(false);
        $xml_writer->startElement('extension');
        $xml_writer->writeAttribute('name', 'transfer-to-user');
        // Set routing context
        $xml_writer->startElement('condition');
        $xml_writer->writeAttribute('field', '${sangoma.routing_context}');
        $xml_writer->writeAttribute('expression', '^$');
        $xml_writer->writeAttribute('break', 'never');
        $xml_writer->StartElement('action');
        $xml_writer->writeAttribute('application', 'set');
        $xml_writer->writeAttribute('data', 'sangoma.routing_context=' . $data['call-routing']);
        $xml_writer->endElement();
        $xml_writer->endElement(); /* end </condition> */
        $xml_writer->startElement('condition');
        $xml_writer->writeAttribute('field', 'destination_number');
        $xml_writer->writeAttribute('expression', '(.*)');
        $xml_writer->StartElement('action');
        $xml_writer->writeAttribute('application', 'transfer');
        $xml_writer->writeAttribute('data', '$1 XML ${sangoma.routing_context}');
        // Close all opened elements
        while ($xml_writer->endElement() !== false) {
            continue;
        }
        // Close document
        $xml_writer->endDocument();
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = "dialplan/core/profiles/profile-" . $profile->name() . ".xml";
        $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));
        return true;
    }
    /**
             * Serializes all SIP Certificates placed in a SIP Profile
             * @param $config_manager
             */
    protected function _serialize_sip_certificates(&$config_manager)
    {
        // Let certificate module do its job :)
        return true;
    }
    protected function _write_profile(&$cfg_mgr, $sip_profile)
    {
        $profile_name = $sip_profile->name();
        $no_limit = false;
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0', 'UTF-8');
        $xml_writer->startElement("profile");
        $xml_writer->writeAttribute("name", $profile_name);
        $this->_write_profile_default_tags($xml_writer, $profile_name);

        $sip_profile_cfg = $sip_profile->get_data_values_exploded(false);
        foreach ($sip_profile_cfg['__all__'] as $k => $v) {
            if ($k == "sip-ip") {
                // Get back ip from interface list
                $transcoding_mode = new Sng_transcoding_config_class($this->node());
                $mgmt_if = $this->node()->hardware()->adapters();
                if (isset($mgmt_if) && isset($mgmt_if[$v]) && strlen($mgmt_if[$v]->ip_address())) {
                    if (!$transcoding_mode->require_transcoding_interface() && Safe_hardware_adapter_class::TRANSCODING_ADAPTER == $mgmt_if[$v]->type()) {
                        $cfg_mgr->add_error('Validating', 'SIP Profile cannot use ' . $v . ' interface when in Hidden Mode.', '/SAFe/nsc_sip_profiles/modify/' . $profile_name, 'Adjust SIP Profile');
                    }
                    $v = $mgmt_if[$v]->ip_address();
                } else {
                    // TODO(wadam) - We're in trouble...
                    $cfg_mgr->add_error('Validating', 'SIP profile ' . $profile_name . ' bind to interface ' . $v . ' without IP address.',
                    // TODO(wadam) - Redirect to Managment IF config page
                    //'/SAFe/ntg_sip_profiles/edit/'.$profile_name,
                    '/admin/network.php', 'Configure');
                    $v = '0.0.0.0';
                    //return false;
                    
                }
            }
            // Do some filtering on param
            switch ($k) {
            case 'transport':
                $this->_write_transport_param($xml_writer, $v);
                $transport_value = $v;
                break;

            case 'domains':
            case 'call-routing':
                break;

            case 'inbound-media-profile':
                if($v) {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../media_profiles/" . $v . "/inbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case 'outbound-media-profile':
                if($v) {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../media_profiles/" . $v . "/outbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case 'sip-tos-value':
                if ($v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'sip-tos-value');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'rtp-tos-value':
                if ($v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'rtp-tos-value');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'enable-load-limit':
                if ($v == "false") $no_limit = true;
                elseif ($v == "true") $no_limit = false;
                $xml_writer->startElement('param');
                $xml_writer->writeAttribute('name', 'enable-load-limit');
                $xml_writer->writeAttribute('value', $v);
                $xml_writer->EndElement();
                break;

            case 'max-sessions':
                if (!$no_limit && $v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'max-sessions');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-high-threshold':
                if (!$no_limit) {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-high-threshold');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-low-threshold':
                if (!$no_limit) {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-low-threshold');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-reject-cause':
                if (!$no_limit) {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-reject-cause');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'load-limit-cause-string':
                if (!$no_limit && $v != "") {
                    $xml_writer->startElement('param');
                    $xml_writer->writeAttribute('name', 'load-limit-cause-string');
                    $xml_writer->writeAttribute('value', $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'rtp-ip':
            case 'ext-rtp-ip':
            case 'ext-sip-ip':
            case 'outbound-proxy':
            case 'apply-nat-acl':
            case 'fqdn-in-contact':
            case 'max-sip-request-length':
                if (strlen(trim($v))) {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;

            case 'sip-dialplan-handler':
                if ('default' != trim($v)) {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;

            default:
                if('__disable__' != $v) {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;
            }
        }
        //Add TLS Parameters
        if (sizeof($sip_profile_cfg['TLS'])) {
            $sip_tls = $sip_profile_cfg['TLS'];
            if ($transport_value == "udptcptls" || $transport_value == "tls") {
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", 'tls');
                $xml_writer->writeAttribute("value", 'true');
                $xml_writer->EndElement();
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", "tls-cert-dir");
                $xml_writer->writeAttribute("value", "/usr/local/nsc/conf/ssl/profiles/" . $profile_name);
                $xml_writer->EndElement();
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", "tls-bind-params");
                $xml_writer->writeAttribute("value", 'transport=tls');
                $xml_writer->EndElement();
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", "tls-verify-depth");
                $xml_writer->writeAttribute("value", '10');
                $xml_writer->EndElement();
                if ($transport_value == "tls") {
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", 'tls-only');
                    $xml_writer->writeAttribute("value", 'true');
                    $xml_writer->EndElement();
                }
                foreach ($sip_tls as $k => $v) {
                    if ($k == 'certificate' || $k == 'certificate-file-name' || $k == 'certificate-file-content') {
                    } else if ('__disable__' != $v){
                        //Print TLS params
                        $xml_writer->startElement("param");
                        $xml_writer->writeAttribute("name", $k);
                        $xml_writer->writeAttribute("value", $v);
                        $xml_writer->EndElement();
                    }
                }
                // Check srtp parameters
                if ($sip_tls['enable-secure-media'] == 'true'){
                    $srtp = $sip_profile_cfg['srtp'];
                    foreach ($srtp as $k => $v) {
                        if ('__disable__' != $v){
                            //Print SRTP params
                            $xml_writer->startElement("param");
                            $xml_writer->writeAttribute("name", $k);
                            $xml_writer->writeAttribute("value", $v);
                            $xml_writer->EndElement();
                        }
                    }
                }
            }
        }
        // Include custom xml
        $xml_writer->startElement("X-PRE-PROCESS");
        $xml_writer->writeAttribute("cmd", "optional_include");
        $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/sip_profiles/' . $profile_name . '.profile.settings.xml');
        $xml_writer->EndElement();
        
        $xml_writer->fullEndElement(); //end settings tag
        // Add the domains
        if (sizeof($sip_profile_cfg['__all__']['domains'])) {
            $sip_domains = $sip_profile_cfg['__all__']['domains'];
            $domains = $this->fs_app()->domain_profiles();
            $xml_writer->startElement("domains");
            foreach ($sip_domains as $sip_domain) {
                // Check domain exists
                if (isset($domains[$sip_domain])) {
                    $domain = $domains[$sip_domain];
                    $xml_writer->startElement("domain");
                    $xml_writer->writeAttribute("name", $domain->name());
                    $xml_writer->writeAttribute("alias", "false");
                    $xml_writer->writeAttribute("parse", "false");
                    $domain_data = $domain->get_data_values(false);
                    // Filter out registration if disable
                    if (isset($domain_data['forward-registration']) && 'true' != $domain_data['forward-registration']) {
                        // unset all forward registration info to avoid invalid config
                        unset($domain_data['registrar-server']);
                        unset($domain_data['registrar-port']);
                        unset($domain_data['registrar-profile']);
                        unset($domain_data['sip-force-expires']);
                        unset($domain_data['register-transport']);
                    } else {
                        // Forward to Profile ?
                        if (!isset($domain_data['registrar-profile']) || 'none' == $domain_data['registrar-profile']) {
                            unset($domain_data['registrar-profile']);
                        }
                    }
                    // In any cases remove forward registration flag
                    unset($domain_data['forward-registration']);
                    foreach ($domain_data as $k => $v) {
                        if($v){
                            $xml_writer->startElement("param");
                            $xml_writer->writeAttribute("name", $k);
                            $xml_writer->writeAttribute("value", $v);
                            $xml_writer->EndElement(); //end param tag
                        }
                    }
                    $xml_writer->EndElement(); //end domain tag
                    
                } else {
                    // TODO(wadam) - Log and error
                    // Should not occurs
                    
                }
            }
            $xml_writer->fullEndElement(); //end domains tag
            
        }
        //Serialize SIP Limits
        $limits = $sip_profile->request_limits();
        $xml_writer->startElement("limits");
        if (!empty($limits)) {
            foreach ($limits as $limit) {
                $limit_data = $limit->get_data_values();
                $xml_writer->startElement('request-rate');
                if ($limit_data['host'] == "") $xml_writer->writeAttribute("host", "ANY");
                else $xml_writer->writeAttribute("host", $limit_data['host']);
                $xml_writer->writeAttribute("method", $limit_data['method']);
                $xml_writer->writeAttribute("rate", $limit_data['limit'] . '/' . $limit_data['period']);
                $xml_writer->EndElement();
            }
        }
        $xml_writer->fullEndElement();
        $xml_writer->fullEndElement(); //end profile tag
        $xml_writer->endDocument();
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = "sip_profiles/" . $profile_name . ".xml";
        $cfg_mgr->add_config(new Safe_configuration_class($file, $xmlstr));
        return true;
    }
    protected function _write_profile_default_tags(&$xml_writer, $profile_name)
    {
        // Check if this profile is bound to some SIP trunks
        $trunks = $this->trunks();
        foreach ($trunks as $trunk) {
            $cfg = $trunk->get_data_values();
            if ($cfg['sip_profile'] == $profile_name) {
                // Include proxy information
                $xml_writer->startElement("gateways");
                $xml_writer->startElement("X-PRE-PROCESS");
                $xml_writer->writeAttribute("cmd", "include");
                $xml_writer->writeAttribute("data", $profile_name . "/*.xml");
                $xml_writer->EndElement();
                $xml_writer->fullEndElement();
                // break we just want to know if at least one gateway is bound
                break;
            }
        }
        // Open settings tag
        $xml_writer->startElement("settings");
        // Include parameters common to all sip profile
        $xml_writer->startElement("X-PRE-PROCESS");
        $xml_writer->writeAttribute("cmd", "include");
        $xml_writer->writeAttribute("data", "common/common.xml");
        $xml_writer->EndElement();
        // Start param list
        $xml_writer->startElement("param");
        $xml_writer->writeAttribute("name", "context");
        //$xml_writer->writeAttribute("value", self::NSC_CORE_ROUTING_CONTEXT);
        $xml_writer->writeAttribute('value', 'profile-' . $profile_name);
        $xml_writer->EndElement();
        $xml_writer->startElement("param");
        $xml_writer->writeAttribute("name", "dialplan");
        $xml_writer->writeAttribute("value", "XML");
        $xml_writer->EndElement();
        $xml_writer->startElement("param");
        $xml_writer->writeAttribute("name", "t38-passthru");
        $xml_writer->writeAttribute("value", "true");
        $xml_writer->EndElement();
    }
    protected function _write_transport_param(&$xml_writer, $transport_value)
    {
        if ($transport_value != "udptcp" && $transport_value != "udptcptls" && $transport_value != "tls") {
            $xml_writer->startElement("param");
            $xml_writer->writeAttribute("name", "bind-params");
            $xml_writer->writeAttribute("value", "transport=" . $transport_value);
            $xml_writer->EndElement();
        }
    }
    protected function _write_trunk(&$config_manager, $sip_trunk)
    {
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0', 'UTF-8');
        $sip_trunk_cfg = $sip_trunk->get_data_values(false, false);
        $profile_name = $sip_trunk->name();
        $xml_writer->startElement("include");
        $xml_writer->startElement("gateway");
        $xml_writer->writeAttribute("name", $profile_name);
        // Latch some values
        $carrier_name = $sip_trunk_cfg['carrier_name'];
        $sip_profile = $sip_trunk_cfg['sip_profile'];
        // Perform some validation with profile
        $profile_obj = $this->api_retrieve_profile($sip_profile);
        if($profile_obj){
            if(!$profile_obj->check_transport($sip_trunk->get_transport())){
                $trunk_obj_def = $this->get_aggregate_object_definition('trunk');
                if($trunk_obj_def['controller_url']){
                    $fix_url = $trunk_obj_def['controller_url'];
                    // TODO: redirect to edit trunk doesn't work...
                    // .'/edit/'.$profile_name;
                }
                $config_manager->add_error(
                    'Validating', 
                    'Trunk ' .$profile_name .' transport '. $sip_trunk->get_transport(true).' incompatible with Profile '.$profile_obj->name().' transport.',
                    $fix_url,
                    'Configure');
            }
        }
        // Remove from cfg list
        unset($sip_trunk_cfg['enabled']);
        unset($sip_trunk_cfg['carrier_name']);
        unset($sip_trunk_cfg['sip_profile']);

        // Check if registration enabled
        if('true' != $sip_trunk_cfg['register']){
            $sip_trunk_cfg['register-proxy'] = "";
        }

        foreach ($sip_trunk_cfg as $k => $v) {
            switch ($k) {
            case 'inbound-media-profile':
                if($v) {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../../media_profiles/" . $v . "/inbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case 'outbound-media-profile':
                if($v) {
                    $xml_writer->startElement("X-PRE-PROCESS");
                    $xml_writer->writeAttribute("cmd", "include");
                    $xml_writer->writeAttribute("data", "../../media_profiles/" . $v . "/outbound.xml");
                    $xml_writer->EndElement();
                }
                break;

            case "call-routing":
                $xml_writer->startElement("param");
                $xml_writer->writeAttribute("name", 'context');
                $xml_writer->writeAttribute("value", "sip-trunk-".$profile_name);
                $xml_writer->EndElement();
                break;

            default:
                if ($v == ""){
                    // Write the empty attribute but commented out
                    $xml_writer->startComment();
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                    $xml_writer->endComment();
                    break;
                }else{
                    $xml_writer->startElement("param");
                    $xml_writer->writeAttribute("name", $k);
                    $xml_writer->writeAttribute("value", $v);
                    $xml_writer->EndElement();
                }
                break;
            }
        }
        // Include custom xml
        $xml_writer->startElement("X-PRE-PROCESS");
        $xml_writer->writeAttribute("cmd", "optional_include");
        $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/sip_profiles/' . $sip_profile.'/'.$profile_name . '.gateway.xml');
        $xml_writer->EndElement();

        $xml_writer->fullEndElement(); //end gateway tag
        $xml_writer->fullEndElement(); //end include tag
        $xml_writer->endDocument();
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = "sip_profiles/" . $sip_profile . "/" . $profile_name . ".xml";
        $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));
        //update carrier gateway sql file
        //update carrier_name, enabled, prefix to the carrier_gateway table
        $prefix = 'sofia/gateway/' . $profile_name . '/';
        $this->_generate_trunk_db_config($config_manager, $prefix, $carrier_name, $sip_trunk);

        // Generates user dialplan
        $cfg = $sip_trunk->get_data_values(false);
        $xml_writer = new XMLWriter();
        $xml_writer->openMemory();
        $xml_writer->startDocument('1.0','UTF-8');

        $xml_writer->startElement("include");
        $xml_writer->startElement("context");
        $xml_writer->writeAttribute("name", "sip-trunk-".$sip_trunk->name());

        $xml_writer->startElement("extension");
        $xml_writer->writeAttribute("name", "transfer-to-profile");

        $xml_writer->startElement("condition");
        $xml_writer->writeAttribute("field", "destination_number");
        $xml_writer->writeAttribute("expression", "(.*)");

        $xml_writer->startElement("action");
        $xml_writer->writeAttribute("application", "set");
        if ($cfg['call-routing'] == 'default') {
            /* set the sangoma routing context empty to let the profile context decide where to go */
            $xml_writer->writeAttribute("data", "sangoma.routing_context=");
        } else {
            $xml_writer->writeAttribute("data", "sangoma.routing_context=" . $cfg['call-routing']);
        }
        $xml_writer->endElement(); //end action tag

        $xml_writer->startElement("action");
        $xml_writer->writeAttribute("application", "transfer");
        $xml_writer->writeAttribute("data", '$1 XML profile-${sofia_profile_name}');
        $xml_writer->endElement(); //end action tag


        $xml_writer->endElement(); //end condition tag
        $xml_writer->endElement(); //end extension tag
        $xml_writer->endElement(); //end context tag
        $xml_writer->endElement(); //end include tag

        $xml_writer->endDocument();
        // Create xml string
        $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
        $file = 'dialplan/core/sip_trunks/'.$sip_trunk->name().'.xml';
        // Instruct configuration manager to create the file
        $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));

        return true;
    }
    protected function _generate_trunk_db_config(&$config_manager, $prefix, $carrier_name, $sip_trunk)
    {
        //build data array
        unset($data);
        $db = $this->db();
        $data['db_class'] = $db;
        $data['data']['prefix'] = $prefix;
        $data['data']['id'] = $sip_trunk->id();
        $sip = $sip_trunk->get_data_values(false);
        if ($carrier_name && "none" != $carrier_name) {
            $data['data']['enabled'] = 1;
            $carrier_profiles = $this->fs_app()->carrier_profiles();
            $carrier = $carrier_profiles[$carrier_name];
            $data['data']['carrier_id'] = $carrier->id();
        } else {
            $data['data']['enabled'] = 0;
            $data['data']['carrier_id'] = "";
        }
        //Determine whether to update or insert a rule
        $status = $sip_trunk->status();
        if ($status == Safe_object_serializer_class::OBJ_STATUS_MODIFIED) {
            $data['primary_key'] = array(
                'carrier_id'
            );
            //Check if there is an existing entry in carrier_gateways for this particular carrier. If there is, it must be
            //removed. There should be no carrier_gateway entry is a sip trunk doesn't use a carrier
            $db->connect();
            $array['id'] = $data['data']['id'];
            $pk = array(
                'id' => $data['data']['id']
            );
            $result = $db->select('carrier_gateway', $pk);
            $db->disconnect();
            if ($sip['carrier_name'] == "none") {
                if (!empty($result)) {
                    foreach ($result as $row) {
                        if ($row->carrier_id != "") {
                            $data['primary_key'] = array(
                                'id'
                            );
                            $config_manager->add_config(new Safe_configuration_class('carrier_gateway', $data, Safe_configuration_class::CFG_DELETE, Safe_configuration_class::CFG_SQL));
                            break;
                        }
                    }
                }
            } else {
                if (!empty($result)) {
                    foreach ($result as $row) {
                        if ($row->carrier_id != "") {
                            $data['primary_key'] = array(
                                'id'
                            );
                            $config_manager->add_config(new Safe_configuration_class('carrier_gateway', $data, Safe_configuration_class::CFG_UPDATE, Safe_configuration_class::CFG_SQL));
                            break;
                        } else {
                            $config_manager->add_config(new Safe_configuration_class('carrier_gateway', $data, Safe_configuration_class::CFG_CREATE, Safe_configuration_class::CFG_SQL));
                            break;
                        }
                    }
                } else $config_manager->add_config(new Safe_configuration_class('carrier_gateway', $data, Safe_configuration_class::CFG_CREATE, Safe_configuration_class::CFG_SQL));
            }
        } elseif ($status == Safe_object_serializer_class::OBJ_STATUS_NEW && $sip['carrier_name'] != "none") {
            $config_manager->add_config(new Safe_configuration_class('carrier_gateway', $data, Safe_configuration_class::CFG_CREATE, Safe_configuration_class::CFG_SQL));
        }
        return true;
    }
    /**
     * start a  SIP Profile
     * @param string $profile_name (SIP profile name to be start)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_start_profile($profile_name, $params = null, &$out_message = null)
    {
        //need to check $this->_sip_profiles loaded or not 
        if (isset($this->_sip_profiles[$profile_name])) {
            $this->eslapi_execute('sofia profile '.$profile_name.' start', '', $out_message);
            if (preg_match("/\bsuccessfully\b/i", $out_message)){
                //Reload XML [Success]
                //NSC_SIP_Profile1 started successfully
                return true;
            }else{
                //Invalid Profile [NSC_SIP_Profile2]
                return false;
            }
        }else{
            $out_message[0] = 'No sip profile found.';
            return false;
        }
    }
    /**
     * stop a  SIP Profile
     * @param string $profile_name (SIP profile name to be stop)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_stop_profile($profile_name, $params = null, &$out_message = null)
    {
        // Set the basic data
        if (isset($this->_sip_profiles[$profile_name])) {
            $this->eslapi_execute('sofia profile '.$profile_name.' stop', '', $out_message);
            if (preg_match("/\bstopping\b/i", $out_message)){
                //Reload XML [Success]
                //stopping: NSC_SIP_Profile1
                return true;
            }else{
                //Invalid Profile [NSC_SIP_Profile2]
                return false;
            }
        }else{
            $out_message[0] = 'No sip profile found.';
            return false;
        }
    }
    /**
     * flush  SIP Profile registrations
     * @param string $profile_name (SIP profile name to be stop)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_flush_registration_profile($profile_name, $params = null, &$out_message=null)
    {
        $call_id = safe_set_default($params,'call_id','');//reg call id
        
        if (isset($this->_sip_profiles[$profile_name])) {
            $this->eslapi_execute('sofia profile ' . $profile_name . ' flush_inbound_reg '.$call_id, '', $out_message);
            //+OK flushing all registrations matching specified call_id
            return true;
        }else{
            $out_message[0] = 'No sip profile found.';
            return false;
        }
    }
    /**
     * Returns the SIP Channel Status
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_channels_profile($profile_name, $params = null, &$out_message=null)
    {
        $uuid = safe_set_default($params,'uuid','');//channel uuid
        $limit = safe_set_default($params,'limit',100);//max return record number
        $offset = safe_set_default($params,'offset',0);//start from record number
        $select =    safe_set_default($params,'select','');//set select 

        $db = $this->_db;
        $db->connect();
        $conditions = "";
        if($uuid!=''){
            $conditions = "uuid = '$uuid'";
        }
        $order_by = 'created';
        $result = $db->select('channels', $conditions, $limit, $offset, $order_by, '', $select, '');
        $db->disconnect();
        if(count($result)>0){
            return $result;
        }else{
            return false;
        }
    }
    /**
     * Returns the SIP Profile Registrations
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_registration_profile($profile_name, $params = null, &$out_message = null)
    {
        $reg_name = safe_set_default($params,'reg_name','');//reg_name
        $limit = safe_set_default($params,'limit',100);//max return record number
        $offset = safe_set_default($params,'offset',0);//start from record number
        $select =    safe_set_default($params,'select','');//set select 
        
        $db = $this->_db;
        $db->connect();
        $conditions = "";
        if($profile_name!=''){
            $conditions = "profile_name ='$profile_name'";
            if($reg_name!=''){
                $conditions = $conditions .' AND '. "CONCAT(sip_user,'@',sip_host) = '$reg_name' ";
            }
        }
        $order_by = '';
        $result = $db->select('sip_registrations', $conditions, $limit, $offset, $order_by, '', $select, '');
        $db->disconnect();
        if(count($result)>0){
            if($select!=''){
                return $result;
            }else{
                $output = null;
                $i=0;
                foreach($result as $item){
                    $output[$i]['call-id']        = $item->call_id;
                    $output[$i]['user']           = $item->sip_user.'@'.$item->sip_host;
                    $output[$i]['contact']        = $item->contact;
                    $output[$i]['thru-contact']   = $item->thru_contact;
                    $output[$i]['agent']          = $item->user_agent;
                    //the $item->expires is different with the sofia return
                    $output[$i]['status']         = $item->status."($item->rpid) exp(1969-12-31 19:00:00) expsecs(-$item->expires)";
                    $output[$i]['host']           = $item->hostname;
                    $output[$i]['network-ip']     = $item->network_ip;
                    $output[$i]['network-port']   = $item->network_port;
                    $output[$i]['sip-auth-user']  = $item->sip_username;
                    $output[$i]['sip-auth-realm'] = $item->sip_realm;
                    $output[$i]['mwi-account']    = $item->mwi_user.'@'.$item->mwi_host;
                    $i++;
                }
            }
            return $output;
        }else{
            return false;
        }
    }
    /**
     * Returns the SIP Profile status
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param array &$out_message (Array output msg)
     */
    public function api_status_profile($profile_name = '', $params = null, &$out_message = null)
    {
        $output = '';
        if($profile_name==''){
            $this->eslapi_execute('sofia xmlstatus', '', $output);
        }else{
            $this->eslapi_execute('sofia xmlstatus profile ' . $profile_name, '', $output);
            
        }
        $xml_output = Safe_xml_to_array($output);
        return $xml_output;
    }
    /**
     * Returns the SIP Trunk Status
     * @param string $profile_name (SIP profile name)
     * @param array $params (parameter array)
     * @param string &$out_message (Array output msg)
     */
    public function api_gateway_status_profile($profile_name = '', $params = null, &$out_message = null)
    {
        $output = '';
        $this->eslapi_execute('sofia xmlstatus gateway ' . $profile_name , '', $output);
        try{
            $xml_output = Safe_xml_to_array($output);
        }catch(exception $e){
            $out_message = $output;
            $xml_output = false;
        }
        return $xml_output;
    }

    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_profile($name, $data=null, &$output = null) {
        $profile = new Fs_sip_profile_class($this->node(), $this->_obj_base_path, $name);
        $profile->configure();
        if($data) {
            $profile->set_data_values($data);
        }
        return $profile;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_profile($name, $data=null, &$output = null) {
        $profiles = $this->profiles();
        return $profiles[$name];
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_profile($name, $data=null, &$output = null) {
        $profile = $this->api_retrieve_profile($name);
        if($profile) {
            if ($profile->validate($data,$output)) {
                if (true == $profile->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_profile($name, $data=null, &$output = null) {
        $profile = $this->api_retrieve_profile($name);
        if($profile) {
            if($profile->can_dispose($output)) {
                // Run validation
                return $profile->dispose();
            }
        }
        return false;
    }
    
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_create_trunk($name, $data=null, &$output = null) {
        $trunk = new Fs_sip_trunk_class($this->node(), $this->_obj_base_path, $name);
        $trunk->configure();
        if($data) {
            $trunk->set_data_values($data);
        }
        return $profile;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_retrieve_trunk($name, $data=null, &$output = null) {
        $trunk = $this->trunks();
        return $trunk[$name];
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_update_trunk($name, $data = null, &$output = null){
        $trunk = $this->api_retrieve_trunk($name);
        if($trunk) {
            if ($trunk->validate($data,$output)) {
                if (true == $trunk->save()) {
                    return true;
                }
            }
        }
        return false;
    }
    /**
     * @brief
     *
     * @param[in out] $name
     * @param[in out] $data
     * @param[in out] $output
     *
     * @return
     */
    public function api_delete_trunk($name, $data=null, &$output = null) {
        $trunk = $this->api_retrieve_trunk($name);
        if($trunk) {
            if($trunk->can_dispose($output)) {
                return $trunk->dispose();
            }
        }
        return false;
    }
    
    public function checklist(&$list)
    {
        if (!$this->profiles()){
            $list[] = array(
                'module' => 'sip',
                'obj_type' => 'profile',
                'description' => 'Not defined',
            );
        }
    }    
}
/* End of file fs_sip_profile_module_class.php */

