<?php
/**
 * FS Radius Module Class
 * @author: Shaunt Libarian
 *
 */

require_once('application/helpers/safe_helper.php');
safe_require_class('safe_service_class');
safe_module_require_class('fs','application');
safe_module_require_class('fs','radius_config');
safe_module_require_class('fs','radius_vsas_config');

class Fs_radius_module_class extends Fs_module_class  {

	private $_node;
	private $_vsas=array();
	
	public function __construct($fs_app) {
		parent::__construct($fs_app, "radius");
		$this->_node = $this->node();
	}
	
	public function configure(){
		// Set the module description
		$this->set_description("RADIUS");
		// Create the config object
		$cfg = new Fs_radius_config_class( $this->_node, $this->path(),$this->name());
		// Synch with DB
		$cfg->configure();
		$cfg->synch();
	
		// Attach config to module
		$this->set_config($cfg);

    // Register aggregated objects
    $this->register_aggregate_object('vsas', 'VSAs');
	
		return parent::configure();
	}
	
	/**
	 * Returns the radius module names
	 * @see Fs_module_class::get_fs_module_name()
	 */
	public function get_fs_module_name() {
		//two module for radius
		$mod['mod_rad_auth.so'] = 'mod_rad_auth.so';
		$mod['mod_radius_cdr.so'] = 'mod_radius_cdr.so'; 
		
		return $mod;
	}
	
	/**
	 * Whether radius module is enabled or disabled
	 * @see Fs_module_class::is_enabled()
	 */
	public function is_enabled($mod_name="") {
		//mod_rad_auth.so
	    //mod_radius_cdr.so -->disabled if accounting turned off
	    
	    $cfg = $this->config()->get_data_values(false);
	
	    if ($mod_name == "mod_rad_auth.so") {
			if("" != $cfg['authserver'] && "" != $cfg['bindaddr'])
				return true;
			else
				return false;
	    }
	   elseif ($mod_name == "mod_radius_cdr.so") {
	      //Check if radius accounting enabled
	      if ($cfg['enable-accounting'] == "true")
	      	return true;
	      else
	      	return false;
	    }
	}
	
  /**
    * @brief Create a specific aggregate object 
    *
    * @param[in out] $name - Name of object
    *
    * @return 
   */
  public function create_aggregate_object($type,$name){
    return new Fs_radius_vsas_config_class(
      $this->_node, $this->fs_app()->object_name().'/'.$this->name().'/vsas', $name);
  }
	/**
	 * Returns VSAS Configuration
	 * @return array
	 * 			Returns an array of VSAS Configuration objects
	 */
	public function vsas() {
		$vsas = Safe_object_serializer_class::get_serializer()->find_objects($this->fs_app()->object_name().'/'.$this->name().'/vsas');

		foreach($vsas as $k=>$v) {
			$v = new Fs_radius_vsas_config_class($this->_node,$this->fs_app()->object_name().'/'.$this->name().'/vsas',$k);
			$v->configure();
			$v->synch();
			$this->_vsas[$k] = $v;
		}

		return $this->_vsas;
	}
	
	/**
	 * Returns the name of the aggregate config object
	 * @return string
	 */
	public function aggregate_config_name(&$short_name=NULL) {
    if(isset($short_name))
      $short_name = 'vsas';
		return 'VSAs';
	}
	

	/**
	 * Serializes the Radius configuration for Radius and VSAS
	 * @see Safe_module_class::generate_config()
	 */
	public function generate_config(&$config_manager){
		$cfg = $this->config();
		if (!empty ($cfg)) {
			// Get our configuration data (unresolved)
			$radius_cfg = $cfg->get_data_values(false);
			$vsas = $this->vsas();
	
			
			$transcoding_mode = new Sng_transcoding_config_class($this->node());
			$mode = $transcoding_mode->ip_mode();
			
			//Create rad_auth.conf.xml
			$xml_writer = new XMLWriter();
			$xml_writer->openMemory();
			$xml_writer->startDocument('1.0','UTF-8');
		
			$xml_writer->startElement('configuration');
			$xml_writer->writeAttribute('name','rad_auth.conf');
			$xml_writer->writeAttribute('description','RADIUS Authentication Configuration');
			
			//Radius Client XML
			$xml_writer->startElement('client');
	
			$xml_writer->startElement('param');
			$xml_writer->writeAttribute('name','authserver');
			$xml_writer->writeAttribute('value',$radius_cfg['authserver'].":".$radius_cfg['auth_port'] . ":" . $radius_cfg['shared-secret']);
			$xml_writer->endElement();
			
			foreach($radius_cfg as $k=>$v) {
				if ($k != 'enable-accounting' && $k != 'acct-port' && $k != 'authserver' && $k != "auth_port" && $k != "shared-secret") {
					//Check interface name as IP not required for radius
					if (!$transcoding_mode->require_transcoding_interface() && $k == "bindaddr" && preg_match("'^[Ss]ng[dD]sp'",$v)) {
						$config_manager->add_error(
								'Validating',
								'Radius cannot use '.$v.' interface when in Hidden Mode.',
								'/SAFe/nsc_radius/modify/nsc/radius',
								'Adjust Radius Configuration');
					}
					
					$xml_writer->startElement('param');
					$xml_writer->writeAttribute('name',$k);
					$xml_writer->writeAttribute('value',$v);
					$xml_writer->endElement();
				}
			}
			
			//Constant Elements
			$xml_writer->startElement('param');
			$xml_writer->writeAttribute('name','dictionary');
			$xml_writer->writeAttribute('value','/usr/local/nsc/etc/radiusclient/dictionary');
			$xml_writer->endElement();
			$xml_writer->startElement('param');
			$xml_writer->writeAttribute('name','seqfile');
			$xml_writer->writeAttribute('value','/usr/local/nsc/run/radius.seq');
			$xml_writer->endElement();
			$xml_writer->startElement('param');
			$xml_writer->writeAttribute('name','radius_deadtime');
			$xml_writer->writeAttribute('value','0');
			$xml_writer->endElement();
			
			$xml_writer->fullEndElement();
			
			//Radius VSAS configuration
			$xml_writer->startElement('vsas');
			if (!empty ($vsas)) {
				foreach ($vsas as $object) {
					if (isset ($object)) {
						$prof_data = $object->get_data_values(false);
						$xml_writer->startElement('param');
						$xml_writer->writeAttribute('name',$object->name());
						$xml_writer->writeAttribute('id',$prof_data['vsa-id']);
						$xml_writer->writeAttribute('value',$prof_data['vsa-value']);
						$xml_writer->writeAttribute('pec',$prof_data['vendor-id']);
						
            $xml_writer->writeAttribute('expr',$prof_data['vsa-value-type']);
            $xml_writer->writeAttribute('direction',$prof_data['radius-message']);
            $xml_writer->endElement();
					}
				}
			}
			$xml_writer->fullEndElement();
			$xml_writer->fullEndElement();
			
			$xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
			$file = 'autoload_configs/rad_auth.conf.xml';
			$config_manager->add_config(new Safe_configuration_class($file, $xmlstr));
			
			//Create mod_radius_cdr.conf.xml if Radius Accounting is turned on
			if ($radius_cfg['enable-accounting'] == "true") {
				$xml_writer = new XMLWriter();
				$xml_writer->openMemory();
				$xml_writer->startDocument('1.0','UTF-8');
				
				$xml_writer->startElement('configuration');
				$xml_writer->writeAttribute('name','mod_radius_cdr.conf');
				$xml_writer->writeAttribute('description','RADIUS CDR Configuration');
				
				$xml_writer->startElement('settings');
				
				$xml_writer->startElement('param');
				$xml_writer->writeAttribute('name','acctserver');
				$xml_writer->writeAttribute('value',$radius_cfg['authserver'].':'.$radius_cfg['acct-port'].':'.$radius_cfg['shared-secret']);
				$xml_writer->endElement();

				$xml_writer->startElement('param');
				$xml_writer->writeAttribute('name','radius_timeout');
				$xml_writer->writeAttribute('value',$radius_cfg['radius_timeout']);
				$xml_writer->endElement();

				$xml_writer->startElement('param');
				$xml_writer->writeAttribute('name','radius_retries');
				$xml_writer->writeAttribute('value',$radius_cfg['radius_retries']);
				$xml_writer->endElement();

				//Constant Elements
				$xml_writer->startElement('param');
				$xml_writer->writeAttribute('name','dictionary');
				$xml_writer->writeAttribute('value','/usr/local/nsc/etc/radiusclient/dictionary.cdr');
				$xml_writer->endElement();
				$xml_writer->startElement('param');
				$xml_writer->writeAttribute('name','seqfile');
				$xml_writer->writeAttribute('value','/usr/local/nsc/run/radiuscdr.seq');
				$xml_writer->endElement();
				$xml_writer->startElement('param');
				$xml_writer->writeAttribute('name','radius_deadtime');
				$xml_writer->writeAttribute('value','0');
				$xml_writer->endElement();
				
				$xml_writer->fullEndElement();
				$xml_writer->fullEndElement();
				
				$xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
				$file = 'autoload_configs/mod_radius_cdr.conf.xml';
				$config_manager->add_config(new Safe_configuration_class($file, $xmlstr));
			}
			else {
				//delete the file if it exists
				$file = $config_manager->directory('configuration').'/autoload_configs/mod_radius_cdr.conf.xml';
				if (file_exists($file) ) {
					$file = 'autoload_configs/mod_radius_cdr.conf.xml';
					$config_manager->add_config(new Safe_configuration_class($file,
						'',
						Safe_configuration_class::CFG_DELETE,
						Safe_configuration_class::CFG_FILE));
				}
			}
		}
		return parent::generate_config($config_manager);
	}
}
?>
