<?php
/**
 * FS SIP Profile Class
 *
 * author: Shaunt Libarian
 */

require_once('application/helpers/safe_helper.php');
safe_require_class('object');
safe_require_class('hardware');
safe_module_require_class('fs','application');
safe_module_require_class('sng','transcoding_config');

class Fs_sip_profile_class extends Safe_configurable_object_class {

  /**
    * @brief
    *
    * @param[in out] $node
    * @param[in out] $parent_name
    * @param[in out] $name
    *
    * @return
   */
    public function __construct($node, $parent_name, $name) {
      // Parent constructor to invoke unserialize if needed
      parent::__construct($parent_name, $name, $node);
    }
    
    /**
      * @brief
      *
      * @return
     */
  protected function configure_general()
  {
    // Default interface name
    $def_if = '';
    // Populate IP enumeration with all management interface IP
    $transcoding_mode = new Sng_transcoding_config_class($this->_node);
    $adapter_type = $transcoding_mode->adapter_type();

    $mgmt_if = $this->_node->hardware()->adapters($adapter_type);
    $if_enum = array();
    foreach($mgmt_if as $if_name=>$if) {
      if(strlen($if->ip_address()))
      {
        if(!strlen($def_if))
          $def_if = $if->name();

        $if_enum[$if->name()] = $if->name() . ' - ' . $if->ip_address();
      }
    }
    // We must have at least one entry or we're in trouble
    if(!strlen($if_enum)){
      // TODO(wadam) - Do something
    }

    // Create object fields
    $this->add_field('user-agent-string', 'User Agent', 'string', '', 50);
    $this->set_field_rules('user-agent-string', 'required');

    $this->add_enum_field('sip-ip', 'SIP IP Address', 'dropdown', $def_if, $if_enum);
    $this->set_field_rules('sip-ip', 'required');
    $this->set_field_help('sip-ip','IP Address being used for SIP signalling.');
    // Add External IP address
    $this->add_field('ext-sip-ip', 'External SIP IP Address', 'string', '', 50);
    $this->set_field_rules('ext-sip-ip','valid_ext_sip_ip');
    $this->set_field_help('ext-sip-ip',
      array("The accepted values are:",
      "1. An IP address string such as '200.1.2.3'",
      "All SIP requests in this profile will use the specified IP address",
      "",
      "2. The string 'auto-nat'",
      "Gateway will use uPNP or NAT-PMP to discover the public IP address it should use",
      "",
      "3. A string starting with 'stun:' followed by a host name or IP, such as: 'stun:my-stun-server.com'",
      "Gateway will use the specified STUN server to discover the public IP address",
      "",
      "4. A string starting with 'host:' followed by a host name, such as 'host:my-small-office.com'",
      "Gateway will resolve the given host name to find out the IP address to use (useful for dynamic DNS)"));

    $this->add_field('sip-port', 'Port', 'string', '5060');
    $this->set_field_rules('sip-port','required|integer|greater_than[0]');
    $this->set_field_help('sip-port','Port being used for SIP Signalling');

    $this->add_enum_field('transport', 'Transport', 'dropdown', 'udptcp',
      array(
        'udp'     => 'UDP',
        'tcp'     => 'TCP',
        'tls' => 'TLS',
        'udptcp'  => 'UDP+TCP',
        'udptcptls' => 'UDP+TCP+TLS'
      ) );
    $this->set_field_help('transport','Transport type being used for SIP.');

    $this->add_field('outbound-proxy', 'Outbound Proxy', 'string', '');
    $this->set_field_rules('outbound-proxy', 'valid_ip_port_domain');
    $this->set_field_help('outbound-proxy', 'Optional outbound proxy');

    // RTP IP address
    $this->add_field('rtp-ip', 'RTP IP address', 'string', '', 25);
    $this->set_field_rules('rtp-ip', 'valid_ip');
    $this->set_field_help('rtp-ip',
      array(
        "This is the RTP IP to be used for the RTP network connection. For this field the only accepted value is an IP address string such as '192.168.100.51'.",
        "",
        "If left empty the same IP address used for the signaling will be used for RTP.",
        "",
        "If RTP is handled via exposed hardware (ie 'exposed' media mode) this parameter is ignored."
	));

    // External RTP IP address
    $this->add_field('ext-rtp-ip', 'External RTP IP address', 'string', '', 50);
    $this->set_field_rules('ext-rtp-ip','valid_ext_sip_ip');
    $this->set_field_help('ext-rtp-ip',
      array(
        "This is the RTP address to be advertised in the SDP for any SDP offer or answer.",
        "",
        "If RTP is handled via exposed hardware (ie 'exposed' media mode) this parameter is ignored.",
        "",
        "The accepted values are:",
        "1. An IP address string such as '200.1.2.4'",
        "",
        "2. The string 'auto-nat'",
        "Gateway will use uPNP or NAT-PMP to discover the public IP address it should use",
        "",
        "3. A string starting with 'stun:' followed by a host name or IP, such as: 'stun:my-stun-server.com'",
        "Gateway will use the specified STUN server to discover the public IP address",
        "",
        "4. A string starting with 'host:' followed by a host name, such as 'host:my-small-office.com'",
        "Gateway will resolve the given host name to find out the IP address to use (useful for dynamic DNS)",
      ));

    // Add General category
    $this->set_field_category('user-agent-string','General');
  }

  protected function configure_timing()
  {
    $this->add_enum_field('enable-timer', 'SIP Session Timer', 'dropdown', 'false', $this->enable_disable_enum());
    $this->set_field_help('enable-timer','Enable or Disable the SIP session timer.');

    $this->add_field('session-timeout', 'Session Expires', 'string', 1800);
    $this->set_field_rules('session-timeout','required[timer]|is_numeric');
    $this->set_field_help('session-timeout','Time in seconds for a specific session to expire.');

    $this->add_field('minimum-session-expires', 'Minimum Session Expires', 'string', 1800);
    $this->set_field_rules('minimum-session-expires','required[timer]|is_numeric');
    $this->set_field_help('minimum-session-expires','Minimum number of sessions that can be expired.');

    $this->add_field('rtcp-audio-interval-msec', 'RTCP Interval', 'string', '5000', 10);
    $this->set_field_rules('rtcp-audio-interval-msec', 'integer|greater_or_equal[100]|less_or_equal[5000]');
    $this->set_field_help('rtcp-audio-interval-msec','RTCP reports interval in msec.');

    // Create groups
    $this->create_group('timer', array('session-timeout', 'minimum-session-expires'));
    $this->conditional_control('enable-timer', 'timer');
    
    // Add Timing category
    $this->set_field_category('enable-timer','Timing');

  }
  protected function configure_interoperability()
  {
    $this->add_enum_field('enable-100rel', '100 Reliability', 'dropdown', 'false', $this->enable_disable_enum());
    $this->set_field_help('enable-100rel','SIP provisional message reliability.');

    $this->add_enum_field('enable-3pcc', '3PCC', 'dropdown', 'false',
		    array('false' => 'Disable', 'proxy' => 'Enable')
		    );
    $this->set_field_help('enable-3pcc','Enable processing of 3PCC (third party call control) to allow processing INVITE messages without an SDP');
    $this->add_enum_field('ignore-183nosdp', 'Ignore 183 without SDP', 'dropdown', 'false', $this->enable_disable_enum());
    $this->set_field_help('ignore-183nosdp','Enable or Disable ignoring 183 messages without a specific SDP.');

    $this->add_field('fqdn-in-contact', 'FQDN in Contact Header', 'string', '', 50);
    $this->set_field_rules('fqdn-in-contact', 'valid_ip_or_domain');
    $this->set_field_help('fqdn-in-contact','Use FQDN instead of IP address in Contact Header.');

    $this->add_field('max-sip-request-length', 'Maximum Sip Request URI Length', 'string', '', 10);
    $this->set_field_rules('max-sip-request-length', 'integer|greater_than[0]');
    $this->set_field_help('max-sip-request-length','Maximum length of Request URI.');

    $this->add_enum_field('notify-refer-on-final-rsp', 'Notify REFER on Final Response', 'dropdown', 'false', $this->enable_disable_enum());
    $this->set_field_help('notify-refer-on-final-rsp','Whether to determine REFER result base on final response. This applies only if a new call leg is created to the referee.');


    // Add Interoperability category
    $this->set_field_category('enable-100rel','Interoperability');
  }
  protected function configure_encryption()
  {
    //TLS Options
    $this->add_enum_field('TLS/tls-version','TLS Version','dropdown','tlsv1',array(
      'sslv23' => 'SSL Version 2 & 3',
      'tlsv1' => 'TLS Version 1'
    ));
    $this->set_field_help('TLS/tls-version','The version being used by SSL/TLS');

    //Insert Certificate stuff here
    $this->add_upload_field('TLS/certificate','TLS Certificate','upload','',30);
    $this->set_field_help('TLS/certificate','TLS/SSL certificate that will be used with this specific sip profile. Must be a valid PEM file.');
    $this->add_field('TLS/certificate-file-name','TLS Certificate File Name','hidden','',60);
    $this->add_field('TLS/certificate-file-content','TLS Certificate','hidden','',60);

    $this->add_field('TLS/tls-passphrase','TLS Passphrase','string','',60);
    $this->set_field_rules('TLS/tls-passphrase','alpha_dash');
    $this->set_field_help('TLS/tls-passphrase','If this SIP Profile is using a private key with a passphrase, you can enter the passphrase here.');

    $this->add_enum_field('TLS/tls-verify-date','Certificate Date Verification','dropdown','true',$this->enable_disable_enum());
    $this->set_field_help('TLS/tls-verify-date','Enable or Disable certificate date verification.');

    $this->add_enum_field('TLS/tls-verify-policy','Certificate Verification Policy','dropdown','out',array(
      'in' => 'Incoming',
      'out' => 'Outgoing',
      'all' => 'Incoming and Outgoing',
      'none' => 'No Verification'
    ));
    $this->set_field_help('TLS/tls-verify-policy','Enable certificate verification policy on incoming, outgoing, or all connections. Can be disabled by selecting No Verification');

    $this->add_field('TLS/tls-sip-port','TLS Port','string',5061,20);
    $this->set_field_rules('TLS/tls-sip-port','greater_than[1024]|less_than[65525]|is_numeric');
    $this->set_field_help('TLS/tls-sip-port','TCP Port required for secure SIP signalling');

    $this->add_enum_field('TLS/enable-secure-media','Secure RTP','dropdown','false',$this->enable_disable_enum());
    $this->set_field_help('TLS/enable-secure-media','Enable or Disable Secure RTP.');

    $this->add_enum_field('srtp/require-secure-rtp','Require Only Secure RTP','dropdown','false',$this->enable_disable_enum());
    $this->set_field_help('srtp/require-secure-rtp','Require inbound leg INVITE to include secure rtp offer.');

    $this->add_enum_field('srtp/support-sdp-secure-avp','Secure AVP','dropdown','false',$this->enable_disable_enum());
    $this->set_field_help('srtp/support-sdp-secure-avp', array('Enable or disable SDP secure AVP (a=crypto in RTP/AVP).','','Note: This option violates RFC 3711.'));

    $this->add_enum_field('srtp/crypto-optional-lifetime','Crypto Life Time','dropdown','__disable__',array(
      '__disable__' => 'Disable',
      '2^48' => 'High',
      '2^31' => 'Medium',
      '2^16' => 'Low',
    ));
    $this->set_field_help('srtp/crypto-optional-lifetime','Number of packets the SRTP crypto master key is valid for (RFC4568 lifetime optional sdp parameter). ');

    $this->add_enum_field('srtp/crypto-optional-mki-length-string','Crypto MKI Length','dropdown','__disable__',array(
      '__disable__' => 'Disable',
      '1:1' => '1:1',
    ));
    $this->set_field_help('srtp/crypto-optional-mki-length-string','The MKI length is the size of the MKI field in the SRTP packet.');

    // SRTP group
    $this->create_group('srtp', array('srtp/require-secure-rtp', 'srtp/support-sdp-secure-avp', 'srtp/crypto-optional-lifetime', 'srtp/crypto-optional-mki-length-string'));
    $this->conditional_control('TLS/enable-secure-media', 'srtp');

    // Add Encryption  category
    $this->set_field_category('TLS/tls-version','Encryption');
  }

  public function configure(){
    // General configuration
    $this->configure_general();
    // General Interop
    $this->configure_interoperability();
    // General Timer
    $this->configure_timing();
    // Encryption configuration
    $this->configure_encryption();

    return parent::configure();
  }

  // Application helpers
  public function get_sip_transport_info(&$transport, &$port,&$ip_addr=null){
    $transport = $this->get_data_value('transport', false);
    $port = $this->get_data_value('sip-port');
    $ip_addr = $this->get_data_value('sip-ip',false);
  }
  /**
   * @brief Check transport compatible with SIP profile transport
   *
   * @param[in out] $transport : transport string (udp, tcp...)
   *
   * @return 
   */
  public function check_transport($transport)
  {
    $_transport = $this->get_data_value('transport', false);
    return (FALSE === strpos($_transport, $transport))?false:true;
  }
  public function get_certificate_info(&$cert_name, &$cert_content)
  {
    if($this->check_transport('tls')){
      $cert_name = $this->get_data_value('TLS/certificate-file-name', false);
      $cert_content = $this->get_data_value('TLS/certificate-file-content', false);
      return ($cert_name)?true:false;
    }
    return false;
  }
}

?>
