<?php
/** vim: set tabstop=4 softtabstop=4 shiftwidth=4 textwidth=80 smarttab expandtab: **/
/** coding: utf-8: **/
/*
 * Copyright (C) 2012  Sangoma Technologies Corp.
 * All Rights Reserved.
 *
 * Author(s)
 * your name <your_name@sangoma.com>
 *
 * This code is Sangoma Technologies Confidential Property.
 * Use of and access to this code is covered by a previously executed
 * non-disclosure agreement between Sangoma Technologies and the Recipient.
 * This code is being supplied for evaluation purposes only and is not to be
 * used for any other purpose.
 */
/**
 * FS Directory Module Class
 * @author William Adam
 */

require_once('application/helpers/safe_helper.php');
safe_require_class('safe_service_class');
safe_module_require_class('fs','application');
safe_module_require_class('fs','domain');
safe_module_require_class('fs','domain_user');

/**
 * @brief FS Directory module class
 */
class Fs_directory_module_class extends Fs_module_class {

    /**
     * @brief Array of configured domains
     */
    private $_domains = array();

    /**
     * @brief Constructor
     *
     * @param[in out] $fs_app
     *
     * @return 
     */
    public function __construct($fs_app) {
        parent::__construct($fs_app, "directory");
    }

    /**
     * @brief Directory module configuration
     *
     * @return 
     */
    public function configure() {
        // Set the module description
        $this->set_description("Directory");

        // Register objects
        $this->register_aggregate_object('domain',
            array(
                'name' => 'Domain',
                'dynamic' => false, /* Domain objs are not dynamic, childs (users) are */
                'base_path' => $this->object_name() . '/domain',
                'has_child' => true,
            )
        );

        return parent::configure();
    }

    /**
     * @brief Return existing domain
     *
     * @return 
     */
    public function domains(){
        $this->_domains = $this->get_aggregate_objects('domain');
        return $this->_domains;
    }

    /**
     * @brief Create a domain
     * Can be overloaded by specialized class to create different type of domain 
     * object
     *
     * @param[in out] $domain_name
     *
     * @return 
     */
    public function create_domain($domain_name){
        return new Fs_domain_class($this->node(), $this->object_name().'/domain', $domain_name);
    }

    /**
     * Decides whether the module is enabled or disabled
     * @see Fs_module_class::is_enabled()
     */
    public function is_enabled($mod_name="") {
        return false;
    }

    public function support_reload(&$need_reload = null) {
        parent::support_reload($need_reload);
        return true;
    }

    /**
     * @brief Generates the domains configuration files and directories
     *
     * @param[in out] $config_manager
     *
     * @return
     */
    public function generate_config(&$config_manager){
        // Cleanup - Remove and create empty domains directories
        $directory = 'dialplan/core/domains';
        $config_manager->add_config(new Safe_configuration_class($directory,
            '',
            Safe_configuration_class::CFG_DELETE,
            Safe_configuration_class::CFG_DIR));
        $config_manager->add_config(new Safe_configuration_class($directory,
            '',
            Safe_configuration_class::CFG_CREATE,
            Safe_configuration_class::CFG_DIR));
        $directory = 'directory';
        $config_manager->add_config(new Safe_configuration_class($directory,
            '',
            Safe_configuration_class::CFG_DELETE,
            Safe_configuration_class::CFG_DIR));
        $config_manager->add_config(new Safe_configuration_class($directory,
            '',
            Safe_configuration_class::CFG_CREATE,
            Safe_configuration_class::CFG_DIR));
        // Get domains
        $domains = $this->domains();

        // For all domains
        foreach($domains as $domain){

            // Create configuration xml tree
            $xml_writer = new XMLWriter();
            $xml_writer->openMemory();
            $xml_writer->startDocument('1.0','UTF-8');

            $xml_writer->startElement("include");
            $xml_writer->startElement("domain");
            $xml_writer->writeAttribute("name", $domain->name());

            // Params section
            $xml_writer->startElement("params");
            $xml_writer->startElement("param");
            $xml_writer->writeAttribute("name", 'dial-string');
            $xml_writer->writeAttribute("value", '{sip_invite_domain=${dialed_domain},presence_id=${dialed_user}@${dialed_domain}}${sofia_contact(${dialed_user}@${dialed_domain})}');
            $xml_writer->EndElement(); //end param tag
            $xml_writer->fullEndElement(); //end params tag

            $xml_writer->startElement("groups");
            $xml_writer->startElement("group");
            $xml_writer->writeAttribute("name", 'default');
            // Add users
            // users tag
            $xml_writer->startElement("users");
            $xml_writer->startElement("X-PRE-PROCESS");
            $xml_writer->writeAttribute("cmd", "optional_include");
            $xml_writer->writeAttribute("data", $domain->name().'/*.xml');
            $xml_writer->EndElement();
            $xml_writer->fullEndElement(); //end users tag

            $xml_writer->fullEndElement(); //end group tag
            $xml_writer->fullEndElement(); //end groups tag
            $xml_writer->fullEndElement(); //end domain tag
            $xml_writer->fullEndElement(); //end include tag
            $xml_writer->endDocument();
            // Create xml string
            $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
            $file = 'directory/'.$domain->name().'.xml';
            // Instruct configuration manager to create the file
            $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));


            // create the dialplan/core/domains/<domain name>.xml include all users 
            // file
            $xml_writer = new XMLWriter();
            $xml_writer->openMemory();
            $xml_writer->startDocument('1.0','UTF-8');

            $xml_writer->startElement("include");

            // Params section
            $xml_writer->startElement("X-PRE-PROCESS");
            $xml_writer->writeAttribute("cmd", "optional_include");
            $xml_writer->writeAttribute("data", $domain->name().'/*.xml');
            $xml_writer->EndElement();

            $xml_writer->endDocument();
            // Create xml string
            $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
            $file = 'dialplan/core/domains/'.$domain->name().'.xml';
            // Instruct configuration manager to create the file
            $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));

            // Handle domain's users
            $this->generate_user_config($config_manager, $domain);
        }
        return parent::generate_config($config_manager);
    }

    /**
        * @brief Reload only for users
        *
        * @param[in out] $config_manager
        *
        * @return 
                 */
    public function reload_generate_config(&$config_manager, $obj_type=null){
        // Get domains
        $domains = $this->domains();

        // For all domains
        foreach($domains as $domain){
            // Handle domain's users
            $this->generate_user_config($config_manager, $domain);
        }
        return parent::generate_config($config_manager);
    }
    public function reload_clear_modified($obj_type=null)
    {
        // Get domains
        $domains = $this->domains();

        // For all domains
        foreach($domains as $domain){
            // the hard way :)
            Safe_object_serializer_class::get_serializer()->clear_modified($domain->object_name() , true);
        }
    }
    public function generate_user_config(&$config_manager, $domain){
        // Delete directory
        $directory = 'directory/'.$domain->name();
        $config_manager->add_config(new Safe_configuration_class($directory,
            '',
            Safe_configuration_class::CFG_DELETE,
            Safe_configuration_class::CFG_DIR));
        $directory = 'dialplan/core/domains/'.$domain->name();
        $config_manager->add_config(new Safe_configuration_class($directory,
            '',
            Safe_configuration_class::CFG_DELETE,
            Safe_configuration_class::CFG_DIR));

        // Handle users
        foreach($domain->users() as $user){
            $cfg = $user->get_data_values(false);
            $xml_writer = new XMLWriter();
            $xml_writer->openMemory();
            $xml_writer->startDocument('1.0','UTF-8');

            $xml_writer->startElement("include");
            $xml_writer->startElement("user");
            $xml_writer->writeAttribute("id", $user->name());

            // Params section
            $xml_writer->startElement("params");
            $xml_writer->startElement("param");
            $xml_writer->writeAttribute("name", 'password');
            $xml_writer->writeAttribute("value", $cfg['password']);
            $xml_writer->EndElement(); //end param tag
            
            // Add custom user params
            $xml_writer->startElement("X-PRE-PROCESS");
            $xml_writer->writeAttribute("cmd", "optional_include");
            $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/directory/'.$domain->name().'/'.$user->id().'.user.params.xml');
            $xml_writer->EndElement();
            $xml_writer->startElement("X-PRE-PROCESS");
            $xml_writer->writeAttribute("cmd", "optional_include");
            $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/directory/'.$domain->name().'/all.user.params.xml');
            $xml_writer->EndElement();

            $xml_writer->fullEndElement(); //end params tag
            // Variables section
            $xml_writer->startElement("variables");
            if($cfg['effective_caller_id_name']){
                $xml_writer->startElement("variable");
                $xml_writer->writeAttribute("name", 'effective_caller_id_name');
                $xml_writer->writeAttribute("value", $cfg['effective_caller_id_name']);
                $xml_writer->EndElement(); //end variable tag
            }
            if($cfg['effective_caller_id_number']){
                $xml_writer->startElement("variable");
                $xml_writer->writeAttribute("name", 'effective_caller_id_number');
                $xml_writer->writeAttribute("value", $cfg['effective_caller_id_number']);
                $xml_writer->EndElement(); //end variable tag
            }
            $xml_writer->startElement("variable");
            $xml_writer->writeAttribute("name", 'user_context');
            $xml_writer->writeAttribute("value", "domain-user-".$user->name()."@".$domain->name());
            $xml_writer->EndElement(); //end variable tag

            // Add custom user variables
            $xml_writer->startElement("X-PRE-PROCESS");
            $xml_writer->writeAttribute("cmd", "optional_include");
            $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/directory/'.$domain->name().'/'.$user->id().'.user.variables.xml');
            $xml_writer->EndElement();
            $xml_writer->startElement("X-PRE-PROCESS");
            $xml_writer->writeAttribute("cmd", "optional_include");
            $xml_writer->writeAttribute("data", '$${base_dir}/conf/custom/directory/'.$domain->name().'/all.user.variables.xml');
            $xml_writer->EndElement();

            $xml_writer->fullEndElement(); //end variables tag

            $xml_writer->fullEndElement(); //end user tag
            $xml_writer->fullEndElement(); //end include tag
            $xml_writer->endDocument();
            // Create xml string
            $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
            //$file = 'directory/'.$domain->name().'/'.$user->name().'.xml';
            $file = 'directory/'.$domain->name().'/'.$user->id().'.xml';
            // Instruct configuration manager to create the file
            $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));

            // Generates user dialplan
            $xml_writer = new XMLWriter();
            $xml_writer->openMemory();
            $xml_writer->startDocument('1.0','UTF-8');

            $xml_writer->startElement("include");
            $xml_writer->startElement("context");
            $xml_writer->writeAttribute("name", "domain-user-".$user->name()."@".$domain->name());

            $xml_writer->startElement("extension");
            $xml_writer->writeAttribute("name", "transfer-to-profile");

            $xml_writer->startElement("condition");
            $xml_writer->writeAttribute("field", "destination_number");
            $xml_writer->writeAttribute("expression", "(.*)");

            $xml_writer->startElement("action");
            $xml_writer->writeAttribute("application", "set");
            if ($cfg['call-routing'] == 'default') {
                /* set the sangoma routing context empty to let the profile context decide where to go */
                $xml_writer->writeAttribute("data", "sangoma.routing_context=");
            } else {
                $xml_writer->writeAttribute("data", "sangoma.routing_context=" . $cfg['call-routing']);
            }
            $xml_writer->endElement(); //end action tag

            $xml_writer->startElement("action");
            $xml_writer->writeAttribute("application", "transfer");
            $xml_writer->writeAttribute("data", '$1 XML profile-${sofia_profile_name}');
            $xml_writer->endElement(); //end action tag


            $xml_writer->endElement(); //end condition tag
            $xml_writer->endElement(); //end extension tag
            $xml_writer->endElement(); //end context tag
            $xml_writer->endElement(); //end include tag

            $xml_writer->endDocument();
            // Create xml string
            $xmlstr = Safe_format_xml($xml_writer->outputMemory(true));
            // $file = 'dialplan/core/domains/'.$domain->name().'/'.$user->name().'.xml';
            $file = 'dialplan/core/domains/'.$domain->name().'/'.$user->id().'.xml';
            // Instruct configuration manager to create the file
            $config_manager->add_config(new Safe_configuration_class($file, $xmlstr));
        }
        return true;
    }

    /**
     * @brief Invoked after a successfull write_config
     *
     * @return 
     */
    public function post_write_config()
    {
        // Reload configuration
        if(!$this->fs_app()->reloadxml()) return false;
        return parent::post_write_config();
    }
}
?>
